/*
 * $RCSfile: RecordModel.java,v $
 *
 * Copyright (c) 1999-2003. Christian Heller. All rights reserved.
 *
 * This software is published under the GPL GNU General Public License.
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * http://www.resmedicinae.org
 * - Information in Medicine -
 */

package resmedicinae.record;

import org.apache.xerces.dom.*;
import org.apache.xerces.parsers.*;
import org.apache.xml.serialize.*;
import resmedicinae.record.topology.*;
import resmedicinae.record.treetable.*;
import org.resmedicinae.domain.healthcare.*;
import org.resmedicinae.domain.healthcare.description.*;
import org.resmedicinae.domain.healthcare.heading.problem.*;
import org.resmedicinae.domain.healthcare.unit.*;
import org.w3c.dom.*;

import cybop.basic.String;

import javax.swing.tree.*;
import java.awt.*;
import java.io.*;
import java.util.*;
import java.util.List;

/**
 * This class represents the Record model. Design patterns: - Application Facade to access Domain Objects
 * - Model of Model View Controller
 * @version $Revision: 1.3 $ $Date: 2003/02/21 09:52:41 $ $Author: tristan24 $
 * @author Christian Heller <christian.heller@tuxtax.de>
 * @author Jens Bohl <info@jens-bohl-professional.de>
 */
public class RecordModel extends HealthRecordViewModel {

    //
    // Children.
    //

    /** The health record. */
    public static final String HEALTH_RECORD = new String("health_record");

    /** The list of record entries as retrieved from the domain. */
    public static final String ENTRIES_LIST = new String("entries_list");

    /** The tree to display the entries in a configurable hierarchy. */
    public static final String ENTRIES_TREE = new String("entries_tree");
    //
    // ?? Jens
    //

    /** Polyon is filled with red plus alpha */
    public static String FILL_POLYGON_ALPHA = "fill_polygon_alpha";

    /** Polyon is filled with red */
    public static String FILL_POLYGON = "fill_polygon";

    /** Polyon is not filled */
    public static String DRAW_POLYGON = "draw_polygon";

    /** The ResMedicinae home directory. */
    public static final String RESMEDICINAE_HOME = "res_medicinae_home";

    /** The Record share directory. */
    public static final String RECORD_SHARE = "record_share";

    /** The name of the directory where the pictures are at. */
    public static final String PICTURE_DIR_NAME = "picture_directory_name";

    /** The relative name of the directory where the pictures are at. */
    public static final String RelativePictureDirectoryName = "relative_picture_directory_name";

    /** The name of the relative directory where the image xml files are at. */
    public static final String RelativeXMLImagesDirectoryName = "relative_xml_images_directory_name";

    /** The name of the relative directory where the HealthRecord xml files are at. */
    public static final String RelativeXMLPatientsDirectoryName = "relative_xml_patients_directory_name";

    /** The name of the relative directory where icons are at. */
    public static final String RelativeIconDirectoryName = "relative_icon_directory_name";

    /** This String represents no entry */
    public static final String NO_ENTRY_STRING = "no_entry";

    /** This color is the background color for components */
    public static final Color BackgroundColor = new Color(204, 204, 204);

    /** The name of the document shown first after start up */
    public static final String StartDocumentName = "start_document_name";

    /** Vector containing all patient documents */
    public static final String XML_PATIENT_DOCUMENTS_VECTOR = "xml_patient_documents_vector";

    /** Array for storing all image file names. This is needed as input for comboBoxes for example. */
    public static final String FILE_NAME_ARRAY = "file_name_array";

    /** Hashtable of patients where key are names and values are patient objects */
    public static final String PATIENT_HASHTABLE = "patient_hashtable";

    /** A vector containing all parsed documents */
    public static final String XML_IMAGE_DOCUMENTS = "xml_image_documents";

    /** A hashtable where the document name is the key and the document itself is the corresponding value. */
    public static final String IMAGE_HASHTABLE = "image_hashtable";

    /** A hashtable where the overlay object name is the key and the object itself is the corresponding value. */
    public static final String OVERLAY_HASHTABLE = "overlay_hashtable";

    /** The toolkit needed */
    public static final String TOOL_KIT = "tool_kit";

    /** The currently loaded document. */
    public static final String CURRENT_DOC = "current_doc";

    /** The vie component */
    public static final String RECORD_VIEW = "record_view";

    /** A vector representing the history of clicked images */
    public static final String HISTORY_VECTORY = "history_vector";

    /** Indicates whether a step is a step back or not. Step means executing method "changeFile" */
    public static final String HISTORY_STEP = "history_step";

    /** The image shown in the panel */
    public static final String CURRENT_IMAGE = "current_image";

    /** A vector containing all overlayRegions */
    public static final String CURRENT_OVERLAY_REGIONS_HASHTABLE = "current_overlay_regions_hashtable";

    /** The index of the region shown. */
    public static final String OVERLAY_REGIONS_INDEX = "overlay_regions_index";


    /** Calendar object for date and time */
    public static final String CALENDAR = "calendar";

    /** Reference to the current shown overlayRegion */
    public static final String CURRENT_OVERLAY_REGION = "current_overlay_region";

    /**
     *The controller of the Record Module. It is needed because of interaction between model and some
     * dialogs hosted by this controller. I am looking for a way to avoid this...
     */
    public static final String CONTROLLER = "controller";

    /** The alternaytive overlays to the current overlay */
    public static final String ALTERNATIVE_OVERLAYS = "alternative_overlays_for_current_overlay";

    /** Temporary string for storing the key of the hashmap */
    public static final String TEMPSTRING = "tempString";
    /**The current tree path.*/
    public static final String CURRENT_TREE_PATH = "current_tree_path";

    /** Constructor. ?? Remove later!! */
    public RecordModel(RecordView v, Record controller) throws Exception {
        tempString = new org.resmedicinae.resmedlib.term.String("tempString");
        setKit(Toolkit.getDefaultToolkit());
        calendar = Calendar.getInstance();
        setCalendar(calendar);
        setView(v);
        setXmlPatientDocuments(parseXMLPatientFiles());
        historyVector = new Vector();
        initializeImageHashtable(parseXMLImageFiles());
        setOverlayRegionIndex(0);
    }
    //
    // create, destroy, set, get methods.
    //

    /**
     * Returns the health record.
     * @return the health record
     */
    public HealthRecord getHealthRecord() {
        //?temporary solution
        return (HealthRecord)get(RecordModel.HEALTH_RECORD);       
    }

    /**
     * Sets the health record.
     * @param healthRecord the health record
     */
    public void setHealthRecord(HealthRecord healthRecord) {
        set(RecordModel.HEALTH_RECORD, healthRecord);
    }

    /*Sets the view
     *@param imageMap the view
    */

    public void setView(RecordView imageMap) {
        this.view = imageMap;
    }

    /**
     *Gets the view
     *@return the view
     */
    public RecordView getView() {
        return this.view;
    }

    /**
     * Initializes the image hashtable with an initial number of xml documents.
     * @param xmlImageDocuments the xml image documents
     * @exception NullPointerException if the xml image documents container is null
     */
    public void initializeImageHashtable(Vector xmlImageDocuments) throws NullPointerException {
        RecordView view = (RecordView)getView();
        setImageHashtable(new Hashtable());
        setOverlayHashtable(new Hashtable());
        Vector allAlternativeOverlays = new Vector();
        if (xmlImageDocuments != null) {
            // For every xmlImageDoc: put name and properties of the image in a hashtable
            // properties e.g. are the overlayRegions and the image own data structure for an image? (class?)
            for (int i = 0; i < xmlImageDocuments.size(); i++) {
                String overlayName = getNameFromDocument((Document) xmlImageDocuments.elementAt(i));
                String description = getDescriptionFromDocument((Document) xmlImageDocuments.elementAt(i));
                Image image = getImageFromDocument((Document) xmlImageDocuments.elementAt(i));
                Hashtable overlayRegions = getOverlayRegionsFromDocument((Document) xmlImageDocuments.elementAt(i));
                Hashtable diseases = getDiseasesFromDocument((Document) xmlImageDocuments.elementAt(i));
                Vector alternativeOverlays = new Vector();
                alternativeOverlays = getAlternativeOverlaysFromDocument((Document) xmlImageDocuments.elementAt(i));
                if(alternativeOverlays!=null){
                    for(int ii=0;ii<alternativeOverlays.size();ii++){
                        if(!(alternativeOverlays.elementAt(ii).equals(overlayName))){
                            allAlternativeOverlays.add(alternativeOverlays.elementAt(ii));
                        }
                    }
                }

                Overlay overlay = new Overlay();
                overlay.set(Overlay.NAME,overlayName);
                overlay.set(Overlay.DESCRIPTION, description);
                overlay.setImage(image);                
                overlay.setOverlayRegionHashtable(overlayRegions);
                overlay.setAlternativeOverlays(alternativeOverlays);
                overlay.set(Overlay.ALTERNATIVE, new Boolean(false));
                overlay.setDiseases(diseases);
                getImageHashtable().put(overlayName, xmlImageDocuments.elementAt(i));
                getOverlayHashtable().put(overlayName, overlay);
                setXmlImageDocuments(xmlImageDocuments);
            }
            //when all overlays are in the system, the allAlternativeOverlays vector is complete. Now these overlays
            //can be marked as "alternative"
            for(int u=0;u<allAlternativeOverlays.size();u++){
                Overlay alternativeOverlay = (Overlay)getOverlayHashtable().get(allAlternativeOverlays.elementAt(u));
                if(alternativeOverlay!=null){
                    alternativeOverlay.setAlternative(true);
                }
            }
        } else {
            throw new NullPointerException("Could not initialize image hashtable. The overlay documents container is null.");
        }
    }

    /**
     * Gets the name from an image document.
     * @param doc the document from which the name should be taken
     * @return the document name
     * @exception NullPointerException if the document is null
     */
    public String getNameFromDocument(Document d) throws NullPointerException {
        if (d != null) {
            NodeList l = d.getElementsByTagName("Name");
            if (l != null) {
                Node n = l.item(0);
                if (n != null) {
                    if (hasNodeValue(n)) {
                        return (getNodeValue(n));
                    } else {
                        System.out.println("Could not get name from overlay document. The name tag has no value.");
                        return RecordStrings.NO_ENTRY;
                    }
                } else {
                    System.out.println("Could not get name from overlay document. The name tag does not exist.");
                    return RecordStrings.NO_ENTRY;
                }
            } else {
                System.out.println("Could not get name from overlay document. The name tag does not exist.");
                return RecordStrings.NO_ENTRY;
            }
        } else {
            throw new NullPointerException("Could not get name from overlay document. The document is null.");
        }
    }
    /**
     *Gets alternative value from an overlay document
     *@param doc the document from which the alternative value should be taken
     *@return true if the overlay is alternative, false if not
     */
    public boolean getAlternativeValueFromDocument(Document doc) {
        if (doc != null) {
            NodeList imageList = doc.getElementsByTagName("IsAlternative");
            if(imageList!=null){
                if (imageList.item(0) != null) {
                    if (hasNodeValue(imageList.item(0))) {
                        String booleanString = getNodeValue(imageList.item(0));
                        if(booleanString.equals("true")){
                            return true;
                        }
                    }
                }
            }
            return false;
        } else {
            throw new NullPointerException("Could not get alternative value from overlay document. The document is null.");
        }
    }
    /**
     *Gets description from an image document
     *@param doc the document from which the description should be taken
     *@return name of document
     */
    public String getDescriptionFromDocument(Document doc) {
        if (doc != null) {
            NodeList imageList = doc.getElementsByTagName("Description");
            if (imageList.item(0) != null) {
                if (hasNodeValue(imageList.item(0))) {
                    return getNodeValue(imageList.item(0));
                } else {
                    System.out.println("Could not get description from orverlay document. The description tag has no value.");
                    return RecordStrings.NO_ENTRY;
                }
            } else {
                System.out.println("Could not get description from overlay document. The description tag does not exist.");
                return RecordStrings.NO_ENTRY;
            }
        } else {
            throw new NullPointerException("Could not get description from overlay document. The document is null.");
        }
    }

    /**
     *Gets image from an overlay document
     *@param doc the document from which the image should be taken
     *@return image of document
     */
    public Image getImageFromDocument(Document doc) {
        if (doc != null) {
            NodeList imageList = doc.getElementsByTagName("Image");
            if (hasNodeValue(imageList.item(0))) {
                return kit.getImage(RecordModel.RelativePictureDirectoryName + getNodeValue(imageList.item(0)));
            } else {
                System.out.println("Could not get image from document. The image tag has no value.");
                return null;
            }
        } else {
            System.out.println("Target Overlay not available.");
            return null;
        }
    }

    /**
     *Gets the alternative overlays from a document
     *@param doc the document from which the alternative overlays should be taken
     *@return vector containing alternative overlays
     */
    public Vector getAlternativeOverlaysFromDocument(Document doc) {
        Vector alternativeOverlaysVector = new Vector();
        Vector returnVector = new Vector();
        if (doc != null) {
            //get root element named <Alternatives>
            NodeList alternativesList = doc.getElementsByTagName("Alternatives");
            Node alternativeNode = alternativesList.item(0);
            if (alternativeNode != null) {
                NodeList alternativeList = alternativeNode.getChildNodes();
                for (int i = 0; i < alternativeList.getLength(); i++) {
                    System.out.println("Node Name: " + alternativeList.item(i).getNodeName());
                    if (alternativeList.item(i).getNodeName().equals("Alternative")) {
                        alternativeOverlaysVector.add(alternativeList.item(i));
                    }
                }
                System.out.println("Es existieren " + alternativeOverlaysVector.size() + " alternative Regionen.");
                for (int e = 0; e < alternativeOverlaysVector.size(); e++) {
                    Node alternativeOverlay = (Node) alternativeOverlaysVector.elementAt(e);
                    String alternativeOverlayName = getNodeValue(alternativeOverlay);
                    System.out.println("Adding " + alternativeOverlayName + " to List.");
                    returnVector.add(alternativeOverlayName);
                }
                return returnVector;
            } else {
                System.out.println("Could not get <Alternatives> Node. The node does not exist.");
                return null;
            }
        } else {
            throw new NullPointerException("Could not get alternative overlays from document. The document is null.");
        }
    }

    /**
     *Gets diseases from an overlay document
     *@param doc the document from which the overlayRegions should be taken
     *@return hashtable with diseases
     */
    public Hashtable getDiseasesFromDocument(Document doc){
        String diseaseName = new String();
        String diseaseDescription = new String();
        Hashtable diseaseHashtable = new Hashtable();
        if (doc != null) {
             NodeList diseaseNodeList = doc.getElementsByTagName("Disease");
             if (diseaseNodeList != null) {
                 for(int i=0;i<diseaseNodeList.getLength();i++){
                     Disease disease = new Disease();
                     NodeList diseaseProperyList = diseaseNodeList.item(i).getChildNodes();
                     for(int ii=0;ii<diseaseProperyList.getLength();ii++){
                         if(diseaseProperyList.item(ii).getNodeName().equals("DiseaseName")){
                            diseaseName = getNodeValue(diseaseProperyList.item(ii));
                         }else
                         if(diseaseProperyList.item(ii).getNodeName().equals("DiseaseDescription")){
                             diseaseDescription = getNodeValue(diseaseProperyList.item(ii));
                             System.out.println("DISEASE-DESCRIPTION: "+diseaseDescription);
                         }
                     }
                     disease.setDiseaseName(diseaseName);
                     disease.setDiseaseDescription(diseaseDescription);
                     diseaseHashtable.put(diseaseName, disease);
                 }
             }
        }
        return diseaseHashtable;
    }

    /**
     *Gets regions from an overlay document
     *@param doc the document from which the overlayRegions should be taken
     *@return vector with overlayRegions
     */
    public Hashtable getOverlayRegionsFromDocument(Document doc) {
        String x, y;
        Vector xvector = new Vector();
        Vector yvector = new Vector();
        Vector overlayRegionVector = new Vector();
        Vector returnVector = new Vector();
        Vector pointVector = new Vector();
        Vector neighbourVector = new Vector();
        String topNeighbour = new String();
        String bottomNeighbour = new String();
        String rightNeighbour = new String();
        String leftNeighbour = new String();
        if (doc != null) {
            //get root element named overlayRegions
            NodeList overlayRegionsList = doc.getElementsByTagName("Overlay");
            if (overlayRegionsList != null) {
                Node overlayRegions = overlayRegionsList.item(0);
                NodeList overlayRegionList = overlayRegions.getChildNodes();
                //get all nodes that are childnodes of overlayRegions and named "overlayRegion"
                //and store them in vector
                for (int i = 0; i < overlayRegionList.getLength(); i++) {
                    if (overlayRegionList.item(i).getNodeName().equals("OverlayRegion")) {
                        overlayRegionVector.add(overlayRegionList.item(i));
                    }
                }
                for (int e = 0; e < overlayRegionVector.size(); e++) {
                    Node overlayRegion = (Node) overlayRegionVector.elementAt(e);
                    NodeList overlayRegionPropertyList = overlayRegion.getChildNodes();
                    pointVector.clear();
                    String overlayRegionName = new String();
                    String overlayRegionTarget = new String();
                    String overlayRegionNumber = new String();
                    for (int r = 0; r < overlayRegionPropertyList.getLength(); r++) {
                        //add points into the point vector
                        if (overlayRegionPropertyList.item(r).getNodeName().equals("Point")) {
                            pointVector.add(overlayRegionPropertyList.item(r));
                        } else
                        //get name of overlayRegion
                            if (overlayRegionPropertyList.item(r).getNodeName().equals("Name")) {
                                if (hasNodeValue(overlayRegionPropertyList.item(r))) {
                                    overlayRegionName = getNodeValue(overlayRegionPropertyList.item(r));
                                } else {
                                    overlayRegionName = RecordStrings.NO_ENTRY;
                                }
                            } else
                            //get destination of overlayRegion
                                if (overlayRegionPropertyList.item(r).getNodeName().equals("TargetImage")) {
                                    if (hasNodeValue(overlayRegionPropertyList.item(r))) {
                                        overlayRegionTarget = getNodeValue(overlayRegionPropertyList.item(r));
                                    } else {
                                        overlayRegionTarget = RecordStrings.NO_ENTRY;
                                    }
                                } else if (overlayRegionPropertyList.item(r).getNodeName().equals("Number")) {
                                    if (hasNodeValue(overlayRegionPropertyList.item(r))) {
                                        overlayRegionNumber = getNodeValue(overlayRegionPropertyList.item(r));
                                    } else {
                                        overlayRegionNumber = RecordStrings.NO_ENTRY;
                                    }
                                } else if (overlayRegionPropertyList.item(r).getNodeName().equals("Neighbours")) {
                                    neighbourVector.add(overlayRegionPropertyList.item(r));
                                    for (int we = 0; we < neighbourVector.size(); we++) {
                                        Node neighbourNode = (Node) neighbourVector.elementAt(we);
                                        NodeList neighbourNodeList = neighbourNode.getChildNodes();
                                        for (int uu = 0; uu < neighbourNodeList.getLength(); uu++) {
                                            Node neighboursAroundNode = neighbourNodeList.item(uu);
                                            if (neighboursAroundNode.getNodeName().equals("Top")) {
                                                if (hasNodeValue(neighboursAroundNode)) {
                                                    topNeighbour = getNodeValue(neighbourNodeList.item(uu));
                                                } else {
                                                    topNeighbour = RecordStrings.NO_ENTRY;
                                                }
                                            } else if (neighboursAroundNode.getNodeName().equals("Bottom")) {
                                                if (hasNodeValue(neighboursAroundNode)) {
                                                    bottomNeighbour = getNodeValue(neighbourNodeList.item(uu));
                                                } else {
                                                    bottomNeighbour = RecordStrings.NO_ENTRY;
                                                }
                                            } else if (neighboursAroundNode.getNodeName().equals("Right")) {
                                                if (hasNodeValue(neighboursAroundNode)) {
                                                    rightNeighbour = getNodeValue(neighbourNodeList.item(uu));
                                                } else {
                                                    rightNeighbour = RecordStrings.NO_ENTRY;
                                                }
                                            } else if (neighboursAroundNode.getNodeName().equals("Left")) {
                                                if (hasNodeValue(neighboursAroundNode)) {
                                                    leftNeighbour = getNodeValue(neighbourNodeList.item(uu));
                                                } else {
                                                    leftNeighbour = RecordStrings.NO_ENTRY;
                                                }
                                            }
                                        }
                                    }
                                }
                    }
                    xvector.clear();
                    yvector.clear();
                    //points of the overlayRegion are examined
                    for (int t = 0; t < pointVector.size(); t++) {
                        Node point = (Node) pointVector.elementAt(t);
                        NodeList coordList = point.getChildNodes();
                        //coordinates of the points are examined
                        int xCoordCounter = 0;
                        int yCoordCounter = 0;
                        for (int u = 0; u < coordList.getLength(); u++) {
                            //coordinates are taken first in a vector ....
                            if (coordList.item(u).getNodeName().equals("X")) {
                                if (hasNodeValue(coordList.item(u))) {
                                    xvector.add(getNodeValue(coordList.item(u)));
                                    xCoordCounter++;
                                } else {
                                    System.out.println("Warning! Missing x-Coordinate in overlay document!");
                                }
                            } else if (coordList.item(u).getNodeName().equals("Y")) {
                                if (hasNodeValue(coordList.item(u))) {
                                    yvector.add(getNodeValue(coordList.item(u)));
                                    yCoordCounter++;
                                } else {
                                    System.out.println("Warning! Missing y-Coordinate in overlay document!");
                                }
                            }
                        }
                    }
                    //...and then into an array
                    int[] xarray = new int[xvector.size()];
                    int[] yarray = new int[yvector.size()];
                    for (int p = 0; p < xvector.size(); p++) {
                        try {
                            xarray[p] = Integer.parseInt((String) xvector.elementAt(p));
                            yarray[p] = Integer.parseInt((String) yvector.elementAt(p));
                        } catch (Exception exception) {
                        }
                    }
                    //new imageoverlayRegions are created with the specific properties
                    OverlayRegion imageMapOverlayRegion = new OverlayRegion(xarray, yarray, xvector.size());
                    imageMapOverlayRegion.set(OverlayRegion.NAME, overlayRegionName);
                    imageMapOverlayRegion.set(OverlayRegion.TARGET, overlayRegionTarget);
                    imageMapOverlayRegion.set(OverlayRegion.NUMBER, overlayRegionNumber);
                    imageMapOverlayRegion.set(OverlayRegion.RIGHT_NEIGHBOUR, rightNeighbour);
                    imageMapOverlayRegion.set(OverlayRegion.LEFT_NEIGHBOUR, leftNeighbour);
                    imageMapOverlayRegion.set(OverlayRegion.TOP_NEIGHBOUR, topNeighbour);
                    imageMapOverlayRegion.set(OverlayRegion.BOTTOM_NEIGHBOUR, bottomNeighbour);
                    returnVector.add(imageMapOverlayRegion);
                }
                Hashtable returnHashtable = new Hashtable(returnVector.size());
                for (int z = 0; z < returnVector.size(); z++) {
                    String overlayRegionName = ((OverlayRegion) returnVector.elementAt(z)).getName();
                    returnHashtable.put(overlayRegionName, (OverlayRegion) returnVector.elementAt(z));
                }
                return returnHashtable;
            } else {
                System.out.println("Overlay besitzt nicht die richtige Struktur. Root-Element <overlay> fehlt.");
                return null;
            }
        } else {
            throw new NullPointerException("Could not get overlay region from document. The document is null");
        }
    }

    /**
     *Increases the number a string contains
     * @param string the string
     * @return the incremented string
     */
    public String incrementString(String string) {
        return (new Integer(Integer.parseInt(string) + 1)).toString();
    }

    /**
     *Tests whether a node has a value or not
     * @param node the node
     * @return true if the node has a value, false if not
     */
    public boolean hasNodeValue(Node node) {
        if (node.getFirstChild() != null) {
            return true;
        } else {
            return false;
        }
    }

    /**
     *Gets the value from a particular node
     * @param node the node
     * @return the node value if the node has one, otherwise a string indicating the node has no value
     */
    public String getNodeValue(Node node) {
        if (hasNodeValue(node)) {
            return node.getFirstChild().getNodeValue();
        } else {
            return RecordStrings.NO_ENTRY;
        }
    }

    /**
     *Sets the value for a particular node
     * @param node the node
     * @return the node value if the node has one, otherwise a string indicating the node has no value
     */
    public void setNodeValue(Node node, String value) {
        if (getCurrentDocument() != null) {
            if (hasNodeValue(node)) {
                node.getFirstChild().setNodeValue(value);
            } else {
                node.appendChild(getCurrentDocument().createTextNode(value));
            }
        } else {
            throw new NullPointerException("Could not set node value. The the current document is null");
        }
    }

    /**
     * Parses the xml file with the given name.
     * @param f the file name
     * @return the document corresponding to the xml file
     */
    public Document parseXMLFile(String f) {
        Document d = null; //?? new DocumentImpl();
        DOMParser p = new DOMParser();
        if (p != null) {
            try {
                p.setFeature("http://xml.org/sax/features/validation", true);
                p.parse(f);
            } catch (Exception e) {
                System.out.println("Could not parse xml file. An Exception occured: " + e);
            }
            d = p.getDocument();
            d.normalize();
        } else {
            throw new NullPointerException("Could not parse xml file. The dom parser is null.");
        }
        return d;
    }

    /**
     * Parses all xml patient files in the xml directory.
     * @return the vector containig xml patient documents parsed from the xml-directory
     */
    public Vector parseXMLPatientFiles() {
        this.xmlPatientDocuments = new Vector();
        File xmlDirectory = new File(RecordModel.RelativeXMLPatientsDirectoryName);
        if (xmlDirectory == null) {
            System.out.print("Pfad zu den Patienten XML-Dateien ist null. Bitte Prfen: ");
            System.out.println(RecordModel.RelativeXMLPatientsDirectoryName);
        }
        //getting all files
        String[] patientArray = xmlDirectory.list(new XMLFileNameFilter());
        if (patientArray == null) {
            System.out.print("Im Pfad zu den Patienten XML-Dateien wurden keine Dateien gefunden. Bitte prfen: ");
            System.out.println(RecordModel.RelativeXMLPatientsDirectoryName);
        } else {
            for (int i = 0; i < patientArray.length; i++) {
                //parse the single file
                this.xmlPatientDocuments.add(parseXMLFile("share/application/healthcare/record/xml/patients/" +
                        patientArray[i]));
            }
        }
        return this.xmlPatientDocuments;
    }

    /**
     * Parses all xml image files in the xml directory.
     * @return the vector containig xml image documents parsed from the xml-directory
     */
    public Vector parseXMLImageFiles() {
        Vector xmlImageDocuments = new Vector();
        File xmlDirectory = new File(RecordModel.RelativeXMLImagesDirectoryName);
        if (xmlDirectory == null) {
            System.out.print("Pfad zu den Overlay XML-Dateien ist null. Bitte Prfen: ");
            System.out.println(RecordModel.RelativeXMLImagesDirectoryName);
        }
        //getting all files
        fileNameArray = xmlDirectory.list(new XMLFileNameFilter());
        if (fileNameArray == null) {
            System.out.print("Im Pfad zu den Overlay XML-Dateien wurden keine Dateien gefunden. Bitte prfen: ");
            System.out.println(RecordModel.RelativeXMLImagesDirectoryName);
        } else {
            for (int i = 0; i < fileNameArray.length; i++) {
                // Parse the single file.
                xmlImageDocuments.add(parseXMLFile("share/application/healthcare/record/xml/overlays/" + fileNameArray[i]));
                // Use only name of file. No extension in fileNameArray.
                fileNameArray[i] = deleteFileExtension(fileNameArray[i]);
            }
        }
        return xmlImageDocuments;
    }

    /**
     * Deletes the filename extension from a given string
     * @return string without extension
     */
    public String deleteFileExtension(String file) {
        return file.substring(0, file.lastIndexOf("."));
    }

    /**
     * Sets all image file names to a specific array
     * @param newFileNameArray the file name array to be set
     */
    public void setFileNames(String[] newFileNameArray) {
        this.fileNameArray = new String[newFileNameArray.length];
        this.fileNameArray = newFileNameArray;
    }

    /**
     * Returns all image file names
     * @return array containing all overlay file names
     */
    public String[] getFileNames() {
        return this.fileNameArray;
    }

    /**
     *Deletes fileName from arry of file names
     *@param fileName file name to be removed from the file name array
     */

    /*public void deleteFileNameFromFileNameArray(String fileName) {
        String[] tempArray = new String[getFileNames().length - 1];
        int tempArrayCounter = 0;
        for (int i = 0; i < getFileNames().length; i++) {
            if (!(getFileNames()[i] == fileName)) {
                tempArray[tempArrayCounter] = getFileNames()[i];
                tempArrayCounter++;
            }
        }
        setFileNames(tempArray);
    }*/

    /**
     *Gets the patient hashtable
     *@return hashtable with patients
     */
    public Hashtable getPatientHashtable() {
        return this.patientHashtable;
    }

    /**
     *Sets the patient hashtable
     *@param patientHashtable the patient hashtable
     */
    public void setPatientHashtable(Hashtable patientHashtable) {
        this.patientHashtable = patientHashtable;
    }

    /**
     *Creates new image document, saves the xml file, updates hashtable and comboBoxes
     *@param newDocument the overlay document to be inserted
     */
    public void insertNewImageDocument(Document newDocument) {
        saveOverlayToXMLFile(newDocument);
        updateImageHashtable(newDocument);
        updateComboBoxes(newDocument);
    }

    /**
     *Deletes image document and the xml file, updates hashtable and comboBoxes
     *@paran documentName the name of the overlay document to be deleted
     */
    public void deleteOverlayDocument(String documentName) {
        File fileToDelete = new File(RecordModel.RelativeXMLImagesDirectoryName + documentName + ".xml");
        fileToDelete.delete();
        //getView().deleteItemFromComboBoxes(documentName);
        deleteTargetOverlayFromXMLFiles(documentName);
        //deleteFileNameFromFileNameArray(documentName);
        this.imageHashtable.remove(documentName);
    }

    /**
     *Gets a document by the specific name
     *@param documentName the name of the document
     *@return document with documentName
     */
    public Document getDocumentByName(String documentName) {
        return (Document) this.imageHashtable.get(documentName);
    }

    /**
     *Deletes a target from all xml files it appears in and writes back the xml file
     *@param targetImage the overlay to be deleted
     */
    public void deleteTargetOverlayFromXMLFiles(String targetImage) {
        for (int u = 0; u < getFileNames().length; u++) {
            Document doc = (Document) this.imageHashtable.get(getFileNames()[u]);
            NodeList targetImageNodeList = doc.getElementsByTagName("TargetImage");
            for (int i = 0; i < targetImageNodeList.getLength(); i++) {
                if (hasNodeValue(targetImageNodeList.item(i))) {
                    if (getNodeValue(targetImageNodeList.item(i)).equals(targetImage)) {
                        //targetImageNodeList.item(i).removeChild(targetImageNodeList.item(i).getFirstChild());
                        setNodeValue(targetImageNodeList.item(i),"NO TARGET OVERLAY GIVEN");
                        getImageHashtable().remove(getFileNames()[u]);
                        getImageHashtable().put(getFileNames()[u], doc);
                        saveOverlayToXMLFile(doc);
                    }
                }
            }
        }
    }

    /**
     * Changes an image file. This image will be shown in the image panel.
     * @param fileName the file name
     * @exception Exception if the hashtable is empty
     * @exception Exception if the file was not found in the hashtable
     */
    public void changeOverlay(String fileName) throws Exception {
        getView().getHistoryBackButton().setEnabled(true);
        if (!getImageHashtable().isEmpty()) {
            if (getImageHashtable().containsKey(fileName)) {
                if ((Document) getImageHashtable().get(fileName) != null) {
                    setCurrentDocument((Document) getImageHashtable().get(fileName));
                } else {
                    System.out.println("Warning!!!!" + fileName + " not in overlay hashtable!!!");
                    System.out.println("Possible reason: a target overlay or alternative overlay referenced by");
                    System.out.println("an overlay does not exist as a xml file. Please make sure that " + fileName + ".xml");
                    System.out.println("exists.");
                }
                if (getImageFromDocument(getCurrentDocument()) != null) {
                    setCurrentImage(getImageFromDocument(getCurrentDocument()));
                }
                setCurrentOverlayRegions(getOverlayRegionsFromDocument(getCurrentDocument()));
                getView().getImagePanel().setImage(getCurrentImage());
                getView().getImagePanel().updateImage();
                String description = getDescriptionFromDocument(getCurrentDocument());
                getView().getImageInfoTextArea().setText(description);
                //this.setCurrentObject(null);
            } else {
                throw new Exception("Could not change file. The file was not found in the hashtable.");
            }
        } else {
            throw new Exception("Could not change file. The hashtable is empty.");
        }
    }

    /**
     * Changes a prticular overlayRegion in the current document.
     * @param oldName the old name of the region
     * @param newName the new name of the region
     * @param newTargetImage the new target of the region
     */
    public void editOverlayRegion(String oldName, String newName, String newTargetImage) throws Exception {
        changeOverlayRegionInDocument(getCurrentDocument(), oldName, newName, newTargetImage);
    }

    /**
     *This method edites the overlayRegion and makes the chanes persistent. Changeable values are
     * for example the overlayRegion name an target.
     *@param doc the document taht should be edited
     * @param oldName the old name of the region
     * @param newName the new name of the region
     * @param newTargetImage the new target of the region
     */
    public void changeOverlayRegionInDocument(Document doc, String oldName, String newName,
                                              String newTargetImage) throws Exception {
        Node nodeToChange = null;
        NodeList overlayRegionNodeList = doc.getElementsByTagName("Name");
        for (int r = 0; r < overlayRegionNodeList.getLength(); r++) {
            if (hasNodeValue(overlayRegionNodeList.item(r))) {
                if (getNodeValue(overlayRegionNodeList.item(r)).equals(oldName)) {
                    nodeToChange = overlayRegionNodeList.item(r).getParentNode();
                }
            }
        }
        if (nodeToChange != null) {
            NodeList overlayRegionPropertyList = nodeToChange.getChildNodes();
            for (int r = 0; r < overlayRegionPropertyList.getLength(); r++) {
                String overlayRegionName;
                String overlayRegionTarget;
                if (overlayRegionPropertyList.item(r).getNodeName().equals("Name")) {
                    setNodeValue(overlayRegionPropertyList.item(r), newName);
                } else if (overlayRegionPropertyList.item(r).getNodeName().equals("TargetImage")) {
                    setNodeValue(overlayRegionPropertyList.item(r), newTargetImage);
                }
            }
        }
        saveOverlayToXMLFile(doc);
        changeOverlay(getNameFromDocument(doc));
        OverlayRegion overlayRegion = (OverlayRegion)getCurrentOverlayRegions().get(oldName);
        overlayRegion.setName(newName);
        overlayRegion.setTarget(newTargetImage);
    }

    public void changeLeftNeighbourOfRegion(Document doc, String overlayRegionName, String leftNeighbour) throws Exception {
        Node nodeToChange = null;
        NodeList overlayRegionNodeList = doc.getElementsByTagName("Name");
        for (int r = 0; r < overlayRegionNodeList.getLength(); r++) {
            if (hasNodeValue(overlayRegionNodeList.item(r))) {
                if (getNodeValue(overlayRegionNodeList.item(r)).equals(overlayRegionName)) {
                    nodeToChange = overlayRegionNodeList.item(r).getParentNode();
                }
            }
        }
        if (nodeToChange != null) {
            NodeList overlayRegionPropertyList = nodeToChange.getChildNodes();
            for (int r = 0; r < overlayRegionPropertyList.getLength(); r++) {
                if (overlayRegionPropertyList.item(r).getNodeName().equals("Neighbours")) {
                    NodeList neighbourNodeList = overlayRegionPropertyList.item(r).getChildNodes();
                    for (int uu = 0; uu < neighbourNodeList.getLength(); uu++) {
                        if (neighbourNodeList.item(uu).getNodeName().equals("Left")) {
                            setNodeValue(neighbourNodeList.item(uu), leftNeighbour);
                        }
                    }
                }
            }
        }
        saveOverlayToXMLFile(doc);
        changeOverlay(getNameFromDocument(doc));
    }

    public void changeRightNeighbourOfRegion(Document doc, String overlayRegionName, String rightNeighbour) throws Exception {
        Node nodeToChange = null;
        NodeList overlayRegionNodeList = doc.getElementsByTagName("Name");
        for (int r = 0; r < overlayRegionNodeList.getLength(); r++) {
            if (hasNodeValue(overlayRegionNodeList.item(r))) {
                if (getNodeValue(overlayRegionNodeList.item(r)).equals(overlayRegionName)) {
                    nodeToChange = overlayRegionNodeList.item(r).getParentNode();
                }
            }
        }
        if (nodeToChange != null) {
            NodeList overlayRegionPropertyList = nodeToChange.getChildNodes();
            for (int r = 0; r < overlayRegionPropertyList.getLength(); r++) {
                if (overlayRegionPropertyList.item(r).getNodeName().equals("Neighbours")) {
                    NodeList neighbourNodeList = overlayRegionPropertyList.item(r).getChildNodes();
                    for (int uu = 0; uu < neighbourNodeList.getLength(); uu++) {
                        if (neighbourNodeList.item(uu).getNodeName().equals("Right")) {
                            setNodeValue(neighbourNodeList.item(uu), rightNeighbour);
                        }
                    }
                }
            }
        }
        saveOverlayToXMLFile(doc);
        changeOverlay(getNameFromDocument(doc));
    }

    public void changeTopNeighbourOfRegion(Document doc, String overlayRegionName, String topNeighbour) throws Exception {
        Node nodeToChange = null;
        NodeList overlayRegionNodeList = doc.getElementsByTagName("Name");
        for (int r = 0; r < overlayRegionNodeList.getLength(); r++) {
            if (hasNodeValue(overlayRegionNodeList.item(r))) {
                if (getNodeValue(overlayRegionNodeList.item(r)).equals(overlayRegionName)) {
                    nodeToChange = overlayRegionNodeList.item(r).getParentNode();
                }
            }
        }
        if (nodeToChange != null) {
            NodeList overlayRegionPropertyList = nodeToChange.getChildNodes();
            for (int r = 0; r < overlayRegionPropertyList.getLength(); r++) {
                if (overlayRegionPropertyList.item(r).getNodeName().equals("Neighbours")) {
                    NodeList neighbourNodeList = overlayRegionPropertyList.item(r).getChildNodes();
                    for (int uu = 0; uu < neighbourNodeList.getLength(); uu++) {
                        if (neighbourNodeList.item(uu).getNodeName().equals("Top")) {
                            setNodeValue(neighbourNodeList.item(uu), topNeighbour);
                        }
                    }
                }
            }
        }
        saveOverlayToXMLFile(doc);
        changeOverlay(getNameFromDocument(doc));
    }

    public void changeBottomNeighbourOfRegion(Document doc, String overlayRegionName,
                                              String bottomNeighbour) throws Exception {
        Node nodeToChange = null;
        NodeList overlayRegionNodeList = doc.getElementsByTagName("Name");
        for (int r = 0; r < overlayRegionNodeList.getLength(); r++) {
            if (hasNodeValue(overlayRegionNodeList.item(r))) {
                if (getNodeValue(overlayRegionNodeList.item(r)).equals(overlayRegionName)) {
                    nodeToChange = overlayRegionNodeList.item(r).getParentNode();
                }
            }
        }
        if (nodeToChange != null) {
            NodeList overlayRegionPropertyList = nodeToChange.getChildNodes();
            for (int r = 0; r < overlayRegionPropertyList.getLength(); r++) {
                if (overlayRegionPropertyList.item(r).getNodeName().equals("Neighbours")) {
                    NodeList neighbourNodeList = overlayRegionPropertyList.item(r).getChildNodes();
                    for (int uu = 0; uu < neighbourNodeList.getLength(); uu++) {
                        if (neighbourNodeList.item(uu).getNodeName().equals("Bottom")) {
                            setNodeValue(neighbourNodeList.item(uu), bottomNeighbour);
                        }
                    }
                }
            }
        }
        saveOverlayToXMLFile(doc);
        changeOverlay(getNameFromDocument(doc));
    }

    /**
     *This method starts removing a region from the current document.
     *@param oldName the name of the region that should be deleted
     */
    public void deleteRegionsFromOverlay(String oldName) throws Exception {
        deleteOverlayRegionInDocument(getCurrentDocument(), oldName);
    }

    /**
     *This method removes a region and makes the changes persistent.
     *@param doc the document ion which the region should be deleted
     * @param oldName the name of the region that should be deleted
     */
    public void deleteOverlayRegionInDocument(Document doc, String oldName) throws Exception {
        Node nodeToDelete = null;
        NodeList overlayRegionNodeList = doc.getElementsByTagName("Name");
        for (int r = 0; r < overlayRegionNodeList.getLength(); r++) {
            if (hasNodeValue(overlayRegionNodeList.item(r))) {
                if (getNodeValue(overlayRegionNodeList.item(r)).equals(oldName)) {
                    nodeToDelete = overlayRegionNodeList.item(r).getParentNode();
                }
            }
        }
        if (nodeToDelete != null) {
            Node parentNode = nodeToDelete.getParentNode();
            parentNode.removeChild(nodeToDelete);
        }
        saveOverlayToXMLFile(doc);
        changeOverlay(getNameFromDocument(doc));
    }

    /**
     *Insert new document in Hashtable
     *@param doc the document whose name should be added to the image hashtable
     */
    public void updateImageHashtable(Document doc) {
        //getting name from document
        String imageName = getNameFromDocument(doc);
        this.imageHashtable.put(imageName, doc);
    }

    /**
     *Updates ComboBoxes when inserting new document. The name of the document will be added to the comboboxes
     *@param doc the document whose name should be added to comboboxes....Deprecated???
     */
    public void updateComboBoxes(Document doc) {
        String imageName = getNameFromDocument(doc);
        //getView().updateComboBoxes(imageName);
    }

    /**
     *Saves a xml file represented by a xml document. The name of the xml file is the name of the document.
     *@param doc the DOM structure to save as an xml file
     */
    public void saveOverlayToXMLFile(Document doc) {
        try {
            String fileName = getNameFromDocument(doc);
            System.out.println("Speicher der Datei " + fileName);
            FileOutputStream out = new FileOutputStream(RecordModel.RelativeXMLImagesDirectoryName + fileName + ".xml");
            XMLSerializer serializer = new XMLSerializer(out, null);
            try {
                serializer.serialize(doc);
                out.flush();
                out.close();
                System.out.println("...erfolgreich.");
            } catch (Exception ee) {
                System.out.println("Document nicht vorhanden:" + ee);
            }
        } catch (IOException ee) {
            System.out.println("Datei kann nicht geschrieben werden:" + ee);
        }
    }

    /** Determines if a change file step is a step back, a history step. return true if step is history step, fasle if not */
    public boolean isHistoryStep() {
        return this.historyStep;
    }

    /** Sets a history step. param historyStep sets whether this step is a history step. History steps are steps back. */
    public void setHistoryStep(boolean historyStep) {
        this.historyStep = historyStep;
    }

    /** One step back. The last element of history vector is taken as and the file is changed to it. */
    public void backToPreviousOverlay() throws Exception {
        if (!getHistoryVector().isEmpty()) {
            int nextToPreviousIndex = getHistoryVector().indexOf(getHistoryVector().lastElement()) - 1;

            /*test whether the last but one element exists, so that the application can change to it*/

            if (nextToPreviousIndex > -1) {
                String previousFileName = (String) getHistoryVector().elementAt(nextToPreviousIndex);
                getHistoryVector().removeElement(getHistoryVector().lastElement());
                setHistoryStep(true);
                changeOverlay(previousFileName);
                setHistoryStep(false);
            }
        }
        if (getHistoryVector().size() < 2) {
            getView().getHistoryBackButton().setEnabled(false);
        }
    }

    /**
     *Saves a region in the particular overlay document and writes back the corresponding xml file.
     *@param name name of the region
     *@param target target of the region, the overlay behind the region
     * @param xCoordinates vector containing the x coordinates of the region
     * @param yCoordinates vector containing the y coordinates of the region
     */
    public void saveOverlayRegion(String name, String target, String number, Vector xCoordinates, Vector yCoordinates,
                                  String topNeighbour, String bottomNeighbour, String leftNeighbour, String rightNeighbour) {
        NodeList overlayRegionsList = getCurrentDocument().getElementsByTagName("Overlay");
        Node overlayRegions = overlayRegionsList.item(0);
        Element newOverlayRegion = getCurrentDocument().createElement("OverlayRegion");
        Element overlayRegionName = getCurrentDocument().createElement("Name");
        overlayRegionName.appendChild(getCurrentDocument().createTextNode(name));
        Element overlayRegionNumber = getCurrentDocument().createElement("Number");
        overlayRegionNumber.appendChild(getCurrentDocument().createTextNode(number));
        Element targetImage = getCurrentDocument().createElement("TargetImage");
        targetImage.appendChild(getCurrentDocument().createTextNode(target));
        Element neighbours = getCurrentDocument().createElement("Neighbours");
        Element neighbourOnTop = getCurrentDocument().createElement("Top");
        neighbourOnTop.appendChild(getCurrentDocument().createTextNode(topNeighbour));
        Element neighbourOnBottom = getCurrentDocument().createElement("Bottom");
        neighbourOnBottom.appendChild(getCurrentDocument().createTextNode(bottomNeighbour));
        Element neighbourOnLeft = getCurrentDocument().createElement("Left");
        neighbourOnLeft.appendChild(getCurrentDocument().createTextNode(leftNeighbour));
        Element neighbourOnRight = getCurrentDocument().createElement("Right");
        neighbourOnRight.appendChild(getCurrentDocument().createTextNode(rightNeighbour));
        neighbours.appendChild(neighbourOnTop);
        neighbours.appendChild(neighbourOnBottom);
        neighbours.appendChild(neighbourOnLeft);
        neighbours.appendChild(neighbourOnRight);
        newOverlayRegion.appendChild(overlayRegionName);
        newOverlayRegion.appendChild(targetImage);
        newOverlayRegion.appendChild(overlayRegionNumber);
        newOverlayRegion.appendChild(neighbours);
        for (int i = 0; i < xCoordinates.size(); i++) {
            Element point = getCurrentDocument().createElement("Point");
            Element xCoord = getCurrentDocument().createElement("X");
            xCoord.appendChild(getCurrentDocument().createTextNode(((Integer) xCoordinates.elementAt(i)).toString()));
            Element yCoord = getCurrentDocument().createElement("Y");
            yCoord.appendChild(getCurrentDocument().createTextNode(((Integer) yCoordinates.elementAt(i)).toString()));
            point.appendChild(xCoord);
            point.appendChild(yCoord);
            newOverlayRegion.appendChild(point);
        }
        overlayRegions.appendChild(newOverlayRegion);
        getCurrentOverlayRegions().clear();
        setCurrentOverlayRegions(getOverlayRegionsFromDocument(getCurrentDocument()));
        //deleteOverlayRegion();
        saveOverlayToXMLFile(getCurrentDocument());
    }

    /**
     *Gets all overlayRegions that are currently in image panel.
     * @return vector containing the current overlayRegions
     */
    public Hashtable getCurrentOverlayRegions() {
        return this.currentOverlayRegions;
    }

    /**
     *Creates a new image document with imageName an pictureName.
     * @param imageName
     * @param pictureName
     * @param pictureDescription
     * @return created document as DOM structure
     */
    public Document createOverlayDocument(String imageName, String pictureName, String pictureDescription) {
        Document doc = new DocumentImpl();
        Element overlay = doc.createElement("Overlay");
        Element name = doc.createElement("Name");
        Element image = doc.createElement("Image");
        Element description = doc.createElement("Description");
        name.appendChild(doc.createTextNode(imageName));
        image.appendChild(doc.createTextNode(pictureName));
        description.appendChild(doc.createTextNode(pictureDescription));
        overlay.appendChild(name);
        overlay.appendChild(image);
        overlay.appendChild(description);
        doc.appendChild(overlay);
        insertNewImageDocument(doc);
        return doc;
    }

    /**
     *Finds matches in patientHashtable.
     * @param stringToMatch the string for which matches should be searched
     * @return vector of matches
     */
    public Vector findMatches(String stringToMatch) {
        Vector returnVector = new Vector();
        Hashtable hashtable = getPatientHashtable();
        for (Enumeration e = hashtable.keys(); e.hasMoreElements();) {
            HealthRecord patient = (HealthRecord) hashtable.get(e.nextElement());
            String patientName = patient.getName().getValue();
            //deleting possible spaces
            patientName = patient.getName().getValue().substring(patientName.lastIndexOf(" ") + 1, patientName.length());
            if (patientName.toLowerCase().startsWith(stringToMatch.toLowerCase())) {
                returnVector.addElement(patient);
            }
        }
        return returnVector;
    }


    /** Adds a patient to hashtable. */
    public void addPatient(HealthRecord patient) {
        this.patientHashtable.put(patient.getIdentifier(), patient);
    }

    /** Gets a random number for a patient id. The randomize algorihm is stoppt when the number is not in patientHashtable. */
    public org.resmedicinae.resmedlib.term.String getRandomPatientId() {
        org.resmedicinae.resmedlib.term.String randomNumberString = new org.resmedicinae.resmedlib.term.String();
        while (true) {
            Random generator = new Random();
            Integer randomNumber = new Integer(generator.nextInt(10000));
            randomNumberString.setValue(randomNumber.toString());
            if (!patientHashtable.containsKey(randomNumberString))
                break;
        }
        return randomNumberString;
    }



    /**
     *Gets the name of a month by his number.
     *@param monthNumber number of month
     * @return String representing the month
     */
    public String getNameOfMonth(int monthNumber) {
        if (monthNumber == 0)
            return "Januar";
        else if (monthNumber == 1)
            return "Februar";
        else if (monthNumber == 2)
            return "Mrz";
        else if (monthNumber == 3)
            return "April";
        else if (monthNumber == 4)
            return "Mai";
        else if (monthNumber == 5)
            return "Juni";
        else if (monthNumber == 6)
            return "Juli";
        else if (monthNumber == 7)
            return "August";
        else if (monthNumber == 8)
            return "September";
        else if (monthNumber == 9)
            return "Oktober";
        else if (monthNumber == 10)
            return "November";
        else if (monthNumber == 11)
            return "Dezember";
        else
            return "NO VALID VALUE";
    }

    /**
     *Adds a marker to the model. This means adding a point of two coordinates to the current partialContact.
     *@param x the x coordinate of the marker
     *@param y the y coordinate of the marker
     */
    public void addMarker(int x, int y) {
        if (getCurrentObject() != null) {
            getCurrentObject().addMarker(x, y);
        } else
            System.out.println("Kein Befund geladen!!!");
    }
    //
    // Special methods.
    //

    /**
     * Loads the health record data from the domain.
     * @param name the patient name
     * @exception NullPointerException if the facades list is null
     */
    public void load(String name) throws NullPointerException {
/*??
       // Ask if wished to save open data.
        //?? ...or just save data by default?
        List l = getFacades();
        if (l != null) {
            Person p = null;
            // Tries to get a person from the first facade in the list.
            // If the returned person is null, the next facade will be used etc.
            // The loop stops as soon as a person is found.
            while (((p == null)) && l.hasNext()) {
                Object o = l.getNext();
                RecordFacade f = null;
                   if (o instanceof RecordLocalFacade) {
                    f = (RecordLocalFacade) l.getNext();
                } else if (o instanceof RecordCorbaFacade) {
                    f = (CorbaFacade) l.getNext();
                } else if (o instanceof RecordRmiFacade) {
                    f = (RmiFacade) l.getNext();
                } else if (o instanceof RecordSoapFacade) {
                    f = (SoapFacade) l.getNext();
                }
                if (f != null) {
                    p = f.getPerson();
                }
            }
            if (p != null) {
                HealthRecord hr = p.getHealthRecord();
                setHealthRecord(hr);
                //Update tree, list and notify the View.
                //?? Code here ...
            } else {
                throw new NullPointerException("Could not load health record. The person is null.");
            }
        } else {
            throw new NullPointerException("Could not load health record. The facades list is null.");
        }
*/
    }

     /** Saves the health record data to the domain. */
    public void save() {
        //?? Write ever single field/data into domain object (HealthRecord).
    }


    /** Increments the overlayRegionIndex */
    public void incrementOverlayRegionIndex() {
        int oldIndex = getOverlayRegionIndex();
        setOverlayRegionIndex(oldIndex + 1);
    }

    /** Decrements the overlayRegionIndex */
    public void decrementOverlayRegionIndex() {
        int oldIndex = getOverlayRegionIndex();
        if (oldIndex > 0) {
            setOverlayRegionIndex(oldIndex - 1);
        }
    }

    /**
     *Test whether overlayRegions can be shown
     * @return true if overlayRegions can be shown, false if not
     */
    public boolean overlayRegionsAvailable() {
        return !getCurrentOverlayRegions().isEmpty();
    }

    public Object[] getCurrentOverlayRegionNames() {
        Vector tempVector = new Vector();
        Enumeration keyEnum = getCurrentOverlayRegions().keys();
        while (keyEnum.hasMoreElements()) {
            tempVector.add(keyEnum.nextElement());
        }
        return tempVector.toArray();
    }

    public void saveAlternativeOverlays(Document doc) {
        Node overlay = doc.getElementsByTagName("Overlay").item(0);
        System.out.println("Overlay: " + overlay);

        /*NodeList alternativesNodeList = doc.getElementsByTagName("Alternatives");
        if(alternativesNodeList.getLength()>0){
            overlay.removeChild(alternativesNodeList.item(0));
            Element alternatives = doc.createElement("Alternatives");
            if(getAlternativeOverlaysForCurrentOverlay()!=null){
                for(int i=0; i<getAlternativeOverlaysForCurrentOverlay().size();i++){
                    String altOverlay = (String)getAlternativeOverlaysForCurrentOverlay().elementAt(i);
                    Element alternative = doc.createElement("Alternative");
                    alternative.appendChild(doc.createTextNode(altOverlay));
                    alternatives.appendChild(alternative);
                }
            }else{
                System.out.println("Could not save alternative overlays. The overlays are null.");
            }
            overlay.appendChild(alternatives);
        }else{
            Element alternatives = doc.createElement("Alternatives");
            if(getAlternativeOverlaysForCurrentOverlay()!=null){
                for(int ii=0; ii<getAlternativeOverlaysForCurrentOverlay().size();ii++){
                    String altOverlay = (String)getAlternativeOverlaysForCurrentOverlay().elementAt(ii);
                    Element alternative = doc.createElement("Alternative");
                    alternative.appendChild(doc.createTextNode(altOverlay));
                    alternatives.appendChild(alternative);
                }
                overlay.appendChild(alternatives);
            }else{
                System.out.println("Could not save alternative overlays. The overlays are null.");
            }
        }*/

        NodeList alternativesNodeList = doc.getElementsByTagName("Alternatives");
        for (int i = 0; i < alternativesNodeList.getLength(); i++) {
            overlay.removeChild(alternativesNodeList.item(i));
        }
        Element alternatives = doc.createElement("Alternatives");
        if (getAlternativeOverlaysForCurrentOverlay() != null) {
            for (int ii = 0; ii < getAlternativeOverlaysForCurrentOverlay().size(); ii++) {
                String altOverlay = (String) getAlternativeOverlaysForCurrentOverlay().elementAt(ii);
                //the overlay itself must not be its own alternative overlay!!! So :
                if(altOverlay!=getNameFromDocument(doc)){
                    System.out.println("Appending alt. overlay: " + altOverlay);
                    Element alternative = doc.createElement("Alternative");
                    alternative.appendChild(doc.createTextNode(altOverlay));
                    alternatives.appendChild(alternative);
                }
            }
            overlay.appendChild(alternatives);
        } else {
            System.out.println("Could not save alternative overlays. There are no alternative overlays.");
        }
        saveOverlayToXMLFile(doc);
    }

    public void removeMarkersFromCurrentObject() {
        if (getCurrentObject() != null) {
            getCurrentObject().getMarkerXCoordinates().clear();
            getCurrentObject().getMarkerYCoordinates().clear();
        }
    }

    public TreeTableRoot getPlansFromCurrentEHR() {
        Object[] problems = getCurrentHealthRecord().getProblems();
        TreeTableRoot root = new TreeTableRoot();
        for (int i = 0; i < problems.length; i++) {
            Problem problem = (Problem) problems[i];
            Object[] episodes = problem.getEpisodes();
            for (int ii = 0; ii < episodes.length; ii++) {
                Episode episode = (Episode) episodes[ii];
                Object[] partialContacts = episode.getPartialContacts();
                for (int iii = 0; iii < partialContacts.length; iii++) {
                    PartialContact partialContact = (PartialContact) partialContacts[iii];
                    Object[] plans = partialContact.getPlans();
                    for (int iiii = 0; iiii < plans.length; iiii++) {
                        root.add(plans[iiii]);
                    }
                }
            }
        }
        return root;
    }

    public TreeTableRoot getObjectivesFromCurrentEHR() {
        Object[] problems = getCurrentHealthRecord().getProblems();
        TreeTableRoot root = new TreeTableRoot();
        for (int i = 0; i < problems.length; i++) {
            Problem problem = (Problem) problems[i];
            Object[] episodes = problem.getEpisodes();
            for (int ii = 0; ii < episodes.length; ii++) {
                Episode episode = (Episode) episodes[ii];
                Object[] partialContacts = episode.getPartialContacts();
                for (int iii = 0; iii < partialContacts.length; iii++) {
                    PartialContact partialContact = (PartialContact) partialContacts[iii];
                    Object[] objectives = partialContact.getObjectives();
                    for (int iiii = 0; iiii < objectives.length; iiii++) {
                        root.add(objectives[iiii]);
                    }
                }
            }
        }
        return root;
    }

    public TreeTableRoot getAssessmentsFromCurrentEHR() {
        Object[] problems = getCurrentHealthRecord().getProblems();
        TreeTableRoot root = new TreeTableRoot();
        for (int i = 0; i < problems.length; i++) {
            Problem problem = (Problem) problems[i];
            Object[] episodes = problem.getEpisodes();
            for (int ii = 0; ii < episodes.length; ii++) {
                Episode episode = (Episode) episodes[ii];
                Object[] partialContacts = episode.getPartialContacts();
                for (int iii = 0; iii < partialContacts.length; iii++) {
                    PartialContact partialContact = (PartialContact) partialContacts[iii];
                    Object[] assessments = partialContact.getAssessments();
                    for (int iiii = 0; iiii < assessments.length; iiii++) {
                        root.add(assessments[iiii]);
                    }
                }
            }
        }
        return root;
    }

    public TreeTableRoot getSubjectivesFromCurrentEHR() {
        Object[] problems = getCurrentHealthRecord().getProblems();
        TreeTableRoot root = new TreeTableRoot();
        for (int i = 0; i < problems.length; i++) {
            Problem problem = (Problem) problems[i];
            Object[] episodes = problem.getEpisodes();
            for (int ii = 0; ii < episodes.length; ii++) {
                Episode episode = (Episode) episodes[ii];
                Object[] partialContacts = episode.getPartialContacts();
                for (int iii = 0; iii < partialContacts.length; iii++) {
                    PartialContact partialContact = (PartialContact) partialContacts[iii];
                    Object[] subjectives = partialContact.getSubjectives();
                    for (int iiii = 0; iiii < subjectives.length; iiii++) {
                        root.add(subjectives[iiii]);
                    }
                }
            }
        }
        return root;
    }

    public void createNewDisease(String diseaseName, String diseaseDescription, Overlay overlay){
        Disease disease = new Disease();
        disease.setDiseaseName(diseaseName);
        disease.setDiseaseDescription(diseaseDescription);
        overlay.addDisease(disease);
        Document doc = getCurrentDocument();
        System.out.println("1");
        if(doc!=null){
            NodeList diseasesList = doc.getElementsByTagName("Diseases");
            NodeList rootList = doc.getElementsByTagName("Overlay");
            if(rootList!=null){
                if(rootList.item(0)!=null){
                    Node root = rootList.item(0);
                    if(diseasesList!=null){
                        Element diseaseElement = doc.createElement("Disease");
                        Element diseaseNameElement = doc.createElement("DiseaseName");
                        diseaseNameElement.appendChild(doc.createTextNode(diseaseName));
                        Element diseaseDescriptionElement  = doc.createElement("DiseaseDescription");
                        diseaseDescriptionElement.appendChild(doc.createTextNode(diseaseDescription));
                        diseaseElement.appendChild(diseaseNameElement);
                        diseaseElement.appendChild(diseaseDescriptionElement);
                        if(diseasesList.item(0)!=null){
                            System.out.println("DISEASE-TYPE: "+diseasesList.item(0).getNodeType());
                            Node diseasesNode = diseasesList.item(0);
                            diseasesNode.appendChild(diseaseElement);
                        }else{
                            //No diseases yet:
                            Element diseasesElement = doc.createElement("Diseases");
                            diseasesElement.appendChild(diseaseElement);
                            root.appendChild(diseasesElement);
                        }
                    }
                    saveOverlayToXMLFile(doc);
                }
            }
        }
    }

    public void removeDiseaseFromOverlay(Disease disease, Overlay overlay){
        String overlayName = overlay.getName();
        Document doc = getDocumentByName(overlayName);
        NodeList diseasesList = doc.getElementsByTagName("Diseases");
        if(diseasesList!=null){
            if(diseasesList.item(0)!=null){
                NodeList diseaseNameList = doc.getElementsByTagName("DiseaseName");
                for(int i=0;i<diseaseNameList.getLength();i++){
                    if(diseaseNameList.item(i)!=null){
                        if(getNodeValue(diseaseNameList.item(i))!=null){
                            if(getNodeValue(diseaseNameList.item(i)).equals(disease.getDiseaseName())){
                                Node parentOfDiseaseName = diseaseNameList.item(i).getParentNode();
                                diseasesList.item(0).removeChild(parentOfDiseaseName);
                            }
                        }
                    }
                }
                saveOverlayToXMLFile(doc);
            }
        }

    }

    public void updateDisease(Disease disease, String newDiseaseName, String newDiseaseDescription){
        Overlay overlay = (Overlay)disease.getParent();
        String overlayName = overlay.getName();
        Document doc = getDocumentByName(overlayName);
        NodeList diseasesList = doc.getElementsByTagName("Diseases");
        if(diseasesList!=null){
            if(diseasesList.item(0)!=null){
                System.out.println("1");
                NodeList diseaseNameList = doc.getElementsByTagName("DiseaseName");
                for(int i=0;i<diseaseNameList.getLength();i++){
                    if(diseaseNameList.item(i)!=null){
                        System.out.println("2");
                        if(getNodeValue(diseaseNameList.item(i))!=null){
                            System.out.println("3");
                            if(getNodeValue(diseaseNameList.item(i)).equals(disease.getDiseaseName())){
                                setNodeValue(diseaseNameList.item(i), newDiseaseName);
                                Node parentNode = diseaseNameList.item(i).getParentNode();
                                NodeList children = parentNode.getChildNodes();
                                System.out.println("4");
                                for(int ii=0;ii<children.getLength();ii++){
                                    if(children.item(ii)!=null){
                                        if(children.item(ii).getNodeName().equals("DiseaseDescription")){
                                            setNodeValue(children.item(ii),newDiseaseDescription);
                                        }
                                    }
                                }
                            }
                        }
                    }
                }
                saveOverlayToXMLFile(doc);
            }
        }

    }

        /**
     *Gets the alternative overlays from a document
     *@param doc the document from which the alternative overlays should be taken
     *@return vector containing alternative overlays
     */
    public Vector getAlternativeOverlaysFromDocument(Document doc) {
        Vector alternativeOverlaysVector = new Vector();
        Vector returnVector = new Vector();
        if (doc != null) {
            //get root element named <Alternatives>
            NodeList alternativesList = doc.getElementsByTagName("Alternatives");
            Node alternativeNode = alternativesList.item(0);
            if (alternativeNode != null) {
                NodeList alternativeList = alternativeNode.getChildNodes();
                for (int i = 0; i < alternativeList.getLength(); i++) {
                    System.out.println("Node Name: " + alternativeList.item(i).getNodeName());
                    if (alternativeList.item(i).getNodeName().equals("Alternative")) {
                        alternativeOverlaysVector.add(alternativeList.item(i));
                    }
                }
                System.out.println("Es existieren " + alternativeOverlaysVector.size() + " alternative Regionen.");
                for (int e = 0; e < alternativeOverlaysVector.size(); e++) {
                    Node alternativeOverlay = (Node) alternativeOverlaysVector.elementAt(e);
                    String alternativeOverlayName = getNodeValue(alternativeOverlay);
                    System.out.println("Adding " + alternativeOverlayName + " to List.");
                    returnVector.add(alternativeOverlayName);
                }
                return returnVector;
            } else {
                System.out.println("Could not get <Alternatives> Node. The node does not exist.");
                return null;
            }
        } else {
            throw new NullPointerException("Could not get alternative overlays from document. The document is null.");
        }
    }
}
