/*
 * $RCSfile: RestoreModel.java,v $
 *
 * Copyright (c) 1999-2002. Christian Heller. All rights reserved.
 *
 * This software is published under the GPL GNU General Public License.
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * http://www.resmedicinae.org
 * - Information in Medicine -
 */

package org.resmedicinae.application.healthcare.restore;

import org.resmedicinae.resmedlib.component.model.*;
import org.resmedicinae.resmedlib.component.view.swing.*;

import javax.swing.*;
import java.io.*;
import java.math.*;
import java.util.*;

/**
 * This class represents the RestoreModel. The model contains all data.
 *
 * @version $Revision: 1.8 $ $Date: 2003/01/07 16:20:46 $ $Author: behr78 $
 * @author Dirk Behrendt <schnoerk@web.de>
 */
public class RestoreModel extends Model {

    private Vector repositoryVector;        // for the view
    private Vector choiceVector;            // for the view
    private Vector dataChoiceVector;
    private Vector restoreChoiceVector;     // for the view
    private Vector dataRestoreChoiceVector;
    private Vector placeInDataChoiceVector;

    private Vector folderVector;

    private Hashtable table;

    private Vector recursiveChoiceBooleanVector;    // elements can be "file","folder","recFolder"
    private Vector recursiveRestoreChoiceBooleanVector; // elements can be "file","folder","recFolder"

    private String backupPathString = "";
    private String restorePathString = "";
    private String storingPathString = "";
    private String informationFileString = "";
    private String reportAreaString;

    private File backupPathFile;
    private File restorePathFile;

    private SwingView view;

    private RandomAccessFile file,file2 = null;

    private boolean repositoryVectorLoadedFromFile = false;
    private boolean repositoryVectorLoadedFromFileIsEmpty = false;
    private boolean copyAll = false;
    private boolean cancelCopy = false;
    boolean bool = false;

    private int countFile = 0;
    private double countBytes = 0;

    public void initialize() {

        setRepositoryVector(createRepositoryVector());
        setChoiceVector(createChoiceVector());
        setRestoreChoiceVector(createRestoreChoiceVector());
        setFolderVector(createFolderVector());
        setRecursiveChoiceBooleanVector(createRecursiveChoiceBooleanVector());
        setRecursiveRestoreChoiceBooleanVector(createRecursiveRestoreChoiceBooleanVector());
        setDataChoiceVector(createDataChoiceVector());
        setDataRestoreChoiceVector(createDataRestoreChoiceVector());
        setPlaceInDataChoiceVector(createPlaceInDataChoiceVector());
        setHashTable(createHashTable());
    }


    /**
     * In this method all information about a selected repository file will be stored in a special text-file. The name of the text-file correspond to the name
     * of the seleted repository file. This special text-file will be stored at a special place (storing path), which the user has to chose.
     * The following information will be stored: <br>
     * - Backup Path  <br>
     * - Restore Path <br>
     * - The Choice Vector (files/directories for backup) <br>
     * - The Recursive Choice Boolean Vector (thre are three possibilities: file, folder or recursive folder)<br>
     * - The Restore Choice Vector (files/directories for restore) <br>
     * - The Recursive Restore Choice Boolean Vector (thre are three possibilities: file, folder or recursive folder) <br>
     * @param str the name of the file (=name of selected repository file), which contains the information of a repository file.
     */
    public void writeToRepositoryFile(String str) {


        try {

            File f = new File(this.getStoringPathString() + File.separator + str + ".res");

            if (f.exists() == true) {

                f.delete();

            } else {

                f.createNewFile();

            }

            file = new RandomAccessFile(this.getStoringPathString() + File.separator + str + ".res", "rw");

            file.writeBytes("BEGIN_OF_FILE\n");
            file.writeBytes("BEGIN_OF_BACKUP_PATH\n");
            this.writeStringsToFile(this.getBackupPathString());
            file.writeBytes("BEGIN_OF_RESTORE_PATH\n");
            this.writeStringsToFile(this.getRestorePathString());
            file.writeBytes("BEGIN_OF_DATA_CHOICE_VECTOR\n");
            this.writeFromVectorToFile(this.dataChoiceVector);
            file.writeBytes("END_OF_DATA_CHOICE_VECTOR\n");
            file.writeBytes("BEGIN_OF_RECURSIVE_CHOICE_BOOLEAN_VECTOR\n");
            this.writeFromVectorToFile(this.recursiveChoiceBooleanVector);
            file.writeBytes("END_OF_RECURSIVE_CHOICE_BOOLEAN_VECTOR\n");
            file.writeBytes("BEGIN_OF_DATA_RESTORE_CHOICE_VECTOR\n");
            this.writeFromVectorToFile(this.dataRestoreChoiceVector);
            file.writeBytes("END_OF_DATA_RESTORE_CHOICE_VECTOR\n");
            file.writeBytes("BEGIN_OF_RECURSIVE_RESTORE_CHOICE_BOOLEAN_VECTOR\n");
            this.writeFromVectorToFile(this.recursiveRestoreChoiceBooleanVector);
            file.writeBytes("END_OF_RECURSIVE_RESTORE_CHOICE_BOOLEAN_VECTOR\n");
            file.writeBytes("END_OF_FILE\n");

            System.out.println("Length of file: " + file.length());
            file.close();

        } catch (Exception e) {
            System.out.println("Fehler: " + e);
        }
        ;

    }


    /**
     * In this method the Information File will be stored at a special place the user has to chose (storing path). The name of this file is "RestoreInfo.res".
     * The file contains the following information: <br>
     * - the names of all created repository files <br>
     * - the storing path <br> (place, where the Information File and all Repository Files will be stored.)
     */
    public void writeRepositoryVectorToFile() {

        try {

            File f = new File(this.getStoringPathString() + File.separator + "RestoreInfo.res");
            System.out.println("The Information file " + this.getStoringPathString() + File.separator + "RestoreInfo.res was stored.");

            if (f.exists() == true) {

                f.delete();

            } else {

                f.createNewFile();

            }

            file = new RandomAccessFile(this.getStoringPathString() + File.separator + "RestoreInfo.res", "rw");
            file.writeBytes("BEGIN_OF_INFORMATION_FILE\n");
            file.writeBytes("BEGIN_OF_REPOSITORY_FILE_INFORMATION\n");
            this.writeFromVectorToFile(this.getRepositoryVector());
            file.writeBytes("END_OF_REPOSITORY_FILE_INFORMATION\n");
            file.writeBytes("BEGIN_OF_STORING_PATH\n");
            file.writeBytes(this.getStoringPathString() + "\n");
            file.writeBytes("END_OF_STORING_PATH\n");
            file.writeBytes("END_OF_INFORMATION_FILE\n");
            file.close();
        } catch (Exception e) {
            System.out.println("Fehler: " + e);
        }
        ;
    }


    /**
     * In this method the Information file RestoreInfo.res will be loaded. <br>
     * (contains all Repository Files and the Storing Path)
     */
    public void readRepositoryVectorFromFile() throws IOException {

        File f = new File(this.getInformationFileString());

        if (!f.exists()) {

            f.createNewFile();

            file = new RandomAccessFile(this.getInformationFileString(), "rw");

            file.writeBytes("BEGIN_OF_INFORMATION_FILE\n");
            file.writeBytes("BEGIN_OF_REPOSITORY_FILE_INFORMATION\n");
            file.writeBytes("END_OF_REPOSITORY_FILE_INFORMATION\n");
            file.writeBytes("BEGIN_OF_STORING_PATH\n");
            file.writeBytes("END_OF_STORING_PATH\n");
            file.writeBytes("END_OF_INFORMATION_FILE\n");
            file.close();
            System.out.println("The file RestoreInfo.res was created.");
            this.setRepositoryVectorLoadedFromFile(true);

            if (this.getRepositoryVector().size() == 0) {

                this.setRepositoryVectorLoadedFromFileIsEmpty(true);

            } else
                this.setRepositoryVectorLoadedFromFileIsEmpty(false);

        } else {

            this.setRepositoryVectorLoadedFromFile(true);
            file2 = new RandomAccessFile(this.getInformationFileString(), "r");

            this.getRepositoryVector().removeAllElements();

            String str = file2.readLine();

            if (str.equals("BEGIN_OF_INFORMATION_FILE")) {

                str = file2.readLine();

                while (!str.equals("END_OF_INFORMATION_FILE")) {

                    if (str.equals("BEGIN_OF_REPOSITORY_FILE_INFORMATION")) {

                        str = file2.readLine();

                        while (!str.equals("END_OF_REPOSITORY_FILE_INFORMATION")) {

                            this.getRepositoryVector().add(str);
                            str = file2.readLine();

                        }

                    }

                    if (str.equals("BEGIN_OF_STORING_PATH")) {

                        str = file2.readLine();

                        while (!str.equals("END_OF_STORING_PATH")) {

                            this.setStoringPathString(str);
                            str = file2.readLine();

                        }

                    }

                    str = file2.readLine();
                }

                file2.close();

            } else
                System.out.println("The stored RestoreInfo file is damaged!");

            try {

                this.updateRepositoryList();
                this.updateBackupOptionRepositoryList();
                this.updateRestoreOptionRepositoryList();


            } catch (Exception e) {
                System.out.println("Fehler: " + e);
            }
            ;

        }

    }


    /**
     * In this method the contents of a vector will be written in a file.
     * @param  vector a vector
     */
    public void writeFromVectorToFile(Vector vector) throws IOException {

        if (vector.isEmpty() == false) {

            int vectorSize = vector.size();

            for (int a = 0; a < vectorSize; a++) {

                file.writeBytes(vector.elementAt(a).toString() + "\n");

            }

        }

    }

    /**
     * In this method a string will be written to a file
     * @param  string - a string
     */
    public void writeStringsToFile(String string) throws IOException {

        if (string != null) {

            file.writeBytes(string + "\n");

        }

    }


    /**
     * In this method all information of a selected repoitory file will be loaded.<br>
     * - Backup Path <br>
     * - Restore Path <br>
     * - Choice <br>
     * - RestoreChoice <br>
     * - Recursive Choice Boolean Vector <br>
     * - Recursive RestoreChoice Boolean Vector <br>
     * @param  repositoryFile the selected repository file
     *
     */
    public void readFromRepositoryFile(String repositoryFile) throws IOException {

        file2 = new RandomAccessFile(this.getStoringPathString() + File.separator + repositoryFile + ".res", "r");
        String str = file2.readLine();

        if (str.equals("BEGIN_OF_FILE")) {

            str = file2.readLine();

            while (!str.equals("END_OF_FILE")) {

                if (str.equals("BEGIN_OF_BACKUP_PATH")) {

                    str = file2.readLine();

                    if (!str.equals("BEGIN_OF_RESTORE_PATH")) {

                        this.setBackupPathString(str);

                    } else
                        this.setBackupPathString("");

                }

                if (str.equals("BEGIN_OF_RESTORE_PATH")) {

                    str = file2.readLine();

                    if (!str.equals("BEGIN_OF_DATA_CHOICE_VECTOR")) {

                        this.setRestorePathString(str);
                        str = file2.readLine();

                    } else
                        this.setRestorePathString("");

                }

                if (str.equals("BEGIN_OF_DATA_CHOICE_VECTOR")) {

                    str = file2.readLine();
                    this.getDataChoiceVector().removeAllElements();

                    while (!str.equals("END_OF_DATA_CHOICE_VECTOR")) {

                        this.getDataChoiceVector().add(str);
                        str = file2.readLine();

                    }

                }

                if (str.equals("BEGIN_OF_RECURSIVE_CHOICE_BOOLEAN_VECTOR")) {

                    str = file2.readLine();
                    this.getRecursiveChoiceBooleanVector().removeAllElements();

                    while (!str.equals("END_OF_RECURSIVE_CHOICE_BOOLEAN_VECTOR")) {

                        this.getRecursiveChoiceBooleanVector().add(str);
                        str = file2.readLine();
                    }

                }


                if (str.equals("BEGIN_OF_DATA_RESTORE_CHOICE_VECTOR")) {

                    str = file2.readLine();
                    this.getDataRestoreChoiceVector().removeAllElements();

                    while (!str.equals("END_OF_DATA_RESTORE_CHOICE_VECTOR")) {

                        this.getDataRestoreChoiceVector().add(str);
                        str = file2.readLine();
                    }

                }


                if (str.equals("BEGIN_OF_RECURSIVE_RESTORE_CHOICE_BOOLEAN_VECTOR")) {

                    str = file2.readLine();
                    this.getRecursiveRestoreChoiceBooleanVector().removeAllElements();

                    while (!str.equals("END_OF_RECURSIVE_RESTORE_CHOICE_BOOLEAN_VECTOR")) {

                        this.getRecursiveRestoreChoiceBooleanVector().add(str);
                        str = file2.readLine();

                    }

                }

                str = file2.readLine();

            }


        } else
            System.out.println("The stored repository file is damaged!");

        file2.close();

        this.updateVectorsForViews();

    }


    /**
     * An empty Repository File will be written. This method will be called, if the user creates a new Repository File.
     * @param  string the created repository file
     *
     */
    public void writeEmptyRepositoryFile(String str) throws IOException {

        File f = new File(this.getStoringPathString() + File.separator + str + ".res");

        if (f.exists() == true) {

            f.delete();

        } else {

            f.createNewFile();

        }

        file = new RandomAccessFile(this.getStoringPathString() + File.separator + str + ".res", "rw");

        file.writeBytes("BEGIN_OF_FILE\n");
        file.writeBytes("BEGIN_OF_BACKUP_PATH\n");
        file.writeBytes("BEGIN_OF_RESTORE_PATH\n");
        file.writeBytes("BEGIN_OF_DATA_CHOICE_VECTOR\n");
        file.writeBytes("END_OF_DATA_CHOICE_VECTOR\n");
        file.writeBytes("BEGIN_OF_RECURSIVE_CHOICE_BOOLEAN_VECTOR\n");
        file.writeBytes("END_OF_RECURSIVE_CHOICE_BOOLEAN_VECTOR\n");
        file.writeBytes("BEGIN_OF_DATA_RESTORE_CHOICE_VECTOR\n");
        file.writeBytes("END_OF_DATA_RESTORE_CHOICE_VECTOR\n");
        file.writeBytes("BEGIN_OF_RECURSIVE_RESTORE_CHOICE_BOOLEAN_VECTOR\n");
        file.writeBytes("END_OF_RECURSIVE_RESTORE_CHOICE_BOOLEAN_VECTOR\n");
        file.writeBytes("END_OF_FILE\n");

        file.close();
    }


    /**
     * In this method data will be prepared for the views. The choice and the RestoreChoice will be manipulated by adding "Recursive",
     * if a folder is recursive. Otherwise it is a normal folder.
     *
     */
    public void updateVectorsForViews() {

        this.getChoiceVector().removeAllElements();
        this.getRestoreChoiceVector().removeAllElements();

        int DataChoiceVectoSize = this.getDataChoiceVector().size();
        int DataRestoreChoiceSize = this.getDataRestoreChoiceVector().size();

        if (this.getDataChoiceVector().isEmpty() == false) {

            for (int a = 0; a < DataChoiceVectoSize; a++) {

                if (this.getRecursiveChoiceBooleanVector().elementAt(a).equals("recFolder")) {

                    this.getChoiceVector().add(this.getDataChoiceVector().elementAt(a) + " (recursive)");

                } else {

                    this.getChoiceVector().add(this.getDataChoiceVector().elementAt(a));

                }

            }

        }


        if (!this.getDataRestoreChoiceVector().isEmpty()) {

            for (int a = 0; a < DataRestoreChoiceSize; a++) {

                if (this.getRecursiveRestoreChoiceBooleanVector().elementAt(a).equals("file")) {

                    String nameOfFile = this.getNameOfFile(this.getDataRestoreChoiceVector().elementAt(a).toString());

                    if (this.getRestorePathString().equals("<default>")) {

                        this.getRestoreChoiceVector().add(this.getDataRestoreChoiceVector().elementAt(a) + nameOfFile);

                    } else
                        this.getRestoreChoiceVector().add(this.getRestorePathString() + File.separator + this.createPathInCorrectOrder(this.getPath(this.getDataRestoreChoiceVector().elementAt(a).toString())) + File.separator + nameOfFile);

                }

                if (this.getRecursiveRestoreChoiceBooleanVector().elementAt(a).equals("folder")) {

                    if (this.getRestorePathString().equals("<default>")) {

                        this.getRestoreChoiceVector().add(this.getDataRestoreChoiceVector().elementAt(a));

                    } else
                        this.getRestoreChoiceVector().add(this.getRestorePathString() + File.separator + this.createPathInCorrectOrder(this.getPath(this.getDataRestoreChoiceVector().elementAt(a).toString())));

                }

                if (this.getRecursiveRestoreChoiceBooleanVector().elementAt(a).equals("recFolder")) {

                    if (this.getRestorePathString().equals("<default>")) {

                        this.getRestoreChoiceVector().add(this.getDataRestoreChoiceVector().elementAt(a) + " (recursive)");

                    } else
                        this.getRestoreChoiceVector().add(this.getRestorePathString() + File.separator + this.createPathInCorrectOrder(this.getPath(this.getDataRestoreChoiceVector().elementAt(a).toString())) + " (recursive)");

                }

            }

        }

        try {

            this.updateChoiceList();
            this.updateBackupOptionBackupPathTextField();
            this.updateBackupOptionChoiceList();
            this.updateRestoreOptionChoiceList();
            this.updateRestoreOptionRestoreChoiceList();
            this.updateRestoreOptionRestorePathTextField();
            this.updateRestoreChoiceList();

        } catch (Exception e) {
            System.out.println("Fehler: " + e);
        }
        ;

    }


    /**
     * The stored Repository File will be deleted.
     * @param str the Repository File
     *
     */
    public void deleteStoredRepositoryFile(String str) {

        File file = new File("c:\\" + str + ".res");

        if (file.exists() == true) {

            if (file.delete() == true) {

                System.out.println("File " + str + ".res successfully deleted.");

            } else
                System.out.println("Could not delete file " + str + ".res !");

        } else
            System.out.println("The file " + str + ".res does not exists");
    }

    /**
     * The user adds a new Repository File. The new file will be added to the Repository vector. The Views will be updated.
     * @param newItem the new Repository File
     *
     */
    public void addToRepositoryVector(String newItem) throws Exception {

        if (getRepositoryVector().contains(newItem)) {

            JFrame frame = new JFrame();

            if (frame != null) {

                JOptionPane.showMessageDialog(frame, "This entry allready exists in the repository!", "Message", JOptionPane.ERROR_MESSAGE);
                System.out.println("This entry allready exists in this vector!");

            } else
                throw new NullPointerException("Could not create frame. The frame is null.");

        } else {

            this.updateBackupOptionRepositoryList();

            getRepositoryVector().add(newItem);

            this.writeRepositoryVectorToFile();
            this.writeEmptyRepositoryFile(newItem);

            System.out.println("Entry '" + newItem + "' was added to the repository.");
            this.updateBackupOptionRepositoryList();
            this.updateRepositoryList();
            this.updateRestoreOptionRepositoryList();

        }

    }


    /**
     * The user deletes a selected Repository File. The file will be removed from the Repository vector. The Views will be updated.
     *
     *
     */
    public void removeFromRepositoryVector() throws Exception {

        RestoreView view = (RestoreView) getView();

        if (this.getRepositoryVector().isEmpty() == false) {

            if (view != null) {

                if (view.getBackupOption().getSelectedRepositoryIndex() != -1) {

                    this.deleteStoredRepositoryFile(getRepositoryVector().elementAt(view.getBackupOption().getSelectedRepositoryIndex()).toString());
                    getRepositoryVector().removeElementAt(view.getBackupOption().getSelectedRepositoryIndex());
                    this.writeRepositoryVectorToFile();
                    view.getBackupOption().updateRepositoryVectorSize();

                    view.getBackupOption().updateRepositoryList();
                    view.updateRepositoryList();
                    view.updateRepositoryList();

                } else {

                    System.out.println("No entry in the repository was selected!");
                    JFrame frame = new JFrame();

                    if (frame != null) JOptionPane.showMessageDialog(frame, "No entry in the repository was selected!", "Message", JOptionPane.ERROR_MESSAGE);

                }


            } else
                throw new Exception("No view available!");


        } else {

            System.out.println("There are no entries in the vector!");
            JFrame frame = new JFrame();

            if (frame != null)
                JOptionPane.showMessageDialog(frame, "There are no entries in the repository!", "Message", JOptionPane.ERROR_MESSAGE);
            else
                throw new NullPointerException("Could not create frame. The frame is null.");
        }
    }


    /**
     * The user adds a new item to the Choice Vector. The item can be: <br>
     * - a file with its path <br>
     * - a directory with its path <br>
     * - a recursive directory with its path <br>
     * The information whether a item is a file/folder/recursive folder is stored separately in the Recursive Choice Boolean Vector. <br>
     * The Choice contains files and/or directories for backup.
     * @param newItem file or directory
     * @param kindOfItem file or folder or recursive folder
     */
    public void addToChoiceVector(File newItem, String kindOfItem) throws Exception {


        if (getChoiceVector().contains(newItem)) {

            JFrame frame = new JFrame();

            if (frame != null) {

                JOptionPane.showMessageDialog(frame, "This entry allready exists in the choice!", "Message", JOptionPane.ERROR_MESSAGE);
                System.out.println("This entry allready exists in this vector!");

            } else
                throw new NullPointerException("Could not create frame. The frame is null.");

        } else {


            if (kindOfItem == "file") {

                getChoiceVector().add(newItem);
                getDataChoiceVector().add(newItem);
                getRecursiveChoiceBooleanVector().add("file");

            }

            if (kindOfItem == "folder") {

                getChoiceVector().add(newItem);
                getDataChoiceVector().add(newItem);
                getRecursiveChoiceBooleanVector().add("folder");

            }

            if (kindOfItem == "recFolder") {

                getChoiceVector().add(newItem + " (recursive)");
                getDataChoiceVector().add(newItem);
                getRecursiveChoiceBooleanVector().add("recFolder");

            }

            System.out.println("Entry '" + newItem + "' was added to the choice.");
            this.updateBackupOptionChoiceList();
            this.updateRestoreOptionChoiceList();
            this.updateChoiceList();

        }

    }


    /**
     * The user adds a new item to the RestoreChoice Vector. The item can be: <br>
     * - a file with its path <br>
     * - a directory with its path <br>
     * - a recursive directory with its path <br>
     * The information whether a item is a file/folder/recursive folder is stored separately in the Recursive Restore Choice Boolean Vector. <br>
     * The RestoreChoice contains files and/or directories for restore. The user can add selected items from the Choice to the RestoreChoice.
     * Normally the Choice will be restored to the place, from where we backuped the data. (The Restore Path is set to <default>). The user can restore the data
     * to another place of his storage device.
     *
     */
    public void addToRestoreChoiceVector() throws Exception {

        RestoreView view = (RestoreView) getView();

        if ((view.getRestoreOption().getSelectedChoiceIndex() == -1) && (view.getRestoreOption().getRestoreOptionTabbedPane().getSelectedIndex() == 0)) {

            this.notPossibleRestoreOptionChoiceTabbedPane();

        } else {

            File file = new File(this.getDataChoiceVector().elementAt(view.getRestoreOption().getSelectedChoiceIndex()).toString());
            Vector vector = this.getPath(this.getDataChoiceVector().elementAt(view.getRestoreOption().getSelectedChoiceIndex()).toString());
            String restoreChoicePath = this.createPathInCorrectOrder(vector);

            // if a file is stored for example at c:\ of your storage device
            String path;
            if (restoreChoicePath.length() == 0) {

                path = "";

            } else
                path = restoreChoicePath + File.separator;

            String nameOfFile = this.getNameOfFile(this.getDataChoiceVector().elementAt(view.getRestoreOption().getSelectedChoiceIndex()).toString());
            String placeOfBackup, string, restorePath;

            if (this.createPathInCorrectOrder(this.getPath(this.getDataChoiceVector().elementAt(view.getRestoreOption().getSelectedChoiceIndex()).toString())).length() == 0) {

                File file1 = new File(this.getDataChoiceVector().elementAt(view.getRestoreOption().getSelectedChoiceIndex()).toString());
                string = file1.getName();

            } else {

                if (this.getRecursiveChoiceBooleanVector().elementAt(view.getRestoreOption().getSelectedChoiceIndex()).equals("file")) {

                    File file1 = new File(this.getDataChoiceVector().elementAt(view.getRestoreOption().getSelectedChoiceIndex()).toString());
                    string = this.createPathInCorrectOrder(this.getPath(this.getDataChoiceVector().elementAt(view.getRestoreOption().getSelectedChoiceIndex()).toString())) + File.separator + file1.getName();

                } else {

                    string = this.createPathInCorrectOrder(this.getPath(this.getDataChoiceVector().elementAt(view.getRestoreOption().getSelectedChoiceIndex()).toString()));

                }
            }


            placeOfBackup = this.getBackupPathString() + File.separator + string;

            if (!this.getRestorePathString().equals("<default>")) {

                if (this.getRecursiveChoiceBooleanVector().elementAt(view.getRestoreOption().getSelectedChoiceIndex()).toString().equals("file")) {

                    restorePath = this.getRestorePathString() + File.separator + path + nameOfFile;

                }
                restorePath = this.getRestorePathString() + File.separator + path;


            } else {

                restorePath = this.getDataChoiceVector().elementAt(view.getRestoreOption().getSelectedChoiceIndex()).toString().toString();
            }

            if (this.getRecursiveChoiceBooleanVector().elementAt(view.getRestoreOption().getSelectedChoiceIndex()).equals("file")) {

                if (this.getRestorePathString().equals("<default>")) {

                    if (this.checkRestoreChoiceContents(this.getDataChoiceVector().elementAt(view.getRestoreOption().getSelectedChoiceIndex()).toString())) {

                        this.getRestoreChoiceVector().add(this.getDataChoiceVector().elementAt(view.getRestoreOption().getSelectedChoiceIndex()).toString());
                        this.getDataRestoreChoiceVector().add(this.getDataChoiceVector().elementAt(view.getRestoreOption().getSelectedChoiceIndex()).toString());

                    }

                } else {

                    if (this.checkRestoreChoiceContents(this.getDataChoiceVector().elementAt(view.getRestoreOption().getSelectedChoiceIndex()).toString())) {

                        this.getRestoreChoiceVector().add(this.getRestorePathString() + File.separator + this.createPathInCorrectOrder(this.getPath(this.getDataChoiceVector().elementAt(view.getRestoreOption().getSelectedChoiceIndex()).toString())) + nameOfFile);
                        this.getDataRestoreChoiceVector().add(this.getDataChoiceVector().elementAt(view.getRestoreOption().getSelectedChoiceIndex()).toString() + File.separator + nameOfFile);

                    }
                }

                this.getRecursiveRestoreChoiceBooleanVector().add("file");

            }

            if (this.getRecursiveChoiceBooleanVector().elementAt(view.getRestoreOption().getSelectedChoiceIndex()).equals("folder")) {

                if (this.getRestorePathString().equals("<default>")) {

                    if (this.checkRestoreChoiceContents(this.getDataChoiceVector().elementAt(view.getRestoreOption().getSelectedChoiceIndex()).toString())) {

                        this.getRestoreChoiceVector().add(this.getDataChoiceVector().elementAt(view.getRestoreOption().getSelectedChoiceIndex()).toString());
                        this.getDataRestoreChoiceVector().add(this.getDataChoiceVector().elementAt(view.getRestoreOption().getSelectedChoiceIndex()).toString());

                    }

                } else {

                    if (this.checkRestoreChoiceContents(this.getDataChoiceVector().elementAt(view.getRestoreOption().getSelectedChoiceIndex()).toString())) {

                        this.getRestoreChoiceVector().add(this.getRestorePathString() + File.separator + this.createPathInCorrectOrder(this.getPath(this.getDataChoiceVector().elementAt(view.getRestoreOption().getSelectedChoiceIndex()).toString())));
                        this.getDataRestoreChoiceVector().add(this.getDataChoiceVector().elementAt(view.getRestoreOption().getSelectedChoiceIndex()).toString());
                    }
                }

                this.getRecursiveRestoreChoiceBooleanVector().add("folder");

            }

            if (this.getRecursiveChoiceBooleanVector().elementAt(view.getRestoreOption().getSelectedChoiceIndex()).equals("recFolder")) {

                if (this.getRestorePathString().equals("<default>")) {

                    if (this.checkRestoreChoiceContents(this.getDataChoiceVector().elementAt(view.getRestoreOption().getSelectedChoiceIndex()).toString())) {

                        this.getRestoreChoiceVector().add(this.getDataChoiceVector().elementAt(view.getRestoreOption().getSelectedChoiceIndex()).toString() + " (recursive)");
                        this.getDataRestoreChoiceVector().add(this.getDataChoiceVector().elementAt(view.getRestoreOption().getSelectedChoiceIndex()).toString());

                    }

                } else {

                    if (this.checkRestoreChoiceContents(this.getDataChoiceVector().elementAt(view.getRestoreOption().getSelectedChoiceIndex()).toString())) {

                        this.getRestoreChoiceVector().add(this.getRestorePathString() + File.separator + this.createPathInCorrectOrder(this.getPath(this.getDataChoiceVector().elementAt(view.getRestoreOption().getSelectedChoiceIndex()).toString())) + " (recursive)");
                        this.getDataRestoreChoiceVector().add(this.getDataChoiceVector().elementAt(view.getRestoreOption().getSelectedChoiceIndex()).toString());
                    }
                }

                this.getRecursiveRestoreChoiceBooleanVector().add("recFolder");

            }

            this.updateRestoreOptionRestoreChoiceList();
            this.updateRestoreChoiceList();
        }

    }


    /**
     * The user takes all items from the Choice to the RestoreChoice. The items can be: <br>
     * - a file with its path <br>
     * - a directory with its path <br>
     * - a recursive directory with its path <br>
     * The information whether a item is a file/folder/recursive folder is stored separately in the Recursive Restore Choice Boolean Vector. <br>
     * The RestoreChoice contains files and/or directories for restore. The user can add selected items from the Choice to the RestoreChoice.
     * Normally the Choice will be restored to the place, from where we backuped the data. (The Restore Path is set to <default>). The user can restore the data
     * to another place of his storage device.
     *
     */
    public void takeAllToRestoreChoiceVector() throws Exception {

        RestoreView view = (RestoreView) getView();

        this.getRestoreChoiceVector().removeAllElements();
        this.getDataRestoreChoiceVector().removeAllElements();
        this.getRecursiveRestoreChoiceBooleanVector().removeAllElements();

        if (this.getChoiceVector().isEmpty() == true) {

            System.out.println("The choice is empty!");

            JFrame frame = new JFrame();

            if (frame != null) {

                JOptionPane.showMessageDialog(frame, "The Choice is empty!", "Message", JOptionPane.ERROR_MESSAGE);

            } else
                throw new NullPointerException("Could not create frame. The frame is null.");

        } else {

            int choiceVectorSize = this.getDataChoiceVector().size();

            for (int a = 0; a < choiceVectorSize; a++) {

                Vector vector = this.getPath(this.getDataChoiceVector().elementAt(a).toString());
                String restoreChoicePath = this.createPathInCorrectOrder(vector);

                // if a file is stored for example at c:\ of your storage device
                String path;
                if (restoreChoicePath.length() == 0) {

                    path = "";

                } else
                    path = restoreChoicePath + File.separator;
                System.out.println(path);


                String nameOfFile = this.getNameOfFile(this.getDataChoiceVector().elementAt(a).toString());

                String placeOfBackup, string, restorePath;

                if (this.createPathInCorrectOrder(this.getPath(this.getDataChoiceVector().elementAt(a).toString())).length() == 0) {

                    File file1 = new File(this.getDataChoiceVector().elementAt(a).toString());
                    string = file1.getName();

                } else {

                    if (this.getRecursiveChoiceBooleanVector().elementAt(a).equals("file")) {

                        File file1 = new File(this.getDataChoiceVector().elementAt(a).toString());
                        string = this.createPathInCorrectOrder(this.getPath(this.getDataChoiceVector().elementAt(a).toString())) + File.separator + file1.getName();

                    } else {

                        string = this.createPathInCorrectOrder(this.getPath(this.getDataChoiceVector().elementAt(a).toString()));

                    }
                }

                placeOfBackup = this.getBackupPathString() + File.separator + string;

                if (!this.getRestorePathString().equals("<default>")) {


                    if (this.getRecursiveChoiceBooleanVector().elementAt(a).equals("file")) {

                        restorePath = this.getRestorePathString() + File.separator + path + nameOfFile;

                    }
                    restorePath = this.getRestorePathString() + File.separator + path;


                } else {

                    restorePath = this.getDataChoiceVector().elementAt(a).toString();
                }

                this.createRestoreChoiceVectors(this.getDataChoiceVector().elementAt(a).toString(), nameOfFile, this.getRecursiveChoiceBooleanVector().elementAt(a).toString());


            }

            this.updateRestoreOptionRestoreChoiceList();
            this.updateRestoreChoiceList();

        }

    }


    /**
     * Get the name of a file.
     *
     * @param string a string that can be: <br>
     * - a file with its path <br>
     * - a directory with its path <br>
     *
     * @return the name of the file or an empty string if it is a directory <br>
     * f.e. "c:\work\file.exe" ---> returns "file.exe"
     *
     */
    public String getNameOfFile(String string) {

        File file = new File(string);
        String nameOfFile;

        if (file.isFile()) {

            nameOfFile = file.getName();
            return nameOfFile;

        } else
            return nameOfFile = "";

    }


    /**
     * Create the Restore Choice Vectors. There is one vector for the views and one vector which will be stored. The vector
     * for the views contains the addition (recursive), if a folder is recursive.
     *
     * @param string a string that can be: <br>
     * - a file with its path <br>
     * - a directory with its path <br>
     * - a recursive directory with its path <br>
     * @param nameOfFile the name of a possible file
     * @param kindOfItem file or folder or recursive folder
     *
     */
    public void createRestoreChoiceVectors(String string, String nameOfFile, String kindOfItem) {

        if (kindOfItem.equals("file")) {

            if (this.getRestorePathString().equals("<default>")) {

                this.getRestoreChoiceVector().add(string);
                this.getDataRestoreChoiceVector().add(string + File.separator);

            } else {

                this.getRestoreChoiceVector().add(this.getRestorePathString() + File.separator + this.createPathInCorrectOrder(this.getPath(string)) + File.separator + nameOfFile);
                this.getDataRestoreChoiceVector().add(string);

            }

            this.getRecursiveRestoreChoiceBooleanVector().add("file");

        }

        if (kindOfItem.equals("folder")) {

            if (this.getRestorePathString().equals("<default>")) {

                this.getRestoreChoiceVector().add(string);
                this.getDataRestoreChoiceVector().add(string + File.separator);

            } else {

                this.getRestoreChoiceVector().add(this.getRestorePathString() + File.separator + this.createPathInCorrectOrder(this.getPath(string)));
                this.getDataRestoreChoiceVector().add(string);

            }


            this.getRecursiveRestoreChoiceBooleanVector().add("folder");

        }

        if (kindOfItem.equals("recFolder")) {

            if (this.getRestorePathString().equals("<default>")) {

                this.getRestoreChoiceVector().add(string + " (recursive)");
                this.getDataRestoreChoiceVector().add(string);

            } else {

                this.getRestoreChoiceVector().add(this.getRestorePathString() + File.separator + this.createPathInCorrectOrder(this.getPath(string + " (recursive)")));
                this.getDataRestoreChoiceVector().add(string);

            }

            this.getRecursiveRestoreChoiceBooleanVector().add("recFolder");
        }

    }


    /**
     * Check if an entry allready exists in the RestoreChoice.
     *
     * @param string a string that can be: <br>
     * - a file with its path <br>
     * - a directory with its path <br>
     * @return true if an entry allready exists, false otherwise
     */
    public boolean checkRestoreChoiceContents(String string) {

        if (getDataRestoreChoiceVector().contains(string)) {

            JFrame frame = new JFrame();

            if (frame != null) {

                JOptionPane.showMessageDialog(frame, "This entry allready exists in the RestorChoice!", "Message", JOptionPane.ERROR_MESSAGE);
                System.out.println("This entry allready exists in the RestorChoice!!");
                return false;

            } else
                throw new NullPointerException("Could not create frame. The frame is null.");

        } else
            return true;

    }


    /**
     * At the RestoreChoice you only can remove elements.
     *
     */
    public void notPossibleRestoreOptionRestoreChoiceTabbedPane() {

        RestoreView view = (RestoreView) getView();

        System.out.println("Only \"Remove selected\" and \"Remove all \" is electable!");

        JFrame frame = new JFrame();

        if (frame != null) {

            JOptionPane.showMessageDialog(frame, "Only \"Remove selected\" and \"Remove all \" is electable!", "Message", JOptionPane.ERROR_MESSAGE);

        } else
            throw new NullPointerException("Could not create frame. The frame is null.");

    }


    /**
     * At the Choice you only can remove elements, if the Choice is not empty and the user has selected an element.
     *
     */
    public void notPossibleRestoreOptionChoiceTabbedPane() {

        RestoreView view = (RestoreView) getView();

        if ((view.getRestoreOption().getSelectedChoiceIndex() == -1) && (view.getRestoreOption().getRestoreOptionTabbedPane().getSelectedIndex() == 0)) {

            JFrame frame = new JFrame();

            if (frame != null) {

                if (this.getChoiceVector().isEmpty() == true) {

                    System.out.println("The choice is empty!");
                    JOptionPane.showMessageDialog(frame, "The Choice is empty!", "Message", JOptionPane.ERROR_MESSAGE);

                } else {

                    System.out.println("No entry in the Choice was selected!");
                    JOptionPane.showMessageDialog(frame, "No entry in the Choice was selected!", "Message", JOptionPane.ERROR_MESSAGE);

                }

            } else
                throw new NullPointerException("Could not create frame. The frame is null.");

        }

    }


    /**
     * Remove the selected element from the RestoreChoice.
     *
     */
    public void removeSelectedFromRestoreChoiceVector() throws Exception {

        RestoreView view = (RestoreView) getView();

        if (view.getRestoreOption().getRestoreOptionTabbedPane().getSelectedIndex() == 0) {

            System.out.println("Entries in the Choice are only removable at the Backup Option Window!");

            JFrame frame = new JFrame();

            if (frame != null) {

                JOptionPane.showMessageDialog(frame, "Entries in the Choice are only removable at the Backup Option Window!", "Message", JOptionPane.ERROR_MESSAGE);

            } else
                throw new NullPointerException("Could not create frame. The frame is null.");

        } else {

            if (this.getRestoreChoiceVector().isEmpty() == false) {

                if ((view.getRestoreOption().getSelectedRestoreChoiceIndex() != -1) && (view.getRestoreOption().getRestoreOptionTabbedPane().getSelectedIndex() == 1)) {

                    getRestoreChoiceVector().removeElementAt(view.getRestoreOption().getSelectedRestoreChoiceIndex());
                    getDataRestoreChoiceVector().removeElementAt(view.getRestoreOption().getSelectedRestoreChoiceIndex());
                    getRecursiveRestoreChoiceBooleanVector().removeElementAt(view.getRestoreOption().getSelectedRestoreChoiceIndex());


                    if (view != null) {

                        view.getRestoreOption().updateRestoreChoiceList();

                    } else
                        throw new Exception("No view available!");

                } else {

                    System.out.println("No entry in the RestoreChoice was selected!");
                    JFrame frame = new JFrame();

                    if (frame != null) JOptionPane.showMessageDialog(frame, "No entry in the RestoreChoice was selected!", "Message", JOptionPane.ERROR_MESSAGE);

                }

            } else {

                System.out.println("The RestoreChoice is empty!");
                JFrame frame = new JFrame();

                if (frame != null) {

                    JOptionPane.showMessageDialog(frame, "The RestoreChoice is empty!", "Message", JOptionPane.ERROR_MESSAGE);

                } else {

                    throw new NullPointerException("Could not create frame. The frame is null.");

                }
            }

        }

    }


    /**
     * Remove all elements from the RestoreChoice.
     *
     */
    public void removeAllFromRestoreChoiceVector() throws Exception {
        RestoreView view = (RestoreView) getView();

        getRestoreChoiceVector().removeAllElements();
        getDataRestoreChoiceVector().removeAllElements();
        getRecursiveRestoreChoiceBooleanVector().removeAllElements();

        if (view != null) {

            view.getRestoreOption().updateRestoreChoiceList();

        } else {

            throw new Exception("No view available!");

        }
    }


    /**
     * Remove the selected element from the Choice.
     *
     */
    public void removeSelectedFromChoiceVector() throws Exception {

        RestoreView view = (RestoreView) getView();

        if (this.getChoiceVector().isEmpty() == false) {

            if (view.getBackupOption().getSelectedChoiceIndex() != -1) {

                if (this.getDataRestoreChoiceVector().contains(this.getDataChoiceVector().elementAt(view.getBackupOption().getSelectedChoiceIndex()))) {

                    int size = this.getDataRestoreChoiceVector().size();

                    for (int a = 0; a < size; a++) {

                        if (this.getDataRestoreChoiceVector().elementAt(a).toString().equals(this.getDataChoiceVector().elementAt(view.getBackupOption().getSelectedChoiceIndex()).toString())) {

                            this.getDataRestoreChoiceVector().removeElementAt(a);
                            this.getRestoreChoiceVector().removeElementAt(a);
                            this.getRecursiveRestoreChoiceBooleanVector().removeElementAt(a);

                        }

                    }

                }

                getChoiceVector().removeElementAt(view.getBackupOption().getSelectedChoiceIndex());
                getDataChoiceVector().removeElementAt(view.getBackupOption().getSelectedChoiceIndex());
                getRecursiveChoiceBooleanVector().removeElementAt(view.getBackupOption().getSelectedChoiceIndex());

                if (view != null) {

                    view.getBackupOption().updateChoiceList();
                    view.getRestoreOption().updateChoiceList();
                    view.getRestoreOption().updateRestoreChoiceList();
                    view.updateChoiceList();

                } else {

                    throw new Exception("No view available!");

                }

            } else {

                System.out.println("No entry in the choice was selected!");
                JFrame frame = new JFrame();

                if (frame != null) {

                    JOptionPane.showMessageDialog(frame, "No entry in the choice was selected!", "Message", JOptionPane.ERROR_MESSAGE);

                }
            }

        } else {

            System.out.println("There are no entries in the vector!");
            JFrame frame = new JFrame();

            if (frame != null) {

                JOptionPane.showMessageDialog(frame, "There are no entries in the choice!", "Message", JOptionPane.ERROR_MESSAGE);

            } else {

                throw new NullPointerException("Could not create frame. The frame is null.");

            }

        }
    }


    /**
     * Remove all elements from the Choice.
     *
     */
    public void removeAllFromChoiceVector() throws Exception {

        RestoreView view = (RestoreView) getView();

        this.getDataChoiceVector().removeAllElements();
        this.getChoiceVector().removeAllElements();
        this.getRecursiveChoiceBooleanVector().removeAllElements();
        this.getDataRestoreChoiceVector().removeAllElements();
        this.getRestoreChoiceVector().removeAllElements();
        this.getRecursiveRestoreChoiceBooleanVector().removeAllElements();

        if (view != null) {

            view.getBackupOption().updateChoiceList();
            view.getRestoreOption().updateChoiceList();
            view.getRestoreOption().updateRestoreChoiceList();
            view.updateChoiceList();

        }

    }


    /**
     * Set the Restore Path to "default". The data will be restored to the place, from where we backuped.
     *
     */
    public void setRestorePathToDefault() {

        RestoreView view = (RestoreView) getView();

        this.setRestorePathString("<default>");
        this.getRestoreChoiceVector().removeAllElements();

        int size = this.getDataRestoreChoiceVector().size();

        for (int a = 0; a < size; a++) {

            if (this.getRecursiveRestoreChoiceBooleanVector().elementAt(a).toString().equals("recFolder")) {

                this.getRestoreChoiceVector().add(this.getDataRestoreChoiceVector().elementAt(a) + "(recursive)");

            } else
                this.getRestoreChoiceVector().add(this.getDataRestoreChoiceVector().elementAt(a));

        }

        try {

            this.updateRestoreOptionRestorePathTextField();
            this.updateRestoreOptionRestoreChoiceList();

        } catch (Exception e) {

            System.out.println("Fehler:" + e);

        }
        ;

    }


    /**
     * Update the RepositoryList at the Backup Option Window.
     *
     */
    public void updateBackupOptionRepositoryList() throws Exception {

        RestoreView view = (RestoreView) getView();

        if (view != null) {

            view.getBackupOption().updateRepositoryList();
            view.getBackupOption().updateRepositoryVectorSize();

        } else {

            throw new Exception("No view available!");

        }
    }


    /**
     * Update the RepositoryList at the Restore Option Window.
     *
     */
    public void updateRestoreOptionRepositoryList() throws Exception {

        RestoreView view = (RestoreView) getView();

        if (view != null) {

            view.getRestoreOption().updateRepositoryList();

        } else {

            throw new Exception("No view available!");

        }
    }


    /**
     * Update the RepositoryList at the Main View.
     *
     */
    public void updateRepositoryList() throws Exception {

        RestoreView view = (RestoreView) getView();

        if (view != null) {

            view.updateRepositoryList();

        } else {

            throw new Exception("No view available!");

        }
    }


    /**
     * Update the ChoiceList at the Backup Option Window.
     *
     */
    public void updateBackupOptionChoiceList() throws Exception {

        RestoreView view = (RestoreView) getView();

        if (view != null) {

            view.getBackupOption().updateChoiceList();

        } else {

            throw new Exception("No view available!");
        }
    }


    /**
     * Update the ChoiceList at the Restore Option Window.
     *
     */
    public void updateRestoreOptionChoiceList() throws Exception {

        RestoreView view = (RestoreView) getView();

        if (view != null) {

            view.getRestoreOption().updateChoiceList();

        } else {

            throw new Exception("No view available!");

        }
    }

    /**
     * Update the RestoreChoiceList at the Restore Option Window.
     *
     */
    public void updateRestoreOptionRestoreChoiceList() throws Exception {

        RestoreView view = (RestoreView) getView();

        if (view != null) {

            view.getRestoreOption().updateRestoreChoiceList();

        } else {

            throw new Exception("No view available!");

        }
    }


    /**
     * Update the RestoreChoiceList at the Main View.
     *
     */
    public void updateRestoreChoiceList() throws Exception {

        RestoreView view = (RestoreView) getView();

        if (view != null) {

            view.updateRestoreChoiceList();

        } else {

            throw new Exception("No view available!");

        }
    }


    /**
     * Update the ChoiceList at the Main View.
     *
     */
    public void updateChoiceList() throws Exception {

        RestoreView view = (RestoreView) getView();

        if (view != null) {

            view.updateChoiceList();

        } else {

            throw new Exception("No view available!");

        }
    }


    /**
     * Update the Backup Path at the Backup Option Window.
     *
     */
    public void updateBackupOptionBackupPathTextField() throws Exception {

        RestoreView view = (RestoreView) getView();

        if (view != null) {

            view.getBackupOption().updateBackupPathTextField();

        } else {

            throw new Exception("No view available!");
        }
    }

    /**
     * Update the Restore Path at the Restore Option Window.
     *
     */
    public void updateRestoreOptionRestorePathTextField() throws Exception {

        RestoreView view = (RestoreView) getView();

        if (view != null) {

            view.getRestoreOption().updateRestorePathTextField();

        } else {

            throw new Exception("No view available!");

        }
    }

    /**
     * Update the Storing Path at the Storing Option Window.
     *
     */
    public void updateStoringOptionStoringPathTextField() throws Exception {

        RestoreView view = (RestoreView) getView();

        if (view != null) {

            view.getStoringOption().updateStoringPathTextField();

        } else {

            throw new Exception("No view available!");

        }
    }


    /**
     * Update the BackupTextArea at the Main View.
     *
     */
    public void updateBackupTextArea() throws Exception {

        RestoreView view = (RestoreView) getView();

        if (view != null) {

            view.updateBackupTextArea();

        } else {

            throw new Exception("No view available!");

        }
    }


    /**
     * Update the RestoreTextArea at the Main View.
     *
     */
    public void updateRestoreTextArea() throws Exception {

        RestoreView view = (RestoreView) getView();

        if (view != null) {

            view.updateRestoreTextArea();

        } else {

            throw new Exception("No view available!");

        }
    }


    /**
     * Update the ErrorTextArea at the Main View.
     *
     */
    public void updateErrorTextArea() throws Exception {

        RestoreView view = (RestoreView) getView();

        if (view != null) {

            view.updateErrorTextArea();

        } else {

            throw new Exception("No view available!");

        }
    }


    /**
     * Update the InfoTextArea at the Main View.
     *
     */
    public void updateInfoTextArea() throws Exception {

        RestoreView view = (RestoreView) getView();

        if (view != null) {

            view.updateInfoTextArea();

        } else {

            throw new Exception("No view available!");

        }
    }


    /**
     * Get all files from the Choice.
     * @return a vector with all files
     *
     */
    public Vector getFilesFromChoice() {

        Vector fileVector = new Vector();

        int choiceVectorSize = this.getDataChoiceVector().size();

        for (int a = 0; a < choiceVectorSize; a++) {

            File file = new File(this.getDataChoiceVector().elementAt(a).toString());

            if (file.isFile()) {

                fileVector.addElement(this.getDataChoiceVector().elementAt(a).toString());

            }

        }

        return fileVector;

    }


    /**
     * Get all folders from the Choice.
     * @return a vector with all folders
     *
     */
    public Vector getFolderFromChoice() {

        Vector folderVector = new Vector();

        int choiceVectorSize = this.getDataChoiceVector().size();

        for (int a = 0; a < choiceVectorSize; a++) {

            File file = new File(this.getDataChoiceVector().elementAt(a).toString());

            if ((file.isDirectory()) && (this.getRecursiveChoiceBooleanVector().elementAt(a).toString().equals("folder"))) {


                folderVector.addElement(this.getDataChoiceVector().elementAt(a).toString());

            }

        }

        return folderVector;

    }


    /**
     * Get all recursive folders from the Choice.
     * @return a vector with all recursive folders
     *
     */
    public Vector getRecursiveFolderFromChoice() {

        Vector recursiveFolderVector = new Vector();

        int choiceVectorSize = this.getDataChoiceVector().size();

        for (int a = 0; a < choiceVectorSize; a++) {

            File file = new File(this.getDataChoiceVector().elementAt(a).toString());

            if ((file.isDirectory()) && (this.getRecursiveChoiceBooleanVector().elementAt(a).toString().equals("recFolder"))) {

                recursiveFolderVector.addElement(this.getDataChoiceVector().elementAt(a).toString());

            }

        }


        return recursiveFolderVector;

    }


    /**
     * Get all files from the RestoreChoice.
     * @return a vector with all files
     *
     */
    public Vector getFilesFromRestoreChoice() {

        Vector fileVector = new Vector();

        int restoreChoiceVectorSize = this.getDataRestoreChoiceVector().size();

        for (int a = 0; a < restoreChoiceVectorSize; a++) {

            File file = new File(this.getDataRestoreChoiceVector().elementAt(a).toString());

            if (this.getRecursiveRestoreChoiceBooleanVector().elementAt(a).equals("file")) {

                String nameOfFile = this.getNameOfFile(this.getDataRestoreChoiceVector().elementAt(a).toString());

                if (this.getRestorePathString().equals("<default>")) {

                    fileVector.addElement(this.getDataRestoreChoiceVector().elementAt(a).toString());

                } else
                    fileVector.addElement(this.getDataRestoreChoiceVector().elementAt(a).toString());

            }

        }

        return fileVector;

    }


    /**
     * Get all folders from the RestoreChoice.
     * @return a vector with all folders
     *
     */
    public Vector getFolderFromRestoreChoice() {

        Vector folderVector = new Vector();

        int dataChoiceVectorSize = this.getDataRestoreChoiceVector().size();

        for (int a = 0; a < dataChoiceVectorSize; a++) {

            if (this.getRecursiveRestoreChoiceBooleanVector().elementAt(a).toString().equals("folder")) {

                folderVector.addElement(this.getDataRestoreChoiceVector().elementAt(a).toString());

            }

        }

        return folderVector;

    }

    /**
     * Get all recursive folders from the RestoreChoice.
     * @return a vector with all recursive folders
     *
     */
    public Vector getRecursiveFolderFromRestoreChoice() {

        Vector recursiveFolderVector = new Vector();

        int dataChoiceVectorSize = this.getDataRestoreChoiceVector().size();

        for (int a = 0; a < dataChoiceVectorSize; a++) {

            if (this.getRecursiveRestoreChoiceBooleanVector().elementAt(a).toString().equals("recFolder")) {


                recursiveFolderVector.addElement(this.getDataRestoreChoiceVector().elementAt(a).toString());

            }

        }

        return recursiveFolderVector;

    }


    /**
     * Get all subfolders from a folder.
     * @param array an array with files and folders
     * @param path the path of an folder or file
     *
     */
    public void getSubFolder(String[] array, String path) {

        int arrayLength = array.length;

        for (int a = 0; a < arrayLength; a++) {

            File file = new File(path + File.separator + array[a].toString());

            if ((file.isDirectory() == true)) {

                this.getFolderVector().addElement(path + File.separator + array[a].toString());

                File file2 = new File(path + File.separator + array[a].toString());

                getSubFolder(file2.list(), path + File.separator + array[a].toString());

            }

        }

    }

    /**
     * Get all files, which are stored in a folder.
     * @param array an array with files and folders
     * @param path the path of an folder or file
     * @return a vector with files
     */
    public Vector getFilesFromFolder(String[] array, String path) {

        Vector fileVector = new Vector();

        int arrayLength = array.length;

        if (arrayLength > 0) {

            for (int a = 0; a < arrayLength; a++) {

                File file = new File(path + File.separator + array[a].toString());

                if (file.isFile()) {

                    fileVector.addElement(path + File.separator + array[a].toString());

                }

            }

        } else {

            File file = new File(path);
            file.mkdirs();

        }

        return fileVector;
    }

    /**
     * Creates a path in a correct order. The method getPath(String str) returns a vector like that: <br>
     * c:\folder1\folder2\folder3 ---> [folder3, folder2, folder1] <br>
     * This method creates a path in correct order. <br>
     * [folder3, folder2, folder1] ---> \folder1\folder2\folder3
     * @param pathVector a vector with folders in a reverse order
     * @return the path in correct order
     * @see method getPath(String str)
     *
     */
    public String createPathInCorrectOrder(Vector pathVector) {

        int pathVectorSize = pathVector.size();

        String fullPath = "";
        String fullPath0 = "";

        if (pathVectorSize > 0) {

            fullPath = pathVector.elementAt(pathVectorSize - 1).toString();
            fullPath0 = pathVector.elementAt(pathVectorSize - 1).toString();

            for (int a = 2; a < pathVectorSize + 2; a++) {

                File file3 = new File(fullPath);

                if ((pathVectorSize - a) > -1) {

                    fullPath = fullPath0 + File.separator + pathVector.elementAt(pathVectorSize - a).toString();
                    fullPath0 = fullPath0 + File.separator + pathVector.elementAt(pathVectorSize - a).toString();

                }

            }

        }

        return fullPath;

    }


    /**
     * Get the path of a file/folder. This method returns a vector like that: <br>
     * c:\folder1\folder2\folder3 ---> [folder3, folder2, folder1] <br>
     * @param str a file/folder
     * @return a vector with the path in reverse order
     */
    public Vector getPath(String str) {

        String path = "";
        String directory = "";
        boolean endOfPath = false;
        Vector pathVector = new Vector();

        File file0 = new File(str);

        if (file0.isFile() == false) {

            path = str;

        } else {

            path = file0.getParent().toString();

        }

        while (endOfPath != true) {

            File file2 = new File(path);

            if (file2.getParent() != null) {

                endOfPath = false;
                path = file2.getParent().toString();

                try {

                    directory = file2.getCanonicalFile().getName();
                    pathVector.addElement(directory);

                } catch (Exception e) {
                    System.out.println("Fehler: " + e);
                }
                ;

            } else {

                endOfPath = true;

            }


        }

        return pathVector;

    }

    /**
     * This method is used for backup and restore process. Prepares source and destination pathes.
     * Calls the method checkFileOccurence(src,dest) to go on in the copy process.
     *
     * @param element the source
     * @param fullPath the path. Like folder1\folder2\folder3
     * @param nameOfFile the name of the file
     * @param str distinguish backup and restore process.
     *
     */
    public void copyFileToDestination(String element, String fullPath, String nameOfFile, String str) throws FileNotFoundException {

        RestoreView view = (RestoreView) getView();
        String source,destination;

        if (str.equals("backup")) {

            destination = this.getBackupPathString() + File.separator + fullPath + File.separator + nameOfFile;
            source = element;

            File src = new File(source);
            File dest = new File(destination);

            if (this.checkFileOccurence(src, dest) == true) {

                this.setReportAreaString("backup  " + src.toString() + "\n");

                try {

                    this.updateBackupTextArea();

                } catch (Exception e) {
                    System.out.println("Fehler: " + e);
                }
                ;

            }

        } else {

            if (this.getRestorePathString().equals("<default>")) {

                destination = element;
                source = element + File.separator + nameOfFile;

                File src = new File(source);
                File dest = new File(destination);

                if (this.checkFileOccurence(src, dest) == true) {

                    this.setReportAreaString("restore  " + source + "\n");

                    try {

                        this.updateRestoreTextArea();

                    } catch (Exception e) {
                        System.out.println("Fehler: " + e);
                    }
                    ;

                }

            } else {

                destination = this.restorePathString + File.separator + fullPath + File.separator + nameOfFile;
                source = element + File.separator + nameOfFile;

                File src = new File(source);
                File dest = new File(destination);

                if (this.checkFileOccurence(src, dest) == true) {

                    this.setReportAreaString("restore  " + source + "\n");

                    try {

                        this.updateRestoreTextArea();

                    } catch (Exception e) {
                        System.out.println("Fehler: " + e);
                    }
                    ;

                }
            }
        }

    }

    /**
     * This method is used for backup and restore process and checks the file occurence.
     * The user has to decide to overwrite existing files or not.
     *
     * @param src the source
     * @param dest the destination
     * @return true if an existing file can be overwrited, false otherwise
     *
     *
     */
    public boolean checkFileOccurence(File src, File dest) {

        if (this.getCopyAll() == false) {

            if (this.getCancelCopy() == false) {

                if (dest.exists()) {

                    System.out.println("This file allready exists!");

                    JFrame frame = new JFrame();
                    int value = JOptionPane.showConfirmDialog(frame, "The file " + dest + " allready exists! Do you want to overwrite it?", "Message", JOptionPane.YES_NO_CANCEL_OPTION);

                    if (value == 0) {

                        if (bool == false) {

                            bool = true;

                            JFrame frame1 = new JFrame();

                            int value2 = JOptionPane.showConfirmDialog(frame1, "Do you want to replace all files?", "Message", JOptionPane.YES_NO_OPTION);

                            if (value2 == 0) {

                                this.setCopyAll(true);

                            } else
                                this.setCopyAll(false);
                        }

                        copy(src, dest);

                        return true;

                    }

                    if (value == 2) {

                        this.setCancelCopy(true);
                        return false;
                    }

                } else {

                    copy(src, dest);
                    return true;

                }

            }
            return false;

        } else {

            copy(src, dest);
            return true;


        }
    }

    /**
     * This method is used for backup and restore process.
     * Copy a file from its source to the destination.
     *
     * @param src the source
     * @param dest the destination
     *
     */
    public void copy(File src, File dest) {

        System.out.println("copy "+src.toString());

        this.setCountFile(this.getCountFile() + 1);

        try {

            FileInputStream in = new FileInputStream(src);
            FileOutputStream out = new FileOutputStream(dest);

            byte[] buffer = new byte[262144];
            int read = 0;

            while (true) {

                read = in.read(buffer);

                Integer i = new Integer(read);

                if (read == -1) {
                    //-1 means EOF
                    break;

                } else
                    out.write(buffer, 0, read);

                this.setCountBytes(this.getCountBytes() + read);

            }

        } catch (Exception e) {
            System.out.println("Fehler: " + e);
        }
        ;

    }

    /**
     * This method is used for backup and restore process. Folders will be created.
     * Normally the source is equal to destination if you use the restore process. (default).
     * But the user can chose another destination.
     *
     * @param pathVector the pathVector like [folder3,folder2,folder1]
     * @param element the source
     * @param string distinguish backup and restore process
     */
    public void createFolder(Vector pathVector, String element, String string) {

        int pathVectorSize = pathVector.size();

        String fullPath0 = "",fullPath = "";

        if (pathVectorSize > 0) {

            if (string.equals("backup")) {

                fullPath = this.getBackupPathString() + File.separator + this.createPathInCorrectOrder(pathVector);

            } else {

                if (this.getRestorePathString().equals("<default>")) {

                    fullPath = element;

                } else

                    fullPath = this.getRestorePathString() + File.separator + this.createPathInCorrectOrder(pathVector);

            }

            File file = new File(fullPath);


            if (!file.exists()) {

                file.mkdirs();

            }

        }

    }


    /**
     * Create an array of files.
     *
     * @param element the source
     * @return an array of files
     */
    public String[] createListFromElement(String element) {

        File file = new File(element);

        return file.list();

    }


    /**
     * Preapares the files for the copy process. Calls the method copyFileToDestination to go on in the copy process.
     * Get the following information: <br>
     * - the path of a file/directory like folder1\folder2\folder3 <br>
     * - the name of the file <br>
     * - the source <br>
     *
     * @param vector a vector with sources (files/directories with their pathes)
     * @param str distinguish the backup and restore process
     */
    public void prepareFilesForCopy(Vector vector, String str) {

        String fullPath, nameOfFile, element;
        int size;

        if (str.equals("restore")) {

            size = vector.size();

            for (int a = 0; a < size; a++) {

                this.createFolder(this.getPath(vector.elementAt(a).toString()), vector.elementAt(a).toString(), "restore");

                if (this.getRestorePathString().equals("<default>")) {

                    element = vector.elementAt(a).toString();

                } else {

                    element = this.getBackupPathString() + File.separator + this.createPathInCorrectOrder(this.getPath(vector.elementAt(a).toString()));

                }

                fullPath = this.createPathInCorrectOrder(this.getPath(vector.elementAt(a).toString()));

                File file = new File(vector.elementAt(a).toString());
                nameOfFile = file.getName();

                try {


                    this.copyFileToDestination(element, fullPath, nameOfFile, "restore");


                } catch (Exception e) {

                    System.out.println("Fehler: " + e);
                    this.setReportAreaString("restore: " + e.getMessage());

                    try {

                        this.updateErrorTextArea();

                    } catch (Exception e1) {
                        System.out.println("Fehler: " + e1);
                    }
                    ;

                }
            }


        } else {

            size = vector.size();

            for (int a = 0; a < size; a++) {

                element = vector.elementAt(a).toString();
                this.createFolder(this.getPath(element), "", "backup");
                fullPath = this.createPathInCorrectOrder(this.getPath(element));

                File file = new File(element);
                nameOfFile = file.getName();

                try {

                    this.copyFileToDestination(element, fullPath, nameOfFile, "backup");

                } catch (Exception e) {

                    System.out.println("Fehler: " + e);

                    this.setReportAreaString("backup: " + e.getMessage());


                    try {

                        this.updateErrorTextArea();

                    } catch (Exception e1) {
                        System.out.println("Fehler: " + e1);
                    }
                    ;

                }
            }
        }
    }


    /**
     * The backup process.<br>
     * - get files from the choice and preapare them for copy <br>
     * - get directories from the choice and copy the files of this directories <br>
     * - get recursive directories from the choice <br>
     *
     */
    public void backupCopy() {

        this.setCancelCopy(false);
        this.setCopyAll(false);
        this.bool = false;
        this.setCountFile(0);
        this.setCountBytes(0);

        RestoreView view = (RestoreView) getView();

        if (this.checkInputValuesForBackup()) {

            this.setReportAreaString("BEGIN OF BACKUP\n");

            try {

                view.getBackupTextArea().setText("");
                this.updateBackupTextArea();

            } catch (Exception e1) {
                System.out.println("Fehler: " + e1);
            }
            ;

            System.out.println("Start of backup process...");

            // get files from the choice and preapare them for copy
            this.prepareFilesForCopy(this.getFilesFromChoice(), "backup");

            // get directories from the choice and copy the files of this directories
            Vector folderVector = this.getFolderFromChoice();
            int folderVectorSize = folderVector.size();

            for (int a = 0; a < folderVectorSize; a++) {

                String[] array = this.createListFromElement(folderVector.elementAt(a).toString());
                String path = folderVector.elementAt(a).toString();

                this.prepareFilesForCopy(this.getFilesFromFolder(array, path), "backup");

            }

            // get recursive directories from the choice
            this.getFolderVector().removeAllElements();

            Vector recursiveFolderVector = new Vector();

            recursiveFolderVector = this.getRecursiveFolderFromChoice();

            int recursiveFolderVectorSize = recursiveFolderVector.size();

            for (int a = 0; a < recursiveFolderVectorSize; a++) {

                this.getFolderVector().removeAllElements();

                this.getFolderVector().addElement(recursiveFolderVector.elementAt(a));

                // get sub-directories in this directory
                String[] array = this.createListFromElement(recursiveFolderVector.elementAt(a).toString());
                String path = recursiveFolderVector.elementAt(a).toString();

                this.getSubFolder(array, path);

                for (int b = 0; b < this.getFolderVector().size(); b++) {

                    this.createFolder(this.getPath(this.getFolderVector().elementAt(b).toString()), "", "backup");

                }


                folderVectorSize = this.getFolderVector().size();
                for (int b = 0; b < folderVectorSize; b++) {

                    File file = new File(this.getFolderVector().elementAt(b).toString());
                    String[] array2 = file.list();
                    String path2 = this.getFolderVector().elementAt(b).toString();

                    this.prepareFilesForCopy(this.getFilesFromFolder(array2, path2), "backup");

                }

            }

            this.setReportAreaString("END OF BACKUP\n");

            try {

                this.updateBackupTextArea();

            } catch (Exception e1) {
                System.out.println("Fehler: " + e1);
            }
            ;

            System.out.println("End of backup process...");

        } else {

            view.getBackupTextArea().setText("");

            this.setReportAreaString("One or more error(s) occured.");

            try {

                this.updateBackupTextArea();

            } catch (Exception e) {
                System.out.println("Fehler: " + e);
            }
            ;

        }

        BigDecimal value1 = new BigDecimal(this.getCountBytes() / 1024 / 1000);
        BigDecimal value2 = new BigDecimal(this.getCountBytes() / 1024);


        this.setReportAreaString(this.getCountFile() + " file(s) were copied. \n" + "Size: " + value2.setScale(2, 2) + " kB  =  " + value1.setScale(2, 2) + " MB");

        try {

            view.getInfoTextArea().setText("");
            this.updateInfoTextArea();

        } catch (Exception e1) {
            System.out.println("Fehler: " + e1);
        }
        ;

    }


    /**
     * The restore process.<br>
     * - get files from the RestoreChoice and preapare them for copy <br>
     * - get directories from the RestoreChoice and copy the files of this directories <br>
     * - get recursive directories from the choice <br>
     *
     */
    public void restoreCopy() {

        this.setCancelCopy(false);
        this.setCopyAll(false);
        this.bool = false;
        this.setCountFile(0);
        this.setCountBytes(0);

        RestoreView view = (RestoreView) getView();

        if (this.checkInputValuesForRestore()) {

            this.setReportAreaString("BEGIN OF RESTORE\n");

            try {

                view.getRestoreTextArea().setText("");
                this.updateRestoreTextArea();

            } catch (Exception e1) {
                System.out.println("Fehler: " + e1);
            }
            ;

            System.out.println("Start of restore copy process...");
            // get files from the restoreChoice and preapare them for copy
            this.prepareFilesForCopy(this.getFilesFromRestoreChoice(), "restore");

            // get directories from the restoreChoice and copy the files of this directories
            Vector folderVector = this.getFolderFromRestoreChoice();
            int folderVectorSize = folderVector.size();

            for (int a = 0; a < folderVectorSize; a++) {

                String[] array = this.createListFromElement(folderVector.elementAt(a).toString());
                String path = folderVector.elementAt(a).toString();

                this.prepareFilesForCopy(this.getFilesFromFolder(array, path), "restore");

            }

            // get recursive directories from the choice
            this.getFolderVector().removeAllElements();

            Vector recursiveFolderVector = new Vector();

            recursiveFolderVector = this.getRecursiveFolderFromRestoreChoice();

            int recursiveFolderVectorSize = recursiveFolderVector.size();

            for (int a = 0; a < recursiveFolderVectorSize; a++) {

                this.getFolderVector().removeAllElements();

                this.getFolderVector().addElement(recursiveFolderVector.elementAt(a));

                // get sub-directories in this directory
                String[] array = this.createListFromElement(recursiveFolderVector.elementAt(a).toString());
                String path = recursiveFolderVector.elementAt(a).toString();

                this.getSubFolder(array, path);

                folderVectorSize = this.getFolderVector().size();

                for (int b = 0; b < folderVectorSize; b++) {

                    File file = new File(this.getFolderVector().elementAt(b).toString());
                    String[] array2 = file.list();
                    String path2 = this.getFolderVector().elementAt(b).toString();
                    this.prepareFilesForCopy(this.getFilesFromFolder(array2, path2), "restore");

                }

            }

            this.setReportAreaString("END OF RESTORE\n");

            try {

                this.updateRestoreTextArea();

            } catch (Exception e1) {
                System.out.println("Fehler: " + e1);
            }
            ;

            System.out.println("End of restore copy process...");

        } else {

            view.getRestoreTextArea().setText("");

            this.setReportAreaString("One or more error(s) occured.");

            try {

                this.updateRestoreTextArea();

            } catch (Exception e) {
                System.out.println("Fehler: " + e);
            }
            ;

        }

        BigDecimal value1 = new BigDecimal(this.getCountBytes() / 1024 / 1000);
        BigDecimal value2 = new BigDecimal(this.getCountBytes() / 1024);

        this.setReportAreaString(this.getCountFile() + " file(s) were copied. \n" + "Size: " + value2.setScale(2, 2) + " kB  =  " + value1.setScale(2, 2) + " MB");

        try {

            view.getInfoTextArea().setText("");
            this.updateInfoTextArea();

        } catch (Exception e1) {
            System.out.println("Fehler: " + e1);
        }
        ;

    }

    /**
     * Check the values before the backup process starts. <br>
     * - ckeck the Backup Path <br>
     * - check the Storing Path <br>
     * - check the Choice
     */
    public boolean checkInputValuesForBackup() {

        RestoreView view = (RestoreView) getView();

        boolean bool = true;

        view.getErrorTextArea().setText("");
        view.getInfoTextArea().setText("");


        // check Backup Path
        File file = new File(this.getBackupPathString());

        if (!file.isDirectory()) {

            System.out.println("No valid Backup Path was selected!");
            this.setReportAreaString("No vaild Backup Path was selected!\n");
            bool = false;

            try {

                this.updateErrorTextArea();

            } catch (Exception e) {
                System.out.println("Fehler: " + e);
            }
            ;

        }

        File file2 = new File(this.getStoringPathString());

        if (!file2.isDirectory()) {

            System.out.println("No vaild Storing Path was selected!");
            this.setReportAreaString("No valid Storing Path was selected!\n");
            bool = false;

            try {

                this.updateErrorTextArea();

            } catch (Exception e) {
                System.out.println("Fehler: " + e);
            }
            ;

        }


        if (this.checkVectorContents(this.getDataChoiceVector(), "Choice") == true) {

            System.out.println("The Choice is valid.");

        } else {

            System.out.println("The Choice is not valid.");
            bool = false;

        }

        return bool;

    }

    /**
     * Check the values before the restore process starts. <br>
     * - ckeck the Restore Path <br>
     * - check the Storing Path <br>
     * - check the RestoreChoice
     */
    public boolean checkInputValuesForRestore() {

        RestoreView view = (RestoreView) getView();

        boolean bool = true;

        view.getErrorTextArea().setText("");
        view.getInfoTextArea().setText("");

        File file = new File(this.getRestorePathString());

        System.out.println("           " + this.getRestorePathString());

        if (!this.getRestorePathString().equals("<default>")) {

            if (!file.isDirectory()) {

                System.out.println("No vaild Restore Path was selected!");
                this.setReportAreaString("No valid Restore Path was selected!\n");
                bool = false;

                try {

                    this.updateErrorTextArea();

                } catch (Exception e) {
                    System.out.println("Fehler: " + e);
                }
                ;

            }

        }

        File file2 = new File(this.getStoringPathString());

        if (!file2.isDirectory()) {

            System.out.println("No vaild Storing Path was selected!");
            this.setReportAreaString("No valid Storing Path was selected!\n");
            bool = false;

            try {

                this.updateErrorTextArea();

            } catch (Exception e) {
                System.out.println("Fehler: " + e);
            }
            ;

        }

        if (this.checkVectorContents(this.getDataRestoreChoiceVector(), "RestoreChoice") == true) {

            System.out.println("The RestoreChoice is valid.");

        } else {

            System.out.println("The RestoreChoice is not valid.");
            bool = false;

        }

        return bool;

    }

    /**
     * Check the Vector contents.<br>
     * - check if the Choice or RestoreChoice is empty
     *
     * @param vector the Choice or the RestoreChoice
     * @return true if the Choice/RestoreChoice is not empty, false otherwise
     *
     */
    public boolean checkVectorContents(Vector vector, String choice) {

        int vectorSize = vector.size();
        boolean bool = true;

        if (vectorSize == 0) {

            bool = false;

            if (choice.equals("Choice")) {

                System.out.println("The Choice is empty. Please create a new one.");
                this.setReportAreaString("The Choice is empty.");

            } else {

                System.out.println("The RestoreChoice is empty. Please create a new one.");
                this.setReportAreaString("The RestoreChoice is empty.");

            }

            try {

                this.updateErrorTextArea();

            } catch (Exception e) {
                System.out.println("Fehler: " + e);
            }
            ;

        }

        return bool;

    }


    public void setView(SwingView view) {

        this.view = view;

    }


    public SwingView getView() {

        return this.view;

    }


    public void setRepositoryVectorLoadedFromFile(boolean bool) {

        this.repositoryVectorLoadedFromFile = bool;

    }


    public boolean getRepositoryVectorLoadedFromFile() {

        return this.repositoryVectorLoadedFromFile;

    }


    public void setRepositoryVectorLoadedFromFileIsEmpty(boolean bool) {

        this.repositoryVectorLoadedFromFileIsEmpty = bool;

    }


    public boolean getRepositoryVectorLoadedFromFileIsEmpty() {

        return this.repositoryVectorLoadedFromFileIsEmpty;

    }


    public void setCopyAll(boolean bool) {

        this.copyAll = bool;

    }

    public boolean getCopyAll() {

        return this.copyAll;

    }


    public void setCancelCopy(boolean bool) {

        this.cancelCopy = bool;

    }


    public boolean getCancelCopy() {

        return this.cancelCopy;

    }


    public Vector createRepositoryVector() {

        return new Vector();

    }


    public Vector getRepositoryVector() {

        return repositoryVector;

    }


    public void setRepositoryVector(Vector listVector) {

        this.repositoryVector = listVector;

    }

    public Hashtable createHashTable() {

        return new Hashtable();

    }


    public Hashtable getHashtable() {

        return this.table;

    }


    public void setHashTable(Hashtable table) {

        this.table = table;

    }

    public Vector createChoiceVector() {

        return new Vector();

    }


    public Vector getChoiceVector() {

        return choiceVector;

    }


    public void setChoiceVector(Vector listVector) {

        this.choiceVector = listVector;

    }

    public Vector createRecursiveChoiceBooleanVector() {

        return new Vector();

    }


    public Vector getRecursiveChoiceBooleanVector() {

        return recursiveChoiceBooleanVector;

    }


    public void setRecursiveChoiceBooleanVector(Vector listVector) {

        this.recursiveChoiceBooleanVector = listVector;

    }

    public Vector createRecursiveRestoreChoiceBooleanVector() {

        return new Vector();

    }


    public Vector getRecursiveRestoreChoiceBooleanVector() {

        return recursiveRestoreChoiceBooleanVector;

    }


    public void setRecursiveRestoreChoiceBooleanVector(Vector listVector) {

        this.recursiveRestoreChoiceBooleanVector = listVector;

    }


    public Vector createDataChoiceVector() {

        return new Vector();

    }


    public Vector getDataChoiceVector() {

        return dataChoiceVector;

    }


    public void setDataChoiceVector(Vector listVector) {

        this.dataChoiceVector = listVector;

    }


    public Vector createRestoreChoiceVector() {

        return new Vector();

    }


    public Vector getRestoreChoiceVector() {

        return restoreChoiceVector;

    }


    public void setRestoreChoiceVector(Vector listVector) {

        this.restoreChoiceVector = listVector;

    }


    public Vector createDataRestoreChoiceVector() {

        return new Vector();

    }


    public Vector getDataRestoreChoiceVector() {

        return dataRestoreChoiceVector;

    }


    public void setDataRestoreChoiceVector(Vector listVector) {

        this.dataRestoreChoiceVector = listVector;

    }


    public Vector createPlaceInDataChoiceVector() {

        return new Vector();

    }


    public Vector getPlaceInDataChoiceVector() {

        return this.placeInDataChoiceVector;

    }


    public void setPlaceInDataChoiceVector(Vector listVector) {

        this.placeInDataChoiceVector = listVector;

    }


    public String getBackupPathString() {

        return backupPathString;

    }


    public void setBackupPathString(String str) {

        this.backupPathString = str;

        try {

            this.updateBackupOptionBackupPathTextField();
        } catch (Exception e) {
            System.out.println("Fehler: " + e);
        }

    }


    public String getStoringPathString() {

        return storingPathString;

    }


    public void setStoringPathString(String str) {

        this.storingPathString = str;

        try {

            this.updateStoringOptionStoringPathTextField();

        } catch (Exception e) {
            System.out.println("Fehler: " + e);
        }


    }


    public String getInformationFileString() {

        return informationFileString;

    }


    public void setInformationFileString(String str) {

        this.informationFileString = str;

    }

    public String getReportAreaString() {

        return reportAreaString;

    }


    public void setReportAreaString(String str) {

        this.reportAreaString = str;

    }

    public String getRestorePathString() {

        return restorePathString;

    }


    public void setRestorePathString(String str) {

        this.restorePathString = str;
        this.getRestoreChoiceVector().removeAllElements();

        int size = this.getDataRestoreChoiceVector().size();

        for (int a = 0; a < size; a++) {

            String nameOfFile = this.getNameOfFile(this.getDataRestoreChoiceVector().elementAt(a).toString());

            this.getRestoreChoiceVector().add(this.getRestorePathString() + File.separator + this.createPathInCorrectOrder(this.getPath(this.getDataRestoreChoiceVector().elementAt(a).toString())) + nameOfFile);

        }

        try {

            this.updateRestoreOptionRestorePathTextField();
        } catch (Exception e) {
            System.out.println("Fehler: " + e);
        }

    }


    public File getBackupPathFile() {

        return this.backupPathFile;

    }


    public void setBackupPathFile(File file) {

        this.backupPathFile = file;

    }


    public File getRestorePathFile() {

        return this.restorePathFile;

    }


    public void setRestorePathFile(File file) {

        this.restorePathFile = file;

    }


    public Vector createFolderVector() {

        return new Vector();

    }


    public Vector getFolderVector() {

        return folderVector;

    }


    public void setFolderVector(Vector folderVector) {

        this.folderVector = folderVector;

    }


    public RandomAccessFile getFile() {

        return this.file;

    }


    public void setFile(RandomAccessFile raf) {

        this.file = raf;

    }


    public int getCountFile() {

        return countFile;

    }


    public void setCountFile(int count) {

        this.countFile = count;

    }


    public double getCountBytes() {

        return countBytes;

    }


    public void setCountBytes(double count) {

        this.countBytes = count;

    }

}

