/*
 * $RCSfile: Logger.java,v $
 *
 * Copyright (c) 1999-2002. The Res Medicinae Developers. All rights reserved.
 *
 * This software is published under the GPL GNU General Public License.
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * http://www.resmedicinae.org
 * - Information in Medicine -
 */

/**
 * This class represents a Logger for the project ResAdress.
 * The Logger is a Singleton.
 * It's encapsulated the java.util.logging.Logger class.
 *
 *
 * @version          $Revision: 1.10 $
 *                   $Date: 2003/01/27 08:10:40 $
 *                   $Author: rholzmueller $
 * @author Rolf Holzmller <rolf.holzmueller@gmx.de>
 */

package org.resmedicinae.application.healthcare.resdata;


public class Logger extends Object {


    /** The Singleton for the Logger */
    static private Logger myLogger;

    /** The named subsystem. */
    static private String LOGNAMESPACE = "org.resmedicinae.application.healthcare.resdata";

    /** The standard log-level. */
    static private Level LOGSTANDARDLEVEL = (Level)Level.ALL;

    /** The constant indicating a console output handler. */
    static public final String CONSOLE_OUTPUT = "console";

    /** The constant indicating a file output handler. */
    static public final String FILE_OUTPUT = "file";


    /** The encapsulated jdk logger. */
    static private java.util.logging.Logger jdkLogger;


    /** The log level. */
    private Level level;

    /** The log file name. */
    private String logfilename = "";


    /**
     * Constructor for Logger.
     */
    private  Logger() {

        super();

        try {

            jdkLogger=null;
            logfilename="";
            level=LOGSTANDARDLEVEL;
            addConsuleHandler();
        }
        catch (Exception e) {
            System.out.println( "Error by the initialiation for the Logger" );
        }
    }


    /**
     * Returns the Singleton for the logger.
     *
     * @return the logger
     */
    static public Logger getLogger() {

        if (myLogger==null) {
            myLogger = new Logger();
        }

        return myLogger;
    }


    /**
     * Returns the jdk logger.
     *
     * @return the jdk logger
     */
    private java.util.logging.Logger getJDKLogger() {

        if (jdkLogger==null) {

            jdkLogger = java.util.logging.Logger.getLogger( LOGNAMESPACE );
            jdkLogger.setLevel( LOGSTANDARDLEVEL );
        }

        return jdkLogger;
    }



    //
    // Level.
    //

    /**
     * Returns the log level.
     *
     * @return the log level
     * @exception NullPointerException if the jdk logger is null
     */
    public Level getLevel() throws NullPointerException {

        return this.level;
    }

    /**
     * Sets the log level.
     *
     * @param level the log level
     * @exception NullPointerException if the jdk logger is null
     */
    public void setLevel(Level level) throws NullPointerException {

        this.level = level;

        // Forward the level to the embedded jdk logger.
        java.util.logging.Logger log = getJDKLogger();

        if (log != null) {

            log.setLevel(level);
        }
    }


    //
    // logfile.
    //

    /**
     * Returns the log file name.
     *
     * @return the logfilename
     */
    public String getJDKLoggerFileName() {

        return this.logfilename;
    }

    /**
     * Sets the log file name.
     *
     * @param plogfile the log file name
     */
    public void setLogFileName(String plogfilename) {

        this.logfilename = plogfilename;
    }


    //
    // Handler container.
    //

    /**
     * Returns the handlers.
     *
     * @return the handlers
     * @exception NullPointerException if the jdk logger is null
     */
    public java.util.logging.Handler[] getHandlers() throws NullPointerException {

        java.util.logging.Handler[] h = null;
        java.util.logging.Logger log = getJDKLogger();

        if (log != null) {

            h = log.getHandlers();

        } else {

            throw new NullPointerException("Could not get handlers. The jdk logger is null.");
        }

        return h;
    }

    /*
     * Removes the handlers so that the handler container is empty.
     *
     * @handlers the handlers
     * @exception NullPointerException if the handlers container is null
     */
    public void destroyHandlers(java.util.logging.Handler[] handlers) throws Exception, NullPointerException {

        if (handlers != null) {

            java.util.logging.Handler h = null;

            for (int i = 0; i < handlers.length; i++) {

                h = handlers[i];

                destroyHandler(h);
                removeHandler(h);
            }

        } else {

            throw new NullPointerException("Could not destroy handlers. The handlers container is null.");
        }
    }

    //
    // Handler.
    //

    /**
     * Adds the handler.
     *
     * @param h the handler
     * @exception NullPointerException if the jdk logger is null
     */
    public void addHandler(java.util.logging.Handler h) throws NullPointerException {

        java.util.logging.Logger log = getJDKLogger();

        if (log != null) {

            log.addHandler(h);

        } else {

            throw new NullPointerException("Could not add handler. The jdk logger is null.");
        }
    }

    /**
     * Removes the handler.
     *
     * @param h the handler
     * @exception NullPointerException if the jdk logger is null
     */
    public void removeHandler(java.util.logging.Handler h) throws NullPointerException {

        java.util.logging.Logger log = getJDKLogger();

        if (log != null) {

            log.removeHandler(h);

        } else {

            throw new NullPointerException("Could not remove handler. The jdk logger is null.");
        }
    }

    /**
     * Creates a handler.
     *
     * @exception Exception if the output is not known
     * @exception NullPointerException if the handler is null
     */
    public void addHandler(String handlertyp) throws Exception, NullPointerException {

        java.util.logging.Handler h = null;

        try {
            if (handlertyp.equals(Logger.CONSOLE_OUTPUT)) {

                h = new java.util.logging.ConsoleHandler();
            }

            else if (handlertyp.equals(Logger.FILE_OUTPUT)) {

                h = new java.util.logging.FileHandler( getJDKLoggerFileName() );
            }

            else {

                //handlertyp nicht implemetiert
                System.out.println( "Der Handlertyp " + handlertyp + " ist in der Methode addHandler nicht definiert");
            }

        } catch (Exception e) {

            throw new Exception("Could not create handler.");
        }

        if (h != null) {

            // Set log level for handler specifying which log messages should be handled by it.
            h.setLevel(getLevel());
            getJDKLogger().addHandler(h);

        } else {

            throw new NullPointerException("Could not create handler. The handler is null.");
        }

    }


    /**
     * Creates a consule handler.
     *
     */
    public void addConsuleHandler() throws Exception, NullPointerException {

        addHandler( Logger.CONSOLE_OUTPUT );
    }

    /**
     * Creates a file handler.
     *
     */
    public void addFileHandler( String filename) throws Exception, NullPointerException {

        setLogFileName( filename );
        addHandler( Logger.FILE_OUTPUT );
    }


    /**
     * Destroys the handler.
     *
     * @param h the handler
     * @exception NullPointerException if the handler is null
     */
    public void destroyHandler(java.util.logging.Handler h) throws NullPointerException {

        if (h != null) {

            // Set log level for handler specifying which log messages should be handled by it.
            h.setLevel(Level.OFF);

        } else {

            throw new NullPointerException("Could not destroy handler. The handler is null.");
        }
    }



    //
    // Loggable.
    //

    /**
     * Logs a message with associated log level.
     *
     * @param lev the level
     * @param msg the message
     * @param t the throwable
     */
    public void log(Level lev, String msg) {

        java.util.logging.Logger l = getJDKLogger();

        if (l != null) {

            l.log(lev, msg, (Throwable)null);
        }
    }

    /**
     * Logs a message with associated throwable information.
     *
     * @param classname
     * @param methodname
     * @param msg the message
     * @param t the throwable
     */
    public void logerror(String classname, String methodname, String msg, Throwable t) {

        java.util.logging.Logger l = getJDKLogger();

        if (l != null) {

            l.log(Level.SEVERE, "classname:" + classname + "---methodname:" + methodname + "---messages:" +msg, t);
        }
    }


}
