/*
 * $RCSfile: EntityRelationshipAssembler.java,v $
 *
 * Copyright (c) 1999-2002. Christian Heller. All rights reserved.
 *
 * This software is published under the GPL GNU General Public License.
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * http://www.resmedicinae.org
 * - Information in Medicine -
 */

package org.resmedicinae.application.healthcare.reform.region.assembler;

import org.resmedicinae.application.healthcare.reform.region.model.*;
import org.resmedicinae.application.healthcare.reform.sqlstatement.create.*;
import org.resmedicinae.application.healthcare.reform.sqlstatement.finder.*;
import org.resmedicinae.application.healthcare.reform.sqlstatement.insertupdate.*;
import org.resmedicinae.application.healthcare.reform.sqlstatement.remove.*;
import org.resmedicinae.application.healthcare.reform.xml.*;
import org.resmedicinae.domain.healthcare.*;
import org.resmedicinae.domain.healthcare.description.*;
import org.resmedicinae.domain.healthcare.heading.problem.*;
import org.resmedicinae.domain.healthcare.unit.*;
import org.resmedicinae.resmedlib.region.assembler.*;

import java.sql.*;
import java.util.*;

/**
 * This class represents an entity relationship model assembler.
 * It can be used to assemble/disassemble an entity relationship model from/to a domain model.
 * @version $Revision: 1.11 $ $Date: 2002/12/17 22:14:39 $ $Author: zone3 $
 * @author Torsten Kunze <zone3@gmx.de>
 * @author Christian Heller <christian.heller@tuxtax.de>
 */
public class EntityRelationshipAssembler extends PersistenceAssembler {
    public static final String ASSEMBLER_KEY = "CONCRETE_ER_ASSEMBLER_KEY";
    private HealthRecordViewModel model;
    private EntityRelationshipModel erModel;

    /**
     *The current healthRecord. This object represents the healthRecord currently in the system.
     * this healthRecord can be used by the application for adding new problems.
     */
    private HealthRecord currentPatient;

    public void initialize() throws Exception {
        try {
            super.initialize();
            /*
              setFinder(createFinder());
              ArrayList arrayList;
              if ((arrayList = getFinder().existTableStructure()).size() > 0) {
                  getFinder().createTableStructure(arrayList);

                  }
            */
        } catch (Exception e) {
            throw new Exception(e);
        }
    }

    public void openConnection(org.resmedicinae.resmedlib.term.String name, org.resmedicinae.resmedlib.term.String pwd) throws Exception {
        try {
            getERModel().setConnection(getERModel().createConnection(name, pwd));
            getERModel().getConnection().setTransactionIsolation(Connection.TRANSACTION_SERIALIZABLE);
            DriverManager.setLoginTimeout(10); // sets the maximum time for connection creation to ten seconds
            checkAndCreateTableStructure();
        } catch (SQLException e) {
            throw new Exception(e.getMessage());
        }
    }

    public void checkAndCreateTableStructure() throws Exception {
        try {
            CreateTableHealthRecords createTableHealthRecords = new CreateTableHealthRecords();
            createTableHealthRecords.initialize();
            createTableHealthRecords.setConnection(getERModel().getConnection());
            if (!createTableHealthRecords.existTable())
                createTableHealthRecords.createTable();
        } catch (SQLException e) {
            throw new Exception("Couldn't find or create table HealthRecords.\n" + e.getMessage());
        }

        try {
            CreateTableProblems createTableProblems = new CreateTableProblems();
            createTableProblems.initialize();
            createTableProblems.setConnection(getERModel().getConnection());
            if (!createTableProblems.existTable())
                createTableProblems.createTable();
        } catch (SQLException e) {
            throw new Exception("Couldn't find or create table Problems.\n" + e.getMessage());
        }

        try {
            CreateTableEpisodes createTableEpisodes = new CreateTableEpisodes();
            createTableEpisodes.initialize();
            createTableEpisodes.setConnection(getERModel().getConnection());
            if (!createTableEpisodes.existTable())
                createTableEpisodes.createTable();
        } catch (SQLException e) {
            throw new Exception("Couldn't find or create table Episodes.\n" + e.getMessage());
        }

        try {
            CreateTablePartialContacts createTablePartialContacts = new CreateTablePartialContacts();
            createTablePartialContacts.initialize();
            createTablePartialContacts.setConnection(getERModel().getConnection());
            if (!createTablePartialContacts.existTable())
                createTablePartialContacts.createTable();
        } catch (SQLException e) {
            throw new Exception("Couldn't find or create table PartialContacts.\n" + e.getMessage());
        }

        try {
            CreateTableObjectives createTableObjectives = new CreateTableObjectives();
            createTableObjectives.initialize();
            createTableObjectives.setConnection(getERModel().getConnection());
            if (!createTableObjectives.existTable())
                createTableObjectives.createTable();
        } catch (SQLException e) {
            throw new Exception("Couldn't find or create table Objectives.\n" + e.getMessage());
        }

        try {
            CreateTableCaves createTableCaves = new CreateTableCaves();
            createTableCaves.initialize();
            createTableCaves.setConnection(getERModel().getConnection());
            if (!createTableCaves.existTable())
                createTableCaves.createTable();
        } catch (SQLException e) {
            throw new Exception("Couldn't find or create table Caves.\n" + e.getMessage());
        }
    }

    public void finishCurrentAssemblerActions() throws Exception {
        if (getERModel() != null) {
            getERModel().destroyConnection();
        }
    }

    public HealthRecordViewModel getModel() {
        return model;
    }

    public void setModel(HealthRecordViewModel model) {
        this.model = model;
    }

    public void setERModel(EntityRelationshipModel erModel) {
        this.erModel = erModel;
    }

    public EntityRelationshipModel getERModel() {
        return this.erModel;
    }

    public void loadHealthRecordIndex() throws Exception {
        System.out.println("in loadHealthRecordIndex()");
        HealthRecordIndex healthRecordIndex = new HealthRecordIndex();
        try {
            FindHealthRecordIndex finder = new FindHealthRecordIndex();
            finder.initialize();
            finder.setConnection(getERModel().getConnection());
            finder.setSqlParameter(new ArrayList());
            ArrayList arrayList = finder.execute();
            //getFinder().findBySQLStatement(SQLFinder.getSqlFinderID(7), null);
            Hashtable healthRecordData;
            Vector v;
            for (int i = 1; i < arrayList.size(); i++) {         // starts at 1, because the first entry 0 is a Vector of column names, not needed here
                v = (Vector) arrayList.get(i);
                healthRecordData = new Hashtable();
                healthRecordData.put(HealthRecordIndex.ID,
                        new org.resmedicinae.resmedlib.term.String(v.elementAt(0).toString()));
                healthRecordData.put(HealthRecordIndex.NAME,
                        new org.resmedicinae.resmedlib.term.String(v.elementAt(1).toString()));
                healthRecordData.put(HealthRecordIndex.FIRSTNAME,
                        new org.resmedicinae.resmedlib.term.String(v.elementAt(2).toString()));
                healthRecordIndex.addHealthRecord(healthRecordData);
            }
            getModel().setHealthRecordIndex(healthRecordIndex);
        } catch (Exception e) {
            throw new Exception("Couldn't create health record index from data base.\n" + e);
        }
    }

    public void saveHealthRecordIndex(HealthRecordIndex healthRecordIndex) {
        // getFinder().updateBySQLStatement(SQLFind.getSqlFinderID(7), null);
    }

    public void saveAllHealthRecords(LoadedHealthRecords loadedHealthRecords) {
    }

    public void saveHealthRecord(HealthRecord healthRecord) throws Exception {
        System.out.println("saveHealthRecord: " + healthRecord.getIdentifier());
        FindOneHealthRecordByID finder = new FindOneHealthRecordByID();
        try {
            finder.initialize();
            finder.setConnection(getERModel().getConnection());
            long t1, t2;
            t1 = System.currentTimeMillis();
            String st = healthRecord.getIdentifier().getValue();
            st = st.substring(8, st.length());
          //  for (int i = 0; i < 10000; i++) {
                //finder.getConnection().setAutoCommit(false);
                ArrayList arrayListFind = new ArrayList();
         //       System.out.println("Entrag: " + i);
         //       healthRecord.setIdentifier(new org.resmedicinae.resmedlib.term.String(st + "_" + i));
                arrayListFind.add(healthRecord.getIdentifier());
                finder.setSqlParameter(arrayListFind);
                arrayListFind = finder.execute();
                //arrayListFind = getFinder().findBySQLStatement(SQLFinder.getSqlFinderID(8), arrayListFind);
                ArrayList arrayListInsertUpdate = new ArrayList();
                if (arrayListFind.size() <= 1)
                    arrayListInsertUpdate.add(healthRecord.getIdentifier());
                //   arrayListInsertUpdate.add(i+"*"+(new HealthRecordIndex().getNewId()));
                arrayListInsertUpdate.add(healthRecord.getName());
                arrayListInsertUpdate.add(healthRecord.getFirstName());
                arrayListInsertUpdate.add(healthRecord.getBirthday());
                arrayListInsertUpdate.add(healthRecord.getStreet());
                arrayListInsertUpdate.add(healthRecord.getZipCode());
                arrayListInsertUpdate.add(healthRecord.getTown());
                arrayListInsertUpdate.add(healthRecord.getCountry());
                arrayListInsertUpdate.add(healthRecord.getPhoneNumber());
                arrayListInsertUpdate.add(healthRecord.getInsurance());
                arrayListInsertUpdate.add(healthRecord.getInsuranceNumber());
                arrayListInsertUpdate.add(healthRecord.getBloodType());
                arrayListInsertUpdate.add(healthRecord.getHeight());
                arrayListInsertUpdate.add(healthRecord.getWeight());
                arrayListInsertUpdate.add(getTimestamp());
                System.out.println("arrayListFind.size(): " + arrayListFind.size());
                if (arrayListFind.size() > 1)
                    arrayListInsertUpdate.add(healthRecord.getIdentifier());
                if (arrayListFind.size() <= 1) {
                    InsertIntoHealthRecords inserter = new InsertIntoHealthRecords();
                    inserter.initialize();
                    inserter.setConnection(getERModel().getConnection());
                    inserter.setSqlParameter(arrayListInsertUpdate);
                    inserter.execute();

                    //getFinder().insertBySQLStatement(SQLFinder.getSqlInsertIntoTableID(0), arrayListInsertUpdate);
                } else {
                    UpdateAnEntityOfHealthRecords updater = new UpdateAnEntityOfHealthRecords();
                    updater.initialize();
                    updater.setConnection(getERModel().getConnection());
                    updater.setSqlParameter(arrayListInsertUpdate);
                    updater.execute();

                    //getFinder().updateBySQLStatement(SQLFinder.getSqlUpdateID(0), arrayListInsertUpdate);
                }
           // }
            System.out.println("healthRecord.hasProblems(): " + healthRecord.hasProblems());
            if (healthRecord.hasProblems()) {
                Object[] problems = healthRecord.getProblems();
                for (int i = 0; i < problems.length; i++)
                    saveProblem(healthRecord.getIdentifier(), (Problem) problems[i]);
            }
            System.out.println("healthRecord.hasCaves(): " + healthRecord.hasCaves());
            if (healthRecord.hasCaves()) {
                Object[] caves = healthRecord.getCaves();
                for (int i = 0; i < caves.length; i++)
                    saveCave(healthRecord.getIdentifier(), (Cave) caves[i]);
            }
            // finder.getConnection().commit();
            // finder.getConnection().setAutoCommit(true);
            t2 = System.currentTimeMillis();
            System.out.println("needed Time to save healthrecord in database in milliseconds: " + (t2 - t1));
        } catch (SQLException e) {
            if (finder.getConnection() != null) {
                try {
                    finder.getConnection().rollback();
                    throw new Exception("Couldn't save health record into data base completely. Rollback was done.\n" + e.getMessage());
                } catch (SQLException ex) {
                    throw new Exception("Couldn't save health record into data base completely. Rollback failed.\n" + e.getMessage());
                }
            } else
                throw new Exception("Couldn't save health record into data base. No database connection found.\n" + e.getMessage());
        }
    }

    public void saveCave(org.resmedicinae.resmedlib.term.String healthRecordID, Cave cave) throws Exception {
        System.out.println("saveCave: " + healthRecordID.getValue());
        FindOneCaveByID finder = new FindOneCaveByID();
        try {
            finder.initialize();
            finder.setConnection(getERModel().getConnection());
//            finder.getConnection().setAutoCommit(false);
            ArrayList arrayListFind = new ArrayList();
            arrayListFind.add(cave.getIdentifier());
            arrayListFind.add(healthRecordID);
            finder.setSqlParameter(arrayListFind);
            arrayListFind = finder.execute();
            ArrayList arrayListInsertUpdate = new ArrayList();
            if (arrayListFind.size() <= 1) {
                arrayListInsertUpdate.add(cave.getIdentifier());
                arrayListInsertUpdate.add(healthRecordID);
            }
            arrayListInsertUpdate.add(cave.getName());
            arrayListInsertUpdate.add(cave.getDescription());
            arrayListInsertUpdate.add(getTimestamp());
            if (arrayListFind.size() > 1) {
                arrayListInsertUpdate.add(cave.getIdentifier());
                arrayListInsertUpdate.add(healthRecordID);
            }
            if (arrayListFind.size() <= 1) {
                InsertIntoCaves inserter = new InsertIntoCaves();
                inserter.initialize();
                inserter.setConnection(getERModel().getConnection());
                inserter.setSqlParameter(arrayListInsertUpdate);
                inserter.execute();

                //getFinder().insertBySQLStatement(SQLFinder.getSqlInsertIntoTableID(1), arrayListInsertUpdate);
            } else {
                UpdateAnEntityOfCaves updater = new UpdateAnEntityOfCaves();
                updater.initialize();
                updater.setConnection(getERModel().getConnection());
                updater.setSqlParameter(arrayListInsertUpdate);
                updater.execute();

                //getFinder().updateBySQLStatement(SQLFinder.getSqlUpdateID(1), arrayListInsertUpdate);
            }
//            finder.getConnection().commit();
//            finder.getConnection().setAutoCommit(true);
        } catch (SQLException e) {
            if (finder.getConnection() != null) {
                try {
                    finder.getConnection().rollback();
                    throw new SQLException("Couldn't save cave into data base completely. Rollback was done.\n" + e.getMessage());
                } catch (SQLException ex) {
                    throw new SQLException("Couldn't save cave into data base completely. Rollback failed.\n" + e.getMessage());
                }
            } else
                throw new SQLException("Couldn't save cave into data base. No database connection found.\n" + e.getMessage());
        }
    }

    public void saveProblem(org.resmedicinae.resmedlib.term.String healthRecordID, Problem problem) throws Exception {
        System.out.println("saveProblem: " + healthRecordID.getValue());
        FindOneProblemByID finder = new FindOneProblemByID();
        try {
            finder.initialize();
            finder.setConnection(getERModel().getConnection());
//            finder.getConnection().setAutoCommit(false);
            ArrayList arrayListFind = new ArrayList();
            arrayListFind.add(problem.getIdentifier());
            arrayListFind.add(healthRecordID);
            finder.setSqlParameter(arrayListFind);
            arrayListFind = finder.execute();
            //arrayListFind = getFinder().findBySQLStatement(SQLFinder.getSqlFinderID(9), arrayListFind);
            ArrayList arrayListInsertUpdate = new ArrayList();
            if (arrayListFind.size() <= 1) {
                arrayListInsertUpdate.add(problem.getIdentifier());
                arrayListInsertUpdate.add(healthRecordID);
            }
            arrayListInsertUpdate.add(problem.getName());
            arrayListInsertUpdate.add(problem.getAnamnese());
            arrayListInsertUpdate.add(getTimestamp());
            if (arrayListFind.size() > 1) {
                arrayListInsertUpdate.add(problem.getIdentifier());
                arrayListInsertUpdate.add(healthRecordID);
            }
            if (arrayListFind.size() <= 1) {
                InsertIntoProblems inserter = new InsertIntoProblems();
                inserter.initialize();
                inserter.setConnection(getERModel().getConnection());
                inserter.setSqlParameter(arrayListInsertUpdate);
                inserter.execute();

                //getFinder().insertBySQLStatement(SQLFinder.getSqlInsertIntoTableID(1), arrayListInsertUpdate);
            } else {
                UpdateAnEntityOfProblems updater = new UpdateAnEntityOfProblems();
                updater.initialize();
                updater.setConnection(getERModel().getConnection());
                updater.setSqlParameter(arrayListInsertUpdate);
                updater.execute();

                //getFinder().updateBySQLStatement(SQLFinder.getSqlUpdateID(1), arrayListInsertUpdate);
            }
            if (problem.hasEpisodes()) {
                Object[] episodes = problem.getEpisodes();
                for (int i = 0; i < episodes.length; i++)
                    saveEpisode(healthRecordID, problem.getIdentifier(), (Episode) episodes[i]);
            }
//            finder.getConnection().commit();
//            finder.getConnection().setAutoCommit(true);
        } catch (SQLException e) {
            if (finder.getConnection() != null) {
                try {
                    finder.getConnection().rollback();
                    throw new SQLException("Couldn't save problem into data base completely. Rollback was done.\n" + e.getMessage());
                } catch (SQLException ex) {
                    throw new SQLException("Couldn't save problem into data base completely. Rollback failed.\n" + e.getMessage());
                }
            } else
                throw new SQLException("Couldn't save problem into data base. No database connection found.\n" + e.getMessage());
        }
    }

    public void saveEpisode(org.resmedicinae.resmedlib.term.String healthRecordID,
                            org.resmedicinae.resmedlib.term.String problemID, Episode episode) throws Exception {
        System.out.println("saveEpisode: " + healthRecordID.getValue());
        FindOneEpisodeByID finder = new FindOneEpisodeByID();
        try {
            finder.initialize();
            finder.setConnection(getERModel().getConnection());
//           finder.getConnection().setAutoCommit(false);
            ArrayList arrayListFind = new ArrayList();
            arrayListFind.add(episode.getIdentifier());
            arrayListFind.add(problemID);
            arrayListFind.add(healthRecordID);
            //arrayListFind = getFinder().findBySQLStatement(SQLFinder.getSqlFinderID(10), arrayListFind);
            finder.setSqlParameter(arrayListFind);
            arrayListFind = finder.execute();
            ArrayList arrayListInsertUpdate = new ArrayList();
            if (arrayListFind.size() <= 1) {
                arrayListInsertUpdate.add(episode.getIdentifier());
                arrayListInsertUpdate.add(problemID);
                arrayListInsertUpdate.add(healthRecordID);
            }
            arrayListInsertUpdate.add(getTimestamp());

/*     add sommething
            arrayListInsertUpdate.add(...);
*/

            if (arrayListFind.size() > 1) {
                arrayListInsertUpdate.add(episode.getIdentifier());
                arrayListInsertUpdate.add(problemID);
                arrayListInsertUpdate.add(healthRecordID);
            }
            if (arrayListFind.size() <= 1) {
                InsertIntoEpisodes inserter = new InsertIntoEpisodes();
                inserter.initialize();
                inserter.setConnection(getERModel().getConnection());
                inserter.setSqlParameter(arrayListInsertUpdate);
                inserter.execute();

                //getFinder().insertBySQLStatement(SQLFinder.getSqlInsertIntoTableID(2), arrayListInsertUpdate);
            } else {
                UpdateAnEntityOfEpisodes updater = new UpdateAnEntityOfEpisodes();
                updater.initialize();
                updater.setConnection(getERModel().getConnection());
                updater.setSqlParameter(arrayListInsertUpdate);
                updater.execute();

                //getFinder().updateBySQLStatement(SQLFinder.getSqlUpdateID(2), arrayListInsertUpdate);
            }
            if (episode.hasPartialContacts()) {
                Object[] partialContacts = episode.getPartialContacts();
                for (int i = 0; i < partialContacts.length; i++)
                    savePartialContact(healthRecordID, problemID, episode.getIdentifier(), (PartialContact) partialContacts[i]);
            }
//            finder.getConnection().commit();
//            finder.getConnection().setAutoCommit(true);
        } catch (SQLException e) {
            if (finder.getConnection() != null) {
                try {
                    finder.getConnection().rollback();
                    throw new SQLException("Couldn't save episode into data base completely. Rollback was done.\n" + e.getMessage());
                } catch (SQLException ex) {
                    throw new SQLException("Couldn't save episode into data base completely. Rollback failed.\n" + e.getMessage());
                }
            } else
                throw new SQLException("Couldn't save episode into data base. No database connection found.\n" + e.getMessage());
        }
    }

    public void savePartialContact(org.resmedicinae.resmedlib.term.String healthRecordID,
                                   org.resmedicinae.resmedlib.term.String problemID, org.resmedicinae.resmedlib.term.String episodeID,
                                   PartialContact partialContact) throws Exception {
        System.out.println("savePartialContact: " + healthRecordID.getValue());
        FindOnePartialContactByID finder = new FindOnePartialContactByID();
        try {
            finder.initialize();
            finder.setConnection(getERModel().getConnection());
//            finder.getConnection().setAutoCommit(false);
            ArrayList arrayListFind = new ArrayList();
            arrayListFind.add(partialContact.getIdentifier());
            arrayListFind.add(episodeID);
            arrayListFind.add(problemID);
            arrayListFind.add(healthRecordID);
            finder.setSqlParameter(arrayListFind);
            arrayListFind = finder.execute();
            //arrayListFind = getFinder().findBySQLStatement(SQLFinder.getSqlFinderID(11), arrayListFind);
            ArrayList arrayListInsertUpdate = new ArrayList();
            if (arrayListFind.size() <= 1) {
                arrayListInsertUpdate.add(partialContact.getIdentifier());
                arrayListInsertUpdate.add(episodeID);
                arrayListInsertUpdate.add(problemID);
                arrayListInsertUpdate.add(healthRecordID);
            }
            arrayListInsertUpdate.add(partialContact.getPartialContactDate());
            System.out.println(">>>>>>>>>>>save subjective: "+((Subjective)partialContact.get(new org.resmedicinae.resmedlib.term.String("subjective"))).getText().getValue());
            System.out.println(">>>>>>>>>>>save assessment: "+((Assessment)partialContact.get(new org.resmedicinae.resmedlib.term.String("assessment"))).getText().getValue());
            System.out.println(">>>>>>>>>>>save plan: "+((Plan)partialContact.get(new org.resmedicinae.resmedlib.term.String("plan"))).getText().getValue());
            arrayListInsertUpdate.add(((Subjective)partialContact.get(new org.resmedicinae.resmedlib.term.String("subjective"))).getText());
            arrayListInsertUpdate.add(((Assessment)partialContact.get(new org.resmedicinae.resmedlib.term.String("assessment"))).getText());
            arrayListInsertUpdate.add(((Plan)partialContact.get(new org.resmedicinae.resmedlib.term.String("plan"))).getText());
            arrayListInsertUpdate.add(getTimestamp());
            if (arrayListFind.size() > 1) {
                arrayListInsertUpdate.add(partialContact.getIdentifier());
                arrayListInsertUpdate.add(episodeID);
                arrayListInsertUpdate.add(problemID);
                arrayListInsertUpdate.add(healthRecordID);
            }
            if (arrayListFind.size() <= 1) {
                InsertIntoPartialContacts inserter = new InsertIntoPartialContacts();
                inserter.initialize();
                inserter.setConnection(getERModel().getConnection());
                inserter.setSqlParameter(arrayListInsertUpdate);
                inserter.execute();

                //getFinder().insertBySQLStatement(SQLFinder.getSqlInsertIntoTableID(3), arrayListInsertUpdate);
            } else {
                UpdateAnEntityOfPartialContacts updater = new UpdateAnEntityOfPartialContacts();
                updater.initialize();
                updater.setConnection(getERModel().getConnection());
                updater.setSqlParameter(arrayListInsertUpdate);
                updater.execute();

                //getFinder().updateBySQLStatement(SQLFinder.getSqlUpdateID(3), arrayListInsertUpdate);
            }
            if (partialContact.hasObjectives()) {
                Object[] objectives = partialContact.getObjectives();
                for (int i = 0; i < objectives.length; i++)
                    saveObjective(healthRecordID, problemID, episodeID, partialContact.getIdentifier(), (Objective) objectives[i]);
            }

//            saveSOAP(healthRecordID, problemID, episodeID, partialContact);
//            finder.getConnection().commit();
//            finder.getConnection().setAutoCommit(true);
        } catch (Exception e) {
            if (finder.getConnection() != null) {
                try {
                    finder.getConnection().rollback();
                    throw new SQLException("Couldn't save partial contact into data base completely. Rollback was done.\n" + e.getMessage());
                } catch (SQLException ex) {
                    throw new SQLException("Couldn't save partial contact into data base completely. Rollback failed.\n" + e.getMessage());
                }
            } else
                throw new SQLException("Couldn't save partial contact into data base. No database connection found.\n" + e.getMessage());
        }
    }
/*
    public void saveSOAP(org.resmedicinae.resmedlib.term.String healthRecordID,
                         org.resmedicinae.resmedlib.term.String problemID, org.resmedicinae.resmedlib.term.String episodeID, PartialContact partialContact) throws Exception {
        try {
            FindOneSOAPByID finder = new FindOneSOAPByID();
            finder.initialize();
            finder.setConnection(getERModel().getConnection());
            ArrayList arrayListFind = new ArrayList();
            arrayListFind.add(new org.resmedicinae.resmedlib.term.String(partialContact.getIdentifier().getValue() + "_0"));
            arrayListFind.add(partialContact.getIdentifier());
            finder.setSqlParameter(arrayListFind);
            arrayListFind = finder.execute();
            //arrayListFind = getFinder().findBySQLStatement(SQLFinder.getSqlFinderID(12), arrayListFind);

            ArrayList arrayListInsertUpdate = new ArrayList();
            if (arrayListFind.size() <= 1) {
                arrayListInsertUpdate.add(new org.resmedicinae.resmedlib.term.String(partialContact.getIdentifier().getValue() + "_0"));
                arrayListInsertUpdate.add(partialContact.getIdentifier());
                arrayListInsertUpdate.add(episodeID);
                arrayListInsertUpdate.add(problemID);
                arrayListInsertUpdate.add(healthRecordID);
            }
            if (partialContact.hasSubjectives()) {
                arrayListInsertUpdate.add(((Subjective) partialContact.getSubjectives()[0]).getText());
            } else
                arrayListInsertUpdate.add(new org.resmedicinae.resmedlib.term.String(null));
            if (partialContact.hasObjectives()) {
                Object[] objectives = partialContact.getObjectives();
                for (int i = 0; i < objectives.length; i++)
                    saveObjective(healthRecordID, problemID, episodeID, partialContact.getIdentifier(), (Objective) objectives[i]);
                arrayListInsertUpdate.add(new org.resmedicinae.resmedlib.term.String("exist, look at table <<objectives>>"));
            } else
                arrayListInsertUpdate.add(new org.resmedicinae.resmedlib.term.String(null));
            if (partialContact.hasAssessments())
                arrayListInsertUpdate.add(((Assessment) partialContact.getAssessments()[0]).getText());
            else
                arrayListInsertUpdate.add(new org.resmedicinae.resmedlib.term.String(null));
            if (partialContact.hasPlans())
                arrayListInsertUpdate.add(((Plan) partialContact.getPlans()[0]).getText());
            else
                arrayListInsertUpdate.add(new org.resmedicinae.resmedlib.term.String(null));
            arrayListInsertUpdate.add(getTimestamp());
            if (arrayListFind.size() > 1) {
                arrayListInsertUpdate.add(new org.resmedicinae.resmedlib.term.String(partialContact.getIdentifier().getValue() + "_0"));
                arrayListInsertUpdate.add(partialContact.getIdentifier());
                arrayListInsertUpdate.add(episodeID);
                arrayListInsertUpdate.add(problemID);
                arrayListInsertUpdate.add(healthRecordID);
            }
//            getFinder().getConnection().setAutoCommit(false);
            if (arrayListFind.size() <= 1) {
                InsertIntoHealthRecords inserter = new InsertIntoHealthRecords();
                inserter.initialize();
                inserter.setConnection(getERModel().getConnection());
                inserter.setSqlParameter(arrayListInsertUpdate);
                inserter.execute();

                //getFinder().insertBySQLStatement(SQLFinder.getSqlInsertIntoTableID(4), arrayListInsertUpdate);
            } else {
                UpdateAnEntityOfHealthRecords updater = new UpdateAnEntityOfHealthRecords();
                updater.initialize();
                updater.setConnection(getERModel().getConnection());
                updater.setSqlParameter(arrayListInsertUpdate);

                //getFinder().updateBySQLStatement(SQLFinder.getSqlUpdateID(4), arrayListInsertUpdate);
            }
//            finder.getConnection().commit();
//            finder.getConnection().setAutoCommit(true);
//            finder.getConnection().setAutoCommit(true);
        } catch (Exception e) {
            if (finder.getConnection() != null) {
                try {
                    finder.getConnection().rollback();
                    throw new SQLException("Couldn't save health record into data base completely. Rollback was done.\n" + e.getMessage());
                } catch (SQLException ex) {
                    throw new SQLException("Couldn't save health record into data base completely. Rollback failed.\n" + e.getMessage());
                }
            } else
                throw new SQLException("Couldn't save health record into data base. No database connection found.\n" + e.getMessage());
        }
    }
*/

    public void saveObjective(org.resmedicinae.resmedlib.term.String healthRecordID,
                              org.resmedicinae.resmedlib.term.String problemID, org.resmedicinae.resmedlib.term.String episodeID,
                              org.resmedicinae.resmedlib.term.String partialContactID, Objective objective) throws Exception {
        System.out.println("saveObjective: " + healthRecordID.getValue());
        FindOneObjectiveByID finder = new FindOneObjectiveByID();
        try {
            finder.initialize();
            finder.setConnection(getERModel().getConnection());
            ArrayList arrayListFind = new ArrayList();
            arrayListFind.add(objective.getIdentifier());
            arrayListFind.add(partialContactID);
            arrayListFind.add(episodeID);
            arrayListFind.add(problemID);
            arrayListFind.add(healthRecordID);
            finder.setSqlParameter(arrayListFind);
            arrayListFind = finder.execute();
            //arrayListFind = getFinder().findBySQLStatement(SQLFinder.getSqlFinderID(21), arrayListFind);
            ArrayList arrayListInsertUpdate = new ArrayList();
            if (arrayListFind.size() <= 1) {
                arrayListInsertUpdate.add(objective.getIdentifier());
                arrayListInsertUpdate.add(partialContactID);
                arrayListInsertUpdate.add(episodeID);
                arrayListInsertUpdate.add(problemID);
                arrayListInsertUpdate.add(healthRecordID);
            }
            arrayListInsertUpdate.add(objective.getText());
            arrayListInsertUpdate.add(getTimestamp());

            if (arrayListFind.size() > 1) {
                arrayListInsertUpdate.add(objective.getIdentifier());
                arrayListInsertUpdate.add(partialContactID);
                arrayListInsertUpdate.add(episodeID);
                arrayListInsertUpdate.add(problemID);
                arrayListInsertUpdate.add(healthRecordID);
            }
//            getFinder().getConnection().setAutoCommit(false);
            if (arrayListFind.size() <= 1) {
                InsertIntoObjectives inserter = new InsertIntoObjectives();
                inserter.initialize();
                inserter.setConnection(getERModel().getConnection());
                inserter.setSqlParameter(arrayListInsertUpdate);
                inserter.execute();

                //getFinder().insertBySQLStatement(SQLFinder.getSqlInsertIntoTableID(5), arrayListInsertUpdate);
            } else {
                UpdateAnEntityOfObjectives updater = new UpdateAnEntityOfObjectives();
                updater.initialize();
                updater.setConnection(getERModel().getConnection());
                updater.setSqlParameter(arrayListInsertUpdate);
                updater.execute();

                //getFinder().updateBySQLStatement(SQLFinder.getSqlUpdateID(9), arrayListInsertUpdate);
            }
//            getFinder().getConnection().commit();
//            getFinder().getConnection().setAutoCommit(true);
        } catch (Exception e) {
            if (finder.getConnection() != null) {
                try {
                    finder.getConnection().rollback();
                    throw new SQLException("Couldn't save objective into data base completely. Rollback was done.\n" + e.getMessage());
                } catch (SQLException ex) {
                    throw new SQLException("Couldn't save objective into data base completely. Rollback failed.\n" + e.getMessage());
                }
            } else
                throw new SQLException("Couldn't save objective into data base. No database connection found.\n" + e.getMessage());
        }
    }

    public HealthRecord loadHealthRecord(org.resmedicinae.resmedlib.term.String healthRecordID) throws Exception {
        System.out.println(">>>>>>>>>>>>>>loadHealthRecord");
        FindOneHealthRecordByID finder = new FindOneHealthRecordByID();
        try {
            finder.initialize();
            finder.setConnection(getERModel().getConnection());
            HealthRecord healthRecord = new HealthRecord();
            long t1, t2;
            ArrayList arrayListFind = new ArrayList();
            arrayListFind.add(healthRecordID);
            t1 = System.currentTimeMillis();
            finder.setSqlParameter(arrayListFind);
            arrayListFind = finder.execute();
            //arrayListFind = getFinder().findBySQLStatement(SQLFinder.getSqlFinderID(8), arrayListFind);
            t2 = System.currentTimeMillis();
            Vector[] v = new Vector[arrayListFind.size()];
            for (int i = 0; i < arrayListFind.size(); i++) {
                v[i] = (Vector) arrayListFind.get(i);
                for (int o = 0; o < v[i].size(); o++)
                    System.out.println(i + ": " + o + ": " + v[i].elementAt(o));
            }
            healthRecord.setIdentifier(new org.resmedicinae.resmedlib.term.String(v[1].elementAt(v[0].indexOf("hrid")).toString()));
            healthRecord.setName(new org.resmedicinae.resmedlib.term.String(v[1].elementAt(v[0].indexOf("surname")).toString()));
            healthRecord.setFirstName(new org.resmedicinae.resmedlib.term.String(v[1].elementAt(v[0].indexOf("firstname")).toString()));
            healthRecord.setBirthday(new org.resmedicinae.resmedlib.term.String(v[1].elementAt(v[0].indexOf("birthday")).toString()));
            healthRecord.setStreet(new org.resmedicinae.resmedlib.term.String(v[1].elementAt(v[0].indexOf("street")).toString()));
            healthRecord.setZipCode(new org.resmedicinae.resmedlib.term.String(v[1].elementAt(v[0].indexOf("zip")).toString()));
            healthRecord.setTown(new org.resmedicinae.resmedlib.term.String(v[1].elementAt(v[0].indexOf("town")).toString()));
            healthRecord.setCountry(new org.resmedicinae.resmedlib.term.String(v[1].elementAt(v[0].indexOf("country")).toString()));
            healthRecord.setPhoneNumber(new org.resmedicinae.resmedlib.term.String(v[1].elementAt(v[0].indexOf("phoneno")).toString()));
            healthRecord.setInsurance(new org.resmedicinae.resmedlib.term.String(v[1].elementAt(v[0].indexOf("insurance")).toString()));
            healthRecord.setInsuranceNumber(new org.resmedicinae.resmedlib.term.String(v[1].elementAt(v[0].indexOf("insurancenumber")).toString()));
            healthRecord.setBloodType(new org.resmedicinae.resmedlib.term.String(v[1].elementAt(v[0].indexOf("bloodtype")).toString()));
            healthRecord.setHeight(new org.resmedicinae.resmedlib.term.String(v[1].elementAt(v[0].indexOf("height")).toString()));
            healthRecord.setWeight(new org.resmedicinae.resmedlib.term.String(v[1].elementAt(v[0].indexOf("weight")).toString()));
            healthRecord.setTimestamp(new org.resmedicinae.resmedlib.term.String(v[1].elementAt(v[0].indexOf("timestamp")).toString()));
            loadCaves(healthRecord);
            loadProblems(healthRecord);
            System.out.println("needed Time in milliseconds: " + (t2 - t1));
            return healthRecord;
        } catch (Exception e) {
            if (finder.getConnection() != null)
                throw new SQLException("Couldn't load health record from data base.\n" + e.getMessage());
            else
                throw new SQLException("Couldn't load health record from data base. No database connection found.\n" + e.getMessage());
        }
    }

    public void loadCaves(HealthRecord healthRecord) throws Exception {
        System.out.println(">>>>>>>>>>>>>>loadCaves");
        FindAllCavesForOneHealthRecord finder = new FindAllCavesForOneHealthRecord();
        try {
            finder.initialize();
            finder.setConnection(getERModel().getConnection());
            Cave cave;
            ArrayList arrayListFind = new ArrayList();
            arrayListFind.add(healthRecord.getIdentifier());
            finder.setSqlParameter(arrayListFind);
            arrayListFind = finder.execute();
            //arrayListFind = getFinder().findBySQLStatement(SQLFinder.getSqlFinderID(18), arrayListFind);
            Vector[] v = new Vector[arrayListFind.size()];
            for (int i = 0; i < arrayListFind.size(); i++) {
                v[i] = (Vector) arrayListFind.get(i);
                for (int o = 0; o < v[i].size(); o++)
                    System.out.println(i + ": " + o + ": " + v[i].elementAt(o));
            }
            for (int i = 1; i < v.length; i++) {
                cave = new Cave();
                cave.setIdentifier(new org.resmedicinae.resmedlib.term.String(v[i].elementAt(v[0].indexOf("caveid")).toString()));
                cave.setName(new org.resmedicinae.resmedlib.term.String(v[i].elementAt(v[0].indexOf("cavename")).toString()));
                cave.setDescription(new org.resmedicinae.resmedlib.term.String(v[i].elementAt(v[0].indexOf("description")).toString()));
                cave.setTimestamp(new org.resmedicinae.resmedlib.term.String(v[i].elementAt(v[0].indexOf("timestamp")).toString()));
                healthRecord.set(cave.getIdentifier(), cave);
            }
        } catch (Exception e) {
            if (finder.getConnection() != null)
                throw new SQLException("Couldn't load problem from data base.\n" + e.getMessage());
            else
                throw new SQLException("Couldn't remove problem from data base. No database connection found.\n" + e.getMessage());
        }
    }

    public void loadProblems(HealthRecord healthRecord) throws Exception {
        System.out.println(">>>>>>>>>>>>>>loadProblems");
        FindAllProblemsForOneHealthRecord finder = new FindAllProblemsForOneHealthRecord();
        try {
            System.out.println("A");
            finder.initialize();
            finder.setConnection(getERModel().getConnection());
            Problem problem;
            ArrayList arrayListFind = new ArrayList();
            arrayListFind.add(healthRecord.getIdentifier());
            finder.setSqlParameter(arrayListFind);
            arrayListFind = finder.execute();
            System.out.println("B");
            //arrayListFind = getFinder().findBySQLStatement(SQLFinder.getSqlFinderID(18), arrayListFind);
            Vector[] v = new Vector[arrayListFind.size()];
            for (int i = 0; i < arrayListFind.size(); i++) {
                v[i] = (Vector) arrayListFind.get(i);
                for (int o = 0; o < v[i].size(); o++)
                    System.out.println(i + ": " + o + ": " + v[i].elementAt(o));
            }
            System.out.println("C");
            for (int i = 1; i < v.length; i++) {
                problem = new Problem();
                problem.setIdentifier(new org.resmedicinae.resmedlib.term.String(v[i].elementAt(v[0].indexOf("problemid")).toString()));
                problem.setName(new org.resmedicinae.resmedlib.term.String(v[i].elementAt(v[0].indexOf("problemname")).toString()));
                problem.setAnamnese(new org.resmedicinae.resmedlib.term.String(v[i].elementAt(v[0].indexOf("anamnese")).toString()));
                problem.setTimestamp(new org.resmedicinae.resmedlib.term.String(v[i].elementAt(v[0].indexOf("timestamp")).toString()));
                loadEpisodes(problem);
                healthRecord.set(problem.getIdentifier(), problem);
            }
        } catch (Exception e) {
            if (finder.getConnection() != null)
                throw new SQLException("Couldn't load problem from data base.\n" + e.getMessage());
            else
                throw new SQLException("Couldn't remove problem from data base. No database connection found.\n" + e.getMessage());
        }
    }

    public void loadEpisodes(Problem problem) throws Exception {
        System.out.println(">>>>>>>>>>>>>>loadEpisodes");
        FindAllEpisodesForOneProblem finder = new FindAllEpisodesForOneProblem();
        try {
            finder.initialize();
            finder.setConnection(getERModel().getConnection());
            Episode episode;
            ArrayList arrayListFind = new ArrayList();
            arrayListFind.add(problem.getIdentifier());
            finder.setSqlParameter(arrayListFind);
            arrayListFind = finder.execute();
            //arrayListFind = getFinder().findBySQLStatement(SQLFinder.getSqlFinderID(19), arrayListFind);
            Vector[] v = new Vector[arrayListFind.size()];
            for (int i = 0; i < arrayListFind.size(); i++) {
                v[i] = (Vector) arrayListFind.get(i);
                for (int o = 0; o < v[i].size(); o++)
                    System.out.println(i + ": " + o + ": " + v[i].elementAt(o));
            }
            for (int i = 1; i < v.length; i++) {
                episode = new Episode();
                episode.setIdentifier(new org.resmedicinae.resmedlib.term.String(v[i].elementAt(v[0].indexOf("episodeid")).toString()));
                episode.setTimestamp(new org.resmedicinae.resmedlib.term.String(v[i].elementAt(v[0].indexOf("timestamp")).toString()));
                loadPartialContacts(episode);
                problem.set(episode.getIdentifier(), episode);
            }
        } catch (Exception e) {
            if (finder.getConnection() != null)
                throw new SQLException("Couldn't load episode from data base.\n" + e.getMessage());
            else
                throw new SQLException("Couldn't remove episode from data base. No database connection found.\n" + e.getMessage());
        }
    }

    public void loadPartialContacts(Episode episode) throws Exception {
        System.out.println(">>>>>>>>>>>>>>loadPartialContacts");
        FindAllPartialContactsForOneEpisode finder = new FindAllPartialContactsForOneEpisode();
        try {
            finder.initialize();
            finder.setConnection(getERModel().getConnection());
            PartialContact partialContact;
            ArrayList arrayListFind = new ArrayList();
            arrayListFind.add(episode.getIdentifier());
            finder.setSqlParameter(arrayListFind);
            arrayListFind = finder.execute();
            //arrayListFind = getFinder().findBySQLStatement(SQLFinder.getSqlFinderID(20), arrayListFind);
            Vector[] v = new Vector[arrayListFind.size()];
            for (int i = 0; i < arrayListFind.size(); i++) {
                v[i] = (Vector) arrayListFind.get(i);
                for (int o = 0; o < v[i].size(); o++)
                    System.out.println(i + ": " + o + ": " + v[i].elementAt(o));
            }
            for (int i = 1; i < v.length; i++) {
                partialContact = new PartialContact();
                partialContact.setIdentifier(new org.resmedicinae.resmedlib.term.String(v[i].elementAt(v[0].indexOf("partialcontactid")).toString()));
                partialContact.setPartialContactDate(new org.resmedicinae.resmedlib.term.String(v[i].elementAt(v[0].indexOf("date")).toString()));
                Subjective subjective = new Subjective(v[i].elementAt(v[0].indexOf("subjective")).toString());
                Assessment assessment = new Assessment(v[i].elementAt(v[0].indexOf("assessment")).toString());
                Plan plan = new Plan(v[i].elementAt(v[0].indexOf("plan")).toString());
                partialContact.set(new org.resmedicinae.resmedlib.term.String("subjective"), subjective);
                partialContact.set(new org.resmedicinae.resmedlib.term.String("assessment"), assessment);
                partialContact.set(new org.resmedicinae.resmedlib.term.String("plan"), plan);
                partialContact.setTimestamp(new org.resmedicinae.resmedlib.term.String(v[i].elementAt(v[0].indexOf("timestamp")).toString()));
                loadObjectives(partialContact);
                //loadSOAP(partialContact);
                episode.set(partialContact.getIdentifier(), partialContact);
            }
        } catch (Exception e) {
            if (finder.getConnection() != null)
                throw new SQLException("Couldn't load partial contact from data base.\n" + e.getMessage());
            else
                throw new SQLException("Couldn't remove partial contact from data base. No database connection found.\n" + e.getMessage());
        }
    }
/*
    public void loadSOAP(PartialContact partialContact) {
        System.out.println(">>>>>>>>>>>>>>loadSOAP");
        FindOneSOAPByID finder = new FindOneSOAPByID();
        finder.initialize();
        finder.setConnection(getERModel().getConnection());
        ArrayList arrayListFind = new ArrayList();
        arrayListFind.add(partialContact.getIdentifier());
        finder.setSqlParameter(arrayListFind);
        arrayListFind = finder.execute();
        //arrayListFind = getFinder().findBySQLStatement(SQLFinder.getSqlFinderID(12), arrayListFind);
        Vector[] v = new Vector[arrayListFind.size()];
        for (int i = 0; i < arrayListFind.size(); i++) {
            v[i] = (Vector) arrayListFind.get(i);
            for (int o = 0; o < v[i].size(); o++)
                System.out.println(i + ": " + o + ": " + v[i].elementAt(o));
        }
        for (int i = 1; i < v.length; i++) {
            Subjective subjective = new Subjective(v[i].elementAt(v[0].indexOf("subjective")).toString());
            Assessment assessment = new Assessment(v[i].elementAt(v[0].indexOf("assessment")).toString());
            Plan plan = new Plan(v[i].elementAt(v[0].indexOf("plan")).toString());
            partialContact.set(new org.resmedicinae.resmedlib.term.String("1"), subjective);
            partialContact.set(new org.resmedicinae.resmedlib.term.String("3"), assessment);
            partialContact.set(new org.resmedicinae.resmedlib.term.String("4"), plan);
        }
        loadObjectives(partialContact);
    }
*/
    public void loadObjectives(PartialContact partialContact) throws Exception {
        System.out.println(">>>>>>>>>>>>>>loadObjectives");
        FindAllObjectivesForOnePartialContact finder = new FindAllObjectivesForOnePartialContact();
        try {
            finder.initialize();
            finder.setConnection(getERModel().getConnection());
            Objective objective;
            ArrayList arrayListFind = new ArrayList();
            arrayListFind.add(partialContact.getIdentifier());
            finder.setSqlParameter(arrayListFind);
            arrayListFind = finder.execute();
            //arrayListFind = getFinder().findBySQLStatement(SQLFinder.getSqlFinderID(21), arrayListFind);
            Vector[] v = new Vector[arrayListFind.size()];
            for (int i = 0; i < arrayListFind.size(); i++) {
                v[i] = (Vector) arrayListFind.get(i);
                for (int o = 0; o < v[i].size(); o++)
                    System.out.println(i + ": " + o + ": " + v[i].elementAt(o));
            }
            for (int i = 1; i < v.length; i++) {
                objective = new Objective();
                objective.setIdentifier(new org.resmedicinae.resmedlib.term.String(v[i].elementAt(v[0].indexOf("objectiveid")).toString()));
                objective.setText(new org.resmedicinae.resmedlib.term.String(v[i].elementAt(v[0].indexOf("text")).toString()));
                objective.setTimestamp(new org.resmedicinae.resmedlib.term.String(v[i].elementAt(v[0].indexOf("timestamp")).toString()));
                partialContact.set(objective.getIdentifier(), objective);
            }
        } catch (Exception e) {
            if (finder.getConnection() != null)
                throw new SQLException("Couldn't load objective from data base.\n" + e.getMessage());
            else
                throw new SQLException("Couldn't remove objective from data base. No database connection found.\n" + e.getMessage());
        }
    }

    public void removeHealthRecord(org.resmedicinae.resmedlib.term.String healthRecordId) throws Exception {
        System.out.println("removeHealthRecord: " + healthRecordId);
        DeleteAnEntityFromHealthRecords deleter = new DeleteAnEntityFromHealthRecords();
        try {
            deleter.initialize();
            deleter.setConnection(getERModel().getConnection());
            long t1, t2;
            t1 = System.currentTimeMillis();
            ArrayList arrayList = new ArrayList();
            arrayList.add(healthRecordId);
            deleter.getConnection().setAutoCommit(false);
            deleter.setSqlParameter(arrayList);
            deleter.execute();
            //getFinder().updateBySQLStatement(SQLFinder.getSqlDeleteDataID(0), arrayList);
            deleter.getConnection().commit();
            deleter.getConnection().setAutoCommit(true);
            t2 = System.currentTimeMillis();
            System.out.println("needed Time in milliseconds: " + (t2 - t1));
        } catch (SQLException e) {
            if (deleter.getConnection() != null) {
                try {
                    deleter.getConnection().rollback();
                    throw new SQLException("Couldn't remove health record from data base completely. Rollback was done.\n" + e.getMessage());
                } catch (SQLException ex) {
                    throw new SQLException("Couldn't remove health record from data base completely. Rollback failed.\n" + e.getMessage());
                }
            } else
                throw new SQLException("Couldn't remove health record from data base. No database connection found.\n" + e.getMessage());
        }
    }

    public void removeCave(org.resmedicinae.resmedlib.term.String caveId) throws Exception {
        System.out.println("removeCave: " + caveId);
        DeleteAnEntityFromCaves deleter = new DeleteAnEntityFromCaves();
        try {
            deleter.initialize();
            deleter.setConnection(getERModel().getConnection());
            long t1, t2;
            t1 = System.currentTimeMillis();
            ArrayList arrayList = new ArrayList();
            arrayList.add(caveId);
            deleter.getConnection().setAutoCommit(false);
            deleter.setSqlParameter(arrayList);
            deleter.execute();
            //getFinder().updateBySQLStatement(SQLFinder.getSqlDeleteDataID(0), arrayList);
            deleter.getConnection().commit();
            deleter.getConnection().setAutoCommit(true);
            t2 = System.currentTimeMillis();
            System.out.println("needed Time in milliseconds: " + (t2 - t1));
        } catch (SQLException e) {
            if (deleter.getConnection() != null) {
                try {
                    deleter.getConnection().rollback();
                    throw new SQLException("Couldn't remove cave from data base completely. Rollback was done.\n" + e.getMessage());
                } catch (SQLException ex) {
                    throw new SQLException("Couldn't remove cave from data base completely. Rollback failed.\n" + e.getMessage());
                }
            } else
                throw new SQLException("Couldn't remove health cave data base. No database connection found.\n" + e.getMessage());
        }
    }

    public void removeProblem(org.resmedicinae.resmedlib.term.String problemId) throws Exception {
        System.out.println("removeProblem: " + problemId);
        DeleteAnEntityFromProblems deleter = new DeleteAnEntityFromProblems();
        try {
            deleter.initialize();
            deleter.setConnection(getERModel().getConnection());
            long t1, t2;
            t1 = System.currentTimeMillis();
            ArrayList arrayList = new ArrayList();
            arrayList.add(problemId);
            deleter.getConnection().setAutoCommit(false);
            deleter.setSqlParameter(arrayList);
            deleter.execute();
            //getFinder().updateBySQLStatement(SQLFinder.getSqlDeleteDataID(0), arrayList);
            deleter.getConnection().commit();
            deleter.getConnection().setAutoCommit(true);
            t2 = System.currentTimeMillis();
            System.out.println("needed Time in milliseconds: " + (t2 - t1));
        } catch (SQLException e) {
            if (deleter.getConnection() != null) {
                try {
                    deleter.getConnection().rollback();
                    throw new SQLException("Couldn't remove problem from data base completely. Rollback was done.\n" + e.getMessage());
                } catch (SQLException ex) {
                    throw new SQLException("Couldn't remove problem from data base completely. Rollback failed.\n" + e.getMessage());
                }
            } else
                throw new SQLException("Couldn't remove health problem data base. No database connection found.\n" + e.getMessage());
        }
    }

    public void removeEpisode(org.resmedicinae.resmedlib.term.String episodeId) throws Exception {
        System.out.println("removeEpisode: " + episodeId);
        DeleteAnEntityFromEpisodes deleter = new DeleteAnEntityFromEpisodes();
        try {
            deleter.initialize();
            deleter.setConnection(getERModel().getConnection());
            long t1, t2;
            t1 = System.currentTimeMillis();
            ArrayList arrayList = new ArrayList();
            arrayList.add(episodeId);
            deleter.getConnection().setAutoCommit(false);
            deleter.setSqlParameter(arrayList);
            deleter.execute();
            //getFinder().updateBySQLStatement(SQLFinder.getSqlDeleteDataID(0), arrayList);
            deleter.getConnection().commit();
            deleter.getConnection().setAutoCommit(true);
            t2 = System.currentTimeMillis();
            System.out.println("needed Time in milliseconds: " + (t2 - t1));
        } catch (SQLException e) {
            if (deleter.getConnection() != null) {
                try {
                    deleter.getConnection().rollback();
                    throw new SQLException("Couldn't remove episode from data base completely. Rollback was done.\n" + e.getMessage());
                } catch (SQLException ex) {
                    throw new SQLException("Couldn't remove episode from data base completely. Rollback failed.\n" + e.getMessage());
                }
            } else
                throw new SQLException("Couldn't remove episode from data base. No database connection found.\n" + e.getMessage());
        }
    }

    public void removePartialContact(org.resmedicinae.resmedlib.term.String partialContactId) throws Exception {
        System.out.println("removePartialContact: " + partialContactId);
        DeleteAnEntityFromPartialContacts deleter = new DeleteAnEntityFromPartialContacts();
        try {
            deleter.initialize();
            deleter.setConnection(getERModel().getConnection());
            long t1, t2;
            t1 = System.currentTimeMillis();
            ArrayList arrayList = new ArrayList();
            arrayList.add(partialContactId);
            deleter.getConnection().setAutoCommit(false);
            deleter.setSqlParameter(arrayList);
            deleter.execute();
            //getFinder().updateBySQLStatement(SQLFinder.getSqlDeleteDataID(0), arrayList);
            deleter.getConnection().commit();
            deleter.getConnection().setAutoCommit(true);
            t2 = System.currentTimeMillis();
            System.out.println("needed Time in milliseconds: " + (t2 - t1));
        } catch (SQLException e) {
            if (deleter.getConnection() != null) {
                try {
                    deleter.getConnection().rollback();
                    throw new SQLException("Couldn't remove partial contact from data base completely. Rollback was done.\n" + e.getMessage());
                } catch (SQLException ex) {
                    throw new SQLException("Couldn't remove partial contact from data base completely. Rollback failed.\n" + e.getMessage());
                }
            } else
                throw new SQLException("Couldn't remove partial contact from data base. No database connection found.\n" + e.getMessage());
        }
    }

    public void removeObjective(org.resmedicinae.resmedlib.term.String objectiveId) throws Exception {
        System.out.println("removeObjective: " + objectiveId);
        DeleteAnEntityFromObjectives deleter = new DeleteAnEntityFromObjectives();
        try {
            deleter.initialize();
            deleter.setConnection(getERModel().getConnection());
            long t1, t2;
            t1 = System.currentTimeMillis();
            ArrayList arrayList = new ArrayList();
            arrayList.add(objectiveId);
            deleter.getConnection().setAutoCommit(false);
            deleter.setSqlParameter(arrayList);
            deleter.execute();
            //getFinder().updateBySQLStatement(SQLFinder.getSqlDeleteDataID(0), arrayList);
            deleter.getConnection().commit();
            deleter.getConnection().setAutoCommit(true);
            t2 = System.currentTimeMillis();
            System.out.println("needed Time in milliseconds: " + (t2 - t1));
        } catch (SQLException e) {
            if (deleter.getConnection() != null) {
                try {
                    deleter.getConnection().rollback();
                    throw new SQLException("Couldn't remove objective from data base completely. Rollback was done.\n" + e.getMessage());
                } catch (SQLException ex) {
                    throw new SQLException("Couldn't remove objective from data base completely. Rollback failed.\n" + e.getMessage());
                }
            } else
                throw new SQLException("Couldn't remove objective from data base. No database connection found.\n" + e.getMessage());
        }
    }

    public org.resmedicinae.resmedlib.term.String getTimestamp() {
        long time = System.currentTimeMillis();
        return new org.resmedicinae.resmedlib.term.String(time + (Math.round(((new Timestamp(time)).getNanos()) / 100000)) + "");
    }
}
