/*
 * $RCSfile: PrintSuit.java,v $
 *
 * Copyright (c) 1999-2002. Torsten Kunze. All rights reserved.
 *
 * This software is published under the GPL GNU General Public License.
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * http://www.resmedicinae.org
 * - Information in Medicine -
 */

package org.resmedicinae.application.healthcare.reform.printer;

import javax.swing.*;
import java.awt.*;
import java.awt.geom.*;
import java.awt.image.*;
import java.awt.print.*;

/**
 * This class is usefull for printing a swing or awt componenent that doesn't implement the Printable or Pageable interface.
 * @version $Revision: 1.10 $ $Date: 2002/10/17 15:43:03 $ $Author: zone3 $
 * @author Torsten Kunze <zone3@gmx.de>
 */
public class PrintSuit implements Printable {
    /** Decides that either a print mask will be printed (false) or the whole component like displayed on the screen (true) */
    private static boolean maskOrScreenshot = false;

    /** The component to print */
    private Component compToPrint;

    /** The name of the print job */
    private String jobName = "Resmedicinae: ";

    /**
     * initializes the print operation without print dialogs; calls the constructor
     * @param comp the component to print
     * @param jobName the name of the print job
     */
    public static void printComponent(Component comp, String jobName, boolean maskOrScreenshot) {
        setMaskOrScreenshot(maskOrScreenshot);
        new PrintSuit(comp, jobName).print();
    }

    /**
     * initializes the print operation with print dialogs; calls the constructor
     * @param comp the component to print
     * @param jobName the name of the print job
     */
    public static void askBeforePrintComponent(Component comp, String jobName, boolean maskOrScreenshot) {
        setMaskOrScreenshot(maskOrScreenshot);
        new PrintSuit(comp, jobName).askBeforePrint();
    }

    /**
     * Constructor
     * @param comp the component to print
     * @param jobName the name of the print job
     */
    public PrintSuit(Component comp, String jobName) {
        this.compToPrint = comp;
        this.jobName = this.jobName + jobName;
    }

    /**
     * opens the print dialogs for some configurations and starts the print operation
     * called intern from printComponent(Component comp, String jobName)
     */
    public void print() {
        PrinterJob printerJob = PrinterJob.getPrinterJob();
        PageFormat formating = new PageFormat();
        Paper paper = new Paper();
        paper.setImageableArea(4, 12, 400, 400);
        formating.setOrientation(PageFormat.PORTRAIT);
        formating.setPaper(paper);
        printerJob.setPrintable(this, formating);
        printerJob.setJobName(this.jobName);
        try {
            printerJob.print();
        } catch (PrinterException pex) {
            pex.printStackTrace();
        }
    }

    /**
     * opens the print dialogs for some configurations and starts the print operation
     * called intern from askBeforePrintComponent(Component comp, String jobName)
     */
    public void askBeforePrint() {
        PrinterJob printerJob = PrinterJob.getPrinterJob();
        PageFormat formating = new PageFormat();
        formating = printerJob.pageDialog(formating);
        printerJob.setPrintable(this);
        printerJob.setJobName(this.jobName);
        if (printerJob.printDialog())
            try {
                printerJob.print();
            } catch (PrinterException pex) {
                pex.printStackTrace();
            }
    }

    /**
     * Inherited method from Printable. Prints the page at the specified index into the specified Graphics context
     * in the specified format. A PrinterJob calls the Printable interface to request
     * that a page be rendered into the context specified by graphics. The format of
     * the page to be drawn is specified by pageFormat. The zero based index of the
     * requested page is specified by pageIndex. If the requested page does not exist
     * then this method returns NO_SUCH_PAGE; otherwise PAGE_EXISTS is returned.
     * The Graphics class or subclass implements the PrinterGraphics interface to provide
     * additional information. If the Printable object aborts the print job then it throws a PrinterException.
     * @param graphics the context into which the page is drawn
     * @param pageFormat the size and orientation of the page being drawn
     * @param pageIndex the zero based index of the page to be drawn
     * @return PAGE_EXISTS if the page is rendered successfully or NO_SUCH_PAGE if pageIndex specifies a non-existent page.
     */
    public int print(Graphics graphics, PageFormat pageFormat, int pageIndex) {
        if (pageIndex >= 1) {
            return Printable.NO_SUCH_PAGE;
        } else {
            Graphics2D graphics2d = (Graphics2D) graphics;
            Graphics2D big;
            double scaleX = 1.0;
            double scaleY = 1.0;
            double scaleBoth = 1.0;
            int fontHeight = 10;
            int widthF = compToPrint.getSize().width;
            int heightF = compToPrint.getSize().height;
            int widthP = (int) pageFormat.getImageableWidth() + 1; // Ermitteln des bedruckbaren Bereiches
            int heightP = (int) pageFormat.getImageableHeight() + 1;
            int x = (int) pageFormat.getImageableX() + 1; // nicht bedruckbaren
            int y = (int) pageFormat.getImageableY() + 1; // Rand ermitteln
            graphics2d.translate(x, y); // Koordinatensystem um diesen Rand verschieben
            graphics2d.setFont(new Font("Times New Roman", Font.PLAIN, fontHeight));
            //	graphics2d.setFont(new Font("SansSerif", Font.PLAIN, fontHeight));
            graphics2d.setColor(Color.black);
            Font fn = graphics2d.getFont();
            FontMetrics fm = graphics.getFontMetrics();
            RepaintManager currentManager = RepaintManager.currentManager(compToPrint);
            currentManager.setDoubleBufferingEnabled(false);
            System.out.println("x: " + x);
            System.out.println("y: " + y);
            System.out.println("widthP: " + widthP);
            System.out.println("heightP: " + heightP);
            if (!getMaskOrScreenshot()) {
                //				****
                ((org.resmedicinae.application.healthcare.reform.formular.script.ScriptView) compToPrint).getPrintMask(graphics2d,
                        new Rectangle(x, y + fontHeight + 1, widthP, heightP), fm);
                //	graphics.drawRect(x,y,widthP-1,heightP-1);
                //				****
            } else {
                BufferedImage bufferedImage = (BufferedImage) compToPrint.createImage(widthF, heightF);
                if (bufferedImage != null) {
                    big = bufferedImage.createGraphics();
                    big.setBackground(compToPrint.getBackground());
                    compToPrint.print(big);
                    if (widthF > widthP)
                        scaleX = (double) widthP / widthF;
                    if (heightF > heightP)
                        scaleY = (double) heightP / heightF;
                    if (scaleX > scaleY)
                        scaleBoth = scaleY;
                    else
                        scaleBoth = scaleX;
                    bufferedImage = scale(scaleBoth, scaleBoth, bufferedImage);
                    graphics2d.drawImage(bufferedImage, 0, 0, null);
                }
            }
            //	compToPrint.paint(graphics);
            currentManager.setDoubleBufferingEnabled(true);
            System.gc();
            return Printable.PAGE_EXISTS;
        }
    }

    /**
     * Scales a buffered image by special factors
     * @param scaleX scale factor for the x axis
     * @param scaleY scale factor for the y axis
     * @param srcImg the image to be scaled
     * @return the scaled buffered image
     */
    public final BufferedImage scale(double scaleX, double scaleY, BufferedImage srcImg) {
        if (scaleX == 1 && scaleY == 1) {
            return srcImg;
        }
        AffineTransformOp op = new AffineTransformOp(AffineTransform.getScaleInstance(scaleX, scaleY), null);
        return op.filter(srcImg, null);
    }

    /**
     * Sets the member variable maskOrScreenshot
     * @param maskOrScreenshot a boolean value
     */
    public static boolean getMaskOrScreenshot() {
        return PrintSuit.maskOrScreenshot;
    }

    /**
     * Returns the member variable maskOrScreenshot
     * @return the value of maskOrScreenshot
     */
    public static void setMaskOrScreenshot(boolean maskOrScreenshot) {
        PrintSuit.maskOrScreenshot = maskOrScreenshot;
    }
}
