/*
 * $RCSfile: TreeTablePanel.java,v $
 *
 * Copyright (c) 1999-2002. Jens Bohl. All rights reserved.
 *
 * This software is published under the GPL GNU General Public License.
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * http://www.resmedicinae.org
 * - Information in Medicine -
 */

package org.resmedicinae.application.healthcare.record.treetable;

import org.resmedicinae.application.healthcare.record.*;
import org.resmedicinae.resmedlib.component.control.*;
import org.resmedicinae.resmedlib.component.view.swing.*;

import javax.swing.*;
import javax.swing.table.*;
import java.awt.*;

/**
 * This panel contains the tree table and a buttonPanel for changing this treeTable. TreeNodes can be created and removed.
 * @author Jens Bohl <info@jens-bohl-professional.de>
 */
public class TreeTablePanel extends ResPanel {
    /** The tree table */
    private JTreeTable treeTable;
    /** The object model. This model is the data structure behind the tree table. */
    private AbstractTreeTableModel objectModel;
    /** The controller of the panel. This is Record. */
    private Controller controller;
     /** The panel containing buttons */
    private ResPanel buttonPanel;
     /** Button for creating a new problem */
    private JButton newProblemButton;
     /** Button for creating a new episode */
    private JButton newEpisodeButton;
     /** Button for creating a new partial contact */
    private JButton newPartialContactButton;
     /** Button for removing a problem */
    private JButton removeProblemButton;
     /** Button for removing an episode */
    private JButton removeEpisodeButton;
     /** Button for removing a partial contact */
    private JButton removePartialContactButton;
     /** ComboBox for changing between particular filters */
    private JComboBox filterComboBox;
     /** Filter label */
    private JLabel filterLabel;
     /** Button for creating a new drug entry */
    private JButton newMedicamentButton;
     /** Button for removing a drug entry */
    private JButton removeMedicamentButton;

     /** The default constructor. */
    public TreeTablePanel() throws NullPointerException {
        super();
    }

    /**
     *Sets the object model as a tree table model
     * @param objectModel the object model
     */
    public void setTreeTableModel(AbstractTreeTableModel objectModel) {
        setObjectModel(objectModel);
        ((JTreeTable) getTreeTable()).setModel(getObjectModel());
        getTreeTable().initialize();
        setTextAreaAsRenderer();
    }

    /** Initialization of all components. */
    public void initialize() {
        setTreeTable(new JTreeTable(getController()));
        JScrollPane treeTableScrollPane = new JScrollPane(treeTable);
        treeTableScrollPane.setPreferredSize(new Dimension(850, 400));
        setButtonPanel(createButtonPanel());
        add(getButtonPanel());
        add(treeTableScrollPane);
    }

    /**
     *Initializes an TextArea as CellRenderer and CellEditor. This is needed because
     * a TableCell should contain more than one row.
     */
    private void setTextAreaAsRenderer() {
        TableColumn column1 = treeTable.getColumnModel().getColumn(1);
        TableColumn column2 = treeTable.getColumnModel().getColumn(2);
        TableColumn column3 = treeTable.getColumnModel().getColumn(3);
        TextAreaCellEditor textAreaCellEditor = new TextAreaCellEditor(3, 10);
        column1.setCellEditor(textAreaCellEditor);
        column1.setCellRenderer(new TextAreaCellRenderer());
        column2.setCellEditor(textAreaCellEditor);
        column2.setCellRenderer(new TextAreaCellRenderer());
        column3.setCellEditor(textAreaCellEditor);
        column3.setCellRenderer(new TextAreaCellRenderer());
    }

    /**
     *Returns the TreeTable component.
     * @return tree table
     */
    public JTreeTable getTreeTable() {
        return this.treeTable;
    }

    /**
     *Sets the TreeTable component.
     * @param treeTable the tree table
     */
    public void setTreeTable(JTreeTable treeTable) {
        this.treeTable = treeTable;
    }

    /**
     *Gets the object model which is the data structure behind the tree table.
     * @return the object model
     */
    public AbstractTreeTableModel getObjectModel() {
        return objectModel;
    }

    /**
     *Sets the object model as datas structure behind the treee table
     * @param objectModel the object model
     */
    public void setObjectModel(AbstractTreeTableModel objectModel) {
        this.objectModel = objectModel;
    }

    /**Sets the controller
     * @param controller the controller
     */
    public void setController(Controller controller) {
        this.controller = controller;
    }

    /**Gets the controller
     * @return the controller
     */
    public Controller getController() {
        return this.controller;
    }

    /**Creates the buttonPanel on the left side of this treeTablePanel
     * @return the buttonPanel
     * */
    public ResPanel createButtonPanel() {
        ResPanel bp = new ResPanel();
        if (bp != null) {
            GridLayout gridLayout = new GridLayout(10, 1);
            gridLayout.setVgap(15);
            bp.setLayout(gridLayout);
            bp.setBorder(BorderFactory.createLineBorder(Color.BLACK));
            setFilterLabel(createFilterLabel());
            setFilterComboBox(createFilterComboBox());
            setNewProblemButton(createNewProblemButton());
            setNewEpisodeButton(createNewEpisodeButton());
            setNewPartialContactButton(createNewPartialContactButton());
            setRemoveProblemButton(createRemoveProblemButton());
            setRemoveEpisodeButton(createRemoveEpisodeButton());
            setRemovePartialContactButton(createRemovePartialContactButton());
            setNewMedicamentButton(createNewMedicamentButton());
            setRemoveMedicamentButton(createRemoveMedicamentButton());
            // bp.add(new JLabel(new ImageIcon(RecordModel.RelativeIconDirectoryName+"mensch.jpg")));
            bp.add(getFilterLabel());
            bp.add(getFilterComboBox());
            bp.add(getNewProblemButton());
            bp.add(getNewEpisodeButton());
            bp.add(getNewPartialContactButton());
            bp.add(getNewMedicamentButton());
            bp.add(getRemoveProblemButton());
            bp.add(getRemoveEpisodeButton());
            bp.add(getRemovePartialContactButton());
            bp.add(getRemoveMedicamentButton());
        } else {
            throw new NullPointerException("Could not create button panel. The panel is null.");
        }
        return bp;
    }

    /**Gets the buttonPanel on the left side of this treeTablePanel
     * @return the buttonPanel
     * */
    public ResPanel getButtonPanel() {
        return buttonPanel;
    }

    /**Sets the buttonPanel on the left side of this treeTablePanel
     * @param buttonPanel the buttonPanel
     * */
    public void setButtonPanel(ResPanel buttonPanel) {
        this.buttonPanel = buttonPanel;
    }

    /**Creates the newProblemButton
     * @return the newProblemButton
     */
    public JButton createNewProblemButton() {
        JButton npb = new JButton();
        if (npb != null) {
            npb.setText("New Problem");
            npb.setEnabled(true);
            npb.setActionCommand(Record.NEW_PROBLEM_ID);
            npb.addActionListener((Record) getController());
        } else {
            throw new NullPointerException("Could not create new problem button. The button is null");
        }
        return npb;
    }

    /**Gets the newProblemButton
     * @return the newProblemButton
     */
    public JButton getNewProblemButton() {
        return newProblemButton;
    }

    /**Sets the newProblemButton
     * @return newProblemButton the newProblemButton
     */
    public void setNewProblemButton(JButton newProblemButton) {
        this.newProblemButton = newProblemButton;
    }

    /**Creates the newEpisodeButton
     * @return the newEpisodeButton
     */
    public JButton createNewEpisodeButton() {
        JButton neb = new JButton();
        if (neb != null) {
            neb.setText("New episode");
            neb.setEnabled(false);
            neb.setActionCommand(Record.NEW_EPISODE_ID);
            neb.addActionListener((Record) getController());
        } else {
            throw new NullPointerException("Could not create new episode button. The button is null");
        }
        return neb;
    }

    /**Gets the newEpisodeButton
     * @return the newEpisodeButton
     */
    public JButton getNewEpisodeButton() {
        return newEpisodeButton;
    }

    /**Sets the newProblemButton
     * @return newProblemButton the newProblemButton
     */
    public void setNewEpisodeButton(JButton newEpisodeButton) {
        this.newEpisodeButton = newEpisodeButton;
    }

    /**Creates the removeProblemButton
     * @return the removeProblemButton
     */
    public JButton createRemoveProblemButton() {
        JButton b = new JButton("Remove problem");
        if (b != null) {
            b.setEnabled(false);
            b.setActionCommand(Record.REMOVE_PROBLEM_ID);
            b.addActionListener((Record) getController());
        } else {
            throw new NullPointerException("Could not create Remove Problem Button. The button is null");
        }
        return b;
    }

    /**Gets the removeProblemButton
     * @return the removeProblemButton
     */
    public JButton getRemoveProblemButton() {
        return removeProblemButton;
    }

    /**Sets the removeProblemButton
     * @return removeProblemButton the removeProblemButton
     */
    public void setRemoveProblemButton(JButton removeProblemButton) {
        this.removeProblemButton = removeProblemButton;
    }

    /**Creates the removeEpisodeButton
     * @return the removeEpisodeButton
     */
    public JButton createRemoveEpisodeButton() {
        JButton b = new JButton("Remove episode");
        if (b != null) {
            b.setEnabled(false);
            b.setActionCommand(Record.REMOVE_EPISODE_ID);
            b.addActionListener((Record) getController());
        } else {
            throw new NullPointerException("Could not create Remove Episode Button. The button is null");
        }
        return b;
    }

    /**Gets the removeEpisodeButton
     * @return the removeEpisodeButton
     */
    public JButton getRemoveEpisodeButton() {
        return removeEpisodeButton;
    }

    /**Sets the removeEpisodeButton
     * @return removeEpisodeButton the removeEpisodeButton
     */
    public void setRemoveEpisodeButton(JButton removeEpisodeButton) {
        this.removeEpisodeButton = removeEpisodeButton;
    }

    /**Creates the newPartialContactButton
     * @return the newPartialContactButton
     */
    public JButton createNewPartialContactButton() {
        JButton b = new JButton("New contact");
        if (b != null) {
            b.setEnabled(false);
            b.setActionCommand(Record.NEW_PARTIAL_CONTACT_ID);
            b.addActionListener((Record) getController());
        } else {
            throw new NullPointerException("Could not create New Partial Contact Button. The button is null");
        }
        return b;
    }

    /**Gets the newPartialContactButton
     * @return the newPartialContactButton
     */
    public JButton getNewPartialContactButton() {
        return newPartialContactButton;
    }

     /**Sets the newPartialContactButton
     * @return newPartialContactButton the newPartialContactButton
     */
    public void setNewPartialContactButton(JButton newPartialContactButton) {
        this.newPartialContactButton = newPartialContactButton;
    }

    /**Creates the removePartialContactButton
     * @return the removePartialContactButton
     */
    public JButton createRemovePartialContactButton() {
        JButton b = new JButton("Remove contact");
        if (b != null) {
            b.setEnabled(false);
            b.setActionCommand(Record.REMOVE_PARTIAL_CONTACT_ID);
            b.addActionListener((Record) getController());
        } else {
            throw new NullPointerException("Could not create New Partial Contact Button. The button is null");
        }
        return b;
    }

    /**Gets the removePartialContactButton
     * @return the removePartialContactButton
     */
    public JButton getRemovePartialContactButton() {
        return removePartialContactButton;
    }

     /**Sets the removePartialContactButton
     * @return removePartialContactButton the removePartialContactButton
     */
    public void setRemovePartialContactButton(JButton removePartialContactButton) {
        this.removePartialContactButton = removePartialContactButton;
    }

    /**Creates the filterComboBox
     * @return the filterComboBox
     */
    public JComboBox createFilterComboBox() {
        Object[] items = {"All", "Subjective", "Objective", "Assessment", "Plan"};
        JComboBox b = new JComboBox(items);
        if (b != null) {
            b.setEnabled(true);
            b.setActionCommand(Record.CHANGE_FILTER_ID);
            b.addActionListener((Record) getController());
        } else {
            throw new NullPointerException("Could not create New Partial Contact Button. The button is null");
        }
        return b;
    }

    /**Gets the filterComboBox
     * @return the filterComboBox
     */
    public JComboBox getFilterComboBox() {
        return filterComboBox;
    }

     /**Sets the filterComboBox
     * @return filterComboBox the filterComboBox
     */
    public void setFilterComboBox(JComboBox filterComboBox) {
        this.filterComboBox = filterComboBox;
    }

    /**Creates the filterLabel
     * @return the filterLabel
     */
    public JLabel createFilterLabel() {
        JLabel l = new JLabel();
        if (l != null) {
            l.setText("Filter");
        } else {
            throw new NullPointerException("Could not create filter label. The label is null.");
        }
        return l;
    }

    /**Gets the filterLabel
     * @return the filterLabel
     */
    public JLabel getFilterLabel() {
        return filterLabel;
    }

    /**Sets the filterLabel
     * @return filterLabel the filterLabel
     */
    public void setFilterLabel(JLabel filterLabel) {
        this.filterLabel = filterLabel;
    }

    public void update() {
        update(getGraphics());
    }

    /**Resets all buttons to the initial state, only newProblemButton is enabled
     *
     */
    public void resetButtonPanel() {
        getRemoveEpisodeButton().setEnabled(false);
        getNewEpisodeButton().setEnabled(false);
        getRemovePartialContactButton().setEnabled(false);
        getNewPartialContactButton().setEnabled(false);
        getRemoveProblemButton().setEnabled(false);
        getNewMedicamentButton().setEnabled(false);
        getRemoveMedicamentButton().setEnabled(false);
        getNewProblemButton().setEnabled(true);
    }

    /**Creates the newMedicamentButton
     * @return the newMedicamentButton
     */
    public JButton createNewMedicamentButton() {
        JButton b = new JButton("New medication");
        if (b != null) {
            b.setEnabled(false);
            b.setActionCommand(Record.NEW_MEDICATION_ID);
            b.addActionListener((Record) getController());
        } else {
            throw new NullPointerException("Could not create New Partial Contact Button. The button is null");
        }
        return b;
    }

    /**Gets the newMedicamentButton
     * @return the newMedicamentButton
     */
    public JButton getNewMedicamentButton() {
        return newMedicamentButton;
    }

    /**Sets the newMedicamentButton
     * @return newMedicamentButton the newMedicamentButton
     */
    public void setNewMedicamentButton(JButton newMedicamentButton) {
        this.newMedicamentButton = newMedicamentButton;
    }

    /**Creates the removeMedicamentButton
     * @return the removeMedicamentButton
     */
    public JButton createRemoveMedicamentButton() {
        JButton b = new JButton("Remove medication");
        if (b != null) {
            b.setEnabled(false);
            b.setActionCommand(Record.REMOVE_MEDICATION_ID);
            b.addActionListener((Record) getController());
        } else {
            throw new NullPointerException("Could not create New Partial Contact Button. The button is null");
        }
        return b;
    }

    /**Gets the removeMedicamentButton
     * @return the removeMedicamentButton
     */
    public JButton getRemoveMedicamentButton() {
        return removeMedicamentButton;
    }

    /**Sets the removeMedicamentButton
     * @return removeMedicamentButton the removeMedicamentButton
     */
    public void setRemoveMedicamentButton(JButton removeMedicamentButton) {
        this.removeMedicamentButton = removeMedicamentButton;
    }
}
