/*
 * $RCSfile: TreeTableModelAdapter.java,v $
 *
 * Copyright (c) 1999-2002. Jens Bohl. All rights reserved.
 *
 * This software is published under the GPL GNU General Public License.
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * http://www.resmedicinae.org
 * - Information in Medicine -
 */

package org.resmedicinae.application.healthcare.record.treetable;

import javax.swing.*;
import javax.swing.event.*;
import javax.swing.table.*;
import javax.swing.tree.*;

/**
 * This class represents the TreeTable model that contains the JTree as well
 * as the object model representing the data structure of the TreeTable.
 *@author Jens Bohl <info@jens-bohl-professional.de>
 */
public class TreeTableModelAdapter extends AbstractTableModel {
    /** The tree component of the tree table */
    private JTree tree;

    /** The tree table model of the tree table */
    private TreeTableModel treeTableModel;

    /**
     *The constructor
     * @param treeTableModel the tree table model
     * @param tree  the tree component of the treetable
     */
    public TreeTableModelAdapter(TreeTableModel treeTableModel, JTree tree) {
        setTree(tree);
        setTreeTableModel(treeTableModel);
        //Initializing a treeExpansion Listener, short initialization
        tree.addTreeExpansionListener(
                new TreeExpansionListener() {
                    public void treeExpanded(TreeExpansionEvent event) {
                        fireTableDataChanged();
                    }

                    public void treeCollapsed(TreeExpansionEvent event) {
                        fireTableDataChanged();
                    }
                });
    }

    /**
     *Returns the column count. TabelModel interface.
     * @return number of columns
     */
    public int getColumnCount() {
        return treeTableModel.getColumnCount();
    }

    /**
     *Returns the column name. Overrides AstractTablelModel.
     * @param column index of column
     * @return column name
     */
    public String getColumnName(int column) {
        return treeTableModel.getColumnName(column);
    }

    /**
     *Returns the class the column object belongs to. Overrides AbstractTableModel.
     * @param column index of column
     * @return class of column
     */
    public Class getColumnClass(int column) {
        return treeTableModel.getColumnClass(column);
    }

    /**
     *Returns the row count. TabelModel interface.
     * @return the row count of the tree
     */
    public int getRowCount() {
        return tree.getRowCount();
    }

    /**
     *Returns the corresponding node for a particular row. Every node is always in one particular row.
     * @param row index of row
     * @return node in this row
     */
    protected Object nodeForRow(int row) {
        TreePath treePath = tree.getPathForRow(row);
        return treePath.getLastPathComponent();
    }

    /**
     *Returns the value of the cell at (row, column). Gets the value from the TreeTableModel with the help of
     * nodeForRow(row). TabelModel interface.
     *@param row index of row
     *@param column index of column
     * @return value at this cell
     */
    public Object getValueAt(int row, int column) {
        return treeTableModel.getValueAt(nodeForRow(row), column);
    }

    /**
     *Returns if the cell is editable. Tests this property of the TreeTableModel with the help of nodeForRow(row).
     * Overrides AbstractTreeTableModel.
     * @param row index of row
     * @param column indes of column
     * @return true,if cell is editable, false if not
     */
    public boolean isCellEditable(int row, int column) {
        return treeTableModel.isCellEditable(nodeForRow(row), column);
    }

    /**
     *Sets the value of the cell at (row, column). Method SetValueAt in TreeTableModel is invoked. Overrides
     * AbstractTreeTableModel.
     * @param row inex of row
     * @param column index of column
     * @param value value for cell
     */
    public void setValueAt(Object value, int row, int column) {
        treeTableModel.setValueAt(value, nodeForRow(row), column);
    }

    /**
     *Gets the JTree component of the tree table model.
     * @return the tree
     */
    public JTree getTree() {
        return tree;
    }

    /**
     *Sets the JTree component of the tree table model.
     *@param tree the JTree component
     */
    public void setTree(JTree tree) {
        this.tree = tree;
    }

    /**
     *Gets the tree table model.
     * @return the treeTableModel
     */
    public TreeTableModel getTreeTableModel() {
        return treeTableModel;
    }

    /**
     *Sets the tree table model
     *@param treeTableModel the tree table model
     */
    public void setTreeTableModel(TreeTableModel treeTableModel) {
        this.treeTableModel = treeTableModel;
    }
}
