/*
 * $RCSfile: TextAreaCellRenderer.java,v $
 *
 * Copyright (c) 1999-2002. Jens Bohl. All rights reserved.
 *
 * This software is published under the GPL GNU General Public License.
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * http://www.resmedicinae.org
 * - Information in Medicine -
 */

package org.resmedicinae.application.healthcare.record.treetable;

import org.resmedicinae.domain.healthcare.unit.*;

import javax.swing.*;
import javax.swing.border.*;
import javax.swing.table.*;
import java.awt.*;

/**
 * This class representes a CellRenderer for the JTable component of the JTreeTable.
 * Basically this renderer is a Swing JTextArea. This renderer is needed to present the text data like subject, object
 * assessments and plans in the right column.
 * @author Jens Bohl <info@jens-bohl-professional.de>
 */
public class TextAreaCellRenderer extends JTextArea implements TableCellRenderer {
    /**Constant representing an etched border*/
    protected static EtchedBorder noFocusBorder;
    /**The unselected foreGround*/
    private Color unselectedForeground;
    /**The unselected backGround*/
    private Color unselectedBackground;

    /**Default constructor*/
    public TextAreaCellRenderer() {
        super();
        noFocusBorder = new EtchedBorder();

        /*Sets the line-wrapping policy of the text area. If set to true the lines
		will be wrapped if they are too long to fit within the allocated width.*/

        setLineWrap(true);

        /*Sets the style of wrapping used if the text area is wrapping lines.
		If set to true the lines will be wrapped at word boundaries (whitespace)
		if they are too long to fit within the allocated width.*/

        setWrapStyleWord(true);
        setOpaque(true);
        //has no effect
        setRows(5);
    }

    /** Returns the renderer component, a JTextArea
     * @param table the table
     * @param value the value of the cell
     * @param isSelected tests whether the cell is selected
     * @param hasFocus tests whether the cell has the focus
     * @param row the row of the cell
     * @param column the column of the cell
     * @return the rendering component, a JTextArea
     * */
    public Component getTableCellRendererComponent(JTable table, Object value, boolean isSelected,
                                                   boolean hasFocus, int row, int column) {
        setValue(value);
        if (isSelected) {
            super.setForeground(table.getSelectionForeground());
            super.setBackground(table.getSelectionBackground());
        } else {
            super.setForeground((unselectedForeground != null) ? unselectedForeground : table.getForeground());
            super.setBackground((unselectedBackground != null) ? unselectedBackground : table.getBackground());
        }
        setFont(table.getFont());
        if (hasFocus) {
            if (table.isCellEditable(row, column)) {
                super.setForeground(UIManager.getColor("Table.focusCellForeground"));
                super.setBackground(UIManager.getColor("Table.focusCellBackground"));
            }
        } else {
        }
        JTreeTable treeTable = (JTreeTable) table;
        Object node = ((TreeTableModelAdapter) treeTable.getModel()).nodeForRow(row);
        if (!table.isCellEditable(row, column)) {
            super.setForeground(table.getSelectionForeground());
        }
        if ((node.getClass() == Medication.class) && (column == 2)) {
            //return new JComboBox();
        } else if ((node.getClass() == Episode.class) && (column == 1)) {
            this.append("(First)");
        } else if ((node.getClass() == Episode.class) && (column == 2)) {
            this.append("(Last)");
        }
        JScrollPane srcollPane = new JScrollPane(this);
        srcollPane.setVerticalScrollBarPolicy(JScrollPane.VERTICAL_SCROLLBAR_AS_NEEDED);
        this.setBorder(BorderFactory.createLineBorder(Color.GRAY));
        return this;
    }

    /** Set the value of the component
     * @param value the value of the component
     * */
    protected void setValue(Object value) {
        setText((value == null) ? "" : value.toString());
    }

    /** Sets the foreground color of the component
     * @param c the foreground color
     * */
    public void setForeground(Color c) {
        super.setForeground(c);
        unselectedForeground = c;
    }

    /** Sets the background color of the component
      * @param c the background color
     * */
    public void setBackground(Color c) {
        super.setBackground(c);
        unselectedBackground = c;
    }

    /** Updates the UI if necessary. */
    public void updateUI() {
        super.updateUI();
        setForeground(null);
        setBackground(null);
    }
}
