/*
 * $RCSfile: JTreeTable.java,v $
 *
 * Copyright (c) 1999-2002. Jens Bohl. All rights reserved.
 *
 * This software is published under the GPL GNU General Public License.
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * http://www.resmedicinae.org
 * - Information in Medicine -
 */

package org.resmedicinae.application.healthcare.record.treetable;

import org.resmedicinae.application.healthcare.record.*;
import org.resmedicinae.resmedlib.component.control.*;
import org.resmedicinae.domain.healthcare.unit.Problem;

import javax.swing.*;
import javax.swing.event.*;
import javax.swing.table.*;
import javax.swing.tree.*;
import java.awt.*;

/**
 * This class implements a TreeTable. Originally this component is not a Swing
 * native component. A TreeTable contains two columns or more columns where the left most one is an
 * ordinary JTree and the others components of a JTable. This split is done by particular
 * TreeTableCellRenderers und TreeTableCellEditors. These two classes are also described in this documentation.
 * Special thanx to Philip Milne at java.sun.com for the TreeTableExample!!!
 * @author Jens Bohl <info@jens-bohl-professional.de>
 */
public class JTreeTable extends JTable {
    /** The renderer for the tree column. */
    private TreeTableCellRenderer tree;
    /** The model for the tree column. */
    private TreeTableModel treeTableModel;
    /** The controller of the panel. This is Record. The controller is needed to add a keyListener and treeSelection
     * Listener.*/
    private Controller controller;

    /** The constructor.
     * @param controller the controller (Record)
     * */
    public JTreeTable(Controller controller) {
        super();
        setController(controller);
    }

    /** Initializes the JTreeTable. */
    public void initialize() {
        //set the selection model
        tree.setSelectionModel(
                new DefaultTreeSelectionModel() {
                    {
                        setSelectionModel(listSelectionModel);
                    }
                });
        tree.setAlignmentY(JTree.TOP_ALIGNMENT);
        setDefaultRenderer(TreeTableModel.class, tree);
        setDefaultEditor(TreeTableModel.class, new TreeTableCellEditor());
        tree.setRowHeight(35);
        setShowGrid(true);
        setRowHeight(35);
        //sets the height and width of space between cells
        setIntercellSpacing(new Dimension(0, 0));
        //sets the width of the columns of the tree table
        getColumnModel().getColumn(0).setPreferredWidth(350);
        getColumnModel().getColumn(1).setPreferredWidth(220);
        getColumnModel().getColumn(2).setPreferredWidth(180);
        getColumnModel().getColumn(3).setPreferredWidth(100);
        DefaultTreeCellRenderer renderer = new DefaultTreeCellRenderer();
        renderer.setLeafIcon(new ImageIcon(RecordModel.RelativeIconDirectoryName + "middle.gif"));
        tree.setCellRenderer(new TreeIconCellRenderer());
        //setSelectionModel(null);
        addKeyListener((Record) getController());
    }

    /**
     *Sets the model of the tree table. The model represents the data shown within the tree table.
     * It can be a xml model or an object model (based on a EHR).
     * @param treeTableModel The tree table model
     */
    public void setModel(TreeTableModel treeTableModel) {
        this.treeTableModel = treeTableModel;
        tree = new TreeTableCellRenderer(treeTableModel, getController());
        super.setModel(new TreeTableModelAdapter(treeTableModel, tree));
    }

    /**
     * Returns the model of this JTreeTable.
     * @return The TreeTableModel
     */
    public TreeTableModel getTreeTableModel() {
        return this.treeTableModel;
    }

    /** Sets the model for the the tree table */
    public void setTreeTableModel(TreeTableModel treeTableModel) {
        this.treeTableModel = treeTableModel;
    }

    /** Sets the tree of the tree table */
    public void setTree(TreeTableCellRenderer tree) {
        this.tree = tree;
    }

    /** Gets the tree from the tree table */
    public JTree getTree() {
        return (JTree) this.tree;
    }

    /**
     * Returns the editing Row of this JTreeTable
     * @return the editing row number
     */
    public int getEditingRow() {
        return (getColumnClass(editingColumn) == TreeTableModel.class) ? -1 : editingRow;
    }

    /**Gets the controller of the MVC
     * @return the controller (Record)
     */
    public Controller getController() {
        return this.controller;
    }

    /**Sets the controller of the MVC
     * @param controller the controller (Record)
     */
    public void setController(Controller controller) {
        this.controller = controller;
    }

    /**
     *This class represents a cell renderer for the tree column of this JTreeTable. A renderer provides
     * the appearance of the component. Basically this renderer is a JTree.
     */
    public class TreeTableCellRenderer extends JTree implements TableCellRenderer {
        /** The visible row. */
        private int visibleRow;

        /**
         *The constructor.
         * @param model the tree model
         */
        public TreeTableCellRenderer(TreeModel model, Controller controller) {
            super(model);
            addTreeSelectionListener((TreeSelectionListener) controller);
        }

        /**
         *Sets the bounds of cell renderer. Overriden from Component.
         * @ param x the x coordinate
         * @param y the y coordinate
         * @param w the width
         * @param h the heigth
         */
        public void setBounds(int x, int y, int w, int h) {
            super.setBounds(x, 0, w, JTreeTable.this.getHeight());
        }

        /**
         *Paints the component. Overriden from JComponent.
         * @param g the graphics context
         */
        public void paint(Graphics g) {
            g.translate(0, -visibleRow * getRowHeight());
            super.paint(g);
        }

        /** Gets the table cell renderer. This renderer is a JTree. Basically from TableCellRenderer.
         * @param table the table
         * @param value the value of the cell
         * @param isSelected tests whether the cell is selected
         * @param hasFocus tests whether the cell has the focus
         * @param row the row of the cell
         * @param column the column of the cell
         * @return the rendering component, a JTree
         * */
        public Component getTableCellRendererComponent(JTable table, Object value, boolean isSelected,
                                                       boolean hasFocus, int row, int column) {
            if (isSelected)
                setBackground(table.getSelectionBackground());
            else
                setBackground(table.getBackground());
            visibleRow = row;
            if((column==1)&&(value.getClass()==Problem.class)){
                setForeground(Color.RED);
            }
            return this;
        }
    }


    /**
     *This class represents a cell editor for this JTeeTable. By editing a particular
     * cell of the JTreeTable the cell editor provides the appearance of the cell. Basically this editor is a JTree.
     * Editor are used when editing a cell.*/

    public class TreeTableCellEditor extends AbstractCellEditor implements TableCellEditor {
        /** Gets the table cell editor component. This component is a JTree.
         * @param table the table
         * @param value the value of the cell
         * @param isSelected tests whether the cell is selected
         * @param hasFocus tests whether the cell has the focus
         * @param row the row of the cell
         * @param column the column of the cell
         * @return the rendering component, a JTree
         * */
        public Component getTableCellEditorComponent(JTable table, Object value, boolean isSelected, int row, int column) {
            return tree;
        }
    }
}
