/*

 * $RCSfile: PatientInfoPanel.java,v $
 *
 * Copyright (c) 1999-2002. Jens Bohl. All rights reserved.
 *
 * This software is published under the GPL GNU General Public License.
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * http://www.resmedicinae.org
 * - Information in Medicine -
 */

package org.resmedicinae.application.healthcare.record.topology;

import org.resmedicinae.application.healthcare.record.*;
import org.resmedicinae.domain.healthcare.*;
import org.resmedicinae.domain.healthcare.description.*;
import org.resmedicinae.domain.healthcare.unit.*;

import javax.swing.*;
import java.awt.*;

/** View component. This panel shows basic patient properties.
 * @author Jens Bohl <info@jens-bohl-professional.de>
 * */
public class PatientInfoPanel extends RecordPanel {
    /** TopologyLabel showing the name of patient */
    private TopologyLabel patientNameLabel;
    /** TopologyLabel showing the first name of patient */
    private TopologyLabel patientFirstNameLabel;
    /** TopologyLabel showing the gender of patient */
    private TopologyLabel patientGenderLabel;
    /** TopologyLabel showing the birthday of patient */
    private TopologyLabel patientBirthdayLabel;
    /** TopologyLabel showing the town of patient */
    private TopologyLabel patientTownLabel;
    /** TopologyLabel showing the adress of patient */
    private TopologyLabel patientAdressLabel;
    /** TopologyLabel showing the adress of patient */
    private TopologyLabel patientTelefonNumberLabel;
    /** TopologyLabel showing the insurance of patient */
    private TopologyLabel patientInsuranceLabel;
    /** TopologyLabel showing the insurance number of patient */
    private TopologyLabel patientInsuranceNumberLabel;
    /** TopologyLabel showing the insurance type of patient */
    private TopologyLabel patientInsuranceTypeLabel;
    /** JList showing the problems of patient */
    private JList problemList;
    /** TopologyScrollPane for problemList */
    private JScrollPane problemListScrollPane;
    /** TopologyTextArea showing the anamneae of patient */
    private TopologyTextArea anamneseTextArea;
    /** Listmodel for problemList */
    private DefaultListModel problemListModel;
    /** Listmodel for caveList */
    private DefaultListModel caveListModel;
    /** Listmodel for medicamentsList */
    private DefaultListModel medicamentsListModel;
    /** JList showing the important information about the patient */
    private JList caveList;
    /** JList showing all medicaments */
    private JList medicamentsList;
    /**ScrollPane for the caveList*/
    private JScrollPane caveListScrollPane;
    /**ScrollPane for the medicamentsList*/
    private JScrollPane medicamentsListScrollPane;
    /** The width of the panel */
    private int panelWidth = 230;
    /** The height of the panel */
    private int panelHeight = 145;
    /** The layout manager */
    private GridBagLayout gridbag;
    /** Action command */
    public static final String CHOOSE_PROBLEM_ID = new String("choose_problem");
    /** Label containing basic information concerning a patient */
    private JLabel basicInformationLabel;
    /** Label containing general information concerning a patient like blood type */
    private JLabel generalInformationLabel;

    /**Default constructor*/
    public PatientInfoPanel() {
        super();
    }

    /** Initialization of components. */
    public void initialize() {
        gridbag = new GridBagLayout();
        setLayout(gridbag);
        GridBagConstraints c = new GridBagConstraints();
        c.fill = GridBagConstraints.VERTICAL;
        setBorder(BorderFactory.createTitledBorder("Patient Data"));
        setToolTipText("General patient information");
        setVisible(true);

        setGeneralInformationLabel(createGeneralInformationLabel());
        setBasicInformationLabel(createBasicInformationLabel());
        setPatientAdressLabel(createPatientAdressLabel());
        setPatientInsuranceLabel(createPatientInsuranceLabel());
        setProblemListModel(createProblemListModel());
        setProblemList(createProblemList());
        setProblemListScrollPane(createProblemListScrollPane());
        setCaveListModel(createCaveListModel());
        setCaveList(createCaveList());
        setCaveListScrollPane(createCaveListScrollPane());
        setMedicamentsListModel(createMedicamentsListModel());
        setMedicamentsList(createMedicamentsList());
        setMedicamentsListScrollPane(createMedicamentsListScollPane());

//        anamneseTextArea = new TopologyTextArea(new Dimension(200, 100));
        //TopologyScrollPane anamneseScrollPane = new TopologyScrollPane(anamneseTextArea, "Anamnese");

        c.anchor = GridBagConstraints.WEST;
        c.gridx = 0;
        c.gridy = 0;
        c.anchor = GridBagConstraints.WEST;
        c.insets = new Insets(5, 5, 5, 5);
        gridbag.setConstraints(basicInformationLabel, c);
        add(basicInformationLabel);

        c.gridx = 0;
        c.gridy = 1;
        gridbag.setConstraints(patientAdressLabel, c);
        add(patientAdressLabel);
        c.gridx = 0;
        c.gridy = 2;
        gridbag.setConstraints(generalInformationLabel, c);
        add(generalInformationLabel);

        c.anchor = GridBagConstraints.WEST;
        c.gridx = 0;
        c.gridy = 3;
        gridbag.setConstraints(patientInsuranceLabel, c);
        add(patientInsuranceLabel);

        c.anchor = GridBagConstraints.WEST;
        c.gridwidth = 1;
        c.gridheight = 4;
        c.gridx = 1;
        c.gridy = 0;
        gridbag.setConstraints(caveListScrollPane, c);
        add(caveListScrollPane);

        c.anchor = GridBagConstraints.WEST;
        c.gridwidth = 1;
        c.gridheight = 4;
        c.gridx = 2;
        c.gridy = 0;
        gridbag.setConstraints(problemListScrollPane, c);
        add(problemListScrollPane);

        c.anchor = GridBagConstraints.WEST;
        c.gridwidth = 1;
        c.gridheight = 4;
        c.gridx = 3;
        c.gridy = 0;
        gridbag.setConstraints(medicamentsListScrollPane, c);
        add(medicamentsListScrollPane);


    }

    /**Gets the preferred size of this panel
     *@return the preferred size
     *
     */
    public Dimension getPreferredSize() {
        return new Dimension(1000, 180);
    }

    /** Changes the current patient. Fills all needed labels and the problemList
     * @param patient the healthRecord to be displayed
     * */
    public void displayPatient(HealthRecord patient) {
        System.out.println("patient:"+patient);
        System.out.println(patient.getFirstName().getValue());
        System.out.println(patient.getName().getValue());
        System.out.println(patient.getBirthday().getValue());
        getBasicInformationLabel().setText(patient.getFirstName().getValue() + ", " + patient.getName().getValue() +
                ", " + patient.getBirthday().getValue());
        getPatientAdressLabel().setText(patient.getStreet().getValue() + ", " + patient.getZipCode().getValue() + ", " + patient.getTown().getValue());
        getPatientInsuranceLabel().setText(patient.getInsurance().getValue());
        getGeneralInformationLabel().setText(patient.getHeight().getValue()+", "+patient.getWeight().getValue()+", "+patient.getBloodType().getValue());
        getCaveListModel().clear();
        Object[] caves = patient.getCaves();
        for (int cavesIndex = 0; cavesIndex < caves.length; cavesIndex++) {
            getCaveListModel().addElement(((Cave) caves[cavesIndex]).getName().getValue());
        }
        problemListModel.removeAllElements();
        Object[] problems = patient.getProblems();
        for (int problemsIndex = 0; problemsIndex < problems.length; problemsIndex++) {
            getProblemListModel().addElement(((Problem) problems[problemsIndex]).getName().getValue());
        }
    }


    /** Adds a problem to the problem list.
     * @param problemName the name of the problem
     * */
    public void addProblemToProblemList(String problemName) {
        this.problemListModel.addElement(problemName);
    }

    /** Enables/Disables problemList and anamnese text area.
     * @param enabled indicates whether probelm list and anamnese textArea should be enabled or not
     * */
    public void setEnabled(boolean enabled) {
        problemList.setEnabled(enabled);
        anamneseTextArea.setEnabled(enabled);
    }

    /** Clears the anamnese text area. */
    public void clearAllTextAreas() {
        anamneseTextArea.setText("");
    }

    /**
     *Gets the patientNameLabel
     * @return patientNameLabel
     */
    public TopologyLabel getPatientNameLabel() {
        return patientNameLabel;
    }

    /**
     *Sets the patientNameLabel
     * @param patientNameLabel the patient name label of the patient info panel
     */
    public void setPatientNameLabel(TopologyLabel patientNameLabel) {
        this.patientNameLabel = patientNameLabel;
    }

    /**
     *Gets the patientFirstNameLabel
     * @return patientFirstNameLabel
     */
    public TopologyLabel getPatientFirstNameLabel() {
        return patientFirstNameLabel;
    }

    /**
     *Sets the patientFirstNameLabel
     * @param patientFirstNameLabel the patient first name label of the patient info panel
     */
    public void setPatientFirstNameLabel(TopologyLabel patientFirstNameLabel) {
        this.patientFirstNameLabel = patientFirstNameLabel;
    }

    /**
     *Gets the patientGenderLabel
     * @return patientGenderLabel
     */
    public TopologyLabel getPatientGenderLabel() {
        return patientGenderLabel;
    }

    /**
     *Sets the patientGenderLabel
     * @param patientGenderLabel the patient gender label of the patient info panel
     */
    public void setPatientGenderLabel(TopologyLabel patientGenderLabel) {
        this.patientGenderLabel = patientGenderLabel;
    }

    /**
     *Gets the patientBirthdayLabel
     * @return patientBirthdayLabel
     */
    public TopologyLabel getPatientBirthdayLabel() {
        return patientBirthdayLabel;
    }

    /**
     *Sets the patientBirthdayLabel
     * @param patientBirthdayLabel the patient birthday label of the patient info panel
     */
    public void setPatientBirthdayLabel(TopologyLabel patientBirthdayLabel) {
        this.patientBirthdayLabel = patientBirthdayLabel;
    }

    /**
     *Gets the patientTownLabel
     * @return patientTownLabel
     */
    public TopologyLabel getPatientTownLabel() {
        return patientTownLabel;
    }

    /**
     *Sets the patientTownLabel
     * @param patientTownLabel the patient town label of the patient info panel
     */
    public void setPatientTownLabel(TopologyLabel patientTownLabel) {
        this.patientTownLabel = patientTownLabel;
    }

    /**
     *Creates the patientAdressLabel
     * @return patientAdressLabel
     */
    public TopologyLabel createPatientAdressLabel(){
        TopologyLabel patientAdressLabel = new TopologyLabel(true);
        patientAdressLabel.setPreferredSize(new Dimension(280, 25));
        return patientAdressLabel;
    }

    /**
     *Gets the patientAdressLabel
     * @return patientAdressLabel
     */
    public TopologyLabel getPatientAdressLabel() {
        return patientAdressLabel;
    }

    /**
     *Sets the patientAdressLabel
     * @param patientAdressLabel the patient adress label of the patient info panel
     */
    public void setPatientAdressLabel(TopologyLabel patientAdressLabel) {
        this.patientAdressLabel = patientAdressLabel;
    }

    /**
     *Gets the patientTelefonNumberLabel
     * @return patientTelefonNumberLabel
     */
    public TopologyLabel getPatientTelefonNumberLabel() {
        return patientTelefonNumberLabel;
    }

    /**
     *Sets the patientTelefonNumberLabel
     * @param patientTelefonNumberLabel the patient telefon number of the patient info panel
     */
    public void setPatientTelefonNumberLabel(TopologyLabel patientTelefonNumberLabel) {
        this.patientTelefonNumberLabel = patientTelefonNumberLabel;
    }

    public TopologyLabel createPatientInsuranceLabel(){
        TopologyLabel patientInsuranceLabel = new TopologyLabel(true);
        patientInsuranceLabel.setPreferredSize(new Dimension(280, 25));
        return patientInsuranceLabel;
    }

    /**
     *Gets the patientInsuranceLabel
     * @param patientInsuranceLabel
     */
    public TopologyLabel getPatientInsuranceLabel() {
        return patientInsuranceLabel;
    }

    /**
     *Sets the patientInsuranceLabel
     * @param patientInsuranceLabel the patient insurance label of the patient info panel
     */
    public void setPatientInsuranceLabel(TopologyLabel patientInsuranceLabel) {
        this.patientInsuranceLabel = patientInsuranceLabel;
    }

    /**
     *Gets the patientInsuranceNumberLabel
     * @return patientInsuranceNumberLabel
     */
    public TopologyLabel getPatientInsuranceNumberLabel() {
        return patientInsuranceNumberLabel;
    }

    /**
     *Sets the patientInsuranceNumberLabel
     * @param patientInsuranceNumberLabel the patient insurance number label of the patient info panel
     */
    public void setPatientInsuranceNumberLabel(TopologyLabel patientInsuranceNumberLabel) {
        this.patientInsuranceNumberLabel = patientInsuranceNumberLabel;
    }

    /**
     *Gets the patientInsuranceTypeLabel
     * @return patientInsuranceTypeLabel
     */
    public TopologyLabel getPatientInsuranceTypeLabel() {
        return patientInsuranceTypeLabel;
    }

    /**
     *Sets the patientInsuranceTypeLabel
     * @param patientInsuranceTypeLabel the patient insurance type label of the patient info panel
     */
    public void setPatientInsuranceTypeLabel(TopologyLabel patientInsuranceTypeLabel) {
        this.patientInsuranceTypeLabel = patientInsuranceTypeLabel;
    }

    /**
     *Creates the problemList
     * @return problemList
     */
    public JList createProblemList(){
        JList problemList = new JList(problemListModel);
        problemList.setPreferredSize(new Dimension(180, 90));
        problemList.setBackground(RecordModel.BackgroundColor);
        problemList.addListSelectionListener((Record)getController());
        return problemList;
    }

    /**
     *Gets the problemList
     * @return problemList
     */
    public JList getProblemList() {
        return problemList;
    }

    /**
     *Sets the problemList
     * @param problemList the problem list of the patient info panel
     */
    public void setProblemList(JList problemList) {
        this.problemList = problemList;
    }

    /**
     *Creates the problemListScrollPane
     * @return problemListScrollPane
     */
    public TopologyScrollPane createProblemListScrollPane(){
        TopologyScrollPane problemListScrollPane = new TopologyScrollPane(getProblemList(), "Problems");
        problemListScrollPane.setPreferredSize(new Dimension(200, 80));
        problemListScrollPane.setHorizontalScrollBarPolicy(JScrollPane.HORIZONTAL_SCROLLBAR_AS_NEEDED);
        problemListScrollPane.setVerticalScrollBarPolicy(JScrollPane.VERTICAL_SCROLLBAR_AS_NEEDED);
        return problemListScrollPane;
    }

    /**
     *Gets the problemListScrollPane
     * @return problemListScrollPane
     */
    public JScrollPane getProblemListScrollPane() {
        return problemListScrollPane;
    }

    /**
     *Sets the problemListScrollPane
     * @param problemListScrollPane  the problem list scrollpane of the patient info panel
     */
    public void setProblemListScrollPane(TopologyScrollPane problemListScrollPane) {
        this.problemListScrollPane = problemListScrollPane;
    }

    /**
     *Gets the anamneseTextArea
     * @return anamneseTextArea
     */
    public TopologyTextArea getAnamneseTextArea() {
        return anamneseTextArea;
    }

    /**
     *Sets the anamneseTextArea
     * @param anamneseTextArea the anamnese text area of the patient info panel
     */
    public void setAnamneseTextArea(TopologyTextArea anamneseTextArea) {
        this.anamneseTextArea = anamneseTextArea;
    }

    /**
     *Creates the problemListModel
     * @return problemListModel
     */
    public DefaultListModel createProblemListModel(){
        return new DefaultListModel();
    }

    /**
     *Gets the problemListModel
     * @return problemListModel
     */
    public DefaultListModel getProblemListModel() {
        return problemListModel;
    }

    /**
     *Sets the problemListModel
     * @param problemListModel the problem list model of the patient info panel
     */
    public void setProblemListModel(DefaultListModel problemListModel) {
        this.problemListModel = problemListModel;
    }

    /**
     *Creates the caveListModel
     * @return caveListModel
     */
    public DefaultListModel createCaveListModel(){
        DefaultListModel caveListModel = new DefaultListModel();
        return caveListModel;
    }

    /**
     *Gets the caveListModel
     * @return caveListModel
     */
    public DefaultListModel getCaveListModel() {
        return this.caveListModel;
    }

    /**
     *Sets the caveListModel
     * @param caveListModel  the cave list model of the patient info panel
     */
    public void setCaveListModel(DefaultListModel caveListModel) {
        this.caveListModel = caveListModel;
    }


    /**
     *Creates the medicamentsListModel
     * @return medicamnetsListModel
     */
    public DefaultListModel createMedicamentsListModel(){
         return new DefaultListModel();
    }

    /**
     *Gets the medicamentsListModel
     * @return medicamnetsListModel
     */
    public DefaultListModel getMedicamentsListModel() {
        return medicamentsListModel;
    }

    /**
     *Sets the medicamentsListModel
     * @param medicamentsListModel the medicamnets list model of the patient info panel
     */
    public void setMedicamentsListModel(DefaultListModel medicamentsListModel) {
        this.medicamentsListModel = medicamentsListModel;
    }

    /**
     *Creates the caveList
     * @return caveList
     */
    public JList createCaveList(){
        JList caveList = new JList(getCaveListModel());
        caveList.setPreferredSize(new Dimension(180, 90));
        caveList.setBackground(RecordModel.BackgroundColor);
        caveList.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
        caveList.addListSelectionListener((Record) getController());
        return caveList;
    }

    /**
     *Gets the caveList
     * @return caveList
     */
    public JList getCaveList() {
        return caveList;
    }

    /**
     *Sets the caveList
     * @param caveList the cave list of the patient info panel
     */
    public void setCaveList(JList caveList) {
        this.caveList = caveList;
    }

    /**
     *Creates the medicamentsList
     * @return medicamentsList
     */
    public JList createMedicamentsList(){
        JList medicamentsList = new JList(medicamentsListModel);
        medicamentsList.setPreferredSize(new Dimension(180, 90));
        medicamentsList.setBackground(RecordModel.BackgroundColor);
        return medicamentsList;
    }

    /**
     *Gets the medicamentsList
     * @return medicamentsList
     */
    public JList getMedicamentsList() {
        return medicamentsList;
    }

    /**
     *Sets the medicamentsList
     * @param medicamentsList the medicaments list of the patient info panel
     */
    public void setMedicamentsList(JList medicamentsList) {
        this.medicamentsList = medicamentsList;
    }

    /**
     *Creates the caveListScrollPane
     * @return caveListScrollPane
     */
    public TopologyScrollPane createCaveListScrollPane(){
        TopologyScrollPane caveListScrollPane = new TopologyScrollPane(getCaveList(), "CAVE!");
        caveListScrollPane.setPreferredSize(new Dimension(200, 80));
        caveListScrollPane.setHorizontalScrollBarPolicy(JScrollPane.HORIZONTAL_SCROLLBAR_AS_NEEDED);
        caveListScrollPane.setVerticalScrollBarPolicy(JScrollPane.VERTICAL_SCROLLBAR_AS_NEEDED);
        return caveListScrollPane;
    }

    /**
     *Gets the caveListScrollPane
     * @return caveListScrollPane
     */
    public JScrollPane getCaveListScrollPane() {
        return caveListScrollPane;
    }

    /**
     *Sets the caveListScrollPane
     * @param caveListScrollPane the cave list scrollpane of the patient info panel
     */
    public void setCaveListScrollPane(TopologyScrollPane caveListScrollPane) {
        this.caveListScrollPane = caveListScrollPane;
    }

    /**
     *Creates the medicamentsListScrollPane
     * @return medicamentsListScrollPane
     */
    public TopologyScrollPane createMedicamentsListScollPane(){
        TopologyScrollPane medicamentsListScrollPane = new TopologyScrollPane(getMedicamentsList(), "Medication");
        medicamentsListScrollPane.setPreferredSize(new Dimension(200, 80));
        medicamentsListScrollPane.setHorizontalScrollBarPolicy(JScrollPane.HORIZONTAL_SCROLLBAR_AS_NEEDED);
        medicamentsListScrollPane.setVerticalScrollBarPolicy(JScrollPane.VERTICAL_SCROLLBAR_AS_NEEDED);
        return medicamentsListScrollPane;
    }

    /**
     *Gets the medicamentsListScrollPane
     * @return medicamentsListScrollPane
     */
    public JScrollPane getMedicamentsListScrollPane() {
        return medicamentsListScrollPane;
    }

    /**
     *Sets the medicamentsListScrollPane
     * @param medicamentsListScrollPane  the medicaments list srcollpane of the patient info panel
     */
    public void setMedicamentsListScrollPane(TopologyScrollPane medicamentsListScrollPane) {
        this.medicamentsListScrollPane = medicamentsListScrollPane;
    }

    /**
     *Gets the panelWidth
     * @return panelWidth
     */
    public int getPanelWidth() {
        return panelWidth;
    }

    /**
     *Sets the panelWidth
     * @param panelWidth  the width of the patient info panel
     */
    public void setPanelWidth(int panelWidth) {
        this.panelWidth = panelWidth;
    }

    /**
     *Gets the panelWidth
     * @return panelWidth
     */
    public int getPanelHeight() {
        return panelHeight;
    }

    /**
     *Sets the panelHeigth
     * @param panelHeigth of the patient info panel
     */
    public void setPanelHeight(int panelHeight) {
        this.panelHeight = panelHeight;
    }

    /**
     *Gets the Layout Manager
     * @return gridbag
     */
    public GridBagLayout getGridbag() {
        return gridbag;
    }

    /**
     *Sets the Layout Manager
     * @param gridbag the layout manager of the patient info panel
     */
    public void setGridbag(GridBagLayout gridbag) {
        this.gridbag = gridbag;
    }


    /**Creates the basicInformationLabel
     * @return the basicInformationLabel
     *
     */
    public JLabel createBasicInformationLabel(){
        JLabel basicInformationLabel = new TopologyLabel(true);
        basicInformationLabel.setPreferredSize(new Dimension(280, 25));
        return basicInformationLabel;
    }

    /**Gets the basicInformationLabel
     * @return the basicInformationLabel
     *
     */
    public JLabel getBasicInformationLabel() {
        return basicInformationLabel;
    }

    /**Sets the generalInformationLabel
     * @param basicInformationLabel the basicInformationLabel
     *
     */
    public void setBasicInformationLabel(JLabel basicInformationLabel) {
        this.basicInformationLabel = basicInformationLabel;
    }

    /**Creates the generalInformationLabel
     * @return the generalInformationLabel
     *
     */
    public JLabel createGeneralInformationLabel(){
        JLabel generalInformationLabel = new TopologyLabel(true);
        generalInformationLabel.setPreferredSize(new Dimension(280, 25));
        return generalInformationLabel;
    }

    /**Gets the generalInformationLabel
     * @return the generalInformationLabel
     *
     */
    public JLabel getGeneralInformationLabel() {
        return generalInformationLabel;
    }

    /**Sets the generalInformationLabel
     * @param generalInformationLabel the generalInformationLabel
     *
     */
    public void setGeneralInformationLabel(JLabel generalInformationLabel) {
        this.generalInformationLabel = generalInformationLabel;
    }
}
