/*
 * $RCSfile: Overlay.java,v $
 *
 * Copyright (c) 1999-2002. Jens Bohl. All rights reserved.
 *
 * This software is published under the GPL GNU General Public License.
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * http://www.resmedicinae.org
 * - Information in Medicine -
 */

package org.resmedicinae.application.healthcare.record.topology;

import org.resmedicinae.resmedlib.Item;

import java.util.Hashtable;
import java.util.Vector;
import java.util.Enumeration;
import java.awt.*;

/**This class represents an overlay. An overlay is the OO-representation of an anatomical image and all it properties.
 * Those are name, description, diseases (in reference on this overlay) and the overlay regions.
 * @author Jens Bohl <info@jens-bohl-professional.de>*
 * */

public class Overlay extends Item{
    /**Name of the overlay*/
    private String name;
    /**Description of the overlay*/
    private String description;
    /**Image of the overlay*/
    private Image image;
    /**Hashtable of overlay regions*/
    private Hashtable overlayRegionHashtable;
    /**Vector of alternative overlays*/
    private Vector alternativeOverlays;
    /**Vector of diseases*/
    private Hashtable diseases;
    /**Boolean indicating wether overlay is alternative or not*/
    private boolean isAlternative;

    /**DefaultConstructor*/
    public Overlay(){
        initialize();
    }

    /**Initialize this object
     * */
    public void initialize(){
        setDiseases(new Hashtable());
        setOverlayRegionHashtable(new Hashtable());
        setAlternativeOverlays(new Vector());
    }
    /**Gets the name of the overlay
     * @return name of the overlay
     * */
    public String getName() {
        return name;
    }

    /**Sets the name of the overlay
     *
     * @param name the name of the overlay
     */
    public void setName(String name) {
        this.name = name;
    }

    /**Gets the description of the overlay
     * @return description of the overlay
     * */
    public String getDescription() {
        return description;
    }

    /**Sets the description of the overlay
     * @param decription description of the overlay
     * */
    public void setDescription(String description) {
        this.description = description;
    }

    /**Gets the image of the overlay
     * @return image of the overlay
     * */
    public Image getImage() {
        return image;
    }

    /**Sets the image of the overlay
     * @param image image of the overlay
     * */
    public void setImage(Image image) {
        this.image = image;
    }

    /**Gets the overlay regions of the overlay
     * @return overlayRegionHashtable containing the overlayRegions of the overlay
     * */
    public Hashtable getOverlayRegionHashtable() {
        return overlayRegionHashtable;
    }

    /**Sets the overlay regions of the overlay
     * @param overlayRegionHashtable hashtable containing the overlayRegions of the overlay
     * */
    public void setOverlayRegionHashtable(Hashtable overlayRegionHashtable) {
        this.overlayRegionHashtable = overlayRegionHashtable;
    }

    /**Return the String representation of this overlay
     * @return the string representation (name) of the overlay
     * */
    public String toString(){
        return getName();
    }

    /**Return the alternative overlays of this overlay
     * @return the alternative overlays of the overlay
     * */
    public Vector getAlternativeOverlays() {
        return alternativeOverlays;
    }

    /**Sets the alternative overlays of the overlay
     * @param alternativeOverlays the alternative overlays of the overlay
     * */
    public void setAlternativeOverlays(Vector alternativeOverlays) {
        this.alternativeOverlays = alternativeOverlays;
    }

    /**Gets all diseases for this overlay
     * @return the diseases of this overlay
     * */
    public Vector getDiseases() {
        Object[] childrenArray = getChildren().values().toArray();
        Vector tempVector = new Vector();
        for (int i = 0; i < childrenArray.length; i++) {
            if (childrenArray[i].getClass().equals(Disease.class)) {
                tempVector.add(childrenArray[i]);
            }
        }
        return tempVector;
    }

    /**Sets all diseases for this overlay
     * @param diseases the diseases of this overlay
     * */
    public void setDiseases(Hashtable diseases) {
        Enumeration keys = diseases.keys();
        while(keys.hasMoreElements()){
            String key = (String)keys.nextElement();
            Disease disease = (Disease)diseases.get(key);
            set(new org.resmedicinae.resmedlib.term.String(key),disease);
        }
    }

    /**Test whether the overlay already has this disease
     * @return true if the overlay has the disease, false if not
     * */
    public boolean hasDisease(String diseaseName){
        return getDiseases().contains(diseaseName);
    }

    /**Adds a disease to an overlay
     * @param disease the disease
     * */
    public void addDisease(Disease disease){
        set(new org.resmedicinae.resmedlib.term.String(disease.getDiseaseName()),disease);
    }
    /**Tests whether this overlay is alternative or not
     * @return true if this overlay is alternative, false if not
     */
    public boolean isAlternative() {
        return isAlternative;
    }

    /**Sets whether this overlay is as an alternative overlay or not
     * @param alternative the boolean value indicating whether this overlay is alternative or not
     *
     */
    public void setAlternative(boolean alternative) {
        isAlternative = alternative;
    }
}
