/*
 * $RCSfile: MakeOverlayRegionDialog.java,v $
 *
 * Copyright (c) 1999-2002. Jens Bohl. All rights reserved.
 *
 * This software is published under the GPL GNU General Public License.
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * http://www.resmedicinae.org
 * - Information in Medicine -
 */

package org.resmedicinae.application.healthcare.record.topology;

import org.resmedicinae.application.healthcare.record.*;
import org.resmedicinae.resmedlib.component.control.*;
import org.resmedicinae.resmedlib.component.view.swing.*;

import javax.swing.*;
import java.awt.*;
import java.awt.event.*;

/** View Component. This dialog is for creating a new overlayRegion.
 * @author Jens Bohl <info@jens-bohl-professional.de>
 *  */
public class MakeOverlayRegionDialog extends SwingView {
    /** Layout Manager of this frame */
    private GridBagLayout gridbag;
    /** Layout constraints of this frame */
    private GridBagConstraints c;
    /** Name label of this overlay region */
    private TopologyLabel polygonNameLabel;
    /** Number label of this overlay region */
    private TopologyLabel polygonNumberLabel;
    /** Number text field of this overlay region */
    private JTextField polygonNumberTextField;
    /** Name text field of this polygon */
    private JTextField polygonNameTextField;
    /** Target label of this polygon */
    private TopologyLabel targetImageLabel;
    /** Panel containing a preview of the target of this polygon */
    private PreviewPanel targetPreviewPanel;
    /** Ok button for commitment */
    private JButton okButton;
    /** Cancel button */
    private JButton cancelButton;
    /** ComboBox for selecting a target */
    private JComboBox targetImageComboBox;
    /** Right neighbour label */
    private TopologyLabel rightNeighbourLabel;
    /** Right neighbour combobox */
    private JComboBox rightNeighbourComboBox;
    /** Right neighbour checkbox */
    private JCheckBox rightNeighbourCheckBox;
    /** Left neighbour label */
    private TopologyLabel leftNeighbourLabel;
    /** Left neighbour combobox */
    private JComboBox leftNeighbourComboBox;
    /** Right neighbour checkbox */
    private JCheckBox leftNeighbourCheckBox;
    /** Top neighbour label */
    private TopologyLabel topNeighbourLabel;
    /** Top neighbour combobox */
    private JComboBox topNeighbourComboBox;
    /** Right neighbour checkbox */
    private JCheckBox topNeighbourCheckBox;
    /** Bottom neighbour label */
    private TopologyLabel bottomNeighbourLabel;
    /** Bottom neighbour combobox */
    private JComboBox bottomNeighbourComboBox;
    /** Right neighbour checkbox */
    private JCheckBox bottomNeighbourCheckBox;
    /** The neighbour panel */
    private ResPanel neighbourPanel;
    /** The top neighbour panel */
    private ResPanel topNeighbourPanel;
    /** The bottom neighbour panel */
    private ResPanel bottomNeighbourPanel;
    /** The left neighbour panel */
    private ResPanel leftNeighbourPanel;
    /** The right neighbour panel */
    private ResPanel rightNeighbourPanel;
    /** Panel containing name and number of region */
    private ResPanel generalRegionInformationPanel;
    /** Panel containing the target of the region */
    private ResPanel targetOverlayPanel;
    /** Label in the infoamtion panel for showing information about dialog handling */
    private JTextArea informationTextArea;
    /**The model of the MVC*/
    private RecordModel model;

    /** Constructor setting location of frame */
    public MakeOverlayRegionDialog() {
    }

    /**
     * Creates a rectangle storing the dialog bounds.
     * @return the rectangle storing the dialog bounds
     */
    public Rectangle createDialogBounds() {
        // Determine width.
        int screenWidth = (int) Math.floor(Toolkit.getDefaultToolkit().getScreenSize().getWidth());
        int w = 600;
        if (w > screenWidth) {
            w = screenWidth;
        }
        // Use centered x position by default.
        int x = (screenWidth - w) / 2;
        // Determine height.
        int screenHeight = (int) Math.floor(Toolkit.getDefaultToolkit().getScreenSize().getHeight());
        int h = 600;
        if (h > screenHeight) {
            h = screenHeight;
        }
        // Use centered y position by default.
        int y = (screenHeight - h) / 2;
        return new Rectangle(x, y, w, h);
    }

    /**
     * Creates a display mode.
     * @return the display mode
     */
    public int createDisplayMode() {
        return DisplayFactory.MODELESS_DIALOG_DISPLAY;
    }

    /**
     * Creates a control issued when the window containing this view is closed.
     * @return the control issued when the window containing this view is closed
     * @exception NullPointerException if the control is null
     */
    public Control createCloseControl() throws NullPointerException {
        Control c = new Control();
        if (c != null) {
            c.setId(Record.DESTROY_LOAD_PATIENT_DIALOG_CONTROL_ID);
            c.setSender((Controller) getController());
        } else {
            throw new NullPointerException("Could not create close control. The control is null.");
        }
        return c;
    }

    /** Finalizes this component. */
    public void finalizz() throws Exception {
        try {
            // Set parameters.
            setBorder(null);
        } finally {
            super.finalizz();
        }
    }

    /** Initialization of components */
    public void initialize() {
        setTitle("Create new overlay region...");
        gridbag = new GridBagLayout();
        setLayout(gridbag);
        GridBagConstraints c = new GridBagConstraints();

        setDisplayMode(DisplayFactory.MODELESS_DIALOG_DISPLAY);
        setDialogBounds(createDialogBounds());
        setGeneralRegionInformationPanel(createGeneralRegionInformationPanel());
        setTargetOverlayPanel(createTargetOverlayPanel());
        setNeighbourPanel(createNeighbourPanel());
        setOkButton(createOkButton());
        setCancelButton(createCancelButton());


        c.anchor = GridBagConstraints.WEST;
        c.insets = new Insets(2, 2, 2, 2);
        c.fill = GridBagConstraints.HORIZONTAL;
        c.gridwidth = 2;
        c.gridx = 0;
        c.gridy = 1;
        gridbag.setConstraints(getGeneralRegionInformationPanel(), c);
        add(getGeneralRegionInformationPanel());

        c.anchor = GridBagConstraints.WEST;
        c.insets = new Insets(2, 2, 2, 2);
        c.fill = GridBagConstraints.HORIZONTAL;
        c.gridwidth = 2;
        c.gridx = 0;
        c.gridy = 2;
        gridbag.setConstraints(getTargetOverlayPanel(), c);
        add(getTargetOverlayPanel());

        c.anchor = GridBagConstraints.WEST;
        c.insets = new Insets(2, 2, 2, 2);
        c.fill = GridBagConstraints.HORIZONTAL;
        c.gridwidth = 2;
        c.gridx = 0;
        c.gridy = 3;
        gridbag.setConstraints(getNeighbourPanel(), c);
        add(getNeighbourPanel());

        c.anchor = GridBagConstraints.EAST;
        c.insets = new Insets(2, 2, 2, 2);
        c.gridwidth = 1;
        c.gridx = 0;
        c.gridy = 4;
        gridbag.setConstraints(getOkButton(), c);
        add(getOkButton());

        c.anchor = GridBagConstraints.EAST;
        c.insets = new Insets(2, 2, 2, 2);
        c.gridwidth = 1;
        c.gridx = 1;
        c.gridy = 4;
        gridbag.setConstraints(getCancelButton(), c);
        add(getCancelButton());
    }

    /**ReInitializes the component. This is needed to reset the properties.*/
    public void reInitialize() {
        RecordModel model = (RecordModel) getModel();
        //first remove all items from all comboboxes
        topNeighbourComboBox.removeAllItems();
        bottomNeighbourComboBox.removeAllItems();
        leftNeighbourComboBox.removeAllItems();
        rightNeighbourComboBox.removeAllItems();
        int currentRegionCount = model.getCurrentOverlayRegionNames().length;
        for (int i = 0; i < currentRegionCount; i++) {
            topNeighbourComboBox.addItem(model.getCurrentOverlayRegionNames()[i]);
            bottomNeighbourComboBox.addItem(model.getCurrentOverlayRegionNames()[i]);
            leftNeighbourComboBox.addItem(model.getCurrentOverlayRegionNames()[i]);
            rightNeighbourComboBox.addItem(model.getCurrentOverlayRegionNames()[i]);
        }
        getOverlayRegionNumberTextField().setText(String.valueOf(currentRegionCount + 1));
    }

    /**Gets the layoutManager
     * @return the layoutManager
     *
     */
    public GridBagLayout getGridbag() {
        return gridbag;
    }

    /**Sets the layoutManager
     * @return gridbag the layoutManager
     *
     */
    public void setGridbag(GridBagLayout gridbag) {
        this.gridbag = gridbag;
    }

    /**Gets the overlayRegionNameLabel
     * @return the overlayRegionNameLabel
     */
    public TopologyLabel getOverlayRegionNameLabel() {
        return polygonNameLabel;
    }

    /**Sets the overlayRegionNameLabel
     * @param the overlayRegionNameLabel
     */
    public void setOverlayRegionNameLabel(TopologyLabel polygonNameLabel) {
        this.polygonNameLabel = polygonNameLabel;
    }

    /**Gets the overlayRegionNameTextField
     * @return the overlayRegionNameTextField
     */
    public JTextField getOverlayRegionNameTextField() {
        return polygonNameTextField;
    }

    /**Sets the overlayRegionNameTextField
     * @param overlayRegionNameTextField the overlayRegionNameTextField
     */
    public void setOverlayRegionNameTextField(JTextField polygonNameTextField) {
        this.polygonNameTextField = polygonNameTextField;
    }

    /**Creates the targetImageLabel
     * @return the targetImageLabel
     */
    public TopologyLabel createTargetImageLabel(){
        return new TopologyLabel(false);
    }

    /**Gets the targetImageLabel
     * @return the targetImageLabel
     */
    public TopologyLabel getTargetImageLabel() {
        return targetImageLabel;
    }

    /**Sets the targetImageLabel
     * @param targetImageLabel the targetImageLabel
     */
    public void setTargetImageLabel(TopologyLabel targetImageLabel) {
        this.targetImageLabel = targetImageLabel;
    }

     /**Creates the ok button
     * @return the ok button
     */
    public JButton createOkButton(){
        JButton b = new JButton("Ok");
        b.setActionCommand(Record.MAKE_OVERLAY_REGION_DIALOG_OK_ID);
        b.addActionListener((Record) getController());
        b.setToolTipText("Create new region");
        return b;
    }

    /**Gets the ok button
    * @return the ok button
    */
    public JButton getOkButton() {
        return okButton;
    }

    /**Sets the ok button
    * @param okButton the ok button
    */
    public void setOkButton(JButton okButton) {
        this.okButton = okButton;
    }

    /**Creates the overlay region target image combobox
     * @return the overlay region target image combobox
     */
    public JComboBox createTargetImageComboBox(){
        JComboBox targetImageComboBox = new JComboBox();
        if (((RecordModel) getModel()).getFileNames() != null) {
            targetImageComboBox = new JComboBox(((RecordModel) getModel()).getFileNames());
        }
        targetImageComboBox.setToolTipText("Choose image");
        return targetImageComboBox;
    }

    /**Gets the overlay region target image combobox
     * @return the overlay region target image combobox
     */
    public JComboBox getTargetImageComboBox() {
        return targetImageComboBox;
    }

    /**Sets the overlay region target image combobox
     * @param targetImageComboBox the overlay region target image combobox
     */
    public void setTargetImageComboBox(JComboBox targetImageComboBox) {
        this.targetImageComboBox = targetImageComboBox;
    }

    /**Gets the overlayNumberLabel
     * @return the overlayNumberLabel
     */
    public TopologyLabel getOverlayRegionNumberLabel() {
        return polygonNumberLabel;
    }

    /**Sets the overlayNumberLabel
     * @param overlayNumberLabel the overlayNumberLabel
     */
    public void setOverlayRegionNumberLabel(TopologyLabel polygonNumberLabel) {
        this.polygonNumberLabel = polygonNumberLabel;
    }

    /**Gets the overlayNumberTextField
     * @return the overlayNumberLabel
     */
    public JTextField getOverlayRegionNumberTextField() {
        return polygonNumberTextField;
    }

    /**Sets the overlayNumberTextField
     * @param overlayNumberLabel the overlayNumberLabel
     */
    public void setOverlayRegionNumberTextField(JTextField polygonNumberTextField) {
        this.polygonNumberTextField = polygonNumberTextField;
    }

    /**Creates the left neighbour label
     * @return the left neighbour label
     */
    public TopologyLabel createRightNeighbourLabel() {
        TopologyLabel l = new TopologyLabel(false);
        l.setText("Right Neighbour");
        return l;
    }

     /**Gets the right neighbour label
     * @return the right neighbour label
     */
    public TopologyLabel getRightNeighbourLabel() {
        return rightNeighbourLabel;
    }

    /**Sets the right neighbour label
     * @param rightNeighBourLabel the right neighbour label
     */
    public void setRightNeighbourLabel(TopologyLabel rightNeighbourLabel) {
        this.rightNeighbourLabel = rightNeighbourLabel;
    }

    /**Creates the left neighbour label
     * @return the left neighbour label
     */
    public TopologyLabel createLeftNeighbourLabel() {
        TopologyLabel l = new TopologyLabel(false);
        l.setText("Left Neighbour");
        return l;
    }

    /**Gets the left neighbour label
     * @return the left neighbour label
     */
    public TopologyLabel getLeftNeighbourLabel() {
        return leftNeighbourLabel;
    }

    /**Sets the left neighbour label
     * @param leftNeighBourLabel the left neighbour label
     */
    public void setLeftNeighbourLabel(TopologyLabel leftNeighbourLabel) {
        this.leftNeighbourLabel = leftNeighbourLabel;
    }

    /**Gets the top neighbour label
     * @return the top neighbour label
     */
    public TopologyLabel getTopNeighbourLabel() {
        return topNeighbourLabel;
    }

    public TopologyLabel createTopNeighbourLabel(){
         TopologyLabel topNeighbourLabel= new TopologyLabel(false);
         topNeighbourLabel.setText("Top Neighbour ");
         return topNeighbourLabel;
    }

     /**Sets the top neighbour label
     * @param topNeighBourLabel the top neighbour label
     */
    public void setTopNeighbourLabel(TopologyLabel topNeighbourLabel) {
        this.topNeighbourLabel = topNeighbourLabel;
    }

    /**Creates the bottom neighbour label
     * @return the bottom neighbour label
     */
    public TopologyLabel createBottomNeighbourLabel(){
        TopologyLabel bottomNeighbourLabel = new TopologyLabel(false);
        bottomNeighbourLabel.setText("Bottom neighbour");
        return bottomNeighbourLabel;
    }

    /**Gets the bottom neighbour label
     * @return the bottom neighbour label
     */
    public TopologyLabel getBottomNeighbourLabel() {
        return bottomNeighbourLabel;
    }

    /**Sets the bottom neighbour label
     * @param bottomNeighBourLabel the bottom neighbour label
     */
    public void setBottomNeighbourLabel(TopologyLabel bottomNeighbourLabel) {
        this.bottomNeighbourLabel = bottomNeighbourLabel;
    }

    /**Gets the right neighbour comboBox
     * @return rightNeighBourComboBox the right neighbour comboBox
     */
    public JComboBox getRightNeighbourComboBox() {
        return rightNeighbourComboBox;
    }

    /**Creates the right neighbour comboBox
     * @return the right neighbour comboBox
     */
    public JComboBox createRightNeighbourComboBox(){
        JComboBox rightNeighbourComboBox = new JComboBox();
        rightNeighbourComboBox.setEnabled(false);
        return rightNeighbourComboBox;
    }

    /**Sets the right neighbour comboBox
     * @param rightNeighBourComboBox the right neighbour comboBox
     */
    public void setRightNeighbourComboBox(JComboBox rightNeighbourComboBox) {
        this.rightNeighbourComboBox = rightNeighbourComboBox;
    }

    /**Creates the right neighbour checkBox
     * @return the right neighbour checkBox
     */
    public JCheckBox createRightNeighbourCheckBox(){
        JCheckBox rightNeighbourCheckBox = new JCheckBox();
        rightNeighbourCheckBox.addItemListener((Record) getController());
        return rightNeighbourCheckBox;
    }

    /**Gets the right neighbour comboBox
    * @return rightNeighBourComboBox the right neighbour comboBox
    */
    public JCheckBox getRightNeighbourCheckBox() {
        return rightNeighbourCheckBox;
    }

    /**Sets the right neighbour comboBox
     * @param rightNeighBourComboBox the right neighbour comboBox
     */
    public void setRightNeighbourCheckBox(JCheckBox rightNeighbourCheckBox) {
        this.rightNeighbourCheckBox = rightNeighbourCheckBox;
    }

     /**Creates the left neighbour comboBox
     * @return the left neighbour comboBox
     */
    public JComboBox createLeftNeighbourComboBox(){
        JComboBox leftNeighbourComboBox = new JComboBox();
        leftNeighbourComboBox.setEnabled(false);
         return leftNeighbourComboBox;
    }

     /**Gets the left neighbour comboBox
     * @return the left neighbour comboBox
     */
    public JComboBox getLeftNeighbourComboBox() {
        return leftNeighbourComboBox;
    }

     /**Sets the left neighbour comboBox
     * @param leftNeighBourComboBox the left neighbour comboBox
     */
    public void setLeftNeighbourComboBox(JComboBox leftNeighbourComboBox) {
        this.leftNeighbourComboBox = leftNeighbourComboBox;
    }

    /**Creates the left neighbour checkBox
     * @return the left neighbour checkBox
     */
    public JCheckBox createLeftNeighbourCheckBox(){
        JCheckBox leftNeighbourCheckBox = new JCheckBox();
        leftNeighbourCheckBox.addItemListener((Record) getController());
        return leftNeighbourCheckBox;
    }

    /**Gets the left neighbour checkBox
     * @return the left neighbour checkBox
     */
    public JCheckBox getLeftNeighbourCheckBox() {
        return leftNeighbourCheckBox;
    }

    /**Sets the left neighbour checkBox
     * @param leftNeighBourCheckBox the left neighbour checkBox
     */
    public void setLeftNeighbourCheckBox(JCheckBox leftNeighbourCheckBox) {
        this.leftNeighbourCheckBox = leftNeighbourCheckBox;
    }

    /**Creates the top neighbour comboBox
     * @return the top neighbour comboBox
     */
    public JComboBox createTopNeighbourComboBox(){
        JComboBox topNeighbourComboBox = new JComboBox();
        topNeighbourComboBox.setMinimumSize(new Dimension(100,50));
        topNeighbourComboBox.setEnabled(false);
        return topNeighbourComboBox;
    }

    /**Gets the top neighbour comboBox
     * @return the top neighbour comboBox
     */
    public JComboBox getTopNeighbourComboBox() {
        return topNeighbourComboBox;
    }

     /**Sets the top neighbour comboBox
     * @param topNeighBourComboBox the top neighbour comboBox
     */
    public void setTopNeighbourComboBox(JComboBox topNeighbourComboBox) {
        this.topNeighbourComboBox = topNeighbourComboBox;
    }

     /**Creates the top neighbour checkBox
     * @return the top neighbour checkBox
     */
    public JCheckBox createTopNeighbourCheckBox(){
        JCheckBox c = new JCheckBox();
        c.addItemListener((Record) getController());
        return c;
    }

    /**Gets the top neighbour checkBox
     * @return the top neighbour checkBox
     */
    public JCheckBox getTopNeighbourCheckBox() {
        return topNeighbourCheckBox;
    }

    /**Sets the top neighbour checkBox
     * @param topNeighBourCheckBox the top neighbour checkBox
     */
    public void setTopNeighbourCheckBox(JCheckBox topNeighbourCheckBox) {
        this.topNeighbourCheckBox = topNeighbourCheckBox;
    }

    /**Creates the bottom neighbour comboBox
     * @return the bottom neighbour comboBox
     */
    public JComboBox createBottomNeighbourComboBox(){
        JComboBox c = bottomNeighbourComboBox = new JComboBox();
        c.setEnabled(false);
        return c;
    }

    /**Gets the bottom neighbour comboBox
     * @return the bottom neighbour comboBox
     */
    public JComboBox getBottomNeighbourComboBox() {
        return bottomNeighbourComboBox;
    }

    /**Sets the bottom neighbour comboBox
     * @param bottomNeighBourComboBox the bottom neighbour comboBox
     */
    public void setBottomNeighbourComboBox(JComboBox bottomNeighbourComboBox) {
        this.bottomNeighbourComboBox = bottomNeighbourComboBox;
    }

    /**Creates the bottom neighbour checkBox
     * @return the bottom neighbour checkBox
     */
    public JCheckBox createBottomNeighbourCheckBox(){
        JCheckBox bottomNeighbourCheckBox = new JCheckBox();
        bottomNeighbourCheckBox.addItemListener((Record) getController());
        return bottomNeighbourCheckBox;
    }

    /**Gets the bottom neighbour checkBox
     * @return the bottom neighbour checkBoox
     */
    public JCheckBox getBottomNeighbourCheckBox() {
        return bottomNeighbourCheckBox;
    }

    /**Sets the bottom neighbour checkBox
     * @param bottomNeighBourComboBox the bottom neighbour checkBox
     */
    public void setBottomNeighbourCheckBox(JCheckBox bottomNeighbourCheckBox) {
        this.bottomNeighbourCheckBox = bottomNeighbourCheckBox;
    }

    /**Creates the neighbour panel.
     * @return the neighbour panel
     */
    public ResPanel createNeighbourPanel() {
        ResPanel p = new ResPanel();
        if (p != null) {
            p.setBorder(BorderFactory.createTitledBorder("Neighbours"));
            GridBagLayout neighbourPanelLayout = new GridBagLayout();
            GridBagConstraints c = new GridBagConstraints();
            p.setLayout(neighbourPanelLayout);
            setTopNeighbourPanel(createTopNeighbourPanel());
            setBottomNeighbourPanel(createBottomNeighbourPanel());
            setLeftNeighbourPanel(createLeftNeighbourPanel());
            setRightNeighbourPanel(createRightNeighbourPanel());
            c.anchor = GridBagConstraints.WEST;
            c.insets = new Insets(2, 2, 2, 2);
            c.gridwidth = 1;
            c.gridx = 0;
            c.gridy = 0;
            neighbourPanelLayout.setConstraints(getTopNeighbourPanel(), c);
            p.add(getTopNeighbourPanel());
            c.anchor = GridBagConstraints.WEST;
            c.insets = new Insets(2, 2, 2, 2);
            c.gridwidth = 1;
            c.gridx = 0;
            c.gridy = 1;
            neighbourPanelLayout.setConstraints(getBottomNeighbourPanel(), c);
            p.add(getBottomNeighbourPanel());
            c.anchor = GridBagConstraints.WEST;
            c.insets = new Insets(2, 2, 2, 2);
            c.gridwidth = 1;
            c.gridx = 0;
            c.gridy = 2;
            neighbourPanelLayout.setConstraints(getLeftNeighbourPanel(), c);
            p.add(getLeftNeighbourPanel());
            c.anchor = GridBagConstraints.WEST;
            c.insets = new Insets(2, 2, 2, 2);
            c.gridwidth = 1;
            c.gridx = 0;
            c.gridy = 3;
            neighbourPanelLayout.setConstraints(getRightNeighbourPanel(), c);
            p.add(getRightNeighbourPanel());
        } else {
            throw new NullPointerException("Could not create NeighbourPanel. The panel is null");
        }
        return p;
    }

    /**Gets the neighbour panel
     * @return the neighbour panel
     */
    public ResPanel getNeighbourPanel() {
        return neighbourPanel;
    }

    /**Sets the neighbour panel
     * @return neighbourPanel the neighbour panel
     */
    public void setNeighbourPanel(ResPanel neighbourPanel) {
        this.neighbourPanel = neighbourPanel;
    }

     /**Creates the top neighbour panel
     * @return the top neighbour panel
     */
    public ResPanel createTopNeighbourPanel() {
        BorderLayout f = new BorderLayout();
        ResPanel tnp = new ResPanel();
        tnp.setPreferredSize(new Dimension(550,50));
        tnp.setLayout(f);
        if (tnp != null) {
            tnp.setBorder(BorderFactory.createTitledBorder("Top"));
            setTopNeighbourLabel(createTopNeighbourLabel());
            setTopNeighbourComboBox(createTopNeighbourComboBox());
            setTopNeighbourCheckBox(createTopNeighbourCheckBox());
            tnp.add(getTopNeighbourCheckBox(),BorderLayout.WEST);
            tnp.add(getTopNeighbourLabel(),BorderLayout.CENTER);
            tnp.add(getTopNeighbourComboBox(),BorderLayout.EAST);
        } else {
            throw new NullPointerException("Could not create topNeighbourPanel. The panel is null");
        }
        return tnp;
    }

    /**Gets the top neighbour panel
     * @return the top neighbour panel
     */
    public ResPanel getTopNeighbourPanel() {
        return topNeighbourPanel;
    }

    /**Sets the top neighbour panel
     * @rparam topNeighbourPanel the top neighbour panel
     */
    public void setTopNeighbourPanel(ResPanel topNeighbourPanel) {
        this.topNeighbourPanel = topNeighbourPanel;
    }

     /**Creates the bottom neighbour panel
     * @return the bottom neighbour panel
     */
    public ResPanel createBottomNeighbourPanel() {
        BorderLayout f = new BorderLayout();
        ResPanel bnp = new ResPanel();
        bnp.setPreferredSize(new Dimension(550,50));
        bnp.setLayout(f);
        if (bnp != null) {
            bnp.setBorder(BorderFactory.createTitledBorder("Bottom"));
            setBottomNeighbourLabel(createBottomNeighbourLabel());
            setBottomNeighbourComboBox(createBottomNeighbourComboBox());
            setBottomNeighbourCheckBox(createBottomNeighbourCheckBox());
            bnp.add(getBottomNeighbourCheckBox(),BorderLayout.WEST);
            bnp.add(getBottomNeighbourLabel(),BorderLayout.CENTER);
            bnp.add(getBottomNeighbourComboBox(),BorderLayout.EAST);
        } else {
            throw new NullPointerException("Could not create bottomNeighbourPanel. The panel is null");
        }
        return bnp;
    }

    /**Gets the bottom neighbour panel
     * @return the bottom neighbour panel
     */
    public ResPanel getBottomNeighbourPanel() {
        return bottomNeighbourPanel;
    }

    /**Sets the bottom neighbour panel
     * @param bottomNeighbourPanel the bottom neighbour panel
     */
    public void setBottomNeighbourPanel(ResPanel bottomNeighbourPanel) {
        this.bottomNeighbourPanel = bottomNeighbourPanel;
    }

    /**Creates the left neighbour panel
     * @return the left neighbour panel
     */
    public ResPanel createLeftNeighbourPanel() {
        BorderLayout f = new BorderLayout();
        ResPanel lnp = new ResPanel();
        lnp.setPreferredSize(new Dimension(550,50));
        lnp.setLayout(f);
        if (lnp != null) {
            lnp.setBorder(BorderFactory.createTitledBorder("Left"));
            setLeftNeighbourLabel(createLeftNeighbourLabel());
            setLeftNeighbourComboBox(createLeftNeighbourComboBox());
            setLeftNeighbourCheckBox(createLeftNeighbourCheckBox());
            lnp.add(getLeftNeighbourCheckBox(),BorderLayout.WEST);
            lnp.add(getLeftNeighbourLabel(),BorderLayout.CENTER);
            lnp.add(getLeftNeighbourComboBox(),BorderLayout.EAST);
        } else {
            throw new NullPointerException("Could not create leftNeighbourPanel. The panel is null");
        }
        return lnp;
    }

     /**Gets the left neighbour panel
     * @return the left neighbour panel
     */
    public ResPanel getLeftNeighbourPanel() {
        return leftNeighbourPanel;
    }

    /**Sets the left neighbour panel
     * @param leftNeighbourPanel the left neighbour panel
     */
    public void setLeftNeighbourPanel(ResPanel leftNeighbourPanel) {
        this.leftNeighbourPanel = leftNeighbourPanel;
    }

    /**Creates the right neighbour panel
     * @return the right neighbour panel
     */
    public ResPanel createRightNeighbourPanel() {
        BorderLayout f = new BorderLayout();
        ResPanel rnp = new ResPanel();
        rnp.setPreferredSize(new Dimension(550,50));
        rnp.setLayout(f);
        if (rnp != null) {
            rnp.setBorder(BorderFactory.createTitledBorder("Right"));
            setRightNeighbourLabel(createRightNeighbourLabel());
            setRightNeighbourComboBox(createRightNeighbourComboBox());
            setRightNeighbourCheckBox(createRightNeighbourCheckBox());
            rnp.add(getRightNeighbourCheckBox(),BorderLayout.WEST);
            rnp.add(getRightNeighbourLabel(),BorderLayout.CENTER);
            rnp.add(getRightNeighbourComboBox(),BorderLayout.EAST);
        } else {
            throw new NullPointerException("Could not create leftNeighbourPanel. The panel is null");
        }
        return rnp;
    }

    /**Gets the right neighbour panel
     * @return the right neighbour panel
     */
    public ResPanel getRightNeighbourPanel() {
        return rightNeighbourPanel;
    }

     /**Sets the right neighbour panel
     * @param rightNeighbourPanel the right neighbour panel
     */
    public void setRightNeighbourPanel(ResPanel rightNeighbourPanel) {
        this.rightNeighbourPanel = rightNeighbourPanel;
    }

    /**Creates the generaInformationPanel
     * @return the generalInformationPanel
     */
    public ResPanel createGeneralRegionInformationPanel() {
        ResPanel p = new ResPanel();
        if (p != null) {
            p.setBorder(BorderFactory.createTitledBorder("General information"));
            GridBagLayout generalRegionInformationLayout = new GridBagLayout();
            GridBagConstraints c = new GridBagConstraints();
            setOverlayRegionNameLabel(new TopologyLabel(false));
            getOverlayRegionNameLabel().setText("Name");
            setOverlayRegionNameTextField(new JTextField(12));
            getOverlayRegionNameTextField().setEnabled(true);
            getOverlayRegionNameTextField().setPreferredSize(new Dimension(200, 20));
            getOverlayRegionNameTextField().setMinimumSize(new Dimension(200, 20));
            getOverlayRegionNameTextField().setMaximumSize(new Dimension(200, 20));
            setOverlayRegionNumberLabel(new TopologyLabel(false));
            getOverlayRegionNumberLabel().setText("Nr.");
            setOverlayRegionNumberTextField(new JTextField(6));
            getOverlayRegionNumberTextField().setEnabled(false);
            //getOverlayRegionNumberTextField().setPreferredSize(new Dimension(30, 20));
            c.anchor = GridBagConstraints.WEST;
            c.fill = GridBagConstraints.HORIZONTAL;
            c.insets = new Insets(2, 2, 2, 2);
            c.gridwidth = 1;
            c.gridx = 0;
            c.gridy = 0;
            generalRegionInformationLayout.setConstraints(getOverlayRegionNameLabel(), c);
            p.add(getOverlayRegionNameLabel());
            c.anchor = GridBagConstraints.EAST;
            c.insets = new Insets(2, 2, 2, 2);
            c.gridwidth = 1;
            c.ipady = 30;
            c.gridx = 0;
            c.gridy = 1;
            generalRegionInformationLayout.setConstraints(getOverlayRegionNameTextField(), c);
            p.add(getOverlayRegionNameTextField());
            c.anchor = GridBagConstraints.WEST;
            c.insets = new Insets(2, 2, 2, 2);
            c.gridwidth = 1;
            c.gridx = 1;
            c.gridy = 0;
            generalRegionInformationLayout.setConstraints(getOverlayRegionNumberLabel(), c);
            p.add(getOverlayRegionNumberLabel());
            c.anchor = GridBagConstraints.EAST;
            c.insets = new Insets(2, 2, 2, 2);
            c.gridwidth = 1;
            c.gridx = 1;
            c.gridy = 1;
            generalRegionInformationLayout.setConstraints(getOverlayRegionNumberTextField(), c);
            p.add(getOverlayRegionNumberTextField());
        }
        return p;
    }

    /**Gets the generaInformationPanel
     * @return the generalInformationPanel
     */
    public ResPanel getGeneralRegionInformationPanel() {
        return generalRegionInformationPanel;
    }

    /**Sets the generaInformationPanel
     * @param generalInformationPanel the generalInformationPanel
     */
    public void setGeneralRegionInformationPanel(ResPanel generalInformationPanel) {
        this.generalRegionInformationPanel = generalInformationPanel;
    }

    /**Creates the targetOverlayPanel
     * @return the targetOverlayPanel
     */
    public ResPanel createTargetOverlayPanel() {
        ResPanel p = new ResPanel();
        if (p != null) {
            p.setBorder(BorderFactory.createTitledBorder("Target Overlay"));
            GridBagLayout targetLayout = new GridBagLayout();
            GridBagConstraints c = new GridBagConstraints();
            setTargetImageLabel(createTargetImageLabel());
            getTargetImageLabel().setText("Name");
            getTargetImageLabel().setMinimumSize(new Dimension(100, 50));
            getTargetImageLabel().setPreferredSize(new Dimension(80, 50));
            if (((RecordModel) getModel()).getFileNames() != null) {
                setTargetImageComboBox(new JComboBox(((RecordModel) getModel()).getFileNames()));
            } else
                setTargetImageComboBox(new JComboBox());
            getTargetImageComboBox().setToolTipText("Target image auswhlen");
            getTargetImageComboBox().addActionListener((Record) getController());
            getTargetImageComboBox().setActionCommand(Record.MAKE_OVERLAY_REGION_DIALOG_CHANGE_PREVIEW_ID);
            setTargetPreviewPanel(createTargetPreviewPanel());
            c.anchor = GridBagConstraints.WEST;
            c.insets = new Insets(2, 2, 2, 2);
            c.gridwidth = 1;
            c.gridx = 0;
            c.gridy = 0;
            targetLayout.setConstraints(getTargetImageLabel(), c);
            p.add(getTargetImageLabel());
            c.anchor = GridBagConstraints.WEST;
            c.insets = new Insets(2, 2, 2, 2);
            c.gridwidth = 1;
            c.gridx = 0;
            c.gridy = 1;
            targetLayout.setConstraints(getTargetImageComboBox(), c);
            p.add(getTargetImageComboBox());
            c.anchor = GridBagConstraints.WEST;
            c.insets = new Insets(2, 2, 2, 2);
            c.gridwidth = 1;
            c.gridx = 0;
            c.gridy = 2;
            targetLayout.setConstraints(getTargetPreviewPanel(), c);
            p.add(getTargetPreviewPanel());
        }
        return p;
    }

     /**Gets the targetOverlayPanel
     * @return the targetOverlayPanel
     */
    public ResPanel getTargetOverlayPanel() {
        return targetOverlayPanel;
    }

     /**Sets the targetOverlayPanel
     * @param targetOverlayPanel the targetOverlayPanel
     */
    public void setTargetOverlayPanel(ResPanel targetOverlayPanel) {
        this.targetOverlayPanel = targetOverlayPanel;
    }

    /**Gets the informationTextArea
     * @return the informationTextArea
     */
    public JTextArea getInformationTextArea() {
        return this.informationTextArea;
    }

    /**Sets the informationTextArea
     * @return informationTextArea the informationTextArea
     */
    public void setInformationTextArea(JTextArea informationTextArea) {
        this.informationTextArea = informationTextArea;
    }

    /**Creates the targetPreviewPanel
     * @return the targetPreviewPanel
     */
    public PreviewPanel createTargetPreviewPanel() {
        PreviewPanel p = new PreviewPanel(new Dimension(100, 100));
        if (p != null) {
            p.initialize();
        } else {
            throw new NullPointerException("Could not create PreviewPanel for MakeOverlayRegionDialog. The panel is null");
        }
        return p;
    }

    /**Gets the targetPreviewPanel
     * @return the targetPreviewPanel
     */
    public PreviewPanel getTargetPreviewPanel() {
        return targetPreviewPanel;
    }

    /**Sets the targetPreviewPanel
     * @return targetPreviewPanel the targetPreviewPanel
     */
    public void setTargetPreviewPanel(PreviewPanel targetPreviewPanel) {
        this.targetPreviewPanel = targetPreviewPanel;
    }

    /**Creates the cancelButton
     * @returm the cancelButton
     * */
    public JButton createCancelButton(){
        JButton b = new JButton("Cancel");
        b.setActionCommand(Record.MAKE_OVERLAY_REGION_DIALOG_CANCEL_ID);
        b.addActionListener((Record) getController());
        b.setToolTipText("Cancel");
        return b;
    }

    /**Gets the cancelButton
     * @returm the cancelButton
     * */
    public JButton getCancelButton() {
        return cancelButton;
    }

    /**Sets the cancelButton
     * @returm cancelButton the cancelButton
     * */
    public void setCancelButton(JButton cancelButton) {
        this.cancelButton = cancelButton;
    }
}
