/*
 * $RCSfile: EditOverlayRegionDialog.java,v $
 *
 * Copyright (c) 1999-2002. Jens Bohl. All rights reserved.
 *
 * This software is published under the GPL GNU General Public License.
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * http://www.resmedicinae.org
 * - Information in Medicine -
 */

package org.resmedicinae.application.healthcare.record.topology;

import org.resmedicinae.application.healthcare.record.*;
import org.resmedicinae.resmedlib.component.control.*;
import org.resmedicinae.resmedlib.component.view.swing.*;

import javax.swing.*;
import java.awt.*;

/**
 * View Component. This dialog is for editing existing overlay regions. The name and target
 * can be changed. The overlay region can be removed as well. This dialog appears after clicking on an overlay region
 * in "edit overlay region mode".
 * @author Jens Bohl <info@jens-bohl-professional.de>
 */
public class EditOverlayRegionDialog extends SwingView {
    /** Layout Manager of this dialog */
    private GridBagLayout gridbag;
    /** Name label of polygon */
    private TopologyLabel nameLabel;
    /** Name textField of the overlay region */
    private JTextField nameTextField;
    /** Target label of the polygon */
    private TopologyLabel targetImageLabel;
    /** Ok Button */
    private JButton okButton;
    /** Delete Button */
    private JButton deleteOverlayRegionButton;
    /** ComboBox containing possibel targets. */
    private JComboBox targetImageComboBox;
    /** String containing the old name of the polygon */
    private String oldName;
    /**Right NeighbourLabel*/
    private TopologyLabel rightNeighbourLabel;
    /** Right neighbour combobox */
    private JComboBox rightNeighbourComboBox;
    /** Right neighbour checkbox */
    private JCheckBox rightNeighbourCheckBox;
    /** Left neighbour label */
    private TopologyLabel leftNeighbourLabel;
    /** Left neighbour combobox */
    private JComboBox leftNeighbourComboBox;
    /** Right neighbour checkbox */
    private JCheckBox leftNeighbourCheckBox;
    /** Top neighbour label */
    private TopologyLabel topNeighbourLabel;
    /** Top neighbour combobox */
    private JComboBox topNeighbourComboBox;
    /** Right neighbour checkbox */
    private JCheckBox topNeighbourCheckBox;
    /** Bottom neighbour label */
    private TopologyLabel bottomNeighbourLabel;
    /** Bottom neighbour combobox */
    private JComboBox bottomNeighbourComboBox;
    /** Right neighbour checkbox */
    private JCheckBox bottomNeighbourCheckBox;
    /** The neighbour panel. This panel conatins all the other neighbour panels.*/
    private ResPanel neighbourPanel;
    /** The top neighbour panel */
    private ResPanel topNeighbourPanel;
    /** The bottom neighbour panel */
    private ResPanel bottomNeighbourPanel;
    /** The left neighbour panel */
    private ResPanel leftNeighbourPanel;
    /** The right neighbour panel */
    private ResPanel rightNeighbourPanel;
    /** The model of MVC */
    private RecordModel model;
    /** Default Constructor */
    public EditOverlayRegionDialog() {
    }

    /** Sets the overlay region name in the textField as well as the oldname oldName. The old name of the overlay
     * region can then be modified.
     * @param overlayRegioName the old name of the overlayRegion
     * */
    public void setOverlayRegionName(String overlayRegioName) {
        nameTextField.setText(overlayRegioName);
        setOldName(overlayRegioName);
    }

    /** Sets the overlayRegion target in the comboBox
     * @param overlayRegionTarget the target of the overlayRegion
     * */
    public void setOverlayRegionTarget(String overlayRegionTarget) {
        targetImageComboBox.setSelectedItem(overlayRegionTarget);
    }

    /** Gets the old name of the overlayRegion. */
    public String getOldName() {
        return this.oldName;
    }

    /** Initialization of components */
    public void initialize() {
        setTitle("Edit overlay region...");
        setGridbag(new GridBagLayout());
        setLayout(getGridbag());

        setDisplayMode(DisplayFactory.MODELESS_DIALOG_DISPLAY);
        setDialogBounds(createDialogBounds());

        GridBagConstraints c = new GridBagConstraints();
        setNameLabel(createNameLabel());
        setNameTextField(createNameTextField());
        setTargetImageLabel(createTargetImageLabel());
        setTargetImageComboBox(createTargetImageComboBox());
        setNeighbourPanel(createNeighbourPanel());
        setOkButton(createOkButton());
        setDeleteOverlayRegionButton(createDeleteOverlayRegionButton());

        c.fill = GridBagConstraints.HORIZONTAL;
        c.anchor = GridBagConstraints.WEST;
        c.insets = new Insets(10, 10, 2, 10);
        c.gridx = 0;
        c.gridy = 0;
        gridbag.setConstraints(getNameLabel(), c);
        add(getNameLabel());

        c.anchor = GridBagConstraints.WEST;
        c.insets = new Insets(10, 10, 2, 10);
        c.gridwidth = 1;
        c.gridx = 1;
        c.gridy = 0;
        gridbag.setConstraints(getNameTextField(), c);
        add(getNameTextField());

        c.anchor = GridBagConstraints.WEST;
        c.insets = new Insets(10, 10, 2, 10);
        c.gridwidth = 1;
        c.gridx = 0;
        c.gridy = 1;
        gridbag.setConstraints(getTargetImageLabel(), c);
        add(getTargetImageLabel());

        c.anchor = GridBagConstraints.WEST;
        c.insets = new Insets(10, 10, 2, 10);
        c.gridwidth = 1;
        c.gridx = 1;
        c.gridy = 1;
        gridbag.setConstraints(getTargetImageComboBox(), c);
        add(getTargetImageComboBox());

        c.anchor = GridBagConstraints.EAST;
        c.insets = new Insets(10, 10, 10, 10);
        c.gridwidth = 2;
        c.gridx = 0;
        c.gridy = 2;
        gridbag.setConstraints(getNeighbourPanel(), c);
        add(getNeighbourPanel());

        c.anchor = GridBagConstraints.EAST;
        c.insets = new Insets(10, 10, 10, 10);
        c.gridwidth = 1;
        c.gridx = 1;
        c.gridy = 3;
        gridbag.setConstraints(getDeleteOverlayRegionButton(), c);
        add(getDeleteOverlayRegionButton());

        c.anchor = GridBagConstraints.WEST;
        c.insets = new Insets(10, 10, 10, 10);
        c.gridx = 0;
        c.gridy = 3;
        gridbag.setConstraints(getOkButton(), c);
        add(getOkButton());

    }

     /**
     * Creates a rectangle storing the dialog bounds.
     * @return the rectangle storing the dialog bounds
     */
    public Rectangle createDialogBounds() {
        // Determine width.
        int screenWidth = (int) Math.floor(Toolkit.getDefaultToolkit().getScreenSize().getWidth());
        int w = 600;
        if (w > screenWidth) {
            w = screenWidth;
        }
        // Use centered x position by default.
        int x = (screenWidth - w) / 2;
        // Determine height.
        int screenHeight = (int) Math.floor(Toolkit.getDefaultToolkit().getScreenSize().getHeight());
        int h = 500;
        if (h > screenHeight) {
            h = screenHeight;
        }
        // Use centered y position by default.
        int y = (screenHeight - h) / 2;
        return new Rectangle(x, y, w, h);
    }

    /**
     * Creates a display mode.
     * @return the display mode
     */
    public int createDisplayMode() {
        return DisplayFactory.MODELESS_DIALOG_DISPLAY;
    }

    /**
     * Creates a control issued when the window containing this view is closed.
     * @return the control issued when the window containing this view is closed
     * @exception NullPointerException if the control is null
     */
    public Control createCloseControl() throws NullPointerException {
        Control c = new Control();
        if (c != null) {
            c.setId(Record.DESTROY_LOAD_PATIENT_DIALOG_CONTROL_ID);
            c.setSender((Controller) getController());
        } else {
            throw new NullPointerException("Could not create close control. The control is null.");
        }
        return c;
    }

    /** Finalizes this component. */
    public void finalizz() throws Exception {
        try {
            // Set parameters.
            setBorder(null);
        } finally {
            super.finalizz();
        }
    }
    /** Gets the layout manager
     * @return the layoutmanager
     */
    public GridBagLayout getGridbag() {
        return gridbag;
    }

    /**Sets the layout manager
     * @param gridbag the layout manager
     */
    public void setGridbag(GridBagLayout gridbag) {
        this.gridbag = gridbag;
    }

    /**Creates the overlay region name label
     * @return the name label
     */
    public TopologyLabel createNameLabel(){
        TopologyLabel l = new TopologyLabel(false);
        l.setText("Name");
        l.setMinimumSize(new Dimension(100, 50));
        l.setPreferredSize(new Dimension(100, 50));
        return l;
    }

    /**Gets the overlay region name label
     * @return the name label
     */
    public TopologyLabel getNameLabel() {
        return nameLabel;
    }

    /**Sets the overlay region name label
     * @param nameLabel the name label
     */
    public void setNameLabel(TopologyLabel nameLabel) {
        this.nameLabel = nameLabel;
    }

    /**Creates the overlay region name taext field
     * @return the name text field
     */
    public JTextField createNameTextField(){
        JTextField nameTextField = new JTextField(15);
        nameTextField.setEditable(true);
        nameTextField.setEnabled(true);
        nameTextField.setMinimumSize(new Dimension(250, 20));
        nameTextField.setPreferredSize(new Dimension(250, 20));
        return nameTextField;
    }

    /**Gets the overlay region name taext field
    * @return the name text field
    */
    public JTextField getNameTextField() {
        return nameTextField;
    }

    /**Sets the overlay region name taext field
    * @param nameTextField the name text field
    */
    public void setNameTextField(JTextField nameTextField) {
        this.nameTextField = nameTextField;
    }

    /**Creates the overlay region target label
     * @return the target label
     */
    public TopologyLabel createTargetImageLabel(){
        TopologyLabel targetImageLabel = new TopologyLabel(false);
        targetImageLabel.setText("Target overlay");
        targetImageLabel.setMinimumSize(new Dimension(100, 50));
        targetImageLabel.setPreferredSize(new Dimension(80, 50));
        return targetImageLabel;
    }

    /**Gets the overlay region target label
     * @return the target label
     */
    public TopologyLabel getTargetImageLabel() {
        return targetImageLabel;
    }

    /**Sets the overlay region target label
     * @param targetImageLabel the target label
     */
    public void setTargetImageLabel(TopologyLabel targetImageLabel) {
        this.targetImageLabel = targetImageLabel;
    }

    /**Creates the ok button
     * @return the ok button
     */
    public JButton createOkButton(){
        JButton okButton = new JButton("Ok");
        okButton.setActionCommand(Record.EDIT_OVERLAY_REGION_DIALOG_OK_ID);
        okButton.addActionListener((Record) getController());
        okButton.setToolTipText("Confirm");
        return okButton;
    }

    /**Gets the ok button
    * @return the ok button
    */
    public JButton getOkButton() {
        return okButton;
    }

    /**Sets the ok button
    * @return okButton the ok button
    */
    public void setOkButton(JButton okButton) {
        this.okButton = okButton;
    }

    /**Creates the delete overlay region button
     * @return the delet overlay region button
     */
    public JButton createDeleteOverlayRegionButton(){
        JButton deleteOverlayRegionButton = new JButton("Remove overlay region");
        deleteOverlayRegionButton.setActionCommand(Record.EDIT_OVERLAY_REGION_DIALOG_DELETE_POLYGON_ID);
        deleteOverlayRegionButton.addActionListener((Record) getController());
        return deleteOverlayRegionButton;
    }

    /**Gets the delete overlay region button
     * @return the delet overlay region button
     */
    public JButton getDeleteOverlayRegionButton() {
        return deleteOverlayRegionButton;
    }

    /**Sets the delete overlay region button
     * @param deleteOverlayRegionButton the delet overlay region button
     */
    public void setDeleteOverlayRegionButton(JButton deleteOverlayRegionButton) {
        this.deleteOverlayRegionButton = deleteOverlayRegionButton;
    }

    /**Creates the overlay region target image combobox
     * @return the overlay region target image combobox
     */
    public JComboBox createTargetImageComboBox(){
        JComboBox targetImageComboBox = new JComboBox();
        if (((RecordModel) getModel()).getFileNames() != null) {
            targetImageComboBox = new JComboBox(((RecordModel) getModel()).getFileNames());
        }
        targetImageComboBox.setToolTipText("Choose image");
        return targetImageComboBox;
    }

    /**Gets the overlay region target image combobox
     * @return the overlay region target image combobox
     */
    public JComboBox getTargetImageComboBox() {
        return targetImageComboBox;
    }

    /**Sets the overlay region target image combobox
     * @param targetImageComboBox the overlay region target image combobox
     */
    public void setTargetImageComboBox(JComboBox targetImageComboBox) {
        this.targetImageComboBox = targetImageComboBox;
    }

    /**Creates the old name for the overlay region
     * @return the name label
     */
    public String createOldName(){
        return new String();
    }


    /**Sets the old name for the selected overlay region
     * @param oldName the old name
     */
    public void setOldName(String oldName) {
        this.oldName = oldName;
    }

    /**Creates the neighbour panel.
     * @return the neighbour panel
     */
    public ResPanel createNeighbourPanel() {
        ResPanel p = new ResPanel();
        if (p != null) {
            p.setBorder(BorderFactory.createTitledBorder("Neighbours"));
            GridBagLayout neighbourPanelLayout = new GridBagLayout();
            GridBagConstraints c = new GridBagConstraints();
            p.setLayout(neighbourPanelLayout);
            setTopNeighbourPanel(createTopNeighbourPanel());
            setBottomNeighbourPanel(createBottomNeighbourPanel());
            setLeftNeighbourPanel(createLeftNeighbourPanel());
            setRightNeighbourPanel(createRightNeighbourPanel());
            c.anchor = GridBagConstraints.WEST;
            c.fill = GridBagConstraints.HORIZONTAL;
            c.insets = new Insets(2, 2, 2, 2);
            c.gridwidth = 1;
            c.gridx = 0;
            c.gridy = 0;
            neighbourPanelLayout.setConstraints(getTopNeighbourPanel(), c);
            p.add(getTopNeighbourPanel());
            c.anchor = GridBagConstraints.WEST;
            c.insets = new Insets(2, 2, 2, 2);
            c.gridwidth = 1;
            c.gridx = 0;
            c.gridy = 1;
            neighbourPanelLayout.setConstraints(getBottomNeighbourPanel(), c);
            p.add(getBottomNeighbourPanel());
            c.anchor = GridBagConstraints.WEST;
            c.insets = new Insets(2, 2, 2, 2);
            c.gridwidth = 1;
            c.gridx = 0;
            c.gridy = 2;
            neighbourPanelLayout.setConstraints(getLeftNeighbourPanel(), c);
            p.add(getLeftNeighbourPanel());
            c.anchor = GridBagConstraints.WEST;
            c.insets = new Insets(2, 2, 2, 2);
            c.gridwidth = 1;
            c.gridx = 0;
            c.gridy = 3;
            neighbourPanelLayout.setConstraints(getRightNeighbourPanel(), c);
            p.add(getRightNeighbourPanel());
        } else {
            throw new NullPointerException("Could not create NeighbourPanel. The panel is null");
        }
        return p;
    }

    /**Gets the neighbour panel
     * @return the neighbour panel
     */
    public ResPanel getNeighbourPanel() {
        return neighbourPanel;
    }

    /**Sets the neighbour panel
     * @return neighbourPanel the neighbour panel
     */
    public void setNeighbourPanel(ResPanel neighbourPanel) {
        this.neighbourPanel = neighbourPanel;
    }

    /**Creates the top neighbour panel
     * @return the top neighbour panel
     */
    public ResPanel createTopNeighbourPanel() {
        BorderLayout f = new BorderLayout();
        ResPanel tnp = new ResPanel();
        tnp.setPreferredSize(new Dimension(550,50));
        tnp.setLayout(f);
        if (tnp != null) {
            tnp.setBorder(BorderFactory.createTitledBorder("Top"));
            setTopNeighbourLabel(createTopNeighbourLabel());
            setTopNeighbourComboBox(createTopNeighbourComboBox());
            setTopNeighbourCheckBox(createTopNeighbourCheckBox());
            tnp.add(getTopNeighbourCheckBox(), BorderLayout.WEST);
            tnp.add(getTopNeighbourLabel(), BorderLayout.CENTER);
            tnp.add(getTopNeighbourComboBox(), BorderLayout.EAST);
        } else {
            throw new NullPointerException("Could not create topNeighbourPanel. The panel is null");
        }
        return tnp;
    }

    /**Gets the top neighbour panel
     * @return the top neighbour panel
     */
    public ResPanel getTopNeighbourPanel() {
        return topNeighbourPanel;
    }

    /**Sets the top neighbour panel
     * @rparam topNeighbourPanel the top neighbour panel
     */
    public void setTopNeighbourPanel(ResPanel topNeighbourPanel) {
        this.topNeighbourPanel = topNeighbourPanel;
    }

    /**Creates the bottom neighbour panel
     * @return the bottom neighbour panel
     */
    public ResPanel createBottomNeighbourPanel() {
        BorderLayout f = new BorderLayout();
        ResPanel bnp = new ResPanel();
        bnp.setLayout(f);
        if (bnp != null) {
            bnp.setBorder(BorderFactory.createTitledBorder("Bottom"));
            setBottomNeighbourLabel(createBottomNeighbourLabel());
            setBottomNeighbourComboBox(createBottomNeighbourComboBox());
            setBottomNeighbourCheckBox(createBottomNeighbourCheckBox());
            bnp.add(getBottomNeighbourCheckBox(), BorderLayout.WEST);
            bnp.add(getBottomNeighbourLabel(), BorderLayout.CENTER);
            bnp.add(getBottomNeighbourComboBox(), BorderLayout.EAST);
        } else {
            throw new NullPointerException("Could not create bottomNeighbourPanel. The panel is null");
        }
        return bnp;
    }

    /**Gets the bottom neighbour panel
     * @return the bottom neighbour panel
     */
    public ResPanel getBottomNeighbourPanel() {
        return bottomNeighbourPanel;
    }

    /**Sets the bottom neighbour panel
     * @param bottomNeighbourPanel the bottom neighbour panel
     */
    public void setBottomNeighbourPanel(ResPanel bottomNeighbourPanel) {
        this.bottomNeighbourPanel = bottomNeighbourPanel;
    }

    /**Creates the left neighbour panel
     * @return the left neighbour panel
     */
    public ResPanel createLeftNeighbourPanel() {
        BorderLayout f = new BorderLayout();
        ResPanel lnp = new ResPanel();
        lnp.setLayout(f);
        if (lnp != null) {
            lnp.setBorder(BorderFactory.createTitledBorder("Left"));
            setLeftNeighbourLabel(createLeftNeighbourLabel());
            setLeftNeighbourComboBox(createLeftNeighbourComboBox());
            setLeftNeighbourCheckBox(createLeftNeighbourCheckBox());

            lnp.add(getLeftNeighbourCheckBox(), BorderLayout.WEST);
            lnp.add(getLeftNeighbourLabel(), BorderLayout.CENTER);
            lnp.add(getLeftNeighbourComboBox(), BorderLayout.EAST);
        } else {
            throw new NullPointerException("Could not create leftNeighbourPanel. The panel is null");
        }
        return lnp;
    }

    /**Gets the left neighbour panel
     * @return the left neighbour panel
     */
    public ResPanel getLeftNeighbourPanel() {
        return leftNeighbourPanel;
    }

    /**Sets the left neighbour panel
     * @param leftNeighbourPanel the left neighbour panel
     */
    public void setLeftNeighbourPanel(ResPanel leftNeighbourPanel) {
        this.leftNeighbourPanel = leftNeighbourPanel;
    }

    /**Creates the right neighbour panel
     * @return the right neighbour panel
     */
    public ResPanel createRightNeighbourPanel() {
        BorderLayout f = new BorderLayout();
        ResPanel rnp = new ResPanel();
        rnp.setLayout(f);
        if (rnp != null) {
            rnp.setBorder(BorderFactory.createTitledBorder("Right"));
            setRightNeighbourLabel(createRightNeighbourLabel());
            setRightNeighbourComboBox(createRightNeighbourComboBox());
            setRightNeighbourCheckBox(createRightNeighbourCheckBox());
            rnp.add(getRightNeighbourCheckBox(), BorderLayout.WEST);
            rnp.add(getRightNeighbourLabel(), BorderLayout.CENTER);
            rnp.add(getRightNeighbourComboBox(), BorderLayout.EAST);
        } else {
            throw new NullPointerException("Could not create leftNeighbourPanel. The panel is null");
        }
        return rnp;
    }

    /**Gets the right neighbour panel
     * @return the right neighbour panel
     */
    public ResPanel getRightNeighbourPanel() {
        return rightNeighbourPanel;
    }

    /**Sets the right neighbour panel
     * @param rightNeighbourPanel the right neighbour panel
     */
    public void setRightNeighbourPanel(ResPanel rightNeighbourPanel) {
        this.rightNeighbourPanel = rightNeighbourPanel;
    }

    /**ReInitializes the component
     * This is needed to reset the dialog properties in respect of the different overlay regions that can be choosed.
     * The different comboBoxes will be reset.
     * */
    public void reInitialize() {
        RecordModel model = (RecordModel) getModel();
        int currentRegionCount = model.getCurrentOverlayRegionNames().length;
        topNeighbourComboBox.removeAllItems();
        bottomNeighbourComboBox.removeAllItems();
        leftNeighbourComboBox.removeAllItems();
        rightNeighbourComboBox.removeAllItems();
        for (int i = 0; i < currentRegionCount; i++) {
            topNeighbourComboBox.addItem(model.getCurrentOverlayRegionNames()[i]);
            bottomNeighbourComboBox.addItem(model.getCurrentOverlayRegionNames()[i]);
            leftNeighbourComboBox.addItem(model.getCurrentOverlayRegionNames()[i]);
            rightNeighbourComboBox.addItem(model.getCurrentOverlayRegionNames()[i]);
        }
    }

    /**Creates the right neighbour Label
     * @return the right neighbour Label
     */
    public TopologyLabel createRightNeighbourLabel(){
        TopologyLabel rightNeighbourLabel = new TopologyLabel(false);
        rightNeighbourLabel.setText("Right neighbour");
        return rightNeighbourLabel;
    }

    /**Gets the right neighbour label
     * @return the right neighbour label
     */
    public TopologyLabel getRightNeighbourLabel() {
        return rightNeighbourLabel;
    }

    /**Sets the right neighbour label
     * @param rightNeighBourLabel the right neighbour label
     */
    public void setRightNeighbourLabel(TopologyLabel rightNeighbourLabel) {
        this.rightNeighbourLabel = rightNeighbourLabel;
    }

    /**Creates the right neighbour comboBox
     * @return the right neighbour comboBox
     */
    public JComboBox createRightNeighbourComboBox(){
        JComboBox rightNeighbourComboBox = new JComboBox();
        rightNeighbourComboBox.setEnabled(false);
        return rightNeighbourComboBox;
    }

    /**Gets the right neighbour comboBox
     * @return rightNeighBourComboBox the right neighbour comboBox
     */
    public JComboBox getRightNeighbourComboBox() {
        return rightNeighbourComboBox;
    }

    /**Sets the right neighbour comboBox
     * @param rightNeighBourComboBox the right neighbour comboBox
     */
    public void setRightNeighbourComboBox(JComboBox rightNeighbourComboBox) {
        this.rightNeighbourComboBox = rightNeighbourComboBox;
    }

    /**Creates the right neighbour checkBox
     * @return the right neighbour checkBox
     */
    public JCheckBox createRightNeighbourCheckBox(){
        JCheckBox rightNeighbourCheckBox = new JCheckBox();
        rightNeighbourCheckBox.addItemListener((Record) getController());
        return rightNeighbourCheckBox;
    }

    /**Gets the right neighbour checkBox
     * @return the right neighbour checkBox
     */
    public JCheckBox getRightNeighbourCheckBox() {
        return rightNeighbourCheckBox;
    }

    /**Sets the right neighbour checkBox
     * @param rightNeighBourCheckBox the right neighbour checkBox
     */
    public void setRightNeighbourCheckBox(JCheckBox rightNeighbourCheckBox) {
        this.rightNeighbourCheckBox = rightNeighbourCheckBox;
    }

    /**Creates the left neighbour label
     * @return the left neighbour label
     */
    public TopologyLabel createLeftNeighbourLabel(){
        TopologyLabel leftNeighbourLabel = new TopologyLabel(false);
        leftNeighbourLabel.setText("Left neighbour");
        return leftNeighbourLabel;
    }

    /**Gets the left neighbour label
     * @return the left neighbour label
     */
    public TopologyLabel getLeftNeighbourLabel() {
        return leftNeighbourLabel;
    }

    /**Sets the left neighbour label
     * @param leftNeighBourLabel the left neighbour label
     */
    public void setLeftNeighbourLabel(TopologyLabel leftNeighbourLabel) {
        this.leftNeighbourLabel = leftNeighbourLabel;
    }

    /**Creates the left neighbour comboBox
     * @return the left neighbour comboBox
     */
    public JComboBox createLeftNeighbourComboBox(){
        JComboBox leftNeighbourComboBox = new JComboBox();
        leftNeighbourComboBox.setEnabled(false);
        return leftNeighbourComboBox;
    }

    /**Gets the left neighbour comboBox
     * @return the left neighbour comboBox
     */
    public JComboBox getLeftNeighbourComboBox() {
        return leftNeighbourComboBox;
    }

    /**Sets the left neighbour comboBox
     * @param leftNeighBourComboBox the left neighbour comboBox
     */
    public void setLeftNeighbourComboBox(JComboBox leftNeighbourComboBox) {
        this.leftNeighbourComboBox = leftNeighbourComboBox;
    }

    /**Creates the left neighbour checkBox
     * @return the left neighbour checkBox
     */
    public JCheckBox createLeftNeighbourCheckBox(){
        JCheckBox leftNeighbourCheckBox = new JCheckBox();
        leftNeighbourCheckBox.addItemListener((Record) getController());
        return leftNeighbourCheckBox;
    }

    /**Gets the left neighbour checkBox
     * @return the left neighbour checkBox
     */
    public JCheckBox getLeftNeighbourCheckBox() {
        return leftNeighbourCheckBox;
    }

    /**Sets the left neighbour checkBox
     * @param leftNeighBourCheckBox the left neighbour checkBox
     */
    public void setLeftNeighbourCheckBox(JCheckBox leftNeighbourCheckBox) {
        this.leftNeighbourCheckBox = leftNeighbourCheckBox;
    }

    /**Creates the top neighbour label
     * @return the top neighbour label
     */
    public TopologyLabel createTopNeighbourLabel(){
        TopologyLabel l = new TopologyLabel(false);
        l.setText("Top neighbour");
        return l;
    }

    /**Gets the top neighbour label
     * @return the top neighbour label
     */
    public TopologyLabel getTopNeighbourLabel() {
        return topNeighbourLabel;
    }

    /**Sets the top neighbour label
     * @param topNeighBourLabel the top neighbour label
     */
    public void setTopNeighbourLabel(TopologyLabel topNeighbourLabel) {
        this.topNeighbourLabel = topNeighbourLabel;
    }

    /**Creates the top neighbour comboBox
     * @return the top neighbour comboBox
     */
    public JComboBox createTopNeighbourComboBox(){
        JComboBox c = new JComboBox();
        c.setEnabled(false);
        return c;
    }

    /**Gets the top neighbour comboBox
     * @return the top neighbour comboBox
     */
    public JComboBox getTopNeighbourComboBox() {
        return topNeighbourComboBox;
    }

    /**Sets the top neighbour comboBox
     * @param topNeighBourComboBox the top neighbour comboBox
     */
    public void setTopNeighbourComboBox(JComboBox topNeighbourComboBox) {
        this.topNeighbourComboBox = topNeighbourComboBox;
    }

    /**Creates the top neighbour checkBox
     * @return the top neighbour checkBox
     */
    public JCheckBox createTopNeighbourCheckBox(){
        JCheckBox c = new JCheckBox();
        c.addItemListener((Record) getController());
        return c;
    }

    /**Gets the top neighbour checkBox
     * @return the top neighbour checkBox
     */
    public JCheckBox getTopNeighbourCheckBox() {
        return topNeighbourCheckBox;
    }

    /**Sets the top neighbour checkBox
     * @param topNeighBourCheckBox the top neighbour checkBox
     */
    public void setTopNeighbourCheckBox(JCheckBox topNeighbourCheckBox) {
        this.topNeighbourCheckBox = topNeighbourCheckBox;
    }

    /**Creates the bottom neighbour label
     * @return the bottom neighbour label
     */
    public TopologyLabel createBottomNeighbourLabel(){
        TopologyLabel bottomNeighbourLabel = new TopologyLabel(false);
        bottomNeighbourLabel.setText("Bottom neighbour");
        return bottomNeighbourLabel;
    }

    /**Gets the bottom neighbour label
     * @return the bottom neighbour label
     */
    public TopologyLabel getBottomNeighbourLabel() {
        return bottomNeighbourLabel;
    }

    /**Sets the bottom neighbour label
     * @param bottomNeighBourLabel the bottom neighbour label
     */
    public void setBottomNeighbourLabel(TopologyLabel bottomNeighbourLabel) {
        this.bottomNeighbourLabel = bottomNeighbourLabel;
    }

    /**Creates the bottom neighbour comboBox
     * @return the bottom neighbour comboBox
     */
    public JComboBox createBottomNeighbourComboBox(){
        JComboBox c = bottomNeighbourComboBox = new JComboBox();
        c.setEnabled(false);
        return c;
    }

    /**Gets the bottom neighbour comboBox
     * @return the bottom neighbour comboBox
     */
    public JComboBox getBottomNeighbourComboBox() {
        return bottomNeighbourComboBox;
    }

    /**Sets the bottom neighbour comboBox
     * @param bottomNeighBourComboBox the bottom neighbour comboBox
     */
    public void setBottomNeighbourComboBox(JComboBox bottomNeighbourComboBox) {
        this.bottomNeighbourComboBox = bottomNeighbourComboBox;
    }

    /**Creates the bottom neighbour checkBox
     * @return the bottom neighbour checkBox
     */
    public JCheckBox createBottomNeighbourCheckBox(){
        JCheckBox bottomNeighbourCheckBox = new JCheckBox();
        bottomNeighbourCheckBox.addItemListener((Record) getController());
        return bottomNeighbourCheckBox;
    }

    /**Gets the bottom neighbour checkBox
     * @return the bottom neighbour checkBoox
     */
    public JCheckBox getBottomNeighbourCheckBox() {
        return bottomNeighbourCheckBox;
    }

    /**Sets the bottom neighbour checkBox
     * @param bottomNeighBourComboBox the bottom neighbour checkBox
     */
    public void setBottomNeighbourCheckBox(JCheckBox bottomNeighbourCheckBox) {
        this.bottomNeighbourCheckBox = bottomNeighbourCheckBox;
    }
}
