/*
 * $RCSfile: AlternativeOverlaysPreviewDialog.java,v $
 *
 * Copyright (c) 1999-2002. Jens Bohl. All rights reserved.
 *
 * This software is published under the GPL GNU General Public License.
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * http://www.resmedicinae.org
 * - Information in Medicine -
 */

package org.resmedicinae.application.healthcare.record.topology;

import org.resmedicinae.application.healthcare.record.*;
import org.resmedicinae.resmedlib.component.control.*;
import org.resmedicinae.resmedlib.component.view.swing.*;

import javax.swing.*;
import java.awt.*;
import java.util.*;

/**View Component.
 * This class gives a preview on all existing alternative overlays for the current overlay. In this dialog
 * one of these overlays can be chosen and this will be the next one shown in the imagePanel.
 * @author Jens Bohl <info@jens-bohl-professional.de>
 */
public class AlternativeOverlaysPreviewDialog extends SwingView {
    /** ButtonGroup for containing preview buttons */
    private ButtonGroup group;

    /** The width of the dialog */
    private int dialogWidth;

    /** The height of the dialog */
    private int dialogHeight;

    /** The x position of dialog */
    private int dialogXPosition;

    /** The y position of dialog */
    private int dialogYPosition;

    /**Default constructor*/
    public AlternativeOverlaysPreviewDialog() {
    }

    public void initialize() {
        setTitle("Choose alternative...");
        setDialogWidth(10);
        setDialogXPosition(150);
        setDialogYPosition(10);

        //FocusListener is needed because of registering...
        //addFocusListener((Record) getController());
        //addKeyListener((Record) getController());

        FlowLayout flowLayout = new FlowLayout();
        flowLayout.setAlignment(FlowLayout.LEFT);
        setLayout(flowLayout);
        setDisplayMode(DisplayFactory.MODELESS_DIALOG_DISPLAY);
        setDialogBounds(createDialogBounds());
    }

    /**This methode reInitializes the dialog. This means getting new alternative overlays after the overlay in the image
     * panel has changed. Because of that, the size of this dialog has to be modified in respect of the number of new
     * alternative overlays. A vector ot previewPanels manages the preview on the alternatives.
     */

    public void reInitialize() {
        RecordModel model = (RecordModel) getModel();
        Vector alternativeOverlays = model.getAlternativeOverlaysForCurrentOverlay();
        setDialogWidth(10);
        setDialogHeight(250);
        if (alternativeOverlays != null) {
            PreviewPanel[] previewPanels = new PreviewPanel[alternativeOverlays.size()];
            PreviewRadioButton[] previewRadioButtons = new PreviewRadioButton[alternativeOverlays.size()];
            group = new ButtonGroup();
            //clearing the panel for reset, getting new alternatives
            removeAll();
            //for every alternative overlay a panel is created
            for (int i = 0; i < alternativeOverlays.size(); i++) {
                JPanel previewContainerPanel = new JPanel();
                previewContainerPanel.setPreferredSize(new Dimension(150, 200));
                String overlayName = (String) alternativeOverlays.elementAt(i);
                previewPanels[i] = new PreviewPanel();
                previewPanels[i].setBorder(BorderFactory.createLineBorder(Color.BLACK));
                previewPanels[i].setImage(model.getImageFromDocument(model.getDocumentByName(overlayName)));
                //only one can be chosen, by mouse or with tab -> radioButtons
                previewRadioButtons[i] = new PreviewRadioButton(overlayName);
                previewRadioButtons[i].addActionListener((Record) getController());
                previewRadioButtons[i].setActionCommand(Record.CHANGE_ALTERNATIVE_OVERLAY_PREVIEW_ID);
                previewRadioButtons[i].addKeyListener((Record) getController());
                group.add(previewRadioButtons[i]);
                previewContainerPanel.add(previewRadioButtons[i]);
                previewContainerPanel.add(previewPanels[i]);
                previewContainerPanel.setBorder(BorderFactory.createLineBorder(Color.BLACK));
                //adding one panel containing an overlay preview
                add(previewContainerPanel);
                int width = (new Double(previewContainerPanel.getPreferredSize().getWidth())).intValue();
                int newWidth = width + getDialogWidth() + 10; // 10 == offset!!!
                setDialogWidth(newWidth);
            }
            if(previewRadioButtons.length>0){
                if (previewRadioButtons[0] != null) {
                    previewRadioButtons[0].setSelected(true);
                }
            }
            setDialogBounds(createDialogBounds());
        } else {
            //if there are no alternatives, clear all, resize and give message
            removeAll();
            add(new JLabel("No alternative overlays available."));
            setDialogWidth(300);
            setDialogHeight(50);
            setDialogBounds(createDialogBounds());
        }
    }

    /**
     * Creates a rectangle storing the dialog bounds.
     * @return the rectangle storing the dialog bounds
     */
    public Rectangle createDialogBounds() {
        // Determine width.
        int screenWidth = (int) Math.floor(Toolkit.getDefaultToolkit().getScreenSize().getWidth());
        int w = getDialogWidth();
        int x = getDialogXPosition();
        int y = getDialogYPosition();
        if (w > screenWidth) {
            w = screenWidth;
        }
        // Determine height.
        int screenHeight = (int) Math.floor(Toolkit.getDefaultToolkit().getScreenSize().getHeight());
        int h = getDialogHeight();
        if (h > screenHeight) {
            h = screenHeight;
        }
        return new Rectangle(x, y, w, h);
    }

    /**
     * Creates a display mode.
     * @return the display mode
     */
    public int createDisplayMode() {
        return DisplayFactory.MODELESS_DIALOG_DISPLAY;
    }


    /**
     * Creates a control issued when the window containing this view is closed.
     * @return the control issued when the window containing this view is closed
     * @exception NullPointerException if the control is null
     */
    public Control createCloseControl() throws NullPointerException {
        Control c = new Control();
        if (c != null) {
            c.setId(Record.DESTROY_LOAD_PATIENT_DIALOG_CONTROL_ID);
            c.setSender((Controller) getController());
        } else {
            throw new NullPointerException("Could not create close control. The control is null.");
        }
        return c;
    }

    /**Gets the buttonGroup for all radiobuttons
     * @return the buttonGroup
     */
    public ButtonGroup getGroup() {
        return group;
    }

    /**Sets the buttonGroup for all radiobuttons
     * @param buttonGroup the buttonGroup
     */
    public void setGroup(ButtonGroup group) {
        this.group = group;
    }

    /**Gets the dialogHeight
     * @return the heigth of the dialog
     */
    public int getDialogHeight() {
        return dialogHeight;
    }

    /**Sets the dialogHeight
     * @param dialogHeight the heigth of the dialog
     */
    public void setDialogHeight(int dialogHeight) {
        this.dialogHeight = dialogHeight;
    }

    /**Gets the dialogWidth
     * @return the width of the dialog
     */
    public int getDialogWidth() {
        return dialogWidth;
    }

    /**Sets the dialogWidth
     * @param the width of the dialog
     */
    public void setDialogWidth(int dialogWidth) {
        this.dialogWidth = dialogWidth;
    }

    /**Gets the x position of the dialog
     * @return the x position
     */
    public int getDialogXPosition() {
        return dialogXPosition;
    }

    /**Sets the x position of the dialog
     * @param the x position
     */
    public void setDialogXPosition(int dialogXPosition) {
        this.dialogXPosition = dialogXPosition;
    }

    /**Gets the y position of the dialog
     * @return the y position
     */
    public int getDialogYPosition() {
        return dialogYPosition;
    }

    /**Sets the y position of the dialog
     * @param the y position
     */
    public void setDialogYPosition(int dialogYPosition) {
        this.dialogYPosition = dialogYPosition;
    }
}
