/*
 * $RCSfile: TreeBuilder.java,v $
 *
 * Copyright (c) 1999-2002. Christian Heller. All rights reserved.
 *
 * This software is published under the GPL GNU General Public License.
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * http://www.resmedicinae.org
 * - Information in Medicine -
 */

package org.resmedicinae.application.common.main;

import java.lang.*;
import org.resmedicinae.resmedlib.component.component.*;
import org.resmedicinae.resmedlib.component.control.*;

/**
 * This class represents a tree builder.
 *
 * @version $Revision: 1.2 $ $Date: 2002/06/12 18:49:34 $ $Author: chrissy $
 * @author Christian Heller <christian.heller@tuxtax.de>
 */
public class TreeBuilder extends Component {

    /*
     * Builds an application tree from the given string array.
     *
     * @param s the string array containing all tree node parameters
     * @return the application tree
     */
    public ApplicationTreeNode build(String[] s) {

        ApplicationTreeNode n = new ApplicationTreeNode("root");
        String category;
        boolean external;
        String location;

        if (s != null) {

            for (int i = 0; i < s.length; i++) {

                category = determineApplicationCategory(s[i]);
                external = determineExternalApplication(s[i]);
                location = determineApplicationLocation(s[i]);

                n = createApplicationTreeBranch(category, external, location, n);
            }

        } else {

            throw new NullPointerException("Could not build tree. The string array is null.");
        }

        return n;
    }

    /**
     * Determines the application category.
     *
     * @param s the source string read from the preference tree
     * @return the application category
     */
    private String determineApplicationCategory(String s) {

        String c = "";

        if ((s != null) && !s.equals("")) {

            // indexOf returns -1 if string not found.
            int begin = 0;
            int end = s.indexOf(",", 0);

            if (end > begin) {

                c = s.substring(begin, end);
            }
        }

        return c;
    }

    /**
     * Determines the flag indicating an external application.
     *
     * @param s the source string read from the preference tree
     * @return the flag indicating an external application
     */
    private boolean determineExternalApplication(String s) {

        boolean e = false;
        String ext = "";

        if ((s != null) && !s.equals("")) {

            // indexOf returns -1 if string not found.
            int begin = s.indexOf(",", 0) + 1;

            if (s.length() > begin) {

                int end = s.indexOf(",", begin);

                // First and second comma must exist.
                if ((begin > 0) && (end > begin)) {

                    ext = s.substring(begin, end);
                }
            }

            if (ext.equals("true")) {

                e = true;
            }
        }

        return e;
    }

    /**
     * Determines the application location.
     *
     * @param s the source string read from the preference tree
     * @return the application location
     */
    private String determineApplicationLocation(String s) {

        String l = "";

        if ((s != null) && !s.equals("")) {

            // indexOf returns -1 if string not found.
            int i = s.indexOf(",", 0) + 1;

            if ((i > 0) && (s.length() > i)) {

                int begin = s.indexOf(",", i) + 1;

                if ((begin > 0) && (begin > i) && (s.length() > begin)) {

                    l = s.substring(begin);
                }
            }
        }

        return l;
    }

    /**
     * Creates branch of tree nodes.
     *
     * @param c the application category tree path
     * @param e the flag indicating an external application
     * @param l the application location
     * @param n the parent tree node
     */
    private ApplicationTreeNode createApplicationTreeBranch(String c, boolean e, String l, ApplicationTreeNode n) {

        String name = determineName(c);
        c = determineRemainingCategory(c);
        ApplicationTreeNode child = getApplicationTreeNode(name, n);

        if (child == null) {

            child = new ApplicationTreeNode(name);
        }

        if (!c.equals("")) {

            child = createApplicationTreeBranch(c, e, l, child);

        } else {

            child.setExternal(e);
            child.setLocation(l);
        }

        n.add(child);

        return n;
    }

    /**
     * Returns the index of the application tree node with the given name.
     *
     * If that application tree node does not exist, -1 will be returned.
     *
     * @param name the application tree node name to search for
     * @param n the parent tree node
     * @return the application tree node with the given name or null if it was not found
     */
    private ApplicationTreeNode getApplicationTreeNode(String name, ApplicationTreeNode n) {

        ApplicationTreeNode child = null;
        ApplicationTreeNode current = null;

        if ((name != null) && (n != null)) {

            for (int i = 0; i < n.getChildCount(); i++) {

                current = (ApplicationTreeNode) n.getChildAt(i);

                if (current != null) {

                    if (name.equals(current.getUserObject())) {

                        child = current;
                        break;
                    }
                }
            }
        }

        return child;
    }

    /**
     * Determines the tree node name.
     *
     * @param c the application category tree path
     * @return the tree node name
     */
    private String determineName(String c) {

        String name = "";

        if ((c != null) && !c.equals("")) {

            if (c.startsWith("/")) {

                c = c.substring(1);
            }

            // indexOf returns -1 if string not found.
            int begin = 0;
            int end = c.indexOf("/", begin);

            if (end > 0) {

                name = c.substring(begin, end);

            } else {

                name = c;
            }
        }

        return name;
    }

    /**
     * Determines the remaining category.
     *
     * @param c the application category tree path
     * @return the remaining category
     */
    private String determineRemainingCategory(String c) {

        String r = "";

        if ((c != null) && !c.equals("")) {

            if (c.startsWith("/")) {

                c = c.substring(1);
            }

            // indexOf returns -1 if string not found.
            int begin = c.indexOf("/", 0) + 1;

            if ((begin > 0) && (c.length() > begin)) {

                r = c.substring(begin);
            }
        }

        return r;
    }
}

