/*
 * $RCSfile: MainApplicationView.java,v $
 *
 * Copyright (c) 1999-2002. Christian Heller. All rights reserved.
 *
 * This software is published under the GPL GNU General Public License.
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * http://www.resmedicinae.org
 * - Information in Medicine -
 */

package org.resmedicinae.application.common.main;

// Do not import java.awt.*! It would conflict with MenuBar etc.
import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.Dimension;
import java.lang.*;
import javax.swing.*;
import javax.swing.tree.*;
import org.resmedicinae.resmedlib.component.configuration.*;
import org.resmedicinae.resmedlib.component.control.*;
import org.resmedicinae.resmedlib.component.view.*;
import org.resmedicinae.resmedlib.component.view.swing.*;

/**
 * This class represents the main application view.
 *
 * It is based on a split pane containing a tree menu on the left.
 * The tree offers a hierarchy of categories with applications available to the system.
 * On the right side of the splitter, there is a container for child application views.
 *
 * It is based on a split pane containing a tree menu on the left.
 * The tree offers a hierarchical list (in categories) of all applications available to the system.
 *
 * @version $Revision: 1.4 $ $Date: 2002/08/09 21:12:40 $ $Author: chrissy $
 * @author Christian Heller <christian.heller@tuxtax.de>
 */
public class MainApplicationView extends SwingView {

    //
    // Attributes.
    //

    /**
     * The split pane.
     *
     * It shares the panel into a left section for the application tree view
     * and a right section for child windows etc.
     */
    private ResSplitPane splitPane;

    /** The application tree view.
     *
     * It displays applications sorted by categories.
     * An application will be started when clicking on one of the leaf tree nodes.
     */
    private ResTree applicationTreeView;

    //
    // Menu bar.
    //

    /**
     * Creates a menu bar.
     *
     * @return the menu bar
     * @exception NullPointerException if the menu bar is null
     */
    public ResMenuBar createMenuBar() throws Exception, NullPointerException {

        MainApplicationMenuBar mb = new MainApplicationMenuBar();

        if (mb != null) {

            mb.setOwner(this);
            mb.setController(getController());
            mb.initialize();

        } else {

            throw new NullPointerException("Could not create menu bar. The menu bar is null.");
        }

        return mb;
    }

    //
    // Tool bar.
    //

    /**
     * Creates a tool bar.
     *
     * @return the tool bar
     * @exception NullPointerException if the tool bar is null
     */
    public ResToolBar createToolBar() throws Exception, NullPointerException {

        ResToolBar tb = new MainApplicationToolBar();

        if (tb != null) {

            tb.initialize();

        } else {

            throw new NullPointerException("Could not create tool bar. The tool bar is null.");
        }

        return tb;
    }

    //
    // Split pane.
    //

    /**
     * Returns the split pane.
     *
     * @return the split pane
     */
    public ResSplitPane getSplitPane() {

        return this.splitPane;
    }

    /**
     * Sets the split pane.
     *
     * @param splitPane the split pane
     */
    public void setSplitPane(ResSplitPane splitPane) {

        this.splitPane = splitPane;
    }

    /**
     * Creates a split pane.
     *
     * @return the split pane
     * @exception NullPointerException if the split pane is null
     */
    public ResSplitPane createSplitPane() throws Exception, NullPointerException {

        ResSplitPane sp = new ResSplitPane();

        if (sp != null) {

            setApplicationTreeView(createApplicationTreeView());

            sp.setLeftComponent(new ResScrollPane(getApplicationTreeView()));
            sp.setRightComponent(null);
            sp.setOrientation(JSplitPane.HORIZONTAL_SPLIT);
            sp.setOneTouchExpandable(true);
            sp.setContinuousLayout(true);
            sp.setDividerLocation(200);
            sp.setPreferredSize(new java.awt.Dimension(800, 600));

        } else {

            throw new NullPointerException("Could not create split pane. The split pane is null.");
        }

        return sp;
    }

    /**
     * Destroys the split pane.
     *
     * @param sp the split pane
     */
    public void destroySplitPane(ResSplitPane sp) throws Exception, NullPointerException {

        if (sp != null) {

            sp.setLeftComponent(null);
            sp.setRightComponent(null);

        } else {

            throw new NullPointerException("Could not destroy split pane. The split pane is null.");
        }
    }

    //
    // Application tree view.
    //

    /**
     * Returns the application tree view.
     *
     * @return the application tree view
     */
    public ResTree getApplicationTreeView() {

        return this.applicationTreeView;
    }

    /**
     * Sets the application tree view.
     *
     * @param applicationTreeView the application tree view
     */
    public void setApplicationTreeView(ResTree applicationTreeView) {

        this.applicationTreeView = applicationTreeView;
    }

    /**
     * Creates an application tree view.
     *
     * @return the application tree view
     * @exception NullPointerException if the application tree view is null
     */
    public ResTree createApplicationTreeView() throws Exception, NullPointerException {

        ResTree v = new ResTree();

        if (v != null) {

            v.setToggleClickCount(1);
            v.setRootVisible(false);
            v.getSelectionModel().setSelectionMode(TreeSelectionModel.SINGLE_TREE_SELECTION);
            v.setMinimumSize(new Dimension(300, 400));
//??            v.setModel(availableApplications);
//??            The model is temporarily set in the controller. See there.

        } else {

            throw new NullPointerException("Could not create application tree view. The application tree view is null.");
        }

        return v;
    }

    /**
     * Destroys the application tree view.
     *
     * @param the application tree view
     * @exception NullPointerException if the application tree view is null
     */
    public void destroyApplicationTreeView(ResTree v) throws Exception, NullPointerException {

        if (v != null) {

        } else {

            throw new NullPointerException("Could not destroy application tree view. The application tree view is null.");
        }
    }

    //
    // Child application component.
    //

    /**
     * Returns the child application component.
     *
     * @return the child application component
     */
    public JComponent getChildApplicationComponent() {

        JComponent c = null;
        JSplitPane sp = getSplitPane();

        if (sp != null) {

            c = (JComponent) sp.getRightComponent();

        } else {

            throw new NullPointerException("Could not get child application component. The split pane is null.");
        }

        return c;
    }

    /**
     * Sets the child application component.
     *
     * @param c the child application component
     */
    public void setChildApplicationComponent(JComponent c) {

        JSplitPane sp = getSplitPane();

        if (sp != null) {

            Component old = sp.getRightComponent();

            if (old != null) {

                sp.remove(old);
            }

            sp.setRightComponent(c);

        } else {

            throw new NullPointerException("Could not set child application component. The split pane is null.");
        }
    }

    //
    // Initializable.
    //

    /**
     * Initializes this component.
     */
    public void initialize() throws Exception {

        super.initialize();

        setSplitPane(createSplitPane());

        //?? Put the following methods to an extra lifecycle method "assemble"?
        add(getSplitPane(), BorderLayout.CENTER);
    }

    /**
     * Finalizes this component.
     */
    public void finalizz() throws Exception {

        //?? Put the following methods to an extra lifecycle method "disassemble"?
        remove(getSplitPane());

        destroySplitPane(getSplitPane());
        setSplitPane(null);

        super.finalizz();
    }
}

