/*
 * $RCSfile: MainApplication.java,v $
 *
 * Copyright (c) 1999-2002. Christian Heller. All rights reserved.
 *
 * This software is published under the GPL GNU General Public License.
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * http://www.resmedicinae.org
 * - Information in Medicine -
 */

package org.resmedicinae.application.common.main;

import java.awt.event.*;
import java.io.*;
import java.lang.*;
import java.util.*;
import java.util.prefs.*;
import javax.swing.event.*;
import javax.swing.tree.*;
import org.resmedicinae.application.common.basic.*;
import org.resmedicinae.resmedlib.component.configuration.*;
import org.resmedicinae.resmedlib.component.component.*;
import org.resmedicinae.resmedlib.component.context.*;
import org.resmedicinae.resmedlib.component.control.*;
import org.resmedicinae.resmedlib.component.logger.*;
import org.resmedicinae.resmedlib.component.model.*;
import org.resmedicinae.resmedlib.component.view.*;
import org.resmedicinae.resmedlib.component.view.swing.*;

/**
 * This class represents a main application.
 *
 * It is able to host child applications which can be common components
 * of the ResMedLib framework or external processes.
 *
 * It basically contains a tree with a hierarchy of all available applications (modules).
 * This tree can be displayed by the corresponding view as menu or tree or the like.
 * On clicking an application, it gets started. The applications can be run standalone,
 * independent from Res Medicinae as well.
 * It is easily possible to write additional applications and integrate them
 * into the system. Read the tutorial "Creating a Res Medicinae Module".
 *
 * The Res Medicinae main application is capable of hosting multiple child applications
 * which get integrated into the main application's view (main frame) and
 * can be displayed as Frames, InternalFrames or TabPages.
 *
 * Design patterns:
 * - Controller in Model View Controller
 *
 * @version $Revision: 1.4 $ $Date: 2002/08/09 21:12:40 $ $Author: chrissy $
 * @author Christian Heller <christian.heller@tuxtax.de>
 */
public class MainApplication extends BasicApplication {

    //
    // Constants.
    //

    /** The show application tree control id. */
    public static final String SHOW_APPLICATION_TREE_CONTROL_ID = "show_application_tree";

    /** The configure application tree control id. */
    public static final String CONFIGURE_APPLICATION_TREE_CONTROL_ID = "configure_application_tree";

    /** The configure res medicinae control id. */
    public static final String CONFIGURE_RES_MEDICINAE_CONTROL_ID = "configure_res_medicinae";

    /** The iconify all control id. */
    public static final String ICONIFY_ALL_CONTROL_ID = "iconify_all";

    /** The close control id. */
    public static final String CLOSE_CONTROL_ID = "close";

    /** The close all control id. */
    public static final String CLOSE_ALL_CONTROL_ID = "close_all";

    /** The cascade control id. */
    public static final String CASCADE_CONTROL_ID = "cascade";

    /** The tile vertically control id. */
    public static final String TILE_VERTICALLY_CONTROL_ID = "tile_vertically";

    /** The tile horizontally control id. */
    public static final String TILE_HORIZONTALLY_CONTROL_ID = "tile_horizontally";

    /** The set frame display control id. */
    public static final String SET_FRAME_DISPLAY_CONTROL_ID = "set_frame_display";

    /** The set internal frame display control id. */
    public static final String SET_INTERNAL_FRAME_DISPLAY_CONTROL_ID = "set_internal_frame_display";

    /** The set tab page display control id. */
    public static final String SET_TAB_PAGE_DISPLAY_CONTROL_ID = "set_tab_page_display";

    //
    // Attributes.
    //

    /** The list of opened applications. */
    private List openedApplications;

    /** The list of opened external applications. */
    private List openedExternalApplications;

    //
    // Name.
    //

    /**
     * Returns the name.
     *
     * @return the name
     */
    public String getName() {

        return "Res Medicinae";
    }

    //
    // Version.
    //

    /**
     * Returns the version.
     *
     * (major).(minor).(<99 = preX, 99 = final).(bug fix)
     *
     * @return the version
     */
    public String getVersion() {

        return "0.0.3.0";
    }

    //
    // Date.
    //

    /**
     * Returns the release date.
     *
     * @return the release date
     */
    public String getDate() {

        return "30.06.2002";
    }

    //
    // Logo.
    //

    /**
     * Returns the logo.
     *
     * @return the logo
     */
    public String getLogo() {

        return "logo.jpeg";
    }

    //
    // Copyright.
    //

    /**
     * Returns the copyright.
     *
     * @return the copyright
     */
    public String getCopyright() {

        return "Copyright (c) 1999-2002. The Res Medicinae Developers. All rights reserved.";
    }

    //
    // Authors.
    //

    /**
     * Returns the authors.
     *
     * @return the authors
     */
    public String getAuthors() {

        return "Christian Heller\n<christian.heller@tuxtax.de>\n- initiated project\n- maintains ResMedLib framework\n- created MainApplication module\n\n";
    }

    //
    // Helpers.
    //

    /**
     * Returns the helpers.
     *
     * @return the helpers
     */
    public String getHelpers() {

        return "Karsten Hilbert\n<karsten.hilbert@gmx.net>\n- maintains the Analysis Document\n- helps out in *many* things\n\n"
            + "Chris Fraser\n<pangaea@comcen.com.au>\n- provided the 'Littlefish' project resources\n\n"
            + "Andrew Arch\n<archa@anz.com>\n- provided the 'Littlefish' project resources\n\n"
            + "Michael Simon\n<michael.simon@gmx.net>\n- created ANT build.xml files\n- created .bat build/start files\n\n"
            + "Henrik Brandes\n<henrik-b@gmx.de>\n- created ANT build.xml files\n- created .bat build/start files\n\n"
            + "Saddia Malik\n<kermitmaliks@web.de>\n- created ANT build.xml files\n- created .bat build/start files\n\n"
            + "Sebastian Hilbert\n<sebastian.hilbert@gmx.net>\n- designed the Res Medicinae logo\n\n"
            + "Johannes Vieweg\n<jvg@gmx.net>\n- provided a PDF version of the Design Document\n\n";
    }

    //
    // Inspirations.
    //

    /**
     * Returns the inspirations.
     *
     * @return the inspirations
     */
    public String getInspirations() {

        return "Scope\nhttp://www.sourceforge.net/projects/scope\n- Hierarchical Model View Controller (HMVC) Design Pattern\n\n"
            + "Apache-Jakarta-Framework\nhttp://jakarta.apache.org\nComponent Lifecycle using Concern Interfaces\n\n"
            + "Linux Kontor\nhttp://www.sourceforge.net/projects/kontor\nEnterprise Resource Planning System\n\n";
    }

    //
    // Opened applications.
    //

    /*
     * Returns the list of opened applications.
     *
     * @return the list of opened applications
     */
    public List getOpenedApplications() {

        return this.openedApplications;
    }

    /**
     * Sets the list of opened applications.
     *
     * @param openedApplications the list of opened applications
     */
    public void setOpenedApplications(List openedApplications) {

        this.openedApplications = openedApplications;
    }

    /*
     * Creates a list for opened applications.
     *
     * @return the list for opened applications
     */
    public List createOpenedApplications() throws Exception, NullPointerException {

        return new Vector();
    }

    /*
     * Destroys the list of opened applications.
     *
     * @param l the list of opened applications
     * @exception NullPointerException if the list is null
     * @exception NullPointerException if the list iterator is null
     */
    public void destroyOpenedApplications(List l) throws Exception, NullPointerException {

        if (l != null) {

            Iterator iter = l.iterator();

            if (iter != null) {

                BasicApplication c;

                while (iter.hasNext()) {

                    c = (BasicApplication) iter.next();

                    l.remove(c);
                    destroyApplication(c);
                }

            } else {

                throw new NullPointerException("Could not destroy the opened applications list. The list iterator is null.");
            }

        } else {

            throw new NullPointerException("Could not destroy the opened applications list. The list is null.");
        }
    }

    //
    // Application.
    //

    /**
     * Creates an application.
     *
     * @param loc the location of the application to be created
     * @param args the command line arguments to run the application
     * @param wp the default work path for the application
     * @return the application
     * @exception NullPointerException if the application class is null
     * @exception NullPointerException if the application is null
     */
    public BasicApplication createApplication(String loc, String[] args, String wp) throws Exception, NullPointerException {

        BasicApplication c = null;
        // Find class by name.
        Class cl = Class.forName(loc);

        if (cl != null) {

            // Create application from given class type.
            c = (BasicApplication) cl.newInstance();

            if (c != null) {

//??            showComponentSplashWindow(c);

                // Contextualize the application.
                log(Level.INFO, "Contextualize the application.");
                Context ctxt = c.createContext();

                String[] locations = extractConfigurationLocations(args);
        
                if (ctxt != null) {
              
                    ctxt.putEntry("parent_view", (View) getView());
                    ctxt.putEntry("system_configuration_location", ""/*??locations[0]*/);
                    ctxt.putEntry("local_configuration_location", ""/*??locations[1]*/);
                    ctxt.putEntry("user_configuration_location", ""/*??locations[2]*/);
        
                } else {
        
                    throw new NullPointerException("Could not put context parameters. The context is null.");
                }

                c.contextualize(ctxt);

                // Compose the application.
                log(Level.INFO, "Compose the application.");
                c.compose(c.createComponentManager());

                // Configure the application.
                log(Level.INFO, "Configure the application.");
                c.configure(c.createConfigurationManager());

                // Initialize the application.
                log(Level.INFO, "Initialize the application.");
                c.initialize();

                // Start the application.
                log(Level.INFO, "Start the application.");
                c.start();
    
            } else {
    
                throw new NullPointerException("Could not create application. The application is null.");
            }

//??            hideComponentSplashWindow(c);

        } else {

            throw new NullPointerException("Could not create application. The application class is null. Probably, a wrong location was given.");
        }

        return c;
    }

    /**
     * Destroys the application.
     *
     * @param c the application
     * @exception NullPointerException if the application is null
     */
    public void destroyApplication(BasicApplication c) throws Exception, NullPointerException {

        if (c != null) {

            // Stop the application.
            log(Level.INFO, "Stop the application.");
            c.stop();
            
            // Finalize the application.
            log(Level.INFO, "Finalize the application.");
            c.finalizz();
            
            // Deconfigure the application.
            log(Level.INFO, "Deconfigure the application.");
            ConfigurationManager conf = c.getConfigurationManager(); 
            
            c.deconfigure(conf);
            c.destroyConfigurationManager(conf);

            // Decompose the application.
            log(Level.INFO, "Decompose the application.");
            ComponentManager comp = c.getComponentManager();
            
            c.decompose(comp);
            c.destroyComponentManager(comp);

            // Decontextualize the application.
            log(Level.INFO, "Decontextualize the application.");
            Context cont = c.getContext();
            
            c.decontextualize(cont);
            c.destroyContext(cont);

        } else {

            throw new NullPointerException("Could not destroy application. The application is null.");
        }
    }

    /**
     * Extracts the configuration locations.
     *
     * @param args the command line arguments
     * @return the configuration locations
     * @exception NullPointerException if the command line parser is null
     */
    private String[] extractConfigurationLocations(String[] args) throws Exception {

        String[] locations = null;
        CommandLineArgumentsParser p = new CommandLineArgumentsParser();

        if (p != null) {

            log(Level.INFO, "Parse command line arguments.");
            locations = p.parse(args);

        } else {

            throw new NullPointerException("Could not parse command line arguments. The command line parser is null.");
        }

        return locations;
    }

    //
    // Opened external applications.
    //

    /*
     * Returns the list of opened external applications.
     *
     * @return the list of opened external applications
     */
    public List getOpenedExternalApplications() {

        return this.openedExternalApplications;
    }

    /**
     * Sets the list of opened external applications.
     *
     * @param openedExternalApplications the list of opened external applications
     */
    public void setOpenedExternalApplications(List openedExternalApplications) {

        this.openedExternalApplications = openedExternalApplications;
    }

    /*
     * Creates a list for opened external applications.
     *
     * @return the list for opened external applications
     */
    public List createOpenedExternalApplications() throws Exception, NullPointerException {

        return new Vector();
    }

    /*
     * Destroys the list of opened external applications.
     *
     * @param l the list of opened external applications
     * @exception NullPointerException if the list is null
     * @exception NullPointerException if the list iterator is null
     * @exception NullPointerException if the process is null
     */
    public void destroyOpenedExternalApplications(List l) throws Exception, NullPointerException {

        if (l != null) {

            Iterator iter = l.iterator();

            if (iter != null) {

                Process p;

                while (iter.hasNext()) {

                    p = (Process) iter.next();

                    l.remove(p);

                    if (p != null) {

                        p.destroy();

                    } else {

                        throw new NullPointerException("Could not destroy the process. The process is null.");
                    }
                }

            } else {

                throw new NullPointerException("Could not destroy the opened external applications list. The list iterator is null.");
            }

        } else {

            throw new NullPointerException("Could not destroy the opened external applications list. The list is null.");
        }
    }

    //
    // External application.
    //

    /*
     * Creates an external application using the given command, arguments and work path.
     *
     * @param cmd the command to execute the application
     * @param args the command line arguments to run the application
     * @param wp the default work path for the application
     * @return the external application process
     * @exception NullPointerException if the runtime is null
     * @exception NullPointerException if the opened applications list is null
     */
    public Process createExternalApplication(String cmd, String[] args, String wp) throws Exception, NullPointerException {

        Process p = null;
        // Get the runtime of this application.
        Runtime r = Runtime.getRuntime();

        if (r != null) {

            try {

                // Executes the specified command and its arguments in a separate process with the specified environment and working directory.
                p = r.exec(cmd/*??, args, new File(wp)*/);

                List l = getOpenedExternalApplications();

                if (l != null) {

                    // Add the application to the list of external applications.
                    l.add(p);

                } else {

                    throw new NullPointerException("Could not create external application. The opened applications list is null.");
                }

            } catch (IOException e) {

                log(Level.SEVERE, "Internal io error while executing command with given arguments.", e);
            }

        } else {

            throw new NullPointerException("Could not create external application. The runtime is null.");
        }

        return p;
    }

    /*
     * Destroys the given external application.
     *
     * @param p the external application to be destroyed
     * @exception NullPointerException if the opened applications list is null
     * @exception NullPointerException if the external application process is null
     */
    public void destroyExternalApplication(Process p) throws Exception, NullPointerException {

        List l = getOpenedExternalApplications();

        if (l != null) {

            // Remove application from list of external applications.
            getOpenedExternalApplications().remove(p);

        } else {

            throw new NullPointerException("Could not destroy external application. The opened applications list is null.");
        }

        if (p != null) {

            // Kill subprocess. External applications don't have a component lifecycle.
            p.destroy();

        } else {

            throw new NullPointerException("Could not destroy external application. The process is null.");
        }
    }

    //
    // System configuration.
    //

    /*
     * Creates the system configuration.
     *
     * @return the system configuration
     * @exception NullPointerException if the system configuration is null
     * @exception NullPointerException if the context is null
     */
    public Configuration createSystemConfiguration() throws Exception, NullPointerException {

        Configuration c = new MainApplicationConfiguration();

        if (c != null) {

            Context ctxt = getContext();
    
            if (ctxt != null) {
    
                ctxt.putEntry("preferences", Preferences.systemNodeForPackage(getClass().getPackage().getClass()));
    
            } else {
    
                throw new NullPointerException("Could not create system configuration. The context is null.");
            }
    
            c.contextualize(ctxt);
            c.initialize();

        } else {

            throw new NullPointerException("Could not create system configuration. The system configuration is null.");
        }

        return c;
    }

    //
    // Local configuration.
    //

    /*
     * Creates the local configuration.
     *
     * @return the local configuration
     * @exception NullPointerException if the local configuration is null
     * @exception NullPointerException if the context is null
     */
    public Configuration createLocalConfiguration() throws Exception, NullPointerException {

        MainApplicationConfiguration c = new MainApplicationConfiguration();

        if (c != null) {

            Context ctxt = getContext();
    
            if (ctxt != null) {
    
                ctxt.putEntry("preferences", Preferences.systemNodeForPackage(getClass().getPackage().getClass()));
    
            } else {
    
                throw new NullPointerException("Could not create local configuration. The context is null.");
            }
    
            c.contextualize(ctxt);
            c.initialize();

        } else {

            throw new NullPointerException("Could not create local configuration. The local configuration is null.");
        }

        return c;
    }

    //
    // User configuration.
    //

    /*
     * Creates the user configuration.
     *
     * @return the user configuration
     * @exception NullPointerException if the user configuration is null
     * @exception NullPointerException if the context is null
     */
    public Configuration createUserConfiguration() throws Exception, NullPointerException {

        Configuration c = new MainApplicationConfiguration();

        if (c != null) {

            Context ctxt = getContext();
    
            if (ctxt != null) {
    
                ctxt.putEntry("preferences", Preferences.userNodeForPackage(getClass().getPackage().getClass()));
    
            } else {
    
                throw new NullPointerException("Could not create user configuration. The context is null.");
            }
    
            c.contextualize(ctxt);
            c.initialize();

        } else {

            throw new NullPointerException("Could not create user configuration. The user configuration is null.");
        }

        return c;
    }

    //
    // Model.
    //

    /**
     * Creates the model.
     *
     * @return the model
     * @exception NullPointerException if the model is null
     */
    public Object createModel() throws Exception, NullPointerException {

        MainApplicationModel m = new MainApplicationModel();

//?? Use this later when Model inherits from Component!
//??        bear(m);

        m.configure(getConfigurationManager());

        return m;
    }

    //
    // View.
    //

    /**
     * Creates the view.
     *
     * @return the view
     * @exception NullPointerException if the view is null
     */
    public View createView() throws Exception, NullPointerException {

        MainApplicationView v = new MainApplicationView();

        if (v != null) {

            v.setController(this);                
            v.configure(getConfigurationManager());
            v.initialize();
            
        } else {
            
            throw new NullPointerException("Could not create swing view. The swing view is null.");
        }

        return v;
    }

    //
    // Configurable.
    //

    /**
     * Configures the system settings.
     *
     * @param c the configuration
     * @exception NullPointerException if the configuration is null
     */
    protected void configureSystemSettings(Configuration c) throws Exception, NullPointerException {

        super.configureSystemSettings(c);

        if (c != null) {

/*??
            setOpenedApplications(((MainApplicationConfiguration) c).getOpenedApplications(createOpenedApplications()));
            setOpenedExternalApplications(((MainApplicationConfiguration) c).getOpenedExternalApplications(createOpenedExternalApplications()));
*/

        } else {

            throw new NullPointerException("Could not configure system settings. The configuration is null.");
        }
    }

    /**
     * Deconfigures the system settings.
     *
     * @param c the configuration
     * @exception NullPointerException if the configuration is null
     */
    protected void deconfigureSystemSettings(Configuration c) throws Exception, NullPointerException {

        try {

            if (c != null) {

/*??
                ((MainApplicationConfiguration) c).setOpenedExternalApplications(getOpenedExternalApplications());
                ((MainApplicationConfiguration) c).setOpenedApplications(getOpenedApplications());
*/

            } else {

                throw new NullPointerException("Could not deconfigure system settings. The configuration is null.");
            }

        } finally {

            super.deconfigureSystemSettings(c);
        }
    }

    //
    // Initializable.
    //

    /**
     * Initializes this component.
     */
    public void initialize() throws Exception {

        super.initialize();

        setOpenedApplications(createOpenedApplications());
        setOpenedExternalApplications(createOpenedExternalApplications());
    }

    /**
     * Finalizes this component.
     */
    public void finalizz() throws Exception {

        try {

            destroyOpenedExternalApplications(getOpenedExternalApplications());
            setOpenedExternalApplications(null);
            destroyOpenedApplications(getOpenedApplications());
            setOpenedApplications(null);

        } finally {

            super.finalizz();
        }
    }

    //
    // Startable.
    //

    /**
     * Starts this component.
     *
     * @exception NullPointerException if the children list is null
     * @exception NullPointerException if the iterator is null
     */
    public void start() throws Exception {

        super.start();

        //?? Somehow conflict! Endless loop!
        //?? Newly created applications are added to the opened applications container
        //?? where they are taken from!! --> introduce additional attribute to just
        //?? store the opened applications configuration setting.
//??        createApplications(getOpenedApplications());
//??        createApplications(getOpenedExternalApplications());
    }

    /**
     * Stops this component.
     */
    public void stop() throws Exception {

        try {
            
//??        destroyApplications(getOpenedExternalApplications());
//??        destroyApplications(getOpenedApplications());

        } finally {
            
            super.stop();
        }
    }

    /*
     * Creates applications.
     *
     * @param l the list containing the applications to be created
     * @exception NullPointerException if the applications list is null
     * @exception NullPointerException if the object is null
     */
    protected void createApplications(List l) throws Exception, NullPointerException {

        if (l != null) {

            Iterator iter = l.iterator();
            Object o = null;

            while (iter.hasNext()) {

                o = iter.next();

                if (o != null) {

//??                    createExternalApplication(o.toString(), null, "");

                    // Add application to this controller's list of child controllers.
                    add(createApplication(o.toString(), null, ""));

                } else {

                    log(Level.WARNING, "Could not create an application. The object is null.");
                }
            }

        } else {

            throw new NullPointerException("Could not create applications. The applications list is null.");
        }
    }

    /*
     * Destroys applications.
     *
     * @param l the list containing the applications to be destroyed
     * @exception NullPointerException if the applications list is null
     * @exception NullPointerException if the object is null
     */
    protected void destroyApplications(List l) throws Exception, NullPointerException {

        if (l != null) {

            Iterator it = l.iterator();
            Object o = null;

            while (it.hasNext()) {

                BasicApplication c = (BasicApplication) it.next();

                // Remove application from list of child controllers.
                remove(c);

                if (c != null) {

//??                    destroyExternalApplication((Process) o);
                    destroyApplication(c);

                } else {

                    log(Level.WARNING, "Could not destroy an application. The object is null.");
                }
            }

        } else {

            throw new NullPointerException("Could not destroy applications. The applications list is null.");
        }
    }

    //
    // Showable.
    //

    /**
     * Shows this component.
     *
     * @param v the view
     * @exception NullPointerException if ??
     */
    public void show(View v) throws Exception {

        super.show(v);

        //?? Only temporary until a ResTree component with Event/Control handling exists in ResMedLib!
        ((MainApplicationView) v).getApplicationTreeView().addTreeSelectionListener(this);
    }

    //
    // Loadable.
    //

    /**
     * Loads this component.
     *
     * @param m the model
     */
    public void load(Object m) throws Exception {

        super.load(m);

        //?? Temporary, until Scope offers tree.
        TreeModel model = new DefaultTreeModel(((MainApplicationModel) getModel()).getAvailableApplications());
        ((MainApplicationView) getView()).getApplicationTreeView().setModel(model);
//??        ((MainApplicationView) getView()).getMenuBar().getApplicationMenu().setModel(model);
    }

    //
    // Controller.
    //

    /**
     * Handles all control events and calls the corresponding methods.
     *
     * @param c the control
     * @exception NullPointerException if the control is null
     */
    protected void handle(Control c) throws Exception, NullPointerException {

        // Parent already handles a number of events that don't have to be catched here once more.
        super.handle(c);

        //?? Do not change org.scopemvc.core.Control into a ResMedLib-Control!
        //?? The Scope lib issues a org.scopemvc.core.Control internal that
        //?? has to be caught here. If the method header is changed, then this
        //?? method does not override the parent method and is treated as new.

        if (c != null) {

            if (((Control) c).matches(MainApplication.CREATE_APPLICATION_CONTROL_ID)) {

                ApplicationTreeNode n = (ApplicationTreeNode) ((Control) c).getSender();
                
                if (n != null) {
                
                    add(createApplication(n.getLocation(), n.getArguments(), n.getWorkPath()));

                } else {

                    throw new NullPointerException("Could not create application. The application tree node is null.");
                }

            } else if (((Control) c).matches(MainApplication.DESTROY_APPLICATION_CONTROL_ID)) {

                BasicApplication a = (BasicApplication) c.getSender();

                // The user initiates a DESTROY_APPLICATION_CONTROL_ID control event
                // in a menu bar, a tool bar or a windowbutton, to close an application.
                // Normally, the application has no handle for this destroy event and
                // hence passes it up to its parent controller which finally destroys the application.
                // However, if the child application is a main application itself,
                // it is capable of destroying applications but it would be wrong
                // to let the application destroy itself!
                // Therefore, it is checked here, if the ctrl application parameter
                // equals this. If not, then this application is the parent of the
                // given ctrl application parameter and it is allowed to destroy the application.
                // If the ctrl parameter equals this application, then the control
                // is set unmatched so that the parent application can handle the control again.
                if (a != this) {
                
                    destroyApplication(a);
                
                } else {
                 
                    ((Control) c).setMatched(false);
                }

            } else if (((Control) c).matches(MainApplication.CONFIGURE_APPLICATION_TREE_CONTROL_ID)) {

                configureApplicationTree();

            } else if (((Control) c).matches(MainApplication.CONFIGURE_RES_MEDICINAE_CONTROL_ID)) {

                configureResMedicinae();

            } else if (((Control) c).matches(MainApplication.SET_FRAME_DISPLAY_CONTROL_ID)) {

                changeDisplayMode(DisplayFactory.FRAME_DISPLAY);

            } else if (((Control) c).matches(MainApplication.SET_INTERNAL_FRAME_DISPLAY_CONTROL_ID)) {

                changeDisplayMode(DisplayFactory.INTERNAL_FRAME_DISPLAY);

            } else if (((Control) c).matches(MainApplication.SET_TAB_PAGE_DISPLAY_CONTROL_ID)) {

                changeDisplayMode(DisplayFactory.TAB_PAGE_DISPLAY);
            }

        } else {

            throw new NullPointerException("Could not handle control. The control is null.");
        }
    }

    /**
     * ?? Temporary, until framework provides easier handling!
     *
     * Handles all action events of the whole application.
     *
     * @param evt the action event
     */                
    public void actionPerformed(ActionEvent evt) {

        super.actionPerformed(evt);

        System.out.println("Action Command main app.");

        try {

            if (evt.getActionCommand().equals(MainApplication.SET_FRAME_DISPLAY_CONTROL_ID)) {
    
                changeDisplayMode(DisplayFactory.FRAME_DISPLAY);
    
            } else if (evt.getActionCommand().equals(MainApplication.SET_INTERNAL_FRAME_DISPLAY_CONTROL_ID)) {
    
                changeDisplayMode(DisplayFactory.INTERNAL_FRAME_DISPLAY);
    
            } else if (evt.getActionCommand().equals(MainApplication.SET_TAB_PAGE_DISPLAY_CONTROL_ID)) {
    
                changeDisplayMode(DisplayFactory.TAB_PAGE_DISPLAY);
            }
        
        } catch (Exception e) {
            
            System.out.println("?? Exception occured! " + e);
        }
    }

    //
    // Application tree configuration dialog.
    //

    /**
     * Configures application tree in a dialog.
     *
     * @exception NullPointerException if the application tree configuration dialog is null
     */
    private void configureApplicationTree() throws NullPointerException {

/*??
        log(Level.CONFIG, "Create configuration dialog.");
        ApplicationTreeConfigurationDialog d = new ApplicationTreeConfigurationDialog(GuiController.getGuiController().getActiveFrame(), "Configure Settings", true, (ApplicationTreeNode) this.mainApplication.getAvailableApplications().getRoot());

        if (d != null) {

            d.setSize(new Dimension(700, 600));

            Dimension dialogSize = d.getPreferredSize();
            Dimension screenSize = Toolkit.getDefaultToolkit().getScreenSize();

            if (dialogSize.height > screenSize.height) {

                dialogSize.height = screenSize.height;
            }

            if (dialogSize.width > screenSize.width) {

                dialogSize.width = screenSize.width;
            }

            d.setLocation((screenSize.width - dialogSize.width) / 2, (screenSize.height - dialogSize.height) / 2);
            d.setVisible(true);

        } else {

            throw new NullPointerException("Could not create about dialog. About dialog is null.");
        }
*/
    }

    //
    // ResMedicinae configuration dialog.
    //

    /**
     * Configures Res Medicinae in a dialog.
     *
     * @exception NullPointerException if the configure Res Medicinae dialog is null
     */
    private void configureResMedicinae() throws NullPointerException {

/*??
        ResMedicinaeConfigurationDialog d = new ResMedicinaeConfigurationDialog(GuiController.getGuiController().getActiveFrame(), "Configure Settings", true, this.mainApplication);

        if (d != null) {

            d.setSize(new Dimension(700, 600));

            Dimension dialogSize = d.getPreferredSize();
            Dimension screenSize = Toolkit.getDefaultToolkit().getScreenSize();

            if (dialogSize.height > screenSize.height) {

                dialogSize.height = screenSize.height;
            }

            if (dialogSize.width > screenSize.width) {

                dialogSize.width = screenSize.width;
            }

            d.setLocation((screenSize.width - dialogSize.width) / 2, (screenSize.height - dialogSize.height) / 2);
            d.setVisible(true);

        } else {

            throw new NullPointerException("Could not create about dialog. About dialog is null.");
        }
*/
    }

    //
    // Change display mode.
    //

    /**
     * Changes the display mode.
     *
     * If the display mode is changed, the following will happen:
     * - the views of all child controllers of this controller are destroyed
     * - the view of this controller is destroyed
     * - the actual display mode value is set
     * - a new view for this controller is created
     * - new views for all child controllers of this controller are created
     *
     * @param mode the display mode
     * @exception NullPointerException if the view is null
     * @exception Exception if the given display mode does not exist
     */
    public void changeDisplayMode(int mode) throws Exception, NullPointerException {

        SwingView v = (SwingView) getView();

        if (v != null) {

            if ((mode == DisplayFactory.FRAME_DISPLAY)
                || (mode == DisplayFactory.INTERNAL_FRAME_DISPLAY)
                || (mode == DisplayFactory.TAB_PAGE_DISPLAY)) {

                if (mode != v.getDisplayMode()) {

                    // Hide view displays of all child controllers of this controller.
                    hideViews();

                    // Set display mode.
                    v.setDisplayMode(mode);
                    v.setChildApplicationComponent(null);

                    // Show view displays for all child controllers of this controller.
                    showViews();
                }

            } else {

                throw new Exception("Could not change display mode. The given display mode does not exist.");
            }

        } else {

            throw new NullPointerException("Could not change display mode. The view is null.");
        }
    }

    /**
     * Show view displays for all application views.
     *
     * @exception NullPointerException if the child controller container is null
     */
    protected void showViews() throws Exception, NullPointerException {

        // Determine child controllers.
        List l = getChildren();

        if (l != null) {

            Object o = null;

            for (int i = 0; i < l.size(); i++) {

                // Determine child controller.
                o = l.get(i);

                if (o instanceof BasicApplication) {

                    ((BasicApplication) o).show((View) ((BasicApplication) o).getView());
                }
            }

        } else {

            throw new NullPointerException("Could not show view. The child controller container is null.");
        }
    }

    /**
     * Hide view displays of all application views.
     *
     * @exception NullPointerException if the child controller container is null
     */
    protected void hideViews() throws Exception, NullPointerException {

        // Determine child controllers.
        List l = getChildren();

        if (l != null) {

            Object o = null;

            for (int i = 0; i < l.size(); i++) {

                // Determine child controller.
                o = l.get(i);

                if (o instanceof BasicApplication) {

                    ((BasicApplication) o).hide((View) ((BasicApplication) o).getView());
                }
            }

        } else {

            throw new NullPointerException("Could not hide view. The child controller container is null.");
        }
    }

    //
    // Temporary event listeners. Will be deleted as soon as Scope/ResMedLib supports Trees.
    //

    /**
     * Listens and reacts to tree selection events.
     *
     * ?? Delete this method as soon as a ResTree component with event/control handling is provided!
     * ?? The direct call of Controller in this method is very unclean design!
     * ?? The Controller should only react to Control Events of the MVC framework!
     *
     * @param evt the tree selection event
     */
    public void valueChanged(TreeSelectionEvent evt) {

        // Determine application tree view as the event's source.
        ResTree tv = (ResTree) evt.getSource();

        if (tv != null) {

            // Determine selected application tree node.
            ApplicationTreeNode n = (ApplicationTreeNode) tv.getLastSelectedPathComponent();

            if (n != null) {

                if (n.isLeaf()) {

                    if (n.isExternal() == false) {

                        try {

                            // Create the application corresponding to the clicked tree node.
                            createApplication(n.getLocation(), n.getArguments(), n.getWorkPath());

                        } catch (Exception e) {

//??                            log(Level.SEVERE, "Error while creating application.", e);
                            System.out.println("Error while creating application." + e);
                        }

                    } else {

                        try {

                            // Create the external application corresponding to the clicked tree node.
                            createExternalApplication(n.getLocation()/*??getCommand()*/, n.getArguments(), n.getWorkPath());

                        } catch (Exception e) {

//??                            log(Level.SEVERE, "Error while creating external application.", e);
                            System.out.println("Error while creating external application." + e);
                        }
                    }
                }
            }
        }
    }
}

