/*
 * $RCSfile: CommandLineArgumentsParser.java,v $
 *
 * Copyright (c) 1999-2002. Christian Heller. All rights reserved.
 *
 * This software is published under the GPL GNU General Public License.
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * http://www.resmedicinae.org
 * - Information in Medicine -
 */

package org.resmedicinae.application.common.main;

import java.lang.*;
import org.resmedicinae.resmedlib.component.component.*;

/**
 * This class represents a parser for command line arguments.
 *
 * It reads these arguments or applet parameters when they are handed over
 * to the system at application start.
 *
 * Valid parameters and their meaning:
 * -system  indicates that the system configuration location is following
 * -local   indicates that the local configuration location is following
 * -user    indicates that the user configuration location is following
 * -help    to displays help information
 *
 * @version $Revision: 1.2 $ $Date: 2002/06/12 18:49:33 $ $Author: chrissy $
 * @author Christian Heller <christian.heller@tuxtax.de>
 */
public class CommandLineArgumentsParser extends Component {

    //
    // Class attributes (constants).
    //

    /** The argument indicating the location of the system configuration. */
    private static final String SYSTEM_CONFIGURATION = "-system";

    /** The argument indicating the location of the local configuration. */
    private static final String LOCAL_CONFIGURATION = "-local";

    /** The argument indicating the location of the user configuration. */
    private static final String USER_CONFIGURATION = "-user";

    /** The argument indicating the location of the user configuration. */
    private static final String HELP = "-help";

    //
    // Parse methods.
    //

    /**
     * Parses the given command line arguments and stores the configuration
     * locations in an array following the order: system, local, user
     *
     * @param args the command line arguments
     * @return the actual configuration locations
     * @exception NullPointerException if the given tui argument is invalid
     */
    public String[] parse(String[] args) throws NullPointerException {

        String[] s = new String[3];

        if (s != null) {

            // Set default values.
            s[0] = "";
            s[1] = "";
            s[2] = "";

            if (args != null) {
                    
                for (int i = 0; i < args.length; i++) {
    
                    if (args[i].equals(CommandLineArgumentsParser.SYSTEM_CONFIGURATION)) {
    
                        s[0] = args[i + 1];
    
                    } else if (args[i].equals(CommandLineArgumentsParser.LOCAL_CONFIGURATION)) {
    
                        s[1] = args[i + 1];
    
                    } else if (args[i].equals(CommandLineArgumentsParser.USER_CONFIGURATION)) {
    
                        s[2] = args[i + 1];
    
                    } else if (args[i].equals(CommandLineArgumentsParser.HELP)) {
    
                        displayHelp();
                    }
                }
            }

        } else {

            throw new NullPointerException("Could not write parsed arguments. The list is null.");
        }

        return s;
    }

    /**
     * Displays command line help information.
     */
    private void displayHelp() {

        System.out.println("Usage: java -jar resmedicinae.jar [-system location] [-local location] [-user location]\n"
            + "where:\n"
            + "[-system] indicates the location of the system configuration,\n"
            + "[-local] indicates the location of the local configuration\n"
            + "[-user] indicates the location of the user configuration\n\n"
            + "If you don't know how to handle this, just omit parameters"
            + "to start the application with default settings.");
    }
}

