/*
 * Copyright (C) 1999-2023. Christian Heller.
 *
 * This file is part of the Cybernetics Oriented Interpreter (CYBOI).
 *
 * CYBOI is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published
 * by the Free Software Foundation, either version 3 of the License,
 * or (at your option) any later version.
 *
 * CYBOI is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty
 * of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with CYBOI. If not, see <http://www.gnu.org/licenses/>.
 *
 * Cybernetics Oriented Programming (CYBOP) <http://www.cybop.org/>
 * CYBOP Developers <cybop-developers@nongnu.org>
 *
 * @version CYBOP 0.26.0 2023-04-04
 * @author Christian Heller <christian.heller@cybop.org>
 */

#ifndef UNICODE_CHARACTER_CODE_MODEL_CONSTANT_HEADER
#define UNICODE_CHARACTER_CODE_MODEL_CONSTANT_HEADER

//
// System interface
//

#include <stddef.h> // wchar_t

//
// Library interface
//

#include "constant.h"

//
// A "Character Set" consists of three parts:
// - Character Repertoire: characters or symbols, e.g. ISO 8859-1 with 256 characters and Unicode with ~ 1 Mio. characters
// - Character Code: table assigning numbers, e.g. a = 97, b = 98, c = 99 etc.
// - Character Encoding: byte sequence storing code numbers, e.g. utf-8 with 97 = 01100001, 98 = 01100010, 99 = 01100011 etc.
//
// This file contains wide character CODE constants.
//

//
// Basic Latin Block
//
// Range:
// hexadecimal: U+0000 - U+007F
// decimal: 0 - 127
//

/**
 * The null unicode character code model. U+0000
 * It is used as string termination in the C programming language.
 */
static wchar_t NULL_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0000 };
static wchar_t* NULL_UNICODE_CHARACTER_CODE_MODEL = NULL_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* NULL_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/** The start of heading unicode character code model. U+0001 */
static wchar_t START_OF_HEADING_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0001 };
static wchar_t* START_OF_HEADING_UNICODE_CHARACTER_CODE_MODEL = START_OF_HEADING_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* START_OF_HEADING_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/** The start of text unicode character code model. U+0002 */
static wchar_t START_OF_TEXT_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0002 };
static wchar_t* START_OF_TEXT_UNICODE_CHARACTER_CODE_MODEL = START_OF_TEXT_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* START_OF_TEXT_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/** The end of text unicode character code model. U+0003 */
static wchar_t END_OF_TEXT_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0003 };
static wchar_t* END_OF_TEXT_UNICODE_CHARACTER_CODE_MODEL = END_OF_TEXT_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* END_OF_TEXT_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/** The end of transmission unicode character code model. U+0004 */
static wchar_t END_OF_TRANSMISSION_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0004 };
static wchar_t* END_OF_TRANSMISSION_UNICODE_CHARACTER_CODE_MODEL = END_OF_TRANSMISSION_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* END_OF_TRANSMISSION_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/** The enquiry unicode character code model. U+0005 */
static wchar_t ENQUIRY_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0005 };
static wchar_t* ENQUIRY_UNICODE_CHARACTER_CODE_MODEL = ENQUIRY_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* ENQUIRY_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/** The acknowledge unicode character code model. U+0006 */
static wchar_t ACKNOWLEDGE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0006 };
static wchar_t* ACKNOWLEDGE_UNICODE_CHARACTER_CODE_MODEL = ACKNOWLEDGE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* ACKNOWLEDGE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/** The bell unicode character code model. U+0007 */
static wchar_t BELL_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0007 };
static wchar_t* BELL_UNICODE_CHARACTER_CODE_MODEL = BELL_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* BELL_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/** The backspace unicode character code model. U+0008 */
static wchar_t BACKSPACE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0008 };
static wchar_t* BACKSPACE_UNICODE_CHARACTER_CODE_MODEL = BACKSPACE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* BACKSPACE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/**
 * The character tabulation unicode character code model.
 *
 * U+0009
 *
 * Alias names:
 * HORIZONTAL TAB
 */
static wchar_t CHARACTER_TABULATION_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0009 };
static wchar_t* CHARACTER_TABULATION_UNICODE_CHARACTER_CODE_MODEL = CHARACTER_TABULATION_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CHARACTER_TABULATION_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/**
 * The line feed unicode character code model.
 *
 * U+000A
 *
 * Alias names:
 * LINE FEED (LF)
 * NEW LINE (NL)
 * END OF LINE (EOL)
 */
static wchar_t LINE_FEED_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x000A };
static wchar_t* LINE_FEED_UNICODE_CHARACTER_CODE_MODEL = LINE_FEED_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LINE_FEED_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/**
 * The line tabulation unicode character code model.
 *
 * U+000B
 *
 * Alias names:
 * VERTICAL TAB
 */
static wchar_t LINE_TABULATION_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x000B };
static wchar_t* LINE_TABULATION_UNICODE_CHARACTER_CODE_MODEL = LINE_TABULATION_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LINE_TABULATION_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/**
 * The form feed unicode character code model.
 *
 * U+000C
 *
 * Alias names:
 * NEW PAGE (NP)
 */
static wchar_t FORM_FEED_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x000C };
static wchar_t* FORM_FEED_UNICODE_CHARACTER_CODE_MODEL = FORM_FEED_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* FORM_FEED_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/** The carriage return unicode character code model. U+000D */
static wchar_t CARRIAGE_RETURN_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x000D };
static wchar_t* CARRIAGE_RETURN_UNICODE_CHARACTER_CODE_MODEL = CARRIAGE_RETURN_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CARRIAGE_RETURN_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/** The shift out unicode character code model. U+000E */
static wchar_t SHIFT_OUT_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x000E };
static wchar_t* SHIFT_OUT_UNICODE_CHARACTER_CODE_MODEL = SHIFT_OUT_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* SHIFT_OUT_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/** The shift in unicode character code model. U+000F */
static wchar_t SHIFT_IN_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x000F };
static wchar_t* SHIFT_IN_UNICODE_CHARACTER_CODE_MODEL = SHIFT_IN_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* SHIFT_IN_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/** The data link escape unicode character code model. U+0010 */
static wchar_t DATA_LINK_ESCAPE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0010 };
static wchar_t* DATA_LINK_ESCAPE_UNICODE_CHARACTER_CODE_MODEL = DATA_LINK_ESCAPE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* DATA_LINK_ESCAPE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/** The device control one unicode character code model. U+0011 */
static wchar_t DEVICE_CONTROL_ONE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0011 };
static wchar_t* DEVICE_CONTROL_ONE_UNICODE_CHARACTER_CODE_MODEL = DEVICE_CONTROL_ONE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* DEVICE_CONTROL_ONE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/** The device control two unicode character code model. U+0012 */
static wchar_t DEVICE_CONTROL_TWO_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0012 };
static wchar_t* DEVICE_CONTROL_TWO_UNICODE_CHARACTER_CODE_MODEL = DEVICE_CONTROL_TWO_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* DEVICE_CONTROL_TWO_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/** The device control three unicode character code model. U+0013 */
static wchar_t DEVICE_CONTROL_THREE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0013 };
static wchar_t* DEVICE_CONTROL_THREE_UNICODE_CHARACTER_CODE_MODEL = DEVICE_CONTROL_THREE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* DEVICE_CONTROL_THREE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/** The device control four unicode character code model. U+0014 */
static wchar_t DEVICE_CONTROL_FOUR_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0014 };
static wchar_t* DEVICE_CONTROL_FOUR_UNICODE_CHARACTER_CODE_MODEL = DEVICE_CONTROL_FOUR_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* DEVICE_CONTROL_FOUR_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/** The negative acknowledge unicode character code model. U+0015 */
static wchar_t NEGATIVE_ACKNOWLEDGE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0015 };
static wchar_t* NEGATIVE_ACKNOWLEDGE_UNICODE_CHARACTER_CODE_MODEL = NEGATIVE_ACKNOWLEDGE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* NEGATIVE_ACKNOWLEDGE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/** The synchronous idle unicode character code model. U+0016 */
static wchar_t SYNCHRONOUS_IDLE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0016 };
static wchar_t* SYNCHRONOUS_IDLE_UNICODE_CHARACTER_CODE_MODEL = SYNCHRONOUS_IDLE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* SYNCHRONOUS_IDLE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/** The end of transmission block unicode character code model. U+0017 */
static wchar_t END_OF_TRANSMISSION_BLOCK_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0017 };
static wchar_t* END_OF_TRANSMISSION_BLOCK_UNICODE_CHARACTER_CODE_MODEL = END_OF_TRANSMISSION_BLOCK_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* END_OF_TRANSMISSION_BLOCK_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/** The cancel unicode character code model. U+0018 */
static wchar_t CANCEL_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0018 };
static wchar_t* CANCEL_UNICODE_CHARACTER_CODE_MODEL = CANCEL_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CANCEL_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/** The end of medium unicode character code model. U+0019 */
static wchar_t END_OF_MEDIUM_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0019 };
static wchar_t* END_OF_MEDIUM_UNICODE_CHARACTER_CODE_MODEL = END_OF_MEDIUM_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* END_OF_MEDIUM_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/** The substitute unicode character code model. U+001A */
static wchar_t SUBSTITUTE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x001A };
static wchar_t* SUBSTITUTE_UNICODE_CHARACTER_CODE_MODEL = SUBSTITUTE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* SUBSTITUTE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/** The escape unicode character code model. U+001B */
static wchar_t ESCAPE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x001B };
static wchar_t* ESCAPE_UNICODE_CHARACTER_CODE_MODEL = ESCAPE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* ESCAPE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/** The file separator unicode character code model. U+001C */
static wchar_t FILE_SEPARATOR_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x001C };
static wchar_t* FILE_SEPARATOR_UNICODE_CHARACTER_CODE_MODEL = FILE_SEPARATOR_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* FILE_SEPARATOR_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/** The group separator unicode character code model. U+001D */
static wchar_t GROUP_SEPARATOR_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x001D };
static wchar_t* GROUP_SEPARATOR_UNICODE_CHARACTER_CODE_MODEL = GROUP_SEPARATOR_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* GROUP_SEPARATOR_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/** The record separator unicode character code model. U+001E */
static wchar_t RECORD_SEPARATOR_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x001E };
static wchar_t* RECORD_SEPARATOR_UNICODE_CHARACTER_CODE_MODEL = RECORD_SEPARATOR_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* RECORD_SEPARATOR_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/** The unit separator unicode character code model. U+001F */
static wchar_t UNIT_SEPARATOR_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x001F };
static wchar_t* UNIT_SEPARATOR_UNICODE_CHARACTER_CODE_MODEL = UNIT_SEPARATOR_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* UNIT_SEPARATOR_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/** The space unicode character code model. U+0020 */
static wchar_t SPACE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0020 };
static wchar_t* SPACE_UNICODE_CHARACTER_CODE_MODEL = SPACE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* SPACE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/** The exclamation mark unicode character code model. U+0021 */
static wchar_t EXCLAMATION_MARK_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0021 };
static wchar_t* EXCLAMATION_MARK_UNICODE_CHARACTER_CODE_MODEL = EXCLAMATION_MARK_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* EXCLAMATION_MARK_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/** The quotation mark unicode character code model. U+0022 */
static wchar_t QUOTATION_MARK_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0022 };
static wchar_t* QUOTATION_MARK_UNICODE_CHARACTER_CODE_MODEL = QUOTATION_MARK_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* QUOTATION_MARK_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/** The number sign unicode character code model. U+0023 */
static wchar_t NUMBER_SIGN_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0023 };
static wchar_t* NUMBER_SIGN_UNICODE_CHARACTER_CODE_MODEL = NUMBER_SIGN_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* NUMBER_SIGN_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/** The dollar sign unicode character code model. U+0024 */
static wchar_t DOLLAR_SIGN_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0024 };
static wchar_t* DOLLAR_SIGN_UNICODE_CHARACTER_CODE_MODEL = DOLLAR_SIGN_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* DOLLAR_SIGN_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/** The percent sign unicode character code model. U+0025 */
static wchar_t PERCENT_SIGN_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0025 };
static wchar_t* PERCENT_SIGN_UNICODE_CHARACTER_CODE_MODEL = PERCENT_SIGN_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* PERCENT_SIGN_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/** The ampersand unicode character code model. U+0026 */
static wchar_t AMPERSAND_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0026 };
static wchar_t* AMPERSAND_UNICODE_CHARACTER_CODE_MODEL = AMPERSAND_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* AMPERSAND_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/** The apostrophe unicode character code model. U+0027 */
static wchar_t APOSTROPHE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0027 };
static wchar_t* APOSTROPHE_UNICODE_CHARACTER_CODE_MODEL = APOSTROPHE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* APOSTROPHE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/** The left parenthesis unicode character code model. U+0028 */
static wchar_t LEFT_PARENTHESIS_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0028 };
static wchar_t* LEFT_PARENTHESIS_UNICODE_CHARACTER_CODE_MODEL = LEFT_PARENTHESIS_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LEFT_PARENTHESIS_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/** The right parenthesis unicode character code model. U+0029 */
static wchar_t RIGHT_PARENTHESIS_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0029 };
static wchar_t* RIGHT_PARENTHESIS_UNICODE_CHARACTER_CODE_MODEL = RIGHT_PARENTHESIS_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* RIGHT_PARENTHESIS_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/** The asterisk unicode character code model. U+002A */
static wchar_t ASTERISK_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x002A };
static wchar_t* ASTERISK_UNICODE_CHARACTER_CODE_MODEL = ASTERISK_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* ASTERISK_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/** The plus sign unicode character code model. U+002B */
static wchar_t PLUS_SIGN_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x002B };
static wchar_t* PLUS_SIGN_UNICODE_CHARACTER_CODE_MODEL = PLUS_SIGN_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* PLUS_SIGN_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/** The comma unicode character code model. U+002C */
static wchar_t COMMA_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x002C };
static wchar_t* COMMA_UNICODE_CHARACTER_CODE_MODEL = COMMA_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* COMMA_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/**
 * The hyphen minus unicode character code model.
 *
 * Unicode category: Punctuation, Dash
 * Unicode: U+002D
 * UTF-8: 0x2D
 * Octal escaped UTF-8: \055
 * Decimal entity reference: &#45;
 *
 * Alias names:
 * hyphen or minus sign
 *
 * Notes:
 * used for either hyphen or minus sign
 *
 * See also:
 * U+2010 HYPHEN
 * U+2011 NON-BREAKING HYPHEN
 * U+2012 FIGURE DASH
 * U+2013 EN DASH
 * U+2212 MINUS SIGN
 */
static wchar_t HYPHEN_MINUS_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x002D };
static wchar_t* HYPHEN_MINUS_UNICODE_CHARACTER_CODE_MODEL = HYPHEN_MINUS_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* HYPHEN_MINUS_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/** The full stop unicode character code model. U+002E */
static wchar_t FULL_STOP_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x002E };
static wchar_t* FULL_STOP_UNICODE_CHARACTER_CODE_MODEL = FULL_STOP_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* FULL_STOP_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/** The solidus (slash) unicode character code model. U+002F */
static wchar_t SOLIDUS_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x002F };
static wchar_t* SOLIDUS_UNICODE_CHARACTER_CODE_MODEL = SOLIDUS_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* SOLIDUS_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/** The digit zero unicode character code model. U+0030 */
static wchar_t DIGIT_ZERO_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0030 };
static wchar_t* DIGIT_ZERO_UNICODE_CHARACTER_CODE_MODEL = DIGIT_ZERO_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* DIGIT_ZERO_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/** The digit one unicode character code model. U+0031 */
static wchar_t DIGIT_ONE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0031 };
static wchar_t* DIGIT_ONE_UNICODE_CHARACTER_CODE_MODEL = DIGIT_ONE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* DIGIT_ONE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/** The digit two unicode character code model. U+0032 */
static wchar_t DIGIT_TWO_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0032 };
static wchar_t* DIGIT_TWO_UNICODE_CHARACTER_CODE_MODEL = DIGIT_TWO_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* DIGIT_TWO_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/** The digit three unicode character code model. U+0033 */
static wchar_t DIGIT_THREE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0033 };
static wchar_t* DIGIT_THREE_UNICODE_CHARACTER_CODE_MODEL = DIGIT_THREE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* DIGIT_THREE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/** The digit four unicode character code model. U+0034 */
static wchar_t DIGIT_FOUR_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0034 };
static wchar_t* DIGIT_FOUR_UNICODE_CHARACTER_CODE_MODEL = DIGIT_FOUR_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* DIGIT_FOUR_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/** The digit five unicode character code model. U+0035 */
static wchar_t DIGIT_FIVE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0035 };
static wchar_t* DIGIT_FIVE_UNICODE_CHARACTER_CODE_MODEL = DIGIT_FIVE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* DIGIT_FIVE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/** The digit six unicode character code model. U+0036 */
static wchar_t DIGIT_SIX_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0036 };
static wchar_t* DIGIT_SIX_UNICODE_CHARACTER_CODE_MODEL = DIGIT_SIX_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* DIGIT_SIX_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/** The digit seven unicode character code model. U+0037 */
static wchar_t DIGIT_SEVEN_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0037 };
static wchar_t* DIGIT_SEVEN_UNICODE_CHARACTER_CODE_MODEL = DIGIT_SEVEN_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* DIGIT_SEVEN_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/** The digit eight unicode character code model. U+0038 */
static wchar_t DIGIT_EIGHT_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0038 };
static wchar_t* DIGIT_EIGHT_UNICODE_CHARACTER_CODE_MODEL = DIGIT_EIGHT_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* DIGIT_EIGHT_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/** The digit nine unicode character code model. U+0039 */
static wchar_t DIGIT_NINE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0039 };
static wchar_t* DIGIT_NINE_UNICODE_CHARACTER_CODE_MODEL = DIGIT_NINE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* DIGIT_NINE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/** The colon unicode character code model. U+003A */
static wchar_t COLON_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x003A };
static wchar_t* COLON_UNICODE_CHARACTER_CODE_MODEL = COLON_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* COLON_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/** The semicolon unicode character code model. U+003B */
static wchar_t SEMICOLON_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x003B };
static wchar_t* SEMICOLON_UNICODE_CHARACTER_CODE_MODEL = SEMICOLON_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* SEMICOLON_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/** The less than sign unicode character code model. U+003C */
static wchar_t LESS_THAN_SIGN_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x003C };
static wchar_t* LESS_THAN_SIGN_UNICODE_CHARACTER_CODE_MODEL = LESS_THAN_SIGN_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LESS_THAN_SIGN_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/**
 * The equals sign unicode character code model.
 *
 * Unicode category: Symbol, Math
 * Unicode: U+003D
 * UTF-8: 0x3D
 * Octal escaped UTF-8: \075
 * Decimal entity reference: &#61;
 *
 * Notes:
 * other related characters: U+2241 NOT TILDE-U+2263 STRICTLY EQUIVALENT TO
 *
 * See also:
 * U+2260 NOT EQUAL TO
 * U+2261 IDENTICAL TO
 */
static wchar_t EQUALS_SIGN_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x003D };
static wchar_t* EQUALS_SIGN_UNICODE_CHARACTER_CODE_MODEL = EQUALS_SIGN_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* EQUALS_SIGN_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/** The greater than sign unicode character code model. U+003E */
static wchar_t GREATER_THAN_SIGN_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x003E };
static wchar_t* GREATER_THAN_SIGN_UNICODE_CHARACTER_CODE_MODEL = GREATER_THAN_SIGN_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* GREATER_THAN_SIGN_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/** The question mark unicode character code model. U+003F */
static wchar_t QUESTION_MARK_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x003F };
static wchar_t* QUESTION_MARK_UNICODE_CHARACTER_CODE_MODEL = QUESTION_MARK_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* QUESTION_MARK_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/** The commercial at unicode character code model. U+0040 */
static wchar_t COMMERCIAL_AT_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0040 };
static wchar_t* COMMERCIAL_AT_UNICODE_CHARACTER_CODE_MODEL = COMMERCIAL_AT_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* COMMERCIAL_AT_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/** The latin capital letter a unicode character code model. U+0041 */
static wchar_t LATIN_CAPITAL_LETTER_A_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0041 };
static wchar_t* LATIN_CAPITAL_LETTER_A_UNICODE_CHARACTER_CODE_MODEL = LATIN_CAPITAL_LETTER_A_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_CAPITAL_LETTER_A_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/** The latin capital letter b unicode character code model. U+0042 */
static wchar_t LATIN_CAPITAL_LETTER_B_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0042 };
static wchar_t* LATIN_CAPITAL_LETTER_B_UNICODE_CHARACTER_CODE_MODEL = LATIN_CAPITAL_LETTER_B_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_CAPITAL_LETTER_B_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/** The latin capital letter c unicode character code model. U+0043 */
static wchar_t LATIN_CAPITAL_LETTER_C_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0043 };
static wchar_t* LATIN_CAPITAL_LETTER_C_UNICODE_CHARACTER_CODE_MODEL = LATIN_CAPITAL_LETTER_C_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_CAPITAL_LETTER_C_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/** The latin capital letter d unicode character code model. U+0044 */
static wchar_t LATIN_CAPITAL_LETTER_D_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0044 };
static wchar_t* LATIN_CAPITAL_LETTER_D_UNICODE_CHARACTER_CODE_MODEL = LATIN_CAPITAL_LETTER_D_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_CAPITAL_LETTER_D_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/** The latin capital letter e unicode character code model. U+0045 */
static wchar_t LATIN_CAPITAL_LETTER_E_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0045 };
static wchar_t* LATIN_CAPITAL_LETTER_E_UNICODE_CHARACTER_CODE_MODEL = LATIN_CAPITAL_LETTER_E_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_CAPITAL_LETTER_E_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/** The latin capital letter f unicode character code model. U+0046 */
static wchar_t LATIN_CAPITAL_LETTER_F_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0046 };
static wchar_t* LATIN_CAPITAL_LETTER_F_UNICODE_CHARACTER_CODE_MODEL = LATIN_CAPITAL_LETTER_F_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_CAPITAL_LETTER_F_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/** The latin capital letter g unicode character code model. U+0047 */
static wchar_t LATIN_CAPITAL_LETTER_G_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0047 };
static wchar_t* LATIN_CAPITAL_LETTER_G_UNICODE_CHARACTER_CODE_MODEL = LATIN_CAPITAL_LETTER_G_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_CAPITAL_LETTER_G_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/** The latin capital letter h unicode character code model. U+0048 */
static wchar_t LATIN_CAPITAL_LETTER_H_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0048 };
static wchar_t* LATIN_CAPITAL_LETTER_H_UNICODE_CHARACTER_CODE_MODEL = LATIN_CAPITAL_LETTER_H_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_CAPITAL_LETTER_H_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/** The latin capital letter i unicode character code model. U+0049 */
static wchar_t LATIN_CAPITAL_LETTER_I_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0049 };
static wchar_t* LATIN_CAPITAL_LETTER_I_UNICODE_CHARACTER_CODE_MODEL = LATIN_CAPITAL_LETTER_I_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_CAPITAL_LETTER_I_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/** The latin capital letter j unicode character code model. U+004A */
static wchar_t LATIN_CAPITAL_LETTER_J_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x004A };
static wchar_t* LATIN_CAPITAL_LETTER_J_UNICODE_CHARACTER_CODE_MODEL = LATIN_CAPITAL_LETTER_J_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_CAPITAL_LETTER_J_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/** The latin capital letter k unicode character code model. U+004B */
static wchar_t LATIN_CAPITAL_LETTER_K_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x004B };
static wchar_t* LATIN_CAPITAL_LETTER_K_UNICODE_CHARACTER_CODE_MODEL = LATIN_CAPITAL_LETTER_K_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_CAPITAL_LETTER_K_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/** The latin capital letter l unicode character code model. U+004C */
static wchar_t LATIN_CAPITAL_LETTER_L_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x004C };
static wchar_t* LATIN_CAPITAL_LETTER_L_UNICODE_CHARACTER_CODE_MODEL = LATIN_CAPITAL_LETTER_L_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_CAPITAL_LETTER_L_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/** The latin capital letter m unicode character code model. U+004D */
static wchar_t LATIN_CAPITAL_LETTER_M_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x004D };
static wchar_t* LATIN_CAPITAL_LETTER_M_UNICODE_CHARACTER_CODE_MODEL = LATIN_CAPITAL_LETTER_M_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_CAPITAL_LETTER_M_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/** The latin capital letter n unicode character code model. U+004E */
static wchar_t LATIN_CAPITAL_LETTER_N_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x004E };
static wchar_t* LATIN_CAPITAL_LETTER_N_UNICODE_CHARACTER_CODE_MODEL = LATIN_CAPITAL_LETTER_N_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_CAPITAL_LETTER_N_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/** The latin capital letter o unicode character code model. U+004F */
static wchar_t LATIN_CAPITAL_LETTER_O_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x004F };
static wchar_t* LATIN_CAPITAL_LETTER_O_UNICODE_CHARACTER_CODE_MODEL = LATIN_CAPITAL_LETTER_O_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_CAPITAL_LETTER_O_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/** The latin capital letter p unicode character code model. U+0050 */
static wchar_t LATIN_CAPITAL_LETTER_P_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0050 };
static wchar_t* LATIN_CAPITAL_LETTER_P_UNICODE_CHARACTER_CODE_MODEL = LATIN_CAPITAL_LETTER_P_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_CAPITAL_LETTER_P_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/** The latin capital letter q unicode character code model. U+0051 */
static wchar_t LATIN_CAPITAL_LETTER_Q_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0051 };
static wchar_t* LATIN_CAPITAL_LETTER_Q_UNICODE_CHARACTER_CODE_MODEL = LATIN_CAPITAL_LETTER_Q_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_CAPITAL_LETTER_Q_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/** The latin capital letter r unicode character code model. U+0052 */
static wchar_t LATIN_CAPITAL_LETTER_R_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0052 };
static wchar_t* LATIN_CAPITAL_LETTER_R_UNICODE_CHARACTER_CODE_MODEL = LATIN_CAPITAL_LETTER_R_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_CAPITAL_LETTER_R_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/** The latin capital letter s unicode character code model. U+0053 */
static wchar_t LATIN_CAPITAL_LETTER_S_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0053 };
static wchar_t* LATIN_CAPITAL_LETTER_S_UNICODE_CHARACTER_CODE_MODEL = LATIN_CAPITAL_LETTER_S_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_CAPITAL_LETTER_S_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/** The latin capital letter t unicode character code model. U+0054 */
static wchar_t LATIN_CAPITAL_LETTER_T_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0054 };
static wchar_t* LATIN_CAPITAL_LETTER_T_UNICODE_CHARACTER_CODE_MODEL = LATIN_CAPITAL_LETTER_T_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_CAPITAL_LETTER_T_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/** The latin capital letter u unicode character code model. U+0055 */
static wchar_t LATIN_CAPITAL_LETTER_U_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0055 };
static wchar_t* LATIN_CAPITAL_LETTER_U_UNICODE_CHARACTER_CODE_MODEL = LATIN_CAPITAL_LETTER_U_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_CAPITAL_LETTER_U_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/** The latin capital letter v unicode character code model. U+0056 */
static wchar_t LATIN_CAPITAL_LETTER_V_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0056 };
static wchar_t* LATIN_CAPITAL_LETTER_V_UNICODE_CHARACTER_CODE_MODEL = LATIN_CAPITAL_LETTER_V_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_CAPITAL_LETTER_V_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/** The latin capital letter w unicode character code model. U+0057 */
static wchar_t LATIN_CAPITAL_LETTER_W_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0057 };
static wchar_t* LATIN_CAPITAL_LETTER_W_UNICODE_CHARACTER_CODE_MODEL = LATIN_CAPITAL_LETTER_W_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_CAPITAL_LETTER_W_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/** The latin capital letter x unicode character code model. U+0058 */
static wchar_t LATIN_CAPITAL_LETTER_X_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0058 };
static wchar_t* LATIN_CAPITAL_LETTER_X_UNICODE_CHARACTER_CODE_MODEL = LATIN_CAPITAL_LETTER_X_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_CAPITAL_LETTER_X_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/** The latin capital letter y unicode character code model. U+0059 */
static wchar_t LATIN_CAPITAL_LETTER_Y_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0059 };
static wchar_t* LATIN_CAPITAL_LETTER_Y_UNICODE_CHARACTER_CODE_MODEL = LATIN_CAPITAL_LETTER_Y_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_CAPITAL_LETTER_Y_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/** The latin capital letter z unicode character code model. U+005A */
static wchar_t LATIN_CAPITAL_LETTER_Z_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x005A };
static wchar_t* LATIN_CAPITAL_LETTER_Z_UNICODE_CHARACTER_CODE_MODEL = LATIN_CAPITAL_LETTER_Z_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_CAPITAL_LETTER_Z_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/** The left square bracket unicode character code model. U+005B */
static wchar_t LEFT_SQUARE_BRACKET_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x005B };
static wchar_t* LEFT_SQUARE_BRACKET_UNICODE_CHARACTER_CODE_MODEL = LEFT_SQUARE_BRACKET_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LEFT_SQUARE_BRACKET_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/** The reverse solidus (backslash) unicode character code model. U+005C */
static wchar_t REVERSE_SOLIDUS_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x005C };
static wchar_t* REVERSE_SOLIDUS_UNICODE_CHARACTER_CODE_MODEL = REVERSE_SOLIDUS_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* REVERSE_SOLIDUS_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/** The right square bracket unicode character code model. U+005D */
static wchar_t RIGHT_SQUARE_BRACKET_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x005D };
static wchar_t* RIGHT_SQUARE_BRACKET_UNICODE_CHARACTER_CODE_MODEL = RIGHT_SQUARE_BRACKET_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* RIGHT_SQUARE_BRACKET_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/** The circumflex accent unicode character code model. U+005E */
static wchar_t CIRCUMFLEX_ACCENT_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x005E };
static wchar_t* CIRCUMFLEX_ACCENT_UNICODE_CHARACTER_CODE_MODEL = CIRCUMFLEX_ACCENT_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CIRCUMFLEX_ACCENT_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/** The low line unicode character code model. U+005F */
static wchar_t LOW_LINE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x005F };
static wchar_t* LOW_LINE_UNICODE_CHARACTER_CODE_MODEL = LOW_LINE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LOW_LINE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/** The grave accent unicode character code model. U+0060 */
static wchar_t GRAVE_ACCENT_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0060 };
static wchar_t* GRAVE_ACCENT_UNICODE_CHARACTER_CODE_MODEL = GRAVE_ACCENT_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* GRAVE_ACCENT_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/** The latin small letter a unicode character code model. U+0061 */
static wchar_t LATIN_SMALL_LETTER_A_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0061 };
static wchar_t* LATIN_SMALL_LETTER_A_UNICODE_CHARACTER_CODE_MODEL = LATIN_SMALL_LETTER_A_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_SMALL_LETTER_A_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/** The latin small letter b unicode character code model. U+0062 */
static wchar_t LATIN_SMALL_LETTER_B_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0062 };
static wchar_t* LATIN_SMALL_LETTER_B_UNICODE_CHARACTER_CODE_MODEL = LATIN_SMALL_LETTER_B_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_SMALL_LETTER_B_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/** The latin small letter c unicode character code model. U+0063 */
static wchar_t LATIN_SMALL_LETTER_C_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0063 };
static wchar_t* LATIN_SMALL_LETTER_C_UNICODE_CHARACTER_CODE_MODEL = LATIN_SMALL_LETTER_C_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_SMALL_LETTER_C_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/** The latin small letter d unicode character code model. U+0064 */
static wchar_t LATIN_SMALL_LETTER_D_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0064 };
static wchar_t* LATIN_SMALL_LETTER_D_UNICODE_CHARACTER_CODE_MODEL = LATIN_SMALL_LETTER_D_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_SMALL_LETTER_D_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/** The latin small letter e unicode character code model. U+0065 */
static wchar_t LATIN_SMALL_LETTER_E_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0065 };
static wchar_t* LATIN_SMALL_LETTER_E_UNICODE_CHARACTER_CODE_MODEL = LATIN_SMALL_LETTER_E_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_SMALL_LETTER_E_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/** The latin small letter f unicode character code model. U+0066 */
static wchar_t LATIN_SMALL_LETTER_F_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0066 };
static wchar_t* LATIN_SMALL_LETTER_F_UNICODE_CHARACTER_CODE_MODEL = LATIN_SMALL_LETTER_F_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_SMALL_LETTER_F_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/** The latin small letter g unicode character code model. U+0067 */
static wchar_t LATIN_SMALL_LETTER_G_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0067 };
static wchar_t* LATIN_SMALL_LETTER_G_UNICODE_CHARACTER_CODE_MODEL = LATIN_SMALL_LETTER_G_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_SMALL_LETTER_G_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/** The latin small letter h unicode character code model. U+0068 */
static wchar_t LATIN_SMALL_LETTER_H_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0068 };
static wchar_t* LATIN_SMALL_LETTER_H_UNICODE_CHARACTER_CODE_MODEL = LATIN_SMALL_LETTER_H_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_SMALL_LETTER_H_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/** The latin small letter i unicode character code model. U+0069 */
static wchar_t LATIN_SMALL_LETTER_I_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0069 };
static wchar_t* LATIN_SMALL_LETTER_I_UNICODE_CHARACTER_CODE_MODEL = LATIN_SMALL_LETTER_I_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_SMALL_LETTER_I_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/** The latin small letter j unicode character code model. U+006A */
static wchar_t LATIN_SMALL_LETTER_J_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x006A };
static wchar_t* LATIN_SMALL_LETTER_J_UNICODE_CHARACTER_CODE_MODEL = LATIN_SMALL_LETTER_J_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_SMALL_LETTER_J_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/** The latin small letter k unicode character code model. U+006B */
static wchar_t LATIN_SMALL_LETTER_K_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x006B };
static wchar_t* LATIN_SMALL_LETTER_K_UNICODE_CHARACTER_CODE_MODEL = LATIN_SMALL_LETTER_K_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_SMALL_LETTER_K_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/** The latin small letter l unicode character code model. U+006C */
static wchar_t LATIN_SMALL_LETTER_L_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x006C };
static wchar_t* LATIN_SMALL_LETTER_L_UNICODE_CHARACTER_CODE_MODEL = LATIN_SMALL_LETTER_L_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_SMALL_LETTER_L_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/** The latin small letter m unicode character code model. U+006D */
static wchar_t LATIN_SMALL_LETTER_M_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x006D };
static wchar_t* LATIN_SMALL_LETTER_M_UNICODE_CHARACTER_CODE_MODEL = LATIN_SMALL_LETTER_M_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_SMALL_LETTER_M_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/** The latin small letter n unicode character code model. U+006E */
static wchar_t LATIN_SMALL_LETTER_N_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x006E };
static wchar_t* LATIN_SMALL_LETTER_N_UNICODE_CHARACTER_CODE_MODEL = LATIN_SMALL_LETTER_N_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_SMALL_LETTER_N_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/** The latin small letter o unicode character code model. U+006F */
static wchar_t LATIN_SMALL_LETTER_O_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x006F };
static wchar_t* LATIN_SMALL_LETTER_O_UNICODE_CHARACTER_CODE_MODEL = LATIN_SMALL_LETTER_O_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_SMALL_LETTER_O_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/** The latin small letter p unicode character code model. U+0070 */
static wchar_t LATIN_SMALL_LETTER_P_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0070 };
static wchar_t* LATIN_SMALL_LETTER_P_UNICODE_CHARACTER_CODE_MODEL = LATIN_SMALL_LETTER_P_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_SMALL_LETTER_P_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/** The latin small letter q unicode character code model. U+0071 */
static wchar_t LATIN_SMALL_LETTER_Q_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0071 };
static wchar_t* LATIN_SMALL_LETTER_Q_UNICODE_CHARACTER_CODE_MODEL = LATIN_SMALL_LETTER_Q_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_SMALL_LETTER_Q_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/** The latin small letter r unicode character code model. U+0072 */
static wchar_t LATIN_SMALL_LETTER_R_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0072 };
static wchar_t* LATIN_SMALL_LETTER_R_UNICODE_CHARACTER_CODE_MODEL = LATIN_SMALL_LETTER_R_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_SMALL_LETTER_R_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/** The latin small letter s unicode character code model. U+0073 */
static wchar_t LATIN_SMALL_LETTER_S_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0073 };
static wchar_t* LATIN_SMALL_LETTER_S_UNICODE_CHARACTER_CODE_MODEL = LATIN_SMALL_LETTER_S_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_SMALL_LETTER_S_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/** The latin small letter t unicode character code model. U+0074 */
static wchar_t LATIN_SMALL_LETTER_T_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0074 };
static wchar_t* LATIN_SMALL_LETTER_T_UNICODE_CHARACTER_CODE_MODEL = LATIN_SMALL_LETTER_T_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_SMALL_LETTER_T_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/** The latin small letter u unicode character code model. U+0075 */
static wchar_t LATIN_SMALL_LETTER_U_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0075 };
static wchar_t* LATIN_SMALL_LETTER_U_UNICODE_CHARACTER_CODE_MODEL = LATIN_SMALL_LETTER_U_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_SMALL_LETTER_U_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/** The latin small letter v unicode character code model. U+0076 */
static wchar_t LATIN_SMALL_LETTER_V_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0076 };
static wchar_t* LATIN_SMALL_LETTER_V_UNICODE_CHARACTER_CODE_MODEL = LATIN_SMALL_LETTER_V_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_SMALL_LETTER_V_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/** The latin small letter w unicode character code model. U+0077 */
static wchar_t LATIN_SMALL_LETTER_W_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0077 };
static wchar_t* LATIN_SMALL_LETTER_W_UNICODE_CHARACTER_CODE_MODEL = LATIN_SMALL_LETTER_W_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_SMALL_LETTER_W_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/** The latin small letter x unicode character code model. U+0078 */
static wchar_t LATIN_SMALL_LETTER_X_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0078 };
static wchar_t* LATIN_SMALL_LETTER_X_UNICODE_CHARACTER_CODE_MODEL = LATIN_SMALL_LETTER_X_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_SMALL_LETTER_X_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/** The latin small letter y unicode character code model. U+0079 */
static wchar_t LATIN_SMALL_LETTER_Y_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0079 };
static wchar_t* LATIN_SMALL_LETTER_Y_UNICODE_CHARACTER_CODE_MODEL = LATIN_SMALL_LETTER_Y_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_SMALL_LETTER_Y_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/** The latin small letter z unicode character code model. U+007A */
static wchar_t LATIN_SMALL_LETTER_Z_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x007A };
static wchar_t* LATIN_SMALL_LETTER_Z_UNICODE_CHARACTER_CODE_MODEL = LATIN_SMALL_LETTER_Z_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_SMALL_LETTER_Z_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/** The left curly bracket unicode character code model. U+007B */
static wchar_t LEFT_CURLY_BRACKET_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x007B };
static wchar_t* LEFT_CURLY_BRACKET_UNICODE_CHARACTER_CODE_MODEL = LEFT_CURLY_BRACKET_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LEFT_CURLY_BRACKET_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/**
 * The vertical line unicode character code model.
 *
 * Unicode category: Symbol, Math
 * Unicode: U+007C
 * UTF-8: 0x7C
 * Octal escaped UTF-8: \174
 * Decimal entity reference: &#124;
 *
 * Alias names:
 * VERTICAL BAR
 *
 * Notes:
 * used in pairs to indicate absolute value
 *
 * See also:
 * U+01C0 LATIN LETTER DENTAL CLICK
 * U+05C0 HEBREW PUNCTUATION PASEQ
 * U+2223 DIVIDES
 * U+2758 LIGHT VERTICAL BAR
 */
static wchar_t VERTICAL_LINE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x007C };
static wchar_t* VERTICAL_LINE_UNICODE_CHARACTER_CODE_MODEL = VERTICAL_LINE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* VERTICAL_LINE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/** The right curly bracket unicode character code model. U+007D */
static wchar_t RIGHT_CURLY_BRACKET_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x007D };
static wchar_t* RIGHT_CURLY_BRACKET_UNICODE_CHARACTER_CODE_MODEL = RIGHT_CURLY_BRACKET_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* RIGHT_CURLY_BRACKET_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/** The tilde unicode character code model. U+007E */
static wchar_t TILDE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x007E };
static wchar_t* TILDE_UNICODE_CHARACTER_CODE_MODEL = TILDE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* TILDE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/** The delete unicode character code model. U+007F */
static wchar_t DELETE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x007F };
static wchar_t* DELETE_UNICODE_CHARACTER_CODE_MODEL = DELETE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* DELETE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

//
// Latin-1 Supplement Block
//
// Range:
// hexadecimal: U+0080 - U+00FF
// decimal: 128 - 255
//

/**
 * The padding character unicode character code model. U+0080
 *
 * Listed as XXX in Unicode. Not part of ISO/IEC 6429 (ECMA-48).
 */
static wchar_t PADDING_CHARACTER_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0080 };
static wchar_t* PADDING_CHARACTER_UNICODE_CHARACTER_CODE_MODEL = PADDING_CHARACTER_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* PADDING_CHARACTER_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/**
 * The high octet preset unicode character code model. U+0081
 *
 * Listed as XXX in Unicode. Not part of ISO/IEC 6429 (ECMA-48).
 */
static wchar_t HIGH_OCTET_PRESET_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0081 };
static wchar_t* HIGH_OCTET_PRESET_UNICODE_CHARACTER_CODE_MODEL = HIGH_OCTET_PRESET_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* HIGH_OCTET_PRESET_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/** The break permitted here unicode character code model. U+0082 */
static wchar_t BREAK_PERMITTED_HERE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0082 };
static wchar_t* BREAK_PERMITTED_HERE_UNICODE_CHARACTER_CODE_MODEL = BREAK_PERMITTED_HERE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* BREAK_PERMITTED_HERE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/** The no break here unicode character code model. U+0083 */
static wchar_t NO_BREAK_HERE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0083 };
static wchar_t* NO_BREAK_HERE_UNICODE_CHARACTER_CODE_MODEL = NO_BREAK_HERE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* NO_BREAK_HERE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/** The index unicode character code model. U+0084 */
static wchar_t INDEX_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0084 };
static wchar_t* INDEX_UNICODE_CHARACTER_CODE_MODEL = INDEX_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* INDEX_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/** The next line unicode character code model. U+0085 */
static wchar_t NEXT_LINE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0085 };
static wchar_t* NEXT_LINE_UNICODE_CHARACTER_CODE_MODEL = NEXT_LINE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* NEXT_LINE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/** The start of selected area unicode character code model. U+0086 */
static wchar_t START_OF_SELECTED_AREA_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0086 };
static wchar_t* START_OF_SELECTED_AREA_UNICODE_CHARACTER_CODE_MODEL = START_OF_SELECTED_AREA_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* START_OF_SELECTED_AREA_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/** The end of selected area unicode character code model. U+0087 */
static wchar_t END_OF_SELECTED_AREA_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0087 };
static wchar_t* END_OF_SELECTED_AREA_UNICODE_CHARACTER_CODE_MODEL = END_OF_SELECTED_AREA_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* END_OF_SELECTED_AREA_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/** The character tabulation set unicode character code model. U+0088 */
static wchar_t CHARACTER_TABULATION_SET_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0088 };
static wchar_t* CHARACTER_TABULATION_SET_UNICODE_CHARACTER_CODE_MODEL = CHARACTER_TABULATION_SET_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CHARACTER_TABULATION_SET_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/** The character tabulation with justification unicode character code model. U+0089 */
static wchar_t CHARACTER_TABULATION_WITH_JUSTIFICATION_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0089 };
static wchar_t* CHARACTER_TABULATION_WITH_JUSTIFICATION_UNICODE_CHARACTER_CODE_MODEL = CHARACTER_TABULATION_WITH_JUSTIFICATION_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CHARACTER_TABULATION_WITH_JUSTIFICATION_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/** The line tabulation set unicode character code model. U+008A */
static wchar_t LINE_TABULATION_SET_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x008A };
static wchar_t* LINE_TABULATION_SET_UNICODE_CHARACTER_CODE_MODEL = LINE_TABULATION_SET_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LINE_TABULATION_SET_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/** The partial line forward unicode character code model. U+008B */
static wchar_t PARTIAL_LINE_FORWARD_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x008B };
static wchar_t* PARTIAL_LINE_FORWARD_UNICODE_CHARACTER_CODE_MODEL = PARTIAL_LINE_FORWARD_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* PARTIAL_LINE_FORWARD_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/** The partial line backward unicode character code model. U+008C */
static wchar_t PARTIAL_LINE_BACKWARD_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x008C };
static wchar_t* PARTIAL_LINE_BACKWARD_UNICODE_CHARACTER_CODE_MODEL = PARTIAL_LINE_BACKWARD_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* PARTIAL_LINE_BACKWARD_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/** The reverse line feed unicode character code model. U+008D */
static wchar_t REVERSE_LINE_FEED_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x008D };
static wchar_t* REVERSE_LINE_FEED_UNICODE_CHARACTER_CODE_MODEL = REVERSE_LINE_FEED_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* REVERSE_LINE_FEED_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/** The single-shift two unicode character code model. U+008E */
static wchar_t SINGLE_SHIFT_TWO_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x008E };
static wchar_t* SINGLE_SHIFT_TWO_UNICODE_CHARACTER_CODE_MODEL = SINGLE_SHIFT_TWO_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* SINGLE_SHIFT_TWO_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/** The single-shift three unicode character code model. U+008F */
static wchar_t SINGLE_SHIFT_THREE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x008F };
static wchar_t* SINGLE_SHIFT_THREE_UNICODE_CHARACTER_CODE_MODEL = SINGLE_SHIFT_THREE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* SINGLE_SHIFT_THREE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/** The device control string unicode character code model. U+0090 */
static wchar_t DEVICE_CONTROL_STRING_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0090 };
static wchar_t* DEVICE_CONTROL_STRING_UNICODE_CHARACTER_CODE_MODEL = DEVICE_CONTROL_STRING_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* DEVICE_CONTROL_STRING_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/** The private use one unicode character code model. U+0091 */
static wchar_t PRIVATE_USE_ONE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0091 };
static wchar_t* PRIVATE_USE_ONE_UNICODE_CHARACTER_CODE_MODEL = PRIVATE_USE_ONE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* PRIVATE_USE_ONE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/** The private use two unicode character code model. U+0092 */
static wchar_t PRIVATE_USE_TWO_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0092 };
static wchar_t* PRIVATE_USE_TWO_UNICODE_CHARACTER_CODE_MODEL = PRIVATE_USE_TWO_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* PRIVATE_USE_TWO_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/** The set transmit state unicode character code model. U+0093 */
static wchar_t SET_TRANSMIT_STATE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0093 };
static wchar_t* SET_TRANSMIT_STATE_UNICODE_CHARACTER_CODE_MODEL = SET_TRANSMIT_STATE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* SET_TRANSMIT_STATE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/** The cancel character unicode character code model. U+0094 */
static wchar_t CANCEL_CHARACTER_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0094 };
static wchar_t* CANCEL_CHARACTER_UNICODE_CHARACTER_CODE_MODEL = CANCEL_CHARACTER_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CANCEL_CHARACTER_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/** The message waiting unicode character code model. U+0095 */
static wchar_t MESSAGE_WAITING_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0095 };
static wchar_t* MESSAGE_WAITING_UNICODE_CHARACTER_CODE_MODEL = MESSAGE_WAITING_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MESSAGE_WAITING_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/** The start of guarded area unicode character code model. U+0096 */
static wchar_t START_OF_GUARDED_AREA_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0096 };
static wchar_t* START_OF_GUARDED_AREA_UNICODE_CHARACTER_CODE_MODEL = START_OF_GUARDED_AREA_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* START_OF_GUARDED_AREA_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/** The end of guarded area unicode character code model. U+0097 */
static wchar_t END_OF_GUARDED_AREA_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0097 };
static wchar_t* END_OF_GUARDED_AREA_UNICODE_CHARACTER_CODE_MODEL = END_OF_GUARDED_AREA_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* END_OF_GUARDED_AREA_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/** The start of string unicode character code model. U+0098 */
static wchar_t START_OF_STRING_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0098 };
static wchar_t* START_OF_STRING_UNICODE_CHARACTER_CODE_MODEL = START_OF_STRING_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* START_OF_STRING_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/**
 * The single graphic character introducer unicode character code model. U+0099
 *
 * Listed as XXX in Unicode. Not part of ISO/IEC 6429.
 */
static wchar_t SINGLE_GRAPHIC_CHARACTER_INTRODUCER_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0099 };
static wchar_t* SINGLE_GRAPHIC_CHARACTER_INTRODUCER_UNICODE_CHARACTER_CODE_MODEL = SINGLE_GRAPHIC_CHARACTER_INTRODUCER_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* SINGLE_GRAPHIC_CHARACTER_INTRODUCER_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/** The single character introducer unicode character code model. U+009A */
static wchar_t SINGLE_CHARACTER_INTRODUCER_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x009A };
static wchar_t* SINGLE_CHARACTER_INTRODUCER_UNICODE_CHARACTER_CODE_MODEL = SINGLE_CHARACTER_INTRODUCER_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* SINGLE_CHARACTER_INTRODUCER_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/** The control sequence introducer unicode character code model. U+009B */
static wchar_t CONTROL_SEQUENCE_INTRODUCER_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x009B };
static wchar_t* CONTROL_SEQUENCE_INTRODUCER_UNICODE_CHARACTER_CODE_MODEL = CONTROL_SEQUENCE_INTRODUCER_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CONTROL_SEQUENCE_INTRODUCER_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/** The string terminator unicode character code model. U+009C */
static wchar_t STRING_TERMINATOR_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x009C };
static wchar_t* STRING_TERMINATOR_UNICODE_CHARACTER_CODE_MODEL = STRING_TERMINATOR_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* STRING_TERMINATOR_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/** The operating system command unicode character code model. U+009D */
static wchar_t OPERATING_SYSTEM_COMMAND_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x009D };
static wchar_t* OPERATING_SYSTEM_COMMAND_UNICODE_CHARACTER_CODE_MODEL = OPERATING_SYSTEM_COMMAND_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* OPERATING_SYSTEM_COMMAND_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/** The privacy message unicode character code model. U+009E */
static wchar_t PRIVACY_MESSAGE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x009E };
static wchar_t* PRIVACY_MESSAGE_UNICODE_CHARACTER_CODE_MODEL = PRIVACY_MESSAGE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* PRIVACY_MESSAGE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/** The application program command unicode character code model. U+009F */
static wchar_t APPLICATION_PROGRAM_COMMAND_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x009F };
static wchar_t* APPLICATION_PROGRAM_COMMAND_UNICODE_CHARACTER_CODE_MODEL = APPLICATION_PROGRAM_COMMAND_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* APPLICATION_PROGRAM_COMMAND_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/**
 * The no-break space unicode character code model.
 *
 * Unicode block: Latin-1 Supplement
 * Unicode category: Separator, Space
 *
 * Unicode: U+00A0
 * UTF-8: 0xC2 0xA0
 * Octal escaped UTF-8: \302\240
 * Decimal entity reference: &#160;
 *
 * Notes:
 * commonly abbreviated as NBSP
 *
 * See also:
 * U+0020 SPACE
 * U+2007 FIGURE SPACE
 * U+202F NARROW NO-BREAK SPACE
 * U+2060 WORD JOINER
 * U+FEFF ZERO WIDTH NO-BREAK SPACE
 */
static wchar_t NO_BREAK_SPACE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x00A0 };
static wchar_t* NO_BREAK_SPACE_UNICODE_CHARACTER_CODE_MODEL = NO_BREAK_SPACE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* NO_BREAK_SPACE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t INVERTED_EXCLAMATION_MARK_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x00A1 };
static wchar_t* INVERTED_EXCLAMATION_MARK_UNICODE_CHARACTER_CODE_MODEL = INVERTED_EXCLAMATION_MARK_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* INVERTED_EXCLAMATION_MARK_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t CENT_SIGN_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x00A2 };
static wchar_t* CENT_SIGN_UNICODE_CHARACTER_CODE_MODEL = CENT_SIGN_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CENT_SIGN_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t POUND_SIGN_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x00A3 };
static wchar_t* POUND_SIGN_UNICODE_CHARACTER_CODE_MODEL = POUND_SIGN_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* POUND_SIGN_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t CURRENCY_SIGN_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x00A4 };
static wchar_t* CURRENCY_SIGN_UNICODE_CHARACTER_CODE_MODEL = CURRENCY_SIGN_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CURRENCY_SIGN_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t YEN_SIGN_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x00A5 };
static wchar_t* YEN_SIGN_UNICODE_CHARACTER_CODE_MODEL = YEN_SIGN_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* YEN_SIGN_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t BROKEN_BAR_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x00A6 };
static wchar_t* BROKEN_BAR_UNICODE_CHARACTER_CODE_MODEL = BROKEN_BAR_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* BROKEN_BAR_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t SECTION_SIGN_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x00A7 };
static wchar_t* SECTION_SIGN_UNICODE_CHARACTER_CODE_MODEL = SECTION_SIGN_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* SECTION_SIGN_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t DIAERESIS_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x00A8 };
static wchar_t* DIAERESIS_UNICODE_CHARACTER_CODE_MODEL = DIAERESIS_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* DIAERESIS_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t COPYRIGHT_SIGN_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x00A9 };
static wchar_t* COPYRIGHT_SIGN_UNICODE_CHARACTER_CODE_MODEL = COPYRIGHT_SIGN_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* COPYRIGHT_SIGN_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t FEMININE_ORDINAL_INDICATOR_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x00AA };
static wchar_t* FEMININE_ORDINAL_INDICATOR_UNICODE_CHARACTER_CODE_MODEL = FEMININE_ORDINAL_INDICATOR_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* FEMININE_ORDINAL_INDICATOR_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LEFT_POINTING_DOUBLE_ANGLE_QUOTATION_MARK_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x00AB };
static wchar_t* LEFT_POINTING_DOUBLE_ANGLE_QUOTATION_MARK_UNICODE_CHARACTER_CODE_MODEL = LEFT_POINTING_DOUBLE_ANGLE_QUOTATION_MARK_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LEFT_POINTING_DOUBLE_ANGLE_QUOTATION_MARK_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t NOT_SIGN_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x00AC };
static wchar_t* NOT_SIGN_UNICODE_CHARACTER_CODE_MODEL = NOT_SIGN_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* NOT_SIGN_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t SOFT_HYPHEN_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x00AD };
static wchar_t* SOFT_HYPHEN_UNICODE_CHARACTER_CODE_MODEL = SOFT_HYPHEN_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* SOFT_HYPHEN_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t REGISTERED_SIGN_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x00AE };
static wchar_t* REGISTERED_SIGN_UNICODE_CHARACTER_CODE_MODEL = REGISTERED_SIGN_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* REGISTERED_SIGN_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MACRON_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x00AF };
static wchar_t* MACRON_UNICODE_CHARACTER_CODE_MODEL = MACRON_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MACRON_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t DEGREE_SIGN_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x00B0 };
static wchar_t* DEGREE_SIGN_UNICODE_CHARACTER_CODE_MODEL = DEGREE_SIGN_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* DEGREE_SIGN_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t PLUS_MINUS_SIGN_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x00B1 };
static wchar_t* PLUS_MINUS_SIGN_UNICODE_CHARACTER_CODE_MODEL = PLUS_MINUS_SIGN_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* PLUS_MINUS_SIGN_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t SUPERSCRIPT_TWO_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x00B2 };
static wchar_t* SUPERSCRIPT_TWO_UNICODE_CHARACTER_CODE_MODEL = SUPERSCRIPT_TWO_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* SUPERSCRIPT_TWO_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t SUPERSCRIPT_THREE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x00B3 };
static wchar_t* SUPERSCRIPT_THREE_UNICODE_CHARACTER_CODE_MODEL = SUPERSCRIPT_THREE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* SUPERSCRIPT_THREE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t ACUTE_ACCENT_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x00B4 };
static wchar_t* ACUTE_ACCENT_UNICODE_CHARACTER_CODE_MODEL = ACUTE_ACCENT_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* ACUTE_ACCENT_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MICRO_SIGN_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x00B5 };
static wchar_t* MICRO_SIGN_UNICODE_CHARACTER_CODE_MODEL = MICRO_SIGN_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MICRO_SIGN_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t PILCROW_SIGN_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x00B6 };
static wchar_t* PILCROW_SIGN_UNICODE_CHARACTER_CODE_MODEL = PILCROW_SIGN_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* PILCROW_SIGN_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MIDDLE_DOT_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x00B7 };
static wchar_t* MIDDLE_DOT_UNICODE_CHARACTER_CODE_MODEL = MIDDLE_DOT_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MIDDLE_DOT_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t CEDILLA_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x00B8 };
static wchar_t* CEDILLA_UNICODE_CHARACTER_CODE_MODEL = CEDILLA_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CEDILLA_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t SUPERSCRIPT_ONE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x00B9 };
static wchar_t* SUPERSCRIPT_ONE_UNICODE_CHARACTER_CODE_MODEL = SUPERSCRIPT_ONE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* SUPERSCRIPT_ONE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MASCULINE_ORDINAL_INDICATOR_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x00BA };
static wchar_t* MASCULINE_ORDINAL_INDICATOR_UNICODE_CHARACTER_CODE_MODEL = MASCULINE_ORDINAL_INDICATOR_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MASCULINE_ORDINAL_INDICATOR_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t RIGHT_POINTING_DOUBLE_ANGLE_QUOTATION_MARK_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x00BB };
static wchar_t* RIGHT_POINTING_DOUBLE_ANGLE_QUOTATION_MARK_UNICODE_CHARACTER_CODE_MODEL = RIGHT_POINTING_DOUBLE_ANGLE_QUOTATION_MARK_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* RIGHT_POINTING_DOUBLE_ANGLE_QUOTATION_MARK_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t VULGAR_FRACTION_ONE_QUARTER_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x00BC };
static wchar_t* VULGAR_FRACTION_ONE_QUARTER_UNICODE_CHARACTER_CODE_MODEL = VULGAR_FRACTION_ONE_QUARTER_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* VULGAR_FRACTION_ONE_QUARTER_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t VULGAR_FRACTION_ONE_HALF_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x00BD };
static wchar_t* VULGAR_FRACTION_ONE_HALF_UNICODE_CHARACTER_CODE_MODEL = VULGAR_FRACTION_ONE_HALF_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* VULGAR_FRACTION_ONE_HALF_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t VULGAR_FRACTION_THREE_QUARTERS_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x00BE };
static wchar_t* VULGAR_FRACTION_THREE_QUARTERS_UNICODE_CHARACTER_CODE_MODEL = VULGAR_FRACTION_THREE_QUARTERS_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* VULGAR_FRACTION_THREE_QUARTERS_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t INVERTED_QUESTION_MARK_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x00BF };
static wchar_t* INVERTED_QUESTION_MARK_UNICODE_CHARACTER_CODE_MODEL = INVERTED_QUESTION_MARK_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* INVERTED_QUESTION_MARK_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_CAPITAL_LETTER_A_WITH_GRAVE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x00C0 };
static wchar_t* LATIN_CAPITAL_LETTER_A_WITH_GRAVE_UNICODE_CHARACTER_CODE_MODEL = LATIN_CAPITAL_LETTER_A_WITH_GRAVE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_CAPITAL_LETTER_A_WITH_GRAVE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_CAPITAL_LETTER_A_WITH_ACUTE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x00C1 };
static wchar_t* LATIN_CAPITAL_LETTER_A_WITH_ACUTE_UNICODE_CHARACTER_CODE_MODEL = LATIN_CAPITAL_LETTER_A_WITH_ACUTE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_CAPITAL_LETTER_A_WITH_ACUTE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_CAPITAL_LETTER_A_WITH_CIRCUMFLEX_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x00C2 };
static wchar_t* LATIN_CAPITAL_LETTER_A_WITH_CIRCUMFLEX_UNICODE_CHARACTER_CODE_MODEL = LATIN_CAPITAL_LETTER_A_WITH_CIRCUMFLEX_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_CAPITAL_LETTER_A_WITH_CIRCUMFLEX_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_CAPITAL_LETTER_A_WITH_TILDE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x00C3 };
static wchar_t* LATIN_CAPITAL_LETTER_A_WITH_TILDE_UNICODE_CHARACTER_CODE_MODEL = LATIN_CAPITAL_LETTER_A_WITH_TILDE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_CAPITAL_LETTER_A_WITH_TILDE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_CAPITAL_LETTER_A_WITH_DIAERESIS_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x00C4 };
static wchar_t* LATIN_CAPITAL_LETTER_A_WITH_DIAERESIS_UNICODE_CHARACTER_CODE_MODEL = LATIN_CAPITAL_LETTER_A_WITH_DIAERESIS_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_CAPITAL_LETTER_A_WITH_DIAERESIS_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_CAPITAL_LETTER_A_WITH_RING_ABOVE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x00C5 };
static wchar_t* LATIN_CAPITAL_LETTER_A_WITH_RING_ABOVE_UNICODE_CHARACTER_CODE_MODEL = LATIN_CAPITAL_LETTER_A_WITH_RING_ABOVE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_CAPITAL_LETTER_A_WITH_RING_ABOVE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_CAPITAL_LETTER_AE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x00C6 };
static wchar_t* LATIN_CAPITAL_LETTER_AE_UNICODE_CHARACTER_CODE_MODEL = LATIN_CAPITAL_LETTER_AE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_CAPITAL_LETTER_AE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_CAPITAL_LETTER_C_WITH_CEDILLA_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x00C7 };
static wchar_t* LATIN_CAPITAL_LETTER_C_WITH_CEDILLA_UNICODE_CHARACTER_CODE_MODEL = LATIN_CAPITAL_LETTER_C_WITH_CEDILLA_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_CAPITAL_LETTER_C_WITH_CEDILLA_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_CAPITAL_LETTER_E_WITH_GRAVE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x00C8 };
static wchar_t* LATIN_CAPITAL_LETTER_E_WITH_GRAVE_UNICODE_CHARACTER_CODE_MODEL = LATIN_CAPITAL_LETTER_E_WITH_GRAVE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_CAPITAL_LETTER_E_WITH_GRAVE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_CAPITAL_LETTER_E_WITH_ACUTE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x00C9 };
static wchar_t* LATIN_CAPITAL_LETTER_E_WITH_ACUTE_UNICODE_CHARACTER_CODE_MODEL = LATIN_CAPITAL_LETTER_E_WITH_ACUTE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_CAPITAL_LETTER_E_WITH_ACUTE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_CAPITAL_LETTER_E_WITH_CIRCUMFLEX_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x00CA };
static wchar_t* LATIN_CAPITAL_LETTER_E_WITH_CIRCUMFLEX_UNICODE_CHARACTER_CODE_MODEL = LATIN_CAPITAL_LETTER_E_WITH_CIRCUMFLEX_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_CAPITAL_LETTER_E_WITH_CIRCUMFLEX_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_CAPITAL_LETTER_E_WITH_DIAERESIS_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x00CB };
static wchar_t* LATIN_CAPITAL_LETTER_E_WITH_DIAERESIS_UNICODE_CHARACTER_CODE_MODEL = LATIN_CAPITAL_LETTER_E_WITH_DIAERESIS_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_CAPITAL_LETTER_E_WITH_DIAERESIS_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_CAPITAL_LETTER_I_WITH_GRAVE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x00CC };
static wchar_t* LATIN_CAPITAL_LETTER_I_WITH_GRAVE_UNICODE_CHARACTER_CODE_MODEL = LATIN_CAPITAL_LETTER_I_WITH_GRAVE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_CAPITAL_LETTER_I_WITH_GRAVE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_CAPITAL_LETTER_I_WITH_ACUTE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x00CD };
static wchar_t* LATIN_CAPITAL_LETTER_I_WITH_ACUTE_UNICODE_CHARACTER_CODE_MODEL = LATIN_CAPITAL_LETTER_I_WITH_ACUTE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_CAPITAL_LETTER_I_WITH_ACUTE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_CAPITAL_LETTER_I_WITH_CIRCUMFLEX_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x00CE };
static wchar_t* LATIN_CAPITAL_LETTER_I_WITH_CIRCUMFLEX_UNICODE_CHARACTER_CODE_MODEL = LATIN_CAPITAL_LETTER_I_WITH_CIRCUMFLEX_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_CAPITAL_LETTER_I_WITH_CIRCUMFLEX_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_CAPITAL_LETTER_I_WITH_DIAERESIS_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x00CF };
static wchar_t* LATIN_CAPITAL_LETTER_I_WITH_DIAERESIS_UNICODE_CHARACTER_CODE_MODEL = LATIN_CAPITAL_LETTER_I_WITH_DIAERESIS_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_CAPITAL_LETTER_I_WITH_DIAERESIS_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_CAPITAL_LETTER_ETH_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x00D0 };
static wchar_t* LATIN_CAPITAL_LETTER_ETH_UNICODE_CHARACTER_CODE_MODEL = LATIN_CAPITAL_LETTER_ETH_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_CAPITAL_LETTER_ETH_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_CAPITAL_LETTER_N_WITH_TILDE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x00D1 };
static wchar_t* LATIN_CAPITAL_LETTER_N_WITH_TILDE_UNICODE_CHARACTER_CODE_MODEL = LATIN_CAPITAL_LETTER_N_WITH_TILDE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_CAPITAL_LETTER_N_WITH_TILDE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_CAPITAL_LETTER_O_WITH_GRAVE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x00D2 };
static wchar_t* LATIN_CAPITAL_LETTER_O_WITH_GRAVE_UNICODE_CHARACTER_CODE_MODEL = LATIN_CAPITAL_LETTER_O_WITH_GRAVE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_CAPITAL_LETTER_O_WITH_GRAVE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_CAPITAL_LETTER_O_WITH_ACUTE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x00D3 };
static wchar_t* LATIN_CAPITAL_LETTER_O_WITH_ACUTE_UNICODE_CHARACTER_CODE_MODEL = LATIN_CAPITAL_LETTER_O_WITH_ACUTE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_CAPITAL_LETTER_O_WITH_ACUTE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_CAPITAL_LETTER_O_WITH_CIRCUMFLEX_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x00D4 };
static wchar_t* LATIN_CAPITAL_LETTER_O_WITH_CIRCUMFLEX_UNICODE_CHARACTER_CODE_MODEL = LATIN_CAPITAL_LETTER_O_WITH_CIRCUMFLEX_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_CAPITAL_LETTER_O_WITH_CIRCUMFLEX_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_CAPITAL_LETTER_O_WITH_TILDE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x00D5 };
static wchar_t* LATIN_CAPITAL_LETTER_O_WITH_TILDE_UNICODE_CHARACTER_CODE_MODEL = LATIN_CAPITAL_LETTER_O_WITH_TILDE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_CAPITAL_LETTER_O_WITH_TILDE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_CAPITAL_LETTER_O_WITH_DIAERESIS_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x00D6 };
static wchar_t* LATIN_CAPITAL_LETTER_O_WITH_DIAERESIS_UNICODE_CHARACTER_CODE_MODEL = LATIN_CAPITAL_LETTER_O_WITH_DIAERESIS_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_CAPITAL_LETTER_O_WITH_DIAERESIS_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MULTIPLICATION_SIGN_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x00D7 };
static wchar_t* MULTIPLICATION_SIGN_UNICODE_CHARACTER_CODE_MODEL = MULTIPLICATION_SIGN_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MULTIPLICATION_SIGN_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_CAPITAL_LETTER_O_WITH_STROKE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x00D8 };
static wchar_t* LATIN_CAPITAL_LETTER_O_WITH_STROKE_UNICODE_CHARACTER_CODE_MODEL = LATIN_CAPITAL_LETTER_O_WITH_STROKE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_CAPITAL_LETTER_O_WITH_STROKE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_CAPITAL_LETTER_U_WITH_GRAVE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x00D9 };
static wchar_t* LATIN_CAPITAL_LETTER_U_WITH_GRAVE_UNICODE_CHARACTER_CODE_MODEL = LATIN_CAPITAL_LETTER_U_WITH_GRAVE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_CAPITAL_LETTER_U_WITH_GRAVE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_CAPITAL_LETTER_U_WITH_ACUTE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x00DA };
static wchar_t* LATIN_CAPITAL_LETTER_U_WITH_ACUTE_UNICODE_CHARACTER_CODE_MODEL = LATIN_CAPITAL_LETTER_U_WITH_ACUTE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_CAPITAL_LETTER_U_WITH_ACUTE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_CAPITAL_LETTER_U_WITH_CIRCUMFLEX_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x00DB };
static wchar_t* LATIN_CAPITAL_LETTER_U_WITH_CIRCUMFLEX_UNICODE_CHARACTER_CODE_MODEL = LATIN_CAPITAL_LETTER_U_WITH_CIRCUMFLEX_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_CAPITAL_LETTER_U_WITH_CIRCUMFLEX_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_CAPITAL_LETTER_U_WITH_DIAERESIS_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x00DC };
static wchar_t* LATIN_CAPITAL_LETTER_U_WITH_DIAERESIS_UNICODE_CHARACTER_CODE_MODEL = LATIN_CAPITAL_LETTER_U_WITH_DIAERESIS_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_CAPITAL_LETTER_U_WITH_DIAERESIS_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_CAPITAL_LETTER_Y_WITH_ACUTE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x00DD };
static wchar_t* LATIN_CAPITAL_LETTER_Y_WITH_ACUTE_UNICODE_CHARACTER_CODE_MODEL = LATIN_CAPITAL_LETTER_Y_WITH_ACUTE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_CAPITAL_LETTER_Y_WITH_ACUTE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_CAPITAL_LETTER_THORN_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x00DE };
static wchar_t* LATIN_CAPITAL_LETTER_THORN_UNICODE_CHARACTER_CODE_MODEL = LATIN_CAPITAL_LETTER_THORN_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_CAPITAL_LETTER_THORN_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_SMALL_LETTER_SHARP_S_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x00DF };
static wchar_t* LATIN_SMALL_LETTER_SHARP_S_UNICODE_CHARACTER_CODE_MODEL = LATIN_SMALL_LETTER_SHARP_S_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_SMALL_LETTER_SHARP_S_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_SMALL_LETTER_A_WITH_GRAVE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x00E0 };
static wchar_t* LATIN_SMALL_LETTER_A_WITH_GRAVE_UNICODE_CHARACTER_CODE_MODEL = LATIN_SMALL_LETTER_A_WITH_GRAVE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_SMALL_LETTER_A_WITH_GRAVE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_SMALL_LETTER_A_WITH_ACUTE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x00E1 };
static wchar_t* LATIN_SMALL_LETTER_A_WITH_ACUTE_UNICODE_CHARACTER_CODE_MODEL = LATIN_SMALL_LETTER_A_WITH_ACUTE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_SMALL_LETTER_A_WITH_ACUTE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_SMALL_LETTER_A_WITH_CIRCUMFLEX_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x00E2 };
static wchar_t* LATIN_SMALL_LETTER_A_WITH_CIRCUMFLEX_UNICODE_CHARACTER_CODE_MODEL = LATIN_SMALL_LETTER_A_WITH_CIRCUMFLEX_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_SMALL_LETTER_A_WITH_CIRCUMFLEX_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_SMALL_LETTER_A_WITH_TILDE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x00E3 };
static wchar_t* LATIN_SMALL_LETTER_A_WITH_TILDE_UNICODE_CHARACTER_CODE_MODEL = LATIN_SMALL_LETTER_A_WITH_TILDE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_SMALL_LETTER_A_WITH_TILDE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_SMALL_LETTER_A_WITH_DIAERESIS_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x00E4 };
static wchar_t* LATIN_SMALL_LETTER_A_WITH_DIAERESIS_UNICODE_CHARACTER_CODE_MODEL = LATIN_SMALL_LETTER_A_WITH_DIAERESIS_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_SMALL_LETTER_A_WITH_DIAERESIS_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_SMALL_LETTER_A_WITH_RING_ABOVE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x00E5 };
static wchar_t* LATIN_SMALL_LETTER_A_WITH_RING_ABOVE_UNICODE_CHARACTER_CODE_MODEL = LATIN_SMALL_LETTER_A_WITH_RING_ABOVE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_SMALL_LETTER_A_WITH_RING_ABOVE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_SMALL_LETTER_AE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x00E6 };
static wchar_t* LATIN_SMALL_LETTER_AE_UNICODE_CHARACTER_CODE_MODEL = LATIN_SMALL_LETTER_AE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_SMALL_LETTER_AE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_SMALL_LETTER_C_WITH_CEDILLA_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x00E7 };
static wchar_t* LATIN_SMALL_LETTER_C_WITH_CEDILLA_UNICODE_CHARACTER_CODE_MODEL = LATIN_SMALL_LETTER_C_WITH_CEDILLA_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_SMALL_LETTER_C_WITH_CEDILLA_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_SMALL_LETTER_E_WITH_GRAVE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x00E8 };
static wchar_t* LATIN_SMALL_LETTER_E_WITH_GRAVE_UNICODE_CHARACTER_CODE_MODEL = LATIN_SMALL_LETTER_E_WITH_GRAVE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_SMALL_LETTER_E_WITH_GRAVE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_SMALL_LETTER_E_WITH_ACUTE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x00E9 };
static wchar_t* LATIN_SMALL_LETTER_E_WITH_ACUTE_UNICODE_CHARACTER_CODE_MODEL = LATIN_SMALL_LETTER_E_WITH_ACUTE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_SMALL_LETTER_E_WITH_ACUTE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_SMALL_LETTER_E_WITH_CIRCUMFLEX_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x00EA };
static wchar_t* LATIN_SMALL_LETTER_E_WITH_CIRCUMFLEX_UNICODE_CHARACTER_CODE_MODEL = LATIN_SMALL_LETTER_E_WITH_CIRCUMFLEX_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_SMALL_LETTER_E_WITH_CIRCUMFLEX_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_SMALL_LETTER_E_WITH_DIAERESIS_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x00EB };
static wchar_t* LATIN_SMALL_LETTER_E_WITH_DIAERESIS_UNICODE_CHARACTER_CODE_MODEL = LATIN_SMALL_LETTER_E_WITH_DIAERESIS_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_SMALL_LETTER_E_WITH_DIAERESIS_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_SMALL_LETTER_I_WITH_GRAVE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x00EC };
static wchar_t* LATIN_SMALL_LETTER_I_WITH_GRAVE_UNICODE_CHARACTER_CODE_MODEL = LATIN_SMALL_LETTER_I_WITH_GRAVE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_SMALL_LETTER_I_WITH_GRAVE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_SMALL_LETTER_I_WITH_ACUTE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x00ED };
static wchar_t* LATIN_SMALL_LETTER_I_WITH_ACUTE_UNICODE_CHARACTER_CODE_MODEL = LATIN_SMALL_LETTER_I_WITH_ACUTE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_SMALL_LETTER_I_WITH_ACUTE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_SMALL_LETTER_I_WITH_CIRCUMFLEX_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x00EE };
static wchar_t* LATIN_SMALL_LETTER_I_WITH_CIRCUMFLEX_UNICODE_CHARACTER_CODE_MODEL = LATIN_SMALL_LETTER_I_WITH_CIRCUMFLEX_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_SMALL_LETTER_I_WITH_CIRCUMFLEX_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_SMALL_LETTER_I_WITH_DIAERESIS_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x00EF };
static wchar_t* LATIN_SMALL_LETTER_I_WITH_DIAERESIS_UNICODE_CHARACTER_CODE_MODEL = LATIN_SMALL_LETTER_I_WITH_DIAERESIS_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_SMALL_LETTER_I_WITH_DIAERESIS_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_SMALL_LETTER_ETH_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x00F0 };
static wchar_t* LATIN_SMALL_LETTER_ETH_UNICODE_CHARACTER_CODE_MODEL = LATIN_SMALL_LETTER_ETH_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_SMALL_LETTER_ETH_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_SMALL_LETTER_N_WITH_TILDE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x00F1 };
static wchar_t* LATIN_SMALL_LETTER_N_WITH_TILDE_UNICODE_CHARACTER_CODE_MODEL = LATIN_SMALL_LETTER_N_WITH_TILDE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_SMALL_LETTER_N_WITH_TILDE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_SMALL_LETTER_O_WITH_GRAVE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x00F2 };
static wchar_t* LATIN_SMALL_LETTER_O_WITH_GRAVE_UNICODE_CHARACTER_CODE_MODEL = LATIN_SMALL_LETTER_O_WITH_GRAVE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_SMALL_LETTER_O_WITH_GRAVE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_SMALL_LETTER_O_WITH_ACUTE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x00F3 };
static wchar_t* LATIN_SMALL_LETTER_O_WITH_ACUTE_UNICODE_CHARACTER_CODE_MODEL = LATIN_SMALL_LETTER_O_WITH_ACUTE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_SMALL_LETTER_O_WITH_ACUTE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_SMALL_LETTER_O_WITH_CIRCUMFLEX_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x00F4 };
static wchar_t* LATIN_SMALL_LETTER_O_WITH_CIRCUMFLEX_UNICODE_CHARACTER_CODE_MODEL = LATIN_SMALL_LETTER_O_WITH_CIRCUMFLEX_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_SMALL_LETTER_O_WITH_CIRCUMFLEX_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_SMALL_LETTER_O_WITH_TILDE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x00F5 };
static wchar_t* LATIN_SMALL_LETTER_O_WITH_TILDE_UNICODE_CHARACTER_CODE_MODEL = LATIN_SMALL_LETTER_O_WITH_TILDE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_SMALL_LETTER_O_WITH_TILDE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_SMALL_LETTER_O_WITH_DIAERESIS_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x00F6 };
static wchar_t* LATIN_SMALL_LETTER_O_WITH_DIAERESIS_UNICODE_CHARACTER_CODE_MODEL = LATIN_SMALL_LETTER_O_WITH_DIAERESIS_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_SMALL_LETTER_O_WITH_DIAERESIS_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t DIVISION_SIGN_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x00F7 };
static wchar_t* DIVISION_SIGN_UNICODE_CHARACTER_CODE_MODEL = DIVISION_SIGN_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* DIVISION_SIGN_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_SMALL_LETTER_O_WITH_STROKE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x00F8 };
static wchar_t* LATIN_SMALL_LETTER_O_WITH_STROKE_UNICODE_CHARACTER_CODE_MODEL = LATIN_SMALL_LETTER_O_WITH_STROKE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_SMALL_LETTER_O_WITH_STROKE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_SMALL_LETTER_U_WITH_GRAVE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x00F9 };
static wchar_t* LATIN_SMALL_LETTER_U_WITH_GRAVE_UNICODE_CHARACTER_CODE_MODEL = LATIN_SMALL_LETTER_U_WITH_GRAVE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_SMALL_LETTER_U_WITH_GRAVE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_SMALL_LETTER_U_WITH_ACUTE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x00FA };
static wchar_t* LATIN_SMALL_LETTER_U_WITH_ACUTE_UNICODE_CHARACTER_CODE_MODEL = LATIN_SMALL_LETTER_U_WITH_ACUTE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_SMALL_LETTER_U_WITH_ACUTE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_SMALL_LETTER_U_WITH_CIRCUMFLEX_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x00FB };
static wchar_t* LATIN_SMALL_LETTER_U_WITH_CIRCUMFLEX_UNICODE_CHARACTER_CODE_MODEL = LATIN_SMALL_LETTER_U_WITH_CIRCUMFLEX_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_SMALL_LETTER_U_WITH_CIRCUMFLEX_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_SMALL_LETTER_U_WITH_DIAERESIS_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x00FC };
static wchar_t* LATIN_SMALL_LETTER_U_WITH_DIAERESIS_UNICODE_CHARACTER_CODE_MODEL = LATIN_SMALL_LETTER_U_WITH_DIAERESIS_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_SMALL_LETTER_U_WITH_DIAERESIS_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_SMALL_LETTER_Y_WITH_ACUTE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x00FD };
static wchar_t* LATIN_SMALL_LETTER_Y_WITH_ACUTE_UNICODE_CHARACTER_CODE_MODEL = LATIN_SMALL_LETTER_Y_WITH_ACUTE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_SMALL_LETTER_Y_WITH_ACUTE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_SMALL_LETTER_THORN_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x00FE };
static wchar_t* LATIN_SMALL_LETTER_THORN_UNICODE_CHARACTER_CODE_MODEL = LATIN_SMALL_LETTER_THORN_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_SMALL_LETTER_THORN_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_SMALL_LETTER_Y_WITH_DIAERESIS_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x00FF };
static wchar_t* LATIN_SMALL_LETTER_Y_WITH_DIAERESIS_UNICODE_CHARACTER_CODE_MODEL = LATIN_SMALL_LETTER_Y_WITH_DIAERESIS_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_SMALL_LETTER_Y_WITH_DIAERESIS_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LESS_THAN_SIGN_WITH_VERTICAL_LINE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x003c, 0x20d2 };
static wchar_t* LESS_THAN_SIGN_WITH_VERTICAL_LINE_UNICODE_CHARACTER_CODE_MODEL = LESS_THAN_SIGN_WITH_VERTICAL_LINE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LESS_THAN_SIGN_WITH_VERTICAL_LINE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_2_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t EQUALS_SIGN_WITH_REVERSE_SLASH_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x003d, 0x20e5 };
static wchar_t* EQUALS_SIGN_WITH_REVERSE_SLASH_UNICODE_CHARACTER_CODE_MODEL = EQUALS_SIGN_WITH_REVERSE_SLASH_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* EQUALS_SIGN_WITH_REVERSE_SLASH_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_2_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t GREATER_THAN_SIGN_WITH_VERTICAL_LINE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x003e, 0x20d2 };
static wchar_t* GREATER_THAN_SIGN_WITH_VERTICAL_LINE_UNICODE_CHARACTER_CODE_MODEL = GREATER_THAN_SIGN_WITH_VERTICAL_LINE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* GREATER_THAN_SIGN_WITH_VERTICAL_LINE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_2_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t FJ_LIGATURE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0066, 0x006a };
static wchar_t* FJ_LIGATURE_UNICODE_CHARACTER_CODE_MODEL = FJ_LIGATURE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* FJ_LIGATURE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_2_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_CAPITAL_LETTER_A_WITH_MACRON_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0100 };
static wchar_t* LATIN_CAPITAL_LETTER_A_WITH_MACRON_UNICODE_CHARACTER_CODE_MODEL = LATIN_CAPITAL_LETTER_A_WITH_MACRON_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_CAPITAL_LETTER_A_WITH_MACRON_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_SMALL_LETTER_A_WITH_MACRON_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0101 };
static wchar_t* LATIN_SMALL_LETTER_A_WITH_MACRON_UNICODE_CHARACTER_CODE_MODEL = LATIN_SMALL_LETTER_A_WITH_MACRON_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_SMALL_LETTER_A_WITH_MACRON_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_CAPITAL_LETTER_A_WITH_BREVE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0102 };
static wchar_t* LATIN_CAPITAL_LETTER_A_WITH_BREVE_UNICODE_CHARACTER_CODE_MODEL = LATIN_CAPITAL_LETTER_A_WITH_BREVE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_CAPITAL_LETTER_A_WITH_BREVE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_SMALL_LETTER_A_WITH_BREVE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0103 };
static wchar_t* LATIN_SMALL_LETTER_A_WITH_BREVE_UNICODE_CHARACTER_CODE_MODEL = LATIN_SMALL_LETTER_A_WITH_BREVE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_SMALL_LETTER_A_WITH_BREVE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_CAPITAL_LETTER_A_WITH_OGONEK_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0104 };
static wchar_t* LATIN_CAPITAL_LETTER_A_WITH_OGONEK_UNICODE_CHARACTER_CODE_MODEL = LATIN_CAPITAL_LETTER_A_WITH_OGONEK_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_CAPITAL_LETTER_A_WITH_OGONEK_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_SMALL_LETTER_A_WITH_OGONEK_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0105 };
static wchar_t* LATIN_SMALL_LETTER_A_WITH_OGONEK_UNICODE_CHARACTER_CODE_MODEL = LATIN_SMALL_LETTER_A_WITH_OGONEK_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_SMALL_LETTER_A_WITH_OGONEK_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_CAPITAL_LETTER_C_WITH_ACUTE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0106 };
static wchar_t* LATIN_CAPITAL_LETTER_C_WITH_ACUTE_UNICODE_CHARACTER_CODE_MODEL = LATIN_CAPITAL_LETTER_C_WITH_ACUTE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_CAPITAL_LETTER_C_WITH_ACUTE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_SMALL_LETTER_C_WITH_ACUTE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0107 };
static wchar_t* LATIN_SMALL_LETTER_C_WITH_ACUTE_UNICODE_CHARACTER_CODE_MODEL = LATIN_SMALL_LETTER_C_WITH_ACUTE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_SMALL_LETTER_C_WITH_ACUTE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_CAPITAL_LETTER_C_WITH_CIRCUMFLEX_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0108 };
static wchar_t* LATIN_CAPITAL_LETTER_C_WITH_CIRCUMFLEX_UNICODE_CHARACTER_CODE_MODEL = LATIN_CAPITAL_LETTER_C_WITH_CIRCUMFLEX_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_CAPITAL_LETTER_C_WITH_CIRCUMFLEX_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_SMALL_LETTER_C_WITH_CIRCUMFLEX_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0109 };
static wchar_t* LATIN_SMALL_LETTER_C_WITH_CIRCUMFLEX_UNICODE_CHARACTER_CODE_MODEL = LATIN_SMALL_LETTER_C_WITH_CIRCUMFLEX_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_SMALL_LETTER_C_WITH_CIRCUMFLEX_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_CAPITAL_LETTER_C_WITH_DOT_ABOVE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x010a };
static wchar_t* LATIN_CAPITAL_LETTER_C_WITH_DOT_ABOVE_UNICODE_CHARACTER_CODE_MODEL = LATIN_CAPITAL_LETTER_C_WITH_DOT_ABOVE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_CAPITAL_LETTER_C_WITH_DOT_ABOVE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_SMALL_LETTER_C_WITH_DOT_ABOVE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x010b };
static wchar_t* LATIN_SMALL_LETTER_C_WITH_DOT_ABOVE_UNICODE_CHARACTER_CODE_MODEL = LATIN_SMALL_LETTER_C_WITH_DOT_ABOVE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_SMALL_LETTER_C_WITH_DOT_ABOVE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_CAPITAL_LETTER_C_WITH_CARON_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x010c };
static wchar_t* LATIN_CAPITAL_LETTER_C_WITH_CARON_UNICODE_CHARACTER_CODE_MODEL = LATIN_CAPITAL_LETTER_C_WITH_CARON_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_CAPITAL_LETTER_C_WITH_CARON_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_SMALL_LETTER_C_WITH_CARON_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x010d };
static wchar_t* LATIN_SMALL_LETTER_C_WITH_CARON_UNICODE_CHARACTER_CODE_MODEL = LATIN_SMALL_LETTER_C_WITH_CARON_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_SMALL_LETTER_C_WITH_CARON_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_CAPITAL_LETTER_D_WITH_CARON_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x010e };
static wchar_t* LATIN_CAPITAL_LETTER_D_WITH_CARON_UNICODE_CHARACTER_CODE_MODEL = LATIN_CAPITAL_LETTER_D_WITH_CARON_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_CAPITAL_LETTER_D_WITH_CARON_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_SMALL_LETTER_D_WITH_CARON_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x010f };
static wchar_t* LATIN_SMALL_LETTER_D_WITH_CARON_UNICODE_CHARACTER_CODE_MODEL = LATIN_SMALL_LETTER_D_WITH_CARON_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_SMALL_LETTER_D_WITH_CARON_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_CAPITAL_LETTER_D_WITH_STROKE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0110 };
static wchar_t* LATIN_CAPITAL_LETTER_D_WITH_STROKE_UNICODE_CHARACTER_CODE_MODEL = LATIN_CAPITAL_LETTER_D_WITH_STROKE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_CAPITAL_LETTER_D_WITH_STROKE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_SMALL_LETTER_D_WITH_STROKE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0111 };
static wchar_t* LATIN_SMALL_LETTER_D_WITH_STROKE_UNICODE_CHARACTER_CODE_MODEL = LATIN_SMALL_LETTER_D_WITH_STROKE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_SMALL_LETTER_D_WITH_STROKE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_CAPITAL_LETTER_E_WITH_MACRON_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0112 };
static wchar_t* LATIN_CAPITAL_LETTER_E_WITH_MACRON_UNICODE_CHARACTER_CODE_MODEL = LATIN_CAPITAL_LETTER_E_WITH_MACRON_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_CAPITAL_LETTER_E_WITH_MACRON_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_SMALL_LETTER_E_WITH_MACRON_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0113 };
static wchar_t* LATIN_SMALL_LETTER_E_WITH_MACRON_UNICODE_CHARACTER_CODE_MODEL = LATIN_SMALL_LETTER_E_WITH_MACRON_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_SMALL_LETTER_E_WITH_MACRON_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_CAPITAL_LETTER_E_WITH_DOT_ABOVE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0116 };
static wchar_t* LATIN_CAPITAL_LETTER_E_WITH_DOT_ABOVE_UNICODE_CHARACTER_CODE_MODEL = LATIN_CAPITAL_LETTER_E_WITH_DOT_ABOVE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_CAPITAL_LETTER_E_WITH_DOT_ABOVE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_SMALL_LETTER_E_WITH_DOT_ABOVE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0117 };
static wchar_t* LATIN_SMALL_LETTER_E_WITH_DOT_ABOVE_UNICODE_CHARACTER_CODE_MODEL = LATIN_SMALL_LETTER_E_WITH_DOT_ABOVE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_SMALL_LETTER_E_WITH_DOT_ABOVE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_CAPITAL_LETTER_E_WITH_OGONEK_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0118 };
static wchar_t* LATIN_CAPITAL_LETTER_E_WITH_OGONEK_UNICODE_CHARACTER_CODE_MODEL = LATIN_CAPITAL_LETTER_E_WITH_OGONEK_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_CAPITAL_LETTER_E_WITH_OGONEK_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_SMALL_LETTER_E_WITH_OGONEK_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0119 };
static wchar_t* LATIN_SMALL_LETTER_E_WITH_OGONEK_UNICODE_CHARACTER_CODE_MODEL = LATIN_SMALL_LETTER_E_WITH_OGONEK_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_SMALL_LETTER_E_WITH_OGONEK_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_CAPITAL_LETTER_E_WITH_CARON_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x011a };
static wchar_t* LATIN_CAPITAL_LETTER_E_WITH_CARON_UNICODE_CHARACTER_CODE_MODEL = LATIN_CAPITAL_LETTER_E_WITH_CARON_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_CAPITAL_LETTER_E_WITH_CARON_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_SMALL_LETTER_E_WITH_CARON_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x011b };
static wchar_t* LATIN_SMALL_LETTER_E_WITH_CARON_UNICODE_CHARACTER_CODE_MODEL = LATIN_SMALL_LETTER_E_WITH_CARON_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_SMALL_LETTER_E_WITH_CARON_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_CAPITAL_LETTER_G_WITH_CIRCUMFLEX_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x011c };
static wchar_t* LATIN_CAPITAL_LETTER_G_WITH_CIRCUMFLEX_UNICODE_CHARACTER_CODE_MODEL = LATIN_CAPITAL_LETTER_G_WITH_CIRCUMFLEX_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_CAPITAL_LETTER_G_WITH_CIRCUMFLEX_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_SMALL_LETTER_G_WITH_CIRCUMFLEX_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x011d };
static wchar_t* LATIN_SMALL_LETTER_G_WITH_CIRCUMFLEX_UNICODE_CHARACTER_CODE_MODEL = LATIN_SMALL_LETTER_G_WITH_CIRCUMFLEX_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_SMALL_LETTER_G_WITH_CIRCUMFLEX_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_CAPITAL_LETTER_G_WITH_BREVE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x011e };
static wchar_t* LATIN_CAPITAL_LETTER_G_WITH_BREVE_UNICODE_CHARACTER_CODE_MODEL = LATIN_CAPITAL_LETTER_G_WITH_BREVE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_CAPITAL_LETTER_G_WITH_BREVE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_SMALL_LETTER_G_WITH_BREVE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x011f };
static wchar_t* LATIN_SMALL_LETTER_G_WITH_BREVE_UNICODE_CHARACTER_CODE_MODEL = LATIN_SMALL_LETTER_G_WITH_BREVE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_SMALL_LETTER_G_WITH_BREVE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_CAPITAL_LETTER_G_WITH_DOT_ABOVE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0120 };
static wchar_t* LATIN_CAPITAL_LETTER_G_WITH_DOT_ABOVE_UNICODE_CHARACTER_CODE_MODEL = LATIN_CAPITAL_LETTER_G_WITH_DOT_ABOVE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_CAPITAL_LETTER_G_WITH_DOT_ABOVE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_SMALL_LETTER_G_WITH_DOT_ABOVE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0121 };
static wchar_t* LATIN_SMALL_LETTER_G_WITH_DOT_ABOVE_UNICODE_CHARACTER_CODE_MODEL = LATIN_SMALL_LETTER_G_WITH_DOT_ABOVE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_SMALL_LETTER_G_WITH_DOT_ABOVE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_CAPITAL_LETTER_G_WITH_CEDILLA_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0122 };
static wchar_t* LATIN_CAPITAL_LETTER_G_WITH_CEDILLA_UNICODE_CHARACTER_CODE_MODEL = LATIN_CAPITAL_LETTER_G_WITH_CEDILLA_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_CAPITAL_LETTER_G_WITH_CEDILLA_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_CAPITAL_LETTER_H_WITH_CIRCUMFLEX_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0124 };
static wchar_t* LATIN_CAPITAL_LETTER_H_WITH_CIRCUMFLEX_UNICODE_CHARACTER_CODE_MODEL = LATIN_CAPITAL_LETTER_H_WITH_CIRCUMFLEX_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_CAPITAL_LETTER_H_WITH_CIRCUMFLEX_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_SMALL_LETTER_H_WITH_CIRCUMFLEX_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0125 };
static wchar_t* LATIN_SMALL_LETTER_H_WITH_CIRCUMFLEX_UNICODE_CHARACTER_CODE_MODEL = LATIN_SMALL_LETTER_H_WITH_CIRCUMFLEX_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_SMALL_LETTER_H_WITH_CIRCUMFLEX_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_CAPITAL_LETTER_H_WITH_STROKE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0126 };
static wchar_t* LATIN_CAPITAL_LETTER_H_WITH_STROKE_UNICODE_CHARACTER_CODE_MODEL = LATIN_CAPITAL_LETTER_H_WITH_STROKE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_CAPITAL_LETTER_H_WITH_STROKE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_SMALL_LETTER_H_WITH_STROKE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0127 };
static wchar_t* LATIN_SMALL_LETTER_H_WITH_STROKE_UNICODE_CHARACTER_CODE_MODEL = LATIN_SMALL_LETTER_H_WITH_STROKE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_SMALL_LETTER_H_WITH_STROKE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_CAPITAL_LETTER_I_WITH_TILDE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0128 };
static wchar_t* LATIN_CAPITAL_LETTER_I_WITH_TILDE_UNICODE_CHARACTER_CODE_MODEL = LATIN_CAPITAL_LETTER_I_WITH_TILDE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_CAPITAL_LETTER_I_WITH_TILDE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_SMALL_LETTER_I_WITH_TILDE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0129 };
static wchar_t* LATIN_SMALL_LETTER_I_WITH_TILDE_UNICODE_CHARACTER_CODE_MODEL = LATIN_SMALL_LETTER_I_WITH_TILDE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_SMALL_LETTER_I_WITH_TILDE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_CAPITAL_LETTER_I_WITH_MACRON_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x012a };
static wchar_t* LATIN_CAPITAL_LETTER_I_WITH_MACRON_UNICODE_CHARACTER_CODE_MODEL = LATIN_CAPITAL_LETTER_I_WITH_MACRON_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_CAPITAL_LETTER_I_WITH_MACRON_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_SMALL_LETTER_I_WITH_MACRON_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x012b };
static wchar_t* LATIN_SMALL_LETTER_I_WITH_MACRON_UNICODE_CHARACTER_CODE_MODEL = LATIN_SMALL_LETTER_I_WITH_MACRON_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_SMALL_LETTER_I_WITH_MACRON_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_CAPITAL_LETTER_I_WITH_OGONEK_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x012e };
static wchar_t* LATIN_CAPITAL_LETTER_I_WITH_OGONEK_UNICODE_CHARACTER_CODE_MODEL = LATIN_CAPITAL_LETTER_I_WITH_OGONEK_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_CAPITAL_LETTER_I_WITH_OGONEK_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_SMALL_LETTER_I_WITH_OGONEK_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x012f };
static wchar_t* LATIN_SMALL_LETTER_I_WITH_OGONEK_UNICODE_CHARACTER_CODE_MODEL = LATIN_SMALL_LETTER_I_WITH_OGONEK_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_SMALL_LETTER_I_WITH_OGONEK_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_CAPITAL_LETTER_I_WITH_DOT_ABOVE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0130 };
static wchar_t* LATIN_CAPITAL_LETTER_I_WITH_DOT_ABOVE_UNICODE_CHARACTER_CODE_MODEL = LATIN_CAPITAL_LETTER_I_WITH_DOT_ABOVE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_CAPITAL_LETTER_I_WITH_DOT_ABOVE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_SMALL_LETTER_DOTLESS_I_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0131 };
static wchar_t* LATIN_SMALL_LETTER_DOTLESS_I_UNICODE_CHARACTER_CODE_MODEL = LATIN_SMALL_LETTER_DOTLESS_I_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_SMALL_LETTER_DOTLESS_I_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_CAPITAL_LIGATURE_IJ_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0132 };
static wchar_t* LATIN_CAPITAL_LIGATURE_IJ_UNICODE_CHARACTER_CODE_MODEL = LATIN_CAPITAL_LIGATURE_IJ_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_CAPITAL_LIGATURE_IJ_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_SMALL_LIGATURE_IJ_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0133 };
static wchar_t* LATIN_SMALL_LIGATURE_IJ_UNICODE_CHARACTER_CODE_MODEL = LATIN_SMALL_LIGATURE_IJ_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_SMALL_LIGATURE_IJ_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_CAPITAL_LETTER_J_WITH_CIRCUMFLEX_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0134 };
static wchar_t* LATIN_CAPITAL_LETTER_J_WITH_CIRCUMFLEX_UNICODE_CHARACTER_CODE_MODEL = LATIN_CAPITAL_LETTER_J_WITH_CIRCUMFLEX_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_CAPITAL_LETTER_J_WITH_CIRCUMFLEX_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_SMALL_LETTER_J_WITH_CIRCUMFLEX_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0135 };
static wchar_t* LATIN_SMALL_LETTER_J_WITH_CIRCUMFLEX_UNICODE_CHARACTER_CODE_MODEL = LATIN_SMALL_LETTER_J_WITH_CIRCUMFLEX_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_SMALL_LETTER_J_WITH_CIRCUMFLEX_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_CAPITAL_LETTER_K_WITH_CEDILLA_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0136 };
static wchar_t* LATIN_CAPITAL_LETTER_K_WITH_CEDILLA_UNICODE_CHARACTER_CODE_MODEL = LATIN_CAPITAL_LETTER_K_WITH_CEDILLA_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_CAPITAL_LETTER_K_WITH_CEDILLA_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_SMALL_LETTER_K_WITH_CEDILLA_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0137 };
static wchar_t* LATIN_SMALL_LETTER_K_WITH_CEDILLA_UNICODE_CHARACTER_CODE_MODEL = LATIN_SMALL_LETTER_K_WITH_CEDILLA_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_SMALL_LETTER_K_WITH_CEDILLA_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_SMALL_LETTER_KRA_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0138 };
static wchar_t* LATIN_SMALL_LETTER_KRA_UNICODE_CHARACTER_CODE_MODEL = LATIN_SMALL_LETTER_KRA_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_SMALL_LETTER_KRA_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_CAPITAL_LETTER_L_WITH_ACUTE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0139 };
static wchar_t* LATIN_CAPITAL_LETTER_L_WITH_ACUTE_UNICODE_CHARACTER_CODE_MODEL = LATIN_CAPITAL_LETTER_L_WITH_ACUTE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_CAPITAL_LETTER_L_WITH_ACUTE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_SMALL_LETTER_L_WITH_ACUTE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x013a };
static wchar_t* LATIN_SMALL_LETTER_L_WITH_ACUTE_UNICODE_CHARACTER_CODE_MODEL = LATIN_SMALL_LETTER_L_WITH_ACUTE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_SMALL_LETTER_L_WITH_ACUTE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_CAPITAL_LETTER_L_WITH_CEDILLA_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x013b };
static wchar_t* LATIN_CAPITAL_LETTER_L_WITH_CEDILLA_UNICODE_CHARACTER_CODE_MODEL = LATIN_CAPITAL_LETTER_L_WITH_CEDILLA_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_CAPITAL_LETTER_L_WITH_CEDILLA_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_SMALL_LETTER_L_WITH_CEDILLA_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x013c };
static wchar_t* LATIN_SMALL_LETTER_L_WITH_CEDILLA_UNICODE_CHARACTER_CODE_MODEL = LATIN_SMALL_LETTER_L_WITH_CEDILLA_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_SMALL_LETTER_L_WITH_CEDILLA_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_CAPITAL_LETTER_L_WITH_CARON_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x013d };
static wchar_t* LATIN_CAPITAL_LETTER_L_WITH_CARON_UNICODE_CHARACTER_CODE_MODEL = LATIN_CAPITAL_LETTER_L_WITH_CARON_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_CAPITAL_LETTER_L_WITH_CARON_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_SMALL_LETTER_L_WITH_CARON_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x013e };
static wchar_t* LATIN_SMALL_LETTER_L_WITH_CARON_UNICODE_CHARACTER_CODE_MODEL = LATIN_SMALL_LETTER_L_WITH_CARON_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_SMALL_LETTER_L_WITH_CARON_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_CAPITAL_LETTER_L_WITH_MIDDLE_DOT_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x013f };
static wchar_t* LATIN_CAPITAL_LETTER_L_WITH_MIDDLE_DOT_UNICODE_CHARACTER_CODE_MODEL = LATIN_CAPITAL_LETTER_L_WITH_MIDDLE_DOT_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_CAPITAL_LETTER_L_WITH_MIDDLE_DOT_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_SMALL_LETTER_L_WITH_MIDDLE_DOT_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0140 };
static wchar_t* LATIN_SMALL_LETTER_L_WITH_MIDDLE_DOT_UNICODE_CHARACTER_CODE_MODEL = LATIN_SMALL_LETTER_L_WITH_MIDDLE_DOT_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_SMALL_LETTER_L_WITH_MIDDLE_DOT_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_CAPITAL_LETTER_L_WITH_STROKE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0141 };
static wchar_t* LATIN_CAPITAL_LETTER_L_WITH_STROKE_UNICODE_CHARACTER_CODE_MODEL = LATIN_CAPITAL_LETTER_L_WITH_STROKE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_CAPITAL_LETTER_L_WITH_STROKE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_SMALL_LETTER_L_WITH_STROKE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0142 };
static wchar_t* LATIN_SMALL_LETTER_L_WITH_STROKE_UNICODE_CHARACTER_CODE_MODEL = LATIN_SMALL_LETTER_L_WITH_STROKE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_SMALL_LETTER_L_WITH_STROKE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_CAPITAL_LETTER_N_WITH_ACUTE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0143 };
static wchar_t* LATIN_CAPITAL_LETTER_N_WITH_ACUTE_UNICODE_CHARACTER_CODE_MODEL = LATIN_CAPITAL_LETTER_N_WITH_ACUTE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_CAPITAL_LETTER_N_WITH_ACUTE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_SMALL_LETTER_N_WITH_ACUTE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0144 };
static wchar_t* LATIN_SMALL_LETTER_N_WITH_ACUTE_UNICODE_CHARACTER_CODE_MODEL = LATIN_SMALL_LETTER_N_WITH_ACUTE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_SMALL_LETTER_N_WITH_ACUTE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_CAPITAL_LETTER_N_WITH_CEDILLA_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0145 };
static wchar_t* LATIN_CAPITAL_LETTER_N_WITH_CEDILLA_UNICODE_CHARACTER_CODE_MODEL = LATIN_CAPITAL_LETTER_N_WITH_CEDILLA_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_CAPITAL_LETTER_N_WITH_CEDILLA_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_SMALL_LETTER_N_WITH_CEDILLA_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0146 };
static wchar_t* LATIN_SMALL_LETTER_N_WITH_CEDILLA_UNICODE_CHARACTER_CODE_MODEL = LATIN_SMALL_LETTER_N_WITH_CEDILLA_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_SMALL_LETTER_N_WITH_CEDILLA_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_CAPITAL_LETTER_N_WITH_CARON_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0147 };
static wchar_t* LATIN_CAPITAL_LETTER_N_WITH_CARON_UNICODE_CHARACTER_CODE_MODEL = LATIN_CAPITAL_LETTER_N_WITH_CARON_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_CAPITAL_LETTER_N_WITH_CARON_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_SMALL_LETTER_N_WITH_CARON_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0148 };
static wchar_t* LATIN_SMALL_LETTER_N_WITH_CARON_UNICODE_CHARACTER_CODE_MODEL = LATIN_SMALL_LETTER_N_WITH_CARON_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_SMALL_LETTER_N_WITH_CARON_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_SMALL_LETTER_N_PRECEDED_BY_APOSTROPHE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0149 };
static wchar_t* LATIN_SMALL_LETTER_N_PRECEDED_BY_APOSTROPHE_UNICODE_CHARACTER_CODE_MODEL = LATIN_SMALL_LETTER_N_PRECEDED_BY_APOSTROPHE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_SMALL_LETTER_N_PRECEDED_BY_APOSTROPHE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_CAPITAL_LETTER_ENG_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x014a };
static wchar_t* LATIN_CAPITAL_LETTER_ENG_UNICODE_CHARACTER_CODE_MODEL = LATIN_CAPITAL_LETTER_ENG_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_CAPITAL_LETTER_ENG_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_SMALL_LETTER_ENG_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x014b };
static wchar_t* LATIN_SMALL_LETTER_ENG_UNICODE_CHARACTER_CODE_MODEL = LATIN_SMALL_LETTER_ENG_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_SMALL_LETTER_ENG_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_CAPITAL_LETTER_O_WITH_MACRON_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x014c };
static wchar_t* LATIN_CAPITAL_LETTER_O_WITH_MACRON_UNICODE_CHARACTER_CODE_MODEL = LATIN_CAPITAL_LETTER_O_WITH_MACRON_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_CAPITAL_LETTER_O_WITH_MACRON_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_SMALL_LETTER_O_WITH_MACRON_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x014d };
static wchar_t* LATIN_SMALL_LETTER_O_WITH_MACRON_UNICODE_CHARACTER_CODE_MODEL = LATIN_SMALL_LETTER_O_WITH_MACRON_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_SMALL_LETTER_O_WITH_MACRON_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_CAPITAL_LETTER_O_WITH_DOUBLE_ACUTE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0150 };
static wchar_t* LATIN_CAPITAL_LETTER_O_WITH_DOUBLE_ACUTE_UNICODE_CHARACTER_CODE_MODEL = LATIN_CAPITAL_LETTER_O_WITH_DOUBLE_ACUTE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_CAPITAL_LETTER_O_WITH_DOUBLE_ACUTE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_SMALL_LETTER_O_WITH_DOUBLE_ACUTE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0151 };
static wchar_t* LATIN_SMALL_LETTER_O_WITH_DOUBLE_ACUTE_UNICODE_CHARACTER_CODE_MODEL = LATIN_SMALL_LETTER_O_WITH_DOUBLE_ACUTE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_SMALL_LETTER_O_WITH_DOUBLE_ACUTE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_CAPITAL_LIGATURE_OE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0152 };
static wchar_t* LATIN_CAPITAL_LIGATURE_OE_UNICODE_CHARACTER_CODE_MODEL = LATIN_CAPITAL_LIGATURE_OE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_CAPITAL_LIGATURE_OE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_SMALL_LIGATURE_OE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0153 };
static wchar_t* LATIN_SMALL_LIGATURE_OE_UNICODE_CHARACTER_CODE_MODEL = LATIN_SMALL_LIGATURE_OE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_SMALL_LIGATURE_OE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_CAPITAL_LETTER_R_WITH_ACUTE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0154 };
static wchar_t* LATIN_CAPITAL_LETTER_R_WITH_ACUTE_UNICODE_CHARACTER_CODE_MODEL = LATIN_CAPITAL_LETTER_R_WITH_ACUTE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_CAPITAL_LETTER_R_WITH_ACUTE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_SMALL_LETTER_R_WITH_ACUTE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0155 };
static wchar_t* LATIN_SMALL_LETTER_R_WITH_ACUTE_UNICODE_CHARACTER_CODE_MODEL = LATIN_SMALL_LETTER_R_WITH_ACUTE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_SMALL_LETTER_R_WITH_ACUTE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_CAPITAL_LETTER_R_WITH_CEDILLA_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0156 };
static wchar_t* LATIN_CAPITAL_LETTER_R_WITH_CEDILLA_UNICODE_CHARACTER_CODE_MODEL = LATIN_CAPITAL_LETTER_R_WITH_CEDILLA_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_CAPITAL_LETTER_R_WITH_CEDILLA_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_SMALL_LETTER_R_WITH_CEDILLA_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0157 };
static wchar_t* LATIN_SMALL_LETTER_R_WITH_CEDILLA_UNICODE_CHARACTER_CODE_MODEL = LATIN_SMALL_LETTER_R_WITH_CEDILLA_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_SMALL_LETTER_R_WITH_CEDILLA_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_CAPITAL_LETTER_R_WITH_CARON_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0158 };
static wchar_t* LATIN_CAPITAL_LETTER_R_WITH_CARON_UNICODE_CHARACTER_CODE_MODEL = LATIN_CAPITAL_LETTER_R_WITH_CARON_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_CAPITAL_LETTER_R_WITH_CARON_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_SMALL_LETTER_R_WITH_CARON_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0159 };
static wchar_t* LATIN_SMALL_LETTER_R_WITH_CARON_UNICODE_CHARACTER_CODE_MODEL = LATIN_SMALL_LETTER_R_WITH_CARON_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_SMALL_LETTER_R_WITH_CARON_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_CAPITAL_LETTER_S_WITH_ACUTE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x015a };
static wchar_t* LATIN_CAPITAL_LETTER_S_WITH_ACUTE_UNICODE_CHARACTER_CODE_MODEL = LATIN_CAPITAL_LETTER_S_WITH_ACUTE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_CAPITAL_LETTER_S_WITH_ACUTE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_SMALL_LETTER_S_WITH_ACUTE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x015b };
static wchar_t* LATIN_SMALL_LETTER_S_WITH_ACUTE_UNICODE_CHARACTER_CODE_MODEL = LATIN_SMALL_LETTER_S_WITH_ACUTE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_SMALL_LETTER_S_WITH_ACUTE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_CAPITAL_LETTER_S_WITH_CIRCUMFLEX_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x015c };
static wchar_t* LATIN_CAPITAL_LETTER_S_WITH_CIRCUMFLEX_UNICODE_CHARACTER_CODE_MODEL = LATIN_CAPITAL_LETTER_S_WITH_CIRCUMFLEX_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_CAPITAL_LETTER_S_WITH_CIRCUMFLEX_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_SMALL_LETTER_S_WITH_CIRCUMFLEX_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x015d };
static wchar_t* LATIN_SMALL_LETTER_S_WITH_CIRCUMFLEX_UNICODE_CHARACTER_CODE_MODEL = LATIN_SMALL_LETTER_S_WITH_CIRCUMFLEX_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_SMALL_LETTER_S_WITH_CIRCUMFLEX_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_CAPITAL_LETTER_S_WITH_CEDILLA_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x015e };
static wchar_t* LATIN_CAPITAL_LETTER_S_WITH_CEDILLA_UNICODE_CHARACTER_CODE_MODEL = LATIN_CAPITAL_LETTER_S_WITH_CEDILLA_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_CAPITAL_LETTER_S_WITH_CEDILLA_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_SMALL_LETTER_S_WITH_CEDILLA_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x015f };
static wchar_t* LATIN_SMALL_LETTER_S_WITH_CEDILLA_UNICODE_CHARACTER_CODE_MODEL = LATIN_SMALL_LETTER_S_WITH_CEDILLA_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_SMALL_LETTER_S_WITH_CEDILLA_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_CAPITAL_LETTER_S_WITH_CARON_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0160 };
static wchar_t* LATIN_CAPITAL_LETTER_S_WITH_CARON_UNICODE_CHARACTER_CODE_MODEL = LATIN_CAPITAL_LETTER_S_WITH_CARON_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_CAPITAL_LETTER_S_WITH_CARON_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_SMALL_LETTER_S_WITH_CARON_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0161 };
static wchar_t* LATIN_SMALL_LETTER_S_WITH_CARON_UNICODE_CHARACTER_CODE_MODEL = LATIN_SMALL_LETTER_S_WITH_CARON_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_SMALL_LETTER_S_WITH_CARON_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_CAPITAL_LETTER_T_WITH_CEDILLA_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0162 };
static wchar_t* LATIN_CAPITAL_LETTER_T_WITH_CEDILLA_UNICODE_CHARACTER_CODE_MODEL = LATIN_CAPITAL_LETTER_T_WITH_CEDILLA_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_CAPITAL_LETTER_T_WITH_CEDILLA_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_SMALL_LETTER_T_WITH_CEDILLA_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0163 };
static wchar_t* LATIN_SMALL_LETTER_T_WITH_CEDILLA_UNICODE_CHARACTER_CODE_MODEL = LATIN_SMALL_LETTER_T_WITH_CEDILLA_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_SMALL_LETTER_T_WITH_CEDILLA_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_CAPITAL_LETTER_T_WITH_CARON_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0164 };
static wchar_t* LATIN_CAPITAL_LETTER_T_WITH_CARON_UNICODE_CHARACTER_CODE_MODEL = LATIN_CAPITAL_LETTER_T_WITH_CARON_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_CAPITAL_LETTER_T_WITH_CARON_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_SMALL_LETTER_T_WITH_CARON_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0165 };
static wchar_t* LATIN_SMALL_LETTER_T_WITH_CARON_UNICODE_CHARACTER_CODE_MODEL = LATIN_SMALL_LETTER_T_WITH_CARON_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_SMALL_LETTER_T_WITH_CARON_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_CAPITAL_LETTER_T_WITH_STROKE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0166 };
static wchar_t* LATIN_CAPITAL_LETTER_T_WITH_STROKE_UNICODE_CHARACTER_CODE_MODEL = LATIN_CAPITAL_LETTER_T_WITH_STROKE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_CAPITAL_LETTER_T_WITH_STROKE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_SMALL_LETTER_T_WITH_STROKE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0167 };
static wchar_t* LATIN_SMALL_LETTER_T_WITH_STROKE_UNICODE_CHARACTER_CODE_MODEL = LATIN_SMALL_LETTER_T_WITH_STROKE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_SMALL_LETTER_T_WITH_STROKE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_CAPITAL_LETTER_U_WITH_TILDE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0168 };
static wchar_t* LATIN_CAPITAL_LETTER_U_WITH_TILDE_UNICODE_CHARACTER_CODE_MODEL = LATIN_CAPITAL_LETTER_U_WITH_TILDE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_CAPITAL_LETTER_U_WITH_TILDE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_SMALL_LETTER_U_WITH_TILDE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0169 };
static wchar_t* LATIN_SMALL_LETTER_U_WITH_TILDE_UNICODE_CHARACTER_CODE_MODEL = LATIN_SMALL_LETTER_U_WITH_TILDE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_SMALL_LETTER_U_WITH_TILDE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_CAPITAL_LETTER_U_WITH_MACRON_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x016a };
static wchar_t* LATIN_CAPITAL_LETTER_U_WITH_MACRON_UNICODE_CHARACTER_CODE_MODEL = LATIN_CAPITAL_LETTER_U_WITH_MACRON_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_CAPITAL_LETTER_U_WITH_MACRON_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_SMALL_LETTER_U_WITH_MACRON_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x016b };
static wchar_t* LATIN_SMALL_LETTER_U_WITH_MACRON_UNICODE_CHARACTER_CODE_MODEL = LATIN_SMALL_LETTER_U_WITH_MACRON_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_SMALL_LETTER_U_WITH_MACRON_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_CAPITAL_LETTER_U_WITH_BREVE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x016c };
static wchar_t* LATIN_CAPITAL_LETTER_U_WITH_BREVE_UNICODE_CHARACTER_CODE_MODEL = LATIN_CAPITAL_LETTER_U_WITH_BREVE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_CAPITAL_LETTER_U_WITH_BREVE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_SMALL_LETTER_U_WITH_BREVE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x016d };
static wchar_t* LATIN_SMALL_LETTER_U_WITH_BREVE_UNICODE_CHARACTER_CODE_MODEL = LATIN_SMALL_LETTER_U_WITH_BREVE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_SMALL_LETTER_U_WITH_BREVE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_CAPITAL_LETTER_U_WITH_RING_ABOVE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x016e };
static wchar_t* LATIN_CAPITAL_LETTER_U_WITH_RING_ABOVE_UNICODE_CHARACTER_CODE_MODEL = LATIN_CAPITAL_LETTER_U_WITH_RING_ABOVE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_CAPITAL_LETTER_U_WITH_RING_ABOVE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_SMALL_LETTER_U_WITH_RING_ABOVE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x016f };
static wchar_t* LATIN_SMALL_LETTER_U_WITH_RING_ABOVE_UNICODE_CHARACTER_CODE_MODEL = LATIN_SMALL_LETTER_U_WITH_RING_ABOVE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_SMALL_LETTER_U_WITH_RING_ABOVE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_CAPITAL_LETTER_U_WITH_DOUBLE_ACUTE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0170 };
static wchar_t* LATIN_CAPITAL_LETTER_U_WITH_DOUBLE_ACUTE_UNICODE_CHARACTER_CODE_MODEL = LATIN_CAPITAL_LETTER_U_WITH_DOUBLE_ACUTE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_CAPITAL_LETTER_U_WITH_DOUBLE_ACUTE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_SMALL_LETTER_U_WITH_DOUBLE_ACUTE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0171 };
static wchar_t* LATIN_SMALL_LETTER_U_WITH_DOUBLE_ACUTE_UNICODE_CHARACTER_CODE_MODEL = LATIN_SMALL_LETTER_U_WITH_DOUBLE_ACUTE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_SMALL_LETTER_U_WITH_DOUBLE_ACUTE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_CAPITAL_LETTER_U_WITH_OGONEK_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0172 };
static wchar_t* LATIN_CAPITAL_LETTER_U_WITH_OGONEK_UNICODE_CHARACTER_CODE_MODEL = LATIN_CAPITAL_LETTER_U_WITH_OGONEK_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_CAPITAL_LETTER_U_WITH_OGONEK_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_SMALL_LETTER_U_WITH_OGONEK_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0173 };
static wchar_t* LATIN_SMALL_LETTER_U_WITH_OGONEK_UNICODE_CHARACTER_CODE_MODEL = LATIN_SMALL_LETTER_U_WITH_OGONEK_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_SMALL_LETTER_U_WITH_OGONEK_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_CAPITAL_LETTER_W_WITH_CIRCUMFLEX_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0174 };
static wchar_t* LATIN_CAPITAL_LETTER_W_WITH_CIRCUMFLEX_UNICODE_CHARACTER_CODE_MODEL = LATIN_CAPITAL_LETTER_W_WITH_CIRCUMFLEX_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_CAPITAL_LETTER_W_WITH_CIRCUMFLEX_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_SMALL_LETTER_W_WITH_CIRCUMFLEX_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0175 };
static wchar_t* LATIN_SMALL_LETTER_W_WITH_CIRCUMFLEX_UNICODE_CHARACTER_CODE_MODEL = LATIN_SMALL_LETTER_W_WITH_CIRCUMFLEX_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_SMALL_LETTER_W_WITH_CIRCUMFLEX_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_CAPITAL_LETTER_Y_WITH_CIRCUMFLEX_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0176 };
static wchar_t* LATIN_CAPITAL_LETTER_Y_WITH_CIRCUMFLEX_UNICODE_CHARACTER_CODE_MODEL = LATIN_CAPITAL_LETTER_Y_WITH_CIRCUMFLEX_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_CAPITAL_LETTER_Y_WITH_CIRCUMFLEX_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_SMALL_LETTER_Y_WITH_CIRCUMFLEX_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0177 };
static wchar_t* LATIN_SMALL_LETTER_Y_WITH_CIRCUMFLEX_UNICODE_CHARACTER_CODE_MODEL = LATIN_SMALL_LETTER_Y_WITH_CIRCUMFLEX_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_SMALL_LETTER_Y_WITH_CIRCUMFLEX_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_CAPITAL_LETTER_Y_WITH_DIAERESIS_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0178 };
static wchar_t* LATIN_CAPITAL_LETTER_Y_WITH_DIAERESIS_UNICODE_CHARACTER_CODE_MODEL = LATIN_CAPITAL_LETTER_Y_WITH_DIAERESIS_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_CAPITAL_LETTER_Y_WITH_DIAERESIS_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_CAPITAL_LETTER_Z_WITH_ACUTE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0179 };
static wchar_t* LATIN_CAPITAL_LETTER_Z_WITH_ACUTE_UNICODE_CHARACTER_CODE_MODEL = LATIN_CAPITAL_LETTER_Z_WITH_ACUTE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_CAPITAL_LETTER_Z_WITH_ACUTE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_SMALL_LETTER_Z_WITH_ACUTE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x017a };
static wchar_t* LATIN_SMALL_LETTER_Z_WITH_ACUTE_UNICODE_CHARACTER_CODE_MODEL = LATIN_SMALL_LETTER_Z_WITH_ACUTE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_SMALL_LETTER_Z_WITH_ACUTE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_CAPITAL_LETTER_Z_WITH_DOT_ABOVE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x017b };
static wchar_t* LATIN_CAPITAL_LETTER_Z_WITH_DOT_ABOVE_UNICODE_CHARACTER_CODE_MODEL = LATIN_CAPITAL_LETTER_Z_WITH_DOT_ABOVE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_CAPITAL_LETTER_Z_WITH_DOT_ABOVE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_SMALL_LETTER_Z_WITH_DOT_ABOVE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x017c };
static wchar_t* LATIN_SMALL_LETTER_Z_WITH_DOT_ABOVE_UNICODE_CHARACTER_CODE_MODEL = LATIN_SMALL_LETTER_Z_WITH_DOT_ABOVE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_SMALL_LETTER_Z_WITH_DOT_ABOVE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_CAPITAL_LETTER_Z_WITH_CARON_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x017d };
static wchar_t* LATIN_CAPITAL_LETTER_Z_WITH_CARON_UNICODE_CHARACTER_CODE_MODEL = LATIN_CAPITAL_LETTER_Z_WITH_CARON_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_CAPITAL_LETTER_Z_WITH_CARON_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_SMALL_LETTER_Z_WITH_CARON_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x017e };
static wchar_t* LATIN_SMALL_LETTER_Z_WITH_CARON_UNICODE_CHARACTER_CODE_MODEL = LATIN_SMALL_LETTER_Z_WITH_CARON_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_SMALL_LETTER_Z_WITH_CARON_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_SMALL_LETTER_F_WITH_HOOK_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0192 };
static wchar_t* LATIN_SMALL_LETTER_F_WITH_HOOK_UNICODE_CHARACTER_CODE_MODEL = LATIN_SMALL_LETTER_F_WITH_HOOK_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_SMALL_LETTER_F_WITH_HOOK_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_CAPITAL_LETTER_Z_WITH_STROKE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x01b5 };
static wchar_t* LATIN_CAPITAL_LETTER_Z_WITH_STROKE_UNICODE_CHARACTER_CODE_MODEL = LATIN_CAPITAL_LETTER_Z_WITH_STROKE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_CAPITAL_LETTER_Z_WITH_STROKE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_SMALL_LETTER_G_WITH_ACUTE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x01f5 };
static wchar_t* LATIN_SMALL_LETTER_G_WITH_ACUTE_UNICODE_CHARACTER_CODE_MODEL = LATIN_SMALL_LETTER_G_WITH_ACUTE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_SMALL_LETTER_G_WITH_ACUTE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_SMALL_LETTER_DOTLESS_J_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0237 };
static wchar_t* LATIN_SMALL_LETTER_DOTLESS_J_UNICODE_CHARACTER_CODE_MODEL = LATIN_SMALL_LETTER_DOTLESS_J_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_SMALL_LETTER_DOTLESS_J_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MODIFIER_LETTER_CIRCUMFLEX_ACCENT_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x02c6 };
static wchar_t* MODIFIER_LETTER_CIRCUMFLEX_ACCENT_UNICODE_CHARACTER_CODE_MODEL = MODIFIER_LETTER_CIRCUMFLEX_ACCENT_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MODIFIER_LETTER_CIRCUMFLEX_ACCENT_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t CARON_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x02c7 };
static wchar_t* CARON_UNICODE_CHARACTER_CODE_MODEL = CARON_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CARON_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t BREVE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x02d8 };
static wchar_t* BREVE_UNICODE_CHARACTER_CODE_MODEL = BREVE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* BREVE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t DOT_ABOVE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x02d9 };
static wchar_t* DOT_ABOVE_UNICODE_CHARACTER_CODE_MODEL = DOT_ABOVE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* DOT_ABOVE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t RING_ABOVE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x02da };
static wchar_t* RING_ABOVE_UNICODE_CHARACTER_CODE_MODEL = RING_ABOVE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* RING_ABOVE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t OGONEK_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x02db };
static wchar_t* OGONEK_UNICODE_CHARACTER_CODE_MODEL = OGONEK_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* OGONEK_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t SMALL_TILDE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x02dc };
static wchar_t* SMALL_TILDE_UNICODE_CHARACTER_CODE_MODEL = SMALL_TILDE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* SMALL_TILDE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t DOUBLE_ACUTE_ACCENT_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x02dd };
static wchar_t* DOUBLE_ACUTE_ACCENT_UNICODE_CHARACTER_CODE_MODEL = DOUBLE_ACUTE_ACCENT_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* DOUBLE_ACUTE_ACCENT_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t COMBINING_INVERTED_BREVE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0311 };
static wchar_t* COMBINING_INVERTED_BREVE_UNICODE_CHARACTER_CODE_MODEL = COMBINING_INVERTED_BREVE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* COMBINING_INVERTED_BREVE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t GREEK_CAPITAL_LETTER_ALPHA_WITH_TONOS_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0386 };
static wchar_t* GREEK_CAPITAL_LETTER_ALPHA_WITH_TONOS_UNICODE_CHARACTER_CODE_MODEL = GREEK_CAPITAL_LETTER_ALPHA_WITH_TONOS_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* GREEK_CAPITAL_LETTER_ALPHA_WITH_TONOS_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t GREEK_CAPITAL_LETTER_EPSILON_WITH_TONOS_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0388 };
static wchar_t* GREEK_CAPITAL_LETTER_EPSILON_WITH_TONOS_UNICODE_CHARACTER_CODE_MODEL = GREEK_CAPITAL_LETTER_EPSILON_WITH_TONOS_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* GREEK_CAPITAL_LETTER_EPSILON_WITH_TONOS_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t GREEK_CAPITAL_LETTER_ETA_WITH_TONOS_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0389 };
static wchar_t* GREEK_CAPITAL_LETTER_ETA_WITH_TONOS_UNICODE_CHARACTER_CODE_MODEL = GREEK_CAPITAL_LETTER_ETA_WITH_TONOS_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* GREEK_CAPITAL_LETTER_ETA_WITH_TONOS_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t GREEK_CAPITAL_LETTER_IOTA_WITH_TONOS_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x038a };
static wchar_t* GREEK_CAPITAL_LETTER_IOTA_WITH_TONOS_UNICODE_CHARACTER_CODE_MODEL = GREEK_CAPITAL_LETTER_IOTA_WITH_TONOS_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* GREEK_CAPITAL_LETTER_IOTA_WITH_TONOS_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t GREEK_CAPITAL_LETTER_OMICRON_WITH_TONOS_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x038c };
static wchar_t* GREEK_CAPITAL_LETTER_OMICRON_WITH_TONOS_UNICODE_CHARACTER_CODE_MODEL = GREEK_CAPITAL_LETTER_OMICRON_WITH_TONOS_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* GREEK_CAPITAL_LETTER_OMICRON_WITH_TONOS_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t GREEK_CAPITAL_LETTER_UPSILON_WITH_TONOS_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x038e };
static wchar_t* GREEK_CAPITAL_LETTER_UPSILON_WITH_TONOS_UNICODE_CHARACTER_CODE_MODEL = GREEK_CAPITAL_LETTER_UPSILON_WITH_TONOS_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* GREEK_CAPITAL_LETTER_UPSILON_WITH_TONOS_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t GREEK_CAPITAL_LETTER_OMEGA_WITH_TONOS_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x038f };
static wchar_t* GREEK_CAPITAL_LETTER_OMEGA_WITH_TONOS_UNICODE_CHARACTER_CODE_MODEL = GREEK_CAPITAL_LETTER_OMEGA_WITH_TONOS_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* GREEK_CAPITAL_LETTER_OMEGA_WITH_TONOS_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t GREEK_SMALL_LETTER_IOTA_WITH_DIALYTIKA_AND_TONOS_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0390 };
static wchar_t* GREEK_SMALL_LETTER_IOTA_WITH_DIALYTIKA_AND_TONOS_UNICODE_CHARACTER_CODE_MODEL = GREEK_SMALL_LETTER_IOTA_WITH_DIALYTIKA_AND_TONOS_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* GREEK_SMALL_LETTER_IOTA_WITH_DIALYTIKA_AND_TONOS_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t GREEK_CAPITAL_LETTER_ALPHA_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0391 };
static wchar_t* GREEK_CAPITAL_LETTER_ALPHA_UNICODE_CHARACTER_CODE_MODEL = GREEK_CAPITAL_LETTER_ALPHA_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* GREEK_CAPITAL_LETTER_ALPHA_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t GREEK_CAPITAL_LETTER_BETA_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0392 };
static wchar_t* GREEK_CAPITAL_LETTER_BETA_UNICODE_CHARACTER_CODE_MODEL = GREEK_CAPITAL_LETTER_BETA_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* GREEK_CAPITAL_LETTER_BETA_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t GREEK_CAPITAL_LETTER_GAMMA_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0393 };
static wchar_t* GREEK_CAPITAL_LETTER_GAMMA_UNICODE_CHARACTER_CODE_MODEL = GREEK_CAPITAL_LETTER_GAMMA_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* GREEK_CAPITAL_LETTER_GAMMA_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t GREEK_CAPITAL_LETTER_DELTA_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0394 };
static wchar_t* GREEK_CAPITAL_LETTER_DELTA_UNICODE_CHARACTER_CODE_MODEL = GREEK_CAPITAL_LETTER_DELTA_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* GREEK_CAPITAL_LETTER_DELTA_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t GREEK_CAPITAL_LETTER_EPSILON_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0395 };
static wchar_t* GREEK_CAPITAL_LETTER_EPSILON_UNICODE_CHARACTER_CODE_MODEL = GREEK_CAPITAL_LETTER_EPSILON_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* GREEK_CAPITAL_LETTER_EPSILON_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t GREEK_CAPITAL_LETTER_ZETA_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0396 };
static wchar_t* GREEK_CAPITAL_LETTER_ZETA_UNICODE_CHARACTER_CODE_MODEL = GREEK_CAPITAL_LETTER_ZETA_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* GREEK_CAPITAL_LETTER_ZETA_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t GREEK_CAPITAL_LETTER_ETA_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0397 };
static wchar_t* GREEK_CAPITAL_LETTER_ETA_UNICODE_CHARACTER_CODE_MODEL = GREEK_CAPITAL_LETTER_ETA_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* GREEK_CAPITAL_LETTER_ETA_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t GREEK_CAPITAL_LETTER_THETA_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0398 };
static wchar_t* GREEK_CAPITAL_LETTER_THETA_UNICODE_CHARACTER_CODE_MODEL = GREEK_CAPITAL_LETTER_THETA_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* GREEK_CAPITAL_LETTER_THETA_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t GREEK_CAPITAL_LETTER_IOTA_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0399 };
static wchar_t* GREEK_CAPITAL_LETTER_IOTA_UNICODE_CHARACTER_CODE_MODEL = GREEK_CAPITAL_LETTER_IOTA_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* GREEK_CAPITAL_LETTER_IOTA_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t GREEK_CAPITAL_LETTER_KAPPA_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x039a };
static wchar_t* GREEK_CAPITAL_LETTER_KAPPA_UNICODE_CHARACTER_CODE_MODEL = GREEK_CAPITAL_LETTER_KAPPA_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* GREEK_CAPITAL_LETTER_KAPPA_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t GREEK_CAPITAL_LETTER_LAMDA_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x039b };
static wchar_t* GREEK_CAPITAL_LETTER_LAMDA_UNICODE_CHARACTER_CODE_MODEL = GREEK_CAPITAL_LETTER_LAMDA_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* GREEK_CAPITAL_LETTER_LAMDA_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t GREEK_CAPITAL_LETTER_MU_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x039c };
static wchar_t* GREEK_CAPITAL_LETTER_MU_UNICODE_CHARACTER_CODE_MODEL = GREEK_CAPITAL_LETTER_MU_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* GREEK_CAPITAL_LETTER_MU_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t GREEK_CAPITAL_LETTER_NU_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x039d };
static wchar_t* GREEK_CAPITAL_LETTER_NU_UNICODE_CHARACTER_CODE_MODEL = GREEK_CAPITAL_LETTER_NU_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* GREEK_CAPITAL_LETTER_NU_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t GREEK_CAPITAL_LETTER_XI_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x039e };
static wchar_t* GREEK_CAPITAL_LETTER_XI_UNICODE_CHARACTER_CODE_MODEL = GREEK_CAPITAL_LETTER_XI_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* GREEK_CAPITAL_LETTER_XI_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t GREEK_CAPITAL_LETTER_OMICRON_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x039f };
static wchar_t* GREEK_CAPITAL_LETTER_OMICRON_UNICODE_CHARACTER_CODE_MODEL = GREEK_CAPITAL_LETTER_OMICRON_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* GREEK_CAPITAL_LETTER_OMICRON_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t GREEK_CAPITAL_LETTER_PI_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x03a0 };
static wchar_t* GREEK_CAPITAL_LETTER_PI_UNICODE_CHARACTER_CODE_MODEL = GREEK_CAPITAL_LETTER_PI_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* GREEK_CAPITAL_LETTER_PI_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t GREEK_CAPITAL_LETTER_RHO_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x03a1 };
static wchar_t* GREEK_CAPITAL_LETTER_RHO_UNICODE_CHARACTER_CODE_MODEL = GREEK_CAPITAL_LETTER_RHO_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* GREEK_CAPITAL_LETTER_RHO_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t GREEK_CAPITAL_LETTER_SIGMA_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x03a3 };
static wchar_t* GREEK_CAPITAL_LETTER_SIGMA_UNICODE_CHARACTER_CODE_MODEL = GREEK_CAPITAL_LETTER_SIGMA_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* GREEK_CAPITAL_LETTER_SIGMA_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t GREEK_CAPITAL_LETTER_TAU_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x03a4 };
static wchar_t* GREEK_CAPITAL_LETTER_TAU_UNICODE_CHARACTER_CODE_MODEL = GREEK_CAPITAL_LETTER_TAU_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* GREEK_CAPITAL_LETTER_TAU_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t GREEK_CAPITAL_LETTER_UPSILON_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x03a5 };
static wchar_t* GREEK_CAPITAL_LETTER_UPSILON_UNICODE_CHARACTER_CODE_MODEL = GREEK_CAPITAL_LETTER_UPSILON_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* GREEK_CAPITAL_LETTER_UPSILON_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t GREEK_CAPITAL_LETTER_PHI_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x03a6 };
static wchar_t* GREEK_CAPITAL_LETTER_PHI_UNICODE_CHARACTER_CODE_MODEL = GREEK_CAPITAL_LETTER_PHI_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* GREEK_CAPITAL_LETTER_PHI_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t GREEK_CAPITAL_LETTER_CHI_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x03a7 };
static wchar_t* GREEK_CAPITAL_LETTER_CHI_UNICODE_CHARACTER_CODE_MODEL = GREEK_CAPITAL_LETTER_CHI_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* GREEK_CAPITAL_LETTER_CHI_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t GREEK_CAPITAL_LETTER_PSI_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x03a8 };
static wchar_t* GREEK_CAPITAL_LETTER_PSI_UNICODE_CHARACTER_CODE_MODEL = GREEK_CAPITAL_LETTER_PSI_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* GREEK_CAPITAL_LETTER_PSI_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t GREEK_CAPITAL_LETTER_OMEGA_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x03a9 };
static wchar_t* GREEK_CAPITAL_LETTER_OMEGA_UNICODE_CHARACTER_CODE_MODEL = GREEK_CAPITAL_LETTER_OMEGA_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* GREEK_CAPITAL_LETTER_OMEGA_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t GREEK_CAPITAL_LETTER_IOTA_WITH_DIALYTIKA_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x03aa };
static wchar_t* GREEK_CAPITAL_LETTER_IOTA_WITH_DIALYTIKA_UNICODE_CHARACTER_CODE_MODEL = GREEK_CAPITAL_LETTER_IOTA_WITH_DIALYTIKA_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* GREEK_CAPITAL_LETTER_IOTA_WITH_DIALYTIKA_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t GREEK_CAPITAL_LETTER_UPSILON_WITH_DIALYTIKA_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x03ab };
static wchar_t* GREEK_CAPITAL_LETTER_UPSILON_WITH_DIALYTIKA_UNICODE_CHARACTER_CODE_MODEL = GREEK_CAPITAL_LETTER_UPSILON_WITH_DIALYTIKA_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* GREEK_CAPITAL_LETTER_UPSILON_WITH_DIALYTIKA_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t GREEK_SMALL_LETTER_ALPHA_WITH_TONOS_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x03ac };
static wchar_t* GREEK_SMALL_LETTER_ALPHA_WITH_TONOS_UNICODE_CHARACTER_CODE_MODEL = GREEK_SMALL_LETTER_ALPHA_WITH_TONOS_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* GREEK_SMALL_LETTER_ALPHA_WITH_TONOS_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t GREEK_SMALL_LETTER_EPSILON_WITH_TONOS_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x03ad };
static wchar_t* GREEK_SMALL_LETTER_EPSILON_WITH_TONOS_UNICODE_CHARACTER_CODE_MODEL = GREEK_SMALL_LETTER_EPSILON_WITH_TONOS_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* GREEK_SMALL_LETTER_EPSILON_WITH_TONOS_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t GREEK_SMALL_LETTER_ETA_WITH_TONOS_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x03ae };
static wchar_t* GREEK_SMALL_LETTER_ETA_WITH_TONOS_UNICODE_CHARACTER_CODE_MODEL = GREEK_SMALL_LETTER_ETA_WITH_TONOS_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* GREEK_SMALL_LETTER_ETA_WITH_TONOS_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t GREEK_SMALL_LETTER_IOTA_WITH_TONOS_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x03af };
static wchar_t* GREEK_SMALL_LETTER_IOTA_WITH_TONOS_UNICODE_CHARACTER_CODE_MODEL = GREEK_SMALL_LETTER_IOTA_WITH_TONOS_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* GREEK_SMALL_LETTER_IOTA_WITH_TONOS_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t GREEK_SMALL_LETTER_UPSILON_WITH_DIALYTIKA_AND_TONOS_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x03b0 };
static wchar_t* GREEK_SMALL_LETTER_UPSILON_WITH_DIALYTIKA_AND_TONOS_UNICODE_CHARACTER_CODE_MODEL = GREEK_SMALL_LETTER_UPSILON_WITH_DIALYTIKA_AND_TONOS_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* GREEK_SMALL_LETTER_UPSILON_WITH_DIALYTIKA_AND_TONOS_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t GREEK_SMALL_LETTER_ALPHA_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x03b1 };
static wchar_t* GREEK_SMALL_LETTER_ALPHA_UNICODE_CHARACTER_CODE_MODEL = GREEK_SMALL_LETTER_ALPHA_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* GREEK_SMALL_LETTER_ALPHA_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t GREEK_SMALL_LETTER_BETA_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x03b2 };
static wchar_t* GREEK_SMALL_LETTER_BETA_UNICODE_CHARACTER_CODE_MODEL = GREEK_SMALL_LETTER_BETA_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* GREEK_SMALL_LETTER_BETA_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t GREEK_SMALL_LETTER_GAMMA_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x03b3 };
static wchar_t* GREEK_SMALL_LETTER_GAMMA_UNICODE_CHARACTER_CODE_MODEL = GREEK_SMALL_LETTER_GAMMA_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* GREEK_SMALL_LETTER_GAMMA_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t GREEK_SMALL_LETTER_DELTA_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x03b4 };
static wchar_t* GREEK_SMALL_LETTER_DELTA_UNICODE_CHARACTER_CODE_MODEL = GREEK_SMALL_LETTER_DELTA_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* GREEK_SMALL_LETTER_DELTA_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t GREEK_SMALL_LETTER_EPSILON_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x03b5 };
static wchar_t* GREEK_SMALL_LETTER_EPSILON_UNICODE_CHARACTER_CODE_MODEL = GREEK_SMALL_LETTER_EPSILON_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* GREEK_SMALL_LETTER_EPSILON_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t GREEK_SMALL_LETTER_ZETA_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x03b6 };
static wchar_t* GREEK_SMALL_LETTER_ZETA_UNICODE_CHARACTER_CODE_MODEL = GREEK_SMALL_LETTER_ZETA_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* GREEK_SMALL_LETTER_ZETA_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t GREEK_SMALL_LETTER_ETA_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x03b7 };
static wchar_t* GREEK_SMALL_LETTER_ETA_UNICODE_CHARACTER_CODE_MODEL = GREEK_SMALL_LETTER_ETA_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* GREEK_SMALL_LETTER_ETA_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t GREEK_SMALL_LETTER_THETA_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x03b8 };
static wchar_t* GREEK_SMALL_LETTER_THETA_UNICODE_CHARACTER_CODE_MODEL = GREEK_SMALL_LETTER_THETA_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* GREEK_SMALL_LETTER_THETA_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t GREEK_SMALL_LETTER_IOTA_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x03b9 };
static wchar_t* GREEK_SMALL_LETTER_IOTA_UNICODE_CHARACTER_CODE_MODEL = GREEK_SMALL_LETTER_IOTA_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* GREEK_SMALL_LETTER_IOTA_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t GREEK_SMALL_LETTER_KAPPA_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x03ba };
static wchar_t* GREEK_SMALL_LETTER_KAPPA_UNICODE_CHARACTER_CODE_MODEL = GREEK_SMALL_LETTER_KAPPA_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* GREEK_SMALL_LETTER_KAPPA_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t GREEK_SMALL_LETTER_LAMDA_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x03bb };
static wchar_t* GREEK_SMALL_LETTER_LAMDA_UNICODE_CHARACTER_CODE_MODEL = GREEK_SMALL_LETTER_LAMDA_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* GREEK_SMALL_LETTER_LAMDA_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t GREEK_SMALL_LETTER_MU_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x03bc };
static wchar_t* GREEK_SMALL_LETTER_MU_UNICODE_CHARACTER_CODE_MODEL = GREEK_SMALL_LETTER_MU_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* GREEK_SMALL_LETTER_MU_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t GREEK_SMALL_LETTER_NU_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x03bd };
static wchar_t* GREEK_SMALL_LETTER_NU_UNICODE_CHARACTER_CODE_MODEL = GREEK_SMALL_LETTER_NU_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* GREEK_SMALL_LETTER_NU_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t GREEK_SMALL_LETTER_XI_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x03be };
static wchar_t* GREEK_SMALL_LETTER_XI_UNICODE_CHARACTER_CODE_MODEL = GREEK_SMALL_LETTER_XI_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* GREEK_SMALL_LETTER_XI_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t GREEK_SMALL_LETTER_OMICRON_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x03bf };
static wchar_t* GREEK_SMALL_LETTER_OMICRON_UNICODE_CHARACTER_CODE_MODEL = GREEK_SMALL_LETTER_OMICRON_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* GREEK_SMALL_LETTER_OMICRON_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t GREEK_SMALL_LETTER_PI_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x03c0 };
static wchar_t* GREEK_SMALL_LETTER_PI_UNICODE_CHARACTER_CODE_MODEL = GREEK_SMALL_LETTER_PI_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* GREEK_SMALL_LETTER_PI_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t GREEK_SMALL_LETTER_RHO_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x03c1 };
static wchar_t* GREEK_SMALL_LETTER_RHO_UNICODE_CHARACTER_CODE_MODEL = GREEK_SMALL_LETTER_RHO_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* GREEK_SMALL_LETTER_RHO_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t GREEK_SMALL_LETTER_FINAL_SIGMA_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x03c2 };
static wchar_t* GREEK_SMALL_LETTER_FINAL_SIGMA_UNICODE_CHARACTER_CODE_MODEL = GREEK_SMALL_LETTER_FINAL_SIGMA_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* GREEK_SMALL_LETTER_FINAL_SIGMA_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t GREEK_SMALL_LETTER_SIGMA_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x03c3 };
static wchar_t* GREEK_SMALL_LETTER_SIGMA_UNICODE_CHARACTER_CODE_MODEL = GREEK_SMALL_LETTER_SIGMA_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* GREEK_SMALL_LETTER_SIGMA_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t GREEK_SMALL_LETTER_TAU_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x03c4 };
static wchar_t* GREEK_SMALL_LETTER_TAU_UNICODE_CHARACTER_CODE_MODEL = GREEK_SMALL_LETTER_TAU_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* GREEK_SMALL_LETTER_TAU_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t GREEK_SMALL_LETTER_UPSILON_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x03c5 };
static wchar_t* GREEK_SMALL_LETTER_UPSILON_UNICODE_CHARACTER_CODE_MODEL = GREEK_SMALL_LETTER_UPSILON_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* GREEK_SMALL_LETTER_UPSILON_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t GREEK_SMALL_LETTER_PHI_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x03c6 };
static wchar_t* GREEK_SMALL_LETTER_PHI_UNICODE_CHARACTER_CODE_MODEL = GREEK_SMALL_LETTER_PHI_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* GREEK_SMALL_LETTER_PHI_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t GREEK_SMALL_LETTER_CHI_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x03c7 };
static wchar_t* GREEK_SMALL_LETTER_CHI_UNICODE_CHARACTER_CODE_MODEL = GREEK_SMALL_LETTER_CHI_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* GREEK_SMALL_LETTER_CHI_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t GREEK_SMALL_LETTER_PSI_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x03c8 };
static wchar_t* GREEK_SMALL_LETTER_PSI_UNICODE_CHARACTER_CODE_MODEL = GREEK_SMALL_LETTER_PSI_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* GREEK_SMALL_LETTER_PSI_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t GREEK_SMALL_LETTER_OMEGA_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x03c9 };
static wchar_t* GREEK_SMALL_LETTER_OMEGA_UNICODE_CHARACTER_CODE_MODEL = GREEK_SMALL_LETTER_OMEGA_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* GREEK_SMALL_LETTER_OMEGA_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t GREEK_SMALL_LETTER_IOTA_WITH_DIALYTIKA_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x03ca };
static wchar_t* GREEK_SMALL_LETTER_IOTA_WITH_DIALYTIKA_UNICODE_CHARACTER_CODE_MODEL = GREEK_SMALL_LETTER_IOTA_WITH_DIALYTIKA_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* GREEK_SMALL_LETTER_IOTA_WITH_DIALYTIKA_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t GREEK_SMALL_LETTER_UPSILON_WITH_DIALYTIKA_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x03cb };
static wchar_t* GREEK_SMALL_LETTER_UPSILON_WITH_DIALYTIKA_UNICODE_CHARACTER_CODE_MODEL = GREEK_SMALL_LETTER_UPSILON_WITH_DIALYTIKA_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* GREEK_SMALL_LETTER_UPSILON_WITH_DIALYTIKA_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t GREEK_SMALL_LETTER_OMICRON_WITH_TONOS_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x03cc };
static wchar_t* GREEK_SMALL_LETTER_OMICRON_WITH_TONOS_UNICODE_CHARACTER_CODE_MODEL = GREEK_SMALL_LETTER_OMICRON_WITH_TONOS_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* GREEK_SMALL_LETTER_OMICRON_WITH_TONOS_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t GREEK_SMALL_LETTER_UPSILON_WITH_TONOS_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x03cd };
static wchar_t* GREEK_SMALL_LETTER_UPSILON_WITH_TONOS_UNICODE_CHARACTER_CODE_MODEL = GREEK_SMALL_LETTER_UPSILON_WITH_TONOS_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* GREEK_SMALL_LETTER_UPSILON_WITH_TONOS_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t GREEK_SMALL_LETTER_OMEGA_WITH_TONOS_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x03ce };
static wchar_t* GREEK_SMALL_LETTER_OMEGA_WITH_TONOS_UNICODE_CHARACTER_CODE_MODEL = GREEK_SMALL_LETTER_OMEGA_WITH_TONOS_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* GREEK_SMALL_LETTER_OMEGA_WITH_TONOS_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t GREEK_THETA_SYMBOL_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x03d1 };
static wchar_t* GREEK_THETA_SYMBOL_UNICODE_CHARACTER_CODE_MODEL = GREEK_THETA_SYMBOL_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* GREEK_THETA_SYMBOL_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t GREEK_UPSILON_WITH_HOOK_SYMBOL_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x03d2 };
static wchar_t* GREEK_UPSILON_WITH_HOOK_SYMBOL_UNICODE_CHARACTER_CODE_MODEL = GREEK_UPSILON_WITH_HOOK_SYMBOL_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* GREEK_UPSILON_WITH_HOOK_SYMBOL_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t GREEK_PHI_SYMBOL_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x03d5 };
static wchar_t* GREEK_PHI_SYMBOL_UNICODE_CHARACTER_CODE_MODEL = GREEK_PHI_SYMBOL_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* GREEK_PHI_SYMBOL_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t GREEK_PI_SYMBOL_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x03d6 };
static wchar_t* GREEK_PI_SYMBOL_UNICODE_CHARACTER_CODE_MODEL = GREEK_PI_SYMBOL_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* GREEK_PI_SYMBOL_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t GREEK_LETTER_DIGAMMA_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x03dc };
static wchar_t* GREEK_LETTER_DIGAMMA_UNICODE_CHARACTER_CODE_MODEL = GREEK_LETTER_DIGAMMA_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* GREEK_LETTER_DIGAMMA_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t GREEK_SMALL_LETTER_DIGAMMA_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x03dd };
static wchar_t* GREEK_SMALL_LETTER_DIGAMMA_UNICODE_CHARACTER_CODE_MODEL = GREEK_SMALL_LETTER_DIGAMMA_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* GREEK_SMALL_LETTER_DIGAMMA_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t GREEK_KAPPA_SYMBOL_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x03f0 };
static wchar_t* GREEK_KAPPA_SYMBOL_UNICODE_CHARACTER_CODE_MODEL = GREEK_KAPPA_SYMBOL_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* GREEK_KAPPA_SYMBOL_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t GREEK_RHO_SYMBOL_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x03f1 };
static wchar_t* GREEK_RHO_SYMBOL_UNICODE_CHARACTER_CODE_MODEL = GREEK_RHO_SYMBOL_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* GREEK_RHO_SYMBOL_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t GREEK_LUNATE_EPSILON_SYMBOL_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x03f5 };
static wchar_t* GREEK_LUNATE_EPSILON_SYMBOL_UNICODE_CHARACTER_CODE_MODEL = GREEK_LUNATE_EPSILON_SYMBOL_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* GREEK_LUNATE_EPSILON_SYMBOL_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t GREEK_REVERSED_LUNATE_EPSILON_SYMBOL_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x03f6 };
static wchar_t* GREEK_REVERSED_LUNATE_EPSILON_SYMBOL_UNICODE_CHARACTER_CODE_MODEL = GREEK_REVERSED_LUNATE_EPSILON_SYMBOL_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* GREEK_REVERSED_LUNATE_EPSILON_SYMBOL_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t CYRILLIC_CAPITAL_LETTER_IO_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0401 };
static wchar_t* CYRILLIC_CAPITAL_LETTER_IO_UNICODE_CHARACTER_CODE_MODEL = CYRILLIC_CAPITAL_LETTER_IO_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CYRILLIC_CAPITAL_LETTER_IO_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t CYRILLIC_CAPITAL_LETTER_DJE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0402 };
static wchar_t* CYRILLIC_CAPITAL_LETTER_DJE_UNICODE_CHARACTER_CODE_MODEL = CYRILLIC_CAPITAL_LETTER_DJE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CYRILLIC_CAPITAL_LETTER_DJE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t CYRILLIC_CAPITAL_LETTER_GJE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0403 };
static wchar_t* CYRILLIC_CAPITAL_LETTER_GJE_UNICODE_CHARACTER_CODE_MODEL = CYRILLIC_CAPITAL_LETTER_GJE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CYRILLIC_CAPITAL_LETTER_GJE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t CYRILLIC_CAPITAL_LETTER_UKRAINIAN_IE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0404 };
static wchar_t* CYRILLIC_CAPITAL_LETTER_UKRAINIAN_IE_UNICODE_CHARACTER_CODE_MODEL = CYRILLIC_CAPITAL_LETTER_UKRAINIAN_IE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CYRILLIC_CAPITAL_LETTER_UKRAINIAN_IE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t CYRILLIC_CAPITAL_LETTER_DZE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0405 };
static wchar_t* CYRILLIC_CAPITAL_LETTER_DZE_UNICODE_CHARACTER_CODE_MODEL = CYRILLIC_CAPITAL_LETTER_DZE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CYRILLIC_CAPITAL_LETTER_DZE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t CYRILLIC_CAPITAL_LETTER_BYELORUSSIAN_UKRAINIAN_I_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0406 };
static wchar_t* CYRILLIC_CAPITAL_LETTER_BYELORUSSIAN_UKRAINIAN_I_UNICODE_CHARACTER_CODE_MODEL = CYRILLIC_CAPITAL_LETTER_BYELORUSSIAN_UKRAINIAN_I_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CYRILLIC_CAPITAL_LETTER_BYELORUSSIAN_UKRAINIAN_I_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t CYRILLIC_CAPITAL_LETTER_YI_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0407 };
static wchar_t* CYRILLIC_CAPITAL_LETTER_YI_UNICODE_CHARACTER_CODE_MODEL = CYRILLIC_CAPITAL_LETTER_YI_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CYRILLIC_CAPITAL_LETTER_YI_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t CYRILLIC_CAPITAL_LETTER_JE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0408 };
static wchar_t* CYRILLIC_CAPITAL_LETTER_JE_UNICODE_CHARACTER_CODE_MODEL = CYRILLIC_CAPITAL_LETTER_JE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CYRILLIC_CAPITAL_LETTER_JE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t CYRILLIC_CAPITAL_LETTER_LJE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0409 };
static wchar_t* CYRILLIC_CAPITAL_LETTER_LJE_UNICODE_CHARACTER_CODE_MODEL = CYRILLIC_CAPITAL_LETTER_LJE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CYRILLIC_CAPITAL_LETTER_LJE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t CYRILLIC_CAPITAL_LETTER_NJE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x040a };
static wchar_t* CYRILLIC_CAPITAL_LETTER_NJE_UNICODE_CHARACTER_CODE_MODEL = CYRILLIC_CAPITAL_LETTER_NJE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CYRILLIC_CAPITAL_LETTER_NJE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t CYRILLIC_CAPITAL_LETTER_TSHE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x040b };
static wchar_t* CYRILLIC_CAPITAL_LETTER_TSHE_UNICODE_CHARACTER_CODE_MODEL = CYRILLIC_CAPITAL_LETTER_TSHE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CYRILLIC_CAPITAL_LETTER_TSHE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t CYRILLIC_CAPITAL_LETTER_KJE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x040c };
static wchar_t* CYRILLIC_CAPITAL_LETTER_KJE_UNICODE_CHARACTER_CODE_MODEL = CYRILLIC_CAPITAL_LETTER_KJE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CYRILLIC_CAPITAL_LETTER_KJE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t CYRILLIC_CAPITAL_LETTER_SHORT_U_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x040e };
static wchar_t* CYRILLIC_CAPITAL_LETTER_SHORT_U_UNICODE_CHARACTER_CODE_MODEL = CYRILLIC_CAPITAL_LETTER_SHORT_U_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CYRILLIC_CAPITAL_LETTER_SHORT_U_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t CYRILLIC_CAPITAL_LETTER_DZHE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x040f };
static wchar_t* CYRILLIC_CAPITAL_LETTER_DZHE_UNICODE_CHARACTER_CODE_MODEL = CYRILLIC_CAPITAL_LETTER_DZHE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CYRILLIC_CAPITAL_LETTER_DZHE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t CYRILLIC_CAPITAL_LETTER_A_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0410 };
static wchar_t* CYRILLIC_CAPITAL_LETTER_A_UNICODE_CHARACTER_CODE_MODEL = CYRILLIC_CAPITAL_LETTER_A_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CYRILLIC_CAPITAL_LETTER_A_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t CYRILLIC_CAPITAL_LETTER_BE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0411 };
static wchar_t* CYRILLIC_CAPITAL_LETTER_BE_UNICODE_CHARACTER_CODE_MODEL = CYRILLIC_CAPITAL_LETTER_BE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CYRILLIC_CAPITAL_LETTER_BE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t CYRILLIC_CAPITAL_LETTER_VE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0412 };
static wchar_t* CYRILLIC_CAPITAL_LETTER_VE_UNICODE_CHARACTER_CODE_MODEL = CYRILLIC_CAPITAL_LETTER_VE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CYRILLIC_CAPITAL_LETTER_VE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t CYRILLIC_CAPITAL_LETTER_GHE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0413 };
static wchar_t* CYRILLIC_CAPITAL_LETTER_GHE_UNICODE_CHARACTER_CODE_MODEL = CYRILLIC_CAPITAL_LETTER_GHE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CYRILLIC_CAPITAL_LETTER_GHE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t CYRILLIC_CAPITAL_LETTER_DE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0414 };
static wchar_t* CYRILLIC_CAPITAL_LETTER_DE_UNICODE_CHARACTER_CODE_MODEL = CYRILLIC_CAPITAL_LETTER_DE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CYRILLIC_CAPITAL_LETTER_DE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t CYRILLIC_CAPITAL_LETTER_IE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0415 };
static wchar_t* CYRILLIC_CAPITAL_LETTER_IE_UNICODE_CHARACTER_CODE_MODEL = CYRILLIC_CAPITAL_LETTER_IE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CYRILLIC_CAPITAL_LETTER_IE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t CYRILLIC_CAPITAL_LETTER_ZHE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0416 };
static wchar_t* CYRILLIC_CAPITAL_LETTER_ZHE_UNICODE_CHARACTER_CODE_MODEL = CYRILLIC_CAPITAL_LETTER_ZHE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CYRILLIC_CAPITAL_LETTER_ZHE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t CYRILLIC_CAPITAL_LETTER_ZE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0417 };
static wchar_t* CYRILLIC_CAPITAL_LETTER_ZE_UNICODE_CHARACTER_CODE_MODEL = CYRILLIC_CAPITAL_LETTER_ZE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CYRILLIC_CAPITAL_LETTER_ZE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t CYRILLIC_CAPITAL_LETTER_I_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0418 };
static wchar_t* CYRILLIC_CAPITAL_LETTER_I_UNICODE_CHARACTER_CODE_MODEL = CYRILLIC_CAPITAL_LETTER_I_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CYRILLIC_CAPITAL_LETTER_I_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t CYRILLIC_CAPITAL_LETTER_SHORT_I_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0419 };
static wchar_t* CYRILLIC_CAPITAL_LETTER_SHORT_I_UNICODE_CHARACTER_CODE_MODEL = CYRILLIC_CAPITAL_LETTER_SHORT_I_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CYRILLIC_CAPITAL_LETTER_SHORT_I_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t CYRILLIC_CAPITAL_LETTER_KA_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x041a };
static wchar_t* CYRILLIC_CAPITAL_LETTER_KA_UNICODE_CHARACTER_CODE_MODEL = CYRILLIC_CAPITAL_LETTER_KA_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CYRILLIC_CAPITAL_LETTER_KA_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t CYRILLIC_CAPITAL_LETTER_EL_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x041b };
static wchar_t* CYRILLIC_CAPITAL_LETTER_EL_UNICODE_CHARACTER_CODE_MODEL = CYRILLIC_CAPITAL_LETTER_EL_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CYRILLIC_CAPITAL_LETTER_EL_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t CYRILLIC_CAPITAL_LETTER_EM_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x041c };
static wchar_t* CYRILLIC_CAPITAL_LETTER_EM_UNICODE_CHARACTER_CODE_MODEL = CYRILLIC_CAPITAL_LETTER_EM_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CYRILLIC_CAPITAL_LETTER_EM_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t CYRILLIC_CAPITAL_LETTER_EN_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x041d };
static wchar_t* CYRILLIC_CAPITAL_LETTER_EN_UNICODE_CHARACTER_CODE_MODEL = CYRILLIC_CAPITAL_LETTER_EN_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CYRILLIC_CAPITAL_LETTER_EN_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t CYRILLIC_CAPITAL_LETTER_O_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x041e };
static wchar_t* CYRILLIC_CAPITAL_LETTER_O_UNICODE_CHARACTER_CODE_MODEL = CYRILLIC_CAPITAL_LETTER_O_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CYRILLIC_CAPITAL_LETTER_O_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t CYRILLIC_CAPITAL_LETTER_PE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x041f };
static wchar_t* CYRILLIC_CAPITAL_LETTER_PE_UNICODE_CHARACTER_CODE_MODEL = CYRILLIC_CAPITAL_LETTER_PE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CYRILLIC_CAPITAL_LETTER_PE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t CYRILLIC_CAPITAL_LETTER_ER_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0420 };
static wchar_t* CYRILLIC_CAPITAL_LETTER_ER_UNICODE_CHARACTER_CODE_MODEL = CYRILLIC_CAPITAL_LETTER_ER_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CYRILLIC_CAPITAL_LETTER_ER_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t CYRILLIC_CAPITAL_LETTER_ES_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0421 };
static wchar_t* CYRILLIC_CAPITAL_LETTER_ES_UNICODE_CHARACTER_CODE_MODEL = CYRILLIC_CAPITAL_LETTER_ES_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CYRILLIC_CAPITAL_LETTER_ES_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t CYRILLIC_CAPITAL_LETTER_TE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0422 };
static wchar_t* CYRILLIC_CAPITAL_LETTER_TE_UNICODE_CHARACTER_CODE_MODEL = CYRILLIC_CAPITAL_LETTER_TE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CYRILLIC_CAPITAL_LETTER_TE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t CYRILLIC_CAPITAL_LETTER_U_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0423 };
static wchar_t* CYRILLIC_CAPITAL_LETTER_U_UNICODE_CHARACTER_CODE_MODEL = CYRILLIC_CAPITAL_LETTER_U_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CYRILLIC_CAPITAL_LETTER_U_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t CYRILLIC_CAPITAL_LETTER_EF_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0424 };
static wchar_t* CYRILLIC_CAPITAL_LETTER_EF_UNICODE_CHARACTER_CODE_MODEL = CYRILLIC_CAPITAL_LETTER_EF_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CYRILLIC_CAPITAL_LETTER_EF_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t CYRILLIC_CAPITAL_LETTER_HA_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0425 };
static wchar_t* CYRILLIC_CAPITAL_LETTER_HA_UNICODE_CHARACTER_CODE_MODEL = CYRILLIC_CAPITAL_LETTER_HA_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CYRILLIC_CAPITAL_LETTER_HA_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t CYRILLIC_CAPITAL_LETTER_TSE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0426 };
static wchar_t* CYRILLIC_CAPITAL_LETTER_TSE_UNICODE_CHARACTER_CODE_MODEL = CYRILLIC_CAPITAL_LETTER_TSE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CYRILLIC_CAPITAL_LETTER_TSE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t CYRILLIC_CAPITAL_LETTER_CHE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0427 };
static wchar_t* CYRILLIC_CAPITAL_LETTER_CHE_UNICODE_CHARACTER_CODE_MODEL = CYRILLIC_CAPITAL_LETTER_CHE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CYRILLIC_CAPITAL_LETTER_CHE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t CYRILLIC_CAPITAL_LETTER_SHA_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0428 };
static wchar_t* CYRILLIC_CAPITAL_LETTER_SHA_UNICODE_CHARACTER_CODE_MODEL = CYRILLIC_CAPITAL_LETTER_SHA_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CYRILLIC_CAPITAL_LETTER_SHA_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t CYRILLIC_CAPITAL_LETTER_SHCHA_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0429 };
static wchar_t* CYRILLIC_CAPITAL_LETTER_SHCHA_UNICODE_CHARACTER_CODE_MODEL = CYRILLIC_CAPITAL_LETTER_SHCHA_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CYRILLIC_CAPITAL_LETTER_SHCHA_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t CYRILLIC_CAPITAL_LETTER_HARD_SIGN_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x042a };
static wchar_t* CYRILLIC_CAPITAL_LETTER_HARD_SIGN_UNICODE_CHARACTER_CODE_MODEL = CYRILLIC_CAPITAL_LETTER_HARD_SIGN_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CYRILLIC_CAPITAL_LETTER_HARD_SIGN_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t CYRILLIC_CAPITAL_LETTER_YERU_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x042b };
static wchar_t* CYRILLIC_CAPITAL_LETTER_YERU_UNICODE_CHARACTER_CODE_MODEL = CYRILLIC_CAPITAL_LETTER_YERU_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CYRILLIC_CAPITAL_LETTER_YERU_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t CYRILLIC_CAPITAL_LETTER_SOFT_SIGN_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x042c };
static wchar_t* CYRILLIC_CAPITAL_LETTER_SOFT_SIGN_UNICODE_CHARACTER_CODE_MODEL = CYRILLIC_CAPITAL_LETTER_SOFT_SIGN_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CYRILLIC_CAPITAL_LETTER_SOFT_SIGN_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t CYRILLIC_CAPITAL_LETTER_E_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x042d };
static wchar_t* CYRILLIC_CAPITAL_LETTER_E_UNICODE_CHARACTER_CODE_MODEL = CYRILLIC_CAPITAL_LETTER_E_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CYRILLIC_CAPITAL_LETTER_E_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t CYRILLIC_CAPITAL_LETTER_YU_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x042e };
static wchar_t* CYRILLIC_CAPITAL_LETTER_YU_UNICODE_CHARACTER_CODE_MODEL = CYRILLIC_CAPITAL_LETTER_YU_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CYRILLIC_CAPITAL_LETTER_YU_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t CYRILLIC_CAPITAL_LETTER_YA_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x042f };
static wchar_t* CYRILLIC_CAPITAL_LETTER_YA_UNICODE_CHARACTER_CODE_MODEL = CYRILLIC_CAPITAL_LETTER_YA_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CYRILLIC_CAPITAL_LETTER_YA_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t CYRILLIC_SMALL_LETTER_A_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0430 };
static wchar_t* CYRILLIC_SMALL_LETTER_A_UNICODE_CHARACTER_CODE_MODEL = CYRILLIC_SMALL_LETTER_A_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CYRILLIC_SMALL_LETTER_A_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t CYRILLIC_SMALL_LETTER_BE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0431 };
static wchar_t* CYRILLIC_SMALL_LETTER_BE_UNICODE_CHARACTER_CODE_MODEL = CYRILLIC_SMALL_LETTER_BE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CYRILLIC_SMALL_LETTER_BE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t CYRILLIC_SMALL_LETTER_VE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0432 };
static wchar_t* CYRILLIC_SMALL_LETTER_VE_UNICODE_CHARACTER_CODE_MODEL = CYRILLIC_SMALL_LETTER_VE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CYRILLIC_SMALL_LETTER_VE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t CYRILLIC_SMALL_LETTER_GHE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0433 };
static wchar_t* CYRILLIC_SMALL_LETTER_GHE_UNICODE_CHARACTER_CODE_MODEL = CYRILLIC_SMALL_LETTER_GHE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CYRILLIC_SMALL_LETTER_GHE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t CYRILLIC_SMALL_LETTER_DE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0434 };
static wchar_t* CYRILLIC_SMALL_LETTER_DE_UNICODE_CHARACTER_CODE_MODEL = CYRILLIC_SMALL_LETTER_DE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CYRILLIC_SMALL_LETTER_DE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t CYRILLIC_SMALL_LETTER_IE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0435 };
static wchar_t* CYRILLIC_SMALL_LETTER_IE_UNICODE_CHARACTER_CODE_MODEL = CYRILLIC_SMALL_LETTER_IE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CYRILLIC_SMALL_LETTER_IE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t CYRILLIC_SMALL_LETTER_ZHE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0436 };
static wchar_t* CYRILLIC_SMALL_LETTER_ZHE_UNICODE_CHARACTER_CODE_MODEL = CYRILLIC_SMALL_LETTER_ZHE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CYRILLIC_SMALL_LETTER_ZHE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t CYRILLIC_SMALL_LETTER_ZE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0437 };
static wchar_t* CYRILLIC_SMALL_LETTER_ZE_UNICODE_CHARACTER_CODE_MODEL = CYRILLIC_SMALL_LETTER_ZE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CYRILLIC_SMALL_LETTER_ZE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t CYRILLIC_SMALL_LETTER_I_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0438 };
static wchar_t* CYRILLIC_SMALL_LETTER_I_UNICODE_CHARACTER_CODE_MODEL = CYRILLIC_SMALL_LETTER_I_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CYRILLIC_SMALL_LETTER_I_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t CYRILLIC_SMALL_LETTER_SHORT_I_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0439 };
static wchar_t* CYRILLIC_SMALL_LETTER_SHORT_I_UNICODE_CHARACTER_CODE_MODEL = CYRILLIC_SMALL_LETTER_SHORT_I_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CYRILLIC_SMALL_LETTER_SHORT_I_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t CYRILLIC_SMALL_LETTER_KA_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x043a };
static wchar_t* CYRILLIC_SMALL_LETTER_KA_UNICODE_CHARACTER_CODE_MODEL = CYRILLIC_SMALL_LETTER_KA_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CYRILLIC_SMALL_LETTER_KA_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t CYRILLIC_SMALL_LETTER_EL_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x043b };
static wchar_t* CYRILLIC_SMALL_LETTER_EL_UNICODE_CHARACTER_CODE_MODEL = CYRILLIC_SMALL_LETTER_EL_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CYRILLIC_SMALL_LETTER_EL_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t CYRILLIC_SMALL_LETTER_EM_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x043c };
static wchar_t* CYRILLIC_SMALL_LETTER_EM_UNICODE_CHARACTER_CODE_MODEL = CYRILLIC_SMALL_LETTER_EM_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CYRILLIC_SMALL_LETTER_EM_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t CYRILLIC_SMALL_LETTER_EN_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x043d };
static wchar_t* CYRILLIC_SMALL_LETTER_EN_UNICODE_CHARACTER_CODE_MODEL = CYRILLIC_SMALL_LETTER_EN_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CYRILLIC_SMALL_LETTER_EN_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t CYRILLIC_SMALL_LETTER_O_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x043e };
static wchar_t* CYRILLIC_SMALL_LETTER_O_UNICODE_CHARACTER_CODE_MODEL = CYRILLIC_SMALL_LETTER_O_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CYRILLIC_SMALL_LETTER_O_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t CYRILLIC_SMALL_LETTER_PE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x043f };
static wchar_t* CYRILLIC_SMALL_LETTER_PE_UNICODE_CHARACTER_CODE_MODEL = CYRILLIC_SMALL_LETTER_PE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CYRILLIC_SMALL_LETTER_PE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t CYRILLIC_SMALL_LETTER_ER_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0440 };
static wchar_t* CYRILLIC_SMALL_LETTER_ER_UNICODE_CHARACTER_CODE_MODEL = CYRILLIC_SMALL_LETTER_ER_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CYRILLIC_SMALL_LETTER_ER_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t CYRILLIC_SMALL_LETTER_ES_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0441 };
static wchar_t* CYRILLIC_SMALL_LETTER_ES_UNICODE_CHARACTER_CODE_MODEL = CYRILLIC_SMALL_LETTER_ES_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CYRILLIC_SMALL_LETTER_ES_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t CYRILLIC_SMALL_LETTER_TE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0442 };
static wchar_t* CYRILLIC_SMALL_LETTER_TE_UNICODE_CHARACTER_CODE_MODEL = CYRILLIC_SMALL_LETTER_TE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CYRILLIC_SMALL_LETTER_TE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t CYRILLIC_SMALL_LETTER_U_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0443 };
static wchar_t* CYRILLIC_SMALL_LETTER_U_UNICODE_CHARACTER_CODE_MODEL = CYRILLIC_SMALL_LETTER_U_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CYRILLIC_SMALL_LETTER_U_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t CYRILLIC_SMALL_LETTER_EF_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0444 };
static wchar_t* CYRILLIC_SMALL_LETTER_EF_UNICODE_CHARACTER_CODE_MODEL = CYRILLIC_SMALL_LETTER_EF_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CYRILLIC_SMALL_LETTER_EF_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t CYRILLIC_SMALL_LETTER_HA_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0445 };
static wchar_t* CYRILLIC_SMALL_LETTER_HA_UNICODE_CHARACTER_CODE_MODEL = CYRILLIC_SMALL_LETTER_HA_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CYRILLIC_SMALL_LETTER_HA_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t CYRILLIC_SMALL_LETTER_TSE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0446 };
static wchar_t* CYRILLIC_SMALL_LETTER_TSE_UNICODE_CHARACTER_CODE_MODEL = CYRILLIC_SMALL_LETTER_TSE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CYRILLIC_SMALL_LETTER_TSE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t CYRILLIC_SMALL_LETTER_CHE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0447 };
static wchar_t* CYRILLIC_SMALL_LETTER_CHE_UNICODE_CHARACTER_CODE_MODEL = CYRILLIC_SMALL_LETTER_CHE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CYRILLIC_SMALL_LETTER_CHE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t CYRILLIC_SMALL_LETTER_SHA_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0448 };
static wchar_t* CYRILLIC_SMALL_LETTER_SHA_UNICODE_CHARACTER_CODE_MODEL = CYRILLIC_SMALL_LETTER_SHA_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CYRILLIC_SMALL_LETTER_SHA_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t CYRILLIC_SMALL_LETTER_SHCHA_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0449 };
static wchar_t* CYRILLIC_SMALL_LETTER_SHCHA_UNICODE_CHARACTER_CODE_MODEL = CYRILLIC_SMALL_LETTER_SHCHA_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CYRILLIC_SMALL_LETTER_SHCHA_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t CYRILLIC_SMALL_LETTER_HARD_SIGN_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x044a };
static wchar_t* CYRILLIC_SMALL_LETTER_HARD_SIGN_UNICODE_CHARACTER_CODE_MODEL = CYRILLIC_SMALL_LETTER_HARD_SIGN_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CYRILLIC_SMALL_LETTER_HARD_SIGN_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t CYRILLIC_SMALL_LETTER_YERU_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x044b };
static wchar_t* CYRILLIC_SMALL_LETTER_YERU_UNICODE_CHARACTER_CODE_MODEL = CYRILLIC_SMALL_LETTER_YERU_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CYRILLIC_SMALL_LETTER_YERU_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t CYRILLIC_SMALL_LETTER_SOFT_SIGN_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x044c };
static wchar_t* CYRILLIC_SMALL_LETTER_SOFT_SIGN_UNICODE_CHARACTER_CODE_MODEL = CYRILLIC_SMALL_LETTER_SOFT_SIGN_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CYRILLIC_SMALL_LETTER_SOFT_SIGN_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t CYRILLIC_SMALL_LETTER_E_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x044d };
static wchar_t* CYRILLIC_SMALL_LETTER_E_UNICODE_CHARACTER_CODE_MODEL = CYRILLIC_SMALL_LETTER_E_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CYRILLIC_SMALL_LETTER_E_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t CYRILLIC_SMALL_LETTER_YU_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x044e };
static wchar_t* CYRILLIC_SMALL_LETTER_YU_UNICODE_CHARACTER_CODE_MODEL = CYRILLIC_SMALL_LETTER_YU_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CYRILLIC_SMALL_LETTER_YU_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t CYRILLIC_SMALL_LETTER_YA_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x044f };
static wchar_t* CYRILLIC_SMALL_LETTER_YA_UNICODE_CHARACTER_CODE_MODEL = CYRILLIC_SMALL_LETTER_YA_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CYRILLIC_SMALL_LETTER_YA_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t CYRILLIC_SMALL_LETTER_IO_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0451 };
static wchar_t* CYRILLIC_SMALL_LETTER_IO_UNICODE_CHARACTER_CODE_MODEL = CYRILLIC_SMALL_LETTER_IO_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CYRILLIC_SMALL_LETTER_IO_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t CYRILLIC_SMALL_LETTER_DJE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0452 };
static wchar_t* CYRILLIC_SMALL_LETTER_DJE_UNICODE_CHARACTER_CODE_MODEL = CYRILLIC_SMALL_LETTER_DJE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CYRILLIC_SMALL_LETTER_DJE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t CYRILLIC_SMALL_LETTER_GJE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0453 };
static wchar_t* CYRILLIC_SMALL_LETTER_GJE_UNICODE_CHARACTER_CODE_MODEL = CYRILLIC_SMALL_LETTER_GJE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CYRILLIC_SMALL_LETTER_GJE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t CYRILLIC_SMALL_LETTER_UKRAINIAN_IE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0454 };
static wchar_t* CYRILLIC_SMALL_LETTER_UKRAINIAN_IE_UNICODE_CHARACTER_CODE_MODEL = CYRILLIC_SMALL_LETTER_UKRAINIAN_IE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CYRILLIC_SMALL_LETTER_UKRAINIAN_IE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t CYRILLIC_SMALL_LETTER_DZE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0455 };
static wchar_t* CYRILLIC_SMALL_LETTER_DZE_UNICODE_CHARACTER_CODE_MODEL = CYRILLIC_SMALL_LETTER_DZE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CYRILLIC_SMALL_LETTER_DZE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t CYRILLIC_SMALL_LETTER_BYELORUSSIAN_UKRAINIAN_I_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0456 };
static wchar_t* CYRILLIC_SMALL_LETTER_BYELORUSSIAN_UKRAINIAN_I_UNICODE_CHARACTER_CODE_MODEL = CYRILLIC_SMALL_LETTER_BYELORUSSIAN_UKRAINIAN_I_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CYRILLIC_SMALL_LETTER_BYELORUSSIAN_UKRAINIAN_I_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t CYRILLIC_SMALL_LETTER_YI_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0457 };
static wchar_t* CYRILLIC_SMALL_LETTER_YI_UNICODE_CHARACTER_CODE_MODEL = CYRILLIC_SMALL_LETTER_YI_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CYRILLIC_SMALL_LETTER_YI_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t CYRILLIC_SMALL_LETTER_JE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0458 };
static wchar_t* CYRILLIC_SMALL_LETTER_JE_UNICODE_CHARACTER_CODE_MODEL = CYRILLIC_SMALL_LETTER_JE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CYRILLIC_SMALL_LETTER_JE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t CYRILLIC_SMALL_LETTER_LJE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x0459 };
static wchar_t* CYRILLIC_SMALL_LETTER_LJE_UNICODE_CHARACTER_CODE_MODEL = CYRILLIC_SMALL_LETTER_LJE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CYRILLIC_SMALL_LETTER_LJE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t CYRILLIC_SMALL_LETTER_NJE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x045a };
static wchar_t* CYRILLIC_SMALL_LETTER_NJE_UNICODE_CHARACTER_CODE_MODEL = CYRILLIC_SMALL_LETTER_NJE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CYRILLIC_SMALL_LETTER_NJE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t CYRILLIC_SMALL_LETTER_TSHE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x045b };
static wchar_t* CYRILLIC_SMALL_LETTER_TSHE_UNICODE_CHARACTER_CODE_MODEL = CYRILLIC_SMALL_LETTER_TSHE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CYRILLIC_SMALL_LETTER_TSHE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t CYRILLIC_SMALL_LETTER_KJE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x045c };
static wchar_t* CYRILLIC_SMALL_LETTER_KJE_UNICODE_CHARACTER_CODE_MODEL = CYRILLIC_SMALL_LETTER_KJE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CYRILLIC_SMALL_LETTER_KJE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t CYRILLIC_SMALL_LETTER_SHORT_U_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x045e };
static wchar_t* CYRILLIC_SMALL_LETTER_SHORT_U_UNICODE_CHARACTER_CODE_MODEL = CYRILLIC_SMALL_LETTER_SHORT_U_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CYRILLIC_SMALL_LETTER_SHORT_U_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t CYRILLIC_SMALL_LETTER_DZHE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x045f };
static wchar_t* CYRILLIC_SMALL_LETTER_DZHE_UNICODE_CHARACTER_CODE_MODEL = CYRILLIC_SMALL_LETTER_DZHE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CYRILLIC_SMALL_LETTER_DZHE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t EN_SPACE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2002 };
static wchar_t* EN_SPACE_UNICODE_CHARACTER_CODE_MODEL = EN_SPACE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* EN_SPACE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t EM_SPACE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2003 };
static wchar_t* EM_SPACE_UNICODE_CHARACTER_CODE_MODEL = EM_SPACE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* EM_SPACE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t THREE_PER_EM_SPACE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2004 };
static wchar_t* THREE_PER_EM_SPACE_UNICODE_CHARACTER_CODE_MODEL = THREE_PER_EM_SPACE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* THREE_PER_EM_SPACE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t FOUR_PER_EM_SPACE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2005 };
static wchar_t* FOUR_PER_EM_SPACE_UNICODE_CHARACTER_CODE_MODEL = FOUR_PER_EM_SPACE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* FOUR_PER_EM_SPACE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t FIGURE_SPACE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2007 };
static wchar_t* FIGURE_SPACE_UNICODE_CHARACTER_CODE_MODEL = FIGURE_SPACE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* FIGURE_SPACE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t PUNCTUATION_SPACE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2008 };
static wchar_t* PUNCTUATION_SPACE_UNICODE_CHARACTER_CODE_MODEL = PUNCTUATION_SPACE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* PUNCTUATION_SPACE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t THIN_SPACE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2009 };
static wchar_t* THIN_SPACE_UNICODE_CHARACTER_CODE_MODEL = THIN_SPACE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* THIN_SPACE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t HAIR_SPACE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x200a };
static wchar_t* HAIR_SPACE_UNICODE_CHARACTER_CODE_MODEL = HAIR_SPACE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* HAIR_SPACE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t ZERO_WIDTH_SPACE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x200b };
static wchar_t* ZERO_WIDTH_SPACE_UNICODE_CHARACTER_CODE_MODEL = ZERO_WIDTH_SPACE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* ZERO_WIDTH_SPACE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t ZERO_WIDTH_NON_JOINER_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x200c };
static wchar_t* ZERO_WIDTH_NON_JOINER_UNICODE_CHARACTER_CODE_MODEL = ZERO_WIDTH_NON_JOINER_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* ZERO_WIDTH_NON_JOINER_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t ZERO_WIDTH_JOINER_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x200d };
static wchar_t* ZERO_WIDTH_JOINER_UNICODE_CHARACTER_CODE_MODEL = ZERO_WIDTH_JOINER_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* ZERO_WIDTH_JOINER_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LEFT_TO_RIGHT_MARK_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x200e };
static wchar_t* LEFT_TO_RIGHT_MARK_UNICODE_CHARACTER_CODE_MODEL = LEFT_TO_RIGHT_MARK_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LEFT_TO_RIGHT_MARK_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t RIGHT_TO_LEFT_MARK_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x200f };
static wchar_t* RIGHT_TO_LEFT_MARK_UNICODE_CHARACTER_CODE_MODEL = RIGHT_TO_LEFT_MARK_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* RIGHT_TO_LEFT_MARK_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t HYPHEN_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2010 };
static wchar_t* HYPHEN_UNICODE_CHARACTER_CODE_MODEL = HYPHEN_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* HYPHEN_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t EN_DASH_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2013 };
static wchar_t* EN_DASH_UNICODE_CHARACTER_CODE_MODEL = EN_DASH_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* EN_DASH_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t EM_DASH_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2014 };
static wchar_t* EM_DASH_UNICODE_CHARACTER_CODE_MODEL = EM_DASH_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* EM_DASH_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t HORIZONTAL_BAR_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2015 };
static wchar_t* HORIZONTAL_BAR_UNICODE_CHARACTER_CODE_MODEL = HORIZONTAL_BAR_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* HORIZONTAL_BAR_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t DOUBLE_VERTICAL_LINE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2016 };
static wchar_t* DOUBLE_VERTICAL_LINE_UNICODE_CHARACTER_CODE_MODEL = DOUBLE_VERTICAL_LINE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* DOUBLE_VERTICAL_LINE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t DOUBLE_LOW_LINE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2017 };
static wchar_t* DOUBLE_LOW_LINE_UNICODE_CHARACTER_CODE_MODEL = DOUBLE_LOW_LINE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* DOUBLE_LOW_LINE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LEFT_SINGLE_QUOTATION_MARK_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2018 };
static wchar_t* LEFT_SINGLE_QUOTATION_MARK_UNICODE_CHARACTER_CODE_MODEL = LEFT_SINGLE_QUOTATION_MARK_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LEFT_SINGLE_QUOTATION_MARK_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t RIGHT_SINGLE_QUOTATION_MARK_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2019 };
static wchar_t* RIGHT_SINGLE_QUOTATION_MARK_UNICODE_CHARACTER_CODE_MODEL = RIGHT_SINGLE_QUOTATION_MARK_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* RIGHT_SINGLE_QUOTATION_MARK_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t SINGLE_LOW_9_QUOTATION_MARK_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x201a };
static wchar_t* SINGLE_LOW_9_QUOTATION_MARK_UNICODE_CHARACTER_CODE_MODEL = SINGLE_LOW_9_QUOTATION_MARK_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* SINGLE_LOW_9_QUOTATION_MARK_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LEFT_DOUBLE_QUOTATION_MARK_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x201c };
static wchar_t* LEFT_DOUBLE_QUOTATION_MARK_UNICODE_CHARACTER_CODE_MODEL = LEFT_DOUBLE_QUOTATION_MARK_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LEFT_DOUBLE_QUOTATION_MARK_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t RIGHT_DOUBLE_QUOTATION_MARK_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x201d };
static wchar_t* RIGHT_DOUBLE_QUOTATION_MARK_UNICODE_CHARACTER_CODE_MODEL = RIGHT_DOUBLE_QUOTATION_MARK_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* RIGHT_DOUBLE_QUOTATION_MARK_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t DOUBLE_LOW_9_QUOTATION_MARK_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x201e };
static wchar_t* DOUBLE_LOW_9_QUOTATION_MARK_UNICODE_CHARACTER_CODE_MODEL = DOUBLE_LOW_9_QUOTATION_MARK_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* DOUBLE_LOW_9_QUOTATION_MARK_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t DAGGER_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2020 };
static wchar_t* DAGGER_UNICODE_CHARACTER_CODE_MODEL = DAGGER_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* DAGGER_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t DOUBLE_DAGGER_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2021 };
static wchar_t* DOUBLE_DAGGER_UNICODE_CHARACTER_CODE_MODEL = DOUBLE_DAGGER_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* DOUBLE_DAGGER_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t BULLET_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2022 };
static wchar_t* BULLET_UNICODE_CHARACTER_CODE_MODEL = BULLET_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* BULLET_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t TWO_DOT_LEADER_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2025 };
static wchar_t* TWO_DOT_LEADER_UNICODE_CHARACTER_CODE_MODEL = TWO_DOT_LEADER_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* TWO_DOT_LEADER_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t HORIZONTAL_ELLIPSIS_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2026 };
static wchar_t* HORIZONTAL_ELLIPSIS_UNICODE_CHARACTER_CODE_MODEL = HORIZONTAL_ELLIPSIS_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* HORIZONTAL_ELLIPSIS_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t PER_MILLE_SIGN_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2030 };
static wchar_t* PER_MILLE_SIGN_UNICODE_CHARACTER_CODE_MODEL = PER_MILLE_SIGN_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* PER_MILLE_SIGN_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t PER_TEN_THOUSAND_SIGN_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2031 };
static wchar_t* PER_TEN_THOUSAND_SIGN_UNICODE_CHARACTER_CODE_MODEL = PER_TEN_THOUSAND_SIGN_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* PER_TEN_THOUSAND_SIGN_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t PRIME_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2032 };
static wchar_t* PRIME_UNICODE_CHARACTER_CODE_MODEL = PRIME_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* PRIME_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t DOUBLE_PRIME_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2033 };
static wchar_t* DOUBLE_PRIME_UNICODE_CHARACTER_CODE_MODEL = DOUBLE_PRIME_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* DOUBLE_PRIME_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t TRIPLE_PRIME_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2034 };
static wchar_t* TRIPLE_PRIME_UNICODE_CHARACTER_CODE_MODEL = TRIPLE_PRIME_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* TRIPLE_PRIME_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t REVERSED_PRIME_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2035 };
static wchar_t* REVERSED_PRIME_UNICODE_CHARACTER_CODE_MODEL = REVERSED_PRIME_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* REVERSED_PRIME_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t SINGLE_LEFT_POINTING_ANGLE_QUOTATION_MARK_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2039 };
static wchar_t* SINGLE_LEFT_POINTING_ANGLE_QUOTATION_MARK_UNICODE_CHARACTER_CODE_MODEL = SINGLE_LEFT_POINTING_ANGLE_QUOTATION_MARK_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* SINGLE_LEFT_POINTING_ANGLE_QUOTATION_MARK_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t SINGLE_RIGHT_POINTING_ANGLE_QUOTATION_MARK_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x203a };
static wchar_t* SINGLE_RIGHT_POINTING_ANGLE_QUOTATION_MARK_UNICODE_CHARACTER_CODE_MODEL = SINGLE_RIGHT_POINTING_ANGLE_QUOTATION_MARK_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* SINGLE_RIGHT_POINTING_ANGLE_QUOTATION_MARK_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t OVERLINE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x203e };
static wchar_t* OVERLINE_UNICODE_CHARACTER_CODE_MODEL = OVERLINE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* OVERLINE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t CARET_INSERTION_POINT_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2041 };
static wchar_t* CARET_INSERTION_POINT_UNICODE_CHARACTER_CODE_MODEL = CARET_INSERTION_POINT_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CARET_INSERTION_POINT_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t HYPHEN_BULLET_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2043 };
static wchar_t* HYPHEN_BULLET_UNICODE_CHARACTER_CODE_MODEL = HYPHEN_BULLET_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* HYPHEN_BULLET_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t FRACTION_SLASH_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2044 };
static wchar_t* FRACTION_SLASH_UNICODE_CHARACTER_CODE_MODEL = FRACTION_SLASH_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* FRACTION_SLASH_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t REVERSED_SEMICOLON_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x204f };
static wchar_t* REVERSED_SEMICOLON_UNICODE_CHARACTER_CODE_MODEL = REVERSED_SEMICOLON_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* REVERSED_SEMICOLON_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t QUADRUPLE_PRIME_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2057 };
static wchar_t* QUADRUPLE_PRIME_UNICODE_CHARACTER_CODE_MODEL = QUADRUPLE_PRIME_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* QUADRUPLE_PRIME_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MEDIUM_MATHEMATICAL_SPACE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x205f };
static wchar_t* MEDIUM_MATHEMATICAL_SPACE_UNICODE_CHARACTER_CODE_MODEL = MEDIUM_MATHEMATICAL_SPACE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MEDIUM_MATHEMATICAL_SPACE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t SPACE_OF_WIDTH_5_18_EM_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x205f, 0x200a };
static wchar_t* SPACE_OF_WIDTH_5_18_EM_UNICODE_CHARACTER_CODE_MODEL = SPACE_OF_WIDTH_5_18_EM_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* SPACE_OF_WIDTH_5_18_EM_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_2_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t WORD_JOINER_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2060 };
static wchar_t* WORD_JOINER_UNICODE_CHARACTER_CODE_MODEL = WORD_JOINER_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* WORD_JOINER_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t FUNCTION_APPLICATION_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2061 };
static wchar_t* FUNCTION_APPLICATION_UNICODE_CHARACTER_CODE_MODEL = FUNCTION_APPLICATION_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* FUNCTION_APPLICATION_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t INVISIBLE_TIMES_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2062 };
static wchar_t* INVISIBLE_TIMES_UNICODE_CHARACTER_CODE_MODEL = INVISIBLE_TIMES_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* INVISIBLE_TIMES_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t INVISIBLE_SEPARATOR_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2063 };
static wchar_t* INVISIBLE_SEPARATOR_UNICODE_CHARACTER_CODE_MODEL = INVISIBLE_SEPARATOR_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* INVISIBLE_SEPARATOR_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t SUPERSCRIPT_LATIN_SMALL_LETTER_N_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x207f };
static wchar_t* SUPERSCRIPT_LATIN_SMALL_LETTER_N_UNICODE_CHARACTER_CODE_MODEL = SUPERSCRIPT_LATIN_SMALL_LETTER_N_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* SUPERSCRIPT_LATIN_SMALL_LETTER_N_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t PESETA_SIGN_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x20a7 };
static wchar_t* PESETA_SIGN_UNICODE_CHARACTER_CODE_MODEL = PESETA_SIGN_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* PESETA_SIGN_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t EURO_SIGN_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x20ac };
static wchar_t* EURO_SIGN_UNICODE_CHARACTER_CODE_MODEL = EURO_SIGN_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* EURO_SIGN_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t COMBINING_THREE_DOTS_ABOVE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x20db };
static wchar_t* COMBINING_THREE_DOTS_ABOVE_UNICODE_CHARACTER_CODE_MODEL = COMBINING_THREE_DOTS_ABOVE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* COMBINING_THREE_DOTS_ABOVE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t COMBINING_FOUR_DOTS_ABOVE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x20dc };
static wchar_t* COMBINING_FOUR_DOTS_ABOVE_UNICODE_CHARACTER_CODE_MODEL = COMBINING_FOUR_DOTS_ABOVE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* COMBINING_FOUR_DOTS_ABOVE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t DOUBLE_STRUCK_CAPITAL_C_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2102 };
static wchar_t* DOUBLE_STRUCK_CAPITAL_C_UNICODE_CHARACTER_CODE_MODEL = DOUBLE_STRUCK_CAPITAL_C_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* DOUBLE_STRUCK_CAPITAL_C_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t CARE_OF_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2105 };
static wchar_t* CARE_OF_UNICODE_CHARACTER_CODE_MODEL = CARE_OF_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CARE_OF_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t SCRIPT_SMALL_G_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x210a };
static wchar_t* SCRIPT_SMALL_G_UNICODE_CHARACTER_CODE_MODEL = SCRIPT_SMALL_G_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* SCRIPT_SMALL_G_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t SCRIPT_CAPITAL_H_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x210b };
static wchar_t* SCRIPT_CAPITAL_H_UNICODE_CHARACTER_CODE_MODEL = SCRIPT_CAPITAL_H_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* SCRIPT_CAPITAL_H_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t BLACK_LETTER_CAPITAL_H_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x210c };
static wchar_t* BLACK_LETTER_CAPITAL_H_UNICODE_CHARACTER_CODE_MODEL = BLACK_LETTER_CAPITAL_H_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* BLACK_LETTER_CAPITAL_H_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t DOUBLE_STRUCK_CAPITAL_H_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x210d };
static wchar_t* DOUBLE_STRUCK_CAPITAL_H_UNICODE_CHARACTER_CODE_MODEL = DOUBLE_STRUCK_CAPITAL_H_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* DOUBLE_STRUCK_CAPITAL_H_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t PLANCK_CONSTANT_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x210e };
static wchar_t* PLANCK_CONSTANT_UNICODE_CHARACTER_CODE_MODEL = PLANCK_CONSTANT_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* PLANCK_CONSTANT_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t PLANCK_CONSTANT_OVER_TWO_PI_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x210f };
static wchar_t* PLANCK_CONSTANT_OVER_TWO_PI_UNICODE_CHARACTER_CODE_MODEL = PLANCK_CONSTANT_OVER_TWO_PI_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* PLANCK_CONSTANT_OVER_TWO_PI_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t SCRIPT_CAPITAL_I_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2110 };
static wchar_t* SCRIPT_CAPITAL_I_UNICODE_CHARACTER_CODE_MODEL = SCRIPT_CAPITAL_I_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* SCRIPT_CAPITAL_I_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t BLACK_LETTER_CAPITAL_I_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2111 };
static wchar_t* BLACK_LETTER_CAPITAL_I_UNICODE_CHARACTER_CODE_MODEL = BLACK_LETTER_CAPITAL_I_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* BLACK_LETTER_CAPITAL_I_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t SCRIPT_CAPITAL_L_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2112 };
static wchar_t* SCRIPT_CAPITAL_L_UNICODE_CHARACTER_CODE_MODEL = SCRIPT_CAPITAL_L_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* SCRIPT_CAPITAL_L_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t SCRIPT_SMALL_L_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2113 };
static wchar_t* SCRIPT_SMALL_L_UNICODE_CHARACTER_CODE_MODEL = SCRIPT_SMALL_L_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* SCRIPT_SMALL_L_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t DOUBLE_STRUCK_CAPITAL_N_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2115 };
static wchar_t* DOUBLE_STRUCK_CAPITAL_N_UNICODE_CHARACTER_CODE_MODEL = DOUBLE_STRUCK_CAPITAL_N_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* DOUBLE_STRUCK_CAPITAL_N_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t NUMERO_SIGN_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2116 };
static wchar_t* NUMERO_SIGN_UNICODE_CHARACTER_CODE_MODEL = NUMERO_SIGN_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* NUMERO_SIGN_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t SOUND_RECORDING_COPYRIGHT_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2117 };
static wchar_t* SOUND_RECORDING_COPYRIGHT_UNICODE_CHARACTER_CODE_MODEL = SOUND_RECORDING_COPYRIGHT_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* SOUND_RECORDING_COPYRIGHT_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t SCRIPT_CAPITAL_P_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2118 };
static wchar_t* SCRIPT_CAPITAL_P_UNICODE_CHARACTER_CODE_MODEL = SCRIPT_CAPITAL_P_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* SCRIPT_CAPITAL_P_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t DOUBLE_STRUCK_CAPITAL_P_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2119 };
static wchar_t* DOUBLE_STRUCK_CAPITAL_P_UNICODE_CHARACTER_CODE_MODEL = DOUBLE_STRUCK_CAPITAL_P_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* DOUBLE_STRUCK_CAPITAL_P_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t DOUBLE_STRUCK_CAPITAL_Q_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x211a };
static wchar_t* DOUBLE_STRUCK_CAPITAL_Q_UNICODE_CHARACTER_CODE_MODEL = DOUBLE_STRUCK_CAPITAL_Q_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* DOUBLE_STRUCK_CAPITAL_Q_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t SCRIPT_CAPITAL_R_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x211b };
static wchar_t* SCRIPT_CAPITAL_R_UNICODE_CHARACTER_CODE_MODEL = SCRIPT_CAPITAL_R_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* SCRIPT_CAPITAL_R_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t BLACK_LETTER_CAPITAL_R_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x211c };
static wchar_t* BLACK_LETTER_CAPITAL_R_UNICODE_CHARACTER_CODE_MODEL = BLACK_LETTER_CAPITAL_R_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* BLACK_LETTER_CAPITAL_R_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t DOUBLE_STRUCK_CAPITAL_R_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x211d };
static wchar_t* DOUBLE_STRUCK_CAPITAL_R_UNICODE_CHARACTER_CODE_MODEL = DOUBLE_STRUCK_CAPITAL_R_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* DOUBLE_STRUCK_CAPITAL_R_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t PRESCRIPTION_TAKE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x211e };
static wchar_t* PRESCRIPTION_TAKE_UNICODE_CHARACTER_CODE_MODEL = PRESCRIPTION_TAKE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* PRESCRIPTION_TAKE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t TRADE_MARK_SIGN_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2122 };
static wchar_t* TRADE_MARK_SIGN_UNICODE_CHARACTER_CODE_MODEL = TRADE_MARK_SIGN_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* TRADE_MARK_SIGN_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t DOUBLE_STRUCK_CAPITAL_Z_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2124 };
static wchar_t* DOUBLE_STRUCK_CAPITAL_Z_UNICODE_CHARACTER_CODE_MODEL = DOUBLE_STRUCK_CAPITAL_Z_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* DOUBLE_STRUCK_CAPITAL_Z_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t INVERTED_OHM_SIGN_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2127 };
static wchar_t* INVERTED_OHM_SIGN_UNICODE_CHARACTER_CODE_MODEL = INVERTED_OHM_SIGN_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* INVERTED_OHM_SIGN_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t BLACK_LETTER_CAPITAL_Z_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2128 };
static wchar_t* BLACK_LETTER_CAPITAL_Z_UNICODE_CHARACTER_CODE_MODEL = BLACK_LETTER_CAPITAL_Z_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* BLACK_LETTER_CAPITAL_Z_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t TURNED_GREEK_SMALL_LETTER_IOTA_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2129 };
static wchar_t* TURNED_GREEK_SMALL_LETTER_IOTA_UNICODE_CHARACTER_CODE_MODEL = TURNED_GREEK_SMALL_LETTER_IOTA_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* TURNED_GREEK_SMALL_LETTER_IOTA_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t SCRIPT_CAPITAL_B_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x212c };
static wchar_t* SCRIPT_CAPITAL_B_UNICODE_CHARACTER_CODE_MODEL = SCRIPT_CAPITAL_B_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* SCRIPT_CAPITAL_B_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t BLACK_LETTER_CAPITAL_C_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x212d };
static wchar_t* BLACK_LETTER_CAPITAL_C_UNICODE_CHARACTER_CODE_MODEL = BLACK_LETTER_CAPITAL_C_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* BLACK_LETTER_CAPITAL_C_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t SCRIPT_SMALL_E_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x212f };
static wchar_t* SCRIPT_SMALL_E_UNICODE_CHARACTER_CODE_MODEL = SCRIPT_SMALL_E_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* SCRIPT_SMALL_E_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t SCRIPT_CAPITAL_E_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2130 };
static wchar_t* SCRIPT_CAPITAL_E_UNICODE_CHARACTER_CODE_MODEL = SCRIPT_CAPITAL_E_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* SCRIPT_CAPITAL_E_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t SCRIPT_CAPITAL_F_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2131 };
static wchar_t* SCRIPT_CAPITAL_F_UNICODE_CHARACTER_CODE_MODEL = SCRIPT_CAPITAL_F_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* SCRIPT_CAPITAL_F_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t SCRIPT_CAPITAL_M_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2133 };
static wchar_t* SCRIPT_CAPITAL_M_UNICODE_CHARACTER_CODE_MODEL = SCRIPT_CAPITAL_M_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* SCRIPT_CAPITAL_M_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t SCRIPT_SMALL_O_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2134 };
static wchar_t* SCRIPT_SMALL_O_UNICODE_CHARACTER_CODE_MODEL = SCRIPT_SMALL_O_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* SCRIPT_SMALL_O_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t ALEF_SYMBOL_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2135 };
static wchar_t* ALEF_SYMBOL_UNICODE_CHARACTER_CODE_MODEL = ALEF_SYMBOL_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* ALEF_SYMBOL_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t BET_SYMBOL_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2136 };
static wchar_t* BET_SYMBOL_UNICODE_CHARACTER_CODE_MODEL = BET_SYMBOL_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* BET_SYMBOL_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t GIMEL_SYMBOL_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2137 };
static wchar_t* GIMEL_SYMBOL_UNICODE_CHARACTER_CODE_MODEL = GIMEL_SYMBOL_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* GIMEL_SYMBOL_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t DALET_SYMBOL_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2138 };
static wchar_t* DALET_SYMBOL_UNICODE_CHARACTER_CODE_MODEL = DALET_SYMBOL_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* DALET_SYMBOL_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t DOUBLE_STRUCK_ITALIC_CAPITAL_D_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2145 };
static wchar_t* DOUBLE_STRUCK_ITALIC_CAPITAL_D_UNICODE_CHARACTER_CODE_MODEL = DOUBLE_STRUCK_ITALIC_CAPITAL_D_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* DOUBLE_STRUCK_ITALIC_CAPITAL_D_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t DOUBLE_STRUCK_ITALIC_SMALL_D_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2146 };
static wchar_t* DOUBLE_STRUCK_ITALIC_SMALL_D_UNICODE_CHARACTER_CODE_MODEL = DOUBLE_STRUCK_ITALIC_SMALL_D_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* DOUBLE_STRUCK_ITALIC_SMALL_D_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t DOUBLE_STRUCK_ITALIC_SMALL_E_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2147 };
static wchar_t* DOUBLE_STRUCK_ITALIC_SMALL_E_UNICODE_CHARACTER_CODE_MODEL = DOUBLE_STRUCK_ITALIC_SMALL_E_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* DOUBLE_STRUCK_ITALIC_SMALL_E_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t DOUBLE_STRUCK_ITALIC_SMALL_I_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2148 };
static wchar_t* DOUBLE_STRUCK_ITALIC_SMALL_I_UNICODE_CHARACTER_CODE_MODEL = DOUBLE_STRUCK_ITALIC_SMALL_I_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* DOUBLE_STRUCK_ITALIC_SMALL_I_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t VULGAR_FRACTION_ONE_THIRD_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2153 };
static wchar_t* VULGAR_FRACTION_ONE_THIRD_UNICODE_CHARACTER_CODE_MODEL = VULGAR_FRACTION_ONE_THIRD_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* VULGAR_FRACTION_ONE_THIRD_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t VULGAR_FRACTION_TWO_THIRDS_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2154 };
static wchar_t* VULGAR_FRACTION_TWO_THIRDS_UNICODE_CHARACTER_CODE_MODEL = VULGAR_FRACTION_TWO_THIRDS_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* VULGAR_FRACTION_TWO_THIRDS_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t VULGAR_FRACTION_ONE_FIFTH_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2155 };
static wchar_t* VULGAR_FRACTION_ONE_FIFTH_UNICODE_CHARACTER_CODE_MODEL = VULGAR_FRACTION_ONE_FIFTH_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* VULGAR_FRACTION_ONE_FIFTH_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t VULGAR_FRACTION_TWO_FIFTHS_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2156 };
static wchar_t* VULGAR_FRACTION_TWO_FIFTHS_UNICODE_CHARACTER_CODE_MODEL = VULGAR_FRACTION_TWO_FIFTHS_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* VULGAR_FRACTION_TWO_FIFTHS_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t VULGAR_FRACTION_THREE_FIFTHS_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2157 };
static wchar_t* VULGAR_FRACTION_THREE_FIFTHS_UNICODE_CHARACTER_CODE_MODEL = VULGAR_FRACTION_THREE_FIFTHS_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* VULGAR_FRACTION_THREE_FIFTHS_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t VULGAR_FRACTION_FOUR_FIFTHS_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2158 };
static wchar_t* VULGAR_FRACTION_FOUR_FIFTHS_UNICODE_CHARACTER_CODE_MODEL = VULGAR_FRACTION_FOUR_FIFTHS_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* VULGAR_FRACTION_FOUR_FIFTHS_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t VULGAR_FRACTION_ONE_SIXTH_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2159 };
static wchar_t* VULGAR_FRACTION_ONE_SIXTH_UNICODE_CHARACTER_CODE_MODEL = VULGAR_FRACTION_ONE_SIXTH_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* VULGAR_FRACTION_ONE_SIXTH_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t VULGAR_FRACTION_FIVE_SIXTHS_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x215a };
static wchar_t* VULGAR_FRACTION_FIVE_SIXTHS_UNICODE_CHARACTER_CODE_MODEL = VULGAR_FRACTION_FIVE_SIXTHS_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* VULGAR_FRACTION_FIVE_SIXTHS_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t VULGAR_FRACTION_ONE_EIGHTH_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x215b };
static wchar_t* VULGAR_FRACTION_ONE_EIGHTH_UNICODE_CHARACTER_CODE_MODEL = VULGAR_FRACTION_ONE_EIGHTH_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* VULGAR_FRACTION_ONE_EIGHTH_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t VULGAR_FRACTION_THREE_EIGHTHS_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x215c };
static wchar_t* VULGAR_FRACTION_THREE_EIGHTHS_UNICODE_CHARACTER_CODE_MODEL = VULGAR_FRACTION_THREE_EIGHTHS_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* VULGAR_FRACTION_THREE_EIGHTHS_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t VULGAR_FRACTION_FIVE_EIGHTHS_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x215d };
static wchar_t* VULGAR_FRACTION_FIVE_EIGHTHS_UNICODE_CHARACTER_CODE_MODEL = VULGAR_FRACTION_FIVE_EIGHTHS_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* VULGAR_FRACTION_FIVE_EIGHTHS_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t VULGAR_FRACTION_SEVEN_EIGHTHS_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x215e };
static wchar_t* VULGAR_FRACTION_SEVEN_EIGHTHS_UNICODE_CHARACTER_CODE_MODEL = VULGAR_FRACTION_SEVEN_EIGHTHS_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* VULGAR_FRACTION_SEVEN_EIGHTHS_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LEFTWARDS_ARROW_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2190 };
static wchar_t* LEFTWARDS_ARROW_UNICODE_CHARACTER_CODE_MODEL = LEFTWARDS_ARROW_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LEFTWARDS_ARROW_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t UPWARDS_ARROW_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2191 };
static wchar_t* UPWARDS_ARROW_UNICODE_CHARACTER_CODE_MODEL = UPWARDS_ARROW_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* UPWARDS_ARROW_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t RIGHTWARDS_ARROW_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2192 };
static wchar_t* RIGHTWARDS_ARROW_UNICODE_CHARACTER_CODE_MODEL = RIGHTWARDS_ARROW_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* RIGHTWARDS_ARROW_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t DOWNWARDS_ARROW_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2193 };
static wchar_t* DOWNWARDS_ARROW_UNICODE_CHARACTER_CODE_MODEL = DOWNWARDS_ARROW_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* DOWNWARDS_ARROW_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LEFT_RIGHT_ARROW_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2194 };
static wchar_t* LEFT_RIGHT_ARROW_UNICODE_CHARACTER_CODE_MODEL = LEFT_RIGHT_ARROW_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LEFT_RIGHT_ARROW_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t UP_DOWN_ARROW_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2195 };
static wchar_t* UP_DOWN_ARROW_UNICODE_CHARACTER_CODE_MODEL = UP_DOWN_ARROW_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* UP_DOWN_ARROW_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t NORTH_WEST_ARROW_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2196 };
static wchar_t* NORTH_WEST_ARROW_UNICODE_CHARACTER_CODE_MODEL = NORTH_WEST_ARROW_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* NORTH_WEST_ARROW_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t NORTH_EAST_ARROW_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2197 };
static wchar_t* NORTH_EAST_ARROW_UNICODE_CHARACTER_CODE_MODEL = NORTH_EAST_ARROW_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* NORTH_EAST_ARROW_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t SOUTH_EAST_ARROW_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2198 };
static wchar_t* SOUTH_EAST_ARROW_UNICODE_CHARACTER_CODE_MODEL = SOUTH_EAST_ARROW_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* SOUTH_EAST_ARROW_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t SOUTH_WEST_ARROW_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2199 };
static wchar_t* SOUTH_WEST_ARROW_UNICODE_CHARACTER_CODE_MODEL = SOUTH_WEST_ARROW_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* SOUTH_WEST_ARROW_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LEFTWARDS_ARROW_WITH_STROKE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x219a };
static wchar_t* LEFTWARDS_ARROW_WITH_STROKE_UNICODE_CHARACTER_CODE_MODEL = LEFTWARDS_ARROW_WITH_STROKE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LEFTWARDS_ARROW_WITH_STROKE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t RIGHTWARDS_ARROW_WITH_STROKE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x219b };
static wchar_t* RIGHTWARDS_ARROW_WITH_STROKE_UNICODE_CHARACTER_CODE_MODEL = RIGHTWARDS_ARROW_WITH_STROKE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* RIGHTWARDS_ARROW_WITH_STROKE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t RIGHTWARDS_WAVE_ARROW_WITH_SLASH_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x219d, 0x0338 };
static wchar_t* RIGHTWARDS_WAVE_ARROW_WITH_SLASH_UNICODE_CHARACTER_CODE_MODEL = RIGHTWARDS_WAVE_ARROW_WITH_SLASH_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* RIGHTWARDS_WAVE_ARROW_WITH_SLASH_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_2_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t RIGHTWARDS_WAVE_ARROW_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x219d };
static wchar_t* RIGHTWARDS_WAVE_ARROW_UNICODE_CHARACTER_CODE_MODEL = RIGHTWARDS_WAVE_ARROW_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* RIGHTWARDS_WAVE_ARROW_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LEFTWARDS_TWO_HEADED_ARROW_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x219e };
static wchar_t* LEFTWARDS_TWO_HEADED_ARROW_UNICODE_CHARACTER_CODE_MODEL = LEFTWARDS_TWO_HEADED_ARROW_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LEFTWARDS_TWO_HEADED_ARROW_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t UPWARDS_TWO_HEADED_ARROW_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x219f };
static wchar_t* UPWARDS_TWO_HEADED_ARROW_UNICODE_CHARACTER_CODE_MODEL = UPWARDS_TWO_HEADED_ARROW_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* UPWARDS_TWO_HEADED_ARROW_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t RIGHTWARDS_TWO_HEADED_ARROW_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x21a0 };
static wchar_t* RIGHTWARDS_TWO_HEADED_ARROW_UNICODE_CHARACTER_CODE_MODEL = RIGHTWARDS_TWO_HEADED_ARROW_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* RIGHTWARDS_TWO_HEADED_ARROW_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t DOWNWARDS_TWO_HEADED_ARROW_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x21a1 };
static wchar_t* DOWNWARDS_TWO_HEADED_ARROW_UNICODE_CHARACTER_CODE_MODEL = DOWNWARDS_TWO_HEADED_ARROW_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* DOWNWARDS_TWO_HEADED_ARROW_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LEFTWARDS_ARROW_WITH_TAIL_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x21a2 };
static wchar_t* LEFTWARDS_ARROW_WITH_TAIL_UNICODE_CHARACTER_CODE_MODEL = LEFTWARDS_ARROW_WITH_TAIL_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LEFTWARDS_ARROW_WITH_TAIL_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t RIGHTWARDS_ARROW_WITH_TAIL_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x21a3 };
static wchar_t* RIGHTWARDS_ARROW_WITH_TAIL_UNICODE_CHARACTER_CODE_MODEL = RIGHTWARDS_ARROW_WITH_TAIL_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* RIGHTWARDS_ARROW_WITH_TAIL_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LEFTWARDS_ARROW_FROM_BAR_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x21a4 };
static wchar_t* LEFTWARDS_ARROW_FROM_BAR_UNICODE_CHARACTER_CODE_MODEL = LEFTWARDS_ARROW_FROM_BAR_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LEFTWARDS_ARROW_FROM_BAR_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t UPWARDS_ARROW_FROM_BAR_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x21a5 };
static wchar_t* UPWARDS_ARROW_FROM_BAR_UNICODE_CHARACTER_CODE_MODEL = UPWARDS_ARROW_FROM_BAR_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* UPWARDS_ARROW_FROM_BAR_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t RIGHTWARDS_ARROW_FROM_BAR_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x21a6 };
static wchar_t* RIGHTWARDS_ARROW_FROM_BAR_UNICODE_CHARACTER_CODE_MODEL = RIGHTWARDS_ARROW_FROM_BAR_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* RIGHTWARDS_ARROW_FROM_BAR_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t DOWNWARDS_ARROW_FROM_BAR_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x21a7 };
static wchar_t* DOWNWARDS_ARROW_FROM_BAR_UNICODE_CHARACTER_CODE_MODEL = DOWNWARDS_ARROW_FROM_BAR_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* DOWNWARDS_ARROW_FROM_BAR_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LEFTWARDS_ARROW_WITH_HOOK_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x21a9 };
static wchar_t* LEFTWARDS_ARROW_WITH_HOOK_UNICODE_CHARACTER_CODE_MODEL = LEFTWARDS_ARROW_WITH_HOOK_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LEFTWARDS_ARROW_WITH_HOOK_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t RIGHTWARDS_ARROW_WITH_HOOK_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x21aa };
static wchar_t* RIGHTWARDS_ARROW_WITH_HOOK_UNICODE_CHARACTER_CODE_MODEL = RIGHTWARDS_ARROW_WITH_HOOK_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* RIGHTWARDS_ARROW_WITH_HOOK_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LEFTWARDS_ARROW_WITH_LOOP_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x21ab };
static wchar_t* LEFTWARDS_ARROW_WITH_LOOP_UNICODE_CHARACTER_CODE_MODEL = LEFTWARDS_ARROW_WITH_LOOP_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LEFTWARDS_ARROW_WITH_LOOP_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t RIGHTWARDS_ARROW_WITH_LOOP_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x21ac };
static wchar_t* RIGHTWARDS_ARROW_WITH_LOOP_UNICODE_CHARACTER_CODE_MODEL = RIGHTWARDS_ARROW_WITH_LOOP_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* RIGHTWARDS_ARROW_WITH_LOOP_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LEFT_RIGHT_WAVE_ARROW_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x21ad };
static wchar_t* LEFT_RIGHT_WAVE_ARROW_UNICODE_CHARACTER_CODE_MODEL = LEFT_RIGHT_WAVE_ARROW_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LEFT_RIGHT_WAVE_ARROW_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LEFT_RIGHT_ARROW_WITH_STROKE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x21ae };
static wchar_t* LEFT_RIGHT_ARROW_WITH_STROKE_UNICODE_CHARACTER_CODE_MODEL = LEFT_RIGHT_ARROW_WITH_STROKE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LEFT_RIGHT_ARROW_WITH_STROKE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t UPWARDS_ARROW_WITH_TIP_LEFTWARDS_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x21b0 };
static wchar_t* UPWARDS_ARROW_WITH_TIP_LEFTWARDS_UNICODE_CHARACTER_CODE_MODEL = UPWARDS_ARROW_WITH_TIP_LEFTWARDS_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* UPWARDS_ARROW_WITH_TIP_LEFTWARDS_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t UPWARDS_ARROW_WITH_TIP_RIGHTWARDS_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x21b1 };
static wchar_t* UPWARDS_ARROW_WITH_TIP_RIGHTWARDS_UNICODE_CHARACTER_CODE_MODEL = UPWARDS_ARROW_WITH_TIP_RIGHTWARDS_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* UPWARDS_ARROW_WITH_TIP_RIGHTWARDS_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t DOWNWARDS_ARROW_WITH_TIP_LEFTWARDS_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x21b2 };
static wchar_t* DOWNWARDS_ARROW_WITH_TIP_LEFTWARDS_UNICODE_CHARACTER_CODE_MODEL = DOWNWARDS_ARROW_WITH_TIP_LEFTWARDS_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* DOWNWARDS_ARROW_WITH_TIP_LEFTWARDS_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t DOWNWARDS_ARROW_WITH_TIP_RIGHTWARDS_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x21b3 };
static wchar_t* DOWNWARDS_ARROW_WITH_TIP_RIGHTWARDS_UNICODE_CHARACTER_CODE_MODEL = DOWNWARDS_ARROW_WITH_TIP_RIGHTWARDS_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* DOWNWARDS_ARROW_WITH_TIP_RIGHTWARDS_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t DOWNWARDS_ARROW_WITH_CORNER_LEFTWARDS_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x21b5 };
static wchar_t* DOWNWARDS_ARROW_WITH_CORNER_LEFTWARDS_UNICODE_CHARACTER_CODE_MODEL = DOWNWARDS_ARROW_WITH_CORNER_LEFTWARDS_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* DOWNWARDS_ARROW_WITH_CORNER_LEFTWARDS_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t ANTICLOCKWISE_TOP_SEMICIRCLE_ARROW_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x21b6 };
static wchar_t* ANTICLOCKWISE_TOP_SEMICIRCLE_ARROW_UNICODE_CHARACTER_CODE_MODEL = ANTICLOCKWISE_TOP_SEMICIRCLE_ARROW_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* ANTICLOCKWISE_TOP_SEMICIRCLE_ARROW_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t CLOCKWISE_TOP_SEMICIRCLE_ARROW_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x21b7 };
static wchar_t* CLOCKWISE_TOP_SEMICIRCLE_ARROW_UNICODE_CHARACTER_CODE_MODEL = CLOCKWISE_TOP_SEMICIRCLE_ARROW_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CLOCKWISE_TOP_SEMICIRCLE_ARROW_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t ANTICLOCKWISE_OPEN_CIRCLE_ARROW_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x21ba };
static wchar_t* ANTICLOCKWISE_OPEN_CIRCLE_ARROW_UNICODE_CHARACTER_CODE_MODEL = ANTICLOCKWISE_OPEN_CIRCLE_ARROW_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* ANTICLOCKWISE_OPEN_CIRCLE_ARROW_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t CLOCKWISE_OPEN_CIRCLE_ARROW_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x21bb };
static wchar_t* CLOCKWISE_OPEN_CIRCLE_ARROW_UNICODE_CHARACTER_CODE_MODEL = CLOCKWISE_OPEN_CIRCLE_ARROW_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CLOCKWISE_OPEN_CIRCLE_ARROW_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LEFTWARDS_HARPOON_WITH_BARB_UPWARDS_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x21bc };
static wchar_t* LEFTWARDS_HARPOON_WITH_BARB_UPWARDS_UNICODE_CHARACTER_CODE_MODEL = LEFTWARDS_HARPOON_WITH_BARB_UPWARDS_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LEFTWARDS_HARPOON_WITH_BARB_UPWARDS_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LEFTWARDS_HARPOON_WITH_BARB_DOWNWARDS_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x21bd };
static wchar_t* LEFTWARDS_HARPOON_WITH_BARB_DOWNWARDS_UNICODE_CHARACTER_CODE_MODEL = LEFTWARDS_HARPOON_WITH_BARB_DOWNWARDS_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LEFTWARDS_HARPOON_WITH_BARB_DOWNWARDS_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t UPWARDS_HARPOON_WITH_BARB_RIGHTWARDS_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x21be };
static wchar_t* UPWARDS_HARPOON_WITH_BARB_RIGHTWARDS_UNICODE_CHARACTER_CODE_MODEL = UPWARDS_HARPOON_WITH_BARB_RIGHTWARDS_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* UPWARDS_HARPOON_WITH_BARB_RIGHTWARDS_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t UPWARDS_HARPOON_WITH_BARB_LEFTWARDS_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x21bf };
static wchar_t* UPWARDS_HARPOON_WITH_BARB_LEFTWARDS_UNICODE_CHARACTER_CODE_MODEL = UPWARDS_HARPOON_WITH_BARB_LEFTWARDS_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* UPWARDS_HARPOON_WITH_BARB_LEFTWARDS_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t RIGHTWARDS_HARPOON_WITH_BARB_UPWARDS_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x21c0 };
static wchar_t* RIGHTWARDS_HARPOON_WITH_BARB_UPWARDS_UNICODE_CHARACTER_CODE_MODEL = RIGHTWARDS_HARPOON_WITH_BARB_UPWARDS_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* RIGHTWARDS_HARPOON_WITH_BARB_UPWARDS_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t RIGHTWARDS_HARPOON_WITH_BARB_DOWNWARDS_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x21c1 };
static wchar_t* RIGHTWARDS_HARPOON_WITH_BARB_DOWNWARDS_UNICODE_CHARACTER_CODE_MODEL = RIGHTWARDS_HARPOON_WITH_BARB_DOWNWARDS_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* RIGHTWARDS_HARPOON_WITH_BARB_DOWNWARDS_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t DOWNWARDS_HARPOON_WITH_BARB_RIGHTWARDS_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x21c2 };
static wchar_t* DOWNWARDS_HARPOON_WITH_BARB_RIGHTWARDS_UNICODE_CHARACTER_CODE_MODEL = DOWNWARDS_HARPOON_WITH_BARB_RIGHTWARDS_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* DOWNWARDS_HARPOON_WITH_BARB_RIGHTWARDS_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t DOWNWARDS_HARPOON_WITH_BARB_LEFTWARDS_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x21c3 };
static wchar_t* DOWNWARDS_HARPOON_WITH_BARB_LEFTWARDS_UNICODE_CHARACTER_CODE_MODEL = DOWNWARDS_HARPOON_WITH_BARB_LEFTWARDS_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* DOWNWARDS_HARPOON_WITH_BARB_LEFTWARDS_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t RIGHTWARDS_ARROW_OVER_LEFTWARDS_ARROW_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x21c4 };
static wchar_t* RIGHTWARDS_ARROW_OVER_LEFTWARDS_ARROW_UNICODE_CHARACTER_CODE_MODEL = RIGHTWARDS_ARROW_OVER_LEFTWARDS_ARROW_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* RIGHTWARDS_ARROW_OVER_LEFTWARDS_ARROW_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t UPWARDS_ARROW_LEFTWARDS_OF_DOWNWARDS_ARROW_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x21c5 };
static wchar_t* UPWARDS_ARROW_LEFTWARDS_OF_DOWNWARDS_ARROW_UNICODE_CHARACTER_CODE_MODEL = UPWARDS_ARROW_LEFTWARDS_OF_DOWNWARDS_ARROW_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* UPWARDS_ARROW_LEFTWARDS_OF_DOWNWARDS_ARROW_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LEFTWARDS_ARROW_OVER_RIGHTWARDS_ARROW_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x21c6 };
static wchar_t* LEFTWARDS_ARROW_OVER_RIGHTWARDS_ARROW_UNICODE_CHARACTER_CODE_MODEL = LEFTWARDS_ARROW_OVER_RIGHTWARDS_ARROW_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LEFTWARDS_ARROW_OVER_RIGHTWARDS_ARROW_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LEFTWARDS_PAIRED_ARROWS_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x21c7 };
static wchar_t* LEFTWARDS_PAIRED_ARROWS_UNICODE_CHARACTER_CODE_MODEL = LEFTWARDS_PAIRED_ARROWS_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LEFTWARDS_PAIRED_ARROWS_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t UPWARDS_PAIRED_ARROWS_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x21c8 };
static wchar_t* UPWARDS_PAIRED_ARROWS_UNICODE_CHARACTER_CODE_MODEL = UPWARDS_PAIRED_ARROWS_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* UPWARDS_PAIRED_ARROWS_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t RIGHTWARDS_PAIRED_ARROWS_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x21c9 };
static wchar_t* RIGHTWARDS_PAIRED_ARROWS_UNICODE_CHARACTER_CODE_MODEL = RIGHTWARDS_PAIRED_ARROWS_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* RIGHTWARDS_PAIRED_ARROWS_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t DOWNWARDS_PAIRED_ARROWS_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x21ca };
static wchar_t* DOWNWARDS_PAIRED_ARROWS_UNICODE_CHARACTER_CODE_MODEL = DOWNWARDS_PAIRED_ARROWS_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* DOWNWARDS_PAIRED_ARROWS_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LEFTWARDS_HARPOON_OVER_RIGHTWARDS_HARPOON_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x21cb };
static wchar_t* LEFTWARDS_HARPOON_OVER_RIGHTWARDS_HARPOON_UNICODE_CHARACTER_CODE_MODEL = LEFTWARDS_HARPOON_OVER_RIGHTWARDS_HARPOON_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LEFTWARDS_HARPOON_OVER_RIGHTWARDS_HARPOON_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t RIGHTWARDS_HARPOON_OVER_LEFTWARDS_HARPOON_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x21cc };
static wchar_t* RIGHTWARDS_HARPOON_OVER_LEFTWARDS_HARPOON_UNICODE_CHARACTER_CODE_MODEL = RIGHTWARDS_HARPOON_OVER_LEFTWARDS_HARPOON_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* RIGHTWARDS_HARPOON_OVER_LEFTWARDS_HARPOON_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LEFTWARDS_DOUBLE_ARROW_WITH_STROKE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x21cd };
static wchar_t* LEFTWARDS_DOUBLE_ARROW_WITH_STROKE_UNICODE_CHARACTER_CODE_MODEL = LEFTWARDS_DOUBLE_ARROW_WITH_STROKE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LEFTWARDS_DOUBLE_ARROW_WITH_STROKE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LEFT_RIGHT_DOUBLE_ARROW_WITH_STROKE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x21ce };
static wchar_t* LEFT_RIGHT_DOUBLE_ARROW_WITH_STROKE_UNICODE_CHARACTER_CODE_MODEL = LEFT_RIGHT_DOUBLE_ARROW_WITH_STROKE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LEFT_RIGHT_DOUBLE_ARROW_WITH_STROKE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t RIGHTWARDS_DOUBLE_ARROW_WITH_STROKE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x21cf };
static wchar_t* RIGHTWARDS_DOUBLE_ARROW_WITH_STROKE_UNICODE_CHARACTER_CODE_MODEL = RIGHTWARDS_DOUBLE_ARROW_WITH_STROKE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* RIGHTWARDS_DOUBLE_ARROW_WITH_STROKE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LEFTWARDS_DOUBLE_ARROW_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x21d0 };
static wchar_t* LEFTWARDS_DOUBLE_ARROW_UNICODE_CHARACTER_CODE_MODEL = LEFTWARDS_DOUBLE_ARROW_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LEFTWARDS_DOUBLE_ARROW_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t UPWARDS_DOUBLE_ARROW_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x21d1 };
static wchar_t* UPWARDS_DOUBLE_ARROW_UNICODE_CHARACTER_CODE_MODEL = UPWARDS_DOUBLE_ARROW_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* UPWARDS_DOUBLE_ARROW_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t RIGHTWARDS_DOUBLE_ARROW_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x21d2 };
static wchar_t* RIGHTWARDS_DOUBLE_ARROW_UNICODE_CHARACTER_CODE_MODEL = RIGHTWARDS_DOUBLE_ARROW_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* RIGHTWARDS_DOUBLE_ARROW_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t DOWNWARDS_DOUBLE_ARROW_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x21d3 };
static wchar_t* DOWNWARDS_DOUBLE_ARROW_UNICODE_CHARACTER_CODE_MODEL = DOWNWARDS_DOUBLE_ARROW_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* DOWNWARDS_DOUBLE_ARROW_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LEFT_RIGHT_DOUBLE_ARROW_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x21d4 };
static wchar_t* LEFT_RIGHT_DOUBLE_ARROW_UNICODE_CHARACTER_CODE_MODEL = LEFT_RIGHT_DOUBLE_ARROW_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LEFT_RIGHT_DOUBLE_ARROW_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t UP_DOWN_DOUBLE_ARROW_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x21d5 };
static wchar_t* UP_DOWN_DOUBLE_ARROW_UNICODE_CHARACTER_CODE_MODEL = UP_DOWN_DOUBLE_ARROW_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* UP_DOWN_DOUBLE_ARROW_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t NORTH_WEST_DOUBLE_ARROW_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x21d6 };
static wchar_t* NORTH_WEST_DOUBLE_ARROW_UNICODE_CHARACTER_CODE_MODEL = NORTH_WEST_DOUBLE_ARROW_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* NORTH_WEST_DOUBLE_ARROW_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t NORTH_EAST_DOUBLE_ARROW_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x21d7 };
static wchar_t* NORTH_EAST_DOUBLE_ARROW_UNICODE_CHARACTER_CODE_MODEL = NORTH_EAST_DOUBLE_ARROW_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* NORTH_EAST_DOUBLE_ARROW_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t SOUTH_EAST_DOUBLE_ARROW_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x21d8 };
static wchar_t* SOUTH_EAST_DOUBLE_ARROW_UNICODE_CHARACTER_CODE_MODEL = SOUTH_EAST_DOUBLE_ARROW_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* SOUTH_EAST_DOUBLE_ARROW_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t SOUTH_WEST_DOUBLE_ARROW_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x21d9 };
static wchar_t* SOUTH_WEST_DOUBLE_ARROW_UNICODE_CHARACTER_CODE_MODEL = SOUTH_WEST_DOUBLE_ARROW_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* SOUTH_WEST_DOUBLE_ARROW_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LEFTWARDS_TRIPLE_ARROW_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x21da };
static wchar_t* LEFTWARDS_TRIPLE_ARROW_UNICODE_CHARACTER_CODE_MODEL = LEFTWARDS_TRIPLE_ARROW_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LEFTWARDS_TRIPLE_ARROW_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t RIGHTWARDS_TRIPLE_ARROW_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x21db };
static wchar_t* RIGHTWARDS_TRIPLE_ARROW_UNICODE_CHARACTER_CODE_MODEL = RIGHTWARDS_TRIPLE_ARROW_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* RIGHTWARDS_TRIPLE_ARROW_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t RIGHTWARDS_SQUIGGLE_ARROW_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x21dd };
static wchar_t* RIGHTWARDS_SQUIGGLE_ARROW_UNICODE_CHARACTER_CODE_MODEL = RIGHTWARDS_SQUIGGLE_ARROW_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* RIGHTWARDS_SQUIGGLE_ARROW_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LEFTWARDS_ARROW_TO_BAR_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x21e4 };
static wchar_t* LEFTWARDS_ARROW_TO_BAR_UNICODE_CHARACTER_CODE_MODEL = LEFTWARDS_ARROW_TO_BAR_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LEFTWARDS_ARROW_TO_BAR_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t RIGHTWARDS_ARROW_TO_BAR_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x21e5 };
static wchar_t* RIGHTWARDS_ARROW_TO_BAR_UNICODE_CHARACTER_CODE_MODEL = RIGHTWARDS_ARROW_TO_BAR_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* RIGHTWARDS_ARROW_TO_BAR_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t DOWNWARDS_ARROW_LEFTWARDS_OF_UPWARDS_ARROW_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x21f5 };
static wchar_t* DOWNWARDS_ARROW_LEFTWARDS_OF_UPWARDS_ARROW_UNICODE_CHARACTER_CODE_MODEL = DOWNWARDS_ARROW_LEFTWARDS_OF_UPWARDS_ARROW_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* DOWNWARDS_ARROW_LEFTWARDS_OF_UPWARDS_ARROW_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LEFTWARDS_OPEN_HEADED_ARROW_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x21fd };
static wchar_t* LEFTWARDS_OPEN_HEADED_ARROW_UNICODE_CHARACTER_CODE_MODEL = LEFTWARDS_OPEN_HEADED_ARROW_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LEFTWARDS_OPEN_HEADED_ARROW_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t RIGHTWARDS_OPEN_HEADED_ARROW_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x21fe };
static wchar_t* RIGHTWARDS_OPEN_HEADED_ARROW_UNICODE_CHARACTER_CODE_MODEL = RIGHTWARDS_OPEN_HEADED_ARROW_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* RIGHTWARDS_OPEN_HEADED_ARROW_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LEFT_RIGHT_OPEN_HEADED_ARROW_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x21ff };
static wchar_t* LEFT_RIGHT_OPEN_HEADED_ARROW_UNICODE_CHARACTER_CODE_MODEL = LEFT_RIGHT_OPEN_HEADED_ARROW_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LEFT_RIGHT_OPEN_HEADED_ARROW_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t FOR_ALL_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2200 };
static wchar_t* FOR_ALL_UNICODE_CHARACTER_CODE_MODEL = FOR_ALL_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* FOR_ALL_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t COMPLEMENT_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2201 };
static wchar_t* COMPLEMENT_UNICODE_CHARACTER_CODE_MODEL = COMPLEMENT_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* COMPLEMENT_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t PARTIAL_DIFFERENTIAL_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2202 };
static wchar_t* PARTIAL_DIFFERENTIAL_UNICODE_CHARACTER_CODE_MODEL = PARTIAL_DIFFERENTIAL_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* PARTIAL_DIFFERENTIAL_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t PARTIAL_DIFFERENTIAL_WITH_SLASH_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2202, 0x0338 };
static wchar_t* PARTIAL_DIFFERENTIAL_WITH_SLASH_UNICODE_CHARACTER_CODE_MODEL = PARTIAL_DIFFERENTIAL_WITH_SLASH_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* PARTIAL_DIFFERENTIAL_WITH_SLASH_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_2_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t THERE_EXISTS_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2203 };
static wchar_t* THERE_EXISTS_UNICODE_CHARACTER_CODE_MODEL = THERE_EXISTS_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* THERE_EXISTS_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t THERE_DOES_NOT_EXIST_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2204 };
static wchar_t* THERE_DOES_NOT_EXIST_UNICODE_CHARACTER_CODE_MODEL = THERE_DOES_NOT_EXIST_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* THERE_DOES_NOT_EXIST_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t EMPTY_SET_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2205 };
static wchar_t* EMPTY_SET_UNICODE_CHARACTER_CODE_MODEL = EMPTY_SET_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* EMPTY_SET_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t NABLA_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2207 };
static wchar_t* NABLA_UNICODE_CHARACTER_CODE_MODEL = NABLA_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* NABLA_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t ELEMENT_OF_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2208 };
static wchar_t* ELEMENT_OF_UNICODE_CHARACTER_CODE_MODEL = ELEMENT_OF_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* ELEMENT_OF_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t NOT_AN_ELEMENT_OF_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2209 };
static wchar_t* NOT_AN_ELEMENT_OF_UNICODE_CHARACTER_CODE_MODEL = NOT_AN_ELEMENT_OF_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* NOT_AN_ELEMENT_OF_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t CONTAINS_AS_MEMBER_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x220b };
static wchar_t* CONTAINS_AS_MEMBER_UNICODE_CHARACTER_CODE_MODEL = CONTAINS_AS_MEMBER_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CONTAINS_AS_MEMBER_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t DOES_NOT_CONTAIN_AS_MEMBER_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x220c };
static wchar_t* DOES_NOT_CONTAIN_AS_MEMBER_UNICODE_CHARACTER_CODE_MODEL = DOES_NOT_CONTAIN_AS_MEMBER_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* DOES_NOT_CONTAIN_AS_MEMBER_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t N_ARY_PRODUCT_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x220f };
static wchar_t* N_ARY_PRODUCT_UNICODE_CHARACTER_CODE_MODEL = N_ARY_PRODUCT_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* N_ARY_PRODUCT_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t N_ARY_COPRODUCT_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2210 };
static wchar_t* N_ARY_COPRODUCT_UNICODE_CHARACTER_CODE_MODEL = N_ARY_COPRODUCT_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* N_ARY_COPRODUCT_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t N_ARY_SUMMATION_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2211 };
static wchar_t* N_ARY_SUMMATION_UNICODE_CHARACTER_CODE_MODEL = N_ARY_SUMMATION_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* N_ARY_SUMMATION_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MINUS_SIGN_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2212 };
static wchar_t* MINUS_SIGN_UNICODE_CHARACTER_CODE_MODEL = MINUS_SIGN_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MINUS_SIGN_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MINUS_OR_PLUS_SIGN_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2213 };
static wchar_t* MINUS_OR_PLUS_SIGN_UNICODE_CHARACTER_CODE_MODEL = MINUS_OR_PLUS_SIGN_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MINUS_OR_PLUS_SIGN_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t DOT_PLUS_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2214 };
static wchar_t* DOT_PLUS_UNICODE_CHARACTER_CODE_MODEL = DOT_PLUS_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* DOT_PLUS_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t SET_MINUS_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2216 };
static wchar_t* SET_MINUS_UNICODE_CHARACTER_CODE_MODEL = SET_MINUS_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* SET_MINUS_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t ASTERISK_OPERATOR_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2217 };
static wchar_t* ASTERISK_OPERATOR_UNICODE_CHARACTER_CODE_MODEL = ASTERISK_OPERATOR_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* ASTERISK_OPERATOR_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t RING_OPERATOR_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2218 };
static wchar_t* RING_OPERATOR_UNICODE_CHARACTER_CODE_MODEL = RING_OPERATOR_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* RING_OPERATOR_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t BULLET_OPERATOR_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2219 };
static wchar_t* BULLET_OPERATOR_UNICODE_CHARACTER_CODE_MODEL = BULLET_OPERATOR_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* BULLET_OPERATOR_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t SQUARE_ROOT_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x221a };
static wchar_t* SQUARE_ROOT_UNICODE_CHARACTER_CODE_MODEL = SQUARE_ROOT_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* SQUARE_ROOT_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t PROPORTIONAL_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x221d };
static wchar_t* PROPORTIONAL_TO_UNICODE_CHARACTER_CODE_MODEL = PROPORTIONAL_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* PROPORTIONAL_TO_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t INFINITY_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x221e };
static wchar_t* INFINITY_UNICODE_CHARACTER_CODE_MODEL = INFINITY_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* INFINITY_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t RIGHT_ANGLE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x221f };
static wchar_t* RIGHT_ANGLE_UNICODE_CHARACTER_CODE_MODEL = RIGHT_ANGLE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* RIGHT_ANGLE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t ANGLE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2220 };
static wchar_t* ANGLE_UNICODE_CHARACTER_CODE_MODEL = ANGLE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* ANGLE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t ANGLE_WITH_VERTICAL_LINE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2220, 0x20d2 };
static wchar_t* ANGLE_WITH_VERTICAL_LINE_UNICODE_CHARACTER_CODE_MODEL = ANGLE_WITH_VERTICAL_LINE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* ANGLE_WITH_VERTICAL_LINE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_2_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MEASURED_ANGLE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2221 };
static wchar_t* MEASURED_ANGLE_UNICODE_CHARACTER_CODE_MODEL = MEASURED_ANGLE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MEASURED_ANGLE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t SPHERICAL_ANGLE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2222 };
static wchar_t* SPHERICAL_ANGLE_UNICODE_CHARACTER_CODE_MODEL = SPHERICAL_ANGLE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* SPHERICAL_ANGLE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t DIVIDES_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2223 };
static wchar_t* DIVIDES_UNICODE_CHARACTER_CODE_MODEL = DIVIDES_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* DIVIDES_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t DOES_NOT_DIVIDE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2224 };
static wchar_t* DOES_NOT_DIVIDE_UNICODE_CHARACTER_CODE_MODEL = DOES_NOT_DIVIDE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* DOES_NOT_DIVIDE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t PARALLEL_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2225 };
static wchar_t* PARALLEL_TO_UNICODE_CHARACTER_CODE_MODEL = PARALLEL_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* PARALLEL_TO_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t NOT_PARALLEL_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2226 };
static wchar_t* NOT_PARALLEL_TO_UNICODE_CHARACTER_CODE_MODEL = NOT_PARALLEL_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* NOT_PARALLEL_TO_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LOGICAL_AND_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2227 };
static wchar_t* LOGICAL_AND_UNICODE_CHARACTER_CODE_MODEL = LOGICAL_AND_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LOGICAL_AND_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LOGICAL_OR_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2228 };
static wchar_t* LOGICAL_OR_UNICODE_CHARACTER_CODE_MODEL = LOGICAL_OR_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LOGICAL_OR_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t INTERSECTION_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2229 };
static wchar_t* INTERSECTION_UNICODE_CHARACTER_CODE_MODEL = INTERSECTION_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* INTERSECTION_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t INTERSECTION_WITH_SERIFS_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2229, 0xfe00 };
static wchar_t* INTERSECTION_WITH_SERIFS_UNICODE_CHARACTER_CODE_MODEL = INTERSECTION_WITH_SERIFS_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* INTERSECTION_WITH_SERIFS_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_2_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t UNION_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x222a };
static wchar_t* UNION_UNICODE_CHARACTER_CODE_MODEL = UNION_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* UNION_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t UNION_WITH_SERIFS_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x222a, 0xfe00 };
static wchar_t* UNION_WITH_SERIFS_UNICODE_CHARACTER_CODE_MODEL = UNION_WITH_SERIFS_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* UNION_WITH_SERIFS_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_2_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t INTEGRAL_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x222b };
static wchar_t* INTEGRAL_UNICODE_CHARACTER_CODE_MODEL = INTEGRAL_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* INTEGRAL_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t DOUBLE_INTEGRAL_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x222c };
static wchar_t* DOUBLE_INTEGRAL_UNICODE_CHARACTER_CODE_MODEL = DOUBLE_INTEGRAL_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* DOUBLE_INTEGRAL_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t TRIPLE_INTEGRAL_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x222d };
static wchar_t* TRIPLE_INTEGRAL_UNICODE_CHARACTER_CODE_MODEL = TRIPLE_INTEGRAL_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* TRIPLE_INTEGRAL_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t CONTOUR_INTEGRAL_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x222e };
static wchar_t* CONTOUR_INTEGRAL_UNICODE_CHARACTER_CODE_MODEL = CONTOUR_INTEGRAL_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CONTOUR_INTEGRAL_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t SURFACE_INTEGRAL_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x222f };
static wchar_t* SURFACE_INTEGRAL_UNICODE_CHARACTER_CODE_MODEL = SURFACE_INTEGRAL_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* SURFACE_INTEGRAL_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t VOLUME_INTEGRAL_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2230 };
static wchar_t* VOLUME_INTEGRAL_UNICODE_CHARACTER_CODE_MODEL = VOLUME_INTEGRAL_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* VOLUME_INTEGRAL_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t CLOCKWISE_INTEGRAL_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2231 };
static wchar_t* CLOCKWISE_INTEGRAL_UNICODE_CHARACTER_CODE_MODEL = CLOCKWISE_INTEGRAL_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CLOCKWISE_INTEGRAL_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t CLOCKWISE_CONTOUR_INTEGRAL_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2232 };
static wchar_t* CLOCKWISE_CONTOUR_INTEGRAL_UNICODE_CHARACTER_CODE_MODEL = CLOCKWISE_CONTOUR_INTEGRAL_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CLOCKWISE_CONTOUR_INTEGRAL_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t ANTICLOCKWISE_CONTOUR_INTEGRAL_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2233 };
static wchar_t* ANTICLOCKWISE_CONTOUR_INTEGRAL_UNICODE_CHARACTER_CODE_MODEL = ANTICLOCKWISE_CONTOUR_INTEGRAL_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* ANTICLOCKWISE_CONTOUR_INTEGRAL_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t THEREFORE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2234 };
static wchar_t* THEREFORE_UNICODE_CHARACTER_CODE_MODEL = THEREFORE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* THEREFORE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t BECAUSE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2235 };
static wchar_t* BECAUSE_UNICODE_CHARACTER_CODE_MODEL = BECAUSE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* BECAUSE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t RATIO_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2236 };
static wchar_t* RATIO_UNICODE_CHARACTER_CODE_MODEL = RATIO_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* RATIO_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t PROPORTION_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2237 };
static wchar_t* PROPORTION_UNICODE_CHARACTER_CODE_MODEL = PROPORTION_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* PROPORTION_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t DOT_MINUS_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2238 };
static wchar_t* DOT_MINUS_UNICODE_CHARACTER_CODE_MODEL = DOT_MINUS_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* DOT_MINUS_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t GEOMETRIC_PROPORTION_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x223a };
static wchar_t* GEOMETRIC_PROPORTION_UNICODE_CHARACTER_CODE_MODEL = GEOMETRIC_PROPORTION_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* GEOMETRIC_PROPORTION_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t HOMOTHETIC_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x223b };
static wchar_t* HOMOTHETIC_UNICODE_CHARACTER_CODE_MODEL = HOMOTHETIC_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* HOMOTHETIC_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t TILDE_OPERATOR_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x223c };
static wchar_t* TILDE_OPERATOR_UNICODE_CHARACTER_CODE_MODEL = TILDE_OPERATOR_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* TILDE_OPERATOR_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t TILDE_OPERATOR_WITH_VERTICAL_LINE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x223c, 0x20d2 };
static wchar_t* TILDE_OPERATOR_WITH_VERTICAL_LINE_UNICODE_CHARACTER_CODE_MODEL = TILDE_OPERATOR_WITH_VERTICAL_LINE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* TILDE_OPERATOR_WITH_VERTICAL_LINE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_2_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t REVERSED_TILDE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x223d };
static wchar_t* REVERSED_TILDE_UNICODE_CHARACTER_CODE_MODEL = REVERSED_TILDE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* REVERSED_TILDE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t REVERSED_TILDE_WITH_UNDERLINE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x223d, 0x0331 };
static wchar_t* REVERSED_TILDE_WITH_UNDERLINE_UNICODE_CHARACTER_CODE_MODEL = REVERSED_TILDE_WITH_UNDERLINE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* REVERSED_TILDE_WITH_UNDERLINE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_2_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t INVERTED_LAZY_S_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x223e };
static wchar_t* INVERTED_LAZY_S_UNICODE_CHARACTER_CODE_MODEL = INVERTED_LAZY_S_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* INVERTED_LAZY_S_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t INVERTED_LAZY_S_WITH_DOUBLE_UNDERLINE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x223e, 0x0333 };
static wchar_t* INVERTED_LAZY_S_WITH_DOUBLE_UNDERLINE_UNICODE_CHARACTER_CODE_MODEL = INVERTED_LAZY_S_WITH_DOUBLE_UNDERLINE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* INVERTED_LAZY_S_WITH_DOUBLE_UNDERLINE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_2_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t SINE_WAVE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x223f };
static wchar_t* SINE_WAVE_UNICODE_CHARACTER_CODE_MODEL = SINE_WAVE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* SINE_WAVE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t WREATH_PRODUCT_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2240 };
static wchar_t* WREATH_PRODUCT_UNICODE_CHARACTER_CODE_MODEL = WREATH_PRODUCT_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* WREATH_PRODUCT_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t NOT_TILDE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2241 };
static wchar_t* NOT_TILDE_UNICODE_CHARACTER_CODE_MODEL = NOT_TILDE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* NOT_TILDE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MINUS_TILDE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2242 };
static wchar_t* MINUS_TILDE_UNICODE_CHARACTER_CODE_MODEL = MINUS_TILDE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MINUS_TILDE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MINUS_TILDE_WITH_SLASH_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2242, 0x0338 };
static wchar_t* MINUS_TILDE_WITH_SLASH_UNICODE_CHARACTER_CODE_MODEL = MINUS_TILDE_WITH_SLASH_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MINUS_TILDE_WITH_SLASH_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_2_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t ASYMPTOTICALLY_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2243 };
static wchar_t* ASYMPTOTICALLY_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL = ASYMPTOTICALLY_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* ASYMPTOTICALLY_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t NOT_ASYMPTOTICALLY_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2244 };
static wchar_t* NOT_ASYMPTOTICALLY_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL = NOT_ASYMPTOTICALLY_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* NOT_ASYMPTOTICALLY_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t APPROXIMATELY_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2245 };
static wchar_t* APPROXIMATELY_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL = APPROXIMATELY_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* APPROXIMATELY_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t APPROXIMATELY_BUT_NOT_ACTUALLY_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2246 };
static wchar_t* APPROXIMATELY_BUT_NOT_ACTUALLY_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL = APPROXIMATELY_BUT_NOT_ACTUALLY_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* APPROXIMATELY_BUT_NOT_ACTUALLY_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t NEITHER_APPROXIMATELY_NOR_ACTUALLY_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2247 };
static wchar_t* NEITHER_APPROXIMATELY_NOR_ACTUALLY_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL = NEITHER_APPROXIMATELY_NOR_ACTUALLY_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* NEITHER_APPROXIMATELY_NOR_ACTUALLY_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t ALMOST_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2248 };
static wchar_t* ALMOST_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL = ALMOST_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* ALMOST_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t NOT_ALMOST_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2249 };
static wchar_t* NOT_ALMOST_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL = NOT_ALMOST_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* NOT_ALMOST_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t ALMOST_EQUAL_OR_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x224a };
static wchar_t* ALMOST_EQUAL_OR_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL = ALMOST_EQUAL_OR_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* ALMOST_EQUAL_OR_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t TRIPLE_TILDE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x224b };
static wchar_t* TRIPLE_TILDE_UNICODE_CHARACTER_CODE_MODEL = TRIPLE_TILDE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* TRIPLE_TILDE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t TRIPLE_TILDE_WITH_SLASH_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x224b, 0x0338 };
static wchar_t* TRIPLE_TILDE_WITH_SLASH_UNICODE_CHARACTER_CODE_MODEL = TRIPLE_TILDE_WITH_SLASH_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* TRIPLE_TILDE_WITH_SLASH_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_2_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t ALL_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x224c };
static wchar_t* ALL_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL = ALL_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* ALL_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t EQUIVALENT_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x224d };
static wchar_t* EQUIVALENT_TO_UNICODE_CHARACTER_CODE_MODEL = EQUIVALENT_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* EQUIVALENT_TO_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t EQUIVALENT_TO_WITH_VERTICAL_LINE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x224d, 0x20d2 };
static wchar_t* EQUIVALENT_TO_WITH_VERTICAL_LINE_UNICODE_CHARACTER_CODE_MODEL = EQUIVALENT_TO_WITH_VERTICAL_LINE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* EQUIVALENT_TO_WITH_VERTICAL_LINE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_2_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t GEOMETRICALLY_EQUIVALENT_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x224e };
static wchar_t* GEOMETRICALLY_EQUIVALENT_TO_UNICODE_CHARACTER_CODE_MODEL = GEOMETRICALLY_EQUIVALENT_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* GEOMETRICALLY_EQUIVALENT_TO_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t GEOMETRICALLY_EQUIVALENT_TO_WITH_SLASH_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x224e, 0x0338 };
static wchar_t* GEOMETRICALLY_EQUIVALENT_TO_WITH_SLASH_UNICODE_CHARACTER_CODE_MODEL = GEOMETRICALLY_EQUIVALENT_TO_WITH_SLASH_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* GEOMETRICALLY_EQUIVALENT_TO_WITH_SLASH_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_2_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t DIFFERENCE_BETWEEN_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x224f };
static wchar_t* DIFFERENCE_BETWEEN_UNICODE_CHARACTER_CODE_MODEL = DIFFERENCE_BETWEEN_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* DIFFERENCE_BETWEEN_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t DIFFERENCE_BETWEEN_WITH_SLASH_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x224f, 0x0338 };
static wchar_t* DIFFERENCE_BETWEEN_WITH_SLASH_UNICODE_CHARACTER_CODE_MODEL = DIFFERENCE_BETWEEN_WITH_SLASH_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* DIFFERENCE_BETWEEN_WITH_SLASH_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_2_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t APPROACHES_THE_LIMIT_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2250 };
static wchar_t* APPROACHES_THE_LIMIT_UNICODE_CHARACTER_CODE_MODEL = APPROACHES_THE_LIMIT_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* APPROACHES_THE_LIMIT_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t APPROACHES_THE_LIMIT_WITH_SLASH_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2250, 0x0338 };
static wchar_t* APPROACHES_THE_LIMIT_WITH_SLASH_UNICODE_CHARACTER_CODE_MODEL = APPROACHES_THE_LIMIT_WITH_SLASH_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* APPROACHES_THE_LIMIT_WITH_SLASH_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_2_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t GEOMETRICALLY_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2251 };
static wchar_t* GEOMETRICALLY_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL = GEOMETRICALLY_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* GEOMETRICALLY_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t APPROXIMATELY_EQUAL_TO_OR_THE_IMAGE_OF_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2252 };
static wchar_t* APPROXIMATELY_EQUAL_TO_OR_THE_IMAGE_OF_UNICODE_CHARACTER_CODE_MODEL = APPROXIMATELY_EQUAL_TO_OR_THE_IMAGE_OF_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* APPROXIMATELY_EQUAL_TO_OR_THE_IMAGE_OF_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t IMAGE_OF_OR_APPROXIMATELY_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2253 };
static wchar_t* IMAGE_OF_OR_APPROXIMATELY_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL = IMAGE_OF_OR_APPROXIMATELY_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* IMAGE_OF_OR_APPROXIMATELY_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t COLON_EQUALS_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2254 };
static wchar_t* COLON_EQUALS_UNICODE_CHARACTER_CODE_MODEL = COLON_EQUALS_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* COLON_EQUALS_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t EQUALS_COLON_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2255 };
static wchar_t* EQUALS_COLON_UNICODE_CHARACTER_CODE_MODEL = EQUALS_COLON_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* EQUALS_COLON_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t RING_IN_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2256 };
static wchar_t* RING_IN_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL = RING_IN_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* RING_IN_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t RING_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2257 };
static wchar_t* RING_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL = RING_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* RING_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t ESTIMATES_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2259 };
static wchar_t* ESTIMATES_UNICODE_CHARACTER_CODE_MODEL = ESTIMATES_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* ESTIMATES_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t EQUIANGULAR_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x225a };
static wchar_t* EQUIANGULAR_TO_UNICODE_CHARACTER_CODE_MODEL = EQUIANGULAR_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* EQUIANGULAR_TO_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t DELTA_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x225c };
static wchar_t* DELTA_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL = DELTA_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* DELTA_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t QUESTIONED_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x225f };
static wchar_t* QUESTIONED_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL = QUESTIONED_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* QUESTIONED_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t NOT_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2260 };
static wchar_t* NOT_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL = NOT_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* NOT_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t IDENTICAL_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2261 };
static wchar_t* IDENTICAL_TO_UNICODE_CHARACTER_CODE_MODEL = IDENTICAL_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* IDENTICAL_TO_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t IDENTICAL_TO_WITH_REVERSE_SLASH_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2261, 0x20e5 };
static wchar_t* IDENTICAL_TO_WITH_REVERSE_SLASH_UNICODE_CHARACTER_CODE_MODEL = IDENTICAL_TO_WITH_REVERSE_SLASH_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* IDENTICAL_TO_WITH_REVERSE_SLASH_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_2_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t NOT_IDENTICAL_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2262 };
static wchar_t* NOT_IDENTICAL_TO_UNICODE_CHARACTER_CODE_MODEL = NOT_IDENTICAL_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* NOT_IDENTICAL_TO_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LESS_THAN_OR_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2264 };
static wchar_t* LESS_THAN_OR_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL = LESS_THAN_OR_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LESS_THAN_OR_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LESS_THAN_OR_EQUAL_TO_WITH_VERTICAL_LINE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2264, 0x20d2 };
static wchar_t* LESS_THAN_OR_EQUAL_TO_WITH_VERTICAL_LINE_UNICODE_CHARACTER_CODE_MODEL = LESS_THAN_OR_EQUAL_TO_WITH_VERTICAL_LINE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LESS_THAN_OR_EQUAL_TO_WITH_VERTICAL_LINE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_2_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t GREATER_THAN_OR_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2265 };
static wchar_t* GREATER_THAN_OR_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL = GREATER_THAN_OR_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* GREATER_THAN_OR_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t GREATER_THAN_OR_EQUAL_TO_WITH_VERTICAL_LINE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2265, 0x20d2 };
static wchar_t* GREATER_THAN_OR_EQUAL_TO_WITH_VERTICAL_LINE_UNICODE_CHARACTER_CODE_MODEL = GREATER_THAN_OR_EQUAL_TO_WITH_VERTICAL_LINE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* GREATER_THAN_OR_EQUAL_TO_WITH_VERTICAL_LINE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_2_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LESS_THAN_OVER_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2266 };
static wchar_t* LESS_THAN_OVER_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL = LESS_THAN_OVER_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LESS_THAN_OVER_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LESS_THAN_OVER_EQUAL_TO_WITH_SLASH_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2266, 0x0338 };
static wchar_t* LESS_THAN_OVER_EQUAL_TO_WITH_SLASH_UNICODE_CHARACTER_CODE_MODEL = LESS_THAN_OVER_EQUAL_TO_WITH_SLASH_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LESS_THAN_OVER_EQUAL_TO_WITH_SLASH_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_2_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t GREATER_THAN_OVER_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2267 };
static wchar_t* GREATER_THAN_OVER_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL = GREATER_THAN_OVER_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* GREATER_THAN_OVER_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t GREATER_THAN_OVER_EQUAL_TO_WITH_SLASH_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2267, 0x0338 };
static wchar_t* GREATER_THAN_OVER_EQUAL_TO_WITH_SLASH_UNICODE_CHARACTER_CODE_MODEL = GREATER_THAN_OVER_EQUAL_TO_WITH_SLASH_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* GREATER_THAN_OVER_EQUAL_TO_WITH_SLASH_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_2_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LESS_THAN_BUT_NOT_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2268 };
static wchar_t* LESS_THAN_BUT_NOT_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL = LESS_THAN_BUT_NOT_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LESS_THAN_BUT_NOT_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LESS_THAN_BUT_NOT_EQUAL_TO___WITH_VERTICAL_STROKE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2268, 0xfe00 };
static wchar_t* LESS_THAN_BUT_NOT_EQUAL_TO___WITH_VERTICAL_STROKE_UNICODE_CHARACTER_CODE_MODEL = LESS_THAN_BUT_NOT_EQUAL_TO___WITH_VERTICAL_STROKE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LESS_THAN_BUT_NOT_EQUAL_TO___WITH_VERTICAL_STROKE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_2_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t GREATER_THAN_BUT_NOT_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2269 };
static wchar_t* GREATER_THAN_BUT_NOT_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL = GREATER_THAN_BUT_NOT_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* GREATER_THAN_BUT_NOT_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t GREATER_THAN_BUT_NOT_EQUAL_TO___WITH_VERTICAL_STROKE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2269, 0xfe00 };
static wchar_t* GREATER_THAN_BUT_NOT_EQUAL_TO___WITH_VERTICAL_STROKE_UNICODE_CHARACTER_CODE_MODEL = GREATER_THAN_BUT_NOT_EQUAL_TO___WITH_VERTICAL_STROKE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* GREATER_THAN_BUT_NOT_EQUAL_TO___WITH_VERTICAL_STROKE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_2_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MUCH_LESS_THAN_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x226a };
static wchar_t* MUCH_LESS_THAN_UNICODE_CHARACTER_CODE_MODEL = MUCH_LESS_THAN_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MUCH_LESS_THAN_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MUCH_LESS_THAN_WITH_SLASH_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x226a, 0x0338 };
static wchar_t* MUCH_LESS_THAN_WITH_SLASH_UNICODE_CHARACTER_CODE_MODEL = MUCH_LESS_THAN_WITH_SLASH_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MUCH_LESS_THAN_WITH_SLASH_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_2_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MUCH_LESS_THAN_WITH_VERTICAL_LINE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x226a, 0x20d2 };
static wchar_t* MUCH_LESS_THAN_WITH_VERTICAL_LINE_UNICODE_CHARACTER_CODE_MODEL = MUCH_LESS_THAN_WITH_VERTICAL_LINE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MUCH_LESS_THAN_WITH_VERTICAL_LINE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_2_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MUCH_GREATER_THAN_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x226b };
static wchar_t* MUCH_GREATER_THAN_UNICODE_CHARACTER_CODE_MODEL = MUCH_GREATER_THAN_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MUCH_GREATER_THAN_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MUCH_GREATER_THAN_WITH_SLASH_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x226b, 0x0338 };
static wchar_t* MUCH_GREATER_THAN_WITH_SLASH_UNICODE_CHARACTER_CODE_MODEL = MUCH_GREATER_THAN_WITH_SLASH_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MUCH_GREATER_THAN_WITH_SLASH_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_2_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MUCH_GREATER_THAN_WITH_VERTICAL_LINE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x226b, 0x20d2 };
static wchar_t* MUCH_GREATER_THAN_WITH_VERTICAL_LINE_UNICODE_CHARACTER_CODE_MODEL = MUCH_GREATER_THAN_WITH_VERTICAL_LINE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MUCH_GREATER_THAN_WITH_VERTICAL_LINE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_2_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t BETWEEN_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x226c };
static wchar_t* BETWEEN_UNICODE_CHARACTER_CODE_MODEL = BETWEEN_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* BETWEEN_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t NOT_EQUIVALENT_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x226d };
static wchar_t* NOT_EQUIVALENT_TO_UNICODE_CHARACTER_CODE_MODEL = NOT_EQUIVALENT_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* NOT_EQUIVALENT_TO_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t NOT_LESS_THAN_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x226e };
static wchar_t* NOT_LESS_THAN_UNICODE_CHARACTER_CODE_MODEL = NOT_LESS_THAN_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* NOT_LESS_THAN_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t NOT_GREATER_THAN_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x226f };
static wchar_t* NOT_GREATER_THAN_UNICODE_CHARACTER_CODE_MODEL = NOT_GREATER_THAN_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* NOT_GREATER_THAN_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t NEITHER_LESS_THAN_NOR_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2270 };
static wchar_t* NEITHER_LESS_THAN_NOR_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL = NEITHER_LESS_THAN_NOR_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* NEITHER_LESS_THAN_NOR_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t NEITHER_GREATER_THAN_NOR_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2271 };
static wchar_t* NEITHER_GREATER_THAN_NOR_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL = NEITHER_GREATER_THAN_NOR_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* NEITHER_GREATER_THAN_NOR_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LESS_THAN_OR_EQUIVALENT_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2272 };
static wchar_t* LESS_THAN_OR_EQUIVALENT_TO_UNICODE_CHARACTER_CODE_MODEL = LESS_THAN_OR_EQUIVALENT_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LESS_THAN_OR_EQUIVALENT_TO_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t GREATER_THAN_OR_EQUIVALENT_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2273 };
static wchar_t* GREATER_THAN_OR_EQUIVALENT_TO_UNICODE_CHARACTER_CODE_MODEL = GREATER_THAN_OR_EQUIVALENT_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* GREATER_THAN_OR_EQUIVALENT_TO_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t NEITHER_LESS_THAN_NOR_EQUIVALENT_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2274 };
static wchar_t* NEITHER_LESS_THAN_NOR_EQUIVALENT_TO_UNICODE_CHARACTER_CODE_MODEL = NEITHER_LESS_THAN_NOR_EQUIVALENT_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* NEITHER_LESS_THAN_NOR_EQUIVALENT_TO_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t NEITHER_GREATER_THAN_NOR_EQUIVALENT_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2275 };
static wchar_t* NEITHER_GREATER_THAN_NOR_EQUIVALENT_TO_UNICODE_CHARACTER_CODE_MODEL = NEITHER_GREATER_THAN_NOR_EQUIVALENT_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* NEITHER_GREATER_THAN_NOR_EQUIVALENT_TO_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LESS_THAN_OR_GREATER_THAN_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2276 };
static wchar_t* LESS_THAN_OR_GREATER_THAN_UNICODE_CHARACTER_CODE_MODEL = LESS_THAN_OR_GREATER_THAN_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LESS_THAN_OR_GREATER_THAN_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t GREATER_THAN_OR_LESS_THAN_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2277 };
static wchar_t* GREATER_THAN_OR_LESS_THAN_UNICODE_CHARACTER_CODE_MODEL = GREATER_THAN_OR_LESS_THAN_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* GREATER_THAN_OR_LESS_THAN_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t NEITHER_LESS_THAN_NOR_GREATER_THAN_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2278 };
static wchar_t* NEITHER_LESS_THAN_NOR_GREATER_THAN_UNICODE_CHARACTER_CODE_MODEL = NEITHER_LESS_THAN_NOR_GREATER_THAN_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* NEITHER_LESS_THAN_NOR_GREATER_THAN_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t NEITHER_GREATER_THAN_NOR_LESS_THAN_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2279 };
static wchar_t* NEITHER_GREATER_THAN_NOR_LESS_THAN_UNICODE_CHARACTER_CODE_MODEL = NEITHER_GREATER_THAN_NOR_LESS_THAN_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* NEITHER_GREATER_THAN_NOR_LESS_THAN_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t PRECEDES_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x227a };
static wchar_t* PRECEDES_UNICODE_CHARACTER_CODE_MODEL = PRECEDES_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* PRECEDES_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t SUCCEEDS_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x227b };
static wchar_t* SUCCEEDS_UNICODE_CHARACTER_CODE_MODEL = SUCCEEDS_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* SUCCEEDS_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t PRECEDES_OR_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x227c };
static wchar_t* PRECEDES_OR_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL = PRECEDES_OR_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* PRECEDES_OR_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t SUCCEEDS_OR_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x227d };
static wchar_t* SUCCEEDS_OR_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL = SUCCEEDS_OR_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* SUCCEEDS_OR_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t PRECEDES_OR_EQUIVALENT_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x227e };
static wchar_t* PRECEDES_OR_EQUIVALENT_TO_UNICODE_CHARACTER_CODE_MODEL = PRECEDES_OR_EQUIVALENT_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* PRECEDES_OR_EQUIVALENT_TO_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t SUCCEEDS_OR_EQUIVALENT_TO_WITH_SLASH_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x227f, 0x0338 };
static wchar_t* SUCCEEDS_OR_EQUIVALENT_TO_WITH_SLASH_UNICODE_CHARACTER_CODE_MODEL = SUCCEEDS_OR_EQUIVALENT_TO_WITH_SLASH_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* SUCCEEDS_OR_EQUIVALENT_TO_WITH_SLASH_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_2_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t SUCCEEDS_OR_EQUIVALENT_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x227f };
static wchar_t* SUCCEEDS_OR_EQUIVALENT_TO_UNICODE_CHARACTER_CODE_MODEL = SUCCEEDS_OR_EQUIVALENT_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* SUCCEEDS_OR_EQUIVALENT_TO_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t DOES_NOT_PRECEDE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2280 };
static wchar_t* DOES_NOT_PRECEDE_UNICODE_CHARACTER_CODE_MODEL = DOES_NOT_PRECEDE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* DOES_NOT_PRECEDE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t DOES_NOT_SUCCEED_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2281 };
static wchar_t* DOES_NOT_SUCCEED_UNICODE_CHARACTER_CODE_MODEL = DOES_NOT_SUCCEED_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* DOES_NOT_SUCCEED_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t SUBSET_OF_WITH_VERTICAL_LINE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2282, 0x20d2 };
static wchar_t* SUBSET_OF_WITH_VERTICAL_LINE_UNICODE_CHARACTER_CODE_MODEL = SUBSET_OF_WITH_VERTICAL_LINE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* SUBSET_OF_WITH_VERTICAL_LINE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_2_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t SUBSET_OF_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2282 };
static wchar_t* SUBSET_OF_UNICODE_CHARACTER_CODE_MODEL = SUBSET_OF_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* SUBSET_OF_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t SUPERSET_OF_WITH_VERTICAL_LINE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2283, 0x20d2 };
static wchar_t* SUPERSET_OF_WITH_VERTICAL_LINE_UNICODE_CHARACTER_CODE_MODEL = SUPERSET_OF_WITH_VERTICAL_LINE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* SUPERSET_OF_WITH_VERTICAL_LINE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_2_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t SUPERSET_OF_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2283 };
static wchar_t* SUPERSET_OF_UNICODE_CHARACTER_CODE_MODEL = SUPERSET_OF_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* SUPERSET_OF_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t NOT_A_SUBSET_OF_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2284 };
static wchar_t* NOT_A_SUBSET_OF_UNICODE_CHARACTER_CODE_MODEL = NOT_A_SUBSET_OF_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* NOT_A_SUBSET_OF_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t NOT_A_SUPERSET_OF_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2285 };
static wchar_t* NOT_A_SUPERSET_OF_UNICODE_CHARACTER_CODE_MODEL = NOT_A_SUPERSET_OF_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* NOT_A_SUPERSET_OF_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t SUBSET_OF_OR_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2286 };
static wchar_t* SUBSET_OF_OR_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL = SUBSET_OF_OR_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* SUBSET_OF_OR_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t SUPERSET_OF_OR_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2287 };
static wchar_t* SUPERSET_OF_OR_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL = SUPERSET_OF_OR_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* SUPERSET_OF_OR_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t NEITHER_A_SUBSET_OF_NOR_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2288 };
static wchar_t* NEITHER_A_SUBSET_OF_NOR_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL = NEITHER_A_SUBSET_OF_NOR_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* NEITHER_A_SUBSET_OF_NOR_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t NEITHER_A_SUPERSET_OF_NOR_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2289 };
static wchar_t* NEITHER_A_SUPERSET_OF_NOR_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL = NEITHER_A_SUPERSET_OF_NOR_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* NEITHER_A_SUPERSET_OF_NOR_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t SUBSET_OF_WITH_NOT_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x228a };
static wchar_t* SUBSET_OF_WITH_NOT_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL = SUBSET_OF_WITH_NOT_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* SUBSET_OF_WITH_NOT_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t SUBSET_OF_WITH_NOT_EQUAL_TO___VARIANT_WITH_STROKE_THROUGH_BOTTOM_MEMBERS_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x228a, 0xfe00 };
static wchar_t* SUBSET_OF_WITH_NOT_EQUAL_TO___VARIANT_WITH_STROKE_THROUGH_BOTTOM_MEMBERS_UNICODE_CHARACTER_CODE_MODEL = SUBSET_OF_WITH_NOT_EQUAL_TO___VARIANT_WITH_STROKE_THROUGH_BOTTOM_MEMBERS_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* SUBSET_OF_WITH_NOT_EQUAL_TO___VARIANT_WITH_STROKE_THROUGH_BOTTOM_MEMBERS_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_2_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t SUPERSET_OF_WITH_NOT_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x228b };
static wchar_t* SUPERSET_OF_WITH_NOT_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL = SUPERSET_OF_WITH_NOT_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* SUPERSET_OF_WITH_NOT_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t SUPERSET_OF_WITH_NOT_EQUAL_TO___VARIANT_WITH_STROKE_THROUGH_BOTTOM_MEMBERS_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x228b, 0xfe00 };
static wchar_t* SUPERSET_OF_WITH_NOT_EQUAL_TO___VARIANT_WITH_STROKE_THROUGH_BOTTOM_MEMBERS_UNICODE_CHARACTER_CODE_MODEL = SUPERSET_OF_WITH_NOT_EQUAL_TO___VARIANT_WITH_STROKE_THROUGH_BOTTOM_MEMBERS_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* SUPERSET_OF_WITH_NOT_EQUAL_TO___VARIANT_WITH_STROKE_THROUGH_BOTTOM_MEMBERS_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_2_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MULTISET_MULTIPLICATION_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x228d };
static wchar_t* MULTISET_MULTIPLICATION_UNICODE_CHARACTER_CODE_MODEL = MULTISET_MULTIPLICATION_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MULTISET_MULTIPLICATION_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MULTISET_UNION_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x228e };
static wchar_t* MULTISET_UNION_UNICODE_CHARACTER_CODE_MODEL = MULTISET_UNION_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MULTISET_UNION_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t SQUARE_IMAGE_OF_WITH_SLASH_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x228f, 0x0338 };
static wchar_t* SQUARE_IMAGE_OF_WITH_SLASH_UNICODE_CHARACTER_CODE_MODEL = SQUARE_IMAGE_OF_WITH_SLASH_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* SQUARE_IMAGE_OF_WITH_SLASH_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_2_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t SQUARE_IMAGE_OF_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x228f };
static wchar_t* SQUARE_IMAGE_OF_UNICODE_CHARACTER_CODE_MODEL = SQUARE_IMAGE_OF_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* SQUARE_IMAGE_OF_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t SQUARE_ORIGINAL_OF_WITH_SLASH_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2290, 0x0338 };
static wchar_t* SQUARE_ORIGINAL_OF_WITH_SLASH_UNICODE_CHARACTER_CODE_MODEL = SQUARE_ORIGINAL_OF_WITH_SLASH_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* SQUARE_ORIGINAL_OF_WITH_SLASH_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_2_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t SQUARE_ORIGINAL_OF_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2290 };
static wchar_t* SQUARE_ORIGINAL_OF_UNICODE_CHARACTER_CODE_MODEL = SQUARE_ORIGINAL_OF_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* SQUARE_ORIGINAL_OF_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t SQUARE_IMAGE_OF_OR_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2291 };
static wchar_t* SQUARE_IMAGE_OF_OR_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL = SQUARE_IMAGE_OF_OR_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* SQUARE_IMAGE_OF_OR_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t SQUARE_ORIGINAL_OF_OR_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2292 };
static wchar_t* SQUARE_ORIGINAL_OF_OR_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL = SQUARE_ORIGINAL_OF_OR_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* SQUARE_ORIGINAL_OF_OR_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t SQUARE_CAP_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2293 };
static wchar_t* SQUARE_CAP_UNICODE_CHARACTER_CODE_MODEL = SQUARE_CAP_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* SQUARE_CAP_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t SQUARE_CAP_WITH_SERIFS_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2293, 0xfe00 };
static wchar_t* SQUARE_CAP_WITH_SERIFS_UNICODE_CHARACTER_CODE_MODEL = SQUARE_CAP_WITH_SERIFS_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* SQUARE_CAP_WITH_SERIFS_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_2_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t SQUARE_CUP_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2294 };
static wchar_t* SQUARE_CUP_UNICODE_CHARACTER_CODE_MODEL = SQUARE_CUP_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* SQUARE_CUP_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t SQUARE_CUP_WITH_SERIFS_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2294, 0xfe00 };
static wchar_t* SQUARE_CUP_WITH_SERIFS_UNICODE_CHARACTER_CODE_MODEL = SQUARE_CUP_WITH_SERIFS_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* SQUARE_CUP_WITH_SERIFS_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_2_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t CIRCLED_PLUS_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2295 };
static wchar_t* CIRCLED_PLUS_UNICODE_CHARACTER_CODE_MODEL = CIRCLED_PLUS_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CIRCLED_PLUS_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t CIRCLED_MINUS_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2296 };
static wchar_t* CIRCLED_MINUS_UNICODE_CHARACTER_CODE_MODEL = CIRCLED_MINUS_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CIRCLED_MINUS_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t CIRCLED_TIMES_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2297 };
static wchar_t* CIRCLED_TIMES_UNICODE_CHARACTER_CODE_MODEL = CIRCLED_TIMES_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CIRCLED_TIMES_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t CIRCLED_DIVISION_SLASH_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2298 };
static wchar_t* CIRCLED_DIVISION_SLASH_UNICODE_CHARACTER_CODE_MODEL = CIRCLED_DIVISION_SLASH_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CIRCLED_DIVISION_SLASH_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t CIRCLED_DOT_OPERATOR_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2299 };
static wchar_t* CIRCLED_DOT_OPERATOR_UNICODE_CHARACTER_CODE_MODEL = CIRCLED_DOT_OPERATOR_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CIRCLED_DOT_OPERATOR_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t CIRCLED_RING_OPERATOR_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x229a };
static wchar_t* CIRCLED_RING_OPERATOR_UNICODE_CHARACTER_CODE_MODEL = CIRCLED_RING_OPERATOR_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CIRCLED_RING_OPERATOR_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t CIRCLED_ASTERISK_OPERATOR_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x229b };
static wchar_t* CIRCLED_ASTERISK_OPERATOR_UNICODE_CHARACTER_CODE_MODEL = CIRCLED_ASTERISK_OPERATOR_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CIRCLED_ASTERISK_OPERATOR_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t CIRCLED_DASH_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x229d };
static wchar_t* CIRCLED_DASH_UNICODE_CHARACTER_CODE_MODEL = CIRCLED_DASH_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CIRCLED_DASH_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t SQUARED_PLUS_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x229e };
static wchar_t* SQUARED_PLUS_UNICODE_CHARACTER_CODE_MODEL = SQUARED_PLUS_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* SQUARED_PLUS_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t SQUARED_MINUS_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x229f };
static wchar_t* SQUARED_MINUS_UNICODE_CHARACTER_CODE_MODEL = SQUARED_MINUS_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* SQUARED_MINUS_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t SQUARED_TIMES_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x22a0 };
static wchar_t* SQUARED_TIMES_UNICODE_CHARACTER_CODE_MODEL = SQUARED_TIMES_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* SQUARED_TIMES_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t SQUARED_DOT_OPERATOR_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x22a1 };
static wchar_t* SQUARED_DOT_OPERATOR_UNICODE_CHARACTER_CODE_MODEL = SQUARED_DOT_OPERATOR_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* SQUARED_DOT_OPERATOR_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t RIGHT_TACK_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x22a2 };
static wchar_t* RIGHT_TACK_UNICODE_CHARACTER_CODE_MODEL = RIGHT_TACK_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* RIGHT_TACK_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LEFT_TACK_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x22a3 };
static wchar_t* LEFT_TACK_UNICODE_CHARACTER_CODE_MODEL = LEFT_TACK_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LEFT_TACK_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t DOWN_TACK_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x22a4 };
static wchar_t* DOWN_TACK_UNICODE_CHARACTER_CODE_MODEL = DOWN_TACK_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* DOWN_TACK_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t UP_TACK_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x22a5 };
static wchar_t* UP_TACK_UNICODE_CHARACTER_CODE_MODEL = UP_TACK_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* UP_TACK_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MODELS_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x22a7 };
static wchar_t* MODELS_UNICODE_CHARACTER_CODE_MODEL = MODELS_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MODELS_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t TRUE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x22a8 };
static wchar_t* TRUE_UNICODE_CHARACTER_CODE_MODEL = TRUE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* TRUE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t FORCES_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x22a9 };
static wchar_t* FORCES_UNICODE_CHARACTER_CODE_MODEL = FORCES_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* FORCES_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t TRIPLE_VERTICAL_BAR_RIGHT_TURNSTILE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x22aa };
static wchar_t* TRIPLE_VERTICAL_BAR_RIGHT_TURNSTILE_UNICODE_CHARACTER_CODE_MODEL = TRIPLE_VERTICAL_BAR_RIGHT_TURNSTILE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* TRIPLE_VERTICAL_BAR_RIGHT_TURNSTILE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t DOUBLE_VERTICAL_BAR_DOUBLE_RIGHT_TURNSTILE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x22ab };
static wchar_t* DOUBLE_VERTICAL_BAR_DOUBLE_RIGHT_TURNSTILE_UNICODE_CHARACTER_CODE_MODEL = DOUBLE_VERTICAL_BAR_DOUBLE_RIGHT_TURNSTILE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* DOUBLE_VERTICAL_BAR_DOUBLE_RIGHT_TURNSTILE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t DOES_NOT_PROVE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x22ac };
static wchar_t* DOES_NOT_PROVE_UNICODE_CHARACTER_CODE_MODEL = DOES_NOT_PROVE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* DOES_NOT_PROVE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t NOT_TRUE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x22ad };
static wchar_t* NOT_TRUE_UNICODE_CHARACTER_CODE_MODEL = NOT_TRUE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* NOT_TRUE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t DOES_NOT_FORCE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x22ae };
static wchar_t* DOES_NOT_FORCE_UNICODE_CHARACTER_CODE_MODEL = DOES_NOT_FORCE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* DOES_NOT_FORCE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t NEGATED_DOUBLE_VERTICAL_BAR_DOUBLE_RIGHT_TURNSTILE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x22af };
static wchar_t* NEGATED_DOUBLE_VERTICAL_BAR_DOUBLE_RIGHT_TURNSTILE_UNICODE_CHARACTER_CODE_MODEL = NEGATED_DOUBLE_VERTICAL_BAR_DOUBLE_RIGHT_TURNSTILE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* NEGATED_DOUBLE_VERTICAL_BAR_DOUBLE_RIGHT_TURNSTILE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t PRECEDES_UNDER_RELATION_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x22b0 };
static wchar_t* PRECEDES_UNDER_RELATION_UNICODE_CHARACTER_CODE_MODEL = PRECEDES_UNDER_RELATION_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* PRECEDES_UNDER_RELATION_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t NORMAL_SUBGROUP_OF_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x22b2 };
static wchar_t* NORMAL_SUBGROUP_OF_UNICODE_CHARACTER_CODE_MODEL = NORMAL_SUBGROUP_OF_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* NORMAL_SUBGROUP_OF_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t CONTAINS_AS_NORMAL_SUBGROUP_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x22b3 };
static wchar_t* CONTAINS_AS_NORMAL_SUBGROUP_UNICODE_CHARACTER_CODE_MODEL = CONTAINS_AS_NORMAL_SUBGROUP_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CONTAINS_AS_NORMAL_SUBGROUP_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t NORMAL_SUBGROUP_OF_OR_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x22b4 };
static wchar_t* NORMAL_SUBGROUP_OF_OR_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL = NORMAL_SUBGROUP_OF_OR_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* NORMAL_SUBGROUP_OF_OR_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t NORMAL_SUBGROUP_OF_OR_EQUAL_TO_WITH_VERTICAL_LINE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x22b4, 0x20d2 };
static wchar_t* NORMAL_SUBGROUP_OF_OR_EQUAL_TO_WITH_VERTICAL_LINE_UNICODE_CHARACTER_CODE_MODEL = NORMAL_SUBGROUP_OF_OR_EQUAL_TO_WITH_VERTICAL_LINE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* NORMAL_SUBGROUP_OF_OR_EQUAL_TO_WITH_VERTICAL_LINE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_2_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t CONTAINS_AS_NORMAL_SUBGROUP_OR_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x22b5 };
static wchar_t* CONTAINS_AS_NORMAL_SUBGROUP_OR_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL = CONTAINS_AS_NORMAL_SUBGROUP_OR_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CONTAINS_AS_NORMAL_SUBGROUP_OR_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t CONTAINS_AS_NORMAL_SUBGROUP_OR_EQUAL_TO_WITH_VERTICAL_LINE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x22b5, 0x20d2 };
static wchar_t* CONTAINS_AS_NORMAL_SUBGROUP_OR_EQUAL_TO_WITH_VERTICAL_LINE_UNICODE_CHARACTER_CODE_MODEL = CONTAINS_AS_NORMAL_SUBGROUP_OR_EQUAL_TO_WITH_VERTICAL_LINE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CONTAINS_AS_NORMAL_SUBGROUP_OR_EQUAL_TO_WITH_VERTICAL_LINE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_2_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t ORIGINAL_OF_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x22b6 };
static wchar_t* ORIGINAL_OF_UNICODE_CHARACTER_CODE_MODEL = ORIGINAL_OF_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* ORIGINAL_OF_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t IMAGE_OF_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x22b7 };
static wchar_t* IMAGE_OF_UNICODE_CHARACTER_CODE_MODEL = IMAGE_OF_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* IMAGE_OF_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MULTIMAP_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x22b8 };
static wchar_t* MULTIMAP_UNICODE_CHARACTER_CODE_MODEL = MULTIMAP_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MULTIMAP_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t HERMITIAN_CONJUGATE_MATRIX_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x22b9 };
static wchar_t* HERMITIAN_CONJUGATE_MATRIX_UNICODE_CHARACTER_CODE_MODEL = HERMITIAN_CONJUGATE_MATRIX_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* HERMITIAN_CONJUGATE_MATRIX_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t INTERCALATE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x22ba };
static wchar_t* INTERCALATE_UNICODE_CHARACTER_CODE_MODEL = INTERCALATE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* INTERCALATE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t XOR_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x22bb };
static wchar_t* XOR_UNICODE_CHARACTER_CODE_MODEL = XOR_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* XOR_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t NOR_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x22bd };
static wchar_t* NOR_UNICODE_CHARACTER_CODE_MODEL = NOR_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* NOR_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t RIGHT_ANGLE_WITH_ARC_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x22be };
static wchar_t* RIGHT_ANGLE_WITH_ARC_UNICODE_CHARACTER_CODE_MODEL = RIGHT_ANGLE_WITH_ARC_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* RIGHT_ANGLE_WITH_ARC_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t RIGHT_TRIANGLE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x22bf };
static wchar_t* RIGHT_TRIANGLE_UNICODE_CHARACTER_CODE_MODEL = RIGHT_TRIANGLE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* RIGHT_TRIANGLE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t N_ARY_LOGICAL_AND_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x22c0 };
static wchar_t* N_ARY_LOGICAL_AND_UNICODE_CHARACTER_CODE_MODEL = N_ARY_LOGICAL_AND_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* N_ARY_LOGICAL_AND_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t N_ARY_LOGICAL_OR_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x22c1 };
static wchar_t* N_ARY_LOGICAL_OR_UNICODE_CHARACTER_CODE_MODEL = N_ARY_LOGICAL_OR_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* N_ARY_LOGICAL_OR_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t N_ARY_INTERSECTION_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x22c2 };
static wchar_t* N_ARY_INTERSECTION_UNICODE_CHARACTER_CODE_MODEL = N_ARY_INTERSECTION_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* N_ARY_INTERSECTION_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t N_ARY_UNION_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x22c3 };
static wchar_t* N_ARY_UNION_UNICODE_CHARACTER_CODE_MODEL = N_ARY_UNION_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* N_ARY_UNION_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t DIAMOND_OPERATOR_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x22c4 };
static wchar_t* DIAMOND_OPERATOR_UNICODE_CHARACTER_CODE_MODEL = DIAMOND_OPERATOR_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* DIAMOND_OPERATOR_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t DOT_OPERATOR_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x22c5 };
static wchar_t* DOT_OPERATOR_UNICODE_CHARACTER_CODE_MODEL = DOT_OPERATOR_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* DOT_OPERATOR_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t STAR_OPERATOR_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x22c6 };
static wchar_t* STAR_OPERATOR_UNICODE_CHARACTER_CODE_MODEL = STAR_OPERATOR_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* STAR_OPERATOR_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t DIVISION_TIMES_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x22c7 };
static wchar_t* DIVISION_TIMES_UNICODE_CHARACTER_CODE_MODEL = DIVISION_TIMES_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* DIVISION_TIMES_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t BOWTIE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x22c8 };
static wchar_t* BOWTIE_UNICODE_CHARACTER_CODE_MODEL = BOWTIE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* BOWTIE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LEFT_NORMAL_FACTOR_SEMIDIRECT_PRODUCT_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x22c9 };
static wchar_t* LEFT_NORMAL_FACTOR_SEMIDIRECT_PRODUCT_UNICODE_CHARACTER_CODE_MODEL = LEFT_NORMAL_FACTOR_SEMIDIRECT_PRODUCT_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LEFT_NORMAL_FACTOR_SEMIDIRECT_PRODUCT_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t RIGHT_NORMAL_FACTOR_SEMIDIRECT_PRODUCT_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x22ca };
static wchar_t* RIGHT_NORMAL_FACTOR_SEMIDIRECT_PRODUCT_UNICODE_CHARACTER_CODE_MODEL = RIGHT_NORMAL_FACTOR_SEMIDIRECT_PRODUCT_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* RIGHT_NORMAL_FACTOR_SEMIDIRECT_PRODUCT_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LEFT_SEMIDIRECT_PRODUCT_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x22cb };
static wchar_t* LEFT_SEMIDIRECT_PRODUCT_UNICODE_CHARACTER_CODE_MODEL = LEFT_SEMIDIRECT_PRODUCT_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LEFT_SEMIDIRECT_PRODUCT_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t RIGHT_SEMIDIRECT_PRODUCT_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x22cc };
static wchar_t* RIGHT_SEMIDIRECT_PRODUCT_UNICODE_CHARACTER_CODE_MODEL = RIGHT_SEMIDIRECT_PRODUCT_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* RIGHT_SEMIDIRECT_PRODUCT_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t REVERSED_TILDE_EQUALS_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x22cd };
static wchar_t* REVERSED_TILDE_EQUALS_UNICODE_CHARACTER_CODE_MODEL = REVERSED_TILDE_EQUALS_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* REVERSED_TILDE_EQUALS_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t CURLY_LOGICAL_OR_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x22ce };
static wchar_t* CURLY_LOGICAL_OR_UNICODE_CHARACTER_CODE_MODEL = CURLY_LOGICAL_OR_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CURLY_LOGICAL_OR_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t CURLY_LOGICAL_AND_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x22cf };
static wchar_t* CURLY_LOGICAL_AND_UNICODE_CHARACTER_CODE_MODEL = CURLY_LOGICAL_AND_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CURLY_LOGICAL_AND_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t DOUBLE_SUBSET_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x22d0 };
static wchar_t* DOUBLE_SUBSET_UNICODE_CHARACTER_CODE_MODEL = DOUBLE_SUBSET_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* DOUBLE_SUBSET_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t DOUBLE_SUPERSET_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x22d1 };
static wchar_t* DOUBLE_SUPERSET_UNICODE_CHARACTER_CODE_MODEL = DOUBLE_SUPERSET_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* DOUBLE_SUPERSET_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t DOUBLE_INTERSECTION_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x22d2 };
static wchar_t* DOUBLE_INTERSECTION_UNICODE_CHARACTER_CODE_MODEL = DOUBLE_INTERSECTION_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* DOUBLE_INTERSECTION_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t DOUBLE_UNION_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x22d3 };
static wchar_t* DOUBLE_UNION_UNICODE_CHARACTER_CODE_MODEL = DOUBLE_UNION_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* DOUBLE_UNION_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t PITCHFORK_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x22d4 };
static wchar_t* PITCHFORK_UNICODE_CHARACTER_CODE_MODEL = PITCHFORK_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* PITCHFORK_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t EQUAL_AND_PARALLEL_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x22d5 };
static wchar_t* EQUAL_AND_PARALLEL_TO_UNICODE_CHARACTER_CODE_MODEL = EQUAL_AND_PARALLEL_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* EQUAL_AND_PARALLEL_TO_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LESS_THAN_WITH_DOT_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x22d6 };
static wchar_t* LESS_THAN_WITH_DOT_UNICODE_CHARACTER_CODE_MODEL = LESS_THAN_WITH_DOT_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LESS_THAN_WITH_DOT_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t GREATER_THAN_WITH_DOT_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x22d7 };
static wchar_t* GREATER_THAN_WITH_DOT_UNICODE_CHARACTER_CODE_MODEL = GREATER_THAN_WITH_DOT_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* GREATER_THAN_WITH_DOT_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t VERY_MUCH_LESS_THAN_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x22d8 };
static wchar_t* VERY_MUCH_LESS_THAN_UNICODE_CHARACTER_CODE_MODEL = VERY_MUCH_LESS_THAN_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* VERY_MUCH_LESS_THAN_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t VERY_MUCH_LESS_THAN_WITH_SLASH_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x22d8, 0x0338 };
static wchar_t* VERY_MUCH_LESS_THAN_WITH_SLASH_UNICODE_CHARACTER_CODE_MODEL = VERY_MUCH_LESS_THAN_WITH_SLASH_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* VERY_MUCH_LESS_THAN_WITH_SLASH_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_2_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t VERY_MUCH_GREATER_THAN_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x22d9 };
static wchar_t* VERY_MUCH_GREATER_THAN_UNICODE_CHARACTER_CODE_MODEL = VERY_MUCH_GREATER_THAN_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* VERY_MUCH_GREATER_THAN_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t VERY_MUCH_GREATER_THAN_WITH_SLASH_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x22d9, 0x0338 };
static wchar_t* VERY_MUCH_GREATER_THAN_WITH_SLASH_UNICODE_CHARACTER_CODE_MODEL = VERY_MUCH_GREATER_THAN_WITH_SLASH_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* VERY_MUCH_GREATER_THAN_WITH_SLASH_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_2_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LESS_THAN_EQUAL_TO_OR_GREATER_THAN_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x22da };
static wchar_t* LESS_THAN_EQUAL_TO_OR_GREATER_THAN_UNICODE_CHARACTER_CODE_MODEL = LESS_THAN_EQUAL_TO_OR_GREATER_THAN_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LESS_THAN_EQUAL_TO_OR_GREATER_THAN_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LESS_THAN_SLANTED_EQUAL_TO_OR_GREATER_THAN_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x22da, 0xfe00 };
static wchar_t* LESS_THAN_SLANTED_EQUAL_TO_OR_GREATER_THAN_UNICODE_CHARACTER_CODE_MODEL = LESS_THAN_SLANTED_EQUAL_TO_OR_GREATER_THAN_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LESS_THAN_SLANTED_EQUAL_TO_OR_GREATER_THAN_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_2_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t GREATER_THAN_EQUAL_TO_OR_LESS_THAN_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x22db };
static wchar_t* GREATER_THAN_EQUAL_TO_OR_LESS_THAN_UNICODE_CHARACTER_CODE_MODEL = GREATER_THAN_EQUAL_TO_OR_LESS_THAN_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* GREATER_THAN_EQUAL_TO_OR_LESS_THAN_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t GREATER_THAN_SLANTED_EQUAL_TO_OR_LESS_THAN_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x22db, 0xfe00 };
static wchar_t* GREATER_THAN_SLANTED_EQUAL_TO_OR_LESS_THAN_UNICODE_CHARACTER_CODE_MODEL = GREATER_THAN_SLANTED_EQUAL_TO_OR_LESS_THAN_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* GREATER_THAN_SLANTED_EQUAL_TO_OR_LESS_THAN_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_2_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t EQUAL_TO_OR_PRECEDES_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x22de };
static wchar_t* EQUAL_TO_OR_PRECEDES_UNICODE_CHARACTER_CODE_MODEL = EQUAL_TO_OR_PRECEDES_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* EQUAL_TO_OR_PRECEDES_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t EQUAL_TO_OR_SUCCEEDS_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x22df };
static wchar_t* EQUAL_TO_OR_SUCCEEDS_UNICODE_CHARACTER_CODE_MODEL = EQUAL_TO_OR_SUCCEEDS_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* EQUAL_TO_OR_SUCCEEDS_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t DOES_NOT_PRECEDE_OR_EQUAL_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x22e0 };
static wchar_t* DOES_NOT_PRECEDE_OR_EQUAL_UNICODE_CHARACTER_CODE_MODEL = DOES_NOT_PRECEDE_OR_EQUAL_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* DOES_NOT_PRECEDE_OR_EQUAL_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t DOES_NOT_SUCCEED_OR_EQUAL_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x22e1 };
static wchar_t* DOES_NOT_SUCCEED_OR_EQUAL_UNICODE_CHARACTER_CODE_MODEL = DOES_NOT_SUCCEED_OR_EQUAL_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* DOES_NOT_SUCCEED_OR_EQUAL_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t NOT_SQUARE_IMAGE_OF_OR_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x22e2 };
static wchar_t* NOT_SQUARE_IMAGE_OF_OR_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL = NOT_SQUARE_IMAGE_OF_OR_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* NOT_SQUARE_IMAGE_OF_OR_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t NOT_SQUARE_ORIGINAL_OF_OR_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x22e3 };
static wchar_t* NOT_SQUARE_ORIGINAL_OF_OR_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL = NOT_SQUARE_ORIGINAL_OF_OR_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* NOT_SQUARE_ORIGINAL_OF_OR_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LESS_THAN_BUT_NOT_EQUIVALENT_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x22e6 };
static wchar_t* LESS_THAN_BUT_NOT_EQUIVALENT_TO_UNICODE_CHARACTER_CODE_MODEL = LESS_THAN_BUT_NOT_EQUIVALENT_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LESS_THAN_BUT_NOT_EQUIVALENT_TO_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t GREATER_THAN_BUT_NOT_EQUIVALENT_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x22e7 };
static wchar_t* GREATER_THAN_BUT_NOT_EQUIVALENT_TO_UNICODE_CHARACTER_CODE_MODEL = GREATER_THAN_BUT_NOT_EQUIVALENT_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* GREATER_THAN_BUT_NOT_EQUIVALENT_TO_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t PRECEDES_BUT_NOT_EQUIVALENT_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x22e8 };
static wchar_t* PRECEDES_BUT_NOT_EQUIVALENT_TO_UNICODE_CHARACTER_CODE_MODEL = PRECEDES_BUT_NOT_EQUIVALENT_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* PRECEDES_BUT_NOT_EQUIVALENT_TO_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t SUCCEEDS_BUT_NOT_EQUIVALENT_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x22e9 };
static wchar_t* SUCCEEDS_BUT_NOT_EQUIVALENT_TO_UNICODE_CHARACTER_CODE_MODEL = SUCCEEDS_BUT_NOT_EQUIVALENT_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* SUCCEEDS_BUT_NOT_EQUIVALENT_TO_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t NOT_NORMAL_SUBGROUP_OF_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x22ea };
static wchar_t* NOT_NORMAL_SUBGROUP_OF_UNICODE_CHARACTER_CODE_MODEL = NOT_NORMAL_SUBGROUP_OF_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* NOT_NORMAL_SUBGROUP_OF_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t DOES_NOT_CONTAIN_AS_NORMAL_SUBGROUP_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x22eb };
static wchar_t* DOES_NOT_CONTAIN_AS_NORMAL_SUBGROUP_UNICODE_CHARACTER_CODE_MODEL = DOES_NOT_CONTAIN_AS_NORMAL_SUBGROUP_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* DOES_NOT_CONTAIN_AS_NORMAL_SUBGROUP_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t NOT_NORMAL_SUBGROUP_OF_OR_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x22ec };
static wchar_t* NOT_NORMAL_SUBGROUP_OF_OR_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL = NOT_NORMAL_SUBGROUP_OF_OR_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* NOT_NORMAL_SUBGROUP_OF_OR_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t DOES_NOT_CONTAIN_AS_NORMAL_SUBGROUP_OR_EQUAL_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x22ed };
static wchar_t* DOES_NOT_CONTAIN_AS_NORMAL_SUBGROUP_OR_EQUAL_UNICODE_CHARACTER_CODE_MODEL = DOES_NOT_CONTAIN_AS_NORMAL_SUBGROUP_OR_EQUAL_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* DOES_NOT_CONTAIN_AS_NORMAL_SUBGROUP_OR_EQUAL_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t VERTICAL_ELLIPSIS_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x22ee };
static wchar_t* VERTICAL_ELLIPSIS_UNICODE_CHARACTER_CODE_MODEL = VERTICAL_ELLIPSIS_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* VERTICAL_ELLIPSIS_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MIDLINE_HORIZONTAL_ELLIPSIS_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x22ef };
static wchar_t* MIDLINE_HORIZONTAL_ELLIPSIS_UNICODE_CHARACTER_CODE_MODEL = MIDLINE_HORIZONTAL_ELLIPSIS_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MIDLINE_HORIZONTAL_ELLIPSIS_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t UP_RIGHT_DIAGONAL_ELLIPSIS_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x22f0 };
static wchar_t* UP_RIGHT_DIAGONAL_ELLIPSIS_UNICODE_CHARACTER_CODE_MODEL = UP_RIGHT_DIAGONAL_ELLIPSIS_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* UP_RIGHT_DIAGONAL_ELLIPSIS_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t DOWN_RIGHT_DIAGONAL_ELLIPSIS_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x22f1 };
static wchar_t* DOWN_RIGHT_DIAGONAL_ELLIPSIS_UNICODE_CHARACTER_CODE_MODEL = DOWN_RIGHT_DIAGONAL_ELLIPSIS_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* DOWN_RIGHT_DIAGONAL_ELLIPSIS_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t ELEMENT_OF_WITH_LONG_HORIZONTAL_STROKE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x22f2 };
static wchar_t* ELEMENT_OF_WITH_LONG_HORIZONTAL_STROKE_UNICODE_CHARACTER_CODE_MODEL = ELEMENT_OF_WITH_LONG_HORIZONTAL_STROKE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* ELEMENT_OF_WITH_LONG_HORIZONTAL_STROKE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t ELEMENT_OF_WITH_VERTICAL_BAR_AT_END_OF_HORIZONTAL_STROKE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x22f3 };
static wchar_t* ELEMENT_OF_WITH_VERTICAL_BAR_AT_END_OF_HORIZONTAL_STROKE_UNICODE_CHARACTER_CODE_MODEL = ELEMENT_OF_WITH_VERTICAL_BAR_AT_END_OF_HORIZONTAL_STROKE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* ELEMENT_OF_WITH_VERTICAL_BAR_AT_END_OF_HORIZONTAL_STROKE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t SMALL_ELEMENT_OF_WITH_VERTICAL_BAR_AT_END_OF_HORIZONTAL_STROKE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x22f4 };
static wchar_t* SMALL_ELEMENT_OF_WITH_VERTICAL_BAR_AT_END_OF_HORIZONTAL_STROKE_UNICODE_CHARACTER_CODE_MODEL = SMALL_ELEMENT_OF_WITH_VERTICAL_BAR_AT_END_OF_HORIZONTAL_STROKE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* SMALL_ELEMENT_OF_WITH_VERTICAL_BAR_AT_END_OF_HORIZONTAL_STROKE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t ELEMENT_OF_WITH_DOT_ABOVE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x22f5 };
static wchar_t* ELEMENT_OF_WITH_DOT_ABOVE_UNICODE_CHARACTER_CODE_MODEL = ELEMENT_OF_WITH_DOT_ABOVE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* ELEMENT_OF_WITH_DOT_ABOVE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t ELEMENT_OF_WITH_DOT_ABOVE_WITH_SLASH_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x22f5, 0x0338 };
static wchar_t* ELEMENT_OF_WITH_DOT_ABOVE_WITH_SLASH_UNICODE_CHARACTER_CODE_MODEL = ELEMENT_OF_WITH_DOT_ABOVE_WITH_SLASH_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* ELEMENT_OF_WITH_DOT_ABOVE_WITH_SLASH_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_2_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t ELEMENT_OF_WITH_OVERBAR_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x22f6 };
static wchar_t* ELEMENT_OF_WITH_OVERBAR_UNICODE_CHARACTER_CODE_MODEL = ELEMENT_OF_WITH_OVERBAR_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* ELEMENT_OF_WITH_OVERBAR_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t SMALL_ELEMENT_OF_WITH_OVERBAR_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x22f7 };
static wchar_t* SMALL_ELEMENT_OF_WITH_OVERBAR_UNICODE_CHARACTER_CODE_MODEL = SMALL_ELEMENT_OF_WITH_OVERBAR_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* SMALL_ELEMENT_OF_WITH_OVERBAR_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t ELEMENT_OF_WITH_TWO_HORIZONTAL_STROKES_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x22f9 };
static wchar_t* ELEMENT_OF_WITH_TWO_HORIZONTAL_STROKES_UNICODE_CHARACTER_CODE_MODEL = ELEMENT_OF_WITH_TWO_HORIZONTAL_STROKES_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* ELEMENT_OF_WITH_TWO_HORIZONTAL_STROKES_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t ELEMENT_OF_WITH_TWO_HORIZONTAL_STROKES_WITH_SLASH_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x22f9, 0x0338 };
static wchar_t* ELEMENT_OF_WITH_TWO_HORIZONTAL_STROKES_WITH_SLASH_UNICODE_CHARACTER_CODE_MODEL = ELEMENT_OF_WITH_TWO_HORIZONTAL_STROKES_WITH_SLASH_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* ELEMENT_OF_WITH_TWO_HORIZONTAL_STROKES_WITH_SLASH_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_2_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t CONTAINS_WITH_LONG_HORIZONTAL_STROKE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x22fa };
static wchar_t* CONTAINS_WITH_LONG_HORIZONTAL_STROKE_UNICODE_CHARACTER_CODE_MODEL = CONTAINS_WITH_LONG_HORIZONTAL_STROKE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CONTAINS_WITH_LONG_HORIZONTAL_STROKE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t CONTAINS_WITH_VERTICAL_BAR_AT_END_OF_HORIZONTAL_STROKE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x22fb };
static wchar_t* CONTAINS_WITH_VERTICAL_BAR_AT_END_OF_HORIZONTAL_STROKE_UNICODE_CHARACTER_CODE_MODEL = CONTAINS_WITH_VERTICAL_BAR_AT_END_OF_HORIZONTAL_STROKE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CONTAINS_WITH_VERTICAL_BAR_AT_END_OF_HORIZONTAL_STROKE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t SMALL_CONTAINS_WITH_VERTICAL_BAR_AT_END_OF_HORIZONTAL_STROKE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x22fc };
static wchar_t* SMALL_CONTAINS_WITH_VERTICAL_BAR_AT_END_OF_HORIZONTAL_STROKE_UNICODE_CHARACTER_CODE_MODEL = SMALL_CONTAINS_WITH_VERTICAL_BAR_AT_END_OF_HORIZONTAL_STROKE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* SMALL_CONTAINS_WITH_VERTICAL_BAR_AT_END_OF_HORIZONTAL_STROKE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t CONTAINS_WITH_OVERBAR_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x22fd };
static wchar_t* CONTAINS_WITH_OVERBAR_UNICODE_CHARACTER_CODE_MODEL = CONTAINS_WITH_OVERBAR_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CONTAINS_WITH_OVERBAR_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t SMALL_CONTAINS_WITH_OVERBAR_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x22fe };
static wchar_t* SMALL_CONTAINS_WITH_OVERBAR_UNICODE_CHARACTER_CODE_MODEL = SMALL_CONTAINS_WITH_OVERBAR_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* SMALL_CONTAINS_WITH_OVERBAR_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t PROJECTIVE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2305 };
static wchar_t* PROJECTIVE_UNICODE_CHARACTER_CODE_MODEL = PROJECTIVE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* PROJECTIVE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t PERSPECTIVE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2306 };
static wchar_t* PERSPECTIVE_UNICODE_CHARACTER_CODE_MODEL = PERSPECTIVE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* PERSPECTIVE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LEFT_CEILING_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2308 };
static wchar_t* LEFT_CEILING_UNICODE_CHARACTER_CODE_MODEL = LEFT_CEILING_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LEFT_CEILING_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t RIGHT_CEILING_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2309 };
static wchar_t* RIGHT_CEILING_UNICODE_CHARACTER_CODE_MODEL = RIGHT_CEILING_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* RIGHT_CEILING_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LEFT_FLOOR_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x230a };
static wchar_t* LEFT_FLOOR_UNICODE_CHARACTER_CODE_MODEL = LEFT_FLOOR_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LEFT_FLOOR_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t RIGHT_FLOOR_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x230b };
static wchar_t* RIGHT_FLOOR_UNICODE_CHARACTER_CODE_MODEL = RIGHT_FLOOR_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* RIGHT_FLOOR_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t BOTTOM_RIGHT_CROP_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x230c };
static wchar_t* BOTTOM_RIGHT_CROP_UNICODE_CHARACTER_CODE_MODEL = BOTTOM_RIGHT_CROP_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* BOTTOM_RIGHT_CROP_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t BOTTOM_LEFT_CROP_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x230d };
static wchar_t* BOTTOM_LEFT_CROP_UNICODE_CHARACTER_CODE_MODEL = BOTTOM_LEFT_CROP_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* BOTTOM_LEFT_CROP_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t TOP_RIGHT_CROP_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x230e };
static wchar_t* TOP_RIGHT_CROP_UNICODE_CHARACTER_CODE_MODEL = TOP_RIGHT_CROP_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* TOP_RIGHT_CROP_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t TOP_LEFT_CROP_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x230f };
static wchar_t* TOP_LEFT_CROP_UNICODE_CHARACTER_CODE_MODEL = TOP_LEFT_CROP_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* TOP_LEFT_CROP_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t REVERSED_NOT_SIGN_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2310 };
static wchar_t* REVERSED_NOT_SIGN_UNICODE_CHARACTER_CODE_MODEL = REVERSED_NOT_SIGN_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* REVERSED_NOT_SIGN_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t ARC_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2312 };
static wchar_t* ARC_UNICODE_CHARACTER_CODE_MODEL = ARC_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* ARC_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t SEGMENT_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2313 };
static wchar_t* SEGMENT_UNICODE_CHARACTER_CODE_MODEL = SEGMENT_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* SEGMENT_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t TELEPHONE_RECORDER_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2315 };
static wchar_t* TELEPHONE_RECORDER_UNICODE_CHARACTER_CODE_MODEL = TELEPHONE_RECORDER_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* TELEPHONE_RECORDER_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t POSITION_INDICATOR_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2316 };
static wchar_t* POSITION_INDICATOR_UNICODE_CHARACTER_CODE_MODEL = POSITION_INDICATOR_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* POSITION_INDICATOR_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t TOP_LEFT_CORNER_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x231c };
static wchar_t* TOP_LEFT_CORNER_UNICODE_CHARACTER_CODE_MODEL = TOP_LEFT_CORNER_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* TOP_LEFT_CORNER_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t TOP_RIGHT_CORNER_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x231d };
static wchar_t* TOP_RIGHT_CORNER_UNICODE_CHARACTER_CODE_MODEL = TOP_RIGHT_CORNER_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* TOP_RIGHT_CORNER_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t BOTTOM_LEFT_CORNER_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x231e };
static wchar_t* BOTTOM_LEFT_CORNER_UNICODE_CHARACTER_CODE_MODEL = BOTTOM_LEFT_CORNER_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* BOTTOM_LEFT_CORNER_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t BOTTOM_RIGHT_CORNER_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x231f };
static wchar_t* BOTTOM_RIGHT_CORNER_UNICODE_CHARACTER_CODE_MODEL = BOTTOM_RIGHT_CORNER_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* BOTTOM_RIGHT_CORNER_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t TOP_HALF_INTEGRAL_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2320 };
static wchar_t* TOP_HALF_INTEGRAL_UNICODE_CHARACTER_CODE_MODEL = TOP_HALF_INTEGRAL_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* TOP_HALF_INTEGRAL_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t BOTTOM_HALF_INTEGRAL_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2321 };
static wchar_t* BOTTOM_HALF_INTEGRAL_UNICODE_CHARACTER_CODE_MODEL = BOTTOM_HALF_INTEGRAL_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* BOTTOM_HALF_INTEGRAL_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t FROWN_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2322 };
static wchar_t* FROWN_UNICODE_CHARACTER_CODE_MODEL = FROWN_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* FROWN_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t SMILE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2323 };
static wchar_t* SMILE_UNICODE_CHARACTER_CODE_MODEL = SMILE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* SMILE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t CYLINDRICITY_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x232d };
static wchar_t* CYLINDRICITY_UNICODE_CHARACTER_CODE_MODEL = CYLINDRICITY_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CYLINDRICITY_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t ALL_AROUND_PROFILE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x232e };
static wchar_t* ALL_AROUND_PROFILE_UNICODE_CHARACTER_CODE_MODEL = ALL_AROUND_PROFILE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* ALL_AROUND_PROFILE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t APL_FUNCTIONAL_SYMBOL_I_BEAM_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2336 };
static wchar_t* APL_FUNCTIONAL_SYMBOL_I_BEAM_UNICODE_CHARACTER_CODE_MODEL = APL_FUNCTIONAL_SYMBOL_I_BEAM_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* APL_FUNCTIONAL_SYMBOL_I_BEAM_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t APL_FUNCTIONAL_SYMBOL_CIRCLE_STILE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x233d };
static wchar_t* APL_FUNCTIONAL_SYMBOL_CIRCLE_STILE_UNICODE_CHARACTER_CODE_MODEL = APL_FUNCTIONAL_SYMBOL_CIRCLE_STILE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* APL_FUNCTIONAL_SYMBOL_CIRCLE_STILE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t APL_FUNCTIONAL_SYMBOL_SLASH_BAR_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x233f };
static wchar_t* APL_FUNCTIONAL_SYMBOL_SLASH_BAR_UNICODE_CHARACTER_CODE_MODEL = APL_FUNCTIONAL_SYMBOL_SLASH_BAR_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* APL_FUNCTIONAL_SYMBOL_SLASH_BAR_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t RIGHT_ANGLE_WITH_DOWNWARDS_ZIGZAG_ARROW_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x237c };
static wchar_t* RIGHT_ANGLE_WITH_DOWNWARDS_ZIGZAG_ARROW_UNICODE_CHARACTER_CODE_MODEL = RIGHT_ANGLE_WITH_DOWNWARDS_ZIGZAG_ARROW_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* RIGHT_ANGLE_WITH_DOWNWARDS_ZIGZAG_ARROW_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t UPPER_LEFT_OR_LOWER_RIGHT_CURLY_BRACKET_SECTION_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x23b0 };
static wchar_t* UPPER_LEFT_OR_LOWER_RIGHT_CURLY_BRACKET_SECTION_UNICODE_CHARACTER_CODE_MODEL = UPPER_LEFT_OR_LOWER_RIGHT_CURLY_BRACKET_SECTION_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* UPPER_LEFT_OR_LOWER_RIGHT_CURLY_BRACKET_SECTION_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t UPPER_RIGHT_OR_LOWER_LEFT_CURLY_BRACKET_SECTION_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x23b1 };
static wchar_t* UPPER_RIGHT_OR_LOWER_LEFT_CURLY_BRACKET_SECTION_UNICODE_CHARACTER_CODE_MODEL = UPPER_RIGHT_OR_LOWER_LEFT_CURLY_BRACKET_SECTION_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* UPPER_RIGHT_OR_LOWER_LEFT_CURLY_BRACKET_SECTION_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t TOP_SQUARE_BRACKET_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x23b4 };
static wchar_t* TOP_SQUARE_BRACKET_UNICODE_CHARACTER_CODE_MODEL = TOP_SQUARE_BRACKET_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* TOP_SQUARE_BRACKET_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t BOTTOM_SQUARE_BRACKET_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x23b5 };
static wchar_t* BOTTOM_SQUARE_BRACKET_UNICODE_CHARACTER_CODE_MODEL = BOTTOM_SQUARE_BRACKET_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* BOTTOM_SQUARE_BRACKET_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t BOTTOM_SQUARE_BRACKET_OVER_TOP_SQUARE_BRACKET_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x23b6 };
static wchar_t* BOTTOM_SQUARE_BRACKET_OVER_TOP_SQUARE_BRACKET_UNICODE_CHARACTER_CODE_MODEL = BOTTOM_SQUARE_BRACKET_OVER_TOP_SQUARE_BRACKET_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* BOTTOM_SQUARE_BRACKET_OVER_TOP_SQUARE_BRACKET_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t TOP_PARENTHESIS_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x23dc };
static wchar_t* TOP_PARENTHESIS_UNICODE_CHARACTER_CODE_MODEL = TOP_PARENTHESIS_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* TOP_PARENTHESIS_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t BOTTOM_PARENTHESIS_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x23dd };
static wchar_t* BOTTOM_PARENTHESIS_UNICODE_CHARACTER_CODE_MODEL = BOTTOM_PARENTHESIS_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* BOTTOM_PARENTHESIS_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t TOP_CURLY_BRACKET_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x23de };
static wchar_t* TOP_CURLY_BRACKET_UNICODE_CHARACTER_CODE_MODEL = TOP_CURLY_BRACKET_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* TOP_CURLY_BRACKET_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t BOTTOM_CURLY_BRACKET_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x23df };
static wchar_t* BOTTOM_CURLY_BRACKET_UNICODE_CHARACTER_CODE_MODEL = BOTTOM_CURLY_BRACKET_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* BOTTOM_CURLY_BRACKET_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t WHITE_TRAPEZIUM_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x23e2 };
static wchar_t* WHITE_TRAPEZIUM_UNICODE_CHARACTER_CODE_MODEL = WHITE_TRAPEZIUM_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* WHITE_TRAPEZIUM_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t ELECTRICAL_INTERSECTION_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x23e7 };
static wchar_t* ELECTRICAL_INTERSECTION_UNICODE_CHARACTER_CODE_MODEL = ELECTRICAL_INTERSECTION_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* ELECTRICAL_INTERSECTION_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t OPEN_BOX_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2423 };
static wchar_t* OPEN_BOX_UNICODE_CHARACTER_CODE_MODEL = OPEN_BOX_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* OPEN_BOX_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t CIRCLED_LATIN_CAPITAL_LETTER_S_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x24c8 };
static wchar_t* CIRCLED_LATIN_CAPITAL_LETTER_S_UNICODE_CHARACTER_CODE_MODEL = CIRCLED_LATIN_CAPITAL_LETTER_S_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CIRCLED_LATIN_CAPITAL_LETTER_S_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t BOX_DRAWINGS_LIGHT_HORIZONTAL_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2500 };
static wchar_t* BOX_DRAWINGS_LIGHT_HORIZONTAL_UNICODE_CHARACTER_CODE_MODEL = BOX_DRAWINGS_LIGHT_HORIZONTAL_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* BOX_DRAWINGS_LIGHT_HORIZONTAL_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t BOX_DRAWINGS_LIGHT_VERTICAL_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2502 };
static wchar_t* BOX_DRAWINGS_LIGHT_VERTICAL_UNICODE_CHARACTER_CODE_MODEL = BOX_DRAWINGS_LIGHT_VERTICAL_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* BOX_DRAWINGS_LIGHT_VERTICAL_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t BOX_DRAWINGS_LIGHT_DOWN_AND_RIGHT_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x250c };
static wchar_t* BOX_DRAWINGS_LIGHT_DOWN_AND_RIGHT_UNICODE_CHARACTER_CODE_MODEL = BOX_DRAWINGS_LIGHT_DOWN_AND_RIGHT_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* BOX_DRAWINGS_LIGHT_DOWN_AND_RIGHT_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t BOX_DRAWINGS_LIGHT_DOWN_AND_LEFT_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2510 };
static wchar_t* BOX_DRAWINGS_LIGHT_DOWN_AND_LEFT_UNICODE_CHARACTER_CODE_MODEL = BOX_DRAWINGS_LIGHT_DOWN_AND_LEFT_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* BOX_DRAWINGS_LIGHT_DOWN_AND_LEFT_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t BOX_DRAWINGS_LIGHT_UP_AND_RIGHT_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2514 };
static wchar_t* BOX_DRAWINGS_LIGHT_UP_AND_RIGHT_UNICODE_CHARACTER_CODE_MODEL = BOX_DRAWINGS_LIGHT_UP_AND_RIGHT_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* BOX_DRAWINGS_LIGHT_UP_AND_RIGHT_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t BOX_DRAWINGS_LIGHT_UP_AND_LEFT_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2518 };
static wchar_t* BOX_DRAWINGS_LIGHT_UP_AND_LEFT_UNICODE_CHARACTER_CODE_MODEL = BOX_DRAWINGS_LIGHT_UP_AND_LEFT_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* BOX_DRAWINGS_LIGHT_UP_AND_LEFT_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t BOX_DRAWINGS_LIGHT_VERTICAL_AND_RIGHT_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x251c };
static wchar_t* BOX_DRAWINGS_LIGHT_VERTICAL_AND_RIGHT_UNICODE_CHARACTER_CODE_MODEL = BOX_DRAWINGS_LIGHT_VERTICAL_AND_RIGHT_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* BOX_DRAWINGS_LIGHT_VERTICAL_AND_RIGHT_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t BOX_DRAWINGS_LIGHT_VERTICAL_AND_LEFT_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2524 };
static wchar_t* BOX_DRAWINGS_LIGHT_VERTICAL_AND_LEFT_UNICODE_CHARACTER_CODE_MODEL = BOX_DRAWINGS_LIGHT_VERTICAL_AND_LEFT_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* BOX_DRAWINGS_LIGHT_VERTICAL_AND_LEFT_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t BOX_DRAWINGS_LIGHT_DOWN_AND_HORIZONTAL_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x252c };
static wchar_t* BOX_DRAWINGS_LIGHT_DOWN_AND_HORIZONTAL_UNICODE_CHARACTER_CODE_MODEL = BOX_DRAWINGS_LIGHT_DOWN_AND_HORIZONTAL_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* BOX_DRAWINGS_LIGHT_DOWN_AND_HORIZONTAL_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t BOX_DRAWINGS_LIGHT_UP_AND_HORIZONTAL_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2534 };
static wchar_t* BOX_DRAWINGS_LIGHT_UP_AND_HORIZONTAL_UNICODE_CHARACTER_CODE_MODEL = BOX_DRAWINGS_LIGHT_UP_AND_HORIZONTAL_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* BOX_DRAWINGS_LIGHT_UP_AND_HORIZONTAL_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t BOX_DRAWINGS_LIGHT_VERTICAL_AND_HORIZONTAL_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x253c };
static wchar_t* BOX_DRAWINGS_LIGHT_VERTICAL_AND_HORIZONTAL_UNICODE_CHARACTER_CODE_MODEL = BOX_DRAWINGS_LIGHT_VERTICAL_AND_HORIZONTAL_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* BOX_DRAWINGS_LIGHT_VERTICAL_AND_HORIZONTAL_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t BOX_DRAWINGS_DOUBLE_HORIZONTAL_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2550 };
static wchar_t* BOX_DRAWINGS_DOUBLE_HORIZONTAL_UNICODE_CHARACTER_CODE_MODEL = BOX_DRAWINGS_DOUBLE_HORIZONTAL_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* BOX_DRAWINGS_DOUBLE_HORIZONTAL_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t BOX_DRAWINGS_DOUBLE_VERTICAL_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2551 };
static wchar_t* BOX_DRAWINGS_DOUBLE_VERTICAL_UNICODE_CHARACTER_CODE_MODEL = BOX_DRAWINGS_DOUBLE_VERTICAL_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* BOX_DRAWINGS_DOUBLE_VERTICAL_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t BOX_DRAWINGS_DOWN_SINGLE_AND_RIGHT_DOUBLE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2552 };
static wchar_t* BOX_DRAWINGS_DOWN_SINGLE_AND_RIGHT_DOUBLE_UNICODE_CHARACTER_CODE_MODEL = BOX_DRAWINGS_DOWN_SINGLE_AND_RIGHT_DOUBLE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* BOX_DRAWINGS_DOWN_SINGLE_AND_RIGHT_DOUBLE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t BOX_DRAWINGS_DOWN_DOUBLE_AND_RIGHT_SINGLE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2553 };
static wchar_t* BOX_DRAWINGS_DOWN_DOUBLE_AND_RIGHT_SINGLE_UNICODE_CHARACTER_CODE_MODEL = BOX_DRAWINGS_DOWN_DOUBLE_AND_RIGHT_SINGLE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* BOX_DRAWINGS_DOWN_DOUBLE_AND_RIGHT_SINGLE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t BOX_DRAWINGS_DOUBLE_DOWN_AND_RIGHT_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2554 };
static wchar_t* BOX_DRAWINGS_DOUBLE_DOWN_AND_RIGHT_UNICODE_CHARACTER_CODE_MODEL = BOX_DRAWINGS_DOUBLE_DOWN_AND_RIGHT_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* BOX_DRAWINGS_DOUBLE_DOWN_AND_RIGHT_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t BOX_DRAWINGS_DOWN_SINGLE_AND_LEFT_DOUBLE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2555 };
static wchar_t* BOX_DRAWINGS_DOWN_SINGLE_AND_LEFT_DOUBLE_UNICODE_CHARACTER_CODE_MODEL = BOX_DRAWINGS_DOWN_SINGLE_AND_LEFT_DOUBLE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* BOX_DRAWINGS_DOWN_SINGLE_AND_LEFT_DOUBLE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t BOX_DRAWINGS_DOWN_DOUBLE_AND_LEFT_SINGLE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2556 };
static wchar_t* BOX_DRAWINGS_DOWN_DOUBLE_AND_LEFT_SINGLE_UNICODE_CHARACTER_CODE_MODEL = BOX_DRAWINGS_DOWN_DOUBLE_AND_LEFT_SINGLE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* BOX_DRAWINGS_DOWN_DOUBLE_AND_LEFT_SINGLE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t BOX_DRAWINGS_DOUBLE_DOWN_AND_LEFT_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2557 };
static wchar_t* BOX_DRAWINGS_DOUBLE_DOWN_AND_LEFT_UNICODE_CHARACTER_CODE_MODEL = BOX_DRAWINGS_DOUBLE_DOWN_AND_LEFT_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* BOX_DRAWINGS_DOUBLE_DOWN_AND_LEFT_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t BOX_DRAWINGS_UP_SINGLE_AND_RIGHT_DOUBLE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2558 };
static wchar_t* BOX_DRAWINGS_UP_SINGLE_AND_RIGHT_DOUBLE_UNICODE_CHARACTER_CODE_MODEL = BOX_DRAWINGS_UP_SINGLE_AND_RIGHT_DOUBLE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* BOX_DRAWINGS_UP_SINGLE_AND_RIGHT_DOUBLE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t BOX_DRAWINGS_UP_DOUBLE_AND_RIGHT_SINGLE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2559 };
static wchar_t* BOX_DRAWINGS_UP_DOUBLE_AND_RIGHT_SINGLE_UNICODE_CHARACTER_CODE_MODEL = BOX_DRAWINGS_UP_DOUBLE_AND_RIGHT_SINGLE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* BOX_DRAWINGS_UP_DOUBLE_AND_RIGHT_SINGLE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t BOX_DRAWINGS_DOUBLE_UP_AND_RIGHT_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x255a };
static wchar_t* BOX_DRAWINGS_DOUBLE_UP_AND_RIGHT_UNICODE_CHARACTER_CODE_MODEL = BOX_DRAWINGS_DOUBLE_UP_AND_RIGHT_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* BOX_DRAWINGS_DOUBLE_UP_AND_RIGHT_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t BOX_DRAWINGS_UP_SINGLE_AND_LEFT_DOUBLE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x255b };
static wchar_t* BOX_DRAWINGS_UP_SINGLE_AND_LEFT_DOUBLE_UNICODE_CHARACTER_CODE_MODEL = BOX_DRAWINGS_UP_SINGLE_AND_LEFT_DOUBLE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* BOX_DRAWINGS_UP_SINGLE_AND_LEFT_DOUBLE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t BOX_DRAWINGS_UP_DOUBLE_AND_LEFT_SINGLE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x255c };
static wchar_t* BOX_DRAWINGS_UP_DOUBLE_AND_LEFT_SINGLE_UNICODE_CHARACTER_CODE_MODEL = BOX_DRAWINGS_UP_DOUBLE_AND_LEFT_SINGLE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* BOX_DRAWINGS_UP_DOUBLE_AND_LEFT_SINGLE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t BOX_DRAWINGS_DOUBLE_UP_AND_LEFT_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x255d };
static wchar_t* BOX_DRAWINGS_DOUBLE_UP_AND_LEFT_UNICODE_CHARACTER_CODE_MODEL = BOX_DRAWINGS_DOUBLE_UP_AND_LEFT_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* BOX_DRAWINGS_DOUBLE_UP_AND_LEFT_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t BOX_DRAWINGS_VERTICAL_SINGLE_AND_RIGHT_DOUBLE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x255e };
static wchar_t* BOX_DRAWINGS_VERTICAL_SINGLE_AND_RIGHT_DOUBLE_UNICODE_CHARACTER_CODE_MODEL = BOX_DRAWINGS_VERTICAL_SINGLE_AND_RIGHT_DOUBLE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* BOX_DRAWINGS_VERTICAL_SINGLE_AND_RIGHT_DOUBLE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t BOX_DRAWINGS_VERTICAL_DOUBLE_AND_RIGHT_SINGLE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x255f };
static wchar_t* BOX_DRAWINGS_VERTICAL_DOUBLE_AND_RIGHT_SINGLE_UNICODE_CHARACTER_CODE_MODEL = BOX_DRAWINGS_VERTICAL_DOUBLE_AND_RIGHT_SINGLE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* BOX_DRAWINGS_VERTICAL_DOUBLE_AND_RIGHT_SINGLE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t BOX_DRAWINGS_DOUBLE_VERTICAL_AND_RIGHT_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2560 };
static wchar_t* BOX_DRAWINGS_DOUBLE_VERTICAL_AND_RIGHT_UNICODE_CHARACTER_CODE_MODEL = BOX_DRAWINGS_DOUBLE_VERTICAL_AND_RIGHT_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* BOX_DRAWINGS_DOUBLE_VERTICAL_AND_RIGHT_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t BOX_DRAWINGS_VERTICAL_SINGLE_AND_LEFT_DOUBLE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2561 };
static wchar_t* BOX_DRAWINGS_VERTICAL_SINGLE_AND_LEFT_DOUBLE_UNICODE_CHARACTER_CODE_MODEL = BOX_DRAWINGS_VERTICAL_SINGLE_AND_LEFT_DOUBLE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* BOX_DRAWINGS_VERTICAL_SINGLE_AND_LEFT_DOUBLE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t BOX_DRAWINGS_VERTICAL_DOUBLE_AND_LEFT_SINGLE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2562 };
static wchar_t* BOX_DRAWINGS_VERTICAL_DOUBLE_AND_LEFT_SINGLE_UNICODE_CHARACTER_CODE_MODEL = BOX_DRAWINGS_VERTICAL_DOUBLE_AND_LEFT_SINGLE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* BOX_DRAWINGS_VERTICAL_DOUBLE_AND_LEFT_SINGLE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t BOX_DRAWINGS_DOUBLE_VERTICAL_AND_LEFT_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2563 };
static wchar_t* BOX_DRAWINGS_DOUBLE_VERTICAL_AND_LEFT_UNICODE_CHARACTER_CODE_MODEL = BOX_DRAWINGS_DOUBLE_VERTICAL_AND_LEFT_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* BOX_DRAWINGS_DOUBLE_VERTICAL_AND_LEFT_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t BOX_DRAWINGS_DOWN_SINGLE_AND_HORIZONTAL_DOUBLE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2564 };
static wchar_t* BOX_DRAWINGS_DOWN_SINGLE_AND_HORIZONTAL_DOUBLE_UNICODE_CHARACTER_CODE_MODEL = BOX_DRAWINGS_DOWN_SINGLE_AND_HORIZONTAL_DOUBLE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* BOX_DRAWINGS_DOWN_SINGLE_AND_HORIZONTAL_DOUBLE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t BOX_DRAWINGS_DOWN_DOUBLE_AND_HORIZONTAL_SINGLE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2565 };
static wchar_t* BOX_DRAWINGS_DOWN_DOUBLE_AND_HORIZONTAL_SINGLE_UNICODE_CHARACTER_CODE_MODEL = BOX_DRAWINGS_DOWN_DOUBLE_AND_HORIZONTAL_SINGLE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* BOX_DRAWINGS_DOWN_DOUBLE_AND_HORIZONTAL_SINGLE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t BOX_DRAWINGS_DOUBLE_DOWN_AND_HORIZONTAL_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2566 };
static wchar_t* BOX_DRAWINGS_DOUBLE_DOWN_AND_HORIZONTAL_UNICODE_CHARACTER_CODE_MODEL = BOX_DRAWINGS_DOUBLE_DOWN_AND_HORIZONTAL_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* BOX_DRAWINGS_DOUBLE_DOWN_AND_HORIZONTAL_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t BOX_DRAWINGS_UP_SINGLE_AND_HORIZONTAL_DOUBLE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2567 };
static wchar_t* BOX_DRAWINGS_UP_SINGLE_AND_HORIZONTAL_DOUBLE_UNICODE_CHARACTER_CODE_MODEL = BOX_DRAWINGS_UP_SINGLE_AND_HORIZONTAL_DOUBLE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* BOX_DRAWINGS_UP_SINGLE_AND_HORIZONTAL_DOUBLE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t BOX_DRAWINGS_UP_DOUBLE_AND_HORIZONTAL_SINGLE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2568 };
static wchar_t* BOX_DRAWINGS_UP_DOUBLE_AND_HORIZONTAL_SINGLE_UNICODE_CHARACTER_CODE_MODEL = BOX_DRAWINGS_UP_DOUBLE_AND_HORIZONTAL_SINGLE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* BOX_DRAWINGS_UP_DOUBLE_AND_HORIZONTAL_SINGLE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t BOX_DRAWINGS_DOUBLE_UP_AND_HORIZONTAL_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2569 };
static wchar_t* BOX_DRAWINGS_DOUBLE_UP_AND_HORIZONTAL_UNICODE_CHARACTER_CODE_MODEL = BOX_DRAWINGS_DOUBLE_UP_AND_HORIZONTAL_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* BOX_DRAWINGS_DOUBLE_UP_AND_HORIZONTAL_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t BOX_DRAWINGS_VERTICAL_SINGLE_AND_HORIZONTAL_DOUBLE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x256a };
static wchar_t* BOX_DRAWINGS_VERTICAL_SINGLE_AND_HORIZONTAL_DOUBLE_UNICODE_CHARACTER_CODE_MODEL = BOX_DRAWINGS_VERTICAL_SINGLE_AND_HORIZONTAL_DOUBLE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* BOX_DRAWINGS_VERTICAL_SINGLE_AND_HORIZONTAL_DOUBLE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t BOX_DRAWINGS_VERTICAL_DOUBLE_AND_HORIZONTAL_SINGLE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x256b };
static wchar_t* BOX_DRAWINGS_VERTICAL_DOUBLE_AND_HORIZONTAL_SINGLE_UNICODE_CHARACTER_CODE_MODEL = BOX_DRAWINGS_VERTICAL_DOUBLE_AND_HORIZONTAL_SINGLE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* BOX_DRAWINGS_VERTICAL_DOUBLE_AND_HORIZONTAL_SINGLE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t BOX_DRAWINGS_DOUBLE_VERTICAL_AND_HORIZONTAL_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x256c };
static wchar_t* BOX_DRAWINGS_DOUBLE_VERTICAL_AND_HORIZONTAL_UNICODE_CHARACTER_CODE_MODEL = BOX_DRAWINGS_DOUBLE_VERTICAL_AND_HORIZONTAL_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* BOX_DRAWINGS_DOUBLE_VERTICAL_AND_HORIZONTAL_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t UPPER_HALF_BLOCK_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2580 };
static wchar_t* UPPER_HALF_BLOCK_UNICODE_CHARACTER_CODE_MODEL = UPPER_HALF_BLOCK_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* UPPER_HALF_BLOCK_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LOWER_HALF_BLOCK_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2584 };
static wchar_t* LOWER_HALF_BLOCK_UNICODE_CHARACTER_CODE_MODEL = LOWER_HALF_BLOCK_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LOWER_HALF_BLOCK_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t FULL_BLOCK_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2588 };
static wchar_t* FULL_BLOCK_UNICODE_CHARACTER_CODE_MODEL = FULL_BLOCK_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* FULL_BLOCK_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LEFT_HALF_BLOCK_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x258c };
static wchar_t* LEFT_HALF_BLOCK_UNICODE_CHARACTER_CODE_MODEL = LEFT_HALF_BLOCK_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LEFT_HALF_BLOCK_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t RIGHT_HALF_BLOCK_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2590 };
static wchar_t* RIGHT_HALF_BLOCK_UNICODE_CHARACTER_CODE_MODEL = RIGHT_HALF_BLOCK_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* RIGHT_HALF_BLOCK_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LIGHT_SHADE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2591 };
static wchar_t* LIGHT_SHADE_UNICODE_CHARACTER_CODE_MODEL = LIGHT_SHADE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LIGHT_SHADE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MEDIUM_SHADE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2592 };
static wchar_t* MEDIUM_SHADE_UNICODE_CHARACTER_CODE_MODEL = MEDIUM_SHADE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MEDIUM_SHADE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t DARK_SHADE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2593 };
static wchar_t* DARK_SHADE_UNICODE_CHARACTER_CODE_MODEL = DARK_SHADE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* DARK_SHADE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t BLACK_SQUARE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x25a0 };
static wchar_t* BLACK_SQUARE_UNICODE_CHARACTER_CODE_MODEL = BLACK_SQUARE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* BLACK_SQUARE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t WHITE_SQUARE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x25a1 };
static wchar_t* WHITE_SQUARE_UNICODE_CHARACTER_CODE_MODEL = WHITE_SQUARE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* WHITE_SQUARE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t BLACK_SMALL_SQUARE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x25aa };
static wchar_t* BLACK_SMALL_SQUARE_UNICODE_CHARACTER_CODE_MODEL = BLACK_SMALL_SQUARE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* BLACK_SMALL_SQUARE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t WHITE_SMALL_SQUARE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x25ab };
static wchar_t* WHITE_SMALL_SQUARE_UNICODE_CHARACTER_CODE_MODEL = WHITE_SMALL_SQUARE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* WHITE_SMALL_SQUARE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t WHITE_RECTANGLE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x25ad };
static wchar_t* WHITE_RECTANGLE_UNICODE_CHARACTER_CODE_MODEL = WHITE_RECTANGLE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* WHITE_RECTANGLE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t BLACK_VERTICAL_RECTANGLE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x25ae };
static wchar_t* BLACK_VERTICAL_RECTANGLE_UNICODE_CHARACTER_CODE_MODEL = BLACK_VERTICAL_RECTANGLE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* BLACK_VERTICAL_RECTANGLE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t WHITE_PARALLELOGRAM_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x25b1 };
static wchar_t* WHITE_PARALLELOGRAM_UNICODE_CHARACTER_CODE_MODEL = WHITE_PARALLELOGRAM_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* WHITE_PARALLELOGRAM_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t WHITE_UP_POINTING_TRIANGLE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x25b3 };
static wchar_t* WHITE_UP_POINTING_TRIANGLE_UNICODE_CHARACTER_CODE_MODEL = WHITE_UP_POINTING_TRIANGLE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* WHITE_UP_POINTING_TRIANGLE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t BLACK_UP_POINTING_SMALL_TRIANGLE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x25b4 };
static wchar_t* BLACK_UP_POINTING_SMALL_TRIANGLE_UNICODE_CHARACTER_CODE_MODEL = BLACK_UP_POINTING_SMALL_TRIANGLE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* BLACK_UP_POINTING_SMALL_TRIANGLE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t WHITE_UP_POINTING_SMALL_TRIANGLE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x25b5 };
static wchar_t* WHITE_UP_POINTING_SMALL_TRIANGLE_UNICODE_CHARACTER_CODE_MODEL = WHITE_UP_POINTING_SMALL_TRIANGLE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* WHITE_UP_POINTING_SMALL_TRIANGLE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t BLACK_RIGHT_POINTING_SMALL_TRIANGLE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x25b8 };
static wchar_t* BLACK_RIGHT_POINTING_SMALL_TRIANGLE_UNICODE_CHARACTER_CODE_MODEL = BLACK_RIGHT_POINTING_SMALL_TRIANGLE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* BLACK_RIGHT_POINTING_SMALL_TRIANGLE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t WHITE_RIGHT_POINTING_SMALL_TRIANGLE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x25b9 };
static wchar_t* WHITE_RIGHT_POINTING_SMALL_TRIANGLE_UNICODE_CHARACTER_CODE_MODEL = WHITE_RIGHT_POINTING_SMALL_TRIANGLE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* WHITE_RIGHT_POINTING_SMALL_TRIANGLE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t WHITE_DOWN_POINTING_TRIANGLE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x25bd };
static wchar_t* WHITE_DOWN_POINTING_TRIANGLE_UNICODE_CHARACTER_CODE_MODEL = WHITE_DOWN_POINTING_TRIANGLE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* WHITE_DOWN_POINTING_TRIANGLE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t BLACK_DOWN_POINTING_SMALL_TRIANGLE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x25be };
static wchar_t* BLACK_DOWN_POINTING_SMALL_TRIANGLE_UNICODE_CHARACTER_CODE_MODEL = BLACK_DOWN_POINTING_SMALL_TRIANGLE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* BLACK_DOWN_POINTING_SMALL_TRIANGLE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t WHITE_DOWN_POINTING_SMALL_TRIANGLE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x25bf };
static wchar_t* WHITE_DOWN_POINTING_SMALL_TRIANGLE_UNICODE_CHARACTER_CODE_MODEL = WHITE_DOWN_POINTING_SMALL_TRIANGLE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* WHITE_DOWN_POINTING_SMALL_TRIANGLE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t BLACK_LEFT_POINTING_SMALL_TRIANGLE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x25c2 };
static wchar_t* BLACK_LEFT_POINTING_SMALL_TRIANGLE_UNICODE_CHARACTER_CODE_MODEL = BLACK_LEFT_POINTING_SMALL_TRIANGLE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* BLACK_LEFT_POINTING_SMALL_TRIANGLE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t WHITE_LEFT_POINTING_SMALL_TRIANGLE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x25c3 };
static wchar_t* WHITE_LEFT_POINTING_SMALL_TRIANGLE_UNICODE_CHARACTER_CODE_MODEL = WHITE_LEFT_POINTING_SMALL_TRIANGLE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* WHITE_LEFT_POINTING_SMALL_TRIANGLE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LOZENGE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x25ca };
static wchar_t* LOZENGE_UNICODE_CHARACTER_CODE_MODEL = LOZENGE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LOZENGE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t WHITE_CIRCLE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x25cb };
static wchar_t* WHITE_CIRCLE_UNICODE_CHARACTER_CODE_MODEL = WHITE_CIRCLE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* WHITE_CIRCLE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t WHITE_UP_POINTING_TRIANGLE_WITH_DOT_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x25ec };
static wchar_t* WHITE_UP_POINTING_TRIANGLE_WITH_DOT_UNICODE_CHARACTER_CODE_MODEL = WHITE_UP_POINTING_TRIANGLE_WITH_DOT_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* WHITE_UP_POINTING_TRIANGLE_WITH_DOT_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LARGE_CIRCLE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x25ef };
static wchar_t* LARGE_CIRCLE_UNICODE_CHARACTER_CODE_MODEL = LARGE_CIRCLE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LARGE_CIRCLE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t UPPER_LEFT_TRIANGLE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x25f8 };
static wchar_t* UPPER_LEFT_TRIANGLE_UNICODE_CHARACTER_CODE_MODEL = UPPER_LEFT_TRIANGLE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* UPPER_LEFT_TRIANGLE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t UPPER_RIGHT_TRIANGLE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x25f9 };
static wchar_t* UPPER_RIGHT_TRIANGLE_UNICODE_CHARACTER_CODE_MODEL = UPPER_RIGHT_TRIANGLE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* UPPER_RIGHT_TRIANGLE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LOWER_LEFT_TRIANGLE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x25fa };
static wchar_t* LOWER_LEFT_TRIANGLE_UNICODE_CHARACTER_CODE_MODEL = LOWER_LEFT_TRIANGLE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LOWER_LEFT_TRIANGLE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t WHITE_MEDIUM_SQUARE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x25fb };
static wchar_t* WHITE_MEDIUM_SQUARE_UNICODE_CHARACTER_CODE_MODEL = WHITE_MEDIUM_SQUARE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* WHITE_MEDIUM_SQUARE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t BLACK_MEDIUM_SQUARE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x25fc };
static wchar_t* BLACK_MEDIUM_SQUARE_UNICODE_CHARACTER_CODE_MODEL = BLACK_MEDIUM_SQUARE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* BLACK_MEDIUM_SQUARE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t BLACK_STAR_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2605 };
static wchar_t* BLACK_STAR_UNICODE_CHARACTER_CODE_MODEL = BLACK_STAR_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* BLACK_STAR_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t WHITE_STAR_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2606 };
static wchar_t* WHITE_STAR_UNICODE_CHARACTER_CODE_MODEL = WHITE_STAR_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* WHITE_STAR_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t BLACK_TELEPHONE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x260e };
static wchar_t* BLACK_TELEPHONE_UNICODE_CHARACTER_CODE_MODEL = BLACK_TELEPHONE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* BLACK_TELEPHONE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t FEMALE_SIGN_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2640 };
static wchar_t* FEMALE_SIGN_UNICODE_CHARACTER_CODE_MODEL = FEMALE_SIGN_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* FEMALE_SIGN_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MALE_SIGN_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2642 };
static wchar_t* MALE_SIGN_UNICODE_CHARACTER_CODE_MODEL = MALE_SIGN_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MALE_SIGN_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t BLACK_SPADE_SUIT_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2660 };
static wchar_t* BLACK_SPADE_SUIT_UNICODE_CHARACTER_CODE_MODEL = BLACK_SPADE_SUIT_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* BLACK_SPADE_SUIT_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t BLACK_CLUB_SUIT_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2663 };
static wchar_t* BLACK_CLUB_SUIT_UNICODE_CHARACTER_CODE_MODEL = BLACK_CLUB_SUIT_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* BLACK_CLUB_SUIT_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t BLACK_HEART_SUIT_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2665 };
static wchar_t* BLACK_HEART_SUIT_UNICODE_CHARACTER_CODE_MODEL = BLACK_HEART_SUIT_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* BLACK_HEART_SUIT_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t BLACK_DIAMOND_SUIT_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2666 };
static wchar_t* BLACK_DIAMOND_SUIT_UNICODE_CHARACTER_CODE_MODEL = BLACK_DIAMOND_SUIT_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* BLACK_DIAMOND_SUIT_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t EIGHTH_NOTE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x266a };
static wchar_t* EIGHTH_NOTE_UNICODE_CHARACTER_CODE_MODEL = EIGHTH_NOTE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* EIGHTH_NOTE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MUSIC_FLAT_SIGN_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x266d };
static wchar_t* MUSIC_FLAT_SIGN_UNICODE_CHARACTER_CODE_MODEL = MUSIC_FLAT_SIGN_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MUSIC_FLAT_SIGN_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MUSIC_NATURAL_SIGN_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x266e };
static wchar_t* MUSIC_NATURAL_SIGN_UNICODE_CHARACTER_CODE_MODEL = MUSIC_NATURAL_SIGN_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MUSIC_NATURAL_SIGN_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MUSIC_SHARP_SIGN_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x266f };
static wchar_t* MUSIC_SHARP_SIGN_UNICODE_CHARACTER_CODE_MODEL = MUSIC_SHARP_SIGN_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MUSIC_SHARP_SIGN_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t CHECK_MARK_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2713 };
static wchar_t* CHECK_MARK_UNICODE_CHARACTER_CODE_MODEL = CHECK_MARK_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CHECK_MARK_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t BALLOT_X_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2717 };
static wchar_t* BALLOT_X_UNICODE_CHARACTER_CODE_MODEL = BALLOT_X_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* BALLOT_X_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MALTESE_CROSS_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2720 };
static wchar_t* MALTESE_CROSS_UNICODE_CHARACTER_CODE_MODEL = MALTESE_CROSS_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MALTESE_CROSS_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t SIX_POINTED_BLACK_STAR_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2736 };
static wchar_t* SIX_POINTED_BLACK_STAR_UNICODE_CHARACTER_CODE_MODEL = SIX_POINTED_BLACK_STAR_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* SIX_POINTED_BLACK_STAR_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LIGHT_VERTICAL_BAR_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2758 };
static wchar_t* LIGHT_VERTICAL_BAR_UNICODE_CHARACTER_CODE_MODEL = LIGHT_VERTICAL_BAR_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LIGHT_VERTICAL_BAR_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LIGHT_LEFT_TORTOISE_SHELL_BRACKET_ORNAMENT_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2772 };
static wchar_t* LIGHT_LEFT_TORTOISE_SHELL_BRACKET_ORNAMENT_UNICODE_CHARACTER_CODE_MODEL = LIGHT_LEFT_TORTOISE_SHELL_BRACKET_ORNAMENT_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LIGHT_LEFT_TORTOISE_SHELL_BRACKET_ORNAMENT_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LIGHT_RIGHT_TORTOISE_SHELL_BRACKET_ORNAMENT_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2773 };
static wchar_t* LIGHT_RIGHT_TORTOISE_SHELL_BRACKET_ORNAMENT_UNICODE_CHARACTER_CODE_MODEL = LIGHT_RIGHT_TORTOISE_SHELL_BRACKET_ORNAMENT_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LIGHT_RIGHT_TORTOISE_SHELL_BRACKET_ORNAMENT_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t REVERSE_SOLIDUS_PRECEDING_SUBSET_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x27c8 };
static wchar_t* REVERSE_SOLIDUS_PRECEDING_SUBSET_UNICODE_CHARACTER_CODE_MODEL = REVERSE_SOLIDUS_PRECEDING_SUBSET_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* REVERSE_SOLIDUS_PRECEDING_SUBSET_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t SUPERSET_PRECEDING_SOLIDUS_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x27c9 };
static wchar_t* SUPERSET_PRECEDING_SOLIDUS_UNICODE_CHARACTER_CODE_MODEL = SUPERSET_PRECEDING_SOLIDUS_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* SUPERSET_PRECEDING_SOLIDUS_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_LEFT_WHITE_SQUARE_BRACKET_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x27e6 };
static wchar_t* MATHEMATICAL_LEFT_WHITE_SQUARE_BRACKET_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_LEFT_WHITE_SQUARE_BRACKET_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_LEFT_WHITE_SQUARE_BRACKET_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_RIGHT_WHITE_SQUARE_BRACKET_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x27e7 };
static wchar_t* MATHEMATICAL_RIGHT_WHITE_SQUARE_BRACKET_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_RIGHT_WHITE_SQUARE_BRACKET_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_RIGHT_WHITE_SQUARE_BRACKET_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_LEFT_ANGLE_BRACKET_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x27e8 };
static wchar_t* MATHEMATICAL_LEFT_ANGLE_BRACKET_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_LEFT_ANGLE_BRACKET_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_LEFT_ANGLE_BRACKET_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_RIGHT_ANGLE_BRACKET_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x27e9 };
static wchar_t* MATHEMATICAL_RIGHT_ANGLE_BRACKET_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_RIGHT_ANGLE_BRACKET_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_RIGHT_ANGLE_BRACKET_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_LEFT_DOUBLE_ANGLE_BRACKET_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x27ea };
static wchar_t* MATHEMATICAL_LEFT_DOUBLE_ANGLE_BRACKET_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_LEFT_DOUBLE_ANGLE_BRACKET_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_LEFT_DOUBLE_ANGLE_BRACKET_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_RIGHT_DOUBLE_ANGLE_BRACKET_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x27eb };
static wchar_t* MATHEMATICAL_RIGHT_DOUBLE_ANGLE_BRACKET_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_RIGHT_DOUBLE_ANGLE_BRACKET_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_RIGHT_DOUBLE_ANGLE_BRACKET_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_LEFT_WHITE_TORTOISE_SHELL_BRACKET_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x27ec };
static wchar_t* MATHEMATICAL_LEFT_WHITE_TORTOISE_SHELL_BRACKET_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_LEFT_WHITE_TORTOISE_SHELL_BRACKET_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_LEFT_WHITE_TORTOISE_SHELL_BRACKET_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_RIGHT_WHITE_TORTOISE_SHELL_BRACKET_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x27ed };
static wchar_t* MATHEMATICAL_RIGHT_WHITE_TORTOISE_SHELL_BRACKET_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_RIGHT_WHITE_TORTOISE_SHELL_BRACKET_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_RIGHT_WHITE_TORTOISE_SHELL_BRACKET_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LONG_LEFTWARDS_ARROW_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x27f5 };
static wchar_t* LONG_LEFTWARDS_ARROW_UNICODE_CHARACTER_CODE_MODEL = LONG_LEFTWARDS_ARROW_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LONG_LEFTWARDS_ARROW_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LONG_RIGHTWARDS_ARROW_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x27f6 };
static wchar_t* LONG_RIGHTWARDS_ARROW_UNICODE_CHARACTER_CODE_MODEL = LONG_RIGHTWARDS_ARROW_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LONG_RIGHTWARDS_ARROW_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LONG_LEFT_RIGHT_ARROW_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x27f7 };
static wchar_t* LONG_LEFT_RIGHT_ARROW_UNICODE_CHARACTER_CODE_MODEL = LONG_LEFT_RIGHT_ARROW_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LONG_LEFT_RIGHT_ARROW_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LONG_LEFTWARDS_DOUBLE_ARROW_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x27f8 };
static wchar_t* LONG_LEFTWARDS_DOUBLE_ARROW_UNICODE_CHARACTER_CODE_MODEL = LONG_LEFTWARDS_DOUBLE_ARROW_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LONG_LEFTWARDS_DOUBLE_ARROW_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LONG_RIGHTWARDS_DOUBLE_ARROW_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x27f9 };
static wchar_t* LONG_RIGHTWARDS_DOUBLE_ARROW_UNICODE_CHARACTER_CODE_MODEL = LONG_RIGHTWARDS_DOUBLE_ARROW_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LONG_RIGHTWARDS_DOUBLE_ARROW_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LONG_LEFT_RIGHT_DOUBLE_ARROW_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x27fa };
static wchar_t* LONG_LEFT_RIGHT_DOUBLE_ARROW_UNICODE_CHARACTER_CODE_MODEL = LONG_LEFT_RIGHT_DOUBLE_ARROW_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LONG_LEFT_RIGHT_DOUBLE_ARROW_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LONG_RIGHTWARDS_ARROW_FROM_BAR_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x27fc };
static wchar_t* LONG_RIGHTWARDS_ARROW_FROM_BAR_UNICODE_CHARACTER_CODE_MODEL = LONG_RIGHTWARDS_ARROW_FROM_BAR_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LONG_RIGHTWARDS_ARROW_FROM_BAR_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LONG_RIGHTWARDS_SQUIGGLE_ARROW_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x27ff };
static wchar_t* LONG_RIGHTWARDS_SQUIGGLE_ARROW_UNICODE_CHARACTER_CODE_MODEL = LONG_RIGHTWARDS_SQUIGGLE_ARROW_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LONG_RIGHTWARDS_SQUIGGLE_ARROW_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LEFTWARDS_DOUBLE_ARROW_WITH_VERTICAL_STROKE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2902 };
static wchar_t* LEFTWARDS_DOUBLE_ARROW_WITH_VERTICAL_STROKE_UNICODE_CHARACTER_CODE_MODEL = LEFTWARDS_DOUBLE_ARROW_WITH_VERTICAL_STROKE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LEFTWARDS_DOUBLE_ARROW_WITH_VERTICAL_STROKE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t RIGHTWARDS_DOUBLE_ARROW_WITH_VERTICAL_STROKE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2903 };
static wchar_t* RIGHTWARDS_DOUBLE_ARROW_WITH_VERTICAL_STROKE_UNICODE_CHARACTER_CODE_MODEL = RIGHTWARDS_DOUBLE_ARROW_WITH_VERTICAL_STROKE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* RIGHTWARDS_DOUBLE_ARROW_WITH_VERTICAL_STROKE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LEFT_RIGHT_DOUBLE_ARROW_WITH_VERTICAL_STROKE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2904 };
static wchar_t* LEFT_RIGHT_DOUBLE_ARROW_WITH_VERTICAL_STROKE_UNICODE_CHARACTER_CODE_MODEL = LEFT_RIGHT_DOUBLE_ARROW_WITH_VERTICAL_STROKE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LEFT_RIGHT_DOUBLE_ARROW_WITH_VERTICAL_STROKE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t RIGHTWARDS_TWO_HEADED_ARROW_FROM_BAR_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2905 };
static wchar_t* RIGHTWARDS_TWO_HEADED_ARROW_FROM_BAR_UNICODE_CHARACTER_CODE_MODEL = RIGHTWARDS_TWO_HEADED_ARROW_FROM_BAR_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* RIGHTWARDS_TWO_HEADED_ARROW_FROM_BAR_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LEFTWARDS_DOUBLE_DASH_ARROW_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x290c };
static wchar_t* LEFTWARDS_DOUBLE_DASH_ARROW_UNICODE_CHARACTER_CODE_MODEL = LEFTWARDS_DOUBLE_DASH_ARROW_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LEFTWARDS_DOUBLE_DASH_ARROW_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t RIGHTWARDS_DOUBLE_DASH_ARROW_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x290d };
static wchar_t* RIGHTWARDS_DOUBLE_DASH_ARROW_UNICODE_CHARACTER_CODE_MODEL = RIGHTWARDS_DOUBLE_DASH_ARROW_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* RIGHTWARDS_DOUBLE_DASH_ARROW_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LEFTWARDS_TRIPLE_DASH_ARROW_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x290e };
static wchar_t* LEFTWARDS_TRIPLE_DASH_ARROW_UNICODE_CHARACTER_CODE_MODEL = LEFTWARDS_TRIPLE_DASH_ARROW_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LEFTWARDS_TRIPLE_DASH_ARROW_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t RIGHTWARDS_TRIPLE_DASH_ARROW_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x290f };
static wchar_t* RIGHTWARDS_TRIPLE_DASH_ARROW_UNICODE_CHARACTER_CODE_MODEL = RIGHTWARDS_TRIPLE_DASH_ARROW_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* RIGHTWARDS_TRIPLE_DASH_ARROW_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t RIGHTWARDS_TWO_HEADED_TRIPLE_DASH_ARROW_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2910 };
static wchar_t* RIGHTWARDS_TWO_HEADED_TRIPLE_DASH_ARROW_UNICODE_CHARACTER_CODE_MODEL = RIGHTWARDS_TWO_HEADED_TRIPLE_DASH_ARROW_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* RIGHTWARDS_TWO_HEADED_TRIPLE_DASH_ARROW_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t RIGHTWARDS_ARROW_WITH_DOTTED_STEM_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2911 };
static wchar_t* RIGHTWARDS_ARROW_WITH_DOTTED_STEM_UNICODE_CHARACTER_CODE_MODEL = RIGHTWARDS_ARROW_WITH_DOTTED_STEM_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* RIGHTWARDS_ARROW_WITH_DOTTED_STEM_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t UPWARDS_ARROW_TO_BAR_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2912 };
static wchar_t* UPWARDS_ARROW_TO_BAR_UNICODE_CHARACTER_CODE_MODEL = UPWARDS_ARROW_TO_BAR_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* UPWARDS_ARROW_TO_BAR_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t DOWNWARDS_ARROW_TO_BAR_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2913 };
static wchar_t* DOWNWARDS_ARROW_TO_BAR_UNICODE_CHARACTER_CODE_MODEL = DOWNWARDS_ARROW_TO_BAR_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* DOWNWARDS_ARROW_TO_BAR_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t RIGHTWARDS_TWO_HEADED_ARROW_WITH_TAIL_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2916 };
static wchar_t* RIGHTWARDS_TWO_HEADED_ARROW_WITH_TAIL_UNICODE_CHARACTER_CODE_MODEL = RIGHTWARDS_TWO_HEADED_ARROW_WITH_TAIL_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* RIGHTWARDS_TWO_HEADED_ARROW_WITH_TAIL_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LEFTWARDS_ARROW_TAIL_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2919 };
static wchar_t* LEFTWARDS_ARROW_TAIL_UNICODE_CHARACTER_CODE_MODEL = LEFTWARDS_ARROW_TAIL_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LEFTWARDS_ARROW_TAIL_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t RIGHTWARDS_ARROW_TAIL_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x291a };
static wchar_t* RIGHTWARDS_ARROW_TAIL_UNICODE_CHARACTER_CODE_MODEL = RIGHTWARDS_ARROW_TAIL_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* RIGHTWARDS_ARROW_TAIL_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LEFTWARDS_DOUBLE_ARROW_TAIL_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x291b };
static wchar_t* LEFTWARDS_DOUBLE_ARROW_TAIL_UNICODE_CHARACTER_CODE_MODEL = LEFTWARDS_DOUBLE_ARROW_TAIL_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LEFTWARDS_DOUBLE_ARROW_TAIL_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t RIGHTWARDS_DOUBLE_ARROW_TAIL_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x291c };
static wchar_t* RIGHTWARDS_DOUBLE_ARROW_TAIL_UNICODE_CHARACTER_CODE_MODEL = RIGHTWARDS_DOUBLE_ARROW_TAIL_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* RIGHTWARDS_DOUBLE_ARROW_TAIL_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LEFTWARDS_ARROW_TO_BLACK_DIAMOND_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x291d };
static wchar_t* LEFTWARDS_ARROW_TO_BLACK_DIAMOND_UNICODE_CHARACTER_CODE_MODEL = LEFTWARDS_ARROW_TO_BLACK_DIAMOND_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LEFTWARDS_ARROW_TO_BLACK_DIAMOND_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t RIGHTWARDS_ARROW_TO_BLACK_DIAMOND_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x291e };
static wchar_t* RIGHTWARDS_ARROW_TO_BLACK_DIAMOND_UNICODE_CHARACTER_CODE_MODEL = RIGHTWARDS_ARROW_TO_BLACK_DIAMOND_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* RIGHTWARDS_ARROW_TO_BLACK_DIAMOND_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LEFTWARDS_ARROW_FROM_BAR_TO_BLACK_DIAMOND_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x291f };
static wchar_t* LEFTWARDS_ARROW_FROM_BAR_TO_BLACK_DIAMOND_UNICODE_CHARACTER_CODE_MODEL = LEFTWARDS_ARROW_FROM_BAR_TO_BLACK_DIAMOND_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LEFTWARDS_ARROW_FROM_BAR_TO_BLACK_DIAMOND_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t RIGHTWARDS_ARROW_FROM_BAR_TO_BLACK_DIAMOND_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2920 };
static wchar_t* RIGHTWARDS_ARROW_FROM_BAR_TO_BLACK_DIAMOND_UNICODE_CHARACTER_CODE_MODEL = RIGHTWARDS_ARROW_FROM_BAR_TO_BLACK_DIAMOND_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* RIGHTWARDS_ARROW_FROM_BAR_TO_BLACK_DIAMOND_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t NORTH_WEST_ARROW_WITH_HOOK_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2923 };
static wchar_t* NORTH_WEST_ARROW_WITH_HOOK_UNICODE_CHARACTER_CODE_MODEL = NORTH_WEST_ARROW_WITH_HOOK_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* NORTH_WEST_ARROW_WITH_HOOK_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t NORTH_EAST_ARROW_WITH_HOOK_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2924 };
static wchar_t* NORTH_EAST_ARROW_WITH_HOOK_UNICODE_CHARACTER_CODE_MODEL = NORTH_EAST_ARROW_WITH_HOOK_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* NORTH_EAST_ARROW_WITH_HOOK_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t SOUTH_EAST_ARROW_WITH_HOOK_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2925 };
static wchar_t* SOUTH_EAST_ARROW_WITH_HOOK_UNICODE_CHARACTER_CODE_MODEL = SOUTH_EAST_ARROW_WITH_HOOK_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* SOUTH_EAST_ARROW_WITH_HOOK_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t SOUTH_WEST_ARROW_WITH_HOOK_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2926 };
static wchar_t* SOUTH_WEST_ARROW_WITH_HOOK_UNICODE_CHARACTER_CODE_MODEL = SOUTH_WEST_ARROW_WITH_HOOK_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* SOUTH_WEST_ARROW_WITH_HOOK_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t NORTH_WEST_ARROW_AND_NORTH_EAST_ARROW_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2927 };
static wchar_t* NORTH_WEST_ARROW_AND_NORTH_EAST_ARROW_UNICODE_CHARACTER_CODE_MODEL = NORTH_WEST_ARROW_AND_NORTH_EAST_ARROW_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* NORTH_WEST_ARROW_AND_NORTH_EAST_ARROW_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t NORTH_EAST_ARROW_AND_SOUTH_EAST_ARROW_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2928 };
static wchar_t* NORTH_EAST_ARROW_AND_SOUTH_EAST_ARROW_UNICODE_CHARACTER_CODE_MODEL = NORTH_EAST_ARROW_AND_SOUTH_EAST_ARROW_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* NORTH_EAST_ARROW_AND_SOUTH_EAST_ARROW_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t SOUTH_EAST_ARROW_AND_SOUTH_WEST_ARROW_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2929 };
static wchar_t* SOUTH_EAST_ARROW_AND_SOUTH_WEST_ARROW_UNICODE_CHARACTER_CODE_MODEL = SOUTH_EAST_ARROW_AND_SOUTH_WEST_ARROW_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* SOUTH_EAST_ARROW_AND_SOUTH_WEST_ARROW_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t SOUTH_WEST_ARROW_AND_NORTH_WEST_ARROW_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x292a };
static wchar_t* SOUTH_WEST_ARROW_AND_NORTH_WEST_ARROW_UNICODE_CHARACTER_CODE_MODEL = SOUTH_WEST_ARROW_AND_NORTH_WEST_ARROW_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* SOUTH_WEST_ARROW_AND_NORTH_WEST_ARROW_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t WAVE_ARROW_POINTING_DIRECTLY_RIGHT_WITH_SLASH_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2933, 0x0338 };
static wchar_t* WAVE_ARROW_POINTING_DIRECTLY_RIGHT_WITH_SLASH_UNICODE_CHARACTER_CODE_MODEL = WAVE_ARROW_POINTING_DIRECTLY_RIGHT_WITH_SLASH_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* WAVE_ARROW_POINTING_DIRECTLY_RIGHT_WITH_SLASH_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_2_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t WAVE_ARROW_POINTING_DIRECTLY_RIGHT_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2933 };
static wchar_t* WAVE_ARROW_POINTING_DIRECTLY_RIGHT_UNICODE_CHARACTER_CODE_MODEL = WAVE_ARROW_POINTING_DIRECTLY_RIGHT_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* WAVE_ARROW_POINTING_DIRECTLY_RIGHT_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t ARROW_POINTING_RIGHTWARDS_THEN_CURVING_DOWNWARDS_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2935 };
static wchar_t* ARROW_POINTING_RIGHTWARDS_THEN_CURVING_DOWNWARDS_UNICODE_CHARACTER_CODE_MODEL = ARROW_POINTING_RIGHTWARDS_THEN_CURVING_DOWNWARDS_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* ARROW_POINTING_RIGHTWARDS_THEN_CURVING_DOWNWARDS_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t ARROW_POINTING_DOWNWARDS_THEN_CURVING_LEFTWARDS_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2936 };
static wchar_t* ARROW_POINTING_DOWNWARDS_THEN_CURVING_LEFTWARDS_UNICODE_CHARACTER_CODE_MODEL = ARROW_POINTING_DOWNWARDS_THEN_CURVING_LEFTWARDS_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* ARROW_POINTING_DOWNWARDS_THEN_CURVING_LEFTWARDS_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t ARROW_POINTING_DOWNWARDS_THEN_CURVING_RIGHTWARDS_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2937 };
static wchar_t* ARROW_POINTING_DOWNWARDS_THEN_CURVING_RIGHTWARDS_UNICODE_CHARACTER_CODE_MODEL = ARROW_POINTING_DOWNWARDS_THEN_CURVING_RIGHTWARDS_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* ARROW_POINTING_DOWNWARDS_THEN_CURVING_RIGHTWARDS_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t RIGHT_SIDE_ARC_CLOCKWISE_ARROW_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2938 };
static wchar_t* RIGHT_SIDE_ARC_CLOCKWISE_ARROW_UNICODE_CHARACTER_CODE_MODEL = RIGHT_SIDE_ARC_CLOCKWISE_ARROW_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* RIGHT_SIDE_ARC_CLOCKWISE_ARROW_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LEFT_SIDE_ARC_ANTICLOCKWISE_ARROW_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2939 };
static wchar_t* LEFT_SIDE_ARC_ANTICLOCKWISE_ARROW_UNICODE_CHARACTER_CODE_MODEL = LEFT_SIDE_ARC_ANTICLOCKWISE_ARROW_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LEFT_SIDE_ARC_ANTICLOCKWISE_ARROW_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t TOP_ARC_CLOCKWISE_ARROW_WITH_MINUS_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x293c };
static wchar_t* TOP_ARC_CLOCKWISE_ARROW_WITH_MINUS_UNICODE_CHARACTER_CODE_MODEL = TOP_ARC_CLOCKWISE_ARROW_WITH_MINUS_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* TOP_ARC_CLOCKWISE_ARROW_WITH_MINUS_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t TOP_ARC_ANTICLOCKWISE_ARROW_WITH_PLUS_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x293d };
static wchar_t* TOP_ARC_ANTICLOCKWISE_ARROW_WITH_PLUS_UNICODE_CHARACTER_CODE_MODEL = TOP_ARC_ANTICLOCKWISE_ARROW_WITH_PLUS_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* TOP_ARC_ANTICLOCKWISE_ARROW_WITH_PLUS_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t RIGHTWARDS_ARROW_WITH_PLUS_BELOW_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2945 };
static wchar_t* RIGHTWARDS_ARROW_WITH_PLUS_BELOW_UNICODE_CHARACTER_CODE_MODEL = RIGHTWARDS_ARROW_WITH_PLUS_BELOW_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* RIGHTWARDS_ARROW_WITH_PLUS_BELOW_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LEFT_RIGHT_ARROW_THROUGH_SMALL_CIRCLE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2948 };
static wchar_t* LEFT_RIGHT_ARROW_THROUGH_SMALL_CIRCLE_UNICODE_CHARACTER_CODE_MODEL = LEFT_RIGHT_ARROW_THROUGH_SMALL_CIRCLE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LEFT_RIGHT_ARROW_THROUGH_SMALL_CIRCLE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t UPWARDS_TWO_HEADED_ARROW_FROM_SMALL_CIRCLE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2949 };
static wchar_t* UPWARDS_TWO_HEADED_ARROW_FROM_SMALL_CIRCLE_UNICODE_CHARACTER_CODE_MODEL = UPWARDS_TWO_HEADED_ARROW_FROM_SMALL_CIRCLE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* UPWARDS_TWO_HEADED_ARROW_FROM_SMALL_CIRCLE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LEFT_BARB_UP_RIGHT_BARB_DOWN_HARPOON_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x294a };
static wchar_t* LEFT_BARB_UP_RIGHT_BARB_DOWN_HARPOON_UNICODE_CHARACTER_CODE_MODEL = LEFT_BARB_UP_RIGHT_BARB_DOWN_HARPOON_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LEFT_BARB_UP_RIGHT_BARB_DOWN_HARPOON_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LEFT_BARB_DOWN_RIGHT_BARB_UP_HARPOON_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x294b };
static wchar_t* LEFT_BARB_DOWN_RIGHT_BARB_UP_HARPOON_UNICODE_CHARACTER_CODE_MODEL = LEFT_BARB_DOWN_RIGHT_BARB_UP_HARPOON_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LEFT_BARB_DOWN_RIGHT_BARB_UP_HARPOON_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LEFT_BARB_UP_RIGHT_BARB_UP_HARPOON_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x294e };
static wchar_t* LEFT_BARB_UP_RIGHT_BARB_UP_HARPOON_UNICODE_CHARACTER_CODE_MODEL = LEFT_BARB_UP_RIGHT_BARB_UP_HARPOON_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LEFT_BARB_UP_RIGHT_BARB_UP_HARPOON_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t UP_BARB_RIGHT_DOWN_BARB_RIGHT_HARPOON_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x294f };
static wchar_t* UP_BARB_RIGHT_DOWN_BARB_RIGHT_HARPOON_UNICODE_CHARACTER_CODE_MODEL = UP_BARB_RIGHT_DOWN_BARB_RIGHT_HARPOON_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* UP_BARB_RIGHT_DOWN_BARB_RIGHT_HARPOON_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LEFT_BARB_DOWN_RIGHT_BARB_DOWN_HARPOON_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2950 };
static wchar_t* LEFT_BARB_DOWN_RIGHT_BARB_DOWN_HARPOON_UNICODE_CHARACTER_CODE_MODEL = LEFT_BARB_DOWN_RIGHT_BARB_DOWN_HARPOON_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LEFT_BARB_DOWN_RIGHT_BARB_DOWN_HARPOON_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t UP_BARB_LEFT_DOWN_BARB_LEFT_HARPOON_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2951 };
static wchar_t* UP_BARB_LEFT_DOWN_BARB_LEFT_HARPOON_UNICODE_CHARACTER_CODE_MODEL = UP_BARB_LEFT_DOWN_BARB_LEFT_HARPOON_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* UP_BARB_LEFT_DOWN_BARB_LEFT_HARPOON_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LEFTWARDS_HARPOON_WITH_BARB_UP_TO_BAR_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2952 };
static wchar_t* LEFTWARDS_HARPOON_WITH_BARB_UP_TO_BAR_UNICODE_CHARACTER_CODE_MODEL = LEFTWARDS_HARPOON_WITH_BARB_UP_TO_BAR_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LEFTWARDS_HARPOON_WITH_BARB_UP_TO_BAR_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t RIGHTWARDS_HARPOON_WITH_BARB_UP_TO_BAR_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2953 };
static wchar_t* RIGHTWARDS_HARPOON_WITH_BARB_UP_TO_BAR_UNICODE_CHARACTER_CODE_MODEL = RIGHTWARDS_HARPOON_WITH_BARB_UP_TO_BAR_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* RIGHTWARDS_HARPOON_WITH_BARB_UP_TO_BAR_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t UPWARDS_HARPOON_WITH_BARB_RIGHT_TO_BAR_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2954 };
static wchar_t* UPWARDS_HARPOON_WITH_BARB_RIGHT_TO_BAR_UNICODE_CHARACTER_CODE_MODEL = UPWARDS_HARPOON_WITH_BARB_RIGHT_TO_BAR_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* UPWARDS_HARPOON_WITH_BARB_RIGHT_TO_BAR_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t DOWNWARDS_HARPOON_WITH_BARB_RIGHT_TO_BAR_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2955 };
static wchar_t* DOWNWARDS_HARPOON_WITH_BARB_RIGHT_TO_BAR_UNICODE_CHARACTER_CODE_MODEL = DOWNWARDS_HARPOON_WITH_BARB_RIGHT_TO_BAR_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* DOWNWARDS_HARPOON_WITH_BARB_RIGHT_TO_BAR_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LEFTWARDS_HARPOON_WITH_BARB_DOWN_TO_BAR_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2956 };
static wchar_t* LEFTWARDS_HARPOON_WITH_BARB_DOWN_TO_BAR_UNICODE_CHARACTER_CODE_MODEL = LEFTWARDS_HARPOON_WITH_BARB_DOWN_TO_BAR_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LEFTWARDS_HARPOON_WITH_BARB_DOWN_TO_BAR_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t RIGHTWARDS_HARPOON_WITH_BARB_DOWN_TO_BAR_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2957 };
static wchar_t* RIGHTWARDS_HARPOON_WITH_BARB_DOWN_TO_BAR_UNICODE_CHARACTER_CODE_MODEL = RIGHTWARDS_HARPOON_WITH_BARB_DOWN_TO_BAR_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* RIGHTWARDS_HARPOON_WITH_BARB_DOWN_TO_BAR_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t UPWARDS_HARPOON_WITH_BARB_LEFT_TO_BAR_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2958 };
static wchar_t* UPWARDS_HARPOON_WITH_BARB_LEFT_TO_BAR_UNICODE_CHARACTER_CODE_MODEL = UPWARDS_HARPOON_WITH_BARB_LEFT_TO_BAR_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* UPWARDS_HARPOON_WITH_BARB_LEFT_TO_BAR_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t DOWNWARDS_HARPOON_WITH_BARB_LEFT_TO_BAR_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2959 };
static wchar_t* DOWNWARDS_HARPOON_WITH_BARB_LEFT_TO_BAR_UNICODE_CHARACTER_CODE_MODEL = DOWNWARDS_HARPOON_WITH_BARB_LEFT_TO_BAR_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* DOWNWARDS_HARPOON_WITH_BARB_LEFT_TO_BAR_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LEFTWARDS_HARPOON_WITH_BARB_UP_FROM_BAR_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x295a };
static wchar_t* LEFTWARDS_HARPOON_WITH_BARB_UP_FROM_BAR_UNICODE_CHARACTER_CODE_MODEL = LEFTWARDS_HARPOON_WITH_BARB_UP_FROM_BAR_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LEFTWARDS_HARPOON_WITH_BARB_UP_FROM_BAR_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t RIGHTWARDS_HARPOON_WITH_BARB_UP_FROM_BAR_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x295b };
static wchar_t* RIGHTWARDS_HARPOON_WITH_BARB_UP_FROM_BAR_UNICODE_CHARACTER_CODE_MODEL = RIGHTWARDS_HARPOON_WITH_BARB_UP_FROM_BAR_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* RIGHTWARDS_HARPOON_WITH_BARB_UP_FROM_BAR_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t UPWARDS_HARPOON_WITH_BARB_RIGHT_FROM_BAR_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x295c };
static wchar_t* UPWARDS_HARPOON_WITH_BARB_RIGHT_FROM_BAR_UNICODE_CHARACTER_CODE_MODEL = UPWARDS_HARPOON_WITH_BARB_RIGHT_FROM_BAR_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* UPWARDS_HARPOON_WITH_BARB_RIGHT_FROM_BAR_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t DOWNWARDS_HARPOON_WITH_BARB_RIGHT_FROM_BAR_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x295d };
static wchar_t* DOWNWARDS_HARPOON_WITH_BARB_RIGHT_FROM_BAR_UNICODE_CHARACTER_CODE_MODEL = DOWNWARDS_HARPOON_WITH_BARB_RIGHT_FROM_BAR_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* DOWNWARDS_HARPOON_WITH_BARB_RIGHT_FROM_BAR_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LEFTWARDS_HARPOON_WITH_BARB_DOWN_FROM_BAR_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x295e };
static wchar_t* LEFTWARDS_HARPOON_WITH_BARB_DOWN_FROM_BAR_UNICODE_CHARACTER_CODE_MODEL = LEFTWARDS_HARPOON_WITH_BARB_DOWN_FROM_BAR_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LEFTWARDS_HARPOON_WITH_BARB_DOWN_FROM_BAR_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t RIGHTWARDS_HARPOON_WITH_BARB_DOWN_FROM_BAR_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x295f };
static wchar_t* RIGHTWARDS_HARPOON_WITH_BARB_DOWN_FROM_BAR_UNICODE_CHARACTER_CODE_MODEL = RIGHTWARDS_HARPOON_WITH_BARB_DOWN_FROM_BAR_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* RIGHTWARDS_HARPOON_WITH_BARB_DOWN_FROM_BAR_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t UPWARDS_HARPOON_WITH_BARB_LEFT_FROM_BAR_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2960 };
static wchar_t* UPWARDS_HARPOON_WITH_BARB_LEFT_FROM_BAR_UNICODE_CHARACTER_CODE_MODEL = UPWARDS_HARPOON_WITH_BARB_LEFT_FROM_BAR_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* UPWARDS_HARPOON_WITH_BARB_LEFT_FROM_BAR_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t DOWNWARDS_HARPOON_WITH_BARB_LEFT_FROM_BAR_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2961 };
static wchar_t* DOWNWARDS_HARPOON_WITH_BARB_LEFT_FROM_BAR_UNICODE_CHARACTER_CODE_MODEL = DOWNWARDS_HARPOON_WITH_BARB_LEFT_FROM_BAR_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* DOWNWARDS_HARPOON_WITH_BARB_LEFT_FROM_BAR_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LEFTWARDS_HARPOON_WITH_BARB_UP_ABOVE_LEFTWARDS_HARPOON_WITH_BARB_DOWN_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2962 };
static wchar_t* LEFTWARDS_HARPOON_WITH_BARB_UP_ABOVE_LEFTWARDS_HARPOON_WITH_BARB_DOWN_UNICODE_CHARACTER_CODE_MODEL = LEFTWARDS_HARPOON_WITH_BARB_UP_ABOVE_LEFTWARDS_HARPOON_WITH_BARB_DOWN_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LEFTWARDS_HARPOON_WITH_BARB_UP_ABOVE_LEFTWARDS_HARPOON_WITH_BARB_DOWN_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t UPWARDS_HARPOON_WITH_BARB_LEFT_BESIDE_UPWARDS_HARPOON_WITH_BARB_RIGHT_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2963 };
static wchar_t* UPWARDS_HARPOON_WITH_BARB_LEFT_BESIDE_UPWARDS_HARPOON_WITH_BARB_RIGHT_UNICODE_CHARACTER_CODE_MODEL = UPWARDS_HARPOON_WITH_BARB_LEFT_BESIDE_UPWARDS_HARPOON_WITH_BARB_RIGHT_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* UPWARDS_HARPOON_WITH_BARB_LEFT_BESIDE_UPWARDS_HARPOON_WITH_BARB_RIGHT_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t RIGHTWARDS_HARPOON_WITH_BARB_UP_ABOVE_RIGHTWARDS_HARPOON_WITH_BARB_DOWN_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2964 };
static wchar_t* RIGHTWARDS_HARPOON_WITH_BARB_UP_ABOVE_RIGHTWARDS_HARPOON_WITH_BARB_DOWN_UNICODE_CHARACTER_CODE_MODEL = RIGHTWARDS_HARPOON_WITH_BARB_UP_ABOVE_RIGHTWARDS_HARPOON_WITH_BARB_DOWN_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* RIGHTWARDS_HARPOON_WITH_BARB_UP_ABOVE_RIGHTWARDS_HARPOON_WITH_BARB_DOWN_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t DOWNWARDS_HARPOON_WITH_BARB_LEFT_BESIDE_DOWNWARDS_HARPOON_WITH_BARB_RIGHT_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2965 };
static wchar_t* DOWNWARDS_HARPOON_WITH_BARB_LEFT_BESIDE_DOWNWARDS_HARPOON_WITH_BARB_RIGHT_UNICODE_CHARACTER_CODE_MODEL = DOWNWARDS_HARPOON_WITH_BARB_LEFT_BESIDE_DOWNWARDS_HARPOON_WITH_BARB_RIGHT_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* DOWNWARDS_HARPOON_WITH_BARB_LEFT_BESIDE_DOWNWARDS_HARPOON_WITH_BARB_RIGHT_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LEFTWARDS_HARPOON_WITH_BARB_UP_ABOVE_RIGHTWARDS_HARPOON_WITH_BARB_UP_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2966 };
static wchar_t* LEFTWARDS_HARPOON_WITH_BARB_UP_ABOVE_RIGHTWARDS_HARPOON_WITH_BARB_UP_UNICODE_CHARACTER_CODE_MODEL = LEFTWARDS_HARPOON_WITH_BARB_UP_ABOVE_RIGHTWARDS_HARPOON_WITH_BARB_UP_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LEFTWARDS_HARPOON_WITH_BARB_UP_ABOVE_RIGHTWARDS_HARPOON_WITH_BARB_UP_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LEFTWARDS_HARPOON_WITH_BARB_DOWN_ABOVE_RIGHTWARDS_HARPOON_WITH_BARB_DOWN_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2967 };
static wchar_t* LEFTWARDS_HARPOON_WITH_BARB_DOWN_ABOVE_RIGHTWARDS_HARPOON_WITH_BARB_DOWN_UNICODE_CHARACTER_CODE_MODEL = LEFTWARDS_HARPOON_WITH_BARB_DOWN_ABOVE_RIGHTWARDS_HARPOON_WITH_BARB_DOWN_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LEFTWARDS_HARPOON_WITH_BARB_DOWN_ABOVE_RIGHTWARDS_HARPOON_WITH_BARB_DOWN_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t RIGHTWARDS_HARPOON_WITH_BARB_UP_ABOVE_LEFTWARDS_HARPOON_WITH_BARB_UP_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2968 };
static wchar_t* RIGHTWARDS_HARPOON_WITH_BARB_UP_ABOVE_LEFTWARDS_HARPOON_WITH_BARB_UP_UNICODE_CHARACTER_CODE_MODEL = RIGHTWARDS_HARPOON_WITH_BARB_UP_ABOVE_LEFTWARDS_HARPOON_WITH_BARB_UP_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* RIGHTWARDS_HARPOON_WITH_BARB_UP_ABOVE_LEFTWARDS_HARPOON_WITH_BARB_UP_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t RIGHTWARDS_HARPOON_WITH_BARB_DOWN_ABOVE_LEFTWARDS_HARPOON_WITH_BARB_DOWN_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2969 };
static wchar_t* RIGHTWARDS_HARPOON_WITH_BARB_DOWN_ABOVE_LEFTWARDS_HARPOON_WITH_BARB_DOWN_UNICODE_CHARACTER_CODE_MODEL = RIGHTWARDS_HARPOON_WITH_BARB_DOWN_ABOVE_LEFTWARDS_HARPOON_WITH_BARB_DOWN_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* RIGHTWARDS_HARPOON_WITH_BARB_DOWN_ABOVE_LEFTWARDS_HARPOON_WITH_BARB_DOWN_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LEFTWARDS_HARPOON_WITH_BARB_UP_ABOVE_LONG_DASH_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x296a };
static wchar_t* LEFTWARDS_HARPOON_WITH_BARB_UP_ABOVE_LONG_DASH_UNICODE_CHARACTER_CODE_MODEL = LEFTWARDS_HARPOON_WITH_BARB_UP_ABOVE_LONG_DASH_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LEFTWARDS_HARPOON_WITH_BARB_UP_ABOVE_LONG_DASH_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LEFTWARDS_HARPOON_WITH_BARB_DOWN_BELOW_LONG_DASH_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x296b };
static wchar_t* LEFTWARDS_HARPOON_WITH_BARB_DOWN_BELOW_LONG_DASH_UNICODE_CHARACTER_CODE_MODEL = LEFTWARDS_HARPOON_WITH_BARB_DOWN_BELOW_LONG_DASH_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LEFTWARDS_HARPOON_WITH_BARB_DOWN_BELOW_LONG_DASH_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t RIGHTWARDS_HARPOON_WITH_BARB_UP_ABOVE_LONG_DASH_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x296c };
static wchar_t* RIGHTWARDS_HARPOON_WITH_BARB_UP_ABOVE_LONG_DASH_UNICODE_CHARACTER_CODE_MODEL = RIGHTWARDS_HARPOON_WITH_BARB_UP_ABOVE_LONG_DASH_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* RIGHTWARDS_HARPOON_WITH_BARB_UP_ABOVE_LONG_DASH_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t RIGHTWARDS_HARPOON_WITH_BARB_DOWN_BELOW_LONG_DASH_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x296d };
static wchar_t* RIGHTWARDS_HARPOON_WITH_BARB_DOWN_BELOW_LONG_DASH_UNICODE_CHARACTER_CODE_MODEL = RIGHTWARDS_HARPOON_WITH_BARB_DOWN_BELOW_LONG_DASH_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* RIGHTWARDS_HARPOON_WITH_BARB_DOWN_BELOW_LONG_DASH_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t UPWARDS_HARPOON_WITH_BARB_LEFT_BESIDE_DOWNWARDS_HARPOON_WITH_BARB_RIGHT_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x296e };
static wchar_t* UPWARDS_HARPOON_WITH_BARB_LEFT_BESIDE_DOWNWARDS_HARPOON_WITH_BARB_RIGHT_UNICODE_CHARACTER_CODE_MODEL = UPWARDS_HARPOON_WITH_BARB_LEFT_BESIDE_DOWNWARDS_HARPOON_WITH_BARB_RIGHT_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* UPWARDS_HARPOON_WITH_BARB_LEFT_BESIDE_DOWNWARDS_HARPOON_WITH_BARB_RIGHT_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t DOWNWARDS_HARPOON_WITH_BARB_LEFT_BESIDE_UPWARDS_HARPOON_WITH_BARB_RIGHT_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x296f };
static wchar_t* DOWNWARDS_HARPOON_WITH_BARB_LEFT_BESIDE_UPWARDS_HARPOON_WITH_BARB_RIGHT_UNICODE_CHARACTER_CODE_MODEL = DOWNWARDS_HARPOON_WITH_BARB_LEFT_BESIDE_UPWARDS_HARPOON_WITH_BARB_RIGHT_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* DOWNWARDS_HARPOON_WITH_BARB_LEFT_BESIDE_UPWARDS_HARPOON_WITH_BARB_RIGHT_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t RIGHT_DOUBLE_ARROW_WITH_ROUNDED_HEAD_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2970 };
static wchar_t* RIGHT_DOUBLE_ARROW_WITH_ROUNDED_HEAD_UNICODE_CHARACTER_CODE_MODEL = RIGHT_DOUBLE_ARROW_WITH_ROUNDED_HEAD_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* RIGHT_DOUBLE_ARROW_WITH_ROUNDED_HEAD_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t EQUALS_SIGN_ABOVE_RIGHTWARDS_ARROW_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2971 };
static wchar_t* EQUALS_SIGN_ABOVE_RIGHTWARDS_ARROW_UNICODE_CHARACTER_CODE_MODEL = EQUALS_SIGN_ABOVE_RIGHTWARDS_ARROW_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* EQUALS_SIGN_ABOVE_RIGHTWARDS_ARROW_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t TILDE_OPERATOR_ABOVE_RIGHTWARDS_ARROW_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2972 };
static wchar_t* TILDE_OPERATOR_ABOVE_RIGHTWARDS_ARROW_UNICODE_CHARACTER_CODE_MODEL = TILDE_OPERATOR_ABOVE_RIGHTWARDS_ARROW_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* TILDE_OPERATOR_ABOVE_RIGHTWARDS_ARROW_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LEFTWARDS_ARROW_ABOVE_TILDE_OPERATOR_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2973 };
static wchar_t* LEFTWARDS_ARROW_ABOVE_TILDE_OPERATOR_UNICODE_CHARACTER_CODE_MODEL = LEFTWARDS_ARROW_ABOVE_TILDE_OPERATOR_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LEFTWARDS_ARROW_ABOVE_TILDE_OPERATOR_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t RIGHTWARDS_ARROW_ABOVE_TILDE_OPERATOR_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2974 };
static wchar_t* RIGHTWARDS_ARROW_ABOVE_TILDE_OPERATOR_UNICODE_CHARACTER_CODE_MODEL = RIGHTWARDS_ARROW_ABOVE_TILDE_OPERATOR_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* RIGHTWARDS_ARROW_ABOVE_TILDE_OPERATOR_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t RIGHTWARDS_ARROW_ABOVE_ALMOST_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2975 };
static wchar_t* RIGHTWARDS_ARROW_ABOVE_ALMOST_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL = RIGHTWARDS_ARROW_ABOVE_ALMOST_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* RIGHTWARDS_ARROW_ABOVE_ALMOST_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LESS_THAN_ABOVE_LEFTWARDS_ARROW_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2976 };
static wchar_t* LESS_THAN_ABOVE_LEFTWARDS_ARROW_UNICODE_CHARACTER_CODE_MODEL = LESS_THAN_ABOVE_LEFTWARDS_ARROW_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LESS_THAN_ABOVE_LEFTWARDS_ARROW_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t GREATER_THAN_ABOVE_RIGHTWARDS_ARROW_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2978 };
static wchar_t* GREATER_THAN_ABOVE_RIGHTWARDS_ARROW_UNICODE_CHARACTER_CODE_MODEL = GREATER_THAN_ABOVE_RIGHTWARDS_ARROW_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* GREATER_THAN_ABOVE_RIGHTWARDS_ARROW_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t SUBSET_ABOVE_RIGHTWARDS_ARROW_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2979 };
static wchar_t* SUBSET_ABOVE_RIGHTWARDS_ARROW_UNICODE_CHARACTER_CODE_MODEL = SUBSET_ABOVE_RIGHTWARDS_ARROW_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* SUBSET_ABOVE_RIGHTWARDS_ARROW_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t SUPERSET_ABOVE_LEFTWARDS_ARROW_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x297b };
static wchar_t* SUPERSET_ABOVE_LEFTWARDS_ARROW_UNICODE_CHARACTER_CODE_MODEL = SUPERSET_ABOVE_LEFTWARDS_ARROW_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* SUPERSET_ABOVE_LEFTWARDS_ARROW_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LEFT_FISH_TAIL_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x297c };
static wchar_t* LEFT_FISH_TAIL_UNICODE_CHARACTER_CODE_MODEL = LEFT_FISH_TAIL_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LEFT_FISH_TAIL_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t RIGHT_FISH_TAIL_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x297d };
static wchar_t* RIGHT_FISH_TAIL_UNICODE_CHARACTER_CODE_MODEL = RIGHT_FISH_TAIL_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* RIGHT_FISH_TAIL_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t UP_FISH_TAIL_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x297e };
static wchar_t* UP_FISH_TAIL_UNICODE_CHARACTER_CODE_MODEL = UP_FISH_TAIL_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* UP_FISH_TAIL_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t DOWN_FISH_TAIL_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x297f };
static wchar_t* DOWN_FISH_TAIL_UNICODE_CHARACTER_CODE_MODEL = DOWN_FISH_TAIL_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* DOWN_FISH_TAIL_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LEFT_WHITE_PARENTHESIS_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2985 };
static wchar_t* LEFT_WHITE_PARENTHESIS_UNICODE_CHARACTER_CODE_MODEL = LEFT_WHITE_PARENTHESIS_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LEFT_WHITE_PARENTHESIS_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t RIGHT_WHITE_PARENTHESIS_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2986 };
static wchar_t* RIGHT_WHITE_PARENTHESIS_UNICODE_CHARACTER_CODE_MODEL = RIGHT_WHITE_PARENTHESIS_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* RIGHT_WHITE_PARENTHESIS_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LEFT_SQUARE_BRACKET_WITH_UNDERBAR_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x298b };
static wchar_t* LEFT_SQUARE_BRACKET_WITH_UNDERBAR_UNICODE_CHARACTER_CODE_MODEL = LEFT_SQUARE_BRACKET_WITH_UNDERBAR_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LEFT_SQUARE_BRACKET_WITH_UNDERBAR_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t RIGHT_SQUARE_BRACKET_WITH_UNDERBAR_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x298c };
static wchar_t* RIGHT_SQUARE_BRACKET_WITH_UNDERBAR_UNICODE_CHARACTER_CODE_MODEL = RIGHT_SQUARE_BRACKET_WITH_UNDERBAR_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* RIGHT_SQUARE_BRACKET_WITH_UNDERBAR_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LEFT_SQUARE_BRACKET_WITH_TICK_IN_TOP_CORNER_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x298d };
static wchar_t* LEFT_SQUARE_BRACKET_WITH_TICK_IN_TOP_CORNER_UNICODE_CHARACTER_CODE_MODEL = LEFT_SQUARE_BRACKET_WITH_TICK_IN_TOP_CORNER_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LEFT_SQUARE_BRACKET_WITH_TICK_IN_TOP_CORNER_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t RIGHT_SQUARE_BRACKET_WITH_TICK_IN_BOTTOM_CORNER_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x298e };
static wchar_t* RIGHT_SQUARE_BRACKET_WITH_TICK_IN_BOTTOM_CORNER_UNICODE_CHARACTER_CODE_MODEL = RIGHT_SQUARE_BRACKET_WITH_TICK_IN_BOTTOM_CORNER_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* RIGHT_SQUARE_BRACKET_WITH_TICK_IN_BOTTOM_CORNER_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LEFT_SQUARE_BRACKET_WITH_TICK_IN_BOTTOM_CORNER_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x298f };
static wchar_t* LEFT_SQUARE_BRACKET_WITH_TICK_IN_BOTTOM_CORNER_UNICODE_CHARACTER_CODE_MODEL = LEFT_SQUARE_BRACKET_WITH_TICK_IN_BOTTOM_CORNER_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LEFT_SQUARE_BRACKET_WITH_TICK_IN_BOTTOM_CORNER_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t RIGHT_SQUARE_BRACKET_WITH_TICK_IN_TOP_CORNER_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2990 };
static wchar_t* RIGHT_SQUARE_BRACKET_WITH_TICK_IN_TOP_CORNER_UNICODE_CHARACTER_CODE_MODEL = RIGHT_SQUARE_BRACKET_WITH_TICK_IN_TOP_CORNER_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* RIGHT_SQUARE_BRACKET_WITH_TICK_IN_TOP_CORNER_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LEFT_ANGLE_BRACKET_WITH_DOT_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2991 };
static wchar_t* LEFT_ANGLE_BRACKET_WITH_DOT_UNICODE_CHARACTER_CODE_MODEL = LEFT_ANGLE_BRACKET_WITH_DOT_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LEFT_ANGLE_BRACKET_WITH_DOT_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t RIGHT_ANGLE_BRACKET_WITH_DOT_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2992 };
static wchar_t* RIGHT_ANGLE_BRACKET_WITH_DOT_UNICODE_CHARACTER_CODE_MODEL = RIGHT_ANGLE_BRACKET_WITH_DOT_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* RIGHT_ANGLE_BRACKET_WITH_DOT_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LEFT_ARC_LESS_THAN_BRACKET_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2993 };
static wchar_t* LEFT_ARC_LESS_THAN_BRACKET_UNICODE_CHARACTER_CODE_MODEL = LEFT_ARC_LESS_THAN_BRACKET_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LEFT_ARC_LESS_THAN_BRACKET_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t RIGHT_ARC_GREATER_THAN_BRACKET_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2994 };
static wchar_t* RIGHT_ARC_GREATER_THAN_BRACKET_UNICODE_CHARACTER_CODE_MODEL = RIGHT_ARC_GREATER_THAN_BRACKET_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* RIGHT_ARC_GREATER_THAN_BRACKET_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t DOUBLE_LEFT_ARC_GREATER_THAN_BRACKET_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2995 };
static wchar_t* DOUBLE_LEFT_ARC_GREATER_THAN_BRACKET_UNICODE_CHARACTER_CODE_MODEL = DOUBLE_LEFT_ARC_GREATER_THAN_BRACKET_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* DOUBLE_LEFT_ARC_GREATER_THAN_BRACKET_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t DOUBLE_RIGHT_ARC_LESS_THAN_BRACKET_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2996 };
static wchar_t* DOUBLE_RIGHT_ARC_LESS_THAN_BRACKET_UNICODE_CHARACTER_CODE_MODEL = DOUBLE_RIGHT_ARC_LESS_THAN_BRACKET_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* DOUBLE_RIGHT_ARC_LESS_THAN_BRACKET_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t VERTICAL_ZIGZAG_LINE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x299a };
static wchar_t* VERTICAL_ZIGZAG_LINE_UNICODE_CHARACTER_CODE_MODEL = VERTICAL_ZIGZAG_LINE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* VERTICAL_ZIGZAG_LINE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t RIGHT_ANGLE_VARIANT_WITH_SQUARE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x299c };
static wchar_t* RIGHT_ANGLE_VARIANT_WITH_SQUARE_UNICODE_CHARACTER_CODE_MODEL = RIGHT_ANGLE_VARIANT_WITH_SQUARE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* RIGHT_ANGLE_VARIANT_WITH_SQUARE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MEASURED_RIGHT_ANGLE_WITH_DOT_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x299d };
static wchar_t* MEASURED_RIGHT_ANGLE_WITH_DOT_UNICODE_CHARACTER_CODE_MODEL = MEASURED_RIGHT_ANGLE_WITH_DOT_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MEASURED_RIGHT_ANGLE_WITH_DOT_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t ANGLE_WITH_UNDERBAR_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x29a4 };
static wchar_t* ANGLE_WITH_UNDERBAR_UNICODE_CHARACTER_CODE_MODEL = ANGLE_WITH_UNDERBAR_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* ANGLE_WITH_UNDERBAR_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t REVERSED_ANGLE_WITH_UNDERBAR_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x29a5 };
static wchar_t* REVERSED_ANGLE_WITH_UNDERBAR_UNICODE_CHARACTER_CODE_MODEL = REVERSED_ANGLE_WITH_UNDERBAR_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* REVERSED_ANGLE_WITH_UNDERBAR_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t OBLIQUE_ANGLE_OPENING_UP_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x29a6 };
static wchar_t* OBLIQUE_ANGLE_OPENING_UP_UNICODE_CHARACTER_CODE_MODEL = OBLIQUE_ANGLE_OPENING_UP_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* OBLIQUE_ANGLE_OPENING_UP_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t OBLIQUE_ANGLE_OPENING_DOWN_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x29a7 };
static wchar_t* OBLIQUE_ANGLE_OPENING_DOWN_UNICODE_CHARACTER_CODE_MODEL = OBLIQUE_ANGLE_OPENING_DOWN_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* OBLIQUE_ANGLE_OPENING_DOWN_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MEASURED_ANGLE_WITH_OPEN_ARM_ENDING_IN_ARROW_POINTING_UP_AND_RIGHT_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x29a8 };
static wchar_t* MEASURED_ANGLE_WITH_OPEN_ARM_ENDING_IN_ARROW_POINTING_UP_AND_RIGHT_UNICODE_CHARACTER_CODE_MODEL = MEASURED_ANGLE_WITH_OPEN_ARM_ENDING_IN_ARROW_POINTING_UP_AND_RIGHT_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MEASURED_ANGLE_WITH_OPEN_ARM_ENDING_IN_ARROW_POINTING_UP_AND_RIGHT_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MEASURED_ANGLE_WITH_OPEN_ARM_ENDING_IN_ARROW_POINTING_UP_AND_LEFT_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x29a9 };
static wchar_t* MEASURED_ANGLE_WITH_OPEN_ARM_ENDING_IN_ARROW_POINTING_UP_AND_LEFT_UNICODE_CHARACTER_CODE_MODEL = MEASURED_ANGLE_WITH_OPEN_ARM_ENDING_IN_ARROW_POINTING_UP_AND_LEFT_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MEASURED_ANGLE_WITH_OPEN_ARM_ENDING_IN_ARROW_POINTING_UP_AND_LEFT_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MEASURED_ANGLE_WITH_OPEN_ARM_ENDING_IN_ARROW_POINTING_DOWN_AND_RIGHT_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x29aa };
static wchar_t* MEASURED_ANGLE_WITH_OPEN_ARM_ENDING_IN_ARROW_POINTING_DOWN_AND_RIGHT_UNICODE_CHARACTER_CODE_MODEL = MEASURED_ANGLE_WITH_OPEN_ARM_ENDING_IN_ARROW_POINTING_DOWN_AND_RIGHT_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MEASURED_ANGLE_WITH_OPEN_ARM_ENDING_IN_ARROW_POINTING_DOWN_AND_RIGHT_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MEASURED_ANGLE_WITH_OPEN_ARM_ENDING_IN_ARROW_POINTING_DOWN_AND_LEFT_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x29ab };
static wchar_t* MEASURED_ANGLE_WITH_OPEN_ARM_ENDING_IN_ARROW_POINTING_DOWN_AND_LEFT_UNICODE_CHARACTER_CODE_MODEL = MEASURED_ANGLE_WITH_OPEN_ARM_ENDING_IN_ARROW_POINTING_DOWN_AND_LEFT_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MEASURED_ANGLE_WITH_OPEN_ARM_ENDING_IN_ARROW_POINTING_DOWN_AND_LEFT_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MEASURED_ANGLE_WITH_OPEN_ARM_ENDING_IN_ARROW_POINTING_RIGHT_AND_UP_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x29ac };
static wchar_t* MEASURED_ANGLE_WITH_OPEN_ARM_ENDING_IN_ARROW_POINTING_RIGHT_AND_UP_UNICODE_CHARACTER_CODE_MODEL = MEASURED_ANGLE_WITH_OPEN_ARM_ENDING_IN_ARROW_POINTING_RIGHT_AND_UP_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MEASURED_ANGLE_WITH_OPEN_ARM_ENDING_IN_ARROW_POINTING_RIGHT_AND_UP_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MEASURED_ANGLE_WITH_OPEN_ARM_ENDING_IN_ARROW_POINTING_LEFT_AND_UP_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x29ad };
static wchar_t* MEASURED_ANGLE_WITH_OPEN_ARM_ENDING_IN_ARROW_POINTING_LEFT_AND_UP_UNICODE_CHARACTER_CODE_MODEL = MEASURED_ANGLE_WITH_OPEN_ARM_ENDING_IN_ARROW_POINTING_LEFT_AND_UP_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MEASURED_ANGLE_WITH_OPEN_ARM_ENDING_IN_ARROW_POINTING_LEFT_AND_UP_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MEASURED_ANGLE_WITH_OPEN_ARM_ENDING_IN_ARROW_POINTING_RIGHT_AND_DOWN_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x29ae };
static wchar_t* MEASURED_ANGLE_WITH_OPEN_ARM_ENDING_IN_ARROW_POINTING_RIGHT_AND_DOWN_UNICODE_CHARACTER_CODE_MODEL = MEASURED_ANGLE_WITH_OPEN_ARM_ENDING_IN_ARROW_POINTING_RIGHT_AND_DOWN_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MEASURED_ANGLE_WITH_OPEN_ARM_ENDING_IN_ARROW_POINTING_RIGHT_AND_DOWN_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MEASURED_ANGLE_WITH_OPEN_ARM_ENDING_IN_ARROW_POINTING_LEFT_AND_DOWN_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x29af };
static wchar_t* MEASURED_ANGLE_WITH_OPEN_ARM_ENDING_IN_ARROW_POINTING_LEFT_AND_DOWN_UNICODE_CHARACTER_CODE_MODEL = MEASURED_ANGLE_WITH_OPEN_ARM_ENDING_IN_ARROW_POINTING_LEFT_AND_DOWN_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MEASURED_ANGLE_WITH_OPEN_ARM_ENDING_IN_ARROW_POINTING_LEFT_AND_DOWN_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t REVERSED_EMPTY_SET_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x29b0 };
static wchar_t* REVERSED_EMPTY_SET_UNICODE_CHARACTER_CODE_MODEL = REVERSED_EMPTY_SET_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* REVERSED_EMPTY_SET_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t EMPTY_SET_WITH_OVERBAR_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x29b1 };
static wchar_t* EMPTY_SET_WITH_OVERBAR_UNICODE_CHARACTER_CODE_MODEL = EMPTY_SET_WITH_OVERBAR_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* EMPTY_SET_WITH_OVERBAR_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t EMPTY_SET_WITH_SMALL_CIRCLE_ABOVE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x29b2 };
static wchar_t* EMPTY_SET_WITH_SMALL_CIRCLE_ABOVE_UNICODE_CHARACTER_CODE_MODEL = EMPTY_SET_WITH_SMALL_CIRCLE_ABOVE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* EMPTY_SET_WITH_SMALL_CIRCLE_ABOVE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t EMPTY_SET_WITH_RIGHT_ARROW_ABOVE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x29b3 };
static wchar_t* EMPTY_SET_WITH_RIGHT_ARROW_ABOVE_UNICODE_CHARACTER_CODE_MODEL = EMPTY_SET_WITH_RIGHT_ARROW_ABOVE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* EMPTY_SET_WITH_RIGHT_ARROW_ABOVE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t EMPTY_SET_WITH_LEFT_ARROW_ABOVE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x29b4 };
static wchar_t* EMPTY_SET_WITH_LEFT_ARROW_ABOVE_UNICODE_CHARACTER_CODE_MODEL = EMPTY_SET_WITH_LEFT_ARROW_ABOVE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* EMPTY_SET_WITH_LEFT_ARROW_ABOVE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t CIRCLE_WITH_HORIZONTAL_BAR_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x29b5 };
static wchar_t* CIRCLE_WITH_HORIZONTAL_BAR_UNICODE_CHARACTER_CODE_MODEL = CIRCLE_WITH_HORIZONTAL_BAR_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CIRCLE_WITH_HORIZONTAL_BAR_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t CIRCLED_VERTICAL_BAR_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x29b6 };
static wchar_t* CIRCLED_VERTICAL_BAR_UNICODE_CHARACTER_CODE_MODEL = CIRCLED_VERTICAL_BAR_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CIRCLED_VERTICAL_BAR_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t CIRCLED_PARALLEL_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x29b7 };
static wchar_t* CIRCLED_PARALLEL_UNICODE_CHARACTER_CODE_MODEL = CIRCLED_PARALLEL_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CIRCLED_PARALLEL_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t CIRCLED_PERPENDICULAR_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x29b9 };
static wchar_t* CIRCLED_PERPENDICULAR_UNICODE_CHARACTER_CODE_MODEL = CIRCLED_PERPENDICULAR_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CIRCLED_PERPENDICULAR_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t CIRCLE_WITH_SUPERIMPOSED_X_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x29bb };
static wchar_t* CIRCLE_WITH_SUPERIMPOSED_X_UNICODE_CHARACTER_CODE_MODEL = CIRCLE_WITH_SUPERIMPOSED_X_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CIRCLE_WITH_SUPERIMPOSED_X_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t CIRCLED_ANTICLOCKWISE_ROTATED_DIVISION_SIGN_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x29bc };
static wchar_t* CIRCLED_ANTICLOCKWISE_ROTATED_DIVISION_SIGN_UNICODE_CHARACTER_CODE_MODEL = CIRCLED_ANTICLOCKWISE_ROTATED_DIVISION_SIGN_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CIRCLED_ANTICLOCKWISE_ROTATED_DIVISION_SIGN_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t CIRCLED_WHITE_BULLET_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x29be };
static wchar_t* CIRCLED_WHITE_BULLET_UNICODE_CHARACTER_CODE_MODEL = CIRCLED_WHITE_BULLET_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CIRCLED_WHITE_BULLET_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t CIRCLED_BULLET_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x29bf };
static wchar_t* CIRCLED_BULLET_UNICODE_CHARACTER_CODE_MODEL = CIRCLED_BULLET_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CIRCLED_BULLET_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t CIRCLED_LESS_THAN_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x29c0 };
static wchar_t* CIRCLED_LESS_THAN_UNICODE_CHARACTER_CODE_MODEL = CIRCLED_LESS_THAN_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CIRCLED_LESS_THAN_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t CIRCLED_GREATER_THAN_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x29c1 };
static wchar_t* CIRCLED_GREATER_THAN_UNICODE_CHARACTER_CODE_MODEL = CIRCLED_GREATER_THAN_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CIRCLED_GREATER_THAN_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t CIRCLE_WITH_SMALL_CIRCLE_TO_THE_RIGHT_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x29c2 };
static wchar_t* CIRCLE_WITH_SMALL_CIRCLE_TO_THE_RIGHT_UNICODE_CHARACTER_CODE_MODEL = CIRCLE_WITH_SMALL_CIRCLE_TO_THE_RIGHT_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CIRCLE_WITH_SMALL_CIRCLE_TO_THE_RIGHT_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t CIRCLE_WITH_TWO_HORIZONTAL_STROKES_TO_THE_RIGHT_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x29c3 };
static wchar_t* CIRCLE_WITH_TWO_HORIZONTAL_STROKES_TO_THE_RIGHT_UNICODE_CHARACTER_CODE_MODEL = CIRCLE_WITH_TWO_HORIZONTAL_STROKES_TO_THE_RIGHT_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CIRCLE_WITH_TWO_HORIZONTAL_STROKES_TO_THE_RIGHT_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t SQUARED_RISING_DIAGONAL_SLASH_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x29c4 };
static wchar_t* SQUARED_RISING_DIAGONAL_SLASH_UNICODE_CHARACTER_CODE_MODEL = SQUARED_RISING_DIAGONAL_SLASH_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* SQUARED_RISING_DIAGONAL_SLASH_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t SQUARED_FALLING_DIAGONAL_SLASH_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x29c5 };
static wchar_t* SQUARED_FALLING_DIAGONAL_SLASH_UNICODE_CHARACTER_CODE_MODEL = SQUARED_FALLING_DIAGONAL_SLASH_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* SQUARED_FALLING_DIAGONAL_SLASH_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t TWO_JOINED_SQUARES_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x29c9 };
static wchar_t* TWO_JOINED_SQUARES_UNICODE_CHARACTER_CODE_MODEL = TWO_JOINED_SQUARES_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* TWO_JOINED_SQUARES_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t TRIANGLE_WITH_SERIFS_AT_BOTTOM_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x29cd };
static wchar_t* TRIANGLE_WITH_SERIFS_AT_BOTTOM_UNICODE_CHARACTER_CODE_MODEL = TRIANGLE_WITH_SERIFS_AT_BOTTOM_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* TRIANGLE_WITH_SERIFS_AT_BOTTOM_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t RIGHT_TRIANGLE_ABOVE_LEFT_TRIANGLE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x29ce };
static wchar_t* RIGHT_TRIANGLE_ABOVE_LEFT_TRIANGLE_UNICODE_CHARACTER_CODE_MODEL = RIGHT_TRIANGLE_ABOVE_LEFT_TRIANGLE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* RIGHT_TRIANGLE_ABOVE_LEFT_TRIANGLE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LEFT_TRIANGLE_BESIDE_VERTICAL_BAR_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x29cf };
static wchar_t* LEFT_TRIANGLE_BESIDE_VERTICAL_BAR_UNICODE_CHARACTER_CODE_MODEL = LEFT_TRIANGLE_BESIDE_VERTICAL_BAR_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LEFT_TRIANGLE_BESIDE_VERTICAL_BAR_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LEFT_TRIANGLE_BESIDE_VERTICAL_BAR_WITH_SLASH_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x29cf, 0x0338 };
static wchar_t* LEFT_TRIANGLE_BESIDE_VERTICAL_BAR_WITH_SLASH_UNICODE_CHARACTER_CODE_MODEL = LEFT_TRIANGLE_BESIDE_VERTICAL_BAR_WITH_SLASH_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LEFT_TRIANGLE_BESIDE_VERTICAL_BAR_WITH_SLASH_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_2_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t VERTICAL_BAR_BESIDE_RIGHT_TRIANGLE_WITH_SLASH_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x29d0, 0x0338 };
static wchar_t* VERTICAL_BAR_BESIDE_RIGHT_TRIANGLE_WITH_SLASH_UNICODE_CHARACTER_CODE_MODEL = VERTICAL_BAR_BESIDE_RIGHT_TRIANGLE_WITH_SLASH_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* VERTICAL_BAR_BESIDE_RIGHT_TRIANGLE_WITH_SLASH_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_2_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t VERTICAL_BAR_BESIDE_RIGHT_TRIANGLE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x29d0 };
static wchar_t* VERTICAL_BAR_BESIDE_RIGHT_TRIANGLE_UNICODE_CHARACTER_CODE_MODEL = VERTICAL_BAR_BESIDE_RIGHT_TRIANGLE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* VERTICAL_BAR_BESIDE_RIGHT_TRIANGLE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t INCOMPLETE_INFINITY_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x29dc };
static wchar_t* INCOMPLETE_INFINITY_UNICODE_CHARACTER_CODE_MODEL = INCOMPLETE_INFINITY_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* INCOMPLETE_INFINITY_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t TIE_OVER_INFINITY_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x29dd };
static wchar_t* TIE_OVER_INFINITY_UNICODE_CHARACTER_CODE_MODEL = TIE_OVER_INFINITY_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* TIE_OVER_INFINITY_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t INFINITY_NEGATED_WITH_VERTICAL_BAR_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x29de };
static wchar_t* INFINITY_NEGATED_WITH_VERTICAL_BAR_UNICODE_CHARACTER_CODE_MODEL = INFINITY_NEGATED_WITH_VERTICAL_BAR_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* INFINITY_NEGATED_WITH_VERTICAL_BAR_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t EQUALS_SIGN_AND_SLANTED_PARALLEL_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x29e3 };
static wchar_t* EQUALS_SIGN_AND_SLANTED_PARALLEL_UNICODE_CHARACTER_CODE_MODEL = EQUALS_SIGN_AND_SLANTED_PARALLEL_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* EQUALS_SIGN_AND_SLANTED_PARALLEL_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t EQUALS_SIGN_AND_SLANTED_PARALLEL_WITH_TILDE_ABOVE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x29e4 };
static wchar_t* EQUALS_SIGN_AND_SLANTED_PARALLEL_WITH_TILDE_ABOVE_UNICODE_CHARACTER_CODE_MODEL = EQUALS_SIGN_AND_SLANTED_PARALLEL_WITH_TILDE_ABOVE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* EQUALS_SIGN_AND_SLANTED_PARALLEL_WITH_TILDE_ABOVE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t IDENTICAL_TO_AND_SLANTED_PARALLEL_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x29e5 };
static wchar_t* IDENTICAL_TO_AND_SLANTED_PARALLEL_UNICODE_CHARACTER_CODE_MODEL = IDENTICAL_TO_AND_SLANTED_PARALLEL_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* IDENTICAL_TO_AND_SLANTED_PARALLEL_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t BLACK_LOZENGE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x29eb };
static wchar_t* BLACK_LOZENGE_UNICODE_CHARACTER_CODE_MODEL = BLACK_LOZENGE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* BLACK_LOZENGE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t RULE_DELAYED_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x29f4 };
static wchar_t* RULE_DELAYED_UNICODE_CHARACTER_CODE_MODEL = RULE_DELAYED_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* RULE_DELAYED_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t SOLIDUS_WITH_OVERBAR_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x29f6 };
static wchar_t* SOLIDUS_WITH_OVERBAR_UNICODE_CHARACTER_CODE_MODEL = SOLIDUS_WITH_OVERBAR_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* SOLIDUS_WITH_OVERBAR_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t N_ARY_CIRCLED_DOT_OPERATOR_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2a00 };
static wchar_t* N_ARY_CIRCLED_DOT_OPERATOR_UNICODE_CHARACTER_CODE_MODEL = N_ARY_CIRCLED_DOT_OPERATOR_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* N_ARY_CIRCLED_DOT_OPERATOR_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t N_ARY_CIRCLED_PLUS_OPERATOR_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2a01 };
static wchar_t* N_ARY_CIRCLED_PLUS_OPERATOR_UNICODE_CHARACTER_CODE_MODEL = N_ARY_CIRCLED_PLUS_OPERATOR_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* N_ARY_CIRCLED_PLUS_OPERATOR_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t N_ARY_CIRCLED_TIMES_OPERATOR_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2a02 };
static wchar_t* N_ARY_CIRCLED_TIMES_OPERATOR_UNICODE_CHARACTER_CODE_MODEL = N_ARY_CIRCLED_TIMES_OPERATOR_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* N_ARY_CIRCLED_TIMES_OPERATOR_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t N_ARY_UNION_OPERATOR_WITH_PLUS_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2a04 };
static wchar_t* N_ARY_UNION_OPERATOR_WITH_PLUS_UNICODE_CHARACTER_CODE_MODEL = N_ARY_UNION_OPERATOR_WITH_PLUS_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* N_ARY_UNION_OPERATOR_WITH_PLUS_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t N_ARY_SQUARE_UNION_OPERATOR_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2a06 };
static wchar_t* N_ARY_SQUARE_UNION_OPERATOR_UNICODE_CHARACTER_CODE_MODEL = N_ARY_SQUARE_UNION_OPERATOR_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* N_ARY_SQUARE_UNION_OPERATOR_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t QUADRUPLE_INTEGRAL_OPERATOR_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2a0c };
static wchar_t* QUADRUPLE_INTEGRAL_OPERATOR_UNICODE_CHARACTER_CODE_MODEL = QUADRUPLE_INTEGRAL_OPERATOR_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* QUADRUPLE_INTEGRAL_OPERATOR_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t FINITE_PART_INTEGRAL_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2a0d };
static wchar_t* FINITE_PART_INTEGRAL_UNICODE_CHARACTER_CODE_MODEL = FINITE_PART_INTEGRAL_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* FINITE_PART_INTEGRAL_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t CIRCULATION_FUNCTION_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2a10 };
static wchar_t* CIRCULATION_FUNCTION_UNICODE_CHARACTER_CODE_MODEL = CIRCULATION_FUNCTION_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CIRCULATION_FUNCTION_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t ANTICLOCKWISE_INTEGRATION_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2a11 };
static wchar_t* ANTICLOCKWISE_INTEGRATION_UNICODE_CHARACTER_CODE_MODEL = ANTICLOCKWISE_INTEGRATION_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* ANTICLOCKWISE_INTEGRATION_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LINE_INTEGRATION_WITH_RECTANGULAR_PATH_AROUND_POLE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2a12 };
static wchar_t* LINE_INTEGRATION_WITH_RECTANGULAR_PATH_AROUND_POLE_UNICODE_CHARACTER_CODE_MODEL = LINE_INTEGRATION_WITH_RECTANGULAR_PATH_AROUND_POLE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LINE_INTEGRATION_WITH_RECTANGULAR_PATH_AROUND_POLE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LINE_INTEGRATION_WITH_SEMICIRCULAR_PATH_AROUND_POLE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2a13 };
static wchar_t* LINE_INTEGRATION_WITH_SEMICIRCULAR_PATH_AROUND_POLE_UNICODE_CHARACTER_CODE_MODEL = LINE_INTEGRATION_WITH_SEMICIRCULAR_PATH_AROUND_POLE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LINE_INTEGRATION_WITH_SEMICIRCULAR_PATH_AROUND_POLE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LINE_INTEGRATION_NOT_INCLUDING_THE_POLE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2a14 };
static wchar_t* LINE_INTEGRATION_NOT_INCLUDING_THE_POLE_UNICODE_CHARACTER_CODE_MODEL = LINE_INTEGRATION_NOT_INCLUDING_THE_POLE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LINE_INTEGRATION_NOT_INCLUDING_THE_POLE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t INTEGRAL_AROUND_A_POINT_OPERATOR_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2a15 };
static wchar_t* INTEGRAL_AROUND_A_POINT_OPERATOR_UNICODE_CHARACTER_CODE_MODEL = INTEGRAL_AROUND_A_POINT_OPERATOR_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* INTEGRAL_AROUND_A_POINT_OPERATOR_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t QUATERNION_INTEGRAL_OPERATOR_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2a16 };
static wchar_t* QUATERNION_INTEGRAL_OPERATOR_UNICODE_CHARACTER_CODE_MODEL = QUATERNION_INTEGRAL_OPERATOR_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* QUATERNION_INTEGRAL_OPERATOR_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t INTEGRAL_WITH_LEFTWARDS_ARROW_WITH_HOOK_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2a17 };
static wchar_t* INTEGRAL_WITH_LEFTWARDS_ARROW_WITH_HOOK_UNICODE_CHARACTER_CODE_MODEL = INTEGRAL_WITH_LEFTWARDS_ARROW_WITH_HOOK_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* INTEGRAL_WITH_LEFTWARDS_ARROW_WITH_HOOK_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t PLUS_SIGN_WITH_SMALL_CIRCLE_ABOVE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2a22 };
static wchar_t* PLUS_SIGN_WITH_SMALL_CIRCLE_ABOVE_UNICODE_CHARACTER_CODE_MODEL = PLUS_SIGN_WITH_SMALL_CIRCLE_ABOVE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* PLUS_SIGN_WITH_SMALL_CIRCLE_ABOVE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t PLUS_SIGN_WITH_CIRCUMFLEX_ACCENT_ABOVE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2a23 };
static wchar_t* PLUS_SIGN_WITH_CIRCUMFLEX_ACCENT_ABOVE_UNICODE_CHARACTER_CODE_MODEL = PLUS_SIGN_WITH_CIRCUMFLEX_ACCENT_ABOVE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* PLUS_SIGN_WITH_CIRCUMFLEX_ACCENT_ABOVE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t PLUS_SIGN_WITH_TILDE_ABOVE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2a24 };
static wchar_t* PLUS_SIGN_WITH_TILDE_ABOVE_UNICODE_CHARACTER_CODE_MODEL = PLUS_SIGN_WITH_TILDE_ABOVE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* PLUS_SIGN_WITH_TILDE_ABOVE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t PLUS_SIGN_WITH_DOT_BELOW_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2a25 };
static wchar_t* PLUS_SIGN_WITH_DOT_BELOW_UNICODE_CHARACTER_CODE_MODEL = PLUS_SIGN_WITH_DOT_BELOW_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* PLUS_SIGN_WITH_DOT_BELOW_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t PLUS_SIGN_WITH_TILDE_BELOW_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2a26 };
static wchar_t* PLUS_SIGN_WITH_TILDE_BELOW_UNICODE_CHARACTER_CODE_MODEL = PLUS_SIGN_WITH_TILDE_BELOW_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* PLUS_SIGN_WITH_TILDE_BELOW_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t PLUS_SIGN_WITH_SUBSCRIPT_TWO_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2a27 };
static wchar_t* PLUS_SIGN_WITH_SUBSCRIPT_TWO_UNICODE_CHARACTER_CODE_MODEL = PLUS_SIGN_WITH_SUBSCRIPT_TWO_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* PLUS_SIGN_WITH_SUBSCRIPT_TWO_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MINUS_SIGN_WITH_COMMA_ABOVE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2a29 };
static wchar_t* MINUS_SIGN_WITH_COMMA_ABOVE_UNICODE_CHARACTER_CODE_MODEL = MINUS_SIGN_WITH_COMMA_ABOVE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MINUS_SIGN_WITH_COMMA_ABOVE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MINUS_SIGN_WITH_DOT_BELOW_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2a2a };
static wchar_t* MINUS_SIGN_WITH_DOT_BELOW_UNICODE_CHARACTER_CODE_MODEL = MINUS_SIGN_WITH_DOT_BELOW_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MINUS_SIGN_WITH_DOT_BELOW_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t PLUS_SIGN_IN_LEFT_HALF_CIRCLE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2a2d };
static wchar_t* PLUS_SIGN_IN_LEFT_HALF_CIRCLE_UNICODE_CHARACTER_CODE_MODEL = PLUS_SIGN_IN_LEFT_HALF_CIRCLE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* PLUS_SIGN_IN_LEFT_HALF_CIRCLE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t PLUS_SIGN_IN_RIGHT_HALF_CIRCLE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2a2e };
static wchar_t* PLUS_SIGN_IN_RIGHT_HALF_CIRCLE_UNICODE_CHARACTER_CODE_MODEL = PLUS_SIGN_IN_RIGHT_HALF_CIRCLE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* PLUS_SIGN_IN_RIGHT_HALF_CIRCLE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t VECTOR_OR_CROSS_PRODUCT_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2a2f };
static wchar_t* VECTOR_OR_CROSS_PRODUCT_UNICODE_CHARACTER_CODE_MODEL = VECTOR_OR_CROSS_PRODUCT_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* VECTOR_OR_CROSS_PRODUCT_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MULTIPLICATION_SIGN_WITH_DOT_ABOVE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2a30 };
static wchar_t* MULTIPLICATION_SIGN_WITH_DOT_ABOVE_UNICODE_CHARACTER_CODE_MODEL = MULTIPLICATION_SIGN_WITH_DOT_ABOVE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MULTIPLICATION_SIGN_WITH_DOT_ABOVE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MULTIPLICATION_SIGN_WITH_UNDERBAR_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2a31 };
static wchar_t* MULTIPLICATION_SIGN_WITH_UNDERBAR_UNICODE_CHARACTER_CODE_MODEL = MULTIPLICATION_SIGN_WITH_UNDERBAR_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MULTIPLICATION_SIGN_WITH_UNDERBAR_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t SMASH_PRODUCT_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2a33 };
static wchar_t* SMASH_PRODUCT_UNICODE_CHARACTER_CODE_MODEL = SMASH_PRODUCT_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* SMASH_PRODUCT_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MULTIPLICATION_SIGN_IN_LEFT_HALF_CIRCLE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2a34 };
static wchar_t* MULTIPLICATION_SIGN_IN_LEFT_HALF_CIRCLE_UNICODE_CHARACTER_CODE_MODEL = MULTIPLICATION_SIGN_IN_LEFT_HALF_CIRCLE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MULTIPLICATION_SIGN_IN_LEFT_HALF_CIRCLE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MULTIPLICATION_SIGN_IN_RIGHT_HALF_CIRCLE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2a35 };
static wchar_t* MULTIPLICATION_SIGN_IN_RIGHT_HALF_CIRCLE_UNICODE_CHARACTER_CODE_MODEL = MULTIPLICATION_SIGN_IN_RIGHT_HALF_CIRCLE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MULTIPLICATION_SIGN_IN_RIGHT_HALF_CIRCLE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t CIRCLED_MULTIPLICATION_SIGN_WITH_CIRCUMFLEX_ACCENT_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2a36 };
static wchar_t* CIRCLED_MULTIPLICATION_SIGN_WITH_CIRCUMFLEX_ACCENT_UNICODE_CHARACTER_CODE_MODEL = CIRCLED_MULTIPLICATION_SIGN_WITH_CIRCUMFLEX_ACCENT_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CIRCLED_MULTIPLICATION_SIGN_WITH_CIRCUMFLEX_ACCENT_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MULTIPLICATION_SIGN_IN_DOUBLE_CIRCLE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2a37 };
static wchar_t* MULTIPLICATION_SIGN_IN_DOUBLE_CIRCLE_UNICODE_CHARACTER_CODE_MODEL = MULTIPLICATION_SIGN_IN_DOUBLE_CIRCLE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MULTIPLICATION_SIGN_IN_DOUBLE_CIRCLE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t CIRCLED_DIVISION_SIGN_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2a38 };
static wchar_t* CIRCLED_DIVISION_SIGN_UNICODE_CHARACTER_CODE_MODEL = CIRCLED_DIVISION_SIGN_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CIRCLED_DIVISION_SIGN_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t PLUS_SIGN_IN_TRIANGLE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2a39 };
static wchar_t* PLUS_SIGN_IN_TRIANGLE_UNICODE_CHARACTER_CODE_MODEL = PLUS_SIGN_IN_TRIANGLE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* PLUS_SIGN_IN_TRIANGLE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MINUS_SIGN_IN_TRIANGLE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2a3a };
static wchar_t* MINUS_SIGN_IN_TRIANGLE_UNICODE_CHARACTER_CODE_MODEL = MINUS_SIGN_IN_TRIANGLE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MINUS_SIGN_IN_TRIANGLE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MULTIPLICATION_SIGN_IN_TRIANGLE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2a3b };
static wchar_t* MULTIPLICATION_SIGN_IN_TRIANGLE_UNICODE_CHARACTER_CODE_MODEL = MULTIPLICATION_SIGN_IN_TRIANGLE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MULTIPLICATION_SIGN_IN_TRIANGLE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t INTERIOR_PRODUCT_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2a3c };
static wchar_t* INTERIOR_PRODUCT_UNICODE_CHARACTER_CODE_MODEL = INTERIOR_PRODUCT_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* INTERIOR_PRODUCT_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t AMALGAMATION_OR_COPRODUCT_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2a3f };
static wchar_t* AMALGAMATION_OR_COPRODUCT_UNICODE_CHARACTER_CODE_MODEL = AMALGAMATION_OR_COPRODUCT_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* AMALGAMATION_OR_COPRODUCT_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t INTERSECTION_WITH_DOT_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2a40 };
static wchar_t* INTERSECTION_WITH_DOT_UNICODE_CHARACTER_CODE_MODEL = INTERSECTION_WITH_DOT_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* INTERSECTION_WITH_DOT_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t UNION_WITH_OVERBAR_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2a42 };
static wchar_t* UNION_WITH_OVERBAR_UNICODE_CHARACTER_CODE_MODEL = UNION_WITH_OVERBAR_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* UNION_WITH_OVERBAR_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t INTERSECTION_WITH_OVERBAR_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2a43 };
static wchar_t* INTERSECTION_WITH_OVERBAR_UNICODE_CHARACTER_CODE_MODEL = INTERSECTION_WITH_OVERBAR_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* INTERSECTION_WITH_OVERBAR_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t INTERSECTION_WITH_LOGICAL_AND_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2a44 };
static wchar_t* INTERSECTION_WITH_LOGICAL_AND_UNICODE_CHARACTER_CODE_MODEL = INTERSECTION_WITH_LOGICAL_AND_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* INTERSECTION_WITH_LOGICAL_AND_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t UNION_WITH_LOGICAL_OR_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2a45 };
static wchar_t* UNION_WITH_LOGICAL_OR_UNICODE_CHARACTER_CODE_MODEL = UNION_WITH_LOGICAL_OR_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* UNION_WITH_LOGICAL_OR_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t UNION_ABOVE_INTERSECTION_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2a46 };
static wchar_t* UNION_ABOVE_INTERSECTION_UNICODE_CHARACTER_CODE_MODEL = UNION_ABOVE_INTERSECTION_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* UNION_ABOVE_INTERSECTION_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t INTERSECTION_ABOVE_UNION_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2a47 };
static wchar_t* INTERSECTION_ABOVE_UNION_UNICODE_CHARACTER_CODE_MODEL = INTERSECTION_ABOVE_UNION_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* INTERSECTION_ABOVE_UNION_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t UNION_ABOVE_BAR_ABOVE_INTERSECTION_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2a48 };
static wchar_t* UNION_ABOVE_BAR_ABOVE_INTERSECTION_UNICODE_CHARACTER_CODE_MODEL = UNION_ABOVE_BAR_ABOVE_INTERSECTION_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* UNION_ABOVE_BAR_ABOVE_INTERSECTION_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t INTERSECTION_ABOVE_BAR_ABOVE_UNION_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2a49 };
static wchar_t* INTERSECTION_ABOVE_BAR_ABOVE_UNION_UNICODE_CHARACTER_CODE_MODEL = INTERSECTION_ABOVE_BAR_ABOVE_UNION_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* INTERSECTION_ABOVE_BAR_ABOVE_UNION_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t UNION_BESIDE_AND_JOINED_WITH_UNION_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2a4a };
static wchar_t* UNION_BESIDE_AND_JOINED_WITH_UNION_UNICODE_CHARACTER_CODE_MODEL = UNION_BESIDE_AND_JOINED_WITH_UNION_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* UNION_BESIDE_AND_JOINED_WITH_UNION_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t INTERSECTION_BESIDE_AND_JOINED_WITH_INTERSECTION_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2a4b };
static wchar_t* INTERSECTION_BESIDE_AND_JOINED_WITH_INTERSECTION_UNICODE_CHARACTER_CODE_MODEL = INTERSECTION_BESIDE_AND_JOINED_WITH_INTERSECTION_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* INTERSECTION_BESIDE_AND_JOINED_WITH_INTERSECTION_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t CLOSED_UNION_WITH_SERIFS_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2a4c };
static wchar_t* CLOSED_UNION_WITH_SERIFS_UNICODE_CHARACTER_CODE_MODEL = CLOSED_UNION_WITH_SERIFS_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CLOSED_UNION_WITH_SERIFS_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t CLOSED_INTERSECTION_WITH_SERIFS_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2a4d };
static wchar_t* CLOSED_INTERSECTION_WITH_SERIFS_UNICODE_CHARACTER_CODE_MODEL = CLOSED_INTERSECTION_WITH_SERIFS_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CLOSED_INTERSECTION_WITH_SERIFS_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t CLOSED_UNION_WITH_SERIFS_AND_SMASH_PRODUCT_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2a50 };
static wchar_t* CLOSED_UNION_WITH_SERIFS_AND_SMASH_PRODUCT_UNICODE_CHARACTER_CODE_MODEL = CLOSED_UNION_WITH_SERIFS_AND_SMASH_PRODUCT_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CLOSED_UNION_WITH_SERIFS_AND_SMASH_PRODUCT_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t DOUBLE_LOGICAL_AND_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2a53 };
static wchar_t* DOUBLE_LOGICAL_AND_UNICODE_CHARACTER_CODE_MODEL = DOUBLE_LOGICAL_AND_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* DOUBLE_LOGICAL_AND_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t DOUBLE_LOGICAL_OR_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2a54 };
static wchar_t* DOUBLE_LOGICAL_OR_UNICODE_CHARACTER_CODE_MODEL = DOUBLE_LOGICAL_OR_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* DOUBLE_LOGICAL_OR_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t TWO_INTERSECTING_LOGICAL_AND_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2a55 };
static wchar_t* TWO_INTERSECTING_LOGICAL_AND_UNICODE_CHARACTER_CODE_MODEL = TWO_INTERSECTING_LOGICAL_AND_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* TWO_INTERSECTING_LOGICAL_AND_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t TWO_INTERSECTING_LOGICAL_OR_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2a56 };
static wchar_t* TWO_INTERSECTING_LOGICAL_OR_UNICODE_CHARACTER_CODE_MODEL = TWO_INTERSECTING_LOGICAL_OR_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* TWO_INTERSECTING_LOGICAL_OR_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t SLOPING_LARGE_OR_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2a57 };
static wchar_t* SLOPING_LARGE_OR_UNICODE_CHARACTER_CODE_MODEL = SLOPING_LARGE_OR_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* SLOPING_LARGE_OR_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t SLOPING_LARGE_AND_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2a58 };
static wchar_t* SLOPING_LARGE_AND_UNICODE_CHARACTER_CODE_MODEL = SLOPING_LARGE_AND_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* SLOPING_LARGE_AND_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LOGICAL_AND_WITH_MIDDLE_STEM_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2a5a };
static wchar_t* LOGICAL_AND_WITH_MIDDLE_STEM_UNICODE_CHARACTER_CODE_MODEL = LOGICAL_AND_WITH_MIDDLE_STEM_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LOGICAL_AND_WITH_MIDDLE_STEM_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LOGICAL_OR_WITH_MIDDLE_STEM_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2a5b };
static wchar_t* LOGICAL_OR_WITH_MIDDLE_STEM_UNICODE_CHARACTER_CODE_MODEL = LOGICAL_OR_WITH_MIDDLE_STEM_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LOGICAL_OR_WITH_MIDDLE_STEM_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LOGICAL_AND_WITH_HORIZONTAL_DASH_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2a5c };
static wchar_t* LOGICAL_AND_WITH_HORIZONTAL_DASH_UNICODE_CHARACTER_CODE_MODEL = LOGICAL_AND_WITH_HORIZONTAL_DASH_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LOGICAL_AND_WITH_HORIZONTAL_DASH_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LOGICAL_OR_WITH_HORIZONTAL_DASH_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2a5d };
static wchar_t* LOGICAL_OR_WITH_HORIZONTAL_DASH_UNICODE_CHARACTER_CODE_MODEL = LOGICAL_OR_WITH_HORIZONTAL_DASH_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LOGICAL_OR_WITH_HORIZONTAL_DASH_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LOGICAL_AND_WITH_UNDERBAR_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2a5f };
static wchar_t* LOGICAL_AND_WITH_UNDERBAR_UNICODE_CHARACTER_CODE_MODEL = LOGICAL_AND_WITH_UNDERBAR_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LOGICAL_AND_WITH_UNDERBAR_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t EQUALS_SIGN_WITH_DOT_BELOW_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2a66 };
static wchar_t* EQUALS_SIGN_WITH_DOT_BELOW_UNICODE_CHARACTER_CODE_MODEL = EQUALS_SIGN_WITH_DOT_BELOW_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* EQUALS_SIGN_WITH_DOT_BELOW_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t TILDE_OPERATOR_WITH_DOT_ABOVE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2a6a };
static wchar_t* TILDE_OPERATOR_WITH_DOT_ABOVE_UNICODE_CHARACTER_CODE_MODEL = TILDE_OPERATOR_WITH_DOT_ABOVE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* TILDE_OPERATOR_WITH_DOT_ABOVE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t CONGRUENT_WITH_DOT_ABOVE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2a6d };
static wchar_t* CONGRUENT_WITH_DOT_ABOVE_UNICODE_CHARACTER_CODE_MODEL = CONGRUENT_WITH_DOT_ABOVE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CONGRUENT_WITH_DOT_ABOVE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t CONGRUENT_WITH_DOT_ABOVE_WITH_SLASH_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2a6d, 0x0338 };
static wchar_t* CONGRUENT_WITH_DOT_ABOVE_WITH_SLASH_UNICODE_CHARACTER_CODE_MODEL = CONGRUENT_WITH_DOT_ABOVE_WITH_SLASH_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CONGRUENT_WITH_DOT_ABOVE_WITH_SLASH_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_2_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t EQUALS_WITH_ASTERISK_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2a6e };
static wchar_t* EQUALS_WITH_ASTERISK_UNICODE_CHARACTER_CODE_MODEL = EQUALS_WITH_ASTERISK_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* EQUALS_WITH_ASTERISK_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t ALMOST_EQUAL_TO_WITH_CIRCUMFLEX_ACCENT_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2a6f };
static wchar_t* ALMOST_EQUAL_TO_WITH_CIRCUMFLEX_ACCENT_UNICODE_CHARACTER_CODE_MODEL = ALMOST_EQUAL_TO_WITH_CIRCUMFLEX_ACCENT_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* ALMOST_EQUAL_TO_WITH_CIRCUMFLEX_ACCENT_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t APPROXIMATELY_EQUAL_OR_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2a70 };
static wchar_t* APPROXIMATELY_EQUAL_OR_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL = APPROXIMATELY_EQUAL_OR_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* APPROXIMATELY_EQUAL_OR_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t APPROXIMATELY_EQUAL_OR_EQUAL_TO_WITH_SLASH_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2a70, 0x0338 };
static wchar_t* APPROXIMATELY_EQUAL_OR_EQUAL_TO_WITH_SLASH_UNICODE_CHARACTER_CODE_MODEL = APPROXIMATELY_EQUAL_OR_EQUAL_TO_WITH_SLASH_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* APPROXIMATELY_EQUAL_OR_EQUAL_TO_WITH_SLASH_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_2_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t EQUALS_SIGN_ABOVE_PLUS_SIGN_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2a71 };
static wchar_t* EQUALS_SIGN_ABOVE_PLUS_SIGN_UNICODE_CHARACTER_CODE_MODEL = EQUALS_SIGN_ABOVE_PLUS_SIGN_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* EQUALS_SIGN_ABOVE_PLUS_SIGN_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t PLUS_SIGN_ABOVE_EQUALS_SIGN_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2a72 };
static wchar_t* PLUS_SIGN_ABOVE_EQUALS_SIGN_UNICODE_CHARACTER_CODE_MODEL = PLUS_SIGN_ABOVE_EQUALS_SIGN_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* PLUS_SIGN_ABOVE_EQUALS_SIGN_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t EQUALS_SIGN_ABOVE_TILDE_OPERATOR_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2a73 };
static wchar_t* EQUALS_SIGN_ABOVE_TILDE_OPERATOR_UNICODE_CHARACTER_CODE_MODEL = EQUALS_SIGN_ABOVE_TILDE_OPERATOR_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* EQUALS_SIGN_ABOVE_TILDE_OPERATOR_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t DOUBLE_COLON_EQUAL_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2a74 };
static wchar_t* DOUBLE_COLON_EQUAL_UNICODE_CHARACTER_CODE_MODEL = DOUBLE_COLON_EQUAL_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* DOUBLE_COLON_EQUAL_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t TWO_CONSECUTIVE_EQUALS_SIGNS_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2a75 };
static wchar_t* TWO_CONSECUTIVE_EQUALS_SIGNS_UNICODE_CHARACTER_CODE_MODEL = TWO_CONSECUTIVE_EQUALS_SIGNS_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* TWO_CONSECUTIVE_EQUALS_SIGNS_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t EQUALS_SIGN_WITH_TWO_DOTS_ABOVE_AND_TWO_DOTS_BELOW_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2a77 };
static wchar_t* EQUALS_SIGN_WITH_TWO_DOTS_ABOVE_AND_TWO_DOTS_BELOW_UNICODE_CHARACTER_CODE_MODEL = EQUALS_SIGN_WITH_TWO_DOTS_ABOVE_AND_TWO_DOTS_BELOW_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* EQUALS_SIGN_WITH_TWO_DOTS_ABOVE_AND_TWO_DOTS_BELOW_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t EQUIVALENT_WITH_FOUR_DOTS_ABOVE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2a78 };
static wchar_t* EQUIVALENT_WITH_FOUR_DOTS_ABOVE_UNICODE_CHARACTER_CODE_MODEL = EQUIVALENT_WITH_FOUR_DOTS_ABOVE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* EQUIVALENT_WITH_FOUR_DOTS_ABOVE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LESS_THAN_WITH_CIRCLE_INSIDE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2a79 };
static wchar_t* LESS_THAN_WITH_CIRCLE_INSIDE_UNICODE_CHARACTER_CODE_MODEL = LESS_THAN_WITH_CIRCLE_INSIDE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LESS_THAN_WITH_CIRCLE_INSIDE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t GREATER_THAN_WITH_CIRCLE_INSIDE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2a7a };
static wchar_t* GREATER_THAN_WITH_CIRCLE_INSIDE_UNICODE_CHARACTER_CODE_MODEL = GREATER_THAN_WITH_CIRCLE_INSIDE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* GREATER_THAN_WITH_CIRCLE_INSIDE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LESS_THAN_WITH_QUESTION_MARK_ABOVE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2a7b };
static wchar_t* LESS_THAN_WITH_QUESTION_MARK_ABOVE_UNICODE_CHARACTER_CODE_MODEL = LESS_THAN_WITH_QUESTION_MARK_ABOVE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LESS_THAN_WITH_QUESTION_MARK_ABOVE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t GREATER_THAN_WITH_QUESTION_MARK_ABOVE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2a7c };
static wchar_t* GREATER_THAN_WITH_QUESTION_MARK_ABOVE_UNICODE_CHARACTER_CODE_MODEL = GREATER_THAN_WITH_QUESTION_MARK_ABOVE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* GREATER_THAN_WITH_QUESTION_MARK_ABOVE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LESS_THAN_OR_SLANTED_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2a7d };
static wchar_t* LESS_THAN_OR_SLANTED_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL = LESS_THAN_OR_SLANTED_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LESS_THAN_OR_SLANTED_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LESS_THAN_OR_SLANTED_EQUAL_TO_WITH_SLASH_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2a7d, 0x0338 };
static wchar_t* LESS_THAN_OR_SLANTED_EQUAL_TO_WITH_SLASH_UNICODE_CHARACTER_CODE_MODEL = LESS_THAN_OR_SLANTED_EQUAL_TO_WITH_SLASH_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LESS_THAN_OR_SLANTED_EQUAL_TO_WITH_SLASH_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_2_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t GREATER_THAN_OR_SLANTED_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2a7e };
static wchar_t* GREATER_THAN_OR_SLANTED_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL = GREATER_THAN_OR_SLANTED_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* GREATER_THAN_OR_SLANTED_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t GREATER_THAN_OR_SLANTED_EQUAL_TO_WITH_SLASH_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2a7e, 0x0338 };
static wchar_t* GREATER_THAN_OR_SLANTED_EQUAL_TO_WITH_SLASH_UNICODE_CHARACTER_CODE_MODEL = GREATER_THAN_OR_SLANTED_EQUAL_TO_WITH_SLASH_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* GREATER_THAN_OR_SLANTED_EQUAL_TO_WITH_SLASH_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_2_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LESS_THAN_OR_SLANTED_EQUAL_TO_WITH_DOT_INSIDE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2a7f };
static wchar_t* LESS_THAN_OR_SLANTED_EQUAL_TO_WITH_DOT_INSIDE_UNICODE_CHARACTER_CODE_MODEL = LESS_THAN_OR_SLANTED_EQUAL_TO_WITH_DOT_INSIDE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LESS_THAN_OR_SLANTED_EQUAL_TO_WITH_DOT_INSIDE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t GREATER_THAN_OR_SLANTED_EQUAL_TO_WITH_DOT_INSIDE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2a80 };
static wchar_t* GREATER_THAN_OR_SLANTED_EQUAL_TO_WITH_DOT_INSIDE_UNICODE_CHARACTER_CODE_MODEL = GREATER_THAN_OR_SLANTED_EQUAL_TO_WITH_DOT_INSIDE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* GREATER_THAN_OR_SLANTED_EQUAL_TO_WITH_DOT_INSIDE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LESS_THAN_OR_SLANTED_EQUAL_TO_WITH_DOT_ABOVE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2a81 };
static wchar_t* LESS_THAN_OR_SLANTED_EQUAL_TO_WITH_DOT_ABOVE_UNICODE_CHARACTER_CODE_MODEL = LESS_THAN_OR_SLANTED_EQUAL_TO_WITH_DOT_ABOVE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LESS_THAN_OR_SLANTED_EQUAL_TO_WITH_DOT_ABOVE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t GREATER_THAN_OR_SLANTED_EQUAL_TO_WITH_DOT_ABOVE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2a82 };
static wchar_t* GREATER_THAN_OR_SLANTED_EQUAL_TO_WITH_DOT_ABOVE_UNICODE_CHARACTER_CODE_MODEL = GREATER_THAN_OR_SLANTED_EQUAL_TO_WITH_DOT_ABOVE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* GREATER_THAN_OR_SLANTED_EQUAL_TO_WITH_DOT_ABOVE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LESS_THAN_OR_SLANTED_EQUAL_TO_WITH_DOT_ABOVE_RIGHT_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2a83 };
static wchar_t* LESS_THAN_OR_SLANTED_EQUAL_TO_WITH_DOT_ABOVE_RIGHT_UNICODE_CHARACTER_CODE_MODEL = LESS_THAN_OR_SLANTED_EQUAL_TO_WITH_DOT_ABOVE_RIGHT_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LESS_THAN_OR_SLANTED_EQUAL_TO_WITH_DOT_ABOVE_RIGHT_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t GREATER_THAN_OR_SLANTED_EQUAL_TO_WITH_DOT_ABOVE_LEFT_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2a84 };
static wchar_t* GREATER_THAN_OR_SLANTED_EQUAL_TO_WITH_DOT_ABOVE_LEFT_UNICODE_CHARACTER_CODE_MODEL = GREATER_THAN_OR_SLANTED_EQUAL_TO_WITH_DOT_ABOVE_LEFT_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* GREATER_THAN_OR_SLANTED_EQUAL_TO_WITH_DOT_ABOVE_LEFT_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LESS_THAN_OR_APPROXIMATE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2a85 };
static wchar_t* LESS_THAN_OR_APPROXIMATE_UNICODE_CHARACTER_CODE_MODEL = LESS_THAN_OR_APPROXIMATE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LESS_THAN_OR_APPROXIMATE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t GREATER_THAN_OR_APPROXIMATE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2a86 };
static wchar_t* GREATER_THAN_OR_APPROXIMATE_UNICODE_CHARACTER_CODE_MODEL = GREATER_THAN_OR_APPROXIMATE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* GREATER_THAN_OR_APPROXIMATE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LESS_THAN_AND_SINGLE_LINE_NOT_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2a87 };
static wchar_t* LESS_THAN_AND_SINGLE_LINE_NOT_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL = LESS_THAN_AND_SINGLE_LINE_NOT_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LESS_THAN_AND_SINGLE_LINE_NOT_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t GREATER_THAN_AND_SINGLE_LINE_NOT_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2a88 };
static wchar_t* GREATER_THAN_AND_SINGLE_LINE_NOT_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL = GREATER_THAN_AND_SINGLE_LINE_NOT_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* GREATER_THAN_AND_SINGLE_LINE_NOT_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LESS_THAN_AND_NOT_APPROXIMATE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2a89 };
static wchar_t* LESS_THAN_AND_NOT_APPROXIMATE_UNICODE_CHARACTER_CODE_MODEL = LESS_THAN_AND_NOT_APPROXIMATE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LESS_THAN_AND_NOT_APPROXIMATE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t GREATER_THAN_AND_NOT_APPROXIMATE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2a8a };
static wchar_t* GREATER_THAN_AND_NOT_APPROXIMATE_UNICODE_CHARACTER_CODE_MODEL = GREATER_THAN_AND_NOT_APPROXIMATE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* GREATER_THAN_AND_NOT_APPROXIMATE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LESS_THAN_ABOVE_DOUBLE_LINE_EQUAL_ABOVE_GREATER_THAN_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2a8b };
static wchar_t* LESS_THAN_ABOVE_DOUBLE_LINE_EQUAL_ABOVE_GREATER_THAN_UNICODE_CHARACTER_CODE_MODEL = LESS_THAN_ABOVE_DOUBLE_LINE_EQUAL_ABOVE_GREATER_THAN_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LESS_THAN_ABOVE_DOUBLE_LINE_EQUAL_ABOVE_GREATER_THAN_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t GREATER_THAN_ABOVE_DOUBLE_LINE_EQUAL_ABOVE_LESS_THAN_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2a8c };
static wchar_t* GREATER_THAN_ABOVE_DOUBLE_LINE_EQUAL_ABOVE_LESS_THAN_UNICODE_CHARACTER_CODE_MODEL = GREATER_THAN_ABOVE_DOUBLE_LINE_EQUAL_ABOVE_LESS_THAN_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* GREATER_THAN_ABOVE_DOUBLE_LINE_EQUAL_ABOVE_LESS_THAN_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LESS_THAN_ABOVE_SIMILAR_OR_EQUAL_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2a8d };
static wchar_t* LESS_THAN_ABOVE_SIMILAR_OR_EQUAL_UNICODE_CHARACTER_CODE_MODEL = LESS_THAN_ABOVE_SIMILAR_OR_EQUAL_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LESS_THAN_ABOVE_SIMILAR_OR_EQUAL_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t GREATER_THAN_ABOVE_SIMILAR_OR_EQUAL_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2a8e };
static wchar_t* GREATER_THAN_ABOVE_SIMILAR_OR_EQUAL_UNICODE_CHARACTER_CODE_MODEL = GREATER_THAN_ABOVE_SIMILAR_OR_EQUAL_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* GREATER_THAN_ABOVE_SIMILAR_OR_EQUAL_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LESS_THAN_ABOVE_SIMILAR_ABOVE_GREATER_THAN_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2a8f };
static wchar_t* LESS_THAN_ABOVE_SIMILAR_ABOVE_GREATER_THAN_UNICODE_CHARACTER_CODE_MODEL = LESS_THAN_ABOVE_SIMILAR_ABOVE_GREATER_THAN_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LESS_THAN_ABOVE_SIMILAR_ABOVE_GREATER_THAN_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t GREATER_THAN_ABOVE_SIMILAR_ABOVE_LESS_THAN_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2a90 };
static wchar_t* GREATER_THAN_ABOVE_SIMILAR_ABOVE_LESS_THAN_UNICODE_CHARACTER_CODE_MODEL = GREATER_THAN_ABOVE_SIMILAR_ABOVE_LESS_THAN_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* GREATER_THAN_ABOVE_SIMILAR_ABOVE_LESS_THAN_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LESS_THAN_ABOVE_GREATER_THAN_ABOVE_DOUBLE_LINE_EQUAL_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2a91 };
static wchar_t* LESS_THAN_ABOVE_GREATER_THAN_ABOVE_DOUBLE_LINE_EQUAL_UNICODE_CHARACTER_CODE_MODEL = LESS_THAN_ABOVE_GREATER_THAN_ABOVE_DOUBLE_LINE_EQUAL_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LESS_THAN_ABOVE_GREATER_THAN_ABOVE_DOUBLE_LINE_EQUAL_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t GREATER_THAN_ABOVE_LESS_THAN_ABOVE_DOUBLE_LINE_EQUAL_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2a92 };
static wchar_t* GREATER_THAN_ABOVE_LESS_THAN_ABOVE_DOUBLE_LINE_EQUAL_UNICODE_CHARACTER_CODE_MODEL = GREATER_THAN_ABOVE_LESS_THAN_ABOVE_DOUBLE_LINE_EQUAL_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* GREATER_THAN_ABOVE_LESS_THAN_ABOVE_DOUBLE_LINE_EQUAL_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LESS_THAN_ABOVE_SLANTED_EQUAL_ABOVE_GREATER_THAN_ABOVE_SLANTED_EQUAL_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2a93 };
static wchar_t* LESS_THAN_ABOVE_SLANTED_EQUAL_ABOVE_GREATER_THAN_ABOVE_SLANTED_EQUAL_UNICODE_CHARACTER_CODE_MODEL = LESS_THAN_ABOVE_SLANTED_EQUAL_ABOVE_GREATER_THAN_ABOVE_SLANTED_EQUAL_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LESS_THAN_ABOVE_SLANTED_EQUAL_ABOVE_GREATER_THAN_ABOVE_SLANTED_EQUAL_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t GREATER_THAN_ABOVE_SLANTED_EQUAL_ABOVE_LESS_THAN_ABOVE_SLANTED_EQUAL_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2a94 };
static wchar_t* GREATER_THAN_ABOVE_SLANTED_EQUAL_ABOVE_LESS_THAN_ABOVE_SLANTED_EQUAL_UNICODE_CHARACTER_CODE_MODEL = GREATER_THAN_ABOVE_SLANTED_EQUAL_ABOVE_LESS_THAN_ABOVE_SLANTED_EQUAL_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* GREATER_THAN_ABOVE_SLANTED_EQUAL_ABOVE_LESS_THAN_ABOVE_SLANTED_EQUAL_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t SLANTED_EQUAL_TO_OR_LESS_THAN_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2a95 };
static wchar_t* SLANTED_EQUAL_TO_OR_LESS_THAN_UNICODE_CHARACTER_CODE_MODEL = SLANTED_EQUAL_TO_OR_LESS_THAN_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* SLANTED_EQUAL_TO_OR_LESS_THAN_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t SLANTED_EQUAL_TO_OR_GREATER_THAN_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2a96 };
static wchar_t* SLANTED_EQUAL_TO_OR_GREATER_THAN_UNICODE_CHARACTER_CODE_MODEL = SLANTED_EQUAL_TO_OR_GREATER_THAN_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* SLANTED_EQUAL_TO_OR_GREATER_THAN_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t SLANTED_EQUAL_TO_OR_LESS_THAN_WITH_DOT_INSIDE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2a97 };
static wchar_t* SLANTED_EQUAL_TO_OR_LESS_THAN_WITH_DOT_INSIDE_UNICODE_CHARACTER_CODE_MODEL = SLANTED_EQUAL_TO_OR_LESS_THAN_WITH_DOT_INSIDE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* SLANTED_EQUAL_TO_OR_LESS_THAN_WITH_DOT_INSIDE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t SLANTED_EQUAL_TO_OR_GREATER_THAN_WITH_DOT_INSIDE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2a98 };
static wchar_t* SLANTED_EQUAL_TO_OR_GREATER_THAN_WITH_DOT_INSIDE_UNICODE_CHARACTER_CODE_MODEL = SLANTED_EQUAL_TO_OR_GREATER_THAN_WITH_DOT_INSIDE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* SLANTED_EQUAL_TO_OR_GREATER_THAN_WITH_DOT_INSIDE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t DOUBLE_LINE_EQUAL_TO_OR_LESS_THAN_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2a99 };
static wchar_t* DOUBLE_LINE_EQUAL_TO_OR_LESS_THAN_UNICODE_CHARACTER_CODE_MODEL = DOUBLE_LINE_EQUAL_TO_OR_LESS_THAN_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* DOUBLE_LINE_EQUAL_TO_OR_LESS_THAN_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t DOUBLE_LINE_EQUAL_TO_OR_GREATER_THAN_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2a9a };
static wchar_t* DOUBLE_LINE_EQUAL_TO_OR_GREATER_THAN_UNICODE_CHARACTER_CODE_MODEL = DOUBLE_LINE_EQUAL_TO_OR_GREATER_THAN_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* DOUBLE_LINE_EQUAL_TO_OR_GREATER_THAN_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t SIMILAR_OR_LESS_THAN_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2a9d };
static wchar_t* SIMILAR_OR_LESS_THAN_UNICODE_CHARACTER_CODE_MODEL = SIMILAR_OR_LESS_THAN_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* SIMILAR_OR_LESS_THAN_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t SIMILAR_OR_GREATER_THAN_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2a9e };
static wchar_t* SIMILAR_OR_GREATER_THAN_UNICODE_CHARACTER_CODE_MODEL = SIMILAR_OR_GREATER_THAN_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* SIMILAR_OR_GREATER_THAN_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t SIMILAR_ABOVE_LESS_THAN_ABOVE_EQUALS_SIGN_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2a9f };
static wchar_t* SIMILAR_ABOVE_LESS_THAN_ABOVE_EQUALS_SIGN_UNICODE_CHARACTER_CODE_MODEL = SIMILAR_ABOVE_LESS_THAN_ABOVE_EQUALS_SIGN_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* SIMILAR_ABOVE_LESS_THAN_ABOVE_EQUALS_SIGN_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t SIMILAR_ABOVE_GREATER_THAN_ABOVE_EQUALS_SIGN_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2aa0 };
static wchar_t* SIMILAR_ABOVE_GREATER_THAN_ABOVE_EQUALS_SIGN_UNICODE_CHARACTER_CODE_MODEL = SIMILAR_ABOVE_GREATER_THAN_ABOVE_EQUALS_SIGN_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* SIMILAR_ABOVE_GREATER_THAN_ABOVE_EQUALS_SIGN_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t DOUBLE_NESTED_LESS_THAN_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2aa1 };
static wchar_t* DOUBLE_NESTED_LESS_THAN_UNICODE_CHARACTER_CODE_MODEL = DOUBLE_NESTED_LESS_THAN_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* DOUBLE_NESTED_LESS_THAN_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t DOUBLE_NESTED_LESS_THAN_WITH_SLASH_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2aa1, 0x0338 };
static wchar_t* DOUBLE_NESTED_LESS_THAN_WITH_SLASH_UNICODE_CHARACTER_CODE_MODEL = DOUBLE_NESTED_LESS_THAN_WITH_SLASH_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* DOUBLE_NESTED_LESS_THAN_WITH_SLASH_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_2_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t DOUBLE_NESTED_GREATER_THAN_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2aa2 };
static wchar_t* DOUBLE_NESTED_GREATER_THAN_UNICODE_CHARACTER_CODE_MODEL = DOUBLE_NESTED_GREATER_THAN_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* DOUBLE_NESTED_GREATER_THAN_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t DOUBLE_NESTED_GREATER_THAN_WITH_SLASH_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2aa2, 0x0338 };
static wchar_t* DOUBLE_NESTED_GREATER_THAN_WITH_SLASH_UNICODE_CHARACTER_CODE_MODEL = DOUBLE_NESTED_GREATER_THAN_WITH_SLASH_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* DOUBLE_NESTED_GREATER_THAN_WITH_SLASH_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_2_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t GREATER_THAN_OVERLAPPING_LESS_THAN_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2aa4 };
static wchar_t* GREATER_THAN_OVERLAPPING_LESS_THAN_UNICODE_CHARACTER_CODE_MODEL = GREATER_THAN_OVERLAPPING_LESS_THAN_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* GREATER_THAN_OVERLAPPING_LESS_THAN_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t GREATER_THAN_BESIDE_LESS_THAN_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2aa5 };
static wchar_t* GREATER_THAN_BESIDE_LESS_THAN_UNICODE_CHARACTER_CODE_MODEL = GREATER_THAN_BESIDE_LESS_THAN_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* GREATER_THAN_BESIDE_LESS_THAN_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LESS_THAN_CLOSED_BY_CURVE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2aa6 };
static wchar_t* LESS_THAN_CLOSED_BY_CURVE_UNICODE_CHARACTER_CODE_MODEL = LESS_THAN_CLOSED_BY_CURVE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LESS_THAN_CLOSED_BY_CURVE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t GREATER_THAN_CLOSED_BY_CURVE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2aa7 };
static wchar_t* GREATER_THAN_CLOSED_BY_CURVE_UNICODE_CHARACTER_CODE_MODEL = GREATER_THAN_CLOSED_BY_CURVE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* GREATER_THAN_CLOSED_BY_CURVE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LESS_THAN_CLOSED_BY_CURVE_ABOVE_SLANTED_EQUAL_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2aa8 };
static wchar_t* LESS_THAN_CLOSED_BY_CURVE_ABOVE_SLANTED_EQUAL_UNICODE_CHARACTER_CODE_MODEL = LESS_THAN_CLOSED_BY_CURVE_ABOVE_SLANTED_EQUAL_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LESS_THAN_CLOSED_BY_CURVE_ABOVE_SLANTED_EQUAL_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t GREATER_THAN_CLOSED_BY_CURVE_ABOVE_SLANTED_EQUAL_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2aa9 };
static wchar_t* GREATER_THAN_CLOSED_BY_CURVE_ABOVE_SLANTED_EQUAL_UNICODE_CHARACTER_CODE_MODEL = GREATER_THAN_CLOSED_BY_CURVE_ABOVE_SLANTED_EQUAL_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* GREATER_THAN_CLOSED_BY_CURVE_ABOVE_SLANTED_EQUAL_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t SMALLER_THAN_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2aaa };
static wchar_t* SMALLER_THAN_UNICODE_CHARACTER_CODE_MODEL = SMALLER_THAN_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* SMALLER_THAN_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LARGER_THAN_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2aab };
static wchar_t* LARGER_THAN_UNICODE_CHARACTER_CODE_MODEL = LARGER_THAN_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LARGER_THAN_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t SMALLER_THAN_OR_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2aac };
static wchar_t* SMALLER_THAN_OR_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL = SMALLER_THAN_OR_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* SMALLER_THAN_OR_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t SMALLER_THAN_OR_SLANTED_EQUAL_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2aac, 0xfe00 };
static wchar_t* SMALLER_THAN_OR_SLANTED_EQUAL_UNICODE_CHARACTER_CODE_MODEL = SMALLER_THAN_OR_SLANTED_EQUAL_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* SMALLER_THAN_OR_SLANTED_EQUAL_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_2_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LARGER_THAN_OR_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2aad };
static wchar_t* LARGER_THAN_OR_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL = LARGER_THAN_OR_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LARGER_THAN_OR_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LARGER_THAN_OR_SLANTED_EQUAL_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2aad, 0xfe00 };
static wchar_t* LARGER_THAN_OR_SLANTED_EQUAL_UNICODE_CHARACTER_CODE_MODEL = LARGER_THAN_OR_SLANTED_EQUAL_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LARGER_THAN_OR_SLANTED_EQUAL_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_2_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t EQUALS_SIGN_WITH_BUMPY_ABOVE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2aae };
static wchar_t* EQUALS_SIGN_WITH_BUMPY_ABOVE_UNICODE_CHARACTER_CODE_MODEL = EQUALS_SIGN_WITH_BUMPY_ABOVE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* EQUALS_SIGN_WITH_BUMPY_ABOVE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t PRECEDES_ABOVE_SINGLE_LINE_EQUALS_SIGN_WITH_SLASH_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2aaf, 0x0338 };
static wchar_t* PRECEDES_ABOVE_SINGLE_LINE_EQUALS_SIGN_WITH_SLASH_UNICODE_CHARACTER_CODE_MODEL = PRECEDES_ABOVE_SINGLE_LINE_EQUALS_SIGN_WITH_SLASH_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* PRECEDES_ABOVE_SINGLE_LINE_EQUALS_SIGN_WITH_SLASH_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_2_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t PRECEDES_ABOVE_SINGLE_LINE_EQUALS_SIGN_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2aaf };
static wchar_t* PRECEDES_ABOVE_SINGLE_LINE_EQUALS_SIGN_UNICODE_CHARACTER_CODE_MODEL = PRECEDES_ABOVE_SINGLE_LINE_EQUALS_SIGN_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* PRECEDES_ABOVE_SINGLE_LINE_EQUALS_SIGN_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t SUCCEEDS_ABOVE_SINGLE_LINE_EQUALS_SIGN_WITH_SLASH_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2ab0, 0x0338 };
static wchar_t* SUCCEEDS_ABOVE_SINGLE_LINE_EQUALS_SIGN_WITH_SLASH_UNICODE_CHARACTER_CODE_MODEL = SUCCEEDS_ABOVE_SINGLE_LINE_EQUALS_SIGN_WITH_SLASH_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* SUCCEEDS_ABOVE_SINGLE_LINE_EQUALS_SIGN_WITH_SLASH_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_2_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t SUCCEEDS_ABOVE_SINGLE_LINE_EQUALS_SIGN_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2ab0 };
static wchar_t* SUCCEEDS_ABOVE_SINGLE_LINE_EQUALS_SIGN_UNICODE_CHARACTER_CODE_MODEL = SUCCEEDS_ABOVE_SINGLE_LINE_EQUALS_SIGN_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* SUCCEEDS_ABOVE_SINGLE_LINE_EQUALS_SIGN_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t PRECEDES_ABOVE_EQUALS_SIGN_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2ab3 };
static wchar_t* PRECEDES_ABOVE_EQUALS_SIGN_UNICODE_CHARACTER_CODE_MODEL = PRECEDES_ABOVE_EQUALS_SIGN_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* PRECEDES_ABOVE_EQUALS_SIGN_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t SUCCEEDS_ABOVE_EQUALS_SIGN_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2ab4 };
static wchar_t* SUCCEEDS_ABOVE_EQUALS_SIGN_UNICODE_CHARACTER_CODE_MODEL = SUCCEEDS_ABOVE_EQUALS_SIGN_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* SUCCEEDS_ABOVE_EQUALS_SIGN_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t PRECEDES_ABOVE_NOT_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2ab5 };
static wchar_t* PRECEDES_ABOVE_NOT_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL = PRECEDES_ABOVE_NOT_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* PRECEDES_ABOVE_NOT_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t SUCCEEDS_ABOVE_NOT_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2ab6 };
static wchar_t* SUCCEEDS_ABOVE_NOT_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL = SUCCEEDS_ABOVE_NOT_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* SUCCEEDS_ABOVE_NOT_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t PRECEDES_ABOVE_ALMOST_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2ab7 };
static wchar_t* PRECEDES_ABOVE_ALMOST_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL = PRECEDES_ABOVE_ALMOST_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* PRECEDES_ABOVE_ALMOST_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t SUCCEEDS_ABOVE_ALMOST_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2ab8 };
static wchar_t* SUCCEEDS_ABOVE_ALMOST_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL = SUCCEEDS_ABOVE_ALMOST_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* SUCCEEDS_ABOVE_ALMOST_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t PRECEDES_ABOVE_NOT_ALMOST_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2ab9 };
static wchar_t* PRECEDES_ABOVE_NOT_ALMOST_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL = PRECEDES_ABOVE_NOT_ALMOST_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* PRECEDES_ABOVE_NOT_ALMOST_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t SUCCEEDS_ABOVE_NOT_ALMOST_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2aba };
static wchar_t* SUCCEEDS_ABOVE_NOT_ALMOST_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL = SUCCEEDS_ABOVE_NOT_ALMOST_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* SUCCEEDS_ABOVE_NOT_ALMOST_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t DOUBLE_PRECEDES_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2abb };
static wchar_t* DOUBLE_PRECEDES_UNICODE_CHARACTER_CODE_MODEL = DOUBLE_PRECEDES_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* DOUBLE_PRECEDES_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t DOUBLE_SUCCEEDS_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2abc };
static wchar_t* DOUBLE_SUCCEEDS_UNICODE_CHARACTER_CODE_MODEL = DOUBLE_SUCCEEDS_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* DOUBLE_SUCCEEDS_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t SUBSET_WITH_DOT_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2abd };
static wchar_t* SUBSET_WITH_DOT_UNICODE_CHARACTER_CODE_MODEL = SUBSET_WITH_DOT_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* SUBSET_WITH_DOT_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t SUPERSET_WITH_DOT_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2abe };
static wchar_t* SUPERSET_WITH_DOT_UNICODE_CHARACTER_CODE_MODEL = SUPERSET_WITH_DOT_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* SUPERSET_WITH_DOT_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t SUBSET_WITH_PLUS_SIGN_BELOW_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2abf };
static wchar_t* SUBSET_WITH_PLUS_SIGN_BELOW_UNICODE_CHARACTER_CODE_MODEL = SUBSET_WITH_PLUS_SIGN_BELOW_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* SUBSET_WITH_PLUS_SIGN_BELOW_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t SUPERSET_WITH_PLUS_SIGN_BELOW_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2ac0 };
static wchar_t* SUPERSET_WITH_PLUS_SIGN_BELOW_UNICODE_CHARACTER_CODE_MODEL = SUPERSET_WITH_PLUS_SIGN_BELOW_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* SUPERSET_WITH_PLUS_SIGN_BELOW_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t SUBSET_WITH_MULTIPLICATION_SIGN_BELOW_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2ac1 };
static wchar_t* SUBSET_WITH_MULTIPLICATION_SIGN_BELOW_UNICODE_CHARACTER_CODE_MODEL = SUBSET_WITH_MULTIPLICATION_SIGN_BELOW_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* SUBSET_WITH_MULTIPLICATION_SIGN_BELOW_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t SUPERSET_WITH_MULTIPLICATION_SIGN_BELOW_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2ac2 };
static wchar_t* SUPERSET_WITH_MULTIPLICATION_SIGN_BELOW_UNICODE_CHARACTER_CODE_MODEL = SUPERSET_WITH_MULTIPLICATION_SIGN_BELOW_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* SUPERSET_WITH_MULTIPLICATION_SIGN_BELOW_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t SUBSET_OF_OR_EQUAL_TO_WITH_DOT_ABOVE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2ac3 };
static wchar_t* SUBSET_OF_OR_EQUAL_TO_WITH_DOT_ABOVE_UNICODE_CHARACTER_CODE_MODEL = SUBSET_OF_OR_EQUAL_TO_WITH_DOT_ABOVE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* SUBSET_OF_OR_EQUAL_TO_WITH_DOT_ABOVE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t SUPERSET_OF_OR_EQUAL_TO_WITH_DOT_ABOVE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2ac4 };
static wchar_t* SUPERSET_OF_OR_EQUAL_TO_WITH_DOT_ABOVE_UNICODE_CHARACTER_CODE_MODEL = SUPERSET_OF_OR_EQUAL_TO_WITH_DOT_ABOVE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* SUPERSET_OF_OR_EQUAL_TO_WITH_DOT_ABOVE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t SUBSET_OF_ABOVE_EQUALS_SIGN_WITH_SLASH_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2ac5, 0x0338 };
static wchar_t* SUBSET_OF_ABOVE_EQUALS_SIGN_WITH_SLASH_UNICODE_CHARACTER_CODE_MODEL = SUBSET_OF_ABOVE_EQUALS_SIGN_WITH_SLASH_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* SUBSET_OF_ABOVE_EQUALS_SIGN_WITH_SLASH_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_2_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t SUBSET_OF_ABOVE_EQUALS_SIGN_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2ac5 };
static wchar_t* SUBSET_OF_ABOVE_EQUALS_SIGN_UNICODE_CHARACTER_CODE_MODEL = SUBSET_OF_ABOVE_EQUALS_SIGN_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* SUBSET_OF_ABOVE_EQUALS_SIGN_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t SUPERSET_OF_ABOVE_EQUALS_SIGN_WITH_SLASH_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2ac6, 0x0338 };
static wchar_t* SUPERSET_OF_ABOVE_EQUALS_SIGN_WITH_SLASH_UNICODE_CHARACTER_CODE_MODEL = SUPERSET_OF_ABOVE_EQUALS_SIGN_WITH_SLASH_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* SUPERSET_OF_ABOVE_EQUALS_SIGN_WITH_SLASH_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_2_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t SUPERSET_OF_ABOVE_EQUALS_SIGN_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2ac6 };
static wchar_t* SUPERSET_OF_ABOVE_EQUALS_SIGN_UNICODE_CHARACTER_CODE_MODEL = SUPERSET_OF_ABOVE_EQUALS_SIGN_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* SUPERSET_OF_ABOVE_EQUALS_SIGN_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t SUBSET_OF_ABOVE_TILDE_OPERATOR_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2ac7 };
static wchar_t* SUBSET_OF_ABOVE_TILDE_OPERATOR_UNICODE_CHARACTER_CODE_MODEL = SUBSET_OF_ABOVE_TILDE_OPERATOR_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* SUBSET_OF_ABOVE_TILDE_OPERATOR_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t SUPERSET_OF_ABOVE_TILDE_OPERATOR_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2ac8 };
static wchar_t* SUPERSET_OF_ABOVE_TILDE_OPERATOR_UNICODE_CHARACTER_CODE_MODEL = SUPERSET_OF_ABOVE_TILDE_OPERATOR_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* SUPERSET_OF_ABOVE_TILDE_OPERATOR_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t SUBSET_OF_ABOVE_NOT_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2acb };
static wchar_t* SUBSET_OF_ABOVE_NOT_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL = SUBSET_OF_ABOVE_NOT_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* SUBSET_OF_ABOVE_NOT_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t SUBSET_OF_ABOVE_NOT_EQUAL_TO___VARIANT_WITH_STROKE_THROUGH_BOTTOM_MEMBERS_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2acb, 0xfe00 };
static wchar_t* SUBSET_OF_ABOVE_NOT_EQUAL_TO___VARIANT_WITH_STROKE_THROUGH_BOTTOM_MEMBERS_UNICODE_CHARACTER_CODE_MODEL = SUBSET_OF_ABOVE_NOT_EQUAL_TO___VARIANT_WITH_STROKE_THROUGH_BOTTOM_MEMBERS_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* SUBSET_OF_ABOVE_NOT_EQUAL_TO___VARIANT_WITH_STROKE_THROUGH_BOTTOM_MEMBERS_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_2_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t SUPERSET_OF_ABOVE_NOT_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2acc };
static wchar_t* SUPERSET_OF_ABOVE_NOT_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL = SUPERSET_OF_ABOVE_NOT_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* SUPERSET_OF_ABOVE_NOT_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t SUPERSET_OF_ABOVE_NOT_EQUAL_TO___VARIANT_WITH_STROKE_THROUGH_BOTTOM_MEMBERS_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2acc, 0xfe00 };
static wchar_t* SUPERSET_OF_ABOVE_NOT_EQUAL_TO___VARIANT_WITH_STROKE_THROUGH_BOTTOM_MEMBERS_UNICODE_CHARACTER_CODE_MODEL = SUPERSET_OF_ABOVE_NOT_EQUAL_TO___VARIANT_WITH_STROKE_THROUGH_BOTTOM_MEMBERS_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* SUPERSET_OF_ABOVE_NOT_EQUAL_TO___VARIANT_WITH_STROKE_THROUGH_BOTTOM_MEMBERS_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_2_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t CLOSED_SUBSET_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2acf };
static wchar_t* CLOSED_SUBSET_UNICODE_CHARACTER_CODE_MODEL = CLOSED_SUBSET_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CLOSED_SUBSET_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t CLOSED_SUPERSET_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2ad0 };
static wchar_t* CLOSED_SUPERSET_UNICODE_CHARACTER_CODE_MODEL = CLOSED_SUPERSET_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CLOSED_SUPERSET_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t CLOSED_SUBSET_OR_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2ad1 };
static wchar_t* CLOSED_SUBSET_OR_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL = CLOSED_SUBSET_OR_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CLOSED_SUBSET_OR_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t CLOSED_SUPERSET_OR_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2ad2 };
static wchar_t* CLOSED_SUPERSET_OR_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL = CLOSED_SUPERSET_OR_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* CLOSED_SUPERSET_OR_EQUAL_TO_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t SUBSET_ABOVE_SUPERSET_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2ad3 };
static wchar_t* SUBSET_ABOVE_SUPERSET_UNICODE_CHARACTER_CODE_MODEL = SUBSET_ABOVE_SUPERSET_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* SUBSET_ABOVE_SUPERSET_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t SUPERSET_ABOVE_SUBSET_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2ad4 };
static wchar_t* SUPERSET_ABOVE_SUBSET_UNICODE_CHARACTER_CODE_MODEL = SUPERSET_ABOVE_SUBSET_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* SUPERSET_ABOVE_SUBSET_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t SUBSET_ABOVE_SUBSET_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2ad5 };
static wchar_t* SUBSET_ABOVE_SUBSET_UNICODE_CHARACTER_CODE_MODEL = SUBSET_ABOVE_SUBSET_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* SUBSET_ABOVE_SUBSET_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t SUPERSET_ABOVE_SUPERSET_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2ad6 };
static wchar_t* SUPERSET_ABOVE_SUPERSET_UNICODE_CHARACTER_CODE_MODEL = SUPERSET_ABOVE_SUPERSET_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* SUPERSET_ABOVE_SUPERSET_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t SUPERSET_BESIDE_SUBSET_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2ad7 };
static wchar_t* SUPERSET_BESIDE_SUBSET_UNICODE_CHARACTER_CODE_MODEL = SUPERSET_BESIDE_SUBSET_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* SUPERSET_BESIDE_SUBSET_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t SUPERSET_BESIDE_AND_JOINED_BY_DASH_WITH_SUBSET_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2ad8 };
static wchar_t* SUPERSET_BESIDE_AND_JOINED_BY_DASH_WITH_SUBSET_UNICODE_CHARACTER_CODE_MODEL = SUPERSET_BESIDE_AND_JOINED_BY_DASH_WITH_SUBSET_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* SUPERSET_BESIDE_AND_JOINED_BY_DASH_WITH_SUBSET_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t ELEMENT_OF_OPENING_DOWNWARDS_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2ad9 };
static wchar_t* ELEMENT_OF_OPENING_DOWNWARDS_UNICODE_CHARACTER_CODE_MODEL = ELEMENT_OF_OPENING_DOWNWARDS_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* ELEMENT_OF_OPENING_DOWNWARDS_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t PITCHFORK_WITH_TEE_TOP_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2ada };
static wchar_t* PITCHFORK_WITH_TEE_TOP_UNICODE_CHARACTER_CODE_MODEL = PITCHFORK_WITH_TEE_TOP_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* PITCHFORK_WITH_TEE_TOP_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t TRANSVERSAL_INTERSECTION_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2adb };
static wchar_t* TRANSVERSAL_INTERSECTION_UNICODE_CHARACTER_CODE_MODEL = TRANSVERSAL_INTERSECTION_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* TRANSVERSAL_INTERSECTION_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t VERTICAL_BAR_DOUBLE_LEFT_TURNSTILE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2ae4 };
static wchar_t* VERTICAL_BAR_DOUBLE_LEFT_TURNSTILE_UNICODE_CHARACTER_CODE_MODEL = VERTICAL_BAR_DOUBLE_LEFT_TURNSTILE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* VERTICAL_BAR_DOUBLE_LEFT_TURNSTILE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LONG_DASH_FROM_LEFT_MEMBER_OF_DOUBLE_VERTICAL_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2ae6 };
static wchar_t* LONG_DASH_FROM_LEFT_MEMBER_OF_DOUBLE_VERTICAL_UNICODE_CHARACTER_CODE_MODEL = LONG_DASH_FROM_LEFT_MEMBER_OF_DOUBLE_VERTICAL_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LONG_DASH_FROM_LEFT_MEMBER_OF_DOUBLE_VERTICAL_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t SHORT_DOWN_TACK_WITH_OVERBAR_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2ae7 };
static wchar_t* SHORT_DOWN_TACK_WITH_OVERBAR_UNICODE_CHARACTER_CODE_MODEL = SHORT_DOWN_TACK_WITH_OVERBAR_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* SHORT_DOWN_TACK_WITH_OVERBAR_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t SHORT_UP_TACK_WITH_UNDERBAR_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2ae8 };
static wchar_t* SHORT_UP_TACK_WITH_UNDERBAR_UNICODE_CHARACTER_CODE_MODEL = SHORT_UP_TACK_WITH_UNDERBAR_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* SHORT_UP_TACK_WITH_UNDERBAR_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t SHORT_UP_TACK_ABOVE_SHORT_DOWN_TACK_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2ae9 };
static wchar_t* SHORT_UP_TACK_ABOVE_SHORT_DOWN_TACK_UNICODE_CHARACTER_CODE_MODEL = SHORT_UP_TACK_ABOVE_SHORT_DOWN_TACK_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* SHORT_UP_TACK_ABOVE_SHORT_DOWN_TACK_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t DOUBLE_UP_TACK_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2aeb };
static wchar_t* DOUBLE_UP_TACK_UNICODE_CHARACTER_CODE_MODEL = DOUBLE_UP_TACK_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* DOUBLE_UP_TACK_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t DOUBLE_STROKE_NOT_SIGN_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2aec };
static wchar_t* DOUBLE_STROKE_NOT_SIGN_UNICODE_CHARACTER_CODE_MODEL = DOUBLE_STROKE_NOT_SIGN_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* DOUBLE_STROKE_NOT_SIGN_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t REVERSED_DOUBLE_STROKE_NOT_SIGN_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2aed };
static wchar_t* REVERSED_DOUBLE_STROKE_NOT_SIGN_UNICODE_CHARACTER_CODE_MODEL = REVERSED_DOUBLE_STROKE_NOT_SIGN_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* REVERSED_DOUBLE_STROKE_NOT_SIGN_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t DOES_NOT_DIVIDE_WITH_REVERSED_NEGATION_SLASH_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2aee };
static wchar_t* DOES_NOT_DIVIDE_WITH_REVERSED_NEGATION_SLASH_UNICODE_CHARACTER_CODE_MODEL = DOES_NOT_DIVIDE_WITH_REVERSED_NEGATION_SLASH_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* DOES_NOT_DIVIDE_WITH_REVERSED_NEGATION_SLASH_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t VERTICAL_LINE_WITH_CIRCLE_ABOVE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2aef };
static wchar_t* VERTICAL_LINE_WITH_CIRCLE_ABOVE_UNICODE_CHARACTER_CODE_MODEL = VERTICAL_LINE_WITH_CIRCLE_ABOVE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* VERTICAL_LINE_WITH_CIRCLE_ABOVE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t VERTICAL_LINE_WITH_CIRCLE_BELOW_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2af0 };
static wchar_t* VERTICAL_LINE_WITH_CIRCLE_BELOW_UNICODE_CHARACTER_CODE_MODEL = VERTICAL_LINE_WITH_CIRCLE_BELOW_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* VERTICAL_LINE_WITH_CIRCLE_BELOW_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t DOWN_TACK_WITH_CIRCLE_BELOW_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2af1 };
static wchar_t* DOWN_TACK_WITH_CIRCLE_BELOW_UNICODE_CHARACTER_CODE_MODEL = DOWN_TACK_WITH_CIRCLE_BELOW_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* DOWN_TACK_WITH_CIRCLE_BELOW_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t PARALLEL_WITH_HORIZONTAL_STROKE_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2af2 };
static wchar_t* PARALLEL_WITH_HORIZONTAL_STROKE_UNICODE_CHARACTER_CODE_MODEL = PARALLEL_WITH_HORIZONTAL_STROKE_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* PARALLEL_WITH_HORIZONTAL_STROKE_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t PARALLEL_WITH_TILDE_OPERATOR_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2af3 };
static wchar_t* PARALLEL_WITH_TILDE_OPERATOR_UNICODE_CHARACTER_CODE_MODEL = PARALLEL_WITH_TILDE_OPERATOR_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* PARALLEL_WITH_TILDE_OPERATOR_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t DOUBLE_SOLIDUS_OPERATOR_WITH_REVERSE_SLASH_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2afd, 0x20e5 };
static wchar_t* DOUBLE_SOLIDUS_OPERATOR_WITH_REVERSE_SLASH_UNICODE_CHARACTER_CODE_MODEL = DOUBLE_SOLIDUS_OPERATOR_WITH_REVERSE_SLASH_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* DOUBLE_SOLIDUS_OPERATOR_WITH_REVERSE_SLASH_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_2_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t DOUBLE_SOLIDUS_OPERATOR_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2afd };
static wchar_t* DOUBLE_SOLIDUS_OPERATOR_UNICODE_CHARACTER_CODE_MODEL = DOUBLE_SOLIDUS_OPERATOR_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* DOUBLE_SOLIDUS_OPERATOR_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_SCRIPT_CAPITAL_A_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd49c };
static wchar_t* MATHEMATICAL_SCRIPT_CAPITAL_A_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_SCRIPT_CAPITAL_A_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_SCRIPT_CAPITAL_A_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_SCRIPT_CAPITAL_C_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd49e };
static wchar_t* MATHEMATICAL_SCRIPT_CAPITAL_C_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_SCRIPT_CAPITAL_C_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_SCRIPT_CAPITAL_C_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_SCRIPT_CAPITAL_D_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd49f };
static wchar_t* MATHEMATICAL_SCRIPT_CAPITAL_D_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_SCRIPT_CAPITAL_D_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_SCRIPT_CAPITAL_D_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_SCRIPT_CAPITAL_G_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd4a2 };
static wchar_t* MATHEMATICAL_SCRIPT_CAPITAL_G_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_SCRIPT_CAPITAL_G_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_SCRIPT_CAPITAL_G_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_SCRIPT_CAPITAL_J_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd4a5 };
static wchar_t* MATHEMATICAL_SCRIPT_CAPITAL_J_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_SCRIPT_CAPITAL_J_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_SCRIPT_CAPITAL_J_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_SCRIPT_CAPITAL_K_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd4a6 };
static wchar_t* MATHEMATICAL_SCRIPT_CAPITAL_K_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_SCRIPT_CAPITAL_K_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_SCRIPT_CAPITAL_K_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_SCRIPT_CAPITAL_N_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd4a9 };
static wchar_t* MATHEMATICAL_SCRIPT_CAPITAL_N_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_SCRIPT_CAPITAL_N_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_SCRIPT_CAPITAL_N_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_SCRIPT_CAPITAL_O_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd4aa };
static wchar_t* MATHEMATICAL_SCRIPT_CAPITAL_O_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_SCRIPT_CAPITAL_O_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_SCRIPT_CAPITAL_O_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_SCRIPT_CAPITAL_P_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd4ab };
static wchar_t* MATHEMATICAL_SCRIPT_CAPITAL_P_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_SCRIPT_CAPITAL_P_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_SCRIPT_CAPITAL_P_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_SCRIPT_CAPITAL_Q_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd4ac };
static wchar_t* MATHEMATICAL_SCRIPT_CAPITAL_Q_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_SCRIPT_CAPITAL_Q_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_SCRIPT_CAPITAL_Q_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_SCRIPT_CAPITAL_S_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd4ae };
static wchar_t* MATHEMATICAL_SCRIPT_CAPITAL_S_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_SCRIPT_CAPITAL_S_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_SCRIPT_CAPITAL_S_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_SCRIPT_CAPITAL_T_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd4af };
static wchar_t* MATHEMATICAL_SCRIPT_CAPITAL_T_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_SCRIPT_CAPITAL_T_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_SCRIPT_CAPITAL_T_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_SCRIPT_CAPITAL_U_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd4b0 };
static wchar_t* MATHEMATICAL_SCRIPT_CAPITAL_U_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_SCRIPT_CAPITAL_U_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_SCRIPT_CAPITAL_U_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_SCRIPT_CAPITAL_V_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd4b1 };
static wchar_t* MATHEMATICAL_SCRIPT_CAPITAL_V_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_SCRIPT_CAPITAL_V_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_SCRIPT_CAPITAL_V_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_SCRIPT_CAPITAL_W_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd4b2 };
static wchar_t* MATHEMATICAL_SCRIPT_CAPITAL_W_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_SCRIPT_CAPITAL_W_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_SCRIPT_CAPITAL_W_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_SCRIPT_CAPITAL_X_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd4b3 };
static wchar_t* MATHEMATICAL_SCRIPT_CAPITAL_X_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_SCRIPT_CAPITAL_X_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_SCRIPT_CAPITAL_X_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_SCRIPT_CAPITAL_Y_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd4b4 };
static wchar_t* MATHEMATICAL_SCRIPT_CAPITAL_Y_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_SCRIPT_CAPITAL_Y_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_SCRIPT_CAPITAL_Y_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_SCRIPT_CAPITAL_Z_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd4b5 };
static wchar_t* MATHEMATICAL_SCRIPT_CAPITAL_Z_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_SCRIPT_CAPITAL_Z_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_SCRIPT_CAPITAL_Z_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_SCRIPT_SMALL_A_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd4b6 };
static wchar_t* MATHEMATICAL_SCRIPT_SMALL_A_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_SCRIPT_SMALL_A_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_SCRIPT_SMALL_A_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_SCRIPT_SMALL_B_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd4b7 };
static wchar_t* MATHEMATICAL_SCRIPT_SMALL_B_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_SCRIPT_SMALL_B_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_SCRIPT_SMALL_B_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_SCRIPT_SMALL_C_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd4b8 };
static wchar_t* MATHEMATICAL_SCRIPT_SMALL_C_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_SCRIPT_SMALL_C_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_SCRIPT_SMALL_C_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_SCRIPT_SMALL_D_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd4b9 };
static wchar_t* MATHEMATICAL_SCRIPT_SMALL_D_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_SCRIPT_SMALL_D_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_SCRIPT_SMALL_D_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_SCRIPT_SMALL_F_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd4bb };
static wchar_t* MATHEMATICAL_SCRIPT_SMALL_F_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_SCRIPT_SMALL_F_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_SCRIPT_SMALL_F_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_SCRIPT_SMALL_H_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd4bd };
static wchar_t* MATHEMATICAL_SCRIPT_SMALL_H_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_SCRIPT_SMALL_H_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_SCRIPT_SMALL_H_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_SCRIPT_SMALL_I_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd4be };
static wchar_t* MATHEMATICAL_SCRIPT_SMALL_I_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_SCRIPT_SMALL_I_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_SCRIPT_SMALL_I_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_SCRIPT_SMALL_J_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd4bf };
static wchar_t* MATHEMATICAL_SCRIPT_SMALL_J_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_SCRIPT_SMALL_J_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_SCRIPT_SMALL_J_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_SCRIPT_SMALL_K_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd4c0 };
static wchar_t* MATHEMATICAL_SCRIPT_SMALL_K_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_SCRIPT_SMALL_K_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_SCRIPT_SMALL_K_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_SCRIPT_SMALL_L_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd4c1 };
static wchar_t* MATHEMATICAL_SCRIPT_SMALL_L_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_SCRIPT_SMALL_L_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_SCRIPT_SMALL_L_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_SCRIPT_SMALL_M_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd4c2 };
static wchar_t* MATHEMATICAL_SCRIPT_SMALL_M_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_SCRIPT_SMALL_M_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_SCRIPT_SMALL_M_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_SCRIPT_SMALL_N_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd4c3 };
static wchar_t* MATHEMATICAL_SCRIPT_SMALL_N_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_SCRIPT_SMALL_N_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_SCRIPT_SMALL_N_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_SCRIPT_SMALL_P_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd4c5 };
static wchar_t* MATHEMATICAL_SCRIPT_SMALL_P_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_SCRIPT_SMALL_P_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_SCRIPT_SMALL_P_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_SCRIPT_SMALL_Q_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd4c6 };
static wchar_t* MATHEMATICAL_SCRIPT_SMALL_Q_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_SCRIPT_SMALL_Q_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_SCRIPT_SMALL_Q_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_SCRIPT_SMALL_R_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd4c7 };
static wchar_t* MATHEMATICAL_SCRIPT_SMALL_R_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_SCRIPT_SMALL_R_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_SCRIPT_SMALL_R_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_SCRIPT_SMALL_S_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd4c8 };
static wchar_t* MATHEMATICAL_SCRIPT_SMALL_S_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_SCRIPT_SMALL_S_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_SCRIPT_SMALL_S_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_SCRIPT_SMALL_T_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd4c9 };
static wchar_t* MATHEMATICAL_SCRIPT_SMALL_T_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_SCRIPT_SMALL_T_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_SCRIPT_SMALL_T_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_SCRIPT_SMALL_U_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd4ca };
static wchar_t* MATHEMATICAL_SCRIPT_SMALL_U_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_SCRIPT_SMALL_U_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_SCRIPT_SMALL_U_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_SCRIPT_SMALL_V_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd4cb };
static wchar_t* MATHEMATICAL_SCRIPT_SMALL_V_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_SCRIPT_SMALL_V_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_SCRIPT_SMALL_V_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_SCRIPT_SMALL_W_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd4cc };
static wchar_t* MATHEMATICAL_SCRIPT_SMALL_W_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_SCRIPT_SMALL_W_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_SCRIPT_SMALL_W_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_SCRIPT_SMALL_X_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd4cd };
static wchar_t* MATHEMATICAL_SCRIPT_SMALL_X_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_SCRIPT_SMALL_X_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_SCRIPT_SMALL_X_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_SCRIPT_SMALL_Y_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd4ce };
static wchar_t* MATHEMATICAL_SCRIPT_SMALL_Y_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_SCRIPT_SMALL_Y_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_SCRIPT_SMALL_Y_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_SCRIPT_SMALL_Z_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd4cf };
static wchar_t* MATHEMATICAL_SCRIPT_SMALL_Z_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_SCRIPT_SMALL_Z_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_SCRIPT_SMALL_Z_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_FRAKTUR_CAPITAL_A_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd504 };
static wchar_t* MATHEMATICAL_FRAKTUR_CAPITAL_A_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_FRAKTUR_CAPITAL_A_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_FRAKTUR_CAPITAL_A_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_FRAKTUR_CAPITAL_B_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd505 };
static wchar_t* MATHEMATICAL_FRAKTUR_CAPITAL_B_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_FRAKTUR_CAPITAL_B_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_FRAKTUR_CAPITAL_B_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_FRAKTUR_CAPITAL_D_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd507 };
static wchar_t* MATHEMATICAL_FRAKTUR_CAPITAL_D_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_FRAKTUR_CAPITAL_D_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_FRAKTUR_CAPITAL_D_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_FRAKTUR_CAPITAL_E_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd508 };
static wchar_t* MATHEMATICAL_FRAKTUR_CAPITAL_E_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_FRAKTUR_CAPITAL_E_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_FRAKTUR_CAPITAL_E_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_FRAKTUR_CAPITAL_F_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd509 };
static wchar_t* MATHEMATICAL_FRAKTUR_CAPITAL_F_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_FRAKTUR_CAPITAL_F_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_FRAKTUR_CAPITAL_F_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_FRAKTUR_CAPITAL_G_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd50a };
static wchar_t* MATHEMATICAL_FRAKTUR_CAPITAL_G_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_FRAKTUR_CAPITAL_G_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_FRAKTUR_CAPITAL_G_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_FRAKTUR_CAPITAL_J_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd50d };
static wchar_t* MATHEMATICAL_FRAKTUR_CAPITAL_J_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_FRAKTUR_CAPITAL_J_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_FRAKTUR_CAPITAL_J_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_FRAKTUR_CAPITAL_K_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd50e };
static wchar_t* MATHEMATICAL_FRAKTUR_CAPITAL_K_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_FRAKTUR_CAPITAL_K_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_FRAKTUR_CAPITAL_K_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_FRAKTUR_CAPITAL_L_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd50f };
static wchar_t* MATHEMATICAL_FRAKTUR_CAPITAL_L_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_FRAKTUR_CAPITAL_L_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_FRAKTUR_CAPITAL_L_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_FRAKTUR_CAPITAL_M_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd510 };
static wchar_t* MATHEMATICAL_FRAKTUR_CAPITAL_M_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_FRAKTUR_CAPITAL_M_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_FRAKTUR_CAPITAL_M_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_FRAKTUR_CAPITAL_N_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd511 };
static wchar_t* MATHEMATICAL_FRAKTUR_CAPITAL_N_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_FRAKTUR_CAPITAL_N_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_FRAKTUR_CAPITAL_N_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_FRAKTUR_CAPITAL_O_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd512 };
static wchar_t* MATHEMATICAL_FRAKTUR_CAPITAL_O_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_FRAKTUR_CAPITAL_O_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_FRAKTUR_CAPITAL_O_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_FRAKTUR_CAPITAL_P_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd513 };
static wchar_t* MATHEMATICAL_FRAKTUR_CAPITAL_P_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_FRAKTUR_CAPITAL_P_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_FRAKTUR_CAPITAL_P_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_FRAKTUR_CAPITAL_Q_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd514 };
static wchar_t* MATHEMATICAL_FRAKTUR_CAPITAL_Q_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_FRAKTUR_CAPITAL_Q_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_FRAKTUR_CAPITAL_Q_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_FRAKTUR_CAPITAL_S_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd516 };
static wchar_t* MATHEMATICAL_FRAKTUR_CAPITAL_S_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_FRAKTUR_CAPITAL_S_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_FRAKTUR_CAPITAL_S_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_FRAKTUR_CAPITAL_T_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd517 };
static wchar_t* MATHEMATICAL_FRAKTUR_CAPITAL_T_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_FRAKTUR_CAPITAL_T_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_FRAKTUR_CAPITAL_T_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_FRAKTUR_CAPITAL_U_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd518 };
static wchar_t* MATHEMATICAL_FRAKTUR_CAPITAL_U_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_FRAKTUR_CAPITAL_U_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_FRAKTUR_CAPITAL_U_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_FRAKTUR_CAPITAL_V_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd519 };
static wchar_t* MATHEMATICAL_FRAKTUR_CAPITAL_V_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_FRAKTUR_CAPITAL_V_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_FRAKTUR_CAPITAL_V_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_FRAKTUR_CAPITAL_W_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd51a };
static wchar_t* MATHEMATICAL_FRAKTUR_CAPITAL_W_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_FRAKTUR_CAPITAL_W_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_FRAKTUR_CAPITAL_W_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_FRAKTUR_CAPITAL_X_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd51b };
static wchar_t* MATHEMATICAL_FRAKTUR_CAPITAL_X_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_FRAKTUR_CAPITAL_X_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_FRAKTUR_CAPITAL_X_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_FRAKTUR_CAPITAL_Y_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd51c };
static wchar_t* MATHEMATICAL_FRAKTUR_CAPITAL_Y_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_FRAKTUR_CAPITAL_Y_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_FRAKTUR_CAPITAL_Y_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_FRAKTUR_SMALL_A_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd51e };
static wchar_t* MATHEMATICAL_FRAKTUR_SMALL_A_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_FRAKTUR_SMALL_A_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_FRAKTUR_SMALL_A_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_FRAKTUR_SMALL_B_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd51f };
static wchar_t* MATHEMATICAL_FRAKTUR_SMALL_B_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_FRAKTUR_SMALL_B_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_FRAKTUR_SMALL_B_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_FRAKTUR_SMALL_C_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd520 };
static wchar_t* MATHEMATICAL_FRAKTUR_SMALL_C_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_FRAKTUR_SMALL_C_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_FRAKTUR_SMALL_C_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_FRAKTUR_SMALL_D_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd521 };
static wchar_t* MATHEMATICAL_FRAKTUR_SMALL_D_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_FRAKTUR_SMALL_D_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_FRAKTUR_SMALL_D_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_FRAKTUR_SMALL_E_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd522 };
static wchar_t* MATHEMATICAL_FRAKTUR_SMALL_E_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_FRAKTUR_SMALL_E_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_FRAKTUR_SMALL_E_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_FRAKTUR_SMALL_F_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd523 };
static wchar_t* MATHEMATICAL_FRAKTUR_SMALL_F_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_FRAKTUR_SMALL_F_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_FRAKTUR_SMALL_F_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_FRAKTUR_SMALL_G_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd524 };
static wchar_t* MATHEMATICAL_FRAKTUR_SMALL_G_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_FRAKTUR_SMALL_G_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_FRAKTUR_SMALL_G_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_FRAKTUR_SMALL_H_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd525 };
static wchar_t* MATHEMATICAL_FRAKTUR_SMALL_H_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_FRAKTUR_SMALL_H_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_FRAKTUR_SMALL_H_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_FRAKTUR_SMALL_I_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd526 };
static wchar_t* MATHEMATICAL_FRAKTUR_SMALL_I_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_FRAKTUR_SMALL_I_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_FRAKTUR_SMALL_I_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_FRAKTUR_SMALL_J_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd527 };
static wchar_t* MATHEMATICAL_FRAKTUR_SMALL_J_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_FRAKTUR_SMALL_J_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_FRAKTUR_SMALL_J_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_FRAKTUR_SMALL_K_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd528 };
static wchar_t* MATHEMATICAL_FRAKTUR_SMALL_K_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_FRAKTUR_SMALL_K_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_FRAKTUR_SMALL_K_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_FRAKTUR_SMALL_L_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd529 };
static wchar_t* MATHEMATICAL_FRAKTUR_SMALL_L_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_FRAKTUR_SMALL_L_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_FRAKTUR_SMALL_L_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_FRAKTUR_SMALL_M_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd52a };
static wchar_t* MATHEMATICAL_FRAKTUR_SMALL_M_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_FRAKTUR_SMALL_M_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_FRAKTUR_SMALL_M_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_FRAKTUR_SMALL_N_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd52b };
static wchar_t* MATHEMATICAL_FRAKTUR_SMALL_N_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_FRAKTUR_SMALL_N_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_FRAKTUR_SMALL_N_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_FRAKTUR_SMALL_O_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd52c };
static wchar_t* MATHEMATICAL_FRAKTUR_SMALL_O_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_FRAKTUR_SMALL_O_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_FRAKTUR_SMALL_O_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_FRAKTUR_SMALL_P_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd52d };
static wchar_t* MATHEMATICAL_FRAKTUR_SMALL_P_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_FRAKTUR_SMALL_P_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_FRAKTUR_SMALL_P_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_FRAKTUR_SMALL_Q_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd52e };
static wchar_t* MATHEMATICAL_FRAKTUR_SMALL_Q_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_FRAKTUR_SMALL_Q_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_FRAKTUR_SMALL_Q_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_FRAKTUR_SMALL_R_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd52f };
static wchar_t* MATHEMATICAL_FRAKTUR_SMALL_R_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_FRAKTUR_SMALL_R_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_FRAKTUR_SMALL_R_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_FRAKTUR_SMALL_S_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd530 };
static wchar_t* MATHEMATICAL_FRAKTUR_SMALL_S_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_FRAKTUR_SMALL_S_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_FRAKTUR_SMALL_S_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_FRAKTUR_SMALL_T_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd531 };
static wchar_t* MATHEMATICAL_FRAKTUR_SMALL_T_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_FRAKTUR_SMALL_T_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_FRAKTUR_SMALL_T_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_FRAKTUR_SMALL_U_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd532 };
static wchar_t* MATHEMATICAL_FRAKTUR_SMALL_U_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_FRAKTUR_SMALL_U_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_FRAKTUR_SMALL_U_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_FRAKTUR_SMALL_V_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd533 };
static wchar_t* MATHEMATICAL_FRAKTUR_SMALL_V_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_FRAKTUR_SMALL_V_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_FRAKTUR_SMALL_V_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_FRAKTUR_SMALL_W_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd534 };
static wchar_t* MATHEMATICAL_FRAKTUR_SMALL_W_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_FRAKTUR_SMALL_W_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_FRAKTUR_SMALL_W_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_FRAKTUR_SMALL_X_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd535 };
static wchar_t* MATHEMATICAL_FRAKTUR_SMALL_X_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_FRAKTUR_SMALL_X_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_FRAKTUR_SMALL_X_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_FRAKTUR_SMALL_Y_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd536 };
static wchar_t* MATHEMATICAL_FRAKTUR_SMALL_Y_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_FRAKTUR_SMALL_Y_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_FRAKTUR_SMALL_Y_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_FRAKTUR_SMALL_Z_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd537 };
static wchar_t* MATHEMATICAL_FRAKTUR_SMALL_Z_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_FRAKTUR_SMALL_Z_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_FRAKTUR_SMALL_Z_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_DOUBLE_STRUCK_CAPITAL_A_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd538 };
static wchar_t* MATHEMATICAL_DOUBLE_STRUCK_CAPITAL_A_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_DOUBLE_STRUCK_CAPITAL_A_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_DOUBLE_STRUCK_CAPITAL_A_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_DOUBLE_STRUCK_CAPITAL_B_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd539 };
static wchar_t* MATHEMATICAL_DOUBLE_STRUCK_CAPITAL_B_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_DOUBLE_STRUCK_CAPITAL_B_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_DOUBLE_STRUCK_CAPITAL_B_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_DOUBLE_STRUCK_CAPITAL_D_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd53b };
static wchar_t* MATHEMATICAL_DOUBLE_STRUCK_CAPITAL_D_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_DOUBLE_STRUCK_CAPITAL_D_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_DOUBLE_STRUCK_CAPITAL_D_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_DOUBLE_STRUCK_CAPITAL_E_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd53c };
static wchar_t* MATHEMATICAL_DOUBLE_STRUCK_CAPITAL_E_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_DOUBLE_STRUCK_CAPITAL_E_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_DOUBLE_STRUCK_CAPITAL_E_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_DOUBLE_STRUCK_CAPITAL_F_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd53d };
static wchar_t* MATHEMATICAL_DOUBLE_STRUCK_CAPITAL_F_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_DOUBLE_STRUCK_CAPITAL_F_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_DOUBLE_STRUCK_CAPITAL_F_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_DOUBLE_STRUCK_CAPITAL_G_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd53e };
static wchar_t* MATHEMATICAL_DOUBLE_STRUCK_CAPITAL_G_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_DOUBLE_STRUCK_CAPITAL_G_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_DOUBLE_STRUCK_CAPITAL_G_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_DOUBLE_STRUCK_CAPITAL_I_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd540 };
static wchar_t* MATHEMATICAL_DOUBLE_STRUCK_CAPITAL_I_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_DOUBLE_STRUCK_CAPITAL_I_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_DOUBLE_STRUCK_CAPITAL_I_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_DOUBLE_STRUCK_CAPITAL_J_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd541 };
static wchar_t* MATHEMATICAL_DOUBLE_STRUCK_CAPITAL_J_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_DOUBLE_STRUCK_CAPITAL_J_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_DOUBLE_STRUCK_CAPITAL_J_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_DOUBLE_STRUCK_CAPITAL_K_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd542 };
static wchar_t* MATHEMATICAL_DOUBLE_STRUCK_CAPITAL_K_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_DOUBLE_STRUCK_CAPITAL_K_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_DOUBLE_STRUCK_CAPITAL_K_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_DOUBLE_STRUCK_CAPITAL_L_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd543 };
static wchar_t* MATHEMATICAL_DOUBLE_STRUCK_CAPITAL_L_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_DOUBLE_STRUCK_CAPITAL_L_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_DOUBLE_STRUCK_CAPITAL_L_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_DOUBLE_STRUCK_CAPITAL_M_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd544 };
static wchar_t* MATHEMATICAL_DOUBLE_STRUCK_CAPITAL_M_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_DOUBLE_STRUCK_CAPITAL_M_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_DOUBLE_STRUCK_CAPITAL_M_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_DOUBLE_STRUCK_CAPITAL_O_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd546 };
static wchar_t* MATHEMATICAL_DOUBLE_STRUCK_CAPITAL_O_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_DOUBLE_STRUCK_CAPITAL_O_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_DOUBLE_STRUCK_CAPITAL_O_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_DOUBLE_STRUCK_CAPITAL_S_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd54a };
static wchar_t* MATHEMATICAL_DOUBLE_STRUCK_CAPITAL_S_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_DOUBLE_STRUCK_CAPITAL_S_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_DOUBLE_STRUCK_CAPITAL_S_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_DOUBLE_STRUCK_CAPITAL_T_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd54b };
static wchar_t* MATHEMATICAL_DOUBLE_STRUCK_CAPITAL_T_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_DOUBLE_STRUCK_CAPITAL_T_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_DOUBLE_STRUCK_CAPITAL_T_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_DOUBLE_STRUCK_CAPITAL_U_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd54c };
static wchar_t* MATHEMATICAL_DOUBLE_STRUCK_CAPITAL_U_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_DOUBLE_STRUCK_CAPITAL_U_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_DOUBLE_STRUCK_CAPITAL_U_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_DOUBLE_STRUCK_CAPITAL_V_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd54d };
static wchar_t* MATHEMATICAL_DOUBLE_STRUCK_CAPITAL_V_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_DOUBLE_STRUCK_CAPITAL_V_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_DOUBLE_STRUCK_CAPITAL_V_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_DOUBLE_STRUCK_CAPITAL_W_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd54e };
static wchar_t* MATHEMATICAL_DOUBLE_STRUCK_CAPITAL_W_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_DOUBLE_STRUCK_CAPITAL_W_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_DOUBLE_STRUCK_CAPITAL_W_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_DOUBLE_STRUCK_CAPITAL_X_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd54f };
static wchar_t* MATHEMATICAL_DOUBLE_STRUCK_CAPITAL_X_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_DOUBLE_STRUCK_CAPITAL_X_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_DOUBLE_STRUCK_CAPITAL_X_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_DOUBLE_STRUCK_CAPITAL_Y_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd550 };
static wchar_t* MATHEMATICAL_DOUBLE_STRUCK_CAPITAL_Y_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_DOUBLE_STRUCK_CAPITAL_Y_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_DOUBLE_STRUCK_CAPITAL_Y_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_DOUBLE_STRUCK_SMALL_A_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd552 };
static wchar_t* MATHEMATICAL_DOUBLE_STRUCK_SMALL_A_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_DOUBLE_STRUCK_SMALL_A_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_DOUBLE_STRUCK_SMALL_A_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_DOUBLE_STRUCK_SMALL_B_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd553 };
static wchar_t* MATHEMATICAL_DOUBLE_STRUCK_SMALL_B_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_DOUBLE_STRUCK_SMALL_B_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_DOUBLE_STRUCK_SMALL_B_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_DOUBLE_STRUCK_SMALL_C_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd554 };
static wchar_t* MATHEMATICAL_DOUBLE_STRUCK_SMALL_C_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_DOUBLE_STRUCK_SMALL_C_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_DOUBLE_STRUCK_SMALL_C_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_DOUBLE_STRUCK_SMALL_D_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd555 };
static wchar_t* MATHEMATICAL_DOUBLE_STRUCK_SMALL_D_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_DOUBLE_STRUCK_SMALL_D_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_DOUBLE_STRUCK_SMALL_D_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_DOUBLE_STRUCK_SMALL_E_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd556 };
static wchar_t* MATHEMATICAL_DOUBLE_STRUCK_SMALL_E_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_DOUBLE_STRUCK_SMALL_E_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_DOUBLE_STRUCK_SMALL_E_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_DOUBLE_STRUCK_SMALL_F_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd557 };
static wchar_t* MATHEMATICAL_DOUBLE_STRUCK_SMALL_F_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_DOUBLE_STRUCK_SMALL_F_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_DOUBLE_STRUCK_SMALL_F_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_DOUBLE_STRUCK_SMALL_G_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd558 };
static wchar_t* MATHEMATICAL_DOUBLE_STRUCK_SMALL_G_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_DOUBLE_STRUCK_SMALL_G_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_DOUBLE_STRUCK_SMALL_G_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_DOUBLE_STRUCK_SMALL_H_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd559 };
static wchar_t* MATHEMATICAL_DOUBLE_STRUCK_SMALL_H_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_DOUBLE_STRUCK_SMALL_H_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_DOUBLE_STRUCK_SMALL_H_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_DOUBLE_STRUCK_SMALL_I_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd55a };
static wchar_t* MATHEMATICAL_DOUBLE_STRUCK_SMALL_I_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_DOUBLE_STRUCK_SMALL_I_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_DOUBLE_STRUCK_SMALL_I_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_DOUBLE_STRUCK_SMALL_J_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd55b };
static wchar_t* MATHEMATICAL_DOUBLE_STRUCK_SMALL_J_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_DOUBLE_STRUCK_SMALL_J_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_DOUBLE_STRUCK_SMALL_J_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_DOUBLE_STRUCK_SMALL_K_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd55c };
static wchar_t* MATHEMATICAL_DOUBLE_STRUCK_SMALL_K_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_DOUBLE_STRUCK_SMALL_K_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_DOUBLE_STRUCK_SMALL_K_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_DOUBLE_STRUCK_SMALL_L_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd55d };
static wchar_t* MATHEMATICAL_DOUBLE_STRUCK_SMALL_L_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_DOUBLE_STRUCK_SMALL_L_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_DOUBLE_STRUCK_SMALL_L_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_DOUBLE_STRUCK_SMALL_M_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd55e };
static wchar_t* MATHEMATICAL_DOUBLE_STRUCK_SMALL_M_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_DOUBLE_STRUCK_SMALL_M_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_DOUBLE_STRUCK_SMALL_M_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_DOUBLE_STRUCK_SMALL_N_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd55f };
static wchar_t* MATHEMATICAL_DOUBLE_STRUCK_SMALL_N_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_DOUBLE_STRUCK_SMALL_N_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_DOUBLE_STRUCK_SMALL_N_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_DOUBLE_STRUCK_SMALL_O_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd560 };
static wchar_t* MATHEMATICAL_DOUBLE_STRUCK_SMALL_O_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_DOUBLE_STRUCK_SMALL_O_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_DOUBLE_STRUCK_SMALL_O_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_DOUBLE_STRUCK_SMALL_P_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd561 };
static wchar_t* MATHEMATICAL_DOUBLE_STRUCK_SMALL_P_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_DOUBLE_STRUCK_SMALL_P_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_DOUBLE_STRUCK_SMALL_P_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_DOUBLE_STRUCK_SMALL_Q_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd562 };
static wchar_t* MATHEMATICAL_DOUBLE_STRUCK_SMALL_Q_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_DOUBLE_STRUCK_SMALL_Q_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_DOUBLE_STRUCK_SMALL_Q_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_DOUBLE_STRUCK_SMALL_R_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd563 };
static wchar_t* MATHEMATICAL_DOUBLE_STRUCK_SMALL_R_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_DOUBLE_STRUCK_SMALL_R_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_DOUBLE_STRUCK_SMALL_R_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_DOUBLE_STRUCK_SMALL_S_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd564 };
static wchar_t* MATHEMATICAL_DOUBLE_STRUCK_SMALL_S_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_DOUBLE_STRUCK_SMALL_S_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_DOUBLE_STRUCK_SMALL_S_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_DOUBLE_STRUCK_SMALL_T_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd565 };
static wchar_t* MATHEMATICAL_DOUBLE_STRUCK_SMALL_T_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_DOUBLE_STRUCK_SMALL_T_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_DOUBLE_STRUCK_SMALL_T_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_DOUBLE_STRUCK_SMALL_U_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd566 };
static wchar_t* MATHEMATICAL_DOUBLE_STRUCK_SMALL_U_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_DOUBLE_STRUCK_SMALL_U_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_DOUBLE_STRUCK_SMALL_U_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_DOUBLE_STRUCK_SMALL_V_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd567 };
static wchar_t* MATHEMATICAL_DOUBLE_STRUCK_SMALL_V_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_DOUBLE_STRUCK_SMALL_V_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_DOUBLE_STRUCK_SMALL_V_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_DOUBLE_STRUCK_SMALL_W_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd568 };
static wchar_t* MATHEMATICAL_DOUBLE_STRUCK_SMALL_W_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_DOUBLE_STRUCK_SMALL_W_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_DOUBLE_STRUCK_SMALL_W_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_DOUBLE_STRUCK_SMALL_X_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd569 };
static wchar_t* MATHEMATICAL_DOUBLE_STRUCK_SMALL_X_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_DOUBLE_STRUCK_SMALL_X_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_DOUBLE_STRUCK_SMALL_X_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_DOUBLE_STRUCK_SMALL_Y_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd56a };
static wchar_t* MATHEMATICAL_DOUBLE_STRUCK_SMALL_Y_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_DOUBLE_STRUCK_SMALL_Y_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_DOUBLE_STRUCK_SMALL_Y_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_DOUBLE_STRUCK_SMALL_Z_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd56b };
static wchar_t* MATHEMATICAL_DOUBLE_STRUCK_SMALL_Z_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_DOUBLE_STRUCK_SMALL_Z_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_DOUBLE_STRUCK_SMALL_Z_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_BOLD_CAPITAL_GAMMA_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd6aa };
static wchar_t* MATHEMATICAL_BOLD_CAPITAL_GAMMA_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_BOLD_CAPITAL_GAMMA_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_BOLD_CAPITAL_GAMMA_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_BOLD_CAPITAL_DELTA_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd6ab };
static wchar_t* MATHEMATICAL_BOLD_CAPITAL_DELTA_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_BOLD_CAPITAL_DELTA_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_BOLD_CAPITAL_DELTA_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_BOLD_CAPITAL_THETA_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd6af };
static wchar_t* MATHEMATICAL_BOLD_CAPITAL_THETA_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_BOLD_CAPITAL_THETA_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_BOLD_CAPITAL_THETA_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_BOLD_CAPITAL_LAMDA_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd6b2 };
static wchar_t* MATHEMATICAL_BOLD_CAPITAL_LAMDA_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_BOLD_CAPITAL_LAMDA_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_BOLD_CAPITAL_LAMDA_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_BOLD_CAPITAL_XI_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd6b5 };
static wchar_t* MATHEMATICAL_BOLD_CAPITAL_XI_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_BOLD_CAPITAL_XI_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_BOLD_CAPITAL_XI_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_BOLD_CAPITAL_PI_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd6b7 };
static wchar_t* MATHEMATICAL_BOLD_CAPITAL_PI_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_BOLD_CAPITAL_PI_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_BOLD_CAPITAL_PI_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_BOLD_CAPITAL_SIGMA_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd6ba };
static wchar_t* MATHEMATICAL_BOLD_CAPITAL_SIGMA_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_BOLD_CAPITAL_SIGMA_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_BOLD_CAPITAL_SIGMA_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_BOLD_CAPITAL_UPSILON_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd6bc };
static wchar_t* MATHEMATICAL_BOLD_CAPITAL_UPSILON_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_BOLD_CAPITAL_UPSILON_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_BOLD_CAPITAL_UPSILON_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_BOLD_CAPITAL_PHI_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd6bd };
static wchar_t* MATHEMATICAL_BOLD_CAPITAL_PHI_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_BOLD_CAPITAL_PHI_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_BOLD_CAPITAL_PHI_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_BOLD_CAPITAL_PSI_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd6bf };
static wchar_t* MATHEMATICAL_BOLD_CAPITAL_PSI_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_BOLD_CAPITAL_PSI_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_BOLD_CAPITAL_PSI_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_BOLD_CAPITAL_OMEGA_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd6c0 };
static wchar_t* MATHEMATICAL_BOLD_CAPITAL_OMEGA_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_BOLD_CAPITAL_OMEGA_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_BOLD_CAPITAL_OMEGA_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_BOLD_SMALL_ALPHA_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd6c2 };
static wchar_t* MATHEMATICAL_BOLD_SMALL_ALPHA_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_BOLD_SMALL_ALPHA_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_BOLD_SMALL_ALPHA_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_BOLD_SMALL_BETA_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd6c3 };
static wchar_t* MATHEMATICAL_BOLD_SMALL_BETA_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_BOLD_SMALL_BETA_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_BOLD_SMALL_BETA_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_BOLD_SMALL_GAMMA_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd6c4 };
static wchar_t* MATHEMATICAL_BOLD_SMALL_GAMMA_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_BOLD_SMALL_GAMMA_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_BOLD_SMALL_GAMMA_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_BOLD_SMALL_DELTA_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd6c5 };
static wchar_t* MATHEMATICAL_BOLD_SMALL_DELTA_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_BOLD_SMALL_DELTA_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_BOLD_SMALL_DELTA_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_BOLD_SMALL_EPSILON_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd6c6 };
static wchar_t* MATHEMATICAL_BOLD_SMALL_EPSILON_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_BOLD_SMALL_EPSILON_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_BOLD_SMALL_EPSILON_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_BOLD_SMALL_ZETA_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd6c7 };
static wchar_t* MATHEMATICAL_BOLD_SMALL_ZETA_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_BOLD_SMALL_ZETA_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_BOLD_SMALL_ZETA_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_BOLD_SMALL_ETA_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd6c8 };
static wchar_t* MATHEMATICAL_BOLD_SMALL_ETA_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_BOLD_SMALL_ETA_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_BOLD_SMALL_ETA_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_BOLD_SMALL_THETA_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd6c9 };
static wchar_t* MATHEMATICAL_BOLD_SMALL_THETA_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_BOLD_SMALL_THETA_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_BOLD_SMALL_THETA_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_BOLD_SMALL_IOTA_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd6ca };
static wchar_t* MATHEMATICAL_BOLD_SMALL_IOTA_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_BOLD_SMALL_IOTA_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_BOLD_SMALL_IOTA_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_BOLD_SMALL_KAPPA_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd6cb };
static wchar_t* MATHEMATICAL_BOLD_SMALL_KAPPA_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_BOLD_SMALL_KAPPA_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_BOLD_SMALL_KAPPA_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_BOLD_SMALL_LAMDA_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd6cc };
static wchar_t* MATHEMATICAL_BOLD_SMALL_LAMDA_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_BOLD_SMALL_LAMDA_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_BOLD_SMALL_LAMDA_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_BOLD_SMALL_MU_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd6cd };
static wchar_t* MATHEMATICAL_BOLD_SMALL_MU_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_BOLD_SMALL_MU_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_BOLD_SMALL_MU_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_BOLD_SMALL_NU_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd6ce };
static wchar_t* MATHEMATICAL_BOLD_SMALL_NU_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_BOLD_SMALL_NU_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_BOLD_SMALL_NU_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_BOLD_SMALL_XI_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd6cf };
static wchar_t* MATHEMATICAL_BOLD_SMALL_XI_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_BOLD_SMALL_XI_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_BOLD_SMALL_XI_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_BOLD_SMALL_PI_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd6d1 };
static wchar_t* MATHEMATICAL_BOLD_SMALL_PI_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_BOLD_SMALL_PI_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_BOLD_SMALL_PI_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_BOLD_SMALL_RHO_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd6d2 };
static wchar_t* MATHEMATICAL_BOLD_SMALL_RHO_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_BOLD_SMALL_RHO_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_BOLD_SMALL_RHO_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_BOLD_SMALL_FINAL_SIGMA_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd6d3 };
static wchar_t* MATHEMATICAL_BOLD_SMALL_FINAL_SIGMA_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_BOLD_SMALL_FINAL_SIGMA_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_BOLD_SMALL_FINAL_SIGMA_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_BOLD_SMALL_SIGMA_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd6d4 };
static wchar_t* MATHEMATICAL_BOLD_SMALL_SIGMA_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_BOLD_SMALL_SIGMA_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_BOLD_SMALL_SIGMA_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_BOLD_SMALL_TAU_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd6d5 };
static wchar_t* MATHEMATICAL_BOLD_SMALL_TAU_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_BOLD_SMALL_TAU_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_BOLD_SMALL_TAU_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_BOLD_SMALL_UPSILON_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd6d6 };
static wchar_t* MATHEMATICAL_BOLD_SMALL_UPSILON_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_BOLD_SMALL_UPSILON_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_BOLD_SMALL_UPSILON_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_BOLD_SMALL_PHI_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd6d7 };
static wchar_t* MATHEMATICAL_BOLD_SMALL_PHI_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_BOLD_SMALL_PHI_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_BOLD_SMALL_PHI_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_BOLD_SMALL_CHI_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd6d8 };
static wchar_t* MATHEMATICAL_BOLD_SMALL_CHI_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_BOLD_SMALL_CHI_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_BOLD_SMALL_CHI_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_BOLD_SMALL_PSI_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd6d9 };
static wchar_t* MATHEMATICAL_BOLD_SMALL_PSI_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_BOLD_SMALL_PSI_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_BOLD_SMALL_PSI_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_BOLD_SMALL_OMEGA_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd6da };
static wchar_t* MATHEMATICAL_BOLD_SMALL_OMEGA_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_BOLD_SMALL_OMEGA_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_BOLD_SMALL_OMEGA_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_BOLD_EPSILON_SYMBOL_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd6dc };
static wchar_t* MATHEMATICAL_BOLD_EPSILON_SYMBOL_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_BOLD_EPSILON_SYMBOL_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_BOLD_EPSILON_SYMBOL_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_BOLD_THETA_SYMBOL_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd6dd };
static wchar_t* MATHEMATICAL_BOLD_THETA_SYMBOL_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_BOLD_THETA_SYMBOL_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_BOLD_THETA_SYMBOL_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_BOLD_KAPPA_SYMBOL_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd6de };
static wchar_t* MATHEMATICAL_BOLD_KAPPA_SYMBOL_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_BOLD_KAPPA_SYMBOL_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_BOLD_KAPPA_SYMBOL_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_BOLD_PHI_SYMBOL_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd6df };
static wchar_t* MATHEMATICAL_BOLD_PHI_SYMBOL_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_BOLD_PHI_SYMBOL_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_BOLD_PHI_SYMBOL_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_BOLD_RHO_SYMBOL_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd6e0 };
static wchar_t* MATHEMATICAL_BOLD_RHO_SYMBOL_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_BOLD_RHO_SYMBOL_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_BOLD_RHO_SYMBOL_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_BOLD_PI_SYMBOL_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd6e1 };
static wchar_t* MATHEMATICAL_BOLD_PI_SYMBOL_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_BOLD_PI_SYMBOL_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_BOLD_PI_SYMBOL_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_BOLD_CAPITAL_DIGAMMA_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd7ca };
static wchar_t* MATHEMATICAL_BOLD_CAPITAL_DIGAMMA_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_BOLD_CAPITAL_DIGAMMA_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_BOLD_CAPITAL_DIGAMMA_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t MATHEMATICAL_BOLD_SMALL_DIGAMMA_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xd7cb };
static wchar_t* MATHEMATICAL_BOLD_SMALL_DIGAMMA_UNICODE_CHARACTER_CODE_MODEL = MATHEMATICAL_BOLD_SMALL_DIGAMMA_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* MATHEMATICAL_BOLD_SMALL_DIGAMMA_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_SMALL_LIGATURE_FF_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xfb00 };
static wchar_t* LATIN_SMALL_LIGATURE_FF_UNICODE_CHARACTER_CODE_MODEL = LATIN_SMALL_LIGATURE_FF_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_SMALL_LIGATURE_FF_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_SMALL_LIGATURE_FI_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xfb01 };
static wchar_t* LATIN_SMALL_LIGATURE_FI_UNICODE_CHARACTER_CODE_MODEL = LATIN_SMALL_LIGATURE_FI_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_SMALL_LIGATURE_FI_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_SMALL_LIGATURE_FL_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xfb02 };
static wchar_t* LATIN_SMALL_LIGATURE_FL_UNICODE_CHARACTER_CODE_MODEL = LATIN_SMALL_LIGATURE_FL_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_SMALL_LIGATURE_FL_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_SMALL_LIGATURE_FFI_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xfb03 };
static wchar_t* LATIN_SMALL_LIGATURE_FFI_UNICODE_CHARACTER_CODE_MODEL = LATIN_SMALL_LIGATURE_FFI_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_SMALL_LIGATURE_FFI_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t LATIN_SMALL_LIGATURE_FFL_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0xfb04 };
static wchar_t* LATIN_SMALL_LIGATURE_FFL_UNICODE_CHARACTER_CODE_MODEL = LATIN_SMALL_LIGATURE_FFL_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* LATIN_SMALL_LIGATURE_FFL_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t BOX_DRAWINGS_LIGHT_ARC_DOWN_AND_RIGHT_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x256D };
static wchar_t* BOX_DRAWINGS_LIGHT_ARC_DOWN_AND_RIGHT_UNICODE_CHARACTER_CODE_MODEL = BOX_DRAWINGS_LIGHT_ARC_DOWN_AND_RIGHT_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* BOX_DRAWINGS_LIGHT_ARC_DOWN_AND_RIGHT_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t BOX_DRAWINGS_LIGHT_ARC_DOWN_AND_LEFT_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x256E };
static wchar_t* BOX_DRAWINGS_LIGHT_ARC_DOWN_AND_LEFT_UNICODE_CHARACTER_CODE_MODEL = BOX_DRAWINGS_LIGHT_ARC_DOWN_AND_LEFT_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* BOX_DRAWINGS_LIGHT_ARC_DOWN_AND_LEFT_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t BOX_DRAWINGS_LIGHT_ARC_UP_AND_LEFT_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x256F };
static wchar_t* BOX_DRAWINGS_LIGHT_ARC_UP_AND_LEFT_UNICODE_CHARACTER_CODE_MODEL = BOX_DRAWINGS_LIGHT_ARC_UP_AND_LEFT_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* BOX_DRAWINGS_LIGHT_ARC_UP_AND_LEFT_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

static wchar_t BOX_DRAWINGS_LIGHT_ARC_UP_AND_RIGHT_UNICODE_CHARACTER_CODE_MODEL_ARRAY[] = { 0x2570 };
static wchar_t* BOX_DRAWINGS_LIGHT_ARC_UP_AND_RIGHT_UNICODE_CHARACTER_CODE_MODEL = BOX_DRAWINGS_LIGHT_ARC_UP_AND_RIGHT_UNICODE_CHARACTER_CODE_MODEL_ARRAY;
static int* BOX_DRAWINGS_LIGHT_ARC_UP_AND_RIGHT_UNICODE_CHARACTER_CODE_MODEL_COUNT = NUMBER_1_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/* UNICODE_CHARACTER_CODE_MODEL_CONSTANT_HEADER */
#endif
