/*
 * Copyright (C) 1999-2022. Christian Heller.
 *
 * This file is part of the Cybernetics Oriented Interpreter (CYBOI).
 *
 * CYBOI is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published
 * by the Free Software Foundation, either version 3 of the License,
 * or (at your option) any later version.
 *
 * CYBOI is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty
 * of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with CYBOI. If not, see <http://www.gnu.org/licenses/>.
 *
 * Cybernetics Oriented Programming (CYBOP) <http://www.cybop.org/>
 * CYBOP Developers <cybop-developers@nongnu.org>
 *
 * @version CYBOP 0.23.0 2022-09-04
 * @author Christian Heller <christian.heller@cybop.org>
 */

#ifndef LOGIFY_LOGIC_CYBOL_FORMAT_CONSTANT_SOURCE
#define LOGIFY_LOGIC_CYBOL_FORMAT_CONSTANT_SOURCE

#include <stddef.h> // wchar_t

#include "../../../../constant/model/cyboi/state/integer_state_cyboi_model.c"

//
// Logify
//
// IANA media type: not defined
// Self-defined media type: logify
// This media type is a CYBOL extension.
//

/**
 * The logify/and logic cybol format.
 *
 * Apply a boolean AND operation.
 *
 * result = x AND y
 *
 * CAUTION! There are several ways to use this operation,
 * with unary or binary operators.
 * This function works like a UNARY operator.
 * The "output" parametre represents the FIRST operand;
 * the "input" parametre the SECOND.
 *
 * Properties:
 * - output (required): the knowledge model in which the output is stored; used as first input operand
 * - input (required): the second input operand
 */
static wchar_t* AND_LOGIFY_LOGIC_CYBOL_FORMAT = L"logify/and";
static int* AND_LOGIFY_LOGIC_CYBOL_FORMAT_COUNT = NUMBER_10_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/**
 * The logify/nand logic cybol format.
 *
 * Apply a boolean NAND operation.
 *
 * result = x NAND y
 *
 * This is a CYBOL extension.
 *
 * Properties:
 * - output (required): the knowledge model in which the output is stored; used as first input operand
 * - input (required): the second input operand
 */
static wchar_t* NAND_LOGIFY_LOGIC_CYBOL_FORMAT = L"logify/nand";
static int* NAND_LOGIFY_LOGIC_CYBOL_FORMAT_COUNT = NUMBER_11_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/**
 * The logify/neg logic cybol format.
 *
 * Apply a boolean NEG operation.
 * When used with Bit operands, then this is the TWO'S COMPLEMENT (all bits negated and added one).
 *
 * result = x NEG y
 *
 * This is a CYBOL extension.
 *
 * Properties:
 * - output (required): the knowledge model in which the output is stored; used as first input operand
 * - input (required): the second input operand
 */
static wchar_t* NEG_LOGIFY_LOGIC_CYBOL_FORMAT = L"logify/neg";
static int* NEG_LOGIFY_LOGIC_CYBOL_FORMAT_COUNT = NUMBER_10_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/**
 * The logify/nor logic cybol format.
 *
 * Apply a boolean NOR operation.
 *
 * result = x NOR y
 *
 * This is a CYBOL extension.
 *
 * Properties:
 * - output (required): the knowledge model in which the output is stored; used as first input operand
 * - input (required): the second input operand
 */
static wchar_t* NOR_LOGIFY_LOGIC_CYBOL_FORMAT = L"logify/nor";
static int* NOR_LOGIFY_LOGIC_CYBOL_FORMAT_COUNT = NUMBER_10_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/**
 * The logify/not logic cybol format.
 *
 * Apply a boolean NOT operation.
 * When used with Bit operands, then this is the ONE'S COMPLEMENT (all bits negated).
 *
 * result = x NOT y
 *
 * This is a CYBOL extension.
 *
 * Properties:
 * - output (required): the knowledge model in which the output is stored; used as first input operand
 * - input (required): the second input operand
 */
static wchar_t* NOT_LOGIFY_LOGIC_CYBOL_FORMAT = L"logify/not";
static int* NOT_LOGIFY_LOGIC_CYBOL_FORMAT_COUNT = NUMBER_10_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/**
 * The logify/or logic cybol format.
 *
 * Apply a boolean OR operation.
 *
 * result = x OR y
 *
 * This is a CYBOL extension.
 *
 * Properties:
 * - output (required): the knowledge model in which the output is stored; used as first input operand
 * - input (required): the second input operand
 */
static wchar_t* OR_LOGIFY_LOGIC_CYBOL_FORMAT = L"logify/or";
static int* OR_LOGIFY_LOGIC_CYBOL_FORMAT_COUNT = NUMBER_9_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/**
 * The logify/xnor logic cybol format.
 *
 * Apply a boolean XNOR operation.
 *
 * result = x XNOR y
 *
 * This is a CYBOL extension.
 *
 * Properties:
 * - output (required): the knowledge model in which the output is stored; used as first input operand
 * - input (required): the second input operand
 */
static wchar_t* XNOR_LOGIFY_LOGIC_CYBOL_FORMAT = L"logify/xnor";
static int* XNOR_LOGIFY_LOGIC_CYBOL_FORMAT_COUNT = NUMBER_11_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/**
 * The logify/xor logic cybol format.
 *
 * Apply a boolean XOR operation.
 *
 * result = x XOR y
 *
 * This is a CYBOL extension.
 *
 * Properties:
 * - output (required): the knowledge model in which the output is stored; used as first input operand
 * - input (required): the second input operand
 */
static wchar_t* XOR_LOGIFY_LOGIC_CYBOL_FORMAT = L"logify/xor";
static int* XOR_LOGIFY_LOGIC_CYBOL_FORMAT_COUNT = NUMBER_10_INTEGER_STATE_CYBOI_MODEL_ARRAY;

/* LOGIFY_LOGIC_CYBOL_FORMAT_CONSTANT_SOURCE */
#endif
