/*
 * Copyright (C) 1999-2020. Christian Heller.
 *
 * This file is part of the Cybernetics Oriented Interpreter (CYBOI).
 *
 * CYBOI is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published
 * by the Free Software Foundation, either version 3 of the License,
 * or (at your option) any later version.
 *
 * CYBOI is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty
 * of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with CYBOI. If not, see <http://www.gnu.org/licenses/>.
 *
 * Cybernetics Oriented Programming (CYBOP) <http://www.cybop.org/>
 * CYBOP Developers <cybop-developers@nongnu.org>
 *
 * @version CYBOP 0.21.0 2020-07-29
 * @author Christian Heller <christian.heller@cybop.org>
 */

#ifndef FILE_UNIX_TERMINAL_WRITER_SOURCE
#define FILE_UNIX_TERMINAL_WRITER_SOURCE

#include <stdio.h> // FILE, fwprintf, fflush

#include "../../../../constant/model/cyboi/log/level_log_cyboi_model.c"
#include "../../../../constant/model/cyboi/state/integer_state_cyboi_model.c"
#include "../../../../constant/model/cyboi/state/pointer_state_cyboi_model.c"
#include "../../../../logger/logger.c"

/**
 * Writes the source to the unix terminal output.
 *
 * @param p0 the destination output file stream
 * @param p1 the source data (null-terminated)
 */
void write_unix_terminal_file(void* p0, void* p1) {

    if (p1 != *NULL_POINTER_STATE_CYBOI_MODEL) {

        char* d = (char*) p1;

        if (p0 != *NULL_POINTER_STATE_CYBOI_MODEL) {

            FILE* f = (FILE*) p0;

            log_message_terminated((void*) DEBUG_LEVEL_LOG_CYBOI_MODEL, (void*) L"Write unix terminal file.");

            //
            // Write to terminal.
            //
            // CAUTION! The character data are printed out at once,
            // using the "fwprintf" function, for two reasons:
            //
            // (1) This is more efficient than using a loop and
            //     writing each single character as done in
            //     "element_file_writer.c".
            //
            // (2) The ansi escape codes are interpreted correctly.
            //
            // CAUTION! The placeholder %s is used, since the data are given
            // as utf-8 multibyte character sequence of type "char".
            // The placeholder %ls would be WRONG here as it expects data
            // of type "wchar_t".
            //
            // CAUTION! The data ought to be null-terminated.
            //
            int e = fwprintf(f, L"%s", d);

            //
            // Test error value.
            //
            // CAUTION! The macro WEOF is an integer value that is
            // returned by fwprintf to indicate an end-of-file
            // condition, or some other error situation.
            //
            // With the GNU library, WEOF is -1. In other libraries,
            // its value may be some other negative number.
            //
            // Therefore, this test checks for negative values in general.
            // The WEOF symbol which is declared in wchar.h then does
            // not have to be considered explicitly here anymore.
            //
            if (e < *NUMBER_0_INTEGER_STATE_CYBOI_MODEL) {

                log_message_terminated((void*) ERROR_LEVEL_LOG_CYBOI_MODEL, (void*) L"Could not write unix terminal file. A write error occured.");
            }

            //
            // Flush any buffered output on the stream to the file.
            //
            // If this was not done here, the buffered output on the
            // stream would only get flushed automatically when either:
            // - one tried to do output and the output buffer is full
            // - the stream was closed
            // - the program terminated by calling exit
            // - a newline was written with the stream being line buffered
            // - an input operation on any stream actually read data from its file
            //
            fflush(f);

        } else {

            log_message_terminated((void*) ERROR_LEVEL_LOG_CYBOI_MODEL, (void*) L"Could not write unix terminal file. The destination output file stream is null.");
        }

    } else {

        log_message_terminated((void*) ERROR_LEVEL_LOG_CYBOI_MODEL, (void*) L"Could not write unix terminal file. The source data is null.");
    }
}

/* FILE_UNIX_TERMINAL_WRITER_SOURCE */
#endif
