/*
 * Copyright (C) 1999-2012. Christian Heller.
 *
 * This file is part of the Cybernetics Oriented Interpreter (CYBOI).
 *
 * CYBOI is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published
 * by the Free Software Foundation, either version 3 of the License,
 * or (at your option) any later version.
 *
 * CYBOI is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty
 * of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with CYBOI. If not, see <http://www.gnu.org/licenses/>.
 *
 * Cybernetics Oriented Programming (CYBOP) <http://www.cybop.org/>
 * Christian Heller <christian.heller@tuxtax.de>
 *
 * @version CYBOP 0.11.0 2012-01-01
 * @author Christian Heller <christian.heller@tuxtax.de>
 */

#ifndef ARRAY_REALLOCATOR_SOURCE
#define ARRAY_REALLOCATOR_SOURCE

#include <stdlib.h>

#include "../../../constant/model/cyboi/state/integer_state_cyboi_model.c"
#include "../../../constant/model/cyboi/state/pointer_state_cyboi_model.c"
#include "../../../constant/type/cyboi/state_cyboi_type.c"
#include "../../../executor/calculator/basic/integer/multiply_integer_calculator.c"
#include "../../../executor/memoriser/size_determiner.c"

/**
 * Reallocates the array.
 *
 * @param p0 the array (pointer reference)
 * @param p1 the count
 * @param p2 the size
 * @param p3 the type
 */
void reallocate_array(void* p0, void* p1, void* p2, void* p3) {

    if (p2 != *NULL_POINTER_STATE_CYBOI_MODEL) {

        int* s = (int*) p2;

        if (p1 != *NULL_POINTER_STATE_CYBOI_MODEL) {

            int* c = (int*) p1;

            if (p0 != *NULL_POINTER_STATE_CYBOI_MODEL) {

                void** a = (void**) p0;

                // CAUTION! Do NOT call the logger here.
                // It uses functions causing circular references.
                // log_message_terminated((void*) DEBUG_LEVEL_LOG_CYBOI_MODEL, (void*) L"Reallocate array.");

                // The memory area.
                int ma = *NUMBER_0_INTEGER_STATE_CYBOI_MODEL;

                // Determine type (type) size.
                determine_size((void*) &ma, p3);

                // Calculate memory area.
                calculate_integer_multiply((void*) &ma, p2);

                if (ma > *NUMBER_0_INTEGER_STATE_CYBOI_MODEL) {

                    //
                    // CAUTION! The memory area (new array size)
                    // MUST NOT be zero or smaller!
                    // If it were equal to zero, then the "realloc"
                    // function call would be equivalent to "free" --
                    // an unwanted side effect that would destroy
                    // allocated memory areas and lead to errors.
                    // Therefore, that case is excluded by this condition.
                    //

                    // The temporary size_t variable.
                    //
                    // CAUTION! It IS NECESSARY because on 64 Bit machines,
                    // the "size_t" type has a size of 8 Byte,
                    // whereas the "int" type has the usual size of 4 Byte.
                    // When trying to cast between the two, memory errors
                    // will occur and the valgrind memcheck tool report:
                    // "Invalid read of size 8".
                    //
                    // CAUTION! Initialise temporary size_t variable with final int value
                    // JUST BEFORE handing that over to the glibc function requiring it.
                    //
                    // CAUTION! Do NOT use cyboi-internal copy functions to achieve that,
                    // because values are casted to int* internally again.
                    size_t tma = ma;

                    // Create a new array with extended size.
                    //
                    // Since the space after the end of the block may be in use,
                    // realloc may find it necessary to copy the block to a
                    // new address where more free space is available.
                    // The value of realloc is the new address of the block.
                    // If the block needs to be moved, realloc copies the old contents.
                    //
                    // CAUTION! The "ma" variable MAY NOT be casted to "size_t",
                    // because it is NOT a pointer, but an integer value!
                    *a = realloc(*a, tma);

                    if (*s > *c) {

                        // CAUTION! If count and size are equal, then nothing
                        // is to be done.
                        // CAUTION! Do NOT change this value if the size is
                        // smaller than the count, because this will result
                        // in a negative value and cause the new array elements
                        // pointer further below to cross the array's boundary!
                        // If the size is smaller than the count, elements
                        // outside the smaller size area are just lost.

                        // The NEW memory area to be initialised.
                        int nma = *NUMBER_0_INTEGER_STATE_CYBOI_MODEL;

                        // Calculate extra array size, which is the given array size
                        // reduced by the existing element count.
                        int es = *s - *c;

                        // Determine type (type) size.
                        determine_size((void*) &nma, p3);

                        // Calculate new memory area.
                        calculate_integer_multiply((void*) &nma, (void*) &es);

                        // The new array elements.
                        void* na = *a + (ma - nma);

                        // The temporary size_t variable.
                        //
                        // CAUTION! It IS NECESSARY because on 64 Bit machines,
                        // the "size_t" type has a size of 8 Byte,
                        // whereas the "int" type has the usual size of 4 Byte.
                        // When trying to cast between the two, memory errors
                        // will occur and the valgrind memcheck tool report:
                        // "Invalid read of size 8".
                        //
                        // CAUTION! Initialise temporary size_t variable with final int value
                        // JUST BEFORE handing that over to the glibc function requiring it.
                        //
                        // CAUTION! Do NOT use cyboi-internal copy functions to achieve that,
                        // because values are casted to int* internally again.
                        size_t tnma = nma;

                        // Initialise ONLY NEW array elements (new memory area)
                        // with zero. Leave existing elements untouched.
                        //
                        // CAUTION! Initialising with zero is essential, since cyboi
                        // frequently tests variables for null pointer values.
                        //
                        // CAUTION! Do NOT use large values, since the zero value gets
                        // converted to an unsigned char inside the "memset" function.
                        memset(na, *NUMBER_0_INTEGER_STATE_CYBOI_MODEL, tnma);
                    }

                } else {

                    log_message_terminated((void*) WARNING_LEVEL_LOG_CYBOI_MODEL, (void*) L"Could not reallocate array. The memory area is not greater than zero.");
                }

            } else {

                log_message_terminated((void*) ERROR_LEVEL_LOG_CYBOI_MODEL, (void*) L"Could not reallocate array. The array is null.");
            }

        } else {

            log_message_terminated((void*) ERROR_LEVEL_LOG_CYBOI_MODEL, (void*) L"Could not reallocate array. The count is null.");
        }

    } else {

        log_message_terminated((void*) ERROR_LEVEL_LOG_CYBOI_MODEL, (void*) L"Could not reallocate array. The size is null.");
    }
}

/* ARRAY_REALLOCATOR_SOURCE */
#endif
