# Session.py
# Copyright (C) 2002 Alex Mercader <alex.mercader@iinet.net.au>
#
# This file is part of Curphoo.
#
# Curphoo is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# Curphoo is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with Curphoo; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
# $Id: Session.py,v 1.6 2005/11/27 20:52:46 mkennedy Exp $

import os
import getpass
import sys
import urllib
import clopt
import pickle
import socket
import time
# --------------------- added 22-Feb-2003 BoW4IamTheBOFH ----------------------------
from time import localtime,strftime
# --------------------- end edit-----------------------------------------------------

CPDIR = os.path.join(os.environ['HOME'], '.curphoo')
CPRC = os.path.join(CPDIR, 'curphoorc')
CPBUDDIES = os.path.join(CPDIR, 'buds')
CPBUDDIESTXT = os.path.join(CPDIR, 'buddies')
# --------------------- added 22-Feb-2003 BoW4IamTheBOFH ----------------------------
CPDATE = strftime('curphoo-%F-%T.log',localtime())

# --------------------- end edit-----------------------------------------------------
CPIGNORESTXT = os.path.join(CPDIR, 'ignore')
# --------------------- added 22-Feb-2003 BoW4IamTheBOFH ----------------------------
CPLOG = os.path.join(CPDIR,CPDATE)
# --------------------- end edit-----------------------------------------------------
CPMYCMDS = os.path.join(CPDIR, 'commands')
CPMYREGEX = os.path.join(CPDIR, 'regex')
CPHIGHSTR = os.path.join(CPDIR, 'highlightstr')
CPHIGHUSER = os.path.join(CPDIR, 'highlightuser')
CPCOLOR = os.path.join(CPDIR, 'color')
CPIGNOREMSG = os.path.join(CPDIR, 'ignoremsg')
CPEXCLUDE = os.path.join(CPDIR, 'exclude')

class Session:


	def __init__(self):
		self.user = ''
		self.password = ''
		self.alias = []
		self.afk = 0
		self.pms_are_ok = 0
		self.pm_recipient = 'no-one' 
		self.login_ok = 0
		self.server = 'cs6.chat.yahoo.com'
		self.room = 'Linux, FreeBSD, Solaris:1'
		self.url = 'http://www.nongnu.org/curphoo/'
		self.profileurl = 'http://profiles.yahoo.com/'
		self.lastChatter = ''
		self.rc = {}
		self.users = []
		self.high = []
		self.highstr = []
		self.buddies = []
		self.ybuddies = []
		self.ignores = {}
		self.ignoremsg = ''
		self.mutes = {}  # temporary ignore
		self.excludes = []  # list of users excluded from ignore
		self.mycmd = {}
		self.myregex = []
		self.babellist = []
		self.babellang_from = 'English' 
		self.babellang_to = 'French' 
		self.cmd = ['JOIN ', 'QUIT ', 'TELL ', 'IGNORE ', 'BUDDY ', 'USERS ', 'AWAY ', 'GOTO ', 'VER', 'UPTIME', 'OPENURL ', 'UINFO ', 'COLOR ', 'MUTE ', 'LOAD ', 'SAVE ', 'OPENPROFILE ', 'EXCLUDE', 'SET ', 'REGEX'] 
		self.colormap = {'BLACK':0, 'RED':1, 'GREEN':2, 'YELLOW':3, 'BLUE':4, 'MAGENTA':5, 'CYAN':6, 'WHITE':7}
		self.colorrevmap = ('BLACK', 'RED', 'GREEN', 'YELLOW', 'BLUE', 'MAGENTA', 'CYAN', 'WHITE')
		self.defcolor = {
'autoignorenotice':'RED',
'awayoffnotice':'WHITE',
'awayonnotice':'WHITE',
'buddylist':'WHITE',
'buddyoffnotice':'BLUE',
'buddyonnotice':'BLUE',
'colorlist':'WHITE',
'commenttext':'WHITE',
'commentuser':'RED',
'defaultnotice':'BLUE',
'emotebullet':'RED',
'emotetext':'WHITE',
'errortext':'RED',
'excludelist':'GREEN',
'excludeoffnotice':'GREEN',
'excludeonnotice':'GREEN',
'help':'WHITE',
'highcommenttext':'WHITE',
'highcommentuser':'RED',
'highoffnotice':'RED',
'highonnotice':'BLUE',
'hslist':'WHITE',
'hulist':'WHITE',
'ignorelist':'RED',
'ignoreoffnotice':'BLUE',
'ignoreonnotice':'RED',
'joinnotice':'BLUE',
'logendnotice':'RED',
'loginnotice':'BLUE',
'logoutnotice':'BLUE',
'logstartnotice':'BLUE',
'mailnotice':'MAGENTA',
'muteoffnotice':'BLUE',
'muteonnotice':'RED',
'mycommenttext':'WHITE',
'mycommentuser':'BLUE',
'myemotebullet':'RED',
'myemotetext':'WHITE',
'newbuddynotice':'RED',
'openurl':'WHITE',
'pmnotice':'YELLOW',
'pmoffnotice':'RED',
'pmonnotice':'BLUE',
'rclist':'WHITE',
'shellcommand':'WHITE',
'statusbar':'BLUE',
'userenternotice':'BLUE',
'userexitnotice':'BLUE',
'userinfo':'WHITE',
'userlist':'WHITE'
}

		self.defrc = {
'username':'',
'password':'',
'room':'Linux, FreeBSD, Solaris:1',
'auto-lowercase':'Y',
'ignore-spambot':'Y',
'ignore-spambot-preignore':'Y',
'ignore-spambot-length':60,
'ignore-spambot-seconds':60,
'ignore-spambot-count':3,
'ignore-spambot-string':'http://',
'scroll-autoignorenotice':'N',
'scroll-awayoffnotice':'N',
'scroll-awayonnotice':'N',
'scroll-excludeoffnotice':'N',
'scroll-excludeonnotice':'N',
'scroll-highoffnotice':'N',
'scroll-highonnotice':'N',
'scroll-ignoreoffnotice':'N',
'scroll-ignoreonnotice':'N',
'scroll-logendnotice':'N',
'scroll-logstartnotice':'N',
'scroll-muteoffnotice':'N',
'scroll-muteonnotice':'N',
'scroll-newbuddynotice':'N',
'scroll-pmnotice':'N',
'scroll-pmoffnotice':'N',
'scroll-pmonnotice':'N',
'scroll-userenternotice':'N',
'scroll-userexitnotice':'N',
'style-comment':'<%s> ',
'style-mycomment':'<%s> ',
'pm-from-all':0
}
		self.pids = []
		self.log = None
		self.time0 = time.time()
		self.userinfo = {}  # stores startime, lastcomment, commentcount
		self.logoutusers = []  # stores last few users that have logged out
		self.colorraw = {}  # stores color profile in raw format
		self.color = {}  # stores color profile in usable form

		# create ~/.curphoo/ if it doesn't exist
		if not os.access(CPDIR, os.F_OK):
			os.mkdir(CPDIR, 0700)
			self.init_rc()
		else:
			# if ~/.curphoo/curphoorc doesn't exist
			if not os.access(CPRC, os.F_OK):
				self.init_rc()
			else:
				self.load_rc()
			
		self.load_opts() # init members with command line switches
		self.load_buds()
		self.load_buddies()  # load your buddies list
		self.load_ignores()  # load list of ignored users
		# NOTE: load_exclude() needs to be called after load_ignores because this will remove users from ignores
		self.load_exclude()  # load list of users excluded from auto ignore (for safety)
		self.load_mycmd()  # load custom commands
		self.load_myregex()  # load ignoring regex
		self.load_highlightstring()  # load list of highlighted strings
		self.load_highlightuser()  # load list of highlighted users
		self.load_color()  # load color profiles
		self.load_ignoremsg()  # load ignore notify message
		
	def clean(self):
		self.save_rc()
		self.save_buddies()
		self.save_ignores()
		self.save_exclude()
		self.save_myregex()
		self.save_highlightstring()
		self.save_highlightuser()
		self.save_color()
		if self.log:
			try:
				self.log.close()
			except IOError:
				pass

	def load_myregex(self):
		if os.access(CPMYREGEX, os.R_OK):
			f = open(CPMYREGEX, 'r')
			while 1:
				s = f.readline()
				if not s:
					break
				else:
					r = s.rstrip()
					if r:
						self.myregex.append(r)
			f.close()

	def save_myregex(self):
		try:
			self.myregex.sort();
			f = open(CPMYREGEX, 'w')
			for s in self.myregex:
				f.write(s + "\n")
			f.close()
		except:
			pass

	def load_mycmd(self):
		self.mycmd = {}
		if os.access(CPMYCMDS, os.R_OK):
			f = open(CPMYCMDS, 'r')
			k = ''
			while 1:
				s = f.readline()
				if s:
					s = s.strip()
					if not s or s.startswith('#'):
						# just a comment or a blank line
						continue
					if s.startswith('::'): # EMOTES
						if s[2:]:
							k = s[2:]
							self.mycmd[k] = ('emote', '')
					elif s.startswith(':'): # COMMENTS
						if s[1:]:
							k = s[1:]
							self.mycmd[k] = ('comment', '')
					elif s.startswith('!'): # EXEC O's
						if s[1:]:
							k = s[1:]
							self.mycmd[k] = ('exec o', '')
					elif s.startswith('|'): # FILTERS
						if s[1:]:
							k = s[1:]
							self.mycmd[k] = ('filter', '')
					else:
						if k:
							if self.mycmd[k][1]:
								self.mycmd[k] = (self.mycmd[k][0], '%s %s' % (self.mycmd[k][1], s))
							else:
								self.mycmd[k] = (self.mycmd[k][0], s)
				else:
					break
			#end while
			f.close()

	def load_buds(self):
		try:
			f = open(CPBUDDIES, 'r')
			self.buddies = pickle.load(f)
			f.close()
		except IOError, pickle.PickleError:
			pass

	def load_buddies(self):
  		try:
			f = open(CPBUDDIESTXT, 'r')
			while 1:
				s = f.readline()
				if s:
					s = s.strip()
					self.buddies.append(s);
				else:
					break
			f.close()
		except:
			pass
	
	def save_buds(self):
		try:
			f = open(CPBUDDIES, 'w')
			pickle.dump(self.buddies, f)
			f.close()
		except:
			pass

	def save_buddies(self):
		try:
			self.buddies.sort();
			f = open(CPBUDDIESTXT, 'w')
			for s in self.buddies:
				f.write(s + "\n")
			f.close()
		except:
			pass

	def load_ignores(self):
  		try:
			f = open(CPIGNORESTXT, 'r')
			while 1:
				s = f.readline()
				if s:
					s = s.strip()
					self.ignores[s] = 1;
				else:
					break
			f.close()
		except:
			pass
	
	def save_ignores(self):
		try:
			sorted_ignores = self.ignores.keys()
			sorted_ignores.sort()
			f = open(CPIGNORESTXT, 'w')
			for s in sorted_ignores:
				f.write(s + "\n")
			f.close()
		except:
			pass

	def init_rc(self):
		self.user = raw_input('Yahoo! Username: ')
		self.rc['username'] = self.user
		self.password = getpass.getpass("       Password: ")
		rep = raw_input("Save profile to ~/.curphoo/curphoorc? [Y/n]: ")[:1].lower()
		if rep in ('', 'y'):
			self.rc['password'] = self.password
		for k in self.defrc.keys():
			if not self.rc.has_key(k) or self.rc[k] == '':
				self.rc[k] = self.defrc[k]
		self.save_rc()

	def load_rc(self):
		# read ~/.curphoo/curphoorc file
		if os.access(CPRC, os.R_OK|os.W_OK):
			f = open(CPRC, 'r')
			try:
				s = f.readline()
				while s:
					scopy = s.lstrip()[:-1]
					if scopy == '':
						s = f.readline()
						continue
					if scopy[0] == '#' or scopy.find('=') == -1:
						s = f.readline()
						continue
					k = scopy.split('=',1)[0]
					k = k.lstrip().strip()
					v = scopy.split('=',1)[1]
					self.rc[k] = v
					s = f.readline()
			finally:
				f.close()
		self.user = self.rc['username']
		self.room = self.rc['room']
		# correct rc setting names  0.3.5 to 0.3.6
		if (self.rc.has_key('comment-style')):
			self.rc['style-comment'] = self.rc['comment-style'][:]
			del self.rc['comment-style']
		if (self.rc.has_key('mycomment-style')):
			self.rc['style-mycomment'] = self.rc['mycomment-style'][:]
			del self.rc['mycomment-style']
		if (self.rc.has_key('cookie')):
			del self.rc['cookie']
		# map new default rc
		for k in self.defrc.keys():
			if not self.rc.has_key(k) or self.rc[k] == '':
				self.rc[k] = self.defrc[k]
		# if a password isn't saved in curphoorc file
		if not self.rc.has_key('password') or self.rc['password'].lstrip().strip() == '':
			self.password = getpass.getpass("Yahoo! Username: %s\n       Password: " % self.rc['username'])
			rep = raw_input("Save profile to ~/.curphoo/curphoorc? [Y/n]: ")[:1].lower()
			if rep in ('', 'y'):
				self.rc['password'] = self.password
		else:
			self.password = self.rc['password']

	def create_path(self):
		os.mkdir(CPDIR, 0700)

	def creat_rc(self):
		if not os.access(CPDIR, os.F_OK):
			self.create_path()

		f = open(CPRC, 'w')
		for k in self.rc.keys():
			f.write("%s=%s\n" % (k, self.rc[k]))
		f.close()

	def save_rc(self):
		# read ~/.curphoo/curphoorc
		lines = []
		if os.access(CPRC, os.F_OK):
			f = open(CPRC, 'r')
			s = f.readline()
			while s:
				s = s.strip()
				lines.append(s)
				s = f.readline()
			f.close()

		rckeys = self.rc.keys()

		f = open(CPRC, 'w')
		for s in lines:
			scopy = s
			sdcopy = scopy.lstrip().strip()
			if scopy == '':
				f.write("\n")
				continue
			if scopy[0] == '#' or scopy.find('=') == -1:
				f.write("%s\n" % s)
				continue
			k = scopy.split('=',1)[0].lstrip().strip()
			v = scopy.split('=',1)[1]
			if self.rc.has_key(k):
				f.write("%s=%s\n" % (k, self.rc[k]))
				del rckeys[rckeys.index(k)]
		rckeys.sort()
		for k in rckeys:
			f.write("%s=%s\n" % (k, self.rc[k]))
		f.close()



	def load_opts(self):
		options = clopt.do()
		if options[0]: self.user = options[0]
		if options[1]: self.room = options[1]
# 		if options[2] in ('5', '6', '7', '8'):
		if options[2] in ('1', '2'):
			self.server = "cs%s.msg.dcn.yahoo.com" % options[2]
		if options[3]:
			try:
				self.log = open(CPLOG,'a')
			except:
				pass
		if options[4]:
			if not options[0]:
				# the username in curphoorc is not wanted
				self.user = ""
				while not self.user:
					self.user=raw_input("Username: ")
					self.user=self.user.strip()
			self.password = ""

	def load_highlightstring(self):
		try:
			f = open(CPHIGHSTR, 'r')
			while 1:
				s = f.readline()
				if s:
					s = s.strip()
					self.highstr.append(s)
				else:
					break
			f.close()
		except:
			pass

	def save_highlightstring(self):
		try:
			self.highstr.sort();
			f = open(CPHIGHSTR, 'w')
			for s in self.highstr:
				f.write(s + "\n")
			f.close()
		except:
			pass

	def load_highlightuser(self):
  		try:
			f = open(CPHIGHUSER, 'r')
			while 1:
				s = f.readline()
				if s:
					s = s.strip()
					self.high.append(s)
				else:
					break
			f.close()
		except:
			pass

	def save_highlightuser(self):
		try:
			self.high.sort();
			f = open(CPHIGHUSER, 'w')
			for s in self.high:
				f.write(s + "\n")
			f.close()
		except:
			pass

	def load_color(self):
		self.color = {}
		self.colorraw = {}
		# if ~/.curphoo/color file doesn't exist
		if not os.access(CPCOLOR, os.F_OK):
			for profile in self.defcolor.keys():
				self.colorraw[profile] = self.defcolor[profile]
				self.color[profile] = self.colormap[self.colorraw[profile].upper()]
			self.save_color()
			return

  		try:
			f = open(CPCOLOR, 'r')
			while 1:
				s = f.readline()
				if s:
					s = s.strip()
					colorname = s.split('=', 2)[0].strip().lstrip()
					colorstr = s.split('=', 2)[1].strip().lstrip()
					self.colorraw[colorname] = colorstr
					self.color[colorname] = self.colormap[colorstr.upper()]
				else:
					break
			f.close()

			# color migration patch for 0.3.2 to >=0.3.3
			if (self.color.has_key('urlopen')):
				self.color['openurl'] = self.color['urlopen']
				del self.color['urlopen']
			# color migration patch for 0.3.5 to >=0.3.6
			if (self.color.has_key('lognotice')):
				del self.color['lognotice']

			# merge in new color profile
			for profile in self.defcolor.keys():
				if not self.color.has_key(profile):
					self.colorraw[profile] = self.defcolor[profile]
					self.color[profile] = self.colormap[self.defcolor[profile].upper()]
		except:
			pass

	def save_color(self):
		try:
			profiles = self.color.keys()
			profiles.sort();
			f = open(CPCOLOR, 'w')
			for s in profiles:
				f.write("%s=%s\n" % (s, self.colorrevmap[self.color[s]]))
			f.close()
		except:
			pass

	# this function needs to parameters: color profile name and color in string
	def set_color(self, colorname, colorvalue):
		self.colorraw[colorname] = colorvalue.upper()
		self.color[colorname] = self.colormap[colorvalue.upper()]

	def load_ignoremsg(self):
		try:
			f = open(CPIGNOREMSG, 'r')
			msglist = f.readlines()
			self.ignoremsg = ''.join(msglist)
			f.close()
		except:
			self.ignoremsg = '*** %s will be ignored'

	def load_exclude(self):
		# load list of users excluded from auto ignore
		try:
			f = open(CPEXCLUDE, 'r')
			while 1:
				s = f.readline()
				if s:
					s = s.strip()
					self.excludes.append(s)
				else:
					break
			f.close()
		except:
			pass

	def save_exclude(self):
		# save list of users excluded from auto ignore
		try:
			self.excludes.sort();
			f = open(CPEXCLUDE, 'w')
			for s in self.high:
				f.write(s + "\n")
			f.close()
		except:
			pass

	def cache_logout_user(self, user):
		if user not in self.logoutusers:
			if len(self.logoutusers) >= 5:
				self.logoutusers = self.logoutusers[1:]
			self.logoutusers.append(user)
