/* this file is part of criawips a gnome presentation application
 *
 * AUTHORS
 *       Sven Herzberg        <herzi@gnome-de.org>
 *
 * Copyright (C) 2004 Sven Herzberg
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "slide-list.h"

#include <inttypes.h>

#include <glib.h>
#include <glib-object.h>
#include <glib/gi18n.h>
#include <gtk/gtk.h>
#include <glade/glade.h>

#include "debug.h"
#include "slide-list-proxy.h"
#include "slide-properties-dialog.h"

enum {
	PROP_0,
	PROP_PRESENTATION
};

enum {
	SIGNAL,
	N_SIGNALS
};

struct _CriaSlideListPrivate {
	CriaPresentation	* presentation;
	GtkTreeModel		* model;
	GladeXML		* xml;
};

GObjectClass* parent_class = NULL;

static	void	cria_slide_list_get_property   (GObject		* object,
						guint		  prop_id,
						GValue		* value,
						GParamSpec	* param_spec);
static void  cria_slide_list_init	       (CriaSlideList	* self);
static void  cria_slide_list_new_slide_after   (CriaSlideList	* self,
						GtkMenuItem	* item);
static void  cria_slide_list_new_slide_before  (CriaSlideList	* self,
						GtkMenuItem	* item);
static	void	cria_slide_list_set_property   (GObject		* object,
						guint		  prop_id,
						const	GValue	* value,
						GParamSpec	* param_spec);
static	void	cria_slide_list_show_properties(CriaSlideList	* self,
						GtkMenuItem	* item);
#if 0
static	guint	cria_slide_list_signals[N_SIGNALS] = { 0 };

static	void	cria_slide_list_signal	       (CriaSlideList	* template,
						const	gchar	* string);
#endif

static void
cria_slide_list_do_popup(CriaSlideList* self, GdkEventButton* event) {
	gint	button;
	guint	time;

	g_assert(CRIA_IS_SLIDE_LIST(self));
	g_assert(CRIA_SLIDE_LIST(self)->priv != NULL);
	g_assert(GLADE_IS_XML(CRIA_SLIDE_LIST(self)->priv->xml));

	if(event) {
		button = event->button;
		time = event->time;
	} else {
		button = 0;
		time = gtk_get_current_event_time();
	}

	gtk_menu_popup(GTK_MENU(glade_xml_get_widget(CRIA_SLIDE_LIST(self)->priv->xml, "popup_slide")),
		       NULL, NULL, NULL, NULL,
		       button, time);
}

static gboolean
cria_slide_list_popup_menu(GtkWidget* self) {
	g_assert(CRIA_IS_SLIDE_LIST(self));

	cria_slide_list_do_popup(CRIA_SLIDE_LIST(self), NULL);

	return TRUE;
}

static gboolean
cria_slide_list_button_press(GtkWidget* self, GdkEventButton* ev) {
	gboolean retval = GTK_WIDGET_CLASS(parent_class)->button_press_event(self, ev);

	if(ev->button == 3 && ev->type == GDK_BUTTON_PRESS) {
		cria_slide_list_do_popup(CRIA_SLIDE_LIST(self), ev);
		retval = TRUE;
	}

	return retval;;
}

static void
cria_slide_list_class_init(CriaSlideListClass* cria_slide_list_class) {
	GObjectClass	* g_object_class;
	GtkWidgetClass	* widget_class;

	/* setting up the GObject class */
	g_object_class = G_OBJECT_CLASS(cria_slide_list_class);
	parent_class = g_type_class_peek_parent(cria_slide_list_class);
#if 0
	/* setting up signal system */
	cria_slide_list_class->signal = cria_slide_list_signal;

	cria_slide_list_signals[SIGNAL] = g_signal_new (
			"signal",
			CRIA_TYPE_SLIDE_LIST,
			G_SIGNAL_RUN_LAST,
			G_STRUCT_OFFSET (
				CriaSlideListClass,
				signal),
			NULL,
			NULL,
			g_cclosure_marshal_VOID__STRING,
			G_TYPE_NONE,
			0);
#endif
	/* setting up property system */
	g_object_class->set_property = cria_slide_list_set_property;
	g_object_class->get_property = cria_slide_list_get_property;

	g_object_class_install_property(g_object_class,
					PROP_PRESENTATION,
					g_param_spec_object("presentation",
							    "Presentation",
							    "The presentation that's currently shown in the slide list",
							    CRIA_TYPE_PRESENTATION,
							    G_PARAM_READWRITE));

	/* setting up the GtkWidget class */
	widget_class = GTK_WIDGET_CLASS(cria_slide_list_class);
	widget_class->button_press_event = cria_slide_list_button_press;
	widget_class->popup_menu = cria_slide_list_popup_menu;
}

/**
 * cria_slide_list_get_presentation:
 * @self: The #CriaSlideList to query
 *
 * Get the presentation that's being displayed by a #CriaSlideList
 *
 * Returns the presentation displayed by a #CriaSlideList
 */
CriaPresentation*
cria_slide_list_get_presentation(CriaSlideList* self) {
	g_return_val_if_fail (CRIA_IS_SLIDE_LIST(self), NULL);
	
	return self->priv->presentation;
}

static void
cria_slide_list_get_property (
		GObject		* object,
		guint		  prop_id,
		GValue		* value,
		GParamSpec	* param_spec)
{
	CriaSlideList	* self;

	self = CRIA_SLIDE_LIST (object);

	switch (prop_id)
	{
	case PROP_PRESENTATION:
		g_value_set_object (
				value,
				self->priv->presentation);
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (
				object,
				prop_id,
				param_spec);
		break;
	}
}

/**
 * cria_slide_list_get_selected:
 * @self: a #CriaSlideList
 *
 * Get the slide that's currently selected on a #CriaSlideList (or the first
 * one if no slide is selected.
 *
 * Returns a #CriaSlide (either the selected or the first one)
 */
CriaSlide*
cria_slide_list_get_selected(CriaSlideList* self) {
	GtkTreeIter	  iter;
	GtkTreeModel	* model;
	CriaSlide	* slide;
	
	g_assert(CRIA_IS_SLIDE_LIST(self));
	g_assert(GTK_IS_TREE_VIEW(self));
	
	if(gtk_tree_selection_get_selected(gtk_tree_view_get_selection(GTK_TREE_VIEW(self)), &model, &iter)) {
		slide = cria_slide_list_proxy_get_slide(CRIA_SLIDE_LIST_PROXY(model), &iter);
	} else {
		slide = cria_presentation_get_slide(self->priv->presentation, 0);
	}
	
	return slide;
}

GType
cria_slide_list_get_type (void)
{
	static GType	type = 0;

	if (!type)
	{
		const GTypeInfo info = {
			sizeof (CriaSlideListClass),
			NULL,	/* base initializer */
			NULL,	/* base finalizer */
			(GClassInitFunc)cria_slide_list_class_init,
			NULL,	/* class finalizer */
			NULL,	/* class data */
			sizeof (CriaSlideList),
			0,
			(GInstanceInitFunc)cria_slide_list_init,
			0
		};

		type = g_type_register_static (
				GTK_TYPE_TREE_VIEW,
				"CriaSlideList",
				&info,
				0);
	}

	return type;
}

static void
cb_renderer_text_edited(CriaSlideList* self, gchar* tree_path, gchar* new_value, GtkCellRendererText* renderer) {
	GtkTreeIter	  iter;
	CriaSlide	* slide;

	g_assert(tree_path != NULL);

	g_return_if_fail(gtk_tree_model_get_iter_from_string(self->priv->model, &iter, tree_path));
	slide = cria_slide_list_proxy_get_slide(CRIA_SLIDE_LIST_PROXY(self->priv->model), &iter);
	g_return_if_fail(CRIA_IS_SLIDE(slide));

#warning "SlideList::cbRendererTextEdited(): FIXME: move this functionality to the SlideListProxy"
	cria_slide_set_title(slide, new_value);
}

static void
cria_slide_list_init(CriaSlideList *self) {
	GtkTreeViewColumn	* column;
	GtkCellRenderer		* renderer;
#warning "SlideList::init(): FIXME: add drag and drop support once the SlideListProxy works"
/*	const GtkTargetEntry target_table [] = {
		{ "GTK_TREE_MODEL_ROW", GTK_TARGET_SAME_WIDGET, 0 }
	};*/
	
	/* setting up private data */
	self->priv = g_new0(CriaSlideListPrivate,1);
#warning "SlideList::init(): FIXME: add SlideSelectionClass"
	gtk_tree_selection_set_mode(gtk_tree_view_get_selection(GTK_TREE_VIEW(self)),
				    GTK_SELECTION_BROWSE);
	
	/* setting up the tree view */
#warning "SlideList::init(): FIXME: add drag and drop support once the SlideListProxy works"
/*	gtk_tree_view_enable_model_drag_source(GTK_TREE_VIEW(self),
					       0,
					       target_table,
					       G_N_ELEMENTS(target_table),
					       GDK_ACTION_MOVE);
	gtk_tree_view_enable_model_drag_dest(GTK_TREE_VIEW(self),
					     target_table,
					     G_N_ELEMENTS(target_table),
					     GDK_ACTION_MOVE);*/
	gtk_tree_view_set_headers_visible(GTK_TREE_VIEW(self), FALSE);
	renderer = gtk_cell_renderer_text_new();
	g_object_set(G_OBJECT(renderer), "editable", TRUE, "editable-set", TRUE, NULL);
	g_signal_connect_swapped(renderer, "edited",
				 G_CALLBACK(cb_renderer_text_edited), self);
	column = gtk_tree_view_column_new();
	gtk_tree_view_column_pack_start(column, renderer, FALSE);
	gtk_tree_view_column_add_attribute(column, renderer,
					   "text", cria_slide_list_proxy_get_title_column());
	gtk_tree_view_column_set_title(column, _("Slides"));
	gtk_tree_view_append_column(GTK_TREE_VIEW(self), column);

	/* setting up the context menu */
	self->priv->xml = glade_xml_new(PACKAGE_DATA_DIR "/" PACKAGE "/data/criawips.glade", "popup_slide", NULL);
	g_signal_connect_swapped(glade_xml_get_widget(self->priv->xml, "popup_slide_menu_new_before"), "activate",
				 G_CALLBACK(cria_slide_list_new_slide_before), self);
	g_signal_connect_swapped(glade_xml_get_widget(self->priv->xml, "popup_slide_menu_new_after"), "activate",
				 G_CALLBACK(cria_slide_list_new_slide_after), self);
	gtk_widget_set_sensitive(glade_xml_get_widget(self->priv->xml, "popup_slide_menu_properties"), FALSE);
	g_signal_connect_swapped(glade_xml_get_widget(self->priv->xml, "popup_slide_menu_properties"), "activate",
				 G_CALLBACK(cria_slide_list_show_properties), self);

	return;
}

GtkWidget*
cria_slide_list_new(CriaPresentation* presentation) {
	CriaSlideList	* self = g_object_new(CRIA_TYPE_SLIDE_LIST,
					      "presentation", presentation,
					      NULL);

	g_assert(GTK_IS_WIDGET(self));

	return GTK_WIDGET(self);
}

static void
cria_slide_list_new_slide(CriaSlideList* self, gboolean after) {
	CriaSlide	* slide;
	gint		  new_pos;
	GtkTreeIter	  iter;
	GtkTreeModel	* model;

	gtk_tree_selection_get_selected(gtk_tree_view_get_selection(GTK_TREE_VIEW(self)),
					&model,
					&iter);
	slide = cria_slide_list_proxy_get_slide(CRIA_SLIDE_LIST_PROXY(model), &iter);

	new_pos = cria_presentation_get_slide_index(cria_slide_list_get_presentation(self), slide);

	if(after) {
		new_pos += 1;
	}

	g_debug("SlideList::newSlide(): new index is %i", new_pos);

	cria_slide_new_pos(cria_slide_list_get_presentation(self), new_pos);
#warning "SlideList::newSlide(): FIXME: get rid of this one. The presentation should emit a signal and the SlideList should listen to it for a redraw"
	cria_slide_list_refresh(self);
}

static void
cria_slide_list_new_slide_after(CriaSlideList* self, GtkMenuItem* item) {
	cria_slide_list_new_slide(self, TRUE);
}

static void
cria_slide_list_new_slide_before(CriaSlideList* self, GtkMenuItem* item) {
	cria_slide_list_new_slide(self, FALSE);
}

void
cria_slide_list_refresh(CriaSlideList* self) {
#warning "SlideList::refresh(): FIXME: refresh automatically and make this method static"
#if 0
	guint		  i, old_length;
	GtkTreePath	* path;
	GtkTreeIter	  iter;

	g_debug("SlideList::refresh(): start");

	/* This is what we do here:
	 * 
	 * 1. count the slides in the list
	 * 2. append the new ones
	 * 3. select the first slide of the new presentation
	 * 4. remove old elements
	 *
	 * If we did this in a different order we would run into problems like
	 * this:
	 *
	 * 1. remove the currently selected slide
	 * 2. this triggers a selection change
	 * 3. this tries to set the slide of the slide editor to a null pointer
	 * 4. nasty things happen
	 */
	
	g_assert(cria_presentation_n_slides(cria_slide_list_get_presentation(self)) > 0);

	old_length = 0;

	if(gtk_tree_model_get_iter_first(GTK_TREE_MODEL(self->priv->store), &iter)) {
		do {
			old_length++;
		} while(gtk_tree_model_iter_next(GTK_TREE_MODEL(self->priv->store), &iter));
	} else {
	}

	g_debug("SlideList::refresh(): old length is %d", old_length);
	
	/* FIXME make one islide-view for each slide */
	for(i = 0; i < cria_presentation_n_slides(cria_slide_list_get_presentation(self)); i++) {
		gtk_tree_store_append(self->priv->store, &iter, NULL);
		gtk_tree_store_set(self->priv->store, &iter,
				COLUMN_PREVIEW, NULL,
				COLUMN_TITLE, cria_slide_get_title(cria_presentation_get_slide(cria_slide_list_get_presentation(self), i)),
				COLUMN_SLIDE_PTR, cria_presentation_get_slide(cria_slide_list_get_presentation(self), i),
				-1);
		g_debug("SlideList::refresh(): added [%i]: (0x%x) %s", i, (uintptr_t)cria_presentation_get_slide(cria_slide_list_get_presentation(self), i), cria_slide_get_title(cria_presentation_get_slide(cria_slide_list_get_presentation(self), i)));
	}

	path = gtk_tree_path_new_from_indices(old_length, -1);
	gtk_tree_selection_select_path(gtk_tree_view_get_selection(GTK_TREE_VIEW(self)),
				       path);
	gtk_tree_path_free(path);

	for(i = 0; i < old_length; i++) {
		char*	title;
		CriaSlide* slide;
		path = gtk_tree_path_new_from_indices(0, -1);
		gtk_tree_model_get_iter(GTK_TREE_MODEL(self->priv->store), &iter, path);
		gtk_tree_model_get(GTK_TREE_MODEL(self->priv->store), &iter,
				   COLUMN_TITLE, &title,
				   COLUMN_SLIDE_PTR, &slide,
				   -1);
		g_debug("SlideList::refresh(): removing Slide at %d: (0x%x) \"%s\"", i+1, (uintptr_t)slide, title);
		gtk_tree_store_remove(self->priv->store, &iter);
		gtk_tree_path_free(path);
	}

	g_debug("SlideList::refresh(): end");
#endif
}

void
cria_slide_list_select_first(CriaSlideList* self) {
	GtkTreeIter	  iter;
	
	g_assert(CRIA_IS_SLIDE_LIST(self));
	g_assert(GTK_IS_TREE_VIEW(self));

	gtk_tree_model_get_iter_first(self->priv->model, &iter);
	gtk_tree_selection_select_iter(gtk_tree_view_get_selection(GTK_TREE_VIEW(self)),
				       &iter);
}

void
cria_slide_list_select_last(CriaSlideList* self) {
	GtkTreePath	* path;
	
	g_assert(CRIA_IS_SLIDE_LIST(self));
	g_assert(GTK_IS_TREE_VIEW(self));
	
	path = gtk_tree_path_new_from_indices(cria_presentation_n_slides(self->priv->presentation) - 1,
					      -1);
	gtk_tree_selection_select_path(gtk_tree_view_get_selection(GTK_TREE_VIEW(self)),
				       path);
	gtk_tree_path_free(path);
}

void
cria_slide_list_select_next(CriaSlideList* self) {
	GtkTreeIter	  iter;
	GtkTreeModel	* store;
	
	g_assert(CRIA_IS_SLIDE_LIST(self));
	g_assert(GTK_IS_TREE_VIEW(self));
	
	gtk_tree_selection_get_selected(gtk_tree_view_get_selection(GTK_TREE_VIEW(self)),
					&store,
					&iter);
	if(gtk_tree_model_iter_next(GTK_TREE_MODEL(store), &iter)) {
		gtk_tree_selection_select_iter(gtk_tree_view_get_selection(GTK_TREE_VIEW(self)),
					       &iter);
	}
}

void
cria_slide_list_select_previous(CriaSlideList* self) {
	gint	  	* indices;
	GtkTreeIter	  iter;
	GtkTreeModel	* model;
	GtkTreePath	* path,
			* newpath;

	gtk_tree_selection_get_selected(gtk_tree_view_get_selection(GTK_TREE_VIEW(self)),
					&model,
					&iter);
	
	path = gtk_tree_model_get_path(model, &iter);
	indices = gtk_tree_path_get_indices(path);
	
	if(*indices > 0) {
		*indices -= 1;
	}

	newpath = gtk_tree_path_new_from_indices(*indices, -1);
	gtk_tree_selection_select_path(gtk_tree_view_get_selection(GTK_TREE_VIEW(self)),
				       newpath);
	gtk_tree_path_free(path);
	gtk_tree_path_free(newpath);
}

/**
 * cria_slide_list_set_presentation:
 * @self: a #CriaSlideList
 * @presentation: a #CriaPresentation
 *
 * Tell @self to display @presentation.
 */
void
cria_slide_list_set_presentation(CriaSlideList* self, CriaPresentation* presentation) {
	g_debug("SlideList::setPresentation(): start");
	
	g_return_if_fail(CRIA_IS_SLIDE_LIST(self));
	g_return_if_fail(CRIA_IS_PRESENTATION(presentation));

	g_assert(cria_presentation_n_slides(presentation) > 0);
	g_assert(CRIA_IS_SLIDE(cria_presentation_get_slide(presentation, 0)));

	if(presentation == self->priv->presentation) {
		return;
	}

	if(self->priv->presentation != NULL) {
		g_object_unref(self->priv->presentation);
	}

	if(presentation != NULL) {
		g_debug("SlideList::setPresentation(): got a real presentation");
		self->priv->presentation = g_object_ref(presentation);
		self->priv->model = GTK_TREE_MODEL(cria_slide_list_proxy_for_presentation(presentation));
		gtk_tree_view_set_model(GTK_TREE_VIEW(self),
					self->priv->model);
	}

	g_object_notify(G_OBJECT(self), "presentation");

	g_debug("SlideList::setPresentation(): end");
}

static void
cria_slide_list_set_property (
		GObject		* object,
		guint		  prop_id,
		const	GValue	* value,
		GParamSpec	* param_spec)
{
	CriaSlideList	* self;
	
	self = CRIA_SLIDE_LIST (object);
	
	switch (prop_id)
	{
	case PROP_PRESENTATION:
		cria_slide_list_set_presentation (
				self,
				g_value_get_object (value));
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (
				object,
				prop_id,
				param_spec);
		break;
	}
}

static void
cria_slide_list_show_properties(CriaSlideList* self, GtkMenuItem* item) {
	CriaSlide	* slide;
	GtkTreeModel	* model;
	GtkTreeIter	  iter;

	g_assert(CRIA_IS_SLIDE_LIST(self));

	gtk_tree_selection_get_selected(gtk_tree_view_get_selection(GTK_TREE_VIEW(self)),
					&model,
					&iter);
	slide = cria_slide_list_proxy_get_slide(CRIA_SLIDE_LIST_PROXY(model), &iter);
	g_return_if_fail(CRIA_IS_SLIDE(slide));
	gtk_widget_show(cria_slide_properties_dialog_new(slide));
}

