/* this file is part of criawips a gnome presentation application
 *
 * AUTHORS
 *       Sven Herzberg        <herzi@gnome-de.org>
 *
 * Copyright (C) 2004 Sven Herzberg
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "block.h"

#include <inttypes.h>
#include <string.h>

#include <glib.h>
#include <glib/gi18n.h>
#include <glib-object.h>

#include <goffice/utils/go-color.h>

#include "debug.h"
#include "enumerations.h"

enum {
	PROP_0,
	PROP_NAME,
	PROP_POS_LEFT,
	PROP_POS_RIGHT,
	PROP_POS_TOP,
	PROP_POS_BOTTOM,
	PROP_ALIGNMENT_HORIZONTAL,
	PROP_ALIGNMENT_VERTICAL,
};

enum {
	SIGNAL,
	N_SIGNALS
};

struct _CriaBlockPrivate {
	gchar			* name,
	// FIXME: get rid of this one once we start to work out the span/block code
				* text;
	GoRect			* position;
	/* FIXME: port the code to GOFont later */
	//const GOFont		* font;
	PangoFontDescription	* font;
	CriaAlignment		  alignment;
	CriaVAlignment		  valignment;
	CriaSlide		* slide;
	GOColor			* color;
};

static	void	cria_block_get_property	       (GObject		* object,
						guint		  prop_id,
						GValue		* value,
						GParamSpec	* param_spec);
static void	cria_block_init		       (CriaBlock	* self);
static	void	cria_block_set_property        (GObject		* object,
						guint		  prop_id,
						const	GValue	* value,
						GParamSpec	* param_spec);
#if 0
/* enable these to add support for signals */
static	guint	cria_block_signals[N_SIGNALS] = { 0 };

static	void	cria_block_signal	       (CriaBlock	* self,
						const	gchar	* string);
#endif

static void
cria_block_class_init (CriaBlockClass	* cria_block_class) {
	GObjectClass	* g_object_class;

#warning "static Block::init(): FIXME: make the default font configurable"
	cria_block_class->default_font = pango_font_description_new();
	pango_font_description_set_family(cria_block_class->default_font, "Sans");
	pango_font_description_set_size(cria_block_class->default_font, 12*8);

	g_object_class = G_OBJECT_CLASS(cria_block_class);
#if 0
	/* setting up signal system */
	cria_block_class->signal = cria_block_signal;

	cria_block_signals[SIGNAL] = g_signal_new (
			"signal",
			CRIA_TYPE_BLOCK,
			G_SIGNAL_RUN_LAST,
			G_STRUCT_OFFSET (
				CriaBlockClass,
				signal),
			NULL,
			NULL,
			g_cclosure_marshal_VOID__STRING,
			G_TYPE_NONE,
			0);
#endif
	/* setting up property system */
	g_object_class->set_property = cria_block_set_property;
	g_object_class->get_property = cria_block_get_property;

	g_object_class_install_property(g_object_class,
					PROP_ALIGNMENT_HORIZONTAL,
					g_param_spec_enum("alignment",
							  "Horizontal Alignment",
							  "The horizontal alignment of embedded elements",
							  CRIA_TYPE_ALIGNMENT,
							  CRIA_ALIGNMENT_UNSET,
							  G_PARAM_READWRITE | G_PARAM_CONSTRUCT));
	g_object_class_install_property(g_object_class,
					PROP_POS_BOTTOM,
					g_param_spec_uint64("bottom",
							    "Bottom",
							    "The width of the block",
							    0,
							    G_MAXUINT64,
							    0,
							    G_PARAM_READWRITE));
	g_object_class_install_property(g_object_class,
					PROP_POS_LEFT,
					g_param_spec_uint64("left",
							    "Left",
							    "The Position of the topleft corner of the layout box",
							    0,
							    G_MAXUINT64,
							    0,
							    G_PARAM_READWRITE));
	g_object_class_install_property(g_object_class,
					PROP_NAME,
					g_param_spec_string("name",
							    "Name",
							    "The name of this block",
							    _("untitled block"),
							    G_PARAM_READWRITE | G_PARAM_CONSTRUCT));
	g_object_class_install_property(g_object_class,
					PROP_POS_RIGHT,
					g_param_spec_uint64("right",
							    "Right",
							    "The height of the block",
							    0,
							    G_MAXUINT64,
							    0,
							    G_PARAM_READWRITE));
	g_object_class_install_property(g_object_class,
					PROP_POS_TOP,
					g_param_spec_uint64("top",
							    "Top",
							    "The vertical position of the block",
							    0,
							    G_MAXUINT64,
							    0,
							    G_PARAM_READWRITE));
	g_object_class_install_property(g_object_class,
					PROP_ALIGNMENT_VERTICAL,
					g_param_spec_enum("valignment",
							  "Vertical Alignment",
							  "The vertical alignment of embedded elements",
							  CRIA_TYPE_VALIGNMENT,
							  CRIA_VALIGNMENT_UNSET,
							  G_PARAM_READWRITE | G_PARAM_CONSTRUCT));
}

static CriaBlock*
get_template(CriaBlock* self) {
	CriaSlide* master;
	CriaBlock* template;
	
	master   = cria_slide_get_master_slide(self->priv->slide);
	template = (master == NULL)?NULL:cria_slide_get_block(master, self->priv->name);

	return template;
}

/**
 * cria_block_get_alignment:
 * @self: the block to get the alignment from
 *
 * Get the horizontal alignment of content in this block
 *
 * Returns the horizontal alignment of content
 */
CriaAlignment
cria_block_get_alignment(CriaBlock* self) {
	CriaBlock* template;
	
	g_assert(self != NULL && CRIA_IS_BLOCK(self));
	g_assert(self->priv != NULL);

	template = get_template(self);

	if(self->priv->alignment == CRIA_ALIGNMENT_UNSET && CRIA_IS_BLOCK(template)) {
		return cria_block_get_alignment(template);
	}

	return self->priv->alignment;
}

/**
 * cria_block_get_color:
 * @self: a #CriaBlock
 * 
 * Get the foreground color of a text block.
 *
 * Returns the color of this text block; if it's not set it returns a default
 * value.
 */
GOColor
cria_block_get_color(CriaBlock* self) {
	CriaBlock	* template;
	
	g_assert(CRIA_IS_BLOCK(self));
	g_assert(self->priv != NULL);

	template = get_template(self);

	if(self->priv->color != NULL) {
		g_debug("Block::getColor(): returning our own");
		return *(self->priv->color);
	} else if (CRIA_IS_BLOCK(template)) {
		g_debug("Block::getColor(): returning the one from the template");
		return cria_block_get_color(template);
	} else {
		/* FIXME: do we want to handle the default here or in the renderer? */
		g_debug("Block::getColor(): returning black");
		return RGBA_BLACK;
	}
}

/**
 * cria_block_get_font:
 * @self: a #CriaBlock
 *
 * Get the font used to render a #CriaBlock.
 *
 * Returns the font used to render this block, NULL if none is set.
 */
const PangoFontDescription*
cria_block_get_font(CriaBlock* self) {
	const PangoFontDescription* font;
	CriaBlock	    * template;
	
	g_assert(self != NULL && CRIA_IS_BLOCK(self));
	g_assert(self->priv != NULL);
	g_assert(self->priv->font != NULL || CRIA_BLOCK_GET_CLASS(self)->default_font != NULL);

	template = get_template(self);

	if(self->priv->font) {
		g_debug("Block::getFont(): returning own font");
		font = self->priv->font;
	} else if(template) {
		g_debug("Block::getFont(): returning from template");
		font = cria_block_get_font(template);
	} else {
		g_debug("Block::getFont(): returing from class");
		font = CRIA_BLOCK_GET_CLASS(self)->default_font;
	}

	g_debug("Block::getFont(): returning %s %i", pango_font_description_get_family(font), pango_font_description_get_size(font) / PANGO_SCALE);

	return font;
}

/**
 * cria_block_get_markup:
 * @self: a #CriaBlock
 *
 * Get the text that should be rendered into a block.
 *
 * Returns the text that should be rendered into a block
 */
const gchar*
cria_block_get_markup(CriaBlock* self) {
#warning "Block::getMarkup(): get rid of me"
	g_assert(self != NULL && CRIA_IS_BLOCK(self));
	g_assert(self->priv != NULL);
	g_assert(self->priv->text != NULL);
	
	return self->priv->text;
}

/**
 * cria_block_get_name:
 * @self: a #CriaBlock
 *
 * Get the name of a block.
 *
 * Returns a block's name.
 */
const gchar*
cria_block_get_name(CriaBlock* self) {
	g_assert(self != NULL && CRIA_IS_BLOCK(self));
	g_assert(self->priv != NULL);
	
	return self->priv->name;
}

/**
 * cria_block_get_position:
 * @self: a #CriaBlock
 *
 * Get the position of a #CriaBlock.
 *
 * Returns the block's position. May return NULL if unset.
 */
GoRect*
cria_block_get_position(CriaBlock* self) {
	GoRect		* retval = NULL;
	CriaSlide	* slide;
	CriaBlock	* template;
	
	g_assert(self != NULL && CRIA_IS_BLOCK(self));
	g_assert(self->priv != NULL);
	g_assert(CRIA_IS_SLIDE(self->priv->slide));

	slide    = self->priv->slide;
	template = get_template(self);

	if(self->priv->position != NULL) {
		g_debug("Block::getPosition(): returning own position");
		retval = g_new0(GoRect,1);
		retval->left   = self->priv->position->left;
		retval->right  = self->priv->position->right;
		retval->top    = self->priv->position->top;
		retval->bottom = self->priv->position->bottom;
	} else if(CRIA_IS_SLIDE(slide) && CRIA_IS_BLOCK(template)) {
		g_debug("Block::getPosition(): returning position from master block");
		retval = template->priv->position;
	} else {
		g_debug("Block::getPosition(): returning NULL");
	}

	return retval;
}

static void
cria_block_get_property(GObject* object, guint prop_id, GValue* value, GParamSpec* param_spec) {
	CriaBlock	* self;

	self = CRIA_BLOCK(object);

	switch (prop_id) {
	case PROP_ALIGNMENT_HORIZONTAL:
		g_value_set_enum(value, cria_block_get_alignment(self));
		break;
	case PROP_ALIGNMENT_VERTICAL:
		g_value_set_enum(value, cria_block_get_valignment(self));
		break;
	case PROP_NAME:
		g_value_set_string(value, cria_block_get_name(self));
		break;
	case PROP_POS_LEFT:
		g_value_set_uint64(value, self->priv->position->left);
		break;
	case PROP_POS_RIGHT:
		g_value_set_uint64(value, self->priv->position->right);
		break;
	case PROP_POS_TOP:
		g_value_set_uint64(value, self->priv->position->top);
		break;
	case PROP_POS_BOTTOM:
		g_value_set_uint64(value, self->priv->position->bottom);
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID(object,
						  prop_id,
						  param_spec);
		break;
	}
}

GType
cria_block_get_type(void) {
	static GType	type = 0;

	if(!type) {
		static const GTypeInfo info = {
			sizeof (CriaBlockClass),
			NULL,	/* base initializer */
			NULL,	/* base finalizer */
			(GClassInitFunc)cria_block_class_init,
			NULL,	/* class finalizer */
			NULL,	/* class data */
			sizeof (CriaBlock),
			0,
			(GInstanceInitFunc)cria_block_init,
			0
		};

		type = g_type_register_static(G_TYPE_OBJECT,
					      "CriaBlock",
					      &info,
					      0);
	}

	return type;
}

/**
 * cria_block_get_valignment:
 * @self: the block to get the alignment from
 *
 * Get the vertical alignment of content in this block
 *
 * Returns the vertical alignment of content
 */
CriaVAlignment
cria_block_get_valignment(CriaBlock* self) {
	CriaBlock* template;
	
	g_assert(self != NULL && CRIA_IS_BLOCK(self));
	g_assert(self->priv != NULL);

	template = get_template(self);

	if(self->priv->valignment == CRIA_VALIGNMENT_UNSET && template) {
		return cria_block_get_valignment(template);
	}

	return self->priv->valignment;
}

static void
cria_block_init(CriaBlock* self) {
	self->priv = g_new0(CriaBlockPrivate,1);
#warning "CriaBlock::init(): FIXME: add configurable default font/size"
	self->priv->font = NULL;
	self->priv->text = g_strdup("");
}

CriaBlock*
cria_block_new(CriaSlide* slide, const gchar* name) {
	CriaBlock	* self;

	g_assert(CRIA_IS_SLIDE(slide));
	
	self = g_object_new(CRIA_TYPE_BLOCK, "name", name, NULL);
	self->priv->slide = g_object_ref(slide);
	
	return self;
}

void
cria_block_set_alignment(CriaBlock* self, CriaAlignment align) {
	g_assert(self != NULL && CRIA_IS_BLOCK(self));
	g_assert(self->priv != NULL);

	self->priv->alignment = align;

	g_object_notify(G_OBJECT(self), "alignment");
}

/**
 * cria_block_set_color:
 * @self: a #CriaBlock
 * @color: a #GOColor
 *
 * Set the foreground color of a text block.
 */
void
cria_block_set_color(CriaBlock* self, GOColor color) {
	g_assert(CRIA_IS_BLOCK(self));
	g_assert(self->priv != NULL);

	if(self->priv->color == NULL) {
		self->priv->color = g_new0(GOColor, 1);
	}

	*(self->priv->color) = color;

#warning "Block::setColor(): to this GObject-ish"
}

void
cria_block_set_font_family(CriaBlock* self, const gchar* font) {
	g_assert(self != NULL && CRIA_IS_BLOCK(self));
	g_assert(self->priv != NULL);

	if(!self->priv->font) {
		self->priv->font = pango_font_description_copy(CRIA_BLOCK_GET_CLASS(self)->default_font);
	}

	pango_font_description_set_family(self->priv->font, font);
	g_debug("Block::setFontFamily(): set to \"%s\" %i", pango_font_description_get_family(self->priv->font), pango_font_description_get_size(self->priv->font));
}

void
cria_block_set_font_size(CriaBlock* self, gint size) {
	g_assert(self != NULL && CRIA_IS_BLOCK(self));
	g_assert(self->priv);

	if(!self->priv->font) {
		self->priv->font = pango_font_description_copy(CRIA_BLOCK_GET_CLASS(self)->default_font);
	}

	pango_font_description_set_size(self->priv->font, size);
	g_debug("Block::setFontSize(): set to \"%s\" %i", pango_font_description_get_family(self->priv->font), pango_font_description_get_size(self->priv->font));
}

void
cria_block_set_name(CriaBlock* self, const gchar* name) {
	g_assert(self != NULL && CRIA_IS_BLOCK(self));
	g_assert(name != NULL);

	if(self->priv->name && !strcmp(name, self->priv->name)) {
		return;
	}

	if (self->priv->name != NULL) {
		g_free(self->priv->name);
	}

	self->priv->name = g_strdup(name);

	g_object_notify(G_OBJECT(self), "name");
}

static void
cria_block_set_pos_bottom(CriaBlock* self, go_unit_t bottom) {
	g_assert(self != NULL && CRIA_IS_BLOCK(self));
	g_assert(self->priv != NULL);

	g_debug("Block::setPosBottom(): %lli", bottom);
	
	if(self->priv->position == NULL) {
		self->priv->position = g_new0(GoRect,1);
	}

	if(bottom != self->priv->position->bottom) {
		self->priv->position->bottom = bottom;

		g_object_notify(G_OBJECT(self), "bottom");
	}
}

static void
cria_block_set_pos_left(CriaBlock* self, go_unit_t left) {
	g_assert(self != NULL && CRIA_IS_BLOCK(self));
	g_assert(self->priv != NULL);
	
	g_debug("Block::setPosLeft(): %lli", left);
	
	if(self->priv->position == NULL) {
		self->priv->position = g_new0(GoRect,1);
	}

	if(left != self->priv->position->left) {
		self->priv->position->left = left;

		g_object_notify(G_OBJECT(self), "left");
	}
}

static void
cria_block_set_pos_right(CriaBlock* self, go_unit_t right) {
	g_assert(self != NULL && CRIA_IS_BLOCK(self));
	g_assert(self->priv != NULL);
	
	g_debug("Block::setPosRight(): %lli", right);
	
	if(self->priv->position == NULL) {
		self->priv->position = g_new0(GoRect,1);
	}

	if(right != self->priv->position->right) {
		self->priv->position->right = right;

		g_object_notify(G_OBJECT(self), "right");
	}
}

static void
cria_block_set_pos_top(CriaBlock* self, go_unit_t top) {
	g_assert(self != NULL && CRIA_IS_BLOCK(self));
	g_assert(self->priv != NULL);
	
	g_debug("Block::setPosTop(): %lli", top);
	
	if(self->priv->position == NULL) {
		self->priv->position = g_new0(GoRect,1);
	}

	if(top != self->priv->position->top) {
		self->priv->position->top = top;

		g_object_notify(G_OBJECT(self), "top");
	}
}

static void
cria_block_set_property(GObject* object, guint prop_id, const GValue* value, GParamSpec* param_spec) {
	CriaBlock	* self;
	
	self = CRIA_BLOCK (object);
	
	switch (prop_id) {
	case PROP_ALIGNMENT_HORIZONTAL:
		cria_block_set_alignment(self, g_value_get_enum(value));
		break;
	case PROP_ALIGNMENT_VERTICAL:
		cria_block_set_valignment(self, g_value_get_enum(value));
		break;
	case PROP_NAME:
		cria_block_set_name(self, g_value_get_string(value));
		break;
	case PROP_POS_LEFT:
		cria_block_set_pos_left(self, g_value_get_uint64(value));
		break;
	case PROP_POS_RIGHT:
		cria_block_set_pos_right(self, g_value_get_uint64(value));
		break;
	case PROP_POS_TOP:
		cria_block_set_pos_top(self, g_value_get_uint64(value));
		break;
	case PROP_POS_BOTTOM:
		cria_block_set_pos_bottom(self, g_value_get_uint64(value));
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID(object,
						  prop_id,
						  param_spec);
		break;
	}
}

void
cria_block_set_text(CriaBlock* self, const gchar* text) {
#warning "Block::setText(): get rid of me"
	g_assert(self != NULL && CRIA_IS_BLOCK(self));
	g_assert(text != NULL && strlen(text));

	if(self->priv->text) {
		g_free(self->priv->text);
	}
	
	self->priv->text = g_strdup(text);
}

void
cria_block_set_valignment(CriaBlock* self, CriaVAlignment valign) {
	g_assert(self != NULL && CRIA_IS_BLOCK(self));
	g_assert(self->priv != NULL);
	
	self->priv->valignment = valign;

	g_object_notify(G_OBJECT(self), "valignment");
}

