/* this file is part of criawips a gnome presentation application
 *
 * AUTHORS
 *       Sven Herzberg        <herzi@gnome-de.org>
 *
 * Copyright (C) 2004 Sven Herzberg
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "renderer.h"

#include <inttypes.h>

#include <glib.h>
#include <glib-object.h>
#include <libgnomecanvas/libgnomecanvas.h>

#include <gnome-canvas-helpers.h>
#include "debug.h"
#include "display.h"
#include "enumerations.h"
#include "image-pool.h"

enum {
	PROP_0,
};

typedef struct _CriaRenderer CriaRenderer;
typedef struct _CriaRendererClass CriaRendererClass;
typedef struct _CriaRendererPrivate CriaRendererPrivate;

static void  cria_renderer_render_background_color     (CriaDisplay	* display,
							CriaSlide	* slide);
static void  cria_renderer_render_background_default   (CriaDisplay	* display,
							CriaSlide	* slide);
static void  cria_renderer_render_background_pixmap    (CriaDisplay	* display,
							CriaSlide	* slide);

#define CRIA_TYPE_RENDERER		(cria_renderer_get_type ())
#define CRIA_RENDERER(object)		(G_TYPE_CHECK_INSTANCE_CAST((object), CRIA_TYPE_RENDERER, CriaRenderer))
#define CRIA_RENDERER_CLASS(klass)	(G_TYPE_CHACK_CLASS_CAST((klass), CRIA_TYPE_RENDERER, CriaRendererClass))
#define CRIA_IS_RENDERER(object)	(G_TYPE_CHECK_INSTANCE_TYPE((object), CRIA_TYPE_RENDERER))
#define CRIA_IS_RENDERER_CLASS(klass)	(G_TYPE_CHECK_CLASS_TYPE((klass), CRIA_TYPE_RENDERER))
#define CRIA_RENDERER_GET_CLASS(object)	(G_TYPE_INSTANCE_GET_CLASS((object), CRIA_TYPE_RENDERER, CriaRendererClass))

struct _CriaRenderer {
	GObject		  base_instance;
};

struct _CriaRendererClass {
	GObjectClass	  base_class;
};

static void
cria_renderer_clear_group(CriaDisplay* display) {
	GList		* it;
	GnomeCanvasGroup* group;

	g_assert(CRIA_IS_DISPLAY(display));

	group = cria_display_get_top_group(display);

	for(it = group->item_list; it; it = it->next) {
		if(GNOME_IS_CANVAS_GROUP(it->data)) {
			gnome_canvas_item_hide(GNOME_CANVAS_ITEM(it->data));
			gtk_object_destroy(GTK_OBJECT(it->data));
			break;
		}
	}

	gnome_canvas_item_new(group,
			      gnome_canvas_group_get_type(),
			      NULL);
}

static GnomeCanvasGroup*
get_render_group_from_display(CriaDisplay* display) {
	GnomeCanvasGroup* group = NULL;
	GList           * it;

	g_assert(CRIA_IS_DISPLAY(display));

	it = cria_display_get_top_group(display)->item_list;

	for(; it; it = it->next) {
		if(GNOME_IS_CANVAS_GROUP(it->data)) {
			group = GNOME_CANVAS_GROUP(it->data);
		}
	}

	g_assert(group);

	return group;
}

static void
cria_renderer_render_background(CriaDisplay* display, CriaSlide* slide) {
	g_assert(CRIA_IS_DISPLAY(display));

	if(cria_slide_get_background_color(slide)) {
		/* create a layer with color */
		cria_renderer_render_background_color(display, slide);
	} else if (FALSE) {
		/* create a layer with a gradient */
#warning "Renderer::renderBackground(): FIXME: implement gradient background"
	} else {
		/* render the default background */
		cria_renderer_render_background_default(display, slide);
	}

	if (cria_slide_get_background(slide)) {
		/* render the image background */
		cria_renderer_render_background_pixmap(display, slide);
	}
}

static void
cria_renderer_render_background_color(CriaDisplay* display, CriaSlide* slide) {
	const GoPoint	* size = cria_slide_get_size(slide);
	GnomeCanvasItem	* item = gnome_canvas_item_new(get_render_group_from_display(display),
						       gnome_canvas_rect_get_type(),
						       "fill-color-rgba", *(cria_slide_get_background_color(slide)),
						       "x1", 0.0,
						       "y1", 0.0,
						       "x2", 0.0 + size->x,
						       "y2", 0.0 + size->y,
						       NULL);
	gnome_canvas_item_show(item);
}

static void
cria_renderer_render_background_default(CriaDisplay* display, CriaSlide* slide) {
	GnomeCanvasItem	* item;
	const GoPoint	* size = cria_slide_get_size(slide);

	item = gnome_canvas_item_new(get_render_group_from_display(display),
				     gnome_canvas_rect_get_type(),
				     "fill-color", "white",
				     "x1", 0.0,
				     "y1", 0.0,
				     "x2", 0.0 + size->x,
				     "y2", 0.0 + size->y,
				     NULL);
	gnome_canvas_item_show(item);
}

static void
cria_renderer_render_background_pixmap(CriaDisplay* display, CriaSlide* slide) {
	if (TRUE) { /* is_stretched */
		const GoPoint	* size = cria_slide_get_size(slide);
		GnomeCanvasGroup* group = get_render_group_from_display(display);
		GError		* error = NULL;
		GdkPixbuf	* pixbuf = cria_image_pool_get_pixbuf(cria_slide_get_background(slide), &error);

		if(pixbuf == NULL) {
#warning "Renderer::renderBackgroundPixmap(): FIXME: add GTK_STOCK_MISSING_IMAGE"
		}

		if(error) {
#warning "Renderer::renderBackgroundPixmap(): FIXME: push out a warning here"	
		}
		
		GnomeCanvasItem	* c_item = gnome_canvas_item_new(group,
								 gnome_canvas_pixbuf_get_type(),
								 "x", 0.0,
								 "y", 0.0,
								 "height", 0.0 + size->y,
								 "height-set", TRUE,
								 "pixbuf", pixbuf,
								 "width", 0.0 + size->x,
								 "width-set", TRUE,
								 NULL);
		gnome_canvas_item_show(c_item);
	} else if (FALSE) { /* is scaled */
#warning "Renderer::renderBackgroundPixmap(): FIXME: implement streched background"
		cria_renderer_render_background_default(display, slide);
	} else if (FALSE) { /* is tiled */
#warning "Renderer::renderBackgroundPixmap(): FIXME: implement tiled background"
		cria_renderer_render_background_default(display, slide);
	} else if (FALSE) { /* is aligned */
#warning "Renderer::renderBackgroundPixmap(): FIXME: implement aligned background"
		cria_renderer_render_background_default(display, slide);
	} else {
		g_assert_not_reached();
	}
}

/* these are type_map[horiz][vert] */
static GtkAnchorType type_map[][4] = {
	// unset
	{ GTK_ANCHOR_NW, GTK_ANCHOR_NW, GTK_ANCHOR_W,      GTK_ANCHOR_SW },
	// left
	{ GTK_ANCHOR_NW, GTK_ANCHOR_NW, GTK_ANCHOR_W,      GTK_ANCHOR_SW },
	// center
	{ GTK_ANCHOR_N,  GTK_ANCHOR_N,  GTK_ANCHOR_CENTER, GTK_ANCHOR_S  },
	// right
	{ GTK_ANCHOR_NE, GTK_ANCHOR_NE, GTK_ANCHOR_E,      GTK_ANCHOR_SE },
	// block
	{ GTK_ANCHOR_NW, GTK_ANCHOR_NW, GTK_ANCHOR_W,      GTK_ANCHOR_SW }
};

static GtkAnchorType
get_anchor_from_alignments(CriaAlignment align, CriaVAlignment valign) {
	return type_map[align][valign];
}

static GtkJustification
get_justification_from_alignment(CriaAlignment align) {
	switch(align) {
	case CRIA_ALIGNMENT_RIGHT:
		return GTK_JUSTIFY_RIGHT;
	case CRIA_ALIGNMENT_CENTER:
		return GTK_JUSTIFY_CENTER;
	case CRIA_ALIGNMENT_LEFT:
	default:
		return GTK_JUSTIFY_LEFT;
	}
}

static gdouble
get_x_from_alignment(CriaAlignment align, go_unit_t left, go_unit_t right) {
	switch(align) {
	case CRIA_ALIGNMENT_RIGHT:
		return 1.0 * right;
	case CRIA_ALIGNMENT_CENTER:
		return 0.5 * (left + right);
	case CRIA_ALIGNMENT_UNSET:
	case CRIA_ALIGNMENT_LEFT:
	case CRIA_ALIGNMENT_JUSTIFY:
		return 1.0 * left;
	}

	g_assert_not_reached();
	return 0.0;
}

static gdouble
get_y_from_valignment(CriaVAlignment align, go_unit_t top, go_unit_t bottom) {
	switch(align) {
	case CRIA_ALIGNMENT_BOTTOM:
		return 1.0 * bottom;
	case CRIA_ALIGNMENT_MIDDLE:
		return 0.5 * (top + bottom);
	case CRIA_VALIGNMENT_UNSET:
	case CRIA_ALIGNMENT_TOP:
		return 1.0 * top;
	}

	g_assert_not_reached();
	return 0.0;
}

static void
cria_renderer_render_block(CriaDisplay* display, CriaSlide* slide, CriaBlock* block) {
	const PangoFontDescription *font;
	GnomeCanvasItem	* item;
	GoRect		* block_position;
	const GoPoint	* slide_size;
	gdouble		  x = 0.0,
			  y = 0.0;
	GtkAnchorType	  anchor;
	GtkJustification  justification;
	GnomeCanvasGroup* group = get_render_group_from_display(display);
	GOColor		  color;
	
	g_debug("Renderer::renderBlock(): %s", cria_block_get_name(block));

	g_assert(group != NULL && GNOME_IS_CANVAS_GROUP(group));
	g_assert(slide != NULL && CRIA_IS_SLIDE(slide));
	g_assert(block != NULL && CRIA_IS_BLOCK(block));

	block_position = cria_block_get_position(block);
	slide_size     = cria_slide_get_size(slide);
	color          = cria_block_get_color(block);

	if(block_position == NULL) {
		block_position = g_new0(GoRect,1);
		block_position->left   = 0;
		block_position->right  = 0;
		block_position->top    = 0;
		block_position->bottom = 0;
	}

	font = cria_block_get_font(block);
	g_debug("Renderer::renderBlock(): Font: %s %i", pango_font_description_get_family(font), pango_font_description_get_size(font));

	g_debug("Renderer::renderBlock(): left:   %lli", block_position->left);
	g_debug("Renderer::renderBlock(): right:  %lli", block_position->right);
	g_debug("Renderer::renderBlock(): top:    %lli", block_position->top);
	g_debug("Renderer::renderBlock(): bottom: %lli", block_position->bottom);

	g_debug("Renderer::renderBlock(): alignment %i", cria_block_get_alignment(block));

	anchor = get_anchor_from_alignments(cria_block_get_alignment(block), cria_block_get_valignment(block));
	x = get_x_from_alignment(cria_block_get_alignment(block), block_position->left, block_position->right);
	y = get_y_from_valignment(cria_block_get_valignment(block), block_position->top, block_position->bottom);
	justification = get_justification_from_alignment(cria_block_get_alignment(block));

#warning "Renderer::renderBlock(): FIXME: Add clipping"
//	"clip", TRUE,
//	"clip-height", height,
//	"clip-width", width,

	item = gnome_canvas_item_new(group,
				     gnome_canvas_text_get_type(),
				     "anchor", anchor,
				     "fill-color-rgba", color,
				     "font-desc", cria_block_get_font(block),
				     "justification", justification,
				     "size-set", TRUE,
				     "text", cria_block_get_markup(block),
				     "x", x,
				     "y", y,
				     NULL);

	//g_free(block_position);

	g_debug("Renderer::renderBlock(): showing canvas item at size %f", 0.0 + pango_font_description_get_size(font));
	gnome_canvas_text_set_zoom_size(GNOME_CANVAS_TEXT(item), 0.0 + pango_font_description_get_size(font));
	gnome_canvas_item_show(item);
	g_debug("Renderer::renderBlock(): done");
}

static void
cria_renderer_render_blocks(CriaDisplay* display, CriaSlide* slide) {
	gchar		**blocks,
			**iterator;

	g_assert(slide != NULL && CRIA_IS_SLIDE(slide));

	blocks = cria_slide_get_block_names(slide);

	for(iterator = blocks; iterator && *iterator; iterator++) {
		g_debug("Renderer::renderBlocks(): rendering \"%s\"", *iterator);
		cria_renderer_render_block(display, slide, cria_slide_get_block(slide, *iterator));
	}

	//g_free(blocks);
}

void
cria_renderer_render_slide(CriaDisplay* display, CriaSlide* slide) {
	g_assert(CRIA_IS_DISPLAY(display));
	g_assert(slide != NULL && CRIA_IS_SLIDE(slide));

	cria_renderer_clear_group(display);
	cria_renderer_render_background(display, slide);
	cria_renderer_render_blocks(display, slide);
}

