/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/* this file is part of criawips a gnome presentation application
 *
 * AUTHORS
 *       Sven Herzberg        <herzi@gnome-de.org>
 *
 * Copyright (C) 2004 Sven Herzberg
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <glib-object.h>
#include <glib.h>
#include <gdk/gdk.h>
#include <libgnome/gnome-i18n.h>

#include "debug.h"
#include "slide.h"

struct _CriaSlidePrivate {
	GHashTable	* blocks;

	CriaLayout	* layout;
	CriaTheme	* theme;
};

enum {
	PROP_0,
	PROP_TITLE,
	PROP_LAYOUT,
	PROP_THEME
};

enum {
	SIGNAL,
	N_SIGNALS
};

static	void	cria_slide_get_property	       (GObject		* object,
						guint		  prop_id,
						GValue		* value,
						GParamSpec	* param_spec);
static	void	cria_slide_init		       (CriaSlide	* self);
static	void	cria_slide_set_property        (GObject		* object,
						guint		  prop_id,
						const	GValue	* value,
						GParamSpec	* param_spec);
#if 0
static	guint	cria_slide_signals[N_SIGNALS] = { 0 };

static	void	cria_slide_signal	       (CriaSlide	* template,
						const	gchar	* string);
#endif
/**
 * cria_slide_add_block:
 * @self: ...
 * @block: ...
 *
 * Adds a #SlideBlock to this slide
 */
void
cria_slide_add_block(CriaSlide* self, CriaSlideBlock* block) {
	g_return_if_fail (CRIA_IS_SLIDE (self));
	g_return_if_fail (CRIA_IS_SLIDE_BLOCK (block));
	g_return_if_fail (cria_slide_block_get_title (block) != NULL);
        g_return_if_fail (cria_slide_get_block (self, cria_slide_block_get_title (block)) == NULL);
	/* FIXME: check that the block is defined in the layout */

	g_hash_table_insert (self->priv->blocks,
			     g_strdup (cria_slide_block_get_title (block)),
			     block);
}

/*
 * cria_slide_class_init:
 * @cria_slide_class: ...
 *
 * Initializes @cria_slide_class.
 */
static void
cria_slide_class_init(CriaSlideClass* cria_slide_class) {
	GObjectClass	* g_object_class;

	g_object_class = G_OBJECT_CLASS(cria_slide_class);
#if 0
	/* setting up signal system */
	cria_slide_class->signal = cria_slide_signal;

	cria_slide_signals[SIGNAL] = g_signal_new (
			"signal",
			CRIA_TYPE_SLIDE,
			G_SIGNAL_RUN_LAST,
			G_STRUCT_OFFSET (
				CriaSlideClass,
				signal),
			NULL,
			NULL,
			g_cclosure_marshal_VOID__STRING,
			G_TYPE_NONE,
			0);
#endif
	/* setting up property system */
	g_object_class->set_property = cria_slide_set_property;
	g_object_class->get_property = cria_slide_get_property;

	g_object_class_install_property(g_object_class,
					PROP_TITLE,
					g_param_spec_string("title",
							    "Title",
							    "The title of that slide",
							    _("untitled"),
							    G_PARAM_READWRITE | G_PARAM_CONSTRUCT));
	g_object_class_install_property(g_object_class,
					PROP_LAYOUT,
					g_param_spec_object("layout",
							    "Layout",
							    "The layout associated to this slide",
							    CRIA_TYPE_LAYOUT,
							    G_PARAM_READWRITE | G_PARAM_CONSTRUCT));
	g_object_class_install_property(g_object_class,
					PROP_THEME,
					g_param_spec_object("theme",
							    "Theme",
							    "The theme associated with this slide",
							    CRIA_TYPE_THEME,
							    G_PARAM_READWRITE | G_PARAM_CONSTRUCT));
}

/**
 * cria_slide_get_block:
 * @self: ...
 * @block: ...
 *
 * Get a block...
 *
 * Returns ...
 */
CriaSlideBlock*
cria_slide_get_block(CriaSlide* self, gchar const* block) {
	g_return_val_if_fail (CRIA_IS_SLIDE (self), NULL);

	return (CriaSlideBlock*) g_hash_table_lookup (self->priv->blocks, block);
}

/**
 * cria_slide_get_display_height:
 * @self: ...
 * @screen: ...
 *
 * Returns the height in pixels that the slide should be rendered to
 */
gdouble
cria_slide_get_display_height(CriaSlide* self, GdkScreen* screen) {
	/* FIXME
	 * - calculate a pair of numbers that are integers and match the ratio
	 *   exactly
	 * - return integer values
	 */
	gdouble	screen_width,
		screen_height,
		retval;
	
/*	g_assert(self != NULL && CRIA_IS_SLIDE(self));
*/	g_assert(screen != NULL && GDK_IS_SCREEN(screen));

	screen_width = gdk_screen_get_width(screen);
	screen_height= gdk_screen_get_height(screen);
	retval = 0.0;
	
	if ((screen_width/screen_height) >= (4.0/3.0)) {
		/* screen is too wide */
		retval = screen_height;
	} else {
		/* screen is too high */
		retval = screen_width*3.0/4.0;
	}
	
	g_debug("Slide::getDisplayHeight(): %f", retval);
	return retval;
}

/**
 * cria_slide_get_display_width:
 * @self: ...
 * @screen: ...
 *
 * Returns the width in pixels that the slide should be rendered to
 */
gdouble
cria_slide_get_display_width(CriaSlide* self, GdkScreen* screen) {
	/* FIXME
	 * - calculate a pair of numbers that are integers and match the ratio
	 *   exactly
	 * - return integer values
	 */
	gdouble screen_width,
		screen_height,
		retval;
	
/*	g_assert(self != NULL && CRIA_IS_SLIDE(self));
*/	g_assert(screen != NULL && GDK_IS_SCREEN(screen));

	screen_width = gdk_screen_get_width(screen);
	screen_height= gdk_screen_get_height(screen);
	retval = 0.0;

	if ((screen_width/screen_height) >= (4.0/3.0)) {
		/* screen is too wide */
		retval = screen_height*4.0/3.0;
	} else {
		/* screen is too high */
		retval = screen_width;
	}
	
	g_debug("Slide::getDisplayWidth(): %f", retval);
	return retval;
}

/**
 * cria_slide_get_layout:
 * @self: ...
 *
 * Get ...
 *
 * Returns ...
 */
CriaLayout*
cria_slide_get_layout (CriaSlide* self) {
	g_return_val_if_fail (CRIA_IS_SLIDE(self), NULL);
	
	return self->priv->layout;
}

/**
 * cria_slide_get_theme:
 * @self: ...
 *
 * ...
 *
 * Returns ...
 */
CriaTheme*
cria_slide_get_theme (CriaSlide* self) {
	g_return_val_if_fail (CRIA_IS_SLIDE(self), NULL);
	
	return self->priv->theme;
}

/**
 * cria_slide_get_title:
 * @self: ...
 *
 * ...
 *
 * Returns ...
 */
const char*
cria_slide_get_title(CriaSlide* self) {
	g_return_val_if_fail (CRIA_IS_SLIDE(self), NULL);
	
	return self->title;
}

/*
 * cria_slide_get_property:
 * @object: ...
 * @prop_id: ...
 * @value: ...
 * @param_spec: ...
 *
 * ...
 */
static void
cria_slide_get_property(GObject* object, guint prop_id, GValue* value, GParamSpec* param_spec) {
	CriaSlide	* self;

	self = CRIA_SLIDE(object);

	switch (prop_id) {
	case PROP_TITLE:
		g_value_set_string(value, self->title);
		break;
	case PROP_THEME:
		g_value_set_object(value, self->priv->theme);
		break;
	case PROP_LAYOUT:
		g_value_set_object(value, self->priv->layout);
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, param_spec);
		break;
	}
}

/**
 * cria_slide_get_type:
 *
 * ...
 *
 * Returns ...
 */
GType
cria_slide_get_type(void) {
	static GType	type = 0;

	if (!type) {
		const GTypeInfo info = {
			sizeof (CriaSlideClass),
			NULL,	/* base initializer */
			NULL,	/* base finalizer */
			(GClassInitFunc)cria_slide_class_init,
			NULL,	/* class finalizer */
			NULL,	/* class data */
			sizeof (CriaSlide),
			0,
			(GInstanceInitFunc)cria_slide_init,
			0
		};

		type = g_type_register_static(G_TYPE_OBJECT,
					      "CriaSlide",
					      &info,
					      0);
	}

	return type;
}

/*
 * cria_slide_init:
 * @self: ...
 *
 * ...
 */
static void
cria_slide_init(CriaSlide* self) {
	self->priv = g_new0(CriaSlidePrivate, 1);
	
	self->priv->blocks = g_hash_table_new_full(g_str_hash,
						   g_str_equal,
						   g_free,
						   g_object_unref);
}

/**
 * cria_slide_set_layout:
 * @self: ...
 * @layout: ...
 *
 * ...
 */
void
cria_slide_set_layout(CriaSlide* self, CriaLayout* layout) {
	g_return_if_fail(CRIA_IS_SLIDE(self));

	if (self->priv->layout != NULL) {
		g_object_unref(self->priv->layout);
	}

	if (layout) {
		self->priv->layout = g_object_ref(layout);
	} else {
		self->priv->layout = layout;
	}

	g_object_notify(G_OBJECT(self), "layout");
}

/**
 * cria_slide_set_theme:
 * @self: ...
 * @theme: ...
 *
 * ...
 */
void
cria_slide_set_theme(CriaSlide* self, CriaTheme* theme) {
	g_return_if_fail(CRIA_IS_SLIDE(self));

	if (self->priv->theme != NULL) {
		g_object_unref(self->priv->theme);
	}

	if (theme) {
		self->priv->theme = g_object_ref(theme);
	} else {
		self->priv->theme = theme;
	}

	g_object_notify(G_OBJECT(self), "theme");
}

/**
 * cria_slide_set_title:
 * @self: ...
 * @title: ...
 *
 * ...
 */
void
cria_slide_set_title(CriaSlide* self, const gchar* title) {
	g_return_if_fail (CRIA_IS_SLIDE(self));

	if (self->title != NULL) {
		g_free (self->title);
	}

	g_debug("cria_slide_set_title(%i): \"%s\"", __LINE__, title);
	self->title = g_strdup(title);

	g_object_notify(G_OBJECT(self), "title");
}

/*
 * cria_slide_set_property:
 * @object: ...
 * @prop_id: ...
 * @value: ...
 * @param_spec: ...
 *
 * ...
 */
static void
cria_slide_set_property(GObject* object, guint prop_id, const GValue* value, GParamSpec* param_spec) {
	CriaSlide	* self;
	
	self = CRIA_SLIDE (object);
	
	switch (prop_id) {
	case PROP_TITLE:
		cria_slide_set_title(self, g_value_get_string (value));
		break;
	case PROP_THEME:
		cria_slide_set_theme(self, g_value_get_object (value));
		break;
	case PROP_LAYOUT:
		cria_slide_set_layout(self, g_value_get_object (value));
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, param_spec);
		break;
	}
}

