/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/* this file is part of criawips a gnome presentation application
 *
 * AUTHORS
 *       Sven Herzberg        <herzi@gnome-de.org>
 *
 * Copyright (C) 2004 Sven Herzberg
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <inttypes.h>

#include <string.h>
#include <glib-object.h>
#include <glib.h>
#include <libgnome/gnome-i18n.h>
#include <gsf/gsf.h>
#include <gsf-gnome/gsf-input-gnomevfs.h>
#include <gsf/gsf-input-stdio.h>

#include "debug.h"
#include "layout.h"
#include "layout-block.h"
#include "presentation.h"
#include "presentation-parser.h"
#include "theme.h"

struct _CriaPresentationPrivate {
	/*GList		* master_slides;*/
	GList		* slides;
	GHashTable	* themes;
};

enum {
	PROP_0,
	PROP_TITLE,
};

enum {
	SIGNAL,
	N_SIGNALS
};

static	void		  cria_presentation_get_property       (GObject		* object,
								guint		  prop_id,
								GValue		* value,
								GParamSpec	* param_spec);
static	CriaPresentation* cria_presentation_new_from_input     (GsfInput	* input,
								GError		**error);
static	void		  cria_presentation_init	       (CriaPresentation* self);
static	void		  cria_presentation_set_property       (GObject		* object,
								guint		  prop_id,
								const	GValue	* value,
								GParamSpec	* param_spec);
#if 0
static	guint		  cria_presentation_signals[N_SIGNALS] = { 0 };

static	void		  cria_presentation_signal	       (CriaPresentation	* template,
								const	gchar	* string);
#endif
/**
 * cria_presentation_add_theme:
 * @self: a CriaPresentation
 * @theme: a CriaTheme
 *
 * Adds the @theme to presentation references by @self.
 */
void
cria_presentation_add_theme (
		CriaPresentation* self,
		CriaTheme	* theme)
{
	g_return_if_fail (CRIA_IS_PRESENTATION (self));
	g_return_if_fail (CRIA_IS_THEME (theme));
	g_return_if_fail (g_hash_table_lookup (self->priv->themes, cria_theme_get_name (theme)) == NULL);

	g_hash_table_insert (
			self->priv->themes,
			g_strdup (cria_theme_get_name (theme)),
			theme);
}

void
cria_presentation_append_slide (
		CriaPresentation* self,
		CriaSlide	* slide)
{
	g_return_if_fail (CRIA_IS_PRESENTATION (self));
	g_return_if_fail (CRIA_IS_SLIDE (slide));

	g_debug ("cria_presentation_append_slide(): appending slide \"%s\" with index %i (dev-count)", cria_slide_get_title (slide), g_list_length (self->priv->slides));
	
	self->priv->slides = g_list_append (
			self->priv->slides,
			slide);
}

static void
cria_presentation_class_init (CriaPresentationClass	* cria_presentation_class)
{
	GObjectClass	* g_object_class;

	g_object_class = G_OBJECT_CLASS(cria_presentation_class);
#if 0
	/* setting up signal system */
	cria_presentation_class->signal = cria_presentation_signal;

	cria_presentation_signals[SIGNAL] = g_signal_new (
			"signal",
			CRIA_TYPE_PRESENTATION,
			G_SIGNAL_RUN_LAST,
			G_STRUCT_OFFSET (
				CriaPresentationClass,
				signal),
			NULL,
			NULL,
			g_cclosure_marshal_VOID__STRING,
			G_TYPE_NONE,
			0);
#endif
	/* setting up property system */
	g_object_class->set_property = cria_presentation_set_property;
	g_object_class->get_property = cria_presentation_get_property;

	g_object_class_install_property(
			g_object_class,
			PROP_TITLE,
			g_param_spec_string(
				"title",
				"Title",
				"The title of the presentation",
				_("untitled"),
				G_PARAM_READWRITE | G_PARAM_CONSTRUCT)
			);
}

static void
cria_presentation_get_property (
		GObject		* object,
		guint		  prop_id,
		GValue		* value,
		GParamSpec	* param_spec)
{
	CriaPresentation	* self;

	self = CRIA_PRESENTATION (object);

	switch (prop_id)
	{
	case PROP_TITLE:
		g_value_set_string(
				value,
				self->title);
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (
				object,
				prop_id,
				param_spec);
		break;
	}
}

CriaSlide*
cria_presentation_get_slide (
		CriaPresentation* self,
		guint		  slide)
{
	CriaSlide	* retval;
	
	g_return_val_if_fail (CRIA_IS_PRESENTATION (self), NULL);
	g_return_val_if_fail (slide < cria_presentation_n_slides (self), NULL);
	
	retval = CRIA_SLIDE (g_list_nth (self->priv->slides, slide)->data);

	return retval;
}

/**
 * cria_presentation_get_slide_index:
 * @self: The presentation to search in
 * @slide: The slide to be found
 *
 * Get the index of a slide in a presentation.
 *
 * Returns the index of the slide if found, -1 otherwise
 */
gint
cria_presentation_get_slide_index(CriaPresentation* self, CriaSlide const* slide) {
	g_assert(self != NULL && CRIA_IS_PRESENTATION(self));
	g_assert(slide != NULL && CRIA_IS_SLIDE(slide));
	g_assert(self->priv != NULL);
	
	return g_list_index(self->priv->slides, slide);
}

CriaTheme*
cria_presentation_get_theme (
		CriaPresentation* self,
		const	char	* theme)
{
	g_return_val_if_fail (CRIA_IS_PRESENTATION (self), NULL);

	return CRIA_THEME (g_hash_table_lookup (self->priv->themes, theme));
}

const char*
cria_presentation_get_title (
		CriaPresentation* self)
{
	g_return_val_if_fail (CRIA_IS_PRESENTATION(self), NULL);
	
	return self->title;
}

GType
cria_presentation_get_type (void)
{
	static GType	type = 0;

	if (!type)
	{
		const GTypeInfo info = {
			sizeof (CriaPresentationClass),
			NULL,	/* base initializer */
			NULL,	/* base finalizer */
			(GClassInitFunc)cria_presentation_class_init,
			NULL,	/* class finalizer */
			NULL,	/* class data */
			sizeof (CriaPresentation),
			0,
			(GInstanceInitFunc)cria_presentation_init,
			0
		};

		type = g_type_register_static (
				G_TYPE_OBJECT,
				"CriaPresentation",
				&info,
				0);
	}

	return type;
}

static void
cria_presentation_init (CriaPresentation *self)
{
	self->priv = g_new0 (CriaPresentationPrivate, 1);

	self->priv->themes = g_hash_table_new_full (
			g_str_hash,
			g_str_equal,
			g_free,
			g_object_unref);
}

CriaPresentation*
cria_presentation_new (void)
{
	return g_object_new (CRIA_TYPE_PRESENTATION, NULL);
}

guint
cria_presentation_n_slides (CriaPresentation        * self)
{
	g_return_val_if_fail (CRIA_IS_PRESENTATION (self), 0);
	
	return g_list_length (self->priv->slides);
}

CriaPresentation*
cria_presentation_new_from_file (const	gchar	* filename, GError**errloc) {
	CriaPresentation* presentation;
	GError		* error = NULL;
	GsfInput	* input = NULL;

	g_return_val_if_fail (filename != NULL, NULL);
	g_return_val_if_fail (strlen (filename) > 0, NULL);
	
	g_debug ("cria_presentation_populate_from_file(%i): start", __LINE__);

	input = gsf_input_stdio_new(filename, &error);

	if (error) {
		*errloc = error;
		return NULL;
	}

	presentation = cria_presentation_new_from_input (input, &error);
	
	if (error) {
		*errloc = error;
		return NULL;
	}

	return presentation;
}

static CriaPresentation*
cria_presentation_new_from_input (
		GsfInput	* input,
		GError		**errloc) {
	CriaPresentation* presentation;
	GError		* error = NULL;
	
	g_debug ("cria_presentation_populate_from_input(%i): start", __LINE__);

	/* TODO add mime type detection */
	/* TODO extract the file reading into a plugin (refer to gnumeric's code
	 * for examples) */
	
	presentation = cria_presentation_parse_input_xml (input, &error);
	
	if (error) {
		*errloc = error;
	}

	return presentation;
}

CriaPresentation*
cria_presentation_new_from_uri(const gchar* text_uri, GError**errloc) {
	CriaPresentation* presentation;
	GError		* error = NULL;
	GsfInput	* input = NULL;

	g_return_val_if_fail (text_uri != NULL, NULL);
	g_return_val_if_fail (strlen (text_uri) > 0, NULL);
	
	input = GSF_INPUT (
			gsf_input_gnomevfs_new (
				text_uri,
				&error)
			);

	if (error) {
		*errloc = error;
		return NULL;
	}

	presentation = cria_presentation_new_from_input (input, &error);

	if (error) {
		*errloc = error;
		return NULL;
	}
	
	return presentation;

}

static void
cria_presentation_set_property (
		GObject		* object,
		guint		  prop_id,
		const	GValue	* value,
		GParamSpec	* param_spec)
{
	CriaPresentation	* self;
	
	self = CRIA_PRESENTATION (object);
	
	switch (prop_id)
	{
	case PROP_TITLE:
		cria_presentation_set_title (
				self,
				g_value_get_string (value));
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (
				object,
				prop_id,
				param_spec);
		break;
	}
}

void
cria_presentation_set_title (
		CriaPresentation* self,
		const gchar	* title)
{
	g_return_if_fail (CRIA_IS_PRESENTATION(self));

	if (self->title != NULL)
		g_free (self->title);

	self->title = g_strdup (title);

	g_object_notify (G_OBJECT (self), "title");
}

void
cria_presentation_add_master_slide(
		CriaPresentation* self,
		CriaSlide	* master_slide) {
#warning "Presentation::addMasterSlide(): implement me"
}

