/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/* this file is part of criawips a gnome presentation application
 *
 * AUTHORS
 *       Sven Herzberg        <herzi@gnome-de.org>
 *
 * Copyright (C) 2004 Sven Herzberg
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <string.h>
#include <glib-object.h>
#include <glib.h>
#include <libgnome/gnome-i18n.h>

#include "debug.h"
#include "layout.h"

struct _CriaLayoutPrivate
{
	GHashTable	* blocks;
};

enum
{
	PROP_0,
	PROP_NAME,
	PROP_DESCRIPTION,
	PROP_BACKGROUND
};

enum
{
	SIGNAL,
	N_SIGNALS
};

static	guint	cria_layout_get_n_blocks       (CriaLayout	* self);
static	void	cria_layout_get_property       (GObject		* object,
						guint		  prop_id,
						GValue		* value,
						GParamSpec	* param_spec);
static	void	cria_layout_init	       (CriaLayout	* layout);
static	void	cria_layout_set_property       (GObject		* object,
						guint		  prop_id,
						const	GValue	* value,
						GParamSpec	* param_spec);
#if 0
static	guint	cria_layout_signals[N_SIGNALS] = { 0 };

static	void	cria_layout_signal	       (CriaLayout	* template,
						const	gchar	* string);
#endif
void
cria_layout_add_block (
		CriaLayout	* self,
		CriaLayoutBlock	* block)
{
	g_return_if_fail (CRIA_IS_LAYOUT (self));
	g_return_if_fail (CRIA_IS_LAYOUT_BLOCK (block));
	g_return_if_fail (cria_layout_block_get_title (block) != NULL);
	g_return_if_fail (cria_layout_get_block (self, cria_layout_block_get_title (block)) == NULL);

	g_hash_table_insert (
			self->priv->blocks,
			g_strdup (cria_layout_block_get_title (block)),
			block);
}

static void
cria_layout_class_init (CriaLayoutClass	* cria_layout_class)
{
	GObjectClass	* g_object_class;

	g_object_class = G_OBJECT_CLASS(cria_layout_class);
#if 0
	/* setting up signal system */
	cria_layout_class->signal = cria_layout_signal;

	cria_layout_signals[SIGNAL] = g_signal_new (
			"signal",
			CRIA_TYPE_LAYOUT,
			G_SIGNAL_RUN_LAST,
			G_STRUCT_OFFSET (
				CriaLayoutClass,
				signal),
			NULL,
			NULL,
			g_cclosure_marshal_VOID__STRING,
			G_TYPE_NONE,
			0);
#endif
	/* setting up property system */
	g_object_class->set_property = cria_layout_set_property;
	g_object_class->get_property = cria_layout_get_property;

	g_object_class_install_property (
			g_object_class,
			PROP_NAME,
			g_param_spec_string (
				"name",
				"Name",
				"The name of this layout (user readable)",
				_("untitled"),
				G_PARAM_READWRITE | G_PARAM_CONSTRUCT)
			);
	g_object_class_install_property (
			g_object_class,
			PROP_DESCRIPTION,
			g_param_spec_string (
				"description",
				"Description",
				"The description of this layout (user readable)",
				_("No description"),
				G_PARAM_READWRITE | G_PARAM_CONSTRUCT)
			);
	g_object_class_install_property (
			g_object_class,
			PROP_BACKGROUND,
			g_param_spec_object (
				"background",
				"Background",
				"The background of slides with this layout",
				GDK_TYPE_PIXBUF,
				G_PARAM_READWRITE | G_PARAM_CONSTRUCT)
			);
}

GdkPixbuf *
cria_layout_get_background (
		CriaLayout	* self)
{
	g_return_val_if_fail (CRIA_IS_LAYOUT (self), NULL);

	return self->background;
}

CriaLayoutBlock *
cria_layout_get_block (
		CriaLayout	* self,
		char const	* title)
{
	g_return_val_if_fail (CRIA_IS_LAYOUT (self), NULL);
	g_return_val_if_fail (title != NULL, NULL);
	g_return_val_if_fail (strlen (title) > 0, NULL);

	return (CriaLayoutBlock*) g_hash_table_lookup (self->priv->blocks, title);
}

static void
populate_array_with_keys (
		gpointer	  key,
		gpointer	  value,
		gpointer	  user_data)
{
	char	**array = (char**)user_data;
	char	* skey = (char*)key; 
	char	**iterator;

	if (!array)
		return;

	for (iterator = array; *iterator; iterator++)
		; /* stepping to the first NULL */

	*iterator = g_strdup (skey);
}

gchar**
cria_layout_get_block_names (
		CriaLayout	* self)
{
	gchar	**retval;

	/* think of the leading NULL */
	retval = g_new0 (gchar*, cria_layout_get_n_blocks (self) + 1);

	g_hash_table_foreach (self->priv->blocks,
			populate_array_with_keys,
			retval);
	
	return retval;
}

const char*
cria_layout_get_name (
		CriaLayout	* self)
{
	g_return_val_if_fail (CRIA_IS_LAYOUT(self), NULL);
	
	return self->name;
}

static guint
cria_layout_get_n_blocks (
		CriaLayout	* self)
{
	return g_hash_table_size (self->priv->blocks);
}

static void
cria_layout_get_property (
		GObject		* object,
		guint		  prop_id,
		GValue		* value,
		GParamSpec	* param_spec)
{
	CriaLayout	* self;

	self = CRIA_LAYOUT (object);

	switch (prop_id)
	{
	case PROP_NAME:
		g_value_set_string (
				value,
				self->name);
		break;
	case PROP_DESCRIPTION:
		g_value_set_string (
				value,
				self->description);
		break;
	case PROP_BACKGROUND:
		g_value_set_object (
				value,
				self->background);
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (
				object,
				prop_id,
				param_spec);
		break;
	}
}

GType
cria_layout_get_type (void)
{
	static GType	type = 0;

	if (!type)
	{
		const GTypeInfo info = {
			sizeof (CriaLayoutClass),
			NULL,	/* base initializer */
			NULL,	/* base finalizer */
			(GClassInitFunc)cria_layout_class_init,
			NULL,	/* class finalizer */
			NULL,	/* class data */
			sizeof (CriaLayout),
			0,
			(GInstanceInitFunc)cria_layout_init,
			0
		};

		type = g_type_register_static (
				G_TYPE_OBJECT,
				"CriaLayout",
				&info,
				0);
	}

	return type;
}

void
cria_layout_init (CriaLayout * self)
{
	self->priv = g_new0 (CriaLayoutPrivate, 1);

	self->priv->blocks = g_hash_table_new_full (
			g_str_hash,
			g_str_equal,
			g_free,
			g_object_unref);
}

void
cria_layout_set_background (
		CriaLayout	* self,
		GdkPixbuf	* background)
{
	g_return_if_fail (CRIA_IS_LAYOUT (self));

	if (self->background)
		g_object_unref (self->background);

	self->background = background;

	g_object_notify (G_OBJECT (self), "background");
}

void
cria_layout_set_description (
		CriaLayout	* self,
		const gchar	* description)
{
	g_return_if_fail (CRIA_IS_LAYOUT (self));

	g_debug ("cria_layout_set_description(): start");

	if (self->description != NULL)
		g_free (self->description);

	self->description = g_strdup (description);

	g_object_notify (G_OBJECT (self), "description");

	g_debug ("cria_layout_set_description(): end");
}

void
cria_layout_set_name (
		CriaLayout	* self,
		const gchar	* name)
{
	g_return_if_fail (CRIA_IS_LAYOUT (self));

	if (self->name != NULL)
		g_free (self->name);

	self->name = g_strdup (name);

	g_object_notify (G_OBJECT (self), "name");
}

static void
cria_layout_set_property (
		GObject		* object,
		guint		  prop_id,
		const	GValue	* value,
		GParamSpec	* param_spec)
{
	CriaLayout	* self;
	
	self = CRIA_LAYOUT (object);
	
	switch (prop_id)
	{
	case PROP_NAME:
		cria_layout_set_name (
				self,
				g_value_get_string (value));
		break;
	case PROP_DESCRIPTION:
		cria_layout_set_description (
				self,
				g_value_get_string (value));
		break;
	case PROP_BACKGROUND:
		cria_layout_set_background (
				self,
				g_value_get_object (value));
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (
				object,
				prop_id,
				param_spec);
		break;
	}
}

