/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/* this file is part of criawips a gnome presentation application
 *
 * AUTHORS
 *       Sven Herzberg        <herzi@gnome-de.org>
 *
 * Copyright (C) 2004 Sven Herzberg
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <glib.h>
#include <glib-object.h>
#include <glib/gi18n.h>
#include <libgnomecanvas/libgnomecanvas.h>

#include "slide-view.h"

enum {
	PROP_0,
	PROP_SLIDE
};

enum {
	SIGNAL_SLIDE_CHANGED,
	N_SIGNALS
};

struct _CriaSlideViewPriv {
	GtkWidget	* canvas;
	GnomeCanvasGroup* group;
	CriaSlide	* slide;
};

static	void	cria_slide_view_get_property	       (GObject		* object,
						guint		  prop_id,
						GValue		* value,
						GParamSpec	* param_spec);
static	void	cria_slide_view_init	       (CriaSlideView	* self);
static	void	cria_slide_view_set_property        (GObject		* object,
						guint		  prop_id,
						const	GValue	* value,
						GParamSpec	* param_spec);

static	guint	cria_slide_view_signals[N_SIGNALS] = { 0 };

static	void	cria_slide_view_slide_changed  (CriaSlideView	* template,
						gpointer	  data);

static void
cria_slide_view_class_init (CriaSlideViewClass	* cria_slide_view_class)
{
	GObjectClass	* g_object_class;

	g_object_class = G_OBJECT_CLASS(cria_slide_view_class);

	/* setting up signal system */
	cria_slide_view_class->slide_changed = cria_slide_view_slide_changed;

	cria_slide_view_signals[SIGNAL_SLIDE_CHANGED] = g_signal_new (
			"slide-changed",
			CRIA_TYPE_SLIDE_VIEW,
			G_SIGNAL_RUN_LAST,
			G_STRUCT_OFFSET (
				CriaSlideViewClass,
				slide_changed),
			NULL,
			NULL,
			g_cclosure_marshal_VOID__VOID,
			G_TYPE_NONE,
			0);

	/* setting up property system */
	g_object_class->set_property = cria_slide_view_set_property;
	g_object_class->get_property = cria_slide_view_get_property;

	g_object_class_install_property (
			g_object_class,
			PROP_SLIDE,
			g_param_spec_object (
				"slide",
				"Slide",
				"The slide that's currently displayed in this view",
				CRIA_TYPE_SLIDE,
				G_PARAM_READWRITE | G_PARAM_CONSTRUCT)
			);
}

CriaSlide*
cria_slide_view_get_slide (CriaSlideView* self) {
	g_return_val_if_fail (CRIA_IS_SLIDE_VIEW(self), NULL);
	
	return self->priv->slide;
}

static void
cria_slide_view_get_property (GObject* object, guint prop_id, GValue* value, GParamSpec* param_spec) {
	CriaSlideView	* self;

	self = CRIA_SLIDE_VIEW (object);

	switch (prop_id) {
	case PROP_SLIDE:
		g_value_set_object (
				value,
				self->priv->slide);
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (
				object,
				prop_id,
				param_spec);
		break;
	}
}

GType
cria_slide_view_get_type (void) {
	static GType	type = 0;

	if (!type) {
		const GTypeInfo info = {
			sizeof (CriaSlideViewClass),
			NULL,	/* base initializer */
			NULL,	/* base finalizer */
			(GClassInitFunc)cria_slide_view_class_init,
			NULL,	/* class finalizer */
			NULL,	/* class data */
			sizeof (CriaSlideView),
			0,
			(GInstanceInitFunc)cria_slide_view_init,
			0
		};

		type = g_type_register_static (
				GTK_TYPE_FRAME,
				"CriaSlideView",
				&info,
				0);
	}

	return type;
}

static void
cria_slide_view_init(CriaSlideView   * self) {
	self->priv = g_new0(CriaSlideViewPriv,1);

	self->priv->canvas = gnome_canvas_new_aa ();
	gtk_widget_show(self->priv->canvas);

	gtk_container_add(
			GTK_CONTAINER(self),
			self->priv->canvas);
	
	gtk_frame_set_shadow_type(
			GTK_FRAME(self),
			GTK_SHADOW_IN);
}

GtkWidget*
cria_slide_view_new(void) {
	return g_object_new(CRIA_TYPE_SLIDE_VIEW, NULL);
}

void
cria_slide_view_set_slide (CriaSlideView* self, CriaSlide* slide) {
	g_assert(CRIA_IS_SLIDE_VIEW(self));
	g_assert(slide == NULL || CRIA_IS_SLIDE(slide));

	if (self->priv->slide != NULL) {
		g_object_unref (self->priv->slide);
	}

	if (slide) {
		self->priv->slide = g_object_ref (slide);
	} else {
		self->priv->slide = NULL;
	}

	g_signal_emit_by_name(self,
			      "slide-changed");
	g_object_notify (G_OBJECT (self), "slide");
}

static void
cria_slide_view_set_property (
		GObject		* object,
		guint		  prop_id,
		const	GValue	* value,
		GParamSpec	* param_spec)
{
	CriaSlideView	* self;
	
	self = CRIA_SLIDE_VIEW (object);
	
	switch (prop_id)
	{
	case PROP_SLIDE:
		cria_slide_view_set_slide (
				self,
				g_value_get_object (value));
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (
				object,
				prop_id,
				param_spec);
		break;
	}
}

static void
cria_slide_view_slide_changed(CriaSlideView* self, gpointer data) {
	/* clear the current canvas */
	if (self->priv->group) {
		g_assert(GNOME_IS_CANVAS_GROUP(self->priv->group));
		gnome_canvas_item_hide(GNOME_CANVAS_ITEM(self->priv->group));
		g_object_unref(self->priv->group);
	}

	self->priv->group = GNOME_CANVAS_GROUP(
			gnome_canvas_item_new(
				gnome_canvas_root(GNOME_CANVAS(self->priv->canvas)),
				gnome_canvas_group_get_type(),
				"x", 0,
				"y", 0,
				NULL)
			);

	if (self->priv->slide) {
		// FIXME render slide
	} else {
		GnomeCanvasItem	* text = gnome_canvas_item_new(self->priv->group,
							       gnome_canvas_text_get_type(),
							       "anchor", GTK_ANCHOR_NORTH_WEST,
							       "text", _("No slide selected"),
							       "fill-color", "black", // FIXME read this color from the gtk theme
							       "x", 0, // FIXME fix the position
							       "y", 0, // FIXME fix the position
							       NULL);
		gdouble		x, y, width, height;

		gnome_canvas_item_get_bounds(
				text,
				&x, &y,
				&width, &height);
		gnome_canvas_set_scroll_region(
				self->priv->canvas,
				x, y,
				width, height);
		gnome_canvas_item_show(text);
	}
}

