/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/* this file is part of criawips a gnome presentation application
 *
 * AUTHORS
 *       Sven Herzberg        <herzi@gnome-de.org>
 *
 * Copyright (C) 2004 Sven Herzberg
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <string.h>

#include <glib.h>
#include <glib-object.h>
#include <glib/gi18n.h>
#include <gtk/gtk.h>
#include <libgnomeui/gnome-about.h>

#include "debug.h"
#include "application.h"
#include "main-window.h"
#include "presentation.h"
#include "slide-list.h"
#include "slide-show.h"

enum {
	PROP_0,
	PROP_PRESENTATION
};

enum {
	SIGNAL,
	N_SIGNALS
};

struct _CriaMainWindowPriv {
	GladeXML	* xml;
	CriaPresentation* presentation;
};

static	void	cria_main_window_get_property  (GObject		* object,
						guint		  prop_id,
						GValue		* value,
						GParamSpec	* param_spec);
static	void	cria_main_window_init	       (CriaMainWindow	* self);
static	void	cria_main_window_set_property  (GObject		* object,
						guint		  prop_id,
						const	GValue	* value,
						GParamSpec	* param_spec);

static	guint	cria_main_window_signals[N_SIGNALS] = { 0 };

static	void	cria_main_window_signal	       (CriaMainWindow	* template,
						const	gchar	* string);
static	void	cria_main_window_open	       (CriaMainWindow	* self,
						gpointer	  data);
static	void	cria_main_window_start_off     (CriaMainWindow	* self);

static void
cria_main_window_class_init (CriaMainWindowClass	* cria_main_window_class)
{
	GObjectClass	* g_object_class;

	g_object_class = G_OBJECT_CLASS(cria_main_window_class);
#if 0
	/* setting up signal system */
	cria_main_window_class->signal = cria_main_window_signal;

	cria_main_window_signals[SIGNAL] = g_signal_new (
			"signal",
			CRIA_TYPE_MAIN_WINDOW,
			G_SIGNAL_RUN_LAST,
			G_STRUCT_OFFSET (
				CriaMainWindowClass,
				signal),
			NULL,
			NULL,
			g_cclosure_marshal_VOID__STRING,
			G_TYPE_NONE,
			0);
#endif
	/* setting up property system */
	g_object_class->set_property = cria_main_window_set_property;
	g_object_class->get_property = cria_main_window_get_property;

	g_object_class_install_property (
			g_object_class,
			PROP_PRESENTATION,
			g_param_spec_object (
				"presentation",
				"Presentation",
				"This is the presentation that's currently assotiated with this main window",
				CRIA_TYPE_PRESENTATION,
				G_PARAM_READWRITE | G_PARAM_CONSTRUCT)
			);
}

CriaMainWindow*
cria_main_window_new(const gchar* filename) {
	CriaMainWindow	* self;

	g_debug("cria_main_window_new(): start");
	
	self = g_object_new(CRIA_TYPE_MAIN_WINDOW, NULL);
	
	if (filename != NULL) {
		CriaPresentation	* p;
		GError			* error = NULL;

		g_debug("cria_main_window_new(): got filename 0x%x(\"%s\")", (int)filename, filename);
		
		p = cria_presentation_new_from_file(filename, &error);

		if (error) {
			GtkWidget	* dialog = glade_xml_get_widget(self->priv->xml, "error_dialog");
			char            * error_message;
			
			error_message = g_strdup_printf(
					"<span weight=\"bold\">The Presentation could not be opened</span>\n\n"
					"For some reason the presentation you wanted to open could not be opened. "
					"The reason was:\n<span style=\"italic\">%s</span>.", error->message);

			gtk_label_set_markup(
					GTK_LABEL(glade_xml_get_widget(self->priv->xml, "error_label_message")),
					error_message);
			g_signal_connect_swapped(
					glade_xml_get_widget(self->priv->xml, "error_button_close"),
					"clicked",
					G_CALLBACK(gtk_widget_hide),
					dialog);
			gtk_dialog_run(GTK_DIALOG(dialog));
			
			g_free(error_message);
			g_error_free(error);
		} else {
			g_debug ("cria_main_window_new(): setting presentation");
		
			g_object_set(G_OBJECT(self), "presentation", p, NULL);

			g_debug ("cria_main_window_new(): done");
		}
	} else {
		g_debug("cria_main_window_new(): filename was NULL, we're done");
	}
	
	g_debug("cria_main_window_new(): in between");
	
	cria_application_register_window(self);

	g_debug("cria_main_window_new(): end");
	return self;
}

CriaPresentation*
cria_main_window_get_presentation (
		CriaMainWindow	* self)
{
	g_return_val_if_fail (CRIA_IS_MAIN_WINDOW(self), NULL);
	
	return self->priv->presentation;
}

static void
cria_main_window_get_property (
		GObject		* object,
		guint		  prop_id,
		GValue		* value,
		GParamSpec	* param_spec)
{
	CriaMainWindow	* self;

	self = CRIA_MAIN_WINDOW (object);

	switch (prop_id)
	{
	case PROP_PRESENTATION:
		g_value_set_object (
				value,
				self->priv->presentation);
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (
				object,
				prop_id,
				param_spec);
		break;
	}
}

GType
cria_main_window_get_type (void)
{
	static GType	type = 0;

	if (!type) {
		const GTypeInfo info = {
			sizeof (CriaMainWindowClass),
			NULL,	/* base initializer */
			NULL,	/* base finalizer */
			(GClassInitFunc)cria_main_window_class_init,
			NULL,	/* class finalizer */
			NULL,	/* class data */
			sizeof (CriaMainWindow),
			0,
			(GInstanceInitFunc)cria_main_window_init,
			0
		};

		type = g_type_register_static (
				G_TYPE_OBJECT,
				"CriaMainWindow",
				&info,
				0);
	}

	return type;
}

static void
cria_main_window_init(CriaMainWindow* self) {
	
	g_debug("cria_main_window_init(): start");
	
	/* creating the private data */
	self->priv = g_new0(CriaMainWindowPriv,1);
	// FIXME add some nice warning if this fails
	self->priv->xml = glade_xml_new(PACKAGE_DATA_DIR "/" PACKAGE "/data/criawips.glade", NULL, NULL);

	/* connecting signals */
	  /* window */
	g_signal_connect(glade_xml_get_widget(self->priv->xml, "main_window"),
			"delete-event",
			G_CALLBACK(criawips_quit),
			NULL);
	  /* menus */
	gtk_widget_set_sensitive(
			glade_xml_get_widget(self->priv->xml, "main_menu_file_new"),
			FALSE);
	g_signal_connect_swapped(glade_xml_get_widget(self->priv->xml, "main_menu_file_open"),
			"activate",
			G_CALLBACK(cria_main_window_open),
			self);
	gtk_widget_set_sensitive(
			glade_xml_get_widget(self->priv->xml, "main_menu_file_save"),
			FALSE);
	gtk_widget_set_sensitive(
			glade_xml_get_widget(self->priv->xml, "main_menu_file_save_as"),
			FALSE);
	gtk_widget_set_sensitive(
			glade_xml_get_widget(self->priv->xml, "main_menu_file_properties"),
			FALSE);
	gtk_widget_set_sensitive(
			glade_xml_get_widget(self->priv->xml, "main_menu_file_close"),
			FALSE);
	g_signal_connect(glade_xml_get_widget(self->priv->xml, "main_menu_file_quit"),
			"activate",
			G_CALLBACK(criawips_quit),
			NULL);

	gtk_widget_set_sensitive(
			glade_xml_get_widget(self->priv->xml, "main_menu_edit_cut"),
			FALSE);
	gtk_widget_set_sensitive(
			glade_xml_get_widget(self->priv->xml, "main_menu_edit_copy"),
			FALSE);
	gtk_widget_set_sensitive(
			glade_xml_get_widget(self->priv->xml, "main_menu_edit_paste"),
			FALSE);
	gtk_widget_set_sensitive(
			glade_xml_get_widget(self->priv->xml, "main_menu_edit_clear"),
			FALSE);
	gtk_widget_set_sensitive(
			glade_xml_get_widget(self->priv->xml, "main_menu_edit_preferences"),
			FALSE);
	
	g_signal_connect_swapped(glade_xml_get_widget(self->priv->xml, "main_menu_view_start_off"),
			"activate",
			G_CALLBACK(cria_main_window_start_off),
			self);

	g_signal_connect_swapped(
			glade_xml_get_widget(self->priv->xml, "main_menu_help_about"),
			"activate",
			G_CALLBACK(cria_main_window_show_about),
			self);
	  /* toolbars */
	g_signal_connect_swapped(
			glade_xml_get_widget(self->priv->xml, "main_tool_button_open"),
			"clicked",
			G_CALLBACK(cria_main_window_open),
			self);
	g_signal_connect_swapped(
			glade_xml_get_widget(self->priv->xml, "main_tool_button_start_off"),
			"clicked",
			G_CALLBACK(cria_main_window_start_off),
			self);

	/* display the window */
	gtk_widget_show(glade_xml_get_widget(self->priv->xml, "main_window"));

	g_debug("cria_main_window_init(): end");
}

static void
cria_main_window_open(CriaMainWindow* self, gpointer d) {
	// FIXME implement code for opening more than one file at a moment
	GtkWidget	* dialog = gtk_file_chooser_dialog_new (
			"Open File",
			NULL,
			GTK_FILE_CHOOSER_ACTION_OPEN,
			GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
			GTK_STOCK_OPEN, GTK_RESPONSE_ACCEPT,
			NULL);

	if (gtk_dialog_run(GTK_DIALOG(dialog))) {
		gtk_widget_hide(dialog);

		while (gtk_events_pending()) {
			gtk_main_iteration();
		}
		
		GSList			* uris	   = gtk_file_chooser_get_uris(GTK_FILE_CHOOSER(dialog));

		if (uris && g_slist_length(uris) == 1) {
			GError	* error = NULL;
			
			cria_main_window_set_presentation(self, cria_presentation_new_from_uri((gchar*)uris->data, &error));

			if (error) {
				// FIXME refactor this smart, this code currently appears three times in the source tree
				GtkWidget	* dialog = glade_xml_get_widget(self->priv->xml, "error_dialog");
				char            * error_message;
				
				error_message = g_strdup_printf(
						"<span weight=\"bold\">The Presentation could not be opened</span>\n\n"
						"For some reason the presentation you wanted to open could not be opened. "
						"The reason was:\n<span style=\"italic\">%s</span>.", error->message);

				gtk_label_set_markup(
						GTK_LABEL(glade_xml_get_widget(self->priv->xml, "error_label_message")),
						error_message);
				g_signal_connect_swapped(
						glade_xml_get_widget(self->priv->xml, "error_button_close"),
						"clicked",
						G_CALLBACK(gtk_widget_hide),
						dialog);
				gtk_dialog_run(GTK_DIALOG(dialog));
				
				g_free(error_message);
				g_error_free(error);
			}
		} else {
			// FIXME implement code for opening more than one file at a moment
		}
	}

	gtk_object_destroy(GTK_OBJECT(dialog));
}

void
cria_main_window_set_presentation (
		CriaMainWindow	* self,
		CriaPresentation* presentation) {
	g_debug("cria_main_window_set_presentation(): start");
	
	g_return_if_fail (CRIA_IS_MAIN_WINDOW(self));
	g_assert(presentation == NULL || CRIA_IS_PRESENTATION(presentation));

	if (self->priv->presentation != NULL) {
		g_object_unref (self->priv->presentation);
	}

	if (presentation) {
		self->priv->presentation = g_object_ref (presentation);
		cria_slide_list_set_presentation(CRIA_SLIDE_LIST(glade_xml_get_widget(self->priv->xml, "main_slide_list")), cria_main_window_get_presentation(self));
	}

	g_object_notify (G_OBJECT (self), "presentation");

	g_debug("cria_main_window_set_presentation(): end");
}

static void
cria_main_window_set_property (
		GObject		* object,
		guint		  prop_id,
		const	GValue	* value,
		GParamSpec	* param_spec) {
	CriaMainWindow	* self;
	
	self = CRIA_MAIN_WINDOW (object);
	
	switch (prop_id) {
	case PROP_PRESENTATION:
		cria_main_window_set_presentation (
				self,
				g_value_get_object (value));
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (
				object,
				prop_id,
				param_spec);
		break;
	}
}

void
cria_main_window_show_about(CriaMainWindow* self) {
	const gchar	* authors[] = {
		"Sven Herzberg",
		NULL
	};

	GdkPixbuf	* pixbuf = NULL;

	const gchar	* documenters[] = {NULL};

	GtkWidget	* dialog = gnome_about_new(
				PACKAGE,
				VERSION,
				_("Copyright (C) 2004 - The criawips development team"),
				_("Criawips is a presentation application for the GNOME "
				"desktop environment which should integrate smoothly into "
				"the desktop"),
				authors,
				documenters,
				(strcmp("translator-credits", _("translator-credits")) == 0) ? NULL : _("translator-credits"),
				pixbuf);
	
	gtk_window_set_transient_for(
			GTK_WINDOW(dialog),
			GTK_WINDOW(glade_xml_get_widget(self->priv->xml, "main_window")));

	gtk_widget_show(dialog);

	if (pixbuf) {
		g_object_unref(pixbuf);
	}
}

static void
cria_main_window_start_off (CriaMainWindow* self) {
	CriaPresentation	* p;

	g_debug("cria_main_window_start_off(): start");
	
	p = cria_main_window_get_presentation(self);

	if (p != NULL && CRIA_IS_PRESENTATION(p)) {
		gtk_widget_show(cria_slide_show_new(p));
	}

	g_debug("cria_main_window_start_off(): end");
}

