/* this file is part of criawips a gnome presentation application
 *
 * AUTHORS
 *       Sven Herzberg        <herzi@gnome-de.org>
 *
 * Copyright (C) 2004 Sven Herzberg
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <widgets/cria-sidebar-model.h>

#include <inttypes.h>
#include <string.h>

#include <glib.h>
#include <glib-object.h>
#include <gtk/gtktreednd.h>

#include <cdebug/cdebug.h>
#include "presentation-view.h"

enum {
	PROP_0,
};

enum {
	N_SIGNALS
};

enum {
	COLUMN_PREVIEW,
	COLUMN_TITLE,
	COLUMN_SLIDE_PTR,
	N_COLUMNS
};

static GType *column_types = NULL;

#warning "FIXME: extract a private header file"
struct _CriaSidebarModelPrivate {
	CriaPresentation	* presentation;
};

static GObjectClass* parent_class = NULL;
static void     cria_sidebar_model_get_property     (GObject			* object,
							guint			  prop_id,
							GValue			* value,
							GParamSpec		* param_spec);
static void     cria_sidebar_model_init	       (CriaSidebarModel	* self);
static void     cria_sidebar_model_init_tree_model_iface
						       (gpointer		  iface,
							gpointer		  data);
static gboolean cria_sidebar_model_iter_children    (GtkTreeModel		* tree_model,
							GtkTreeIter		* iter,
							GtkTreeIter		* parent);
static gboolean cria_sidebar_model_iter_has_child   (GtkTreeModel		* tree_model,
							GtkTreeIter		* iter);
static gboolean cria_sidebar_model_iter_next	       (GtkTreeModel		* tree_model,
							GtkTreeIter		* iter);
static gint     cria_sidebar_model_iter_n_children  (GtkTreeModel		* tree_model,
							GtkTreeIter		* iter);
static gboolean cria_sidebar_model_iter_nth_child   (GtkTreeModel		* tree_model,
							GtkTreeIter		* iter,
							GtkTreeIter		* parent,
							gint			  n);
static gboolean cria_sidebar_model_iter_parent      (GtkTreeModel		* tree_model,
							GtkTreeIter		* iter,
							GtkTreeIter		* child);
static void     cria_sidebar_model_ref_node         (GtkTreeModel		* tree_model,
							GtkTreeIter		* iter);
static void     cria_sidebar_model_set_property     (GObject			* object,
							guint			  prop_id,
							const GValue		* value,
							GParamSpec		* param_spec);
static void     cria_sidebar_model_unref_node       (GtkTreeModel		* tree_model,
							GtkTreeIter		* iter);
#if 0
/* enable these to add support for signals */
static	guint	cria_sidebar_model_signals[N_SIGNALS] = { 0 };

static	void	cria_sidebar_model_signal	       (CriaSidebarModel	* self,
						const	gchar	* string);
#endif

static void
cslp_finalize(GObject* object) {
	CriaSidebarModel* self = CRIA_SIDEBAR_MODEL(object);

	if(self->priv->presentation) {
		g_object_unref(self->priv->presentation);
		self->priv->presentation = NULL;
	}

	parent_class->finalize(object);
}

static void
cria_sidebar_model_class_init(CriaSidebarModelClass* cria_sidebar_model_class) {
	GObjectClass	* g_object_class;

	parent_class = g_type_class_peek_parent(cria_sidebar_model_class);
#if 0
	/* setting up signal system */
	cria_sidebar_model_class->signal = cria_sidebar_model_signal;

	cria_sidebar_model_signals[SIGNAL] = g_signal_new (
			"signal",
			CRIA_TYPE_SIDEBAR_MODEL,
			G_SIGNAL_RUN_LAST,
			G_STRUCT_OFFSET (
				CriaSidebarModelClass,
				signal),
			NULL,
			NULL,
			g_cclosure_marshal_VOID__STRING,
			G_TYPE_NONE,
			0);
#endif
	/* setting up property system */
	g_object_class = G_OBJECT_CLASS(cria_sidebar_model_class);
	g_object_class->finalize = cslp_finalize;
	g_object_class->get_property = cria_sidebar_model_get_property;
	g_object_class->set_property = cria_sidebar_model_set_property;

	/* setting up the PresentationView interface */
	_cria_presentation_view_install_properties(g_object_class);

	/* setting up the slide list proxy class */
#warning "SlideListProxy::classInit(): FIXME: initialize this a bit more dynamically"
	column_types = g_new0(GType, N_COLUMNS);
	column_types[0] = GDK_TYPE_PIXBUF;
	column_types[1] = G_TYPE_STRING;
	column_types[2] = CRIA_TYPE_SLIDE;
}

/**
 * cria_sidebar_model_for_presentation:
 * @presentation: a #CriaPresentation
 *
 * Get a #CriaSidebarModel for @presentation.
 *
 * Returns a #CriaSidebarModel that provides a #GtkTreeModel for a #CriaPresentation.
 */
CriaSidebarModel*
cria_sidebar_model_for_presentation(CriaPresentation* presentation) {
#warning "SlideBarModel::forPresentation(): FIXME: build a hash tree and do not always create a new proxy"
	return g_object_new(CRIA_TYPE_SIDEBAR_MODEL, "presentation", presentation, NULL);
}

static GType
cria_sidebar_model_get_column_type(GtkTreeModel* tree_model, gint index) {
	g_return_val_if_fail(0 <= index && index < N_COLUMNS, 0);

	return column_types[index];
}

static GtkTreeModelFlags
cria_sidebar_model_get_flags(GtkTreeModel* tree_model) {
	g_return_val_if_fail(CRIA_IS_SIDEBAR_MODEL(tree_model), 0);

#warning "getFlags(): FIXME: check whether we want GTK_TREE_MODEL_ITERS_PERSIST too"
	return GTK_TREE_MODEL_LIST_ONLY;
}

static gboolean
cria_sidebar_model_get_iter(GtkTreeModel* tree_model, GtkTreeIter* iter, GtkTreePath* path) {
	gint		    i;
	CriaSidebarModel* self;
	
	g_return_val_if_fail(CRIA_IS_SIDEBAR_MODEL(tree_model), FALSE);
	g_return_val_if_fail(gtk_tree_path_get_depth(path) > 0, FALSE);

	self = CRIA_SIDEBAR_MODEL(tree_model);

	i = gtk_tree_path_get_indices(path)[0];

	if(i >= cria_slide_list_n_slides(CRIA_SLIDE_LIST(self->priv->presentation))) {
		return FALSE;
	}
	
	iter->user_data = cria_slide_list_get(CRIA_SLIDE_LIST(self->priv->presentation), i);

	if(iter->user_data) {
		return TRUE;
	}

	return FALSE;
}

static gint
cria_sidebar_model_get_n_columns(GtkTreeModel *tree_model) {
	g_return_val_if_fail(CRIA_IS_SIDEBAR_MODEL(tree_model), 0);

	return N_COLUMNS;
}

static GtkTreePath*
cria_sidebar_model_get_path(GtkTreeModel* tree_model, GtkTreeIter* iter) {
	GtkTreePath       * path = NULL;
	CriaSidebarModel* self;
	gint		    index;

	g_assert(CRIA_IS_SIDEBAR_MODEL(tree_model));
	self = CRIA_SIDEBAR_MODEL(tree_model);

	index = cria_slide_list_index(CRIA_SLIDE_LIST(self->priv->presentation), CRIA_SLIDE(iter->user_data));
	g_return_val_if_fail(index >= 0, path);

	path = gtk_tree_path_new();
	gtk_tree_path_append_index(path, index);
	return path;
}

CriaPresentation*
cria_sidebar_model_get_presentation(CriaSidebarModel* self) {
	g_return_val_if_fail(CRIA_IS_SIDEBAR_MODEL(self), NULL);
	
	return self->priv->presentation;
}

static void
cria_sidebar_model_get_property(GObject* object, guint prop_id, GValue* value, GParamSpec* param_spec) {
	CriaSidebarModel	* self;

	self = CRIA_SIDEBAR_MODEL(object);

	switch(prop_id) {
	case CRIA_PRESENTATION_VIEW_PROP_PRESENTATION:
		g_value_set_object(value, self->priv->presentation);
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID(object,
						  prop_id,
						  param_spec);
		break;
	}
}

/**
 * cria_sidebar_model_get_slide:
 * @self: a #CriaSidebarModel
 * @iter: a #GtkTreeIter
 *
 * Get the slide represented by an Iterator.
 *
 * Returns the #CriaSlide that's being referred to by the iterator.
 */
CriaSlide*
cria_sidebar_model_get_slide(CriaSidebarModel* self, GtkTreeIter* iter) {
	return iter->user_data;
}

/**
 * cria_sidebar_model_get_title_column:
 * 
 * Get the column that should be used by a #GtkTreeView to get the title of a
 * slide.
 *
 * Returns the title column index.
 */
gint
cria_sidebar_model_get_title_column(void) {
	return COLUMN_TITLE;
}

static gboolean
cslp_row_draggable(GtkTreeDragSource* source, GtkTreePath* path) {
	/* as long as we only display slides, there's no problem here */
	return TRUE;
}

static CriaSlideList*
cslp_container_for_path_depth(CriaSidebarModel* self, GtkTreePath* path, gint depth) {
	CriaSlideList* c = CRIA_SLIDE_LIST(self->priv->presentation);
	guint i = 0;

	g_return_val_if_fail(gtk_tree_path_get_depth(path)  >= depth, NULL);

	for(i = 0; i < depth; i++) {
		guint num = gtk_tree_path_get_indices(path)[i];
		if(num >= cria_slide_list_n_slides(c)) {
			return NULL;
		}
		c = CRIA_SLIDE_LIST(cria_slide_list_get(c, num));
	}

	g_return_val_if_fail(CRIA_IS_SLIDE_LIST(c), NULL);
	return c;
}

static CriaSlideList*
cslp_container_for_path(CriaSidebarModel* self, GtkTreePath* path) {
	return cslp_container_for_path_depth(self, path, gtk_tree_path_get_depth(path));
}

static gboolean
cslp_drag_data_get(GtkTreeDragSource* source, GtkTreePath* path, GtkSelectionData* data) {
	CriaSlideList* c;

	c = cslp_container_for_path(CRIA_SIDEBAR_MODEL(source), path);
	g_return_val_if_fail(CRIA_IS_SLIDE(c), FALSE);
	data->data = (guchar*)g_new0(CriaSlide*,1);
	*((CriaSlide**)data->data) = g_object_ref(c);
	data->length = sizeof(CriaSlide*);

	return TRUE;
}

static gboolean
cslp_drag_data_delete(GtkTreeDragSource* source, GtkTreePath* path) {
	CriaSlideList* c;

	c = cslp_container_for_path_depth(CRIA_SIDEBAR_MODEL(source),
					  path,
					  gtk_tree_path_get_depth(path) - 1);
	cria_slide_list_remove(c, gtk_tree_path_get_indices(path)[gtk_tree_path_get_depth(path) - 1]);
	return TRUE;
}

static void
cslp_init_tree_drag_iface(gpointer interface) {
	GtkTreeDragSourceIface* iface = interface;

	iface->row_draggable    = cslp_row_draggable;
	iface->drag_data_get    = cslp_drag_data_get;
	iface->drag_data_delete = cslp_drag_data_delete;
}

static gboolean
cslp_drag_data_received(GtkTreeDragDest* dest, GtkTreePath* path, GtkSelectionData* data) {
	CriaSlide* slide, * copy;
	
	slide = *((CriaSlide**)(data->data));
	copy = cria_slide_copy(slide,
			       cslp_container_for_path_depth(CRIA_SIDEBAR_MODEL(dest),
							     path,
							     gtk_tree_path_get_depth(path) - 1),
			       gtk_tree_path_get_indices(path)[gtk_tree_path_get_depth(path) - 1]);
	g_object_unref(slide);

	return TRUE;
}

static gboolean
cslp_row_drop_possible(GtkTreeDragDest* dest, GtkTreePath* path, GtkSelectionData* data) {
#warning "dragDataReceived(): FIXME: enable this for slides too"
	return gtk_tree_path_get_depth(path) == 1;
}

static void
cslp_init_tree_drag_dest(gpointer interface) {
	GtkTreeDragDestIface* iface = interface;

	iface->drag_data_received = cslp_drag_data_received;
	iface->row_drop_possible  = cslp_row_drop_possible;
}

GType
cria_sidebar_model_get_type(void) {
	static GType	type = 0;

	if(!type) {
		static const GTypeInfo info = {
			sizeof(CriaSidebarModelClass),
			NULL,	/* base initializer */
			NULL,	/* base finalizer */
			(GClassInitFunc)cria_sidebar_model_class_init,
			NULL,	/* class finalizer */
			NULL,	/* class data */
			sizeof(CriaSidebarModel),
			0,
			(GInstanceInitFunc)cria_sidebar_model_init,
			0
		};

		static const GInterfaceInfo slide_view_info = {
			NULL, /* init */
			NULL, /* finalize */
			NULL  /* data */
		};

		static const GInterfaceInfo tree_model_info = {
			cria_sidebar_model_init_tree_model_iface, /* init */
			NULL, /* finalize */
			NULL  /* data */
		};

		static const GInterfaceInfo tree_drag_info = {
			(GInterfaceInitFunc)cslp_init_tree_drag_iface, /* init */
			NULL, /* finalize */
			NULL  /* data */
		};

		static const GInterfaceInfo tree_drop_info = {
			(GInterfaceInitFunc)cslp_init_tree_drag_dest, /* init */
			NULL, /* finalize */
			NULL  /* data */
		};

		type = g_type_register_static(G_TYPE_OBJECT,
					      "CriaSidebarModel",
					      &info,
					      0);
		g_type_add_interface_static(type,
					    CRIA_TYPE_PRESENTATION_VIEW,
					    &slide_view_info);
		g_type_add_interface_static(type,
					    GTK_TYPE_TREE_MODEL,
					    &tree_model_info);
		g_type_add_interface_static(type,
					    GTK_TYPE_TREE_DRAG_SOURCE,
					    &tree_drag_info);
		g_type_add_interface_static(type,
					    GTK_TYPE_TREE_DRAG_DEST,
					    &tree_drop_info);
	}

	return type;
}

static void
cria_sidebar_model_get_value(GtkTreeModel* tree_model, GtkTreeIter* iter, gint column, GValue* value) {
	g_return_if_fail(0 <= column && column < N_COLUMNS);

	g_value_init(value, column_types[column]);

	switch(column) {
	case COLUMN_PREVIEW:
#warning "SlideListProxy::getValue(): add support for preview pixmaps"
		g_value_set_object(value, NULL);
		break;
	case COLUMN_TITLE:
		g_value_set_string(value, cria_slide_get_title(CRIA_SLIDE(iter->user_data)));
		break;
	case COLUMN_SLIDE_PTR:
		g_value_set_object(value, CRIA_SLIDE(iter->user_data));
		break;
	default:
		cdebugo(tree_model, "getValue()", "unknown column '%i'", column);
		g_assert_not_reached();
	}
}

static void
cria_sidebar_model_init(CriaSidebarModel* self) {
	g_return_if_fail(CRIA_IS_SIDEBAR_MODEL(self));

	self->priv = g_new0(CriaSidebarModelPrivate,1);
}

static void
cria_sidebar_model_init_tree_model_iface(gpointer iface, gpointer data) {
	GtkTreeModelIface* tree_model_class = iface;
	g_assert(G_TYPE_FROM_INTERFACE(tree_model_class) == GTK_TYPE_TREE_MODEL);

	tree_model_class->get_column_type = cria_sidebar_model_get_column_type;
	tree_model_class->get_flags       = cria_sidebar_model_get_flags;
	tree_model_class->get_iter        = cria_sidebar_model_get_iter;
	tree_model_class->get_n_columns   = cria_sidebar_model_get_n_columns;
	tree_model_class->get_path        = cria_sidebar_model_get_path;
	tree_model_class->get_value       = cria_sidebar_model_get_value;
	tree_model_class->iter_children   = cria_sidebar_model_iter_children;
	tree_model_class->iter_has_child  = cria_sidebar_model_iter_has_child;
	tree_model_class->iter_n_children = cria_sidebar_model_iter_n_children;
	tree_model_class->iter_next       = cria_sidebar_model_iter_next;
	tree_model_class->iter_nth_child  = cria_sidebar_model_iter_nth_child;
	tree_model_class->iter_parent     = cria_sidebar_model_iter_parent;
	tree_model_class->ref_node        = cria_sidebar_model_ref_node;
	tree_model_class->unref_node      = cria_sidebar_model_unref_node;
}

static gboolean
cria_sidebar_model_iter_children(GtkTreeModel* tree_model, GtkTreeIter *iter, GtkTreeIter* parent) {
	return FALSE;
}

static gboolean
cria_sidebar_model_iter_has_child(GtkTreeModel* tree_model, GtkTreeIter* iter) {
	return FALSE;
}

static gboolean
cria_sidebar_model_iter_next(GtkTreeModel* tree_model, GtkTreeIter* iter) {
	gint             n;
	gboolean         retval = FALSE;
	CriaPresentation*presentation;
	CriaSlide       *slide;

	g_return_val_if_fail(CRIA_IS_SIDEBAR_MODEL(tree_model), retval);
	
	presentation = CRIA_SIDEBAR_MODEL(tree_model)->priv->presentation;
	slide        = CRIA_SLIDE(iter->user_data);
	
	if(slide) {
		n = cria_slide_list_index(CRIA_SLIDE_LIST(presentation), slide);
	} else {
		n = 0;
	}

	if((n + 1) < cria_slide_list_n_slides(CRIA_SLIDE_LIST(presentation))) {
		iter->user_data = cria_slide_list_get(CRIA_SLIDE_LIST(presentation), n + 1);
		retval = TRUE;
	}

	return retval;
}

static gint
cria_sidebar_model_iter_n_children(GtkTreeModel* tree_model, GtkTreeIter* iter) {
	g_return_val_if_fail(CRIA_IS_SIDEBAR_MODEL(tree_model), 0);
	
	if(iter == NULL) {
		return cria_slide_list_n_slides(CRIA_SLIDE_LIST(CRIA_SIDEBAR_MODEL(tree_model)->priv->presentation));
	} else {
		return 0;
	}
}

static gboolean
cria_sidebar_model_iter_nth_child(GtkTreeModel* tree_model, GtkTreeIter* iter, GtkTreeIter* parent, gint n) {
	gboolean retval = FALSE;
	
	g_return_val_if_fail(CRIA_IS_SIDEBAR_MODEL(tree_model), retval);
	cdebugo(tree_model, "iterNthChild()", "iter(0x%x), parent(0x%x), n(%i)", (uintptr_t)iter, (uintptr_t)parent, n);

	if(parent == NULL) {
		GtkTreePath* path = gtk_tree_path_new();
		gtk_tree_path_append_index(path, n);
		retval = cria_sidebar_model_get_iter(tree_model, iter, path);
		gtk_tree_path_free(path);
	} else {
		g_assert_not_reached();
	}
	
	return retval;
}

static gboolean
cria_sidebar_model_iter_parent(GtkTreeModel* tree_model, GtkTreeIter* iter, GtkTreeIter* child) {
	return FALSE;
}

static void
cria_sidebar_model_ref_node(GtkTreeModel* tree_model, GtkTreeIter* iter) {
	g_object_ref(CRIA_SLIDE(iter->user_data));
}

static void
cb_presentation_inserted_slide(CriaSidebarModel* self, gint new_position, CriaSlide* slide, CriaPresentation* presentation) {
	GtkTreeIter	  iter;
	GtkTreePath	* path;
	
	g_return_if_fail(CRIA_IS_SIDEBAR_MODEL(self));
	
	path = gtk_tree_path_new();
	gtk_tree_path_append_index(path, new_position);
	gtk_tree_model_get_iter(GTK_TREE_MODEL(self), &iter, path);
	gtk_tree_model_row_inserted(GTK_TREE_MODEL(self), path, &iter);
	gtk_tree_path_free(path);
}

static void
csm_slide_deleted(CriaSidebarModel* self) {
#warning "FIXME: add implementation"
}

static void
cria_sidebar_model_set_presentation(CriaSidebarModel* self, CriaPresentation* presentation) {
	g_return_if_fail(CRIA_IS_SIDEBAR_MODEL(self));
	g_return_if_fail(CRIA_IS_PRESENTATION(presentation));

	if(self->priv->presentation != NULL) {
		g_free(self->priv->presentation);
	}

	self->priv->presentation = g_object_ref(presentation);
	g_signal_connect_swapped(self->priv->presentation, "slide-deleted",
				 G_CALLBACK(csm_slide_deleted), self);
	g_signal_connect_swapped(self->priv->presentation, "inserted-slide",
				 G_CALLBACK(cb_presentation_inserted_slide), self);

	g_object_notify(G_OBJECT(self), "presentation");
}

static void
cria_sidebar_model_set_property(GObject* object, guint prop_id, const GValue* value, GParamSpec* param_spec) {
	CriaSidebarModel	* self;
	
	self = CRIA_SIDEBAR_MODEL(object);
	
	switch(prop_id) {
	case CRIA_PRESENTATION_VIEW_PROP_PRESENTATION:
		cria_sidebar_model_set_presentation(self, g_value_get_object(value));
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, param_spec);
		break;
	}
}

static void
cria_sidebar_model_unref_node(GtkTreeModel* tree_model, GtkTreeIter* iter) {
	g_object_unref(CRIA_SLIDE(iter->user_data));
}

