/* this file is part of criawips, a gnome presentation application
 *
 * AUTHORS
 *       Sven Herzberg        <herzi@gnome-de.org>
 *
 * Copyright (C) 2004,2005 Sven Herzberg
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "slide-renderer.h"
#include "slide-renderer-priv.h"

#include <inttypes.h>

#include <glib.h>
#include <glib-object.h>
#include <libgnomecanvas/libgnomecanvas.h>

#include <cdebug.h>

#include <rendering/cria-block-renderer.h>

#include "background-renderer.h"
#include "enumerations.h"
#include "image-pool.h"
#include "slide-editor.h"
#include "slide-view.h"

enum {
	PROP_0,
};

enum {
	N_SIGNALS
};

static GObjectClass* parent_class = NULL;

static void cria_slide_renderer_get_property   (GObject		* object,
						guint		  prop_id,
						GValue		* value,
						GParamSpec	* param_spec);
static void cria_slide_renderer_set_property   (GObject		* object,
						guint		  prop_id,
						const	GValue	* value,
						GParamSpec	* param_spec);
#if 0
/* enable these to add support for signals */
static	guint	cria_slide_renderer_signals[N_SIGNALS] = { 0 };

static	void	cria_slide_renderer_signal	       (CriaSlideRenderer	* self,
						const	gchar	* string);
#endif

static void
csr_update_focus_layer_size(CriaSlideRenderer* self) {
	GoPoint* size;
	g_return_if_fail(GNOME_IS_CANVAS_ITEM(self->focus_layer));
	g_return_if_fail(CRIA_IS_SLIDE(self->slide));
	
	size = cria_slide_get_size(self->slide);
	gnome_canvas_item_set(self->focus_layer,
			"x1", 0.0,
			"y1", 0.0,
			"x2", 0.0 + size->x,
			"y2", 0.0 + size->y,
			NULL);
	g_free(size);
}

static void
csr_ensure_stacking(CriaSlideRenderer* self) {
	if(self->frame) {
		gnome_canvas_item_raise_to_top(self->frame);
	}

	if(self->focus_layer) {
		gnome_canvas_item_lower_to_bottom(self->focus_layer);
	}
	if(self->background_renderer) {
		gnome_canvas_item_lower_to_bottom(GNOME_CANVAS_ITEM(self->background_renderer));
	}
	if(self->shadow) {
		gnome_canvas_item_lower_to_bottom(self->shadow);
	}
}

static void
csr_ensure_focus_layer(CriaSlideRenderer* self) {
	g_return_if_fail(GNOME_CANVAS_ITEM(self)->canvas);
	if(!self->focus_layer) {
		self->focus_layer = gnome_canvas_item_new(GNOME_CANVAS_GROUP(self),
							  gnome_canvas_rect_get_type(),
							  NULL);
		csr_update_focus_layer_size(self);
		csr_ensure_stacking(self);
	}
}


static gdouble
csr_point(GnomeCanvasItem* self, gdouble x, gdouble y, int cx, int cy, GnomeCanvasItem** actual_item) {
	gdouble retval = FALSE;

	if(actual_item && *actual_item == GNOME_CANVAS_ITEM(CRIA_SLIDE_RENDERER(self)->background_renderer)) {
		/* don't focus the background renderer, focus ourselves */
		*actual_item = self;
	}

	if(!retval && GNOME_CANVAS_ITEM_CLASS(parent_class)->point) {
		retval = GNOME_CANVAS_ITEM_CLASS(parent_class)->point(self, x, y, cx, cy, actual_item);
	}
	
	return retval;
}

static gboolean
csr_button_press_event(CriaItem* item, GdkEventButton* ev) {
	gboolean retval = FALSE;

	if(ev->type == GDK_BUTTON_PRESS && ev->button == 1) {
		cria_item_grab_focus(item);
		retval = TRUE;
	}
	
	return retval;
}

static gboolean
csr_focus_in_event(CriaItem* item, GdkEventFocus* ev) {
	CriaSlideRenderer* self = CRIA_SLIDE_RENDERER(item);
	guint		   color;
	GdkColor           gdk_color;
	gdk_color = GTK_WIDGET(GNOME_CANVAS_ITEM(self)->canvas)->style->base[GTK_STATE_SELECTED];
	gnome_canvas_item_set(self->frame,
		     /* MEMO text would get style->text[GTK_STATE_SELECTED] */
		     "outline-color-gdk", &gdk_color,
		     "width-pixels", 2,
		     NULL);
	/* The focus layer doesn't work, because the canvas doesn't support semi-transparent rectangles */
	csr_ensure_focus_layer(self);
	color = GNOME_CANVAS_COLOR_A(gdk_color.red >> 2, gdk_color.green >> 2, gdk_color.blue >> 2, 0x11);
	gnome_canvas_item_set(self->focus_layer,
		     "fill-color-rgba", &color,
		     NULL);
	gnome_canvas_item_show(self->focus_layer);
	return TRUE;
}

static void
csr_frame_unfocused(CriaSlideRenderer* self) {
	gnome_canvas_item_set(self->frame,
		     /* MEMO text would get style->text[GTK_STATE_ACTIVE] */
		     "outline-color-gdk", &GTK_WIDGET(GNOME_CANVAS_ITEM(self)->canvas)->style->black,
		     "width-pixels", 1,
		     NULL);
}

static gboolean
csr_focus_out_event(CriaItem* item, GdkEventFocus* ev) {
	CriaSlideRenderer* self = CRIA_SLIDE_RENDERER(item);
	
	csr_frame_unfocused(self);
	/* The focus layer doesn't work because the canvas doesn't support semi-transparent rectangles */
/*	if(self->focus_layer) {
		gnome_canvas_item_hide(self->focus_layer);
	}
*/	return TRUE;
}

static void
csr_finalize(GObject* object) {
	CriaSlideRenderer* self = CRIA_SLIDE_RENDERER(object);
	
	g_object_unref(self->slide);
	self->slide = NULL;

	parent_class->finalize(object);
}

static void
cria_slide_renderer_class_init(CriaSlideRendererClass* cria_slide_renderer_class) {
	GObjectClass	    * g_object_class;
	GnomeCanvasItemClass* canvas_item_class;
	CriaItemClass       * item_class;

	parent_class = g_type_class_peek_parent(cria_slide_renderer_class);

	/* setting up the object class */
	g_object_class = G_OBJECT_CLASS(cria_slide_renderer_class);
	g_object_class->finalize = csr_finalize;
#if 0
	/* setting up signal system */
	cria_slide_renderer_class->signal = cria_slide_renderer_signal;

	cria_slide_renderer_signals[SIGNAL] = g_signal_new("signal",
						  CRIA_TYPE_SLIDE_RENDERER,
						  G_SIGNAL_RUN_LAST,
						  G_STRUCT_OFFSET(CriaSlideRendererClass,
							  	  signal),
						  NULL,
						  NULL,
						  g_cclosure_marshal_VOID__STRING,
						  G_TYPE_NONE,
						  0);
#endif
	/* setting up the object class */
	g_object_class->set_property = cria_slide_renderer_set_property;
	g_object_class->get_property = cria_slide_renderer_get_property;

	/* setting up the canvas item class */
	canvas_item_class = GNOME_CANVAS_ITEM_CLASS(cria_slide_renderer_class);
	canvas_item_class->point = csr_point;

	/* setting up the cria item class */
	item_class = CRIA_ITEM_CLASS(cria_slide_renderer_class);
	item_class->button_press_event = csr_button_press_event;
	item_class->focus_in_event = csr_focus_in_event;
	item_class->focus_out_event = csr_focus_out_event;

	/* setting up the slide view interface */
	_cria_slide_view_install_properties(g_object_class);
}

static void
csr_update_frame_size(CriaSlideRenderer* self) {
	GoPoint* size;
	g_return_if_fail(GNOME_IS_CANVAS_ITEM(self->frame));
	g_return_if_fail(CRIA_IS_SLIDE(self->slide));

	size = cria_slide_get_size(self->slide);
	gnome_canvas_item_set(self->frame,
		     "x1", 0.0,
		     "y1", 0.0,
		     "x2", 0.0 + size->x,
		     "y2", 0.0 + size->y,
		     NULL);
	g_free(size);
}

static void
csr_ensure_frame(CriaSlideRenderer* self) {
	if(!self->frame) {
		self->frame = gnome_canvas_item_new(GNOME_CANVAS_GROUP(self),
						    gnome_canvas_rect_get_type(),
						    NULL);
		csr_frame_unfocused(self);

		if(self->slide) {
			csr_update_frame_size(self);
		}
		gnome_canvas_item_raise_to_top(self->frame);
		gnome_canvas_item_show(self->frame);
	}
}

static void
csr_update_shadow_size(CriaSlideRenderer* self) {
	GoPoint* size;
	g_return_if_fail(GNOME_IS_CANVAS_ITEM(self->shadow));
	g_return_if_fail(CRIA_IS_SLIDE(self->slide));

	size = cria_slide_get_size(self->slide);
	gnome_canvas_item_set(self->shadow,
		     "x1", 0.0 + SLIDE_EDITOR_SHADOW_OFFSET,
		     "y1", 0.0 + SLIDE_EDITOR_SHADOW_OFFSET,
		     "x2", 0.0 + (size->x + SLIDE_EDITOR_SHADOW_OFFSET),
		     "y2", 0.0 + (size->y + SLIDE_EDITOR_SHADOW_OFFSET),
		     NULL);
	g_free(size);
}

static void
csr_ensure_shadow(CriaSlideRenderer* self) {
	if(!self->shadow) {
		/* add the shadow */
		self->shadow = gnome_canvas_item_new(GNOME_CANVAS_GROUP(self),
						   gnome_canvas_rect_get_type(),
						   "fill-color", "gray",
#warning "ensureShadow(): FIXME: set the fill-color to semi-transparent black instead of gray"
						   NULL);

		if(self->slide) {
			csr_update_shadow_size(self);
		}
		gnome_canvas_item_lower_to_bottom(self->shadow);
		gnome_canvas_item_show(self->shadow);
	}
}

static void
cria_slide_renderer_get_property(GObject* object, guint prop_id, GValue* value, GParamSpec* param_spec) {
	CriaSlideRenderer	* self;

	self = CRIA_SLIDE_RENDERER(object);

	switch(prop_id) {
	case CRIA_SLIDE_VIEW_PROP_SLIDE:
		g_value_set_object(value, cria_slide_renderer_get_slide(self));
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, param_spec);
		break;
	}
}

CriaSlide*
cria_slide_renderer_get_slide(CriaSlideRenderer* self) {
	g_return_val_if_fail(CRIA_IS_SLIDE_RENDERER(self), NULL);
	
	return self->slide;
}

static void
cb_interactive_changed(CriaSlideRenderer* self) {
	if(cria_item_is_interactive(CRIA_ITEM(self))) {
		csr_ensure_frame(self);
		csr_ensure_shadow(self);
		csr_ensure_stacking(self);
	} else {
		if(self->frame) {
			gnome_canvas_item_hide(self->frame);
		}

		if(self->shadow) {
			gnome_canvas_item_hide(self->shadow);
		}
	}
}

static void
cria_slide_renderer_init(CriaSlideRenderer* self) {
	CRIA_ITEM_SET_FLAG(CRIA_ITEM(self), CRIA_CAN_FOCUS);
	self->background_renderer = cria_background_renderer_new();

	g_signal_connect(self, "notify::interactive", G_CALLBACK(cb_interactive_changed), NULL);
}

GType
cria_slide_renderer_get_type(void) {
	static GType	type = 0;

	if(!type) {
		static const GTypeInfo info = {
			sizeof(CriaSlideRendererClass),
			NULL,	/* base initializer */
			NULL,	/* base finalizer */
			(GClassInitFunc)cria_slide_renderer_class_init,
			NULL,	/* class finalizer */
			NULL,	/* class data */
			sizeof(CriaSlideRenderer),
			0,
			(GInstanceInitFunc)cria_slide_renderer_init,
			0
		};

		static const GInterfaceInfo slide_view_info = {
			NULL, /* GInterfaceInit */
			NULL, /* GInterfaceFinalize */
			NULL  /* GInterfaceData */
		};

		type = g_type_register_static(CRIA_TYPE_ITEM,
					      "CriaSlideRenderer",
					      &info,
					      0);
		g_type_add_interface_static(type, CRIA_TYPE_SLIDE_VIEW, &slide_view_info);
	}

	return type;
}

/**
 * cria_slide_renderer_new:
 * @display: a #CriaCanvas
 * @interactive: specifies whether the renderer should allow editing
 *
 * Create a new renderer for a #CriaSlide.
 *
 * Returns the new slide renderer.
 */
CriaSlideRenderer*
cria_slide_renderer_new(CriaCanvas* canvas, gboolean interactive) {
	g_return_val_if_fail(CRIA_IS_CANVAS(canvas), NULL);
	return CRIA_SLIDE_RENDERER(gnome_canvas_item_new(gnome_canvas_root(GNOME_CANVAS(canvas)),
							 CRIA_TYPE_SLIDE_RENDERER,
							 "interactive", interactive,
							 NULL));
}

static void
cria_slide_renderer_set_property(GObject* object, guint prop_id, const GValue* value, GParamSpec* param_spec) {
	CriaSlideRenderer	* self;
	
	self = CRIA_SLIDE_RENDERER(object);
	
	switch(prop_id) {
	case CRIA_SLIDE_VIEW_PROP_SLIDE:
		cria_slide_renderer_set_slide(self, g_value_get_object(value));
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID(object,
						  prop_id,
						  param_spec);
		break;
	}
}

static void
update_block_renderers(CriaSlideRenderer* self) {
	/* This is the algorithm:
	 * 1. iterate over the existing renderers until
	 *    a) there are no renderers left
	 *    b) there are no blocks left
	 * 2. create new renderers if b) or hide renderers if a)
	 */

	g_return_if_fail(CRIA_IS_SLIDE(self->slide));

	GList   * it;
	gchar   **blocks;
	gchar   **it2;

	it2 = blocks = cria_slide_get_block_names(self->slide);

	for(it = self->block_renderers; it && *it2; it = it->next) {
		cdebugo(self, "updateBlockRenderers()", "Rendering block: %s", *it2);
		cria_block_renderer_set_block(CRIA_BLOCK_RENDERER(it->data), cria_slide_get_block(self->slide, *it2));
		cria_block_renderer_show(CRIA_BLOCK_RENDERER(it->data));
		/* be careful another iteration is done at the end of the following line */
		it2++;
	}

	if(*it2 == NULL && it != NULL) {
		for(; it; it = it->next) {
			cdebugo(self, "updateBlockRenderers()", "Hiding a block renderer");
			cria_block_renderer_hide(CRIA_BLOCK_RENDERER(it->data));
		}
	} else if (*it2 != NULL && it == NULL) {
		for(; it2 && *it2; it2++) {
			cdebugo(self, "updateBlockRenderers()", "Creating a block renderer for %s", *it2);
			self->block_renderers = g_list_prepend(self->block_renderers,
							       cria_block_renderer_new(CRIA_ITEM(self),
										       cria_item_is_interactive(CRIA_ITEM(self))));
			cria_block_renderer_set_block(CRIA_BLOCK_RENDERER(self->block_renderers->data),
						      cria_slide_get_block(self->slide, *it2));
		}
	}

	g_strfreev(blocks);
}

static void
csr_ensure_background_renderer(CriaSlideRenderer* self) {
	gnome_canvas_item_construct(GNOME_CANVAS_ITEM(self->background_renderer),
				    GNOME_CANVAS_GROUP(self),
				    NULL, NULL);
	csr_ensure_stacking(self);

	gnome_canvas_item_show(GNOME_CANVAS_ITEM(self->background_renderer));
}

static void
update_slide_size(CriaSlideRenderer* self) {
	g_return_if_fail(CRIA_IS_SLIDE_RENDERER(self));
	g_return_if_fail(CRIA_IS_SLIDE(self->slide));
	
	if(self->frame) {
		csr_update_frame_size(self);
	}
	if(self->shadow) {
		csr_update_shadow_size(self);
	}
	if(self->focus_layer) {
		csr_update_focus_layer_size(self);
	}
	if(self->background_renderer) {
		GoPoint* size = cria_slide_get_size(self->slide);
		GoRect   rect;
		
		rect.top    = 0;
		rect.left   = 0;
		rect.right  = size->x;
		rect.bottom = size->y;
		g_free(size);
		
		cria_background_renderer_set_position(self->background_renderer, &rect);
	}
}

void
cria_slide_renderer_set_slide(CriaSlideRenderer* self, CriaSlide* slide) {
	g_return_if_fail(CRIA_IS_SLIDE_RENDERER(self));
	g_return_if_fail(CRIA_IS_SLIDE(slide));

	if(self->slide != NULL) {
		g_object_unref(self->slide);
	}

	self->slide = g_object_ref(slide);

	g_object_notify(G_OBJECT(self), "slide");

	g_return_if_fail(GNOME_CANVAS_ITEM(self)->canvas);
	if(!GNOME_CANVAS_ITEM(self->background_renderer)->canvas) {
		csr_ensure_background_renderer(self);
	}
	cria_background_renderer_set_background(self->background_renderer, cria_slide_get_background(self->slide, FALSE));

	update_slide_size(self);

	update_block_renderers(self);
}

