/* this file is part of criawips a gnome presentation application
 *
 * AUTHORS
 *       Sven Herzberg        <herzi@gnome-de.org>
 *
 * Copyright (C) 2004,2005 Sven Herzberg
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "slide-editor.h"
#include "slide-editor-priv.h"

#include <inttypes.h>
#include <string.h>

#include <glib.h>
#include <glib-object.h>
#include <glib/gi18n.h>
#include <libgnomecanvas/libgnomecanvas.h>
#include <ccc/canvas.h>

#include <cdebug.h>

#include "presentation.h"
#include "slide-renderer.h"
#include "slide-view.h"

enum {
	PROP_0,
};

static GObjectClass* parent_class = NULL;

static gboolean cria_slide_editor_focus_in	       (GtkWidget	* self,
							GdkEventFocus	* ev);
static gboolean cria_slide_editor_focus_out	       (GtkWidget	* self,
							GdkEventFocus	* ev);
static	void	cria_slide_editor_get_property         (GObject		* object,
					        	guint		  prop_id,
					        	GValue		* value,
					        	GParamSpec	* param_spec);
static	void	cria_slide_editor_init	               (CriaSlideEditor	* self);
static	void	cria_slide_editor_set_property         (GObject		* object,
						        guint		  prop_id,
						        const	GValue	* value,
						        GParamSpec	* param_spec);
static gboolean	cria_slide_editor_popup_menu	       (GtkWidget	* self);
static	void	cria_slide_editor_slide_changed        (CriaSlideEditor	* self,
						        gpointer	  data);

static void
cse_finalize(GObject* object) {
	CriaSlideEditor* self = CRIA_SLIDE_EDITOR(object);

	if(self->slide) {
		g_object_unref(self->slide);
		self->slide = NULL;
	}

	parent_class->finalize(object);
}

static void
cria_slide_editor_class_init(CriaSlideEditorClass* cria_slide_editor_class) {
	GObjectClass	* g_object_class;
	GtkWidgetClass	* widget_class;

	/* setting up the object class */
	g_object_class = G_OBJECT_CLASS(cria_slide_editor_class);
	g_object_class->finalize = cse_finalize;
	g_object_class->set_property = cria_slide_editor_set_property;
	g_object_class->get_property = cria_slide_editor_get_property;

	/* setting up the widget class */
	widget_class = GTK_WIDGET_CLASS(cria_slide_editor_class);
	widget_class->focus_in_event = cria_slide_editor_focus_in;
	widget_class->focus_out_event = cria_slide_editor_focus_out;
	widget_class->popup_menu = cria_slide_editor_popup_menu;

	/* setting up the gnome canvas class */
	parent_class = g_type_class_peek_parent(cria_slide_editor_class);

	/* setting up the slide view interface */
	_cria_slide_view_install_properties(g_object_class);
}

static gboolean
cria_slide_editor_focus_in(GtkWidget* self, GdkEventFocus* ev) {
	cdebugo(self, "focusIn()", "run");
	return GTK_WIDGET_CLASS(parent_class)->focus_in_event(self, ev);
}

static gboolean
cria_slide_editor_focus_out(GtkWidget* self, GdkEventFocus* ev) {
	cdebugo(self, "focusOut()", "run");
	return GTK_WIDGET_CLASS(parent_class)->focus_out_event(self, ev);
}

CriaSlide*
cria_slide_editor_get_slide(CriaSlideEditor* self) {
	g_return_val_if_fail(CRIA_IS_SLIDE_EDITOR(self), NULL);
	
	return self->slide;
}

static void
cria_slide_editor_get_property(GObject* object, guint prop_id, GValue* value, GParamSpec* param_spec) {
	CriaSlideEditor	* self;

	self = CRIA_SLIDE_EDITOR (object);

	switch(prop_id) {
	case CRIA_SLIDE_VIEW_PROP_SLIDE:
		g_value_set_object(value, self->slide);
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, param_spec);
		break;
	}
}

GType
cria_slide_editor_get_type(void) {
	static GType	type = 0;

	if(!type) {
		static const GTypeInfo info = {
			sizeof (CriaSlideEditorClass),
			NULL,	/* base initializer */
			NULL,	/* base finalizer */
			(GClassInitFunc)cria_slide_editor_class_init,
			NULL,	/* class finalizer */
			NULL,	/* class data */
			sizeof (CriaSlideEditor),
			0,
			(GInstanceInitFunc)cria_slide_editor_init,
			0
		};

		static const GInterfaceInfo view_info = {
			NULL, /* GInterfaceInitFunc */
			NULL, /* GInterfaceFinalizeFunc */
			NULL  /* data */
		};

		type = g_type_register_static(CRIA_TYPE_SLIDE_DISPLAY,
					      "CriaSlideEditor",
					      &info,
					      0);
		g_type_add_interface_static(type, CRIA_TYPE_SLIDE_VIEW, &view_info);
	}

	return type;
}

static void
cria_slide_editor_init(CriaSlideEditor   * self) {
	/* set anti aliased mode before creating any items */
	g_object_set(self, "aa", TRUE, NULL);

#warning "init(): FIXME: try to delay the creation until the editor is completely initialized"
	/* prepare the slide display widget */
	self->slide_renderer = cria_slide_renderer_new(CRIA_CANVAS(self), TRUE);
	gnome_canvas_item_show(GNOME_CANVAS_ITEM(self->slide_renderer));

	/* connect neccessary signals */
	g_signal_connect(self, "notify::slide",
			 G_CALLBACK(cria_slide_editor_slide_changed), NULL);

	GTK_WIDGET_CAN_FOCUS(self);
}

/**
 * cria_slide_editor_new:
 * @slide: a #CriaSlide
 *
 * Create a new slide editor, connected to @slide.
 *
 * Returns the new slide editor.
 */
GtkWidget*
cria_slide_editor_new(CriaSlide* slide) {
	GObject*	self = NULL;
	g_return_val_if_fail(slide == NULL || CRIA_IS_SLIDE(slide), GTK_WIDGET(self));
	
	self = g_object_new(CRIA_TYPE_SLIDE_EDITOR, NULL);
	
	if(slide) {
		cria_slide_editor_set_slide(CRIA_SLIDE_EDITOR(self), slide);
	}

	cria_canvas_set_padding(CRIA_CANVAS(self), SLIDE_EDITOR_PADDING);
	return GTK_WIDGET(self);
}

static gboolean
cria_slide_editor_popup_menu(GtkWidget* widget) {
#warning "SlideEditor::popupMenu(): find out where the cursor is and populate the menu"
#warning "SlideEditor::popupMenu(): show the menu"
	return FALSE;
}

void
cria_slide_editor_set_slide(CriaSlideEditor* self, CriaSlide* slide) {
	g_return_if_fail(CRIA_IS_SLIDE_EDITOR(self));
	g_return_if_fail(CRIA_IS_SLIDE(slide));
	
	cdebugo(self, "setSlide()", "start");

	if(self->slide != slide) {
		GoPoint	   * slide_s;
		CcRectangle  rect;
		
		if(self->slide != NULL) {
			g_object_unref(self->slide);
		}

		self->slide = g_object_ref(slide);

		slide_s = cria_slide_get_size(self->slide);

		rect.x = 0.0;
		rect.y = 0.0;
#warning "setSlide(): FIXME: make this floating point only"
#warning "setSlide(): FIXME: use the extents of the slide renderer later"
		rect.w = 0.0 + (slide_s->x + SLIDE_EDITOR_SHADOW_OFFSET);
		rect.h = 0.0 + (slide_s->y + SLIDE_EDITOR_SHADOW_OFFSET);
		cria_canvas_set_extents(CRIA_CANVAS(self), &rect);
		g_free(slide_s);
	
		g_object_notify(G_OBJECT(self), "slide");
	}
	
	cdebugo(self, "setSlide()", "end");
}

static void
cria_slide_editor_set_property(GObject* object, guint prop_id, const GValue* value, GParamSpec* param_spec) {
	CriaSlideEditor	* self;
	
	self = CRIA_SLIDE_EDITOR(object);
	
	switch(prop_id) {
	case CRIA_SLIDE_VIEW_PROP_SLIDE:
		cria_slide_editor_set_slide(self, g_value_get_object(value));
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, param_spec);
		break;
	}
}

static void
cria_slide_editor_slide_changed(CriaSlideEditor* self, gpointer data) {
	g_return_if_fail(CRIA_IS_SLIDE_EDITOR(self));

	cria_slide_renderer_set_slide(self->slide_renderer, cria_slide_editor_get_slide(self));
}

