/* this file is part of criawips, a gnome presentation application
 *
 * AUTHORS
 *       Sven Herzberg        <herzi@gnome-de.org>
 *
 * Copyright (C) 2004,2005 Sven Herzberg
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "slide-display.h"

#include <glib.h>
#include <glib-object.h>
#include <goffice/utils/go-units.h>
#include <cdebug/cdebug.h>

#include <utils/cria-units.h>

enum {
	PROP_0,
	PROP_SLIDE_ZOOM
};

enum {
	SIGNAL,
	N_SIGNALS
};

static GObjectClass* parent_class = NULL;

static void cria_slide_display_get_property	       (GObject		* object,
						guint		  prop_id,
						GValue		* value,
						GParamSpec	* param_spec);
static void cria_slide_display_set_property	       (GObject		* object,
						guint		  prop_id,
						const GValue	* value,
						GParamSpec	* param_spec);
#if 0
/* enable these to add support for signals */
static	guint	cria_slide_display_signals[N_SIGNALS] = { 0 };

static	void	cria_slide_display_signal	       (CriaSlideDisplay	* self,
						const	gchar	* string);
#endif

/*
 * cria_slide_get_display_size:
 * @self: a #CriaSlideDisplay
 *
 * Get the size that the slide will take on its screen.
 *
 * Returns a #GoPoint defining the size of the screen in pixels. Don't forget
 * to g_free it.
 */
static GoPoint*
cria_slide_display_get_size(CriaSlideDisplay* self) {
	GoPoint  * retval;
	GdkScreen* screen = gtk_widget_get_screen(GTK_WIDGET(self));

	g_return_val_if_fail(CRIA_IS_SLIDE_DISPLAY(self), NULL);
	g_return_val_if_fail(GDK_IS_SCREEN(screen), NULL);
	
#warning "getSize(): FIXME: make this code slide-agnostic"

	retval = g_new0(GoPoint,1);
	retval->x = gdk_screen_get_width(screen);
	retval->y = gdk_screen_get_height(screen);

	cdebugo(self, "getDisplaySize()", "got resolution %llix%lli (%f)", retval->x, retval->y, 1.0 * retval->x / retval->y);

#warning "Slide::getDisplaySize(): FIXME: get the ratio from the presentation"
#define RATIO_X 4
#define RATIO_Y 3
	if ((1.0 * retval->x / retval->y) >= (1.0 * RATIO_X / RATIO_Y)) {
		/* screen is too wide */
		retval->y -= (retval->y % RATIO_Y);
		retval->x  = retval->y / RATIO_Y * RATIO_X;
		cdebugo(self, "getDisplaySize()", "The screen is too wide, display resolution is %llix%lli", retval->x, retval->y);
	} else {
		/* screen is too high */
		retval->x -= (retval->x % RATIO_X);
		retval->y  = retval->x / RATIO_X * RATIO_Y;
		cdebugo(self, "getDisplaySize()", "The screen is too high, display resolution is %llix%lli", retval->x, retval->y);
	}
	
	return retval;
#undef RATIO_X
#undef RATIO_Y
}

static void
cria_slide_display_class_init(CriaSlideDisplayClass* cria_slide_display_class) {
	GObjectClass	* g_object_class;

	parent_class = g_type_class_peek_parent(cria_slide_display_class);

	/* setting up the object class system */
	g_object_class = G_OBJECT_CLASS(cria_slide_display_class);
	g_object_class->get_property = cria_slide_display_get_property;
	g_object_class->set_property = cria_slide_display_set_property;

	g_object_class_install_property(g_object_class,
					PROP_SLIDE_ZOOM,
					g_param_spec_double("slide-zoom",
							    "Slide Zoom",
							    "This attribute specifies the zoom value in a way that "
							    "100% are the full screen zoom.",
							    G_MINDOUBLE,
							    G_MAXDOUBLE,
							    0.75,
							    G_PARAM_READWRITE));
}

static void
cria_slide_display_get_property(GObject* object, guint prop_id, GValue* value, GParamSpec* param_spec) {
	CriaSlideDisplay	* self;

	self = CRIA_SLIDE_DISPLAY(object);

	switch(prop_id) {
	case PROP_SLIDE_ZOOM:
		g_value_set_double(value, cria_slide_display_get_slide_zoom(self));
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID(object,
						  prop_id,
						  param_spec);
		break;
	}
}

static void
notify_zoom_cb(CriaSlideDisplay* self) {
	g_object_notify(G_OBJECT(self), "slide-zoom");
}

static void
cria_slide_display_init(CriaSlideDisplay* self) {
	g_signal_connect(self, "notify::zoom", G_CALLBACK(notify_zoom_cb), NULL);
}

GType
cria_slide_display_get_type(void) {
	static GType	type = 0;

	if(!type) {
		static const GTypeInfo info = {
			sizeof(CriaSlideDisplayClass),
			NULL,	/* base initializer */
			NULL,	/* base finalizer */
			(GClassInitFunc)cria_slide_display_class_init,
			NULL,	/* class finalizer */
			NULL,	/* class data */
			sizeof(CriaSlideDisplay),
			0,
			(GInstanceInitFunc)cria_slide_display_init,
			0
		};

		type = g_type_register_static(CRIA_TYPE_CANVAS,
					      "CriaSlideDisplay",
					      &info,
					      0);
	}

	return type;
}

gdouble
cria_slide_display_get_slide_zoom(CriaSlideDisplay* self) {
	GoPoint* display = cria_slide_display_get_size(self),
		 slide;
	gdouble retval;
       
	slide.x = 5760;
	slide.y = 4320;
#warning "getSlideZoom(): FIXME: get the slide size from the slide"
	retval = cria_canvas_get_zoom(CRIA_CANVAS(self)) * slide.x / display->x;
	g_free(display);
	return retval;
}

/* we get a percentage here (0.0 - 1.0) and we want 100% to be fullscreen */
void
cria_slide_display_set_slide_zoom(CriaSlideDisplay* self, gdouble slide_zoom) {
	GoPoint* display;
	GoPoint  slide;
	
	g_return_if_fail(CRIA_IS_SLIDE_DISPLAY(self));
	
	display = cria_slide_display_get_size(self);
#warning "setSlideZoom(): FIXME: get the slide size from the slide"
	slide.x = 5760;
	slide.y = 4320;

	cria_canvas_set_zoom(CRIA_CANVAS(self), slide_zoom * display->x / slide.x);
	
	g_free(display);
}

static void
cria_slide_display_set_property(GObject* object, guint prop_id, const GValue* value, GParamSpec* param_spec) {
	CriaSlideDisplay	* self;
	
	self = CRIA_SLIDE_DISPLAY(object);
	
	switch(prop_id) {
	case PROP_SLIDE_ZOOM:
		cria_slide_display_set_slide_zoom(self, g_value_get_double(value));
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, param_spec);
		break;
	}
}

CriaSlideDisplay*
cria_slide_display_new(void) {
	return CRIA_SLIDE_DISPLAY(g_object_new(CRIA_TYPE_SLIDE_DISPLAY, NULL));
}

