/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/* this file is part of criawips a gnome presentation application
 *
 * AUTHORS
 *       Sven Herzberg        <herzi@gnome-de.org>
 *
 * Copyright (C) 2004 Sven Herzberg
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <glib-object.h>
#include <glib.h>
#include <libgnome/gnome-i18n.h>

#include "debug.h"
#include "slide.h"

struct _CriaSlidePrivate
{
	GHashTable	* blocks;
};

enum
{
	PROP_0,
	PROP_TITLE,
	PROP_LAYOUT,
	PROP_THEME
};

enum
{
	SIGNAL,
	N_SIGNALS
};

static	void	cria_slide_get_property	       (GObject		* object,
						guint		  prop_id,
						GValue		* value,
						GParamSpec	* param_spec);
static	void	cria_slide_init		       (CriaSlide	* self);
static	void	cria_slide_set_property        (GObject		* object,
						guint		  prop_id,
						const	GValue	* value,
						GParamSpec	* param_spec);

static	guint	cria_slide_signals[N_SIGNALS] = { 0 };

static	void	cria_slide_signal	       (CriaSlide	* template,
						const	gchar	* string);

void
cria_slide_add_block (
		CriaSlide	* self,
		CriaSlideBlock	* block)
{
	g_return_if_fail (CRIA_IS_SLIDE (self));
	g_return_if_fail (CRIA_IS_SLIDE_BLOCK (block));
	g_return_if_fail (cria_slide_block_get_title (block) != NULL);
        g_return_if_fail (cria_slide_get_block (self, cria_slide_block_get_title (block)) == NULL);

	g_hash_table_insert (self->priv->blocks,
			g_strdup (cria_slide_block_get_title (block)),
			block);
}

static void
cria_slide_class_init (CriaSlideClass	* cria_slide_class)
{
	GObjectClass	* g_object_class;

	g_object_class = G_OBJECT_CLASS(cria_slide_class);
#if 0
	/* setting up signal system */
	cria_slide_class->signal = cria_slide_signal;

	cria_slide_signals[SIGNAL] = g_signal_new (
			"signal",
			CRIA_TYPE_SLIDE,
			G_SIGNAL_RUN_LAST,
			G_STRUCT_OFFSET (
				CriaSlideClass,
				signal),
			NULL,
			NULL,
			g_cclosure_marshal_VOID__STRING,
			G_TYPE_NONE,
			0);
#endif
	/* setting up property system */
	g_object_class->set_property = cria_slide_set_property;
	g_object_class->get_property = cria_slide_get_property;

	g_object_class_install_property (
			g_object_class,
			PROP_TITLE,
			g_param_spec_string (
				"title",
				"Title",
				"The title of that slide",
				_("untitled"),
				G_PARAM_READWRITE | G_PARAM_CONSTRUCT)
			);
	g_object_class_install_property (
			g_object_class,
			PROP_LAYOUT,
			g_param_spec_string (
				"layout",
				"Layout",
				"The layout associated to this slide",
				NULL,
				G_PARAM_READWRITE | G_PARAM_CONSTRUCT)
			);
	g_object_class_install_property (
			g_object_class,
			PROP_THEME,
			g_param_spec_string (
				"theme",
				"Theme",
				"The theme associated with this slide",
				NULL,
				G_PARAM_READWRITE | G_PARAM_CONSTRUCT)
			);
}

CriaSlideBlock*
cria_slide_get_block (
		CriaSlide	* self,
		gchar const	* block)
{
	g_return_val_if_fail (CRIA_IS_SLIDE (self), NULL);

	return (CriaSlideBlock*) g_hash_table_lookup (self->priv->blocks, block);
}

const char*
cria_slide_get_layout (
		CriaSlide	* self)
{
	g_return_val_if_fail (CRIA_IS_SLIDE(self), NULL);
	
	return self->layout;
}

const char*
cria_slide_get_theme (
		CriaSlide	* self)
{
	g_return_val_if_fail (CRIA_IS_SLIDE(self), NULL);
	
	return self->theme;
}

const char*
cria_slide_get_title (
		CriaSlide	* self)
{
	g_return_val_if_fail (CRIA_IS_SLIDE(self), NULL);
	
	return self->title;
}

static void
cria_slide_get_property (
		GObject		* object,
		guint		  prop_id,
		GValue		* value,
		GParamSpec	* param_spec)
{
	CriaSlide	* self;

	self = CRIA_SLIDE (object);

	switch (prop_id)
	{
	case PROP_TITLE:
		g_value_set_string (
				value,
				self->title);
		break;
	case PROP_THEME:
		g_value_set_string (
				value,
				self->theme);
		break;
	case PROP_LAYOUT:
		g_value_set_string (
				value,
				self->layout);
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (
				object,
				prop_id,
				param_spec);
		break;
	}
}

GType
cria_slide_get_type (void)
{
	static GType	type = 0;

	if (!type)
	{
		const GTypeInfo info = {
			sizeof (CriaSlideClass),
			NULL,	/* base initializer */
			NULL,	/* base finalizer */
			(GClassInitFunc)cria_slide_class_init,
			NULL,	/* class finalizer */
			NULL,	/* class data */
			sizeof (CriaSlide),
			0,
			(GInstanceInitFunc)cria_slide_init,
			0
		};

		type = g_type_register_static (
				G_TYPE_OBJECT,
				"CriaSlide",
				&info,
				0);
	}

	return type;
}

static void
cria_slide_init (CriaSlide * self)
{
	self->priv = g_new0 (CriaSlidePrivate, 1);
	self->priv->blocks = g_hash_table_new_full (
			g_str_hash,
			g_str_equal,
			g_free,
			g_object_unref);
}

void
cria_slide_set_layout (
		CriaSlide	* self,
		const gchar	* layout)
{
	g_return_if_fail (CRIA_IS_SLIDE(self));

	if (self->layout != NULL)
		g_free (self->layout);

	g_debug ("cria_slide_set_layout(%i): \"%s\"", __LINE__, layout);
	self->layout = g_strdup (layout);

	g_object_notify (G_OBJECT (self), "layout");
}

void
cria_slide_set_theme (
		CriaSlide	* self,
		const gchar	* theme)
{
	g_return_if_fail (CRIA_IS_SLIDE(self));

	if (self->theme != NULL)
		g_free (self->theme);

	g_debug ("cria_slide_set_theme(%i): \"%s\"", __LINE__, theme);
	self->theme = g_strdup (theme);

	g_object_notify (G_OBJECT (self), "theme");
}

void
cria_slide_set_title (
		CriaSlide	* self,
		const gchar	* title)
{
	g_return_if_fail (CRIA_IS_SLIDE(self));

	if (self->title != NULL)
		g_free (self->title);

	g_debug ("cria_slide_set_title(%i): \"%s\"", __LINE__, title);
	self->title = g_strdup (title);

	g_object_notify (G_OBJECT (self), "title");
}

static void
cria_slide_set_property (
		GObject		* object,
		guint		  prop_id,
		const	GValue	* value,
		GParamSpec	* param_spec)
{
	CriaSlide	* self;
	
	self = CRIA_SLIDE (object);
	
	switch (prop_id)
	{
	case PROP_TITLE:
		cria_slide_set_title (
				self,
				g_value_get_string (value));
		break;
	case PROP_THEME:
		cria_slide_set_theme (
				self,
				g_value_get_string (value));
		break;
	case PROP_LAYOUT:
		cria_slide_set_layout (
				self,
				g_value_get_string (value));
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (
				object,
				prop_id,
				param_spec);
		break;
	}
}

