/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/* this file is part of criawips a gnome presentation application
 *
 * AUTHORS
 *       Sven Herzberg        <herzi@gnome-de.org>
 *
 * Copyright (C) 2004 Sven Herzberg
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <string.h>

#include <glib-object.h>
#include <glib.h>
#include <libgnome/gnome-i18n.h>

#include "debug.h"
#include "enumerations.h"
#include "layout-block.h"

enum
{
	PROP_0,
	PROP_TITLE,
	PROP_POS_X,
	PROP_POS_X_UNIT,
	PROP_POS_Y,
	PROP_POS_Y_UNIT,
	PROP_WIDTH,
	PROP_WIDTH_UNIT,
	PROP_HEIGHT,
	PROP_HEIGHT_UNIT,
	PROP_ALIGNMENT,
	PROP_VALIGNMENT,
	PROP_DEFAULT_TEXT,
	PROP_FONT_FAMILY,
	PROP_FONT_SIZE
};

enum
{
	SIGNAL,
	N_SIGNALS
};

static	void	cria_layout_block_get_property (GObject		* object,
						guint		  prop_id,
						GValue		* value,
						GParamSpec	* param_spec);
static	void	cria_layout_block_set_property (GObject		* object,
						guint		  prop_id,
						const	GValue	* value,
						GParamSpec	* param_spec);

static	guint	cria_layout_block_signals[N_SIGNALS] = { 0 };

static	void	cria_layout_block_signal       (CriaLayoutBlock	* template,
						const	gchar	* string);

static void
cria_layout_block_class_init (CriaLayoutBlockClass	* cria_layout_block_class)
{
	GObjectClass	* g_object_class;

	g_object_class = G_OBJECT_CLASS(cria_layout_block_class);
#if 0
	/* setting up signal system */
	cria_layout_block_class->signal = cria_layout_block_signal;

	cria_layout_block_signals[SIGNAL] = g_signal_new (
			"signal",
			CRIA_TYPE_LAYOUT_BLOCK,
			G_SIGNAL_RUN_LAST,
			G_STRUCT_OFFSET (
				CriaLayoutBlockClass,
				signal),
			NULL,
			NULL,
			g_cclosure_marshal_VOID__STRING,
			G_TYPE_NONE,
			0);
#endif
	/* setting up property system */
	g_object_class->set_property = cria_layout_block_set_property;
	g_object_class->get_property = cria_layout_block_get_property;

	g_object_class_install_property (
			g_object_class,
			PROP_TITLE,
			g_param_spec_string (
				"title",
				"Title",
				// TODO add documentation
				"",
				_("untitled"),
				G_PARAM_READWRITE | G_PARAM_CONSTRUCT)
			);
	g_object_class_install_property (
			g_object_class,
			PROP_POS_X,
			g_param_spec_double (
				"pos-x",
				"X-Position",
				// TODO add documentation
				"",
				-G_MAXDOUBLE,
				G_MAXDOUBLE,
				0,
				G_PARAM_READWRITE | G_PARAM_CONSTRUCT)
			);
	g_object_class_install_property (
			g_object_class,
			PROP_POS_X_UNIT,
			g_param_spec_enum (
				"pos-x-unit",
				"X-Position",
				// TODO add documentation
				"",
				CRIA_TYPE_UNIT,
				CRIA_UNIT_PIXEL,
				G_PARAM_READWRITE | G_PARAM_CONSTRUCT)
			);
	g_object_class_install_property (
			g_object_class,
			PROP_POS_Y,
			g_param_spec_double (
				"pos-y",
				"Y-Position",
				// TODO add documentation
				"",
				-G_MAXDOUBLE,
				G_MAXDOUBLE,
				0,
				G_PARAM_READWRITE | G_PARAM_CONSTRUCT)
			);
	g_object_class_install_property (
			g_object_class,
			PROP_POS_Y_UNIT,
			g_param_spec_enum (
				"pos-y-unit",
				"X-Position",
				// TODO add documentation
				"",
				CRIA_TYPE_UNIT,
				CRIA_UNIT_PIXEL,
				G_PARAM_READWRITE | G_PARAM_CONSTRUCT)
			);
	g_object_class_install_property (
			g_object_class,
			PROP_WIDTH,
			g_param_spec_double (
				"width",
				"Width",
				// TODO add documentation
				"",
				-G_MAXDOUBLE,
				G_MAXDOUBLE,
				0,
				G_PARAM_READWRITE | G_PARAM_CONSTRUCT)
			);
	g_object_class_install_property (
			g_object_class,
			PROP_WIDTH_UNIT,
			g_param_spec_enum (
				"width-unit",
				"X-Position",
				// TODO add documentation
				"",
				CRIA_TYPE_UNIT,
				CRIA_UNIT_PIXEL,
				G_PARAM_READWRITE | G_PARAM_CONSTRUCT)
			);
	g_object_class_install_property (
			g_object_class,
			PROP_HEIGHT,
			g_param_spec_double (
				"height",
				"Height",
				// TODO add documentation
				"",
				-G_MAXDOUBLE,
				G_MAXDOUBLE,
				0.,
				G_PARAM_READWRITE | G_PARAM_CONSTRUCT)
			);
	g_object_class_install_property (
			g_object_class,
			PROP_HEIGHT_UNIT,
			g_param_spec_enum (
				"height-unit",
				"X-Position",
				// TODO add documentation
				"",
				CRIA_TYPE_UNIT,
				CRIA_UNIT_PIXEL,
				G_PARAM_READWRITE | G_PARAM_CONSTRUCT)
			);
	g_object_class_install_property (
			g_object_class,
			PROP_ALIGNMENT,
			g_param_spec_enum (
				"alignment",
				"Horizontal Alignment",
				// TODO add documentation
				"",
				CRIA_TYPE_ALIGNMENT,
				CRIA_ALIGNMENT_LEFT,
				G_PARAM_READWRITE | G_PARAM_CONSTRUCT)
			);
	g_object_class_install_property (
			g_object_class,
			PROP_VALIGNMENT,
			g_param_spec_enum (
				"valignment",
				"Vertical Alignment",
				// TODO add documentation
				"",
				CRIA_TYPE_VALIGNMENT,
				CRIA_ALIGNMENT_TOP,
				G_PARAM_READWRITE | G_PARAM_CONSTRUCT)
			);
	g_object_class_install_property (
			g_object_class,
			PROP_DEFAULT_TEXT,
			g_param_spec_string (
				"default-text",
				"Default Text",
				// TODO add documentation
				"",
				_("this block has no content"),
				G_PARAM_READWRITE | G_PARAM_CONSTRUCT)
			);
	g_object_class_install_property (
			g_object_class,
			PROP_FONT_FAMILY,
			g_param_spec_string (
				"family",
				"Font Family",
				// TODO add documentation
				"",
				"Sans",
				G_PARAM_READWRITE | G_PARAM_CONSTRUCT)
			);
	g_object_class_install_property (
			g_object_class,
			PROP_FONT_SIZE,
			g_param_spec_double (
				"size",
				"Font Size",
				// TODO add documentation
				"",
				-G_MINDOUBLE,
				G_MAXDOUBLE,
				// TODO add some configurable default size
				0.,
				G_PARAM_READWRITE | G_PARAM_CONSTRUCT)
			);
}

CriaAlignment
cria_layout_block_get_alignment (CriaLayoutBlock *self)
{
	g_return_val_if_fail (CRIA_IS_LAYOUT_BLOCK (self), CRIA_ALIGNMENT_LEFT);

	return self->alignment;
}

const char *
cria_layout_block_get_default_text (CriaLayoutBlock *self)
{
	g_return_val_if_fail (CRIA_IS_LAYOUT_BLOCK (self), NULL);

	return self->default_text;
}

const char *
cria_layout_block_get_font_family (CriaLayoutBlock * self)
{
	// TODO return the default value
	g_return_val_if_fail (CRIA_IS_LAYOUT_BLOCK (self), NULL);

	return self->font_family;
}

gdouble
cria_layout_block_get_font_size (CriaLayoutBlock * self)
{
	// TODO return the default value
	g_return_val_if_fail (CRIA_IS_LAYOUT_BLOCK (self), 0);

	g_debug ("cria_layout_block_get_font_size(): got %f", self->font_size);
	return self->font_size;
}

gdouble
cria_layout_block_get_height (CriaLayoutBlock *self)
{
	g_return_val_if_fail (CRIA_IS_LAYOUT_BLOCK (self), 0);

	return self->height;
}

gdouble
cria_layout_block_get_pos_x (CriaLayoutBlock *self)
{
	g_return_val_if_fail (CRIA_IS_LAYOUT_BLOCK (self), 0);

	return self->pos_x;
}

gdouble
cria_layout_block_get_pos_y (CriaLayoutBlock *self)
{
	g_return_val_if_fail (CRIA_IS_LAYOUT_BLOCK (self), 0);

	return self->pos_y;
}

static void
cria_layout_block_get_property (
		GObject		* object,
		guint		  prop_id,
		GValue		* value,
		GParamSpec	* param_spec)
{
	CriaLayoutBlock	* self;

	self = CRIA_LAYOUT_BLOCK (object);

	switch (prop_id)
	{
	case PROP_TITLE:
		g_value_set_string (
				value,
				self->title);
		break;
	case PROP_POS_X:
		g_value_set_double (
				value,
				self->pos_x);
		break;
	case PROP_POS_X_UNIT:
		g_value_set_enum (
				value,
				self->pos_x_unit);
		break;
	case PROP_POS_Y:
		g_value_set_double (
				value,
				self->pos_y);
		break;
	case PROP_POS_Y_UNIT:
		g_value_set_enum (
				value,
				self->pos_y_unit);
		break;
	case PROP_WIDTH:
		g_value_set_double (
				value,
				self->width);
		break;
	case PROP_WIDTH_UNIT:
		g_value_set_enum (
				value,
				self->width_unit);
		break;
	case PROP_HEIGHT:
		g_value_set_double (
				value,
				self->height);
		break;
	case PROP_HEIGHT_UNIT:
		g_value_set_enum (
				value,
				self->height_unit);
		break;
	case PROP_ALIGNMENT:
		g_value_set_enum (
				value,
				self->alignment);
		break;
	case PROP_VALIGNMENT:
		g_value_set_enum (
				value,
				self->valignment);
		break;
	case PROP_DEFAULT_TEXT:
		g_value_set_string (
				value,
				self->default_text);
		break;
	case PROP_FONT_FAMILY:
		g_value_set_string (
				value,
				self->font_family);
		break;
	case PROP_FONT_SIZE:
		g_debug ("cria_layout_block_get_property(font size): getting...");
		g_value_set_double (
				value,
				self->font_size);
		g_debug ("cria_layout_block_get_property(font size): got %f", self->font_size);
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (
				object,
				prop_id,
				param_spec);
		break;
	}
}

const char*
cria_layout_block_get_title (
		CriaLayoutBlock	* self)
{
	g_return_val_if_fail (CRIA_IS_LAYOUT_BLOCK(self), NULL);
	
	return self->title;
}

GType
cria_layout_block_get_type (void)
{
	static GType	type = 0;

	if (!type)
	{
		const GTypeInfo info = {
			sizeof (CriaLayoutBlockClass),
			NULL,	/* base initializer */
			NULL,	/* base finalizer */
			(GClassInitFunc)cria_layout_block_class_init,
			NULL,	/* class finalizer */
			NULL,	/* class data */
			sizeof (CriaLayoutBlock),
			0,
			NULL,
			0
		};

		type = g_type_register_static (
				G_TYPE_OBJECT,
				"CriaLayoutBlock",
				&info,
				0);
	}

	return type;
}

gdouble
cria_layout_block_get_width (CriaLayoutBlock *self)
{
	g_return_val_if_fail (CRIA_IS_LAYOUT_BLOCK (self), 0);
	
	return self->width;
}

void
cria_layout_block_set_alignment (
		CriaLayoutBlock	* self,
		CriaAlignment	  alignment)
{
	g_return_if_fail (CRIA_IS_LAYOUT_BLOCK (self));

	if (self->alignment != alignment)
	{
		self->alignment = alignment;

		g_object_notify (G_OBJECT (self), "alignment");
	}
}

void
cria_layout_block_set_default_text (
		CriaLayoutBlock * self,
		char const	* text)
{
	g_return_if_fail (CRIA_IS_LAYOUT_BLOCK (self));
	
	if (self->default_text)
		g_free (self->default_text);

	self->default_text = g_strdup (text);

	g_object_notify (
			G_OBJECT (self),
			"default-text");
}

void
cria_layout_block_set_font_family (
		CriaLayoutBlock * self,
		char const	* text)
{
	g_return_if_fail (CRIA_IS_LAYOUT_BLOCK (self));
	
	if (self->font_family)
		g_free (self->font_family);

	self->font_family = g_strdup (text);

	g_object_notify (
			G_OBJECT (self),
			"family");
}

void
cria_layout_block_set_font_size (
		CriaLayoutBlock	* self,
		gdouble		  font_size)
{
	g_return_if_fail (CRIA_IS_LAYOUT_BLOCK (self));

	if (self->font_size != font_size)
	{
		self->font_size = font_size;

		g_debug ("cria_layout_block_set_font_size(%s:%i): font size is %f", __FILE__, __LINE__, font_size);
		g_object_notify (G_OBJECT (self), "size");
	}

	
}

void
cria_layout_block_set_height (
		CriaLayoutBlock	* self,
		gdouble		  height)
{
	g_return_if_fail (CRIA_IS_LAYOUT_BLOCK (self));

	if (self->height != height)
	{
		self->height = height;

		g_object_notify (G_OBJECT (self), "height");
	}

	
}

void
cria_layout_block_set_height_unit (
		CriaLayoutBlock * self,
		CriaUnit          unit)
{
	g_return_if_fail (CRIA_IS_LAYOUT_BLOCK (self));

	if (self->height_unit != unit)
	{
		self->height_unit = unit;

		g_object_notify (G_OBJECT (self), "height-unit");
	}
}
void
cria_layout_block_set_pos_x (
		CriaLayoutBlock	* self,
		gdouble		  pos_x)
{
	g_return_if_fail (CRIA_IS_LAYOUT_BLOCK (self));

	if (self->pos_x != pos_x)
	{
		self->pos_x = pos_x;

		g_object_notify (G_OBJECT (self), "pos-x");
	}
}

void
cria_layout_block_set_pos_x_unit (
		CriaLayoutBlock	* self,
		CriaUnit	  unit)
{
	g_return_if_fail (CRIA_IS_LAYOUT_BLOCK (self));

	if (self->pos_x_unit != unit)
	{
		self->pos_x_unit = unit;

		g_object_notify (G_OBJECT (self), "pos-x-unit");
	}
}

void
cria_layout_block_set_pos_y (
		CriaLayoutBlock	* self,
		gdouble		  pos_y)
{
	g_return_if_fail (CRIA_IS_LAYOUT_BLOCK (self));

	if (self->pos_y != pos_y)
	{
		self->pos_y = pos_y;

		g_object_notify (G_OBJECT (self), "pos-y");
	}
}

void
cria_layout_block_set_pos_y_unit (
		CriaLayoutBlock * self,
		CriaUnit          unit)
{
	g_return_if_fail (CRIA_IS_LAYOUT_BLOCK (self));

	if (self->pos_y_unit != unit)
	{
		self->pos_y_unit = unit;

		g_object_notify (G_OBJECT (self), "pos-x-unit");
	}
}

static void
cria_layout_block_set_property (
		GObject		* object,
		guint		  prop_id,
		const	GValue	* value,
		GParamSpec	* param_spec)
{
	CriaLayoutBlock	* self;
	
	self = CRIA_LAYOUT_BLOCK (object);
	
	switch (prop_id)
	{
	case PROP_TITLE:
		cria_layout_block_set_title (
				self,
				g_value_get_string (value));
		break;
	case PROP_POS_X:
		cria_layout_block_set_pos_x (
				self,
				g_value_get_double (value));
		break;
	case PROP_POS_X_UNIT:
		cria_layout_block_set_pos_x_unit (
				self,
				g_value_get_enum (value));
		break;
	case PROP_POS_Y:
		cria_layout_block_set_pos_y (
				self,
				g_value_get_double (value));
		break;
	case PROP_POS_Y_UNIT:
		cria_layout_block_set_pos_y_unit (
				self,
				g_value_get_enum (value));
		break;
	case PROP_WIDTH:
		cria_layout_block_set_width (
				self,
				g_value_get_double (value));
		break;
	case PROP_WIDTH_UNIT:
		cria_layout_block_set_width_unit (
				self,
				g_value_get_enum (value));
		break;
	case PROP_HEIGHT:
		cria_layout_block_set_height (
				self,
				g_value_get_double (value));
		break;
	case PROP_HEIGHT_UNIT:
		cria_layout_block_set_height_unit (
				self,
				g_value_get_enum (value));
		break;
	case PROP_ALIGNMENT:
		cria_layout_block_set_alignment (
				self,
				g_value_get_enum (value));
		break;
	case PROP_VALIGNMENT:
		cria_layout_block_set_valignment (
				self,
				g_value_get_enum (value));
		break;
	case PROP_DEFAULT_TEXT:
		cria_layout_block_set_default_text (
				self,
				g_value_get_string (value));
		break;
	case PROP_FONT_FAMILY:
		cria_layout_block_set_font_family (
				self,
				g_value_get_string (value));
		break;
	case PROP_FONT_SIZE:
		cria_layout_block_set_font_size (
				self,
				g_value_get_double (value));
		g_debug ("cria_layout_block_set_property(font size): done");
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (
				object,
				prop_id,
				param_spec);
		break;
	}
}

void
cria_layout_block_set_title (
		CriaLayoutBlock	* self,
		const gchar	* title)
{
	g_return_if_fail (CRIA_IS_LAYOUT_BLOCK (self));

	if (self->title != NULL)
		g_free (self->title);

	self->title = g_strdup (title);

	g_debug ("cria_layout_block_set_title(): title ox%x", (guint) self->title);
	g_object_notify (G_OBJECT (self), "title");
}

void
cria_layout_block_set_valignment (
		CriaLayoutBlock	* self,
		CriaVAlignment	  valignment)
{
	g_return_if_fail (CRIA_IS_LAYOUT_BLOCK (self));

	if (self->valignment != valignment)
	{
		self->valignment = valignment;

		g_object_notify (G_OBJECT (self), "valignment");
	}
}

void
cria_layout_block_set_width (
		CriaLayoutBlock	* self,
		gdouble		  width)
{
	g_return_if_fail (CRIA_IS_LAYOUT_BLOCK (self));

	if (self->width != width)
	{
		self->width = width;

		g_object_notify (G_OBJECT (self), "width");
	}
}

void
cria_layout_block_set_width_unit (
		CriaLayoutBlock * self,
		CriaUnit          unit)
{
	g_return_if_fail (CRIA_IS_LAYOUT_BLOCK (self));

	if (self->width_unit != unit)
	{
		self->width_unit = unit;

		g_object_notify (G_OBJECT (self), "width-unit");
	}
}

/** enumeration hadling **/

CriaAlignment
cria_alignment_from_string (char const * string)
{
	CriaAlignment	  retval = CRIA_ALIGNMENT_LEFT;
	
	if (string == NULL)
		g_debug ("cria_alignment_from_string(%s:%i): unknown alignment tag \"%s\"", __FILE__, __LINE__, string);
	else if (!strcmp (string, "left"))
	{
		g_debug ("cria_alignment_from_string(%s:%i): \"%s\" means \"left\"", __FILE__, __LINE__, string);
		retval = CRIA_ALIGNMENT_LEFT;
	}
	else if (!strcmp (string, "right"))
	{
		g_debug ("cria_alignment_from_string(%s:%i): \"%s\" means \"right\"", __FILE__, __LINE__, string);
		retval = CRIA_ALIGNMENT_RIGHT;
	}
	else if (!strcmp (string, "center"))
	{
		g_debug ("cria_alignment_from_string(%s:%i): \"%s\" means \"center\"", __FILE__, __LINE__, string);
		retval = CRIA_ALIGNMENT_CENTER;
	}
	
	return retval;
}

CriaUnit
cria_unit_unit_from_string (char const * string)
{
	char		* start;
	CriaUnit	  retval = CRIA_UNIT_PIXEL;

	g_ascii_strtod (string, &start);

	if (!strcmp (start, "%"))
	{
		retval = CRIA_UNIT_PERCENT;
		g_debug ("cria_unit_unit_from_string(%s): read percent", string);
	}
	else if (!strcmp (start, "px"))
	{
		retval = CRIA_UNIT_PIXEL;
		g_debug ("cria_unit_unit_from_string(%s): read pixel", string);
	}
	else
	{
		g_warning ("Unit \"%s\" not supported", start);
		retval = CRIA_UNIT_PIXEL;
	}


	return retval;
}

gdouble
cria_unit_val_from_string (char const * string)
{
	gdouble	  retval = g_ascii_strtod (string, NULL);

	g_debug ("cria_unit_val_from_string(%s): read %f", string, retval);

	return retval;
}

CriaVAlignment
cria_valignment_from_string (char const * string)
{
#warning implement stub method
	return CRIA_ALIGNMENT_TOP;
}
