/*
Copyright (C)  2005  Daniele Zelante

This file is part of copyblock.

copyblock is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

copyblock is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with copyblock; if not, write to the Free Software
Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/
/*@LICENSE*/
// $Id: copyblock.cxx,v 1.1 2005/11/14 22:06:31 zeldan Exp $

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <time.h>
#include <fcntl.h>
#include <string.h>
#include <assert.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <algorithm>

#ifdef __FreeBSD__
#include <sys/ioctl.h>
#include <sys/disk.h> 
#endif




#ifndef _LP64
	#define OFF_FRM "%lld"
#else
	#define OFF_FRM "%ld"
#endif


static const char * LICENSE =
"Copyright (C)  2005  Daniele Zelante \n"
" \n"    
"This program is free software; you can redistribute it and/or modify \n"
"it under the terms of the GNU General Public License as published by \n"
"the Free Software Foundation; either version 2 of the License, or \n"
"(at your option) any later version. \n"
" \n"
"This program is distributed in the hope that it will be useful, \n"
"but WITHOUT ANY WARRANTY; without even the implied warranty of \n"
"MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the \n"
"GNU General Public License for more details. \n"
" \n"
"You should have received a copy of the GNU General Public License \n"
"along with this program; if not, write to the Free Software \n"
"Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA \n";


static const char * ZELDAN = "\033[1;31mz\033[1;33me\033[1;32ml\033[1;36md\033[1;34ma\033[1;35mn\033[0;39m";


void banner()
{
	fprintf(
			stderr,
			"product : %s \n"
			"version : %s \n"
			"build   : %s \n"
			"author  : [%s] \n"
			"email   : <zeldan@email.it> \n\n"
			,
			PRODUCT,
			RELEASE,
			TIMESTAMP,
			ZELDAN
			);
}
	
void license()
{
	fprintf(stderr,"license : GPLv2 \n\n%s\n",LICENSE);
}

	


void syntax()
{
	fprintf(stderr,"syntax : %s {-r[z]|-w[s]} [-b bsize] [-m msize] [-V] filename \n",PRODUCT);
	fprintf(stderr,"-r[z]    : read from file and writes to stdout ('z' to write sparse)\n");
	fprintf(stderr,"-w[s]    : write to file reading from stdin ('s' to write sync)\n");
	fprintf(stderr,"-b bsize : set blocksize in KiB, defaults to 64 KiB \n");
	fprintf(stderr,"-m msize : set maximum size in KiB to read or write \n");
	fprintf(stderr,"-V       : display version info \n");
	fprintf(stderr,"-L       : display license info \n");
	fprintf(stderr,"filename : block device or regular file to work on \n");
	fprintf(stderr,"\n");
	exit(3);
}


void myfatal(const char * str)
{
	fprintf(stderr,"%s\n",str);
	exit(2);
}
	
void myerror(const char * str)
{
	perror(str);
	exit(1);
}

void logerror(const char * str)
{
	perror(str);
}
		
size_t read_n(const int fd, char * buf, const size_t count)
{
	size_t done = 0;
	while(done<count)
	{
		ssize_t rd = read(fd,buf+done,count-done);
		if(rd==-1) logerror("\nerror: read");
		if(rd<=0) break;
		size_t rdu = static_cast<size_t>(rd);
		assert(rdu<=count);
		if(rdu < count-done)
			fprintf(stderr,"\nwarn: slow read (%zu < %zu)\n",rdu,count-done);
		done += rdu;
	}
	return done;
}

size_t write_n(const int fd, const char * buf, const size_t count)
{
	size_t done = 0;
	while(done<count)
	{
		ssize_t wr = write(fd,buf+done,count-done);
		if(wr==-1) logerror("\nerror: write");
		if(wr<=0) break;
		size_t wru = static_cast<size_t>(wr);
		if(wru < count-done) 
			fprintf(stderr,"\nwarn: slow write (%zu < %zu)\n",wru,count-done);
		done += wru;
	}
	return done;
}




void printsize(const off_t size)
{
	const double sizeK = size/1024.0;
	const double sizeM = sizeK/1024.0;
	const double sizeG = sizeM/1024.0;
		
	fprintf(stderr,"size = " OFF_FRM " B | %.02f KiB | %.02f MiB | %.02f GiB\n",
			size,sizeK,sizeM,sizeG);

}

void printprogress(time_t timebase, const off_t total, const off_t part, off_t * s0, time_t * t0)
{
	const time_t t1 = time(0);

	if(t1!=*t0)
	{
		const int progress = static_cast<int>((100.0*part) / total + 0.5);
		const double speed = static_cast<double>(part-*s0)/(t1-*t0);
		const double speedM = speed/(1024.0*1024.0);

		const double avgspeed = static_cast<double>(part)/(t1-timebase);
		const double avgspeedM = avgspeed/(1024.0*1024.0);

		int eta = static_cast<int>((total-part) / avgspeed);
		const int etaD = eta / (60*60*24);
		eta -= etaD * (60*60*24);
		const int etaH = eta / (60*60);
		eta -= etaH * (60*60);
		const int etaM = eta / 60;
		eta -= etaM * 60;
		const int etaS = eta;
			
		fprintf(stderr,"\r%2d%% : ETA = %dd,%02dh,%02dm,%02ds : AVG_RATE = %.02f MiB/s : RATE = %.02f MiB/s     \r",
			progress,etaD,etaH,etaM,etaS,avgspeedM,speedM);
	
		*s0 = part;
		*t0 = t1;
	}

}

struct fileinfo_t
{
	fileinfo_t() : _ok(false), _regular(false), _sizeable(false), _size(0) {}

	bool _ok;	
	bool _regular;
	bool _sizeable;
	off_t _size;
};


fileinfo_t fileinfo(const int fd)
{
	fileinfo_t fi;
	
	struct stat s;
	if(-1==fstat(fd,&s)) myerror("fstat");

	fi._ok = S_ISREG(s.st_mode) || S_ISBLK(s.st_mode) || S_ISCHR(s.st_mode);
	fi._regular = S_ISREG(s.st_mode);
	fi._size = s.st_size;
	if(fi._size) fi._sizeable = true;

	#ifdef __FreeBSD__
	if(fi._ok && !fi._regular && !fi._sizeable)
	{
		if(-1!=ioctl(fd,DIOCGMEDIASIZE,&fi._size))
			fi._sizeable = true;
	}
	#endif


	if(!fi._sizeable)
	{
		bool ok = true;
		off_t x = lseek(fd,0,SEEK_END);
		if(x==-1) ok=false;
		if(-1==lseek(fd,0,SEEK_SET)) ok=false;
		if(ok)
		{
			fi._size = x;
			if(fi._size) fi._sizeable = true;	
		}
	}

	return fi;
}






bool isdirty(const char * buf, size_t len)
{
	size_t n;
	for(n=0; n<len; ++n)
		if(buf[n]) return true;
	return false;
}

void blockread(const char * filename, const size_t blocksize, const off_t maxsize, bool sparse)
{
	fileinfo_t nfo1 = fileinfo(1);
	if(!nfo1._regular && sparse) 
	{
		fprintf(stderr,"warn: ignoring [-z] when writing to non-regular file \n");
		sparse = false;
	}
	
	const int fdi = open(filename,O_RDONLY);
	if(fdi==-1) myerror("open");
	
	fileinfo_t fdinfo = fileinfo(fdi);
	if(!fdinfo._ok) myfatal("invalid input file type");
	
	off_t srcsize = 0;
		
	if(!fdinfo._sizeable)
	{
		if(maxsize)
			srcsize = maxsize;
		else
			myfatal("[-m size] is mandatory when source file size is unknown");
	}
	else
	{
		srcsize = fdinfo._size;
			
		if(maxsize)
		{
			if(maxsize > srcsize)
				fprintf(stderr,"warn: requested size greater than real size \n");
			if(maxsize < srcsize)
			{
				fprintf(
						stderr,
						"info: truncating size from "OFF_FRM" to "OFF_FRM" \n",
						srcsize,maxsize
						);
				srcsize = maxsize;
			}
		}
	}
		
	printsize(srcsize);
	char * buffer = new char[blocksize];
	time_t t0 = time(0);
	time_t timebase = t0;
	off_t oldpart = 0;
	off_t part = 0;
	
	while(part<srcsize)
	{
		printprogress(timebase,srcsize,part,&oldpart,&t0);

		const size_t rdreq = static_cast<size_t>(std::min(static_cast<off_t>(blocksize), srcsize-part));
		const size_t rd = read_n(fdi, buffer, rdreq);
		
		if(rd<rdreq)
		{
			fprintf(stderr,"\nerror: unexpected end of device \n");
			break;
		}

		part += rd;
		
		if(sparse && rd==blocksize && part<srcsize && !isdirty(buffer,rd))
		{
			if(-1==lseek(1,rd,SEEK_CUR)) 
				myerror("\nseek");
		}
		else
		{
			const size_t wr = write_n(1,buffer,rd);
			if(wr<rd)
			{
				fprintf(stderr,"\nerror: unable to write \n");
				break;
			}
		}
		
	}

	delete[] buffer;
	
}


void blockwrite(const char * filename, const size_t blocksize, const off_t maxsize, int syncr)
{
	int fdo = open(filename,O_RDWR|(syncr ? O_SYNC : 0));
	if(fdo==-1) myerror("open");

	fileinfo_t fdonfo = fileinfo(fdo);
	if(!fdonfo._ok) myfatal("invalid output file type \n");
	
	fileinfo_t nfo0 = fileinfo(0);

	off_t devsize = 0;
	
	if(!fdonfo._sizeable && !nfo0._sizeable && !maxsize)
		myfatal("[-m size] is mandatory when both target and stream sizes are unknown");
		
	if(fdonfo._sizeable && nfo0._sizeable)
	{
		if(fdonfo._size!=nfo0._size)
			fprintf(stderr,"warn: target size is "OFF_FRM", stream size is "OFF_FRM", using smaller one \n", fdonfo._size, nfo0._size);

		devsize=std::min(fdonfo._size,nfo0._size);
	}
	
	if(fdonfo._sizeable && !nfo0._sizeable) devsize=fdonfo._size;
	if(!fdonfo._sizeable && nfo0._sizeable) devsize=nfo0._size;
	

	if(maxsize)
	{
		if(maxsize > devsize)
			fprintf(stderr,"warn: requested size greater than real size \n");
		if(maxsize < devsize)
		{
			fprintf(
				stderr,
				"info: truncating size from "OFF_FRM" to "OFF_FRM" \n"
				,devsize,maxsize
				);
			devsize = maxsize;
		}
	}


	printsize(devsize);
	char * buffer = new char[blocksize];
	time_t t0 = time(0);
	time_t timebase = t0;
	off_t oldpart = 0;
	off_t part = 0;

	while(part<devsize)
	{
		printprogress(timebase,devsize,part,&oldpart,&t0);
		
		const size_t rdreq = static_cast<size_t>(std::min(static_cast<off_t>(blocksize), devsize-part));
		const size_t rd = read_n(0, buffer, rdreq);
		if(rd<rdreq)
		{
			fprintf(stderr,"\nerror: unexpected end of file\n");
			break;
		}

		part += rd;

		size_t wr = write_n(fdo, buffer, rd);
		if(wr<rd)
		{
			fprintf(stderr,"\nerror: unable to write \n");
			break;
		}
				
		if(rd<blocksize) break;

	}

	delete[] buffer;
}	


int main(int argc, char ** argv)
{
	assert(sizeof(off_t)==8);
		
	const char * optstring = "LVrwszb:m:";
	bool writeop = false;
	bool readop = false;
	bool syncr = false;
	bool sparse = false;
	size_t blocksize = 0;
	off_t maxsize = 0;
	
	for(;;)
	{
		const int opt = getopt(argc,argv,optstring);
		if(opt==-1) break;
		if(opt=='V') banner();
		if(opt=='L') license();
		if(opt=='r') readop = true;
		if(opt=='w') writeop = true;
		if(opt=='s') syncr = true;
		if(opt=='z') sparse = true;
		if(opt=='b')
		{
			if(blocksize!=0) syntax();
			blocksize = atoi(optarg) * 1024;
			if(blocksize==0) syntax();
		}
	
		if(opt=='m')
		{
			if(maxsize!=0) syntax();
			maxsize = (static_cast<int64_t>(atoi(optarg))) * 1024;
			if(maxsize==0) syntax();
		}
	}

	if(optind+1!=argc) syntax();
	const char * filename = argv[optind];
	
	if(readop==writeop) syntax();
	if((sparse && writeop) || (readop && syncr)) syntax();
	
	if(blocksize==0) blocksize=64*1024;


	fprintf(stderr,"%s \"%s\" ; blocksize %zu KiB",
			readop ? "reading" : "writing",
			filename,
			blocksize/1024);

	if(maxsize)
		fprintf(stderr," ; maxsize "OFF_FRM" KiB",maxsize/1024);
	if(syncr)
		fprintf(stderr," ; sync");
	
	fprintf(stderr,"\n");

	if(readop)
		blockread(filename,blocksize,maxsize,sparse);
	if(writeop)
		blockwrite(filename,blocksize,maxsize,syncr);

	fprintf(stderr,"\nOK\n");
	
	return 0;
}
	
