﻿program consoleSudoku;
{
Description:
------------
Program name:   consoleSudoku
Actual version: 0.11

The program 'consoleSudoku' is an easy to use Sudoku program for normal 9x9 
Sudoku’s. The program is designed to run in a console.

Features:
- consoleSudoku can generate Sudoku’s in a wide range of difficulties.
- It is also possible to enter a Sudoku field manually.
- To support the manually solving of a Sudoku, the program have optionally
  two methods of checking the input-numbers.
- The program can make a 'sheet' with six Sudoku’s on it, written to a 
  file and ready to print out.
- The program supports the English and the German language.

The program was developed with FreePascal.

You may find some other useful information after the license.


Copyright (C) <2016>  <author: Peter Rossnagel>  <contact: pewrn@t-online.de>
-----------------------------------------------------------------------------
This source is free software; you can redistribute it and/or modify it under
the terms of the GNU General Public License version 3 as published by the Free
Software Foundation.

This code is distributed in the hope that it will be useful, but WITHOUT ANY
WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
details.

A copy of the GNU General Public License is available on the World Wide Web
at <http://www.gnu.org/copyleft/gpl.html>. You can also obtain it by writing
to the Free Software Foundation, Inc., 59 Temple Place - Suite 330, Boston,
MA 02111-1307, USA.
}

{ //begin of the man-page text (without any tags)
<-                                                                     ->
"consoleSudoku" "6" "2016-05-28" "consoleSudoku manual" "games"
#########################################################################
NAME
consoleSudoku - a program for playing sudoku.

SYNOPSIS
consoleSudoku [options]

DESCRIPTION
The program 'consoleSudoku' is an easy to use Sudoku program for normal 9x9
Sudoku’s. This program can generate Sudoku’s in a wide range of difficulties.
The generation process of a Sudoku normally takes only a few seconds.
It is also possible to enter a Sudoku field manually.
To support the manually solving of a Sudoku, the program have optionally
two methods of checking the input-numbers.
The program can also make a 'sheet' with six Sudoku’s on it, written to a
file and ready to print out.
The consoleSudoku program is designed to run in a console.

OPTIONS
-d, -displaymethod [normal|alternative]
Changes the screen-output method.
This maybe useful if problems occurs.
Default display-method is normal.

-h, -help
Show the help message and exit.

-l, -language [ger|eng]
Select a language.
The program supports the English (eng) and the German (ger) language.
Default language is English (eng).

-m, -mode [mark|gray|blue]
How to figure out the Sudoku field.
Tag (mark) the fix numbers with a "!".
Or use a color (gray or blue) and the marker "!" is not used.
Default mode is mark.

-p, -makeprintablesudokus [easy|middle|hard|mixed]
make the file "printme.sudoku.txt" with six generated Sudoku’s, ready to print out.
The difficult level can be selected by:
(easy)    generates difficult levels: 1, 1, 2, 2, 2, 3
(middle)  generates difficult levels: 3, 4, 4, 5, 6, 5
(hard)    generates difficult levels: 5, 6, 6, 7, 7, 8
(mixed)   generates difficult levels: 2, 3, 4, 5, 6, 7
1:(>45) 2:(41-45) 3:(37-40) 4:(34-36) 5:(32-33) 6:(30-31) 7:(28-29) 8:(<=28)
The program knows the difficult levels 1 to 8. For example difficult level 5 means,
a generated Sudoku has 32 or 33 occupied fields.
If no parameter is given, mixed is used as default

EXAMPLES
consoleSudoku
Normally no option is needed to play consoleSudoku game.

consoleSudoku -language ger -m blue
Program uses the German language and to figure out the Sudoku-field a blue color is used.

consoleSudoku -p
Generate six Sudoku’s with parameter mixed as default, written to the file "printme.sudoku.txt".

FILES
"printme.sudoku.txt"
When generated by the option -makeprintablesudokus, the file can be found in the working directory.

COMMANDS
In interactive mode, the following key bindings are available:

   <ESC>, x - (exit, or back to main menu)
   q, e -     (quit, enter)
   g - (generate Sudoku)
   n - (new Sudoku manually)
   +/- (change difficult level)
   s - (solve the Sudoku)
   l - (change language)
   t - (toggle, change checking methods) / (toggle look)

BUGS
Send bug reports to <pewrn@t-online.de>

AUTHOR
The program was written in FreePascal by Peter Roßnagel.
Contact: pewrn@t-online.de
#########################################################################
<-                                                                     ->
} // end of the man-page text



// Beginn: April.2016
// Letzte Aenderung: 31.Mai.2016
//
// Ver.001
// - grundlegende Bildschirmausgabe von Suduko mit Rahmen und Hilfslinien
// - internes TestSudoku
// - Integritaetspruefung des Sudokus
// - Recursiver Loesungsalgorithmus
// Ver.002
// - unbenutzer Code entfernt, Dokumentation vereinheitlicht
// - recursiver Loesungsalgorithmus:
//       - suche freies Feld optimiert
//       - um zweite Loesungsstrategie erweitert
//       - Pruefung auf eindeutige Loesung
// - text_screen_output: Um Anzeige eines Eingabecursors erweitert
// - haendische Eingabe eines neuen Sudokus
// Ver.003
// - Startbildschirm mit Auswahlmenue, ein internes TestSudoku fuer Menue belegt
// - Zufallssuche nach belegten und freien Feld, für die Sudoku-Generierung benoetigt
// - Hauptprog. grundlegender Ablauf erstellt
// - Ausgabesteuerung mittels ScreenMessages
// Ver.004
// - haendisches loesen
// Ver.005
// - generiere Sudoku mit vorgegebenen Schwierigkeitsgrad!
// - Menue fuer Generierung, Auswahl des Schwierigkeitsgrades
// Ver.006
// - Optimierung und Fine-Tuning der Sudoku-Generierung
// Ver.007
// - Umstellung auf mehrsprachen Unterstuetzung!
// Ver.008
// - Englischer Sprachtext eingefuegt
// - Lizenz-Text eingefuegt
// - Hauptprogramm in procedure main_sudoku verschoben
// Ver.009
// - Programmname geändert auf: >>> "consoleSudoku",
// Ver. 0.10, Programmname: consoleSudoku
// - Code fuer das Haendling der Konsole eingefuegt
// - Farbschema
// - Startparameter: -l -language <ger/eng>
//                   -d -displaymethod <normal/alternative>
//                   -m -mode <mark/gray/blue>
// Ver. 0.11
// - Generierung von sechs Sudokus in einer Datei zum Ausdrucken
//           -p -makeprintablesudokus <easy/middle/hard/mixed>
// - Dokumentation vervollstaendigt, Korrekturen   
//
// Formatierung für Listing-Ausdruch unter Office, l-/r-Rand o,5 cm, Schr. 10 Pkt. Spalte 95->



{$mode objfpc}{$H+}                      // Nuetzliche Voreinstellungen
                                         // fuer 'Programme'
uses                                     // sagt die IDE
  {$IFDEF UNIX}{$IFDEF UseCThreads}      //
  cthreads,                              //
  {$ENDIF}{$ENDIF}                       //
  Classes, SysUtils, CustApp,            //

{uses  SysUtils,}                        // fuer: sleep
       Crt;                              // fuer: ClrScr ,gotoXY

const // grundlegende Sudoku Einstellungen
      max_tries_genDifficultLevel= 10;   // Max. Anz. d. Versuche das gef. Sudoku zu generieren
      min_SudokuFields_used      = 23;   // Einfordern einer Mindestanzahl belegter Feldpos.
      screenFlag1_useBlank       = 1;    // <>1 - Bildschirmausg. mit "0" f. unbel. Feldpos.
                                         // 1   - eine leere Stelle
      max_difficult_levels       = 8;    // Anz. der definierten Schwierigkeitsstufen


const // Multilanguage * Mehrsprachen Unterstuetzung
      max_languages              = 2;    // Anz. der unterstuetzten Sprachen
      ger=1; eng=2;
      default_language           = eng;  // gewaehlte Sprache: 1 - deutsch, 2 - engl.
      max_txt                    = 20;   // max. Anz. von Textmitteilungen pro Kategorie
      max_prog_areas             = 7;    // Anzahlt der definierten Prog.-Bereiche
      menu=1; gmen=2; entr=3;            // definiert Prog.-Bereiche mit Ausgabe
      main=4; gene=5; edit=6;            // s.o.
      difc=7;                            // s.o.
      waitMSec_onGeneralSelection=15;    // Schlafzeit in 1/1000 sek. fuer Menues
      waitMSec_onEditSudoku=5;           // Schlafzeit in ms für Sudoku aktiv bearbeiten
      default_mark_fixSudokuPositions=1; // 1 - fixe Zahlen des Sudokus mit „!x“ markiern
      default_color_fixSudokuPositions=0;// 1 - benutze: LightGray, 2 - benutze: Blue,
                                         // 0 - benutze System defaults
      default_displayMethod= 1;          // 0 - erzwinge immer Bildschirmneuaufb. mit: ClrScr
                                         // 1 - regulaerer Bildschirmaufb. ueber: gotoXY
                                         // 2 - scrollen, fuer debug-Zwecke
      default_fileName=    'printme.sudoku.txt'; //sechs gen. Sudokus zum Ausdruck in dieser Datei


const // Alle debugPrintFlags... Manipulieren die Bildschirmausgabe zur Fehlersuche
      // ==========================================================================
      // 0 - ist der Standardwert
      // Wenn debugPrintFlagsgesetzt werden, dann ggf. auch default_displayMethod=2 setzen!
      debugPrintFlag1_integrityErrors= 0; // 1 - Integritaetspruef.: Ausgabe gefundener Fehler
      debugPrintFlag2_integrity_OK=    0; // 1 - Integritaetspruefung: Bildschirmausgabe einer
                                          //     Gut-Mitteilung, falls die Definition erfuellt
      debugPrintFlag4_solve_emptyPos=  0; // 1 - Loesungsalgorithmus (solve):
                                          //     Ausgabe der gefundenen freien Feldposition
      debugPrintFlag5_solve_RecCall=   0; // 1 - Ausgabe der Parameter des rekursiven Aufrufs
      debugPrintFlag6_solve_SolvMsg=   0; // 1 - Ausgabe wenn Sudoku GELOEST wurde!
      debugPrintFlag7_solve_clearly=   0; // 1 - Loes.Alg. (solve_clearly): Loesung mehrdeutig
      debugPrintFlag10_random=         0; // 1 - erlaube Infos -->get_a_random_FieldPosition
      debugPrintFlag12_edit=           0; // 1 - Details zur eingegebenen Taste u. Position
      debugPrintFlag14_randomLine=     0; // 1 - Ausg. von setOneLineWithRandomNumbers
      debugPrintFlag16_generateFound=  0; // 1 - Ausg. wenn ein Sudoku generiert wurde
      debugPrintFlag17_genFoundBetter= 0; // 1 - Ausg. wenn ein besser generiertes gef. wurde
      debugPrintFlag20_parameterFound= 0; // 1 - gibt erkannte uebergebene Startparameter aus

type sudokuField            = array [1..9, 1..9] of integer; // Das Sudokufeld mit Z. 0-9
     integrityErrorField    = array [1..3] of integer;       // Fuer die Fehlerauswertung
     ScreenMessage =    record                               // Fuer die Bildschirmausgabe
                          Head, Status, Hint, KeyInfo, KeyInfo2: string
                        end;
     CursorPosition =   record                               // Der Cursor auf dem Sudoku
                          x, y: integer
                        end;
     oneDifficultLevel= record                               // Schwierigkeitsgrade des S.
                          MsgText: string;
                          lowValue, highValue, callValue: integer
                        end;
     ScrMessages= array [1..max_languages, 1..max_prog_areas,// fuer die Multi-
                         1..max_txt] of ScreenMessage;       // Sprachunterstuezung
     DifficultLevels = array[1..max_difficult_levels] of oneDifficultLevel;

type // TMySudoku, generiert von der IDE fuer Konsole-Anwendung
     TMySudoku = class(TCustomApplication)
     protected
       procedure DoRun; override;
     public
       constructor Create(TheOwner: TComponent); override;
       destructor Destroy; override;
       procedure WriteHelp; virtual;
     end;

//  global genutzte Variablen
//  -------------------------
var lang: integer;                     // Die gewaehlte Sprache
    PositionsExamined   : Real;        // Zaehlt die rekurs. Aufr. des Loesungsalg.
    fixSudoku           : sudokufield; // Das zu loes. Sudoku ohne irgendw. Aenderungen
    scrolling_displayMethod: integer;  // Methode den Bildschirmaufbau durchzufuehren
    mark_fixSudokuPositions: integer;  // markiert ggf. die fixen Zahlen mit !x
    color_fixSudokuPositions: integer; // Farbschema verwenden?

var Application: TMySudoku;            {  generiert von der IDE fuer Konsole-Anwendung }



procedure set_ScreenMessages(var Msg: ScrMessages; language: integer);
// Alle regulaer auszugebenen Mitteilungstexte sind hier zentral zusammengefuehrt!
// >>>> Multilanguage support * Mehrsprachen Unterstuetzung <<<<
// In:  -language: Als Index auf die definierten Mitteilungstexte (Msg)
// Out: -Msg: Die Mitteilungstexte in der gewuenschten Sprache
// Beachte die als globale Konstanten definierte Werte:
//       ger=1;  eng=2;              // derzeit unterstuetzte Sprachen
//       menu=1; gmen=2; entr=3;     // Programmbereiche mit Bildschirmausgabe
//       main=4; gene=5; edit=6;     // s.o.
//       difc=7;                     // s.o.
var i_prog, i_txt, i_lang: integer;

begin
  for i_lang:=1 to max_languages do
    for i_prog:=1 to max_prog_areas do
      for i_txt:=1 to max_txt do
      begin                        // Initialisiere mit einer Fehlermitteilung
        Msg[i_lang,i_prog,i_txt].Head:=    'Fehler bei den Mitteilungstexten';
        Msg[i_lang,i_prog,i_txt].Status:=  'Fehler bei den Mitteilungstexten';
        Msg[i_lang,i_prog,i_txt].Hint:=    'Fehler bei den Mitteilungstexten';
        Msg[i_lang,i_prog,i_txt].KeyInfo:= 'Fehler bei den Mitteilungstexten';
        Msg[i_lang,i_prog,i_txt].KeyInfo2:='Fehler bei den Mitteilungstexten';
      end;

// **************************************************
// *** ger:   Die deutschen Bildschirmausgaben    ***
// **************************************************
// Text-Bildschirmausgaben des Unterprogrammes 'mainMenue'
// Rahmenbreite:            '**************************************'
  Msg[ger,menu,1].Head:=    '>>> >>>> >>> Auswahlmenü  <<< <<<< <<<';
  Msg[ger,menu,1].Status:=  'Stat.: Natürlich kein legales Sudoku! ';
  Msg[ger,menu,1].Hint:=    'Hinw.: l- Sprache, t- Aussehen ändern ';
  Msg[ger,menu,1].KeyInfo:= 'n - Sudoku händisch eingeben          ';
  Msg[ger,menu,1].KeyInfo2:='g - Sudoku generieren lassen          ';

// Text-Bildschirmausgaben des Unterprogrammes 'generiereSudokuMenue'
  Msg[ger,gmen,1].Head:=    '>>> >> >>> Generiere Sudoku <<< << <<<';
  Msg[ger,gmen,1].Status:=  'Stat.: Level: ';
  Msg[ger,gmen,1].Hint:=    'Hinw.: Dies KANN einige Zeit dauern!  ';
  Msg[ger,gmen,2].Hint:=    'Hinw.: Eingabe erwartet...            ';
  Msg[ger,gmen,1].KeyInfo:= '+/- Level ändern, g - generiere       ';
  Msg[ger,gmen,1].KeyInfo2:='<ESC>, x - ENDE                       ';

// Text-Bildschirmausgaben des Unterprogrammes 'enterNewSudoku'
  Msg[ger,entr,1].Head:=    '>> Ein zu lösendes Sudoku eingeben <<';
  Msg[ger,entr,1].Status:=  'Stat.: OK (Feldintegrität)            ';
  //                  z. B. 'Zahl 'x' verletzt Integrität   '
  Msg[ger,entr,2].Status:=  'Stat.: Zahl ';
  Msg[ger,entr,3].Status:=  ' verletzt Integrität   ';
  Msg[ger,entr,4].Status:=  'Stat.: >>>> PRÜFE DIE LÖSBARKEIT! <<<<';
  Msg[ger,entr,5].Status:=  'Stat.: Lösbar, aber nicht eind. lösbar';
  Msg[ger,entr,6].Status:=  'Stat.: >>> >>> NICHT LÖSBAR!!! <<< <<<';
  Msg[ger,entr,7].Status:=  'Stat.: Nicht genuegend Felder belegt! ';
  Msg[ger,entr,1].Hint:=    'Hinw.: Eingabe: 0-9 und Cursor-Tasten ';
  Msg[ger,entr,2].Hint:=    'Hinw.: Andere Zahl oder Feld belegen  ';
  Msg[ger,entr,3].Hint:=    'Hinw.: Dies kann einige Zeit dauern!  ';
  Msg[ger,entr,4].Hint:=    'Hinw.: Weitere Feldbelegungen erf.    ';
  Msg[ger,entr,5].Hint:=    'Hinw.: Änderungen an den Feldbel. erf.';
  Msg[ger,entr,6].Hint:=    'Hinw.: Weitere Felder belegen         ';
  Msg[ger,entr,1].KeyInfo:= 'e, q     - Eingabe abschließen        ';
  Msg[ger,entr,1].KeyInfo2:='<ESC>, x - alle Eingaben verwerfen    ';

// Text-Bildschirmausgaben des Unterprogrammes 'main_sudoku'
  Msg[ger,main,1].Head:=    '';
  Msg[ger,main,1].Status:=  'Stat.: ';
  Msg[ger,main,1].Hint:=    '>>> Die SUDOKUS befinden sich in der Datei: ';
  Msg[ger,main,1].KeyInfo:= '';
  Msg[ger,main,1].KeyInfo2:='';

// Text-Bildschirmausgaben des Unterprogrammes 'gen_sudoku_with_difficult_level'
  Msg[ger,gene,1].Head:=    '>>> >> >>> Generiere Sudoku <<< << <<<';
  Msg[ger,gene,1].Status:=  'Stat.: Suche:                         ';
  Msg[ger,gene,2].Status:=  'Stat.: Suche: ';
  Msg[ger,gene,1].Hint:=    'Hinw.:       - Bitte warten -         ';
  //                 z. B.: 'Vers.: 1/10, akt.: 30, best: 23'
  Msg[ger,gene,2].Hint:=    'Hinw.: Ver.: ';
  Msg[ger,gene,3].Hint:=    ', akt.: ';
  Msg[ger,gene,4].Hint:=    ', best: ';
  Msg[ger,gene,1].KeyInfo:= 'Hinw.: Dies kann einige Zeit dauern!  ';
  Msg[ger,gene,1].KeyInfo2:='Hinw.:       - Bitte warten -         ';

// Text-Bildschirmausgaben des Unterprogrammes 'edit_to_solve_sudoku'
  Msg[ger,edit,1].Head:=    '>>> >>> >>>  Sudoku lösen  <<< <<< <<<';
  Msg[ger,edit,1].Status:=  'Stat.: >>>> PRÜFE DIE LÖSBARKEIT! <<<<';
  Msg[ger,edit,2].Status:=  'Stat.: Ungel., keine Eingabeprüfung   ';
  Msg[ger,edit,3].Status:=  'Stat.: Ungel., Eing.-Integritätsprüf. ';
  Msg[ger,edit,4].Status:=  'Stat.: Ungel., Eing.-Korrektheitsprüf.';
  Msg[ger,edit,5].Status:=  'Stat.: Glückwunsch! Lösung gefunden.  ';
  Msg[ger,edit,6].Status:=  'Stat.: Computerlösung                 ';
  Msg[ger,edit,1].Hint:=    'Hinw.: Dies KANN einige Zeit dauern!  ';
  Msg[ger,edit,2].Hint:=    'Hinw.: Sudoku ist eindeutig lösbar    ';
  //                  z. B. 'Felder belegt: 30, frei: 51 '
  Msg[ger,edit,3].Hint:=    'Hinw.: Felder belegt: ';
  Msg[ger,edit,4].Hint:=    ', frei: ';
  //                  z. B. 'Zahl x verletzt Integrität   '
  Msg[ger,edit,5].Hint:=    'Hinw.: Zahl ';
  Msg[ger,edit,6].Hint:=    ' verletzt Integrität   ';
  Msg[ger,edit,7].Hint:=    'Hinw.: Feld belegen                   ';
  //                  z. B. 'Zahl 5 verletzt Korrektheit!'
  Msg[ger,edit,8].Hint:=    'Hinw.: Zahl ';
  Msg[ger,edit,9].Hint:=    ' verletzt Korrektheit!';
  Msg[ger,edit,10].Hint:=   'Hinw.: Feld belegen    ';
  Msg[ger,edit,11].Hint:=   'Hinw.: Einige Eing. wurden korrigiert ';
  Msg[ger,edit,12].Hint:=   'Hinw.: =============================  ';
  Msg[ger,edit,13].Hint:=   'Hinw.: Mehr Glück nächstes Mal        ';
  Msg[ger,edit,1].KeyInfo:= 's - lösen, t - Eingabeprüfung ändern  ';
  Msg[ger,edit,2].KeyInfo:= 'e, q     - Hauptmenü                  ';
  Msg[ger,edit,1].KeyInfo2:='Eingabe: 0 - 9 und Cursor-Tasten      ';
  Msg[ger,edit,2].KeyInfo2:='<ESC>, x - Rückkehr zum Hauptmenü     ';

 // Text-Mitteilungen des Unterprogrammes 'set_difficult_level_definitions'
  Msg[ger,difc,1].Status:=  'sehr leicht ';
  Msg[ger,difc,2].Status:=  'leicht  -   ';
  Msg[ger,difc,3].Status:=  'leicht  -   ';
  Msg[ger,difc,4].Status:=  'mittel  -   ';
  Msg[ger,difc,5].Status:=  'mittel  -   ';
  Msg[ger,difc,6].Status:=  'schwer  -   ';
  Msg[ger,difc,7].Status:=  'sehr schwer ';
  Msg[ger,difc,8].Status:=  'sehr schwer ';


  // **************************************************
  // *** eng:   The english screen messages         ***
  // **************************************************
  // screen messages from the subroutines 'mainMenue'
  // Rahmenbreite:            '**************************************'
  Msg[eng,menu,1].Head:=    '>>> >>> >>> Selection menu <<< <<< <<<';
  Msg[eng,menu,1].Status:=  'Stat.: NOT a legal Sudoku!            ';
  Msg[eng,menu,1].Hint:=    'Hint: l-Change language, t-Toggle look';
  Msg[eng,menu,1].KeyInfo:= 'n - Enter a new Sudoku                ';
  Msg[eng,menu,1].KeyInfo2:='g - Generate a Sudoku                 ';

  // screen messages from the subroutines 'generiereSudokuMenue'
  Msg[eng,gmen,1].Head:=    '>>> >> >> Generate a Sudoku << <<< <<<';
  Msg[eng,gmen,1].Status:=  'Stat.: Level: ';
  Msg[eng,gmen,1].Hint:=    'Hint:  This could take a while!       ';
  Msg[eng,gmen,2].Hint:=    'Hint:  Waiting for input...           ';
  Msg[eng,gmen,1].KeyInfo:= '+/- Change level, g - Generate        ';
  Msg[eng,gmen,1].KeyInfo2:='<ESC>, x - EXIT                       ';

 // screen messages from the subroutines  'enterNewSudoku'
  Msg[eng,entr,1].Head:=    '>>>> >>>> Enter a new Sudoku <<<< <<<<';
  Msg[eng,entr,1].Status:=  'Stat.: OK (field-integrity)           ';
  //                  z. B. 'Zahl 'x' verletzt Integrität   '
  Msg[eng,entr,2].Status:=  'Stat.: Number ';
  Msg[eng,entr,3].Status:=  ' hurt integrity      ';
  Msg[eng,entr,4].Status:=  'Stat.: > Check: Can Sudoku be solved?<';
  Msg[eng,entr,5].Status:=  'Stat.: Can be solved, but not unique  ';
  Msg[eng,entr,6].Status:=  'Stat.: >> >>> cannot solved !!! <<< <<';
  Msg[eng,entr,7].Status:=  'Stat.: Too many free field-positions! ';
  Msg[eng,entr,1].Hint:=    'Hint:  Use: 0-9 and cursor-keys       ';
  Msg[eng,entr,2].Hint:=    'Hint:  Choose another field or number ';
  Msg[eng,entr,3].Hint:=    'Hint:  >>> This CAN take a while! <<< ';
  Msg[eng,entr,4].Hint:=    'Hint:  Occupy more field-positions    ';
  Msg[eng,entr,5].Hint:=    'Hint:  Some positions have to change  ';
  Msg[eng,entr,6].Hint:=    'Hint:  Occupy more field-positions    ';
  Msg[eng,entr,1].KeyInfo:= 'e, q   - accept input                 ';
  Msg[eng,entr,1].KeyInfo2:='<ESC>, x - Exit, dismiss input        ';

  // screen messages from the subroutines 'main_sudoku'
  Msg[eng,main,1].Head:=    '';
  Msg[eng,main,1].Status:=  'Stat.: ';
  Msg[eng,main,1].Hint:=    '>>> The SUDOKUS can be found in the file: ';
  Msg[eng,main,1].KeyInfo:= '';
  Msg[eng,main,1].KeyInfo2:='';

  // screen messages from the subroutines 'gen_sudoku_with_difficult_level'
  Msg[eng,gene,1].Head:=    '>>> >> >> Generate a Sudoku << <<< <<<';
  Msg[eng,gene,1].Status:=  'Stat.: Searching:                     ';
  Msg[eng,gene,2].Status:=  'Stat.: Searching: ';
  Msg[eng,gene,1].Hint:=    'Hint:        - Please wait -          ';
  //                 z. B.: 'Vers.: 1/10, akt.: 30, best: 23'
  Msg[eng,gene,2].Hint:=    'Hint: Search: ';
  Msg[eng,gene,3].Hint:=    ', act.: ';
  Msg[eng,gene,4].Hint:=    ', best: ';
  Msg[eng,gene,1].KeyInfo:= 'Hint: This can take a while!          ';
  Msg[eng,gene,1].KeyInfo2:='Hint:        - Please wait -          ';

  // screen messages from the subroutines 'edit_to_solve_sudoku'
  Msg[eng,edit,1].Head:=    '>>> >>> >> Solve the Sudoku << <<< <<<';
  Msg[eng,edit,1].Status:=  'Stat.: >>> Checking for a solution <<<';
  Msg[eng,edit,2].Status:=  'Stat.: Unsolved, no checking on input ';
  Msg[eng,edit,3].Status:=  'Stat.: Unsolved, integrity-check is on';
  Msg[eng,edit,4].Status:=  'Stat.: Unsolved, check correctness    ';
  Msg[eng,edit,5].Status:=  'Stat.: Congratulation! Solution found!';
  Msg[eng,edit,6].Status:=  'Stat.: Computer give this solution    ';
  Msg[eng,edit,1].Hint:=    'Hint:  This could take a while!       ';
  Msg[eng,edit,2].Hint:=    'Hint:  Sudoku has one unique solution ';
  //                  z. B. 'Felder belegt: 30, frei: 51 '
  Msg[eng,edit,3].Hint:=    'Hint:  Fields occupied: ';
  Msg[eng,edit,4].Hint:=    ', free: ';
  //                  z. B. 'Zahl x verletzt Integrität   '
  Msg[eng,edit,5].Hint:=    'Hint:  Number ';
  Msg[eng,edit,6].Hint:=    ' hurt integrity      ';
  Msg[eng,edit,7].Hint:=    'Hint:  Occupy this field-position     ';
  //                  z. B. 'Zahl 5 verletzt Korrektheit!'
  Msg[eng,edit,8].Hint:=    'Hint:  Number ';
  Msg[eng,edit,9].Hint:=    ' hurt correctness!';
  Msg[eng,edit,10].Hint:=   'Hint:  Occupy this field-position     ';
  Msg[eng,edit,11].Hint:=   'Hint: Some entries have been corrected';
  Msg[eng,edit,12].Hint:=   'Hint:  ===============================';
  Msg[eng,edit,13].Hint:=   'Hint:  More luck next time...         ';
  Msg[eng,edit,1].KeyInfo:= 's - Solve, t - Toggle input mode      ';
  Msg[eng,edit,2].KeyInfo:= 'e, q     - Go back to main menu       ';
  Msg[eng,edit,1].KeyInfo2:='Input: 0 - 9 and cursor-keys          ';
  Msg[eng,edit,2].KeyInfo2:='<ESC>, x - EXIT to main menu          ';

  // Text messages from the subroutines 'set_difficult_level_definitions'
  Msg[eng,difc,1].Status:=  'very easy - ';
  Msg[eng,difc,2].Status:=  'easy    -   ';
  Msg[eng,difc,3].Status:=  'easy    -   ';
  Msg[eng,difc,4].Status:=  'middle   -  ';
  Msg[eng,difc,5].Status:=  'middle   -  ';
  Msg[eng,difc,6].Status:=  'hard    -   ';
  Msg[eng,difc,7].Status:=  'very hard - ';
  Msg[eng,difc,8].Status:=  'very hard - ';

end;



procedure set_difficult_level_definitions(var difficult: DifficultLevels);
// Definiert die moeglichen Schwierigkeitsgrade,
// Gibt die Vorgaben fuer die Generierung von Sudokus.
// Zum Vergleich in Zeitschrift: 44-leicht, 32-mittel, 28-knifflig, 28-raffiniert
// In:  - difficult: unberuecksichtigt
//      - lang (global): Index der gewuenschen Sprache
// Out: - difficult: Die definierten Schwierigkeitsstufen
//
// Hinweis: - max_difficult_levels (global): Die maximale Anzahl der definierten
//            Schwierigkeitsstufen ist als Konstante hinterlegt.
var Msg: ScrMessages;
begin
  set_ScreenMessages(Msg, lang);

  difficult[1].callValue:= 46;
  difficult[1].lowValue := 46; difficult[1].highValue:= 60;
  difficult[1].MsgText  := Msg[lang,difc,1].Status     //'sehr leicht '
                         +' (>45)  ' + '     ';

  difficult[2].callValue:= 42;
  difficult[2].lowValue := 41; difficult[2].highValue:= 45;
  difficult[2].MsgText  := Msg[lang,difc,2].Status     //'leicht      '
                         +' (41-45)' + '     ';

  difficult[3].callValue:= 38;
  difficult[3].lowValue := 37; difficult[3].highValue:= 40;
  difficult[3].MsgText  := Msg[lang,difc,3].Status     //'leicht      '
                         +' (37-40)' + '     ';

  difficult[4].callValue:= 34;
  difficult[4].lowValue := 34; difficult[4].highValue:= 36;
  difficult[4].MsgText  := Msg[lang,difc,4].Status     //'mittel      '
                         +' (34-36)' + '     ';

  difficult[5].callValue:= 32;
  difficult[5].lowValue := 32; difficult[5].highValue:= 33;
  difficult[5].MsgText  := Msg[lang,difc,5].Status     //'mittel      '
                         +' (32-33)' + '     ';

  difficult[6].callValue:= 30;
  difficult[6].lowValue := 30; difficult[6].highValue:= 31;
  difficult[6].MsgText  := Msg[lang,difc,6].Status     //'schwer      '
                         +' (30-31)' + '     ';

  difficult[7].callValue:= 28;
  difficult[7].lowValue := 28; difficult[7].highValue:= 29;
  difficult[7].MsgText  := Msg[lang,difc,7].Status     //'sehr schwer '
                         +' (28-29)' + '     ';

  difficult[8].callValue:= 15;
  difficult[8].lowValue := 15; difficult[8].highValue:= 28;
  difficult[8].MsgText  := Msg[lang,difc,8].Status     //'sehr schwer '
                         +' (<=28) ' + '     ';
end;



procedure load_internal_Sudoku(field_nr: integer; var s: sudokuField);
// Lade ein internes vordefiniertes Sudokufeld
// Das Feld Nr. 0 und 1 wird waehrend des Menues angezeigt!
// Felder Nr. 2 und 3 dienten zu Test und Pruefzwecke waehrend der Entwicklung!
// In:  - field_nr: Die Indexnummer des zu ladenden vordefinierten Sudukufeldes
// Out: - sudokuField: Das ueber die field_nr indizierte, vordefinierte Sudokufeld

begin
  if field_nr=0 then
  begin
    // Fuers Menue, kein legales Sudoku Spielfeld
    s[1,9]:=1;s[2,9]:=1;s[3,9]:=1;s[4,9]:=2;s[5,9]:=2;s[6,9]:=2;s[7,9]:=3;s[8,9]:=3;s[9,9]:=3;
    s[1,8]:=1;s[2,8]:=1;s[3,8]:=1;s[4,8]:=2;s[5,8]:=2;s[6,8]:=2;s[7,8]:=3;s[8,8]:=3;s[9,8]:=3;
    s[1,7]:=1;s[2,7]:=1;s[3,7]:=1;s[4,7]:=2;s[5,7]:=2;s[6,7]:=2;s[7,7]:=3;s[8,7]:=3;s[9,7]:=3;
    s[1,6]:=4;s[2,6]:=4;s[3,6]:=4;s[4,6]:=5;s[5,6]:=5;s[6,6]:=5;s[7,6]:=6;s[8,6]:=6;s[9,6]:=6;
    s[1,5]:=4;s[2,5]:=4;s[3,5]:=4;s[4,5]:=5;s[5,5]:=5;s[6,5]:=5;s[7,5]:=6;s[8,5]:=6;s[9,5]:=6;
    s[1,4]:=4;s[2,4]:=4;s[3,4]:=4;s[4,4]:=5;s[5,4]:=5;s[6,4]:=5;s[7,4]:=6;s[8,4]:=6;s[9,4]:=6;
    s[1,3]:=7;s[2,3]:=7;s[3,3]:=7;s[4,3]:=8;s[5,3]:=8;s[6,3]:=8;s[7,3]:=9;s[8,3]:=9;s[9,3]:=9;
    s[1,2]:=7;s[2,2]:=7;s[3,2]:=7;s[4,2]:=8;s[5,2]:=8;s[6,2]:=8;s[7,2]:=9;s[8,2]:=9;s[9,2]:=9;
    s[1,1]:=7;s[2,1]:=7;s[3,1]:=7;s[4,1]:=8;s[5,1]:=8;s[6,1]:=8;s[7,1]:=9;s[8,1]:=9;s[9,1]:=9;

  end; if field_nr=1 then
  begin
    // Fuers Menue, kein legales Sudoku Spielfeld
    s[1,9]:=0;s[2,9]:=0;s[3,9]:=0;s[4,9]:=0;s[5,9]:=0;s[6,9]:=0;s[7,9]:=0;s[8,9]:=0;s[9,9]:=0;
    s[1,8]:=0;s[2,8]:=1;s[3,8]:=0;s[4,8]:=0;s[5,8]:=2;s[6,8]:=0;s[7,8]:=0;s[8,8]:=3;s[9,8]:=0;
    s[1,7]:=0;s[2,7]:=0;s[3,7]:=0;s[4,7]:=0;s[5,7]:=0;s[6,7]:=0;s[7,7]:=0;s[8,7]:=0;s[9,7]:=0;
    s[1,6]:=0;s[2,6]:=0;s[3,6]:=0;s[4,6]:=5;s[5,6]:=5;s[6,6]:=5;s[7,6]:=0;s[8,6]:=0;s[9,6]:=0;
    s[1,5]:=0;s[2,5]:=4;s[3,5]:=0;s[4,5]:=5;s[5,5]:=5;s[6,5]:=5;s[7,5]:=0;s[8,5]:=6;s[9,5]:=0;
    s[1,4]:=0;s[2,4]:=0;s[3,4]:=0;s[4,4]:=5;s[5,4]:=5;s[6,4]:=5;s[7,4]:=0;s[8,4]:=0;s[9,4]:=0;
    s[1,3]:=0;s[2,3]:=0;s[3,3]:=0;s[4,3]:=0;s[5,3]:=0;s[6,3]:=0;s[7,3]:=0;s[8,3]:=0;s[9,3]:=0;
    s[1,2]:=0;s[2,2]:=7;s[3,2]:=0;s[4,2]:=0;s[5,2]:=8;s[6,2]:=0;s[7,2]:=0;s[8,2]:=9;s[9,2]:=0;
    s[1,1]:=0;s[2,1]:=0;s[3,1]:=0;s[4,1]:=0;s[5,1]:=0;s[6,1]:=0;s[7,1]:=0;s[8,1]:=0;s[9,1]:=0;

  end; if field_nr=2 then
  begin
    // Ein legales mittelschweres Sudoku
    s[1,9]:=0;s[2,9]:=0;s[3,9]:=6;s[4,9]:=8;s[5,9]:=0;s[6,9]:=0;s[7,9]:=4;s[8,9]:=7;s[9,9]:=0;
    s[1,8]:=0;s[2,8]:=0;s[3,8]:=0;s[4,8]:=0;s[5,8]:=0;s[6,8]:=0;s[7,8]:=8;s[8,8]:=0;s[9,8]:=0;
    s[1,7]:=7;s[2,7]:=2;s[3,7]:=8;s[4,7]:=3;s[5,7]:=0;s[6,7]:=0;s[7,7]:=0;s[8,7]:=0;s[9,7]:=0;
    s[1,6]:=0;s[2,6]:=0;s[3,6]:=0;s[4,6]:=2;s[5,6]:=0;s[6,6]:=0;s[7,6]:=7;s[8,6]:=0;s[9,6]:=0;
    s[1,5]:=4;s[2,5]:=0;s[3,5]:=2;s[4,5]:=0;s[5,5]:=5;s[6,5]:=7;s[7,5]:=0;s[8,5]:=0;s[9,5]:=8;
    s[1,4]:=0;s[2,4]:=6;s[3,4]:=7;s[4,4]:=0;s[5,4]:=8;s[6,4]:=9;s[7,4]:=0;s[8,4]:=0;s[9,4]:=5;
    s[1,3]:=0;s[2,3]:=0;s[3,3]:=9;s[4,3]:=0;s[5,3]:=4;s[6,3]:=1;s[7,3]:=2;s[8,3]:=5;s[9,3]:=0;
    s[1,2]:=0;s[2,2]:=0;s[3,2]:=1;s[4,2]:=6;s[5,2]:=2;s[6,2]:=0;s[7,2]:=0;s[8,2]:=0;s[9,2]:=0;
    s[1,1]:=0;s[2,1]:=7;s[3,1]:=0;s[4,1]:=0;s[5,1]:=9;s[6,1]:=0;s[7,1]:=0;s[8,1]:=1;s[9,1]:=0;

  end; if field_nr=3 then
  begin
    // Das Sudodu #2 mit eingebauten Fehlern: Position (2,8) Wert von 0 auf 8 geaendert
    //                                        Position (1,9) Wert von 0 auf 7 geaendert
    s[1,9]:=7;s[2,9]:=0;s[3,9]:=6;s[4,9]:=8;s[5,9]:=0;s[6,9]:=0;s[7,9]:=4;s[8,9]:=7;s[9,9]:=0;
    s[1,8]:=0;s[2,8]:=8;s[3,8]:=0;s[4,8]:=0;s[5,8]:=0;s[6,8]:=0;s[7,8]:=8;s[8,8]:=0;s[9,8]:=0;
    s[1,7]:=7;s[2,7]:=2;s[3,7]:=8;s[4,7]:=3;s[5,7]:=0;s[6,7]:=0;s[7,7]:=0;s[8,7]:=0;s[9,7]:=0;
    s[1,6]:=0;s[2,6]:=0;s[3,6]:=0;s[4,6]:=2;s[5,6]:=0;s[6,6]:=0;s[7,6]:=7;s[8,6]:=0;s[9,6]:=0;
    s[1,5]:=4;s[2,5]:=0;s[3,5]:=2;s[4,5]:=0;s[5,5]:=5;s[6,5]:=7;s[7,5]:=0;s[8,5]:=0;s[9,5]:=8;
    s[1,4]:=0;s[2,4]:=6;s[3,4]:=7;s[4,4]:=0;s[5,4]:=8;s[6,4]:=9;s[7,4]:=0;s[8,4]:=0;s[9,4]:=5;
    s[1,3]:=0;s[2,3]:=0;s[3,3]:=9;s[4,3]:=0;s[5,3]:=4;s[6,3]:=1;s[7,3]:=2;s[8,3]:=5;s[9,3]:=0;
    s[1,2]:=0;s[2,2]:=0;s[3,2]:=1;s[4,2]:=6;s[5,2]:=2;s[6,2]:=0;s[7,2]:=0;s[8,2]:=0;s[9,2]:=0;
    s[1,1]:=0;s[2,1]:=7;s[3,1]:=0;s[4,1]:=0;s[5,1]:=9;s[6,1]:=0;s[7,1]:=0;s[8,1]:=1;s[9,1]:=0;
  end;
end;



procedure clear_ScreenMessages(var ScrMsg: ScreenMessage);
// Loescht alle Bildschirmausgabetexte/Mitteilungstexte.
// ScreenMessages -> Head, Status, Hint, KeyInfo, KeyInfo2
// Out: - ScrMsg: Die geloeschen Mitteilungstexte
begin
// Rahmenbreite:   '**************************************'
  ScrMsg.Head:=    '                                      ';
  ScrMsg.Status:=  '                                      ';
  ScrMsg.Hint:=    '                                      ';
  ScrMsg.KeyInfo:= '                                      ';
  ScrMsg.KeyInfo2:='                                      ';
end;



procedure clear_sudoku(var sudoku: sudokuField);
// Initialisiere (loesche) das Sudoku-Feld.
// out: - sudoku: mit Null belegtes Sudoku-Feld,
//                Null entspricht einer leeren/freien Position im Sudoku.
var     x, y         : integer;
begin
  for y:= 1 to 9 do
    for x:= 1 to 9 do
      sudoku[x,y]:= 0;
end;



function count_fields_used(sudoku: sudokuField):integer;
// Zaehlt wieviele Sudoku-Felder belegt sind.
// -> Dies gibt z.B. Auskunft ueber den Schwierigkeitsgrad eines loesbaren Sudokus.
// In: - sudoku: belegt mit 0 bis 9, Null entspr. einer freien Position im Sudoku.
// Out: - count_fields_used: wieviele Feldpositionen im Sudoku sind belegt (nicht leer).
var     x, y         : integer;
begin
  count_fields_used:=0;
  for y:= 1 to 9 do
    for x:= 1 to 9 do
      if (sudoku[x,y] <> 0) then count_fields_used:=count_fields_used+1;
end;



function count_fields_free(sudoku: sudokuField):integer;
// Zaehlt wieviele Sudoku-Felder belegt sind.
// In: - sudoku: belegt mit 0 bis 9, Null entspr. einer freien Position im Sudoku.
// Out: - count_fields_used: wieviele Feldpositionen im Sudoku sind leer.
var     x, y         : integer;
begin
  count_fields_free:=0;
  for y:= 1 to 9 do
    for x:= 1 to 9 do
      if (sudoku[x,y] = 0) then count_fields_free:=count_fields_free+1;
end;



function are_Sudokus_same(sudoku1, sudoku2 : sudokuField): boolean;
// Vergleicht zwei Sudoku-Felder ob diese identisch sind.
// In:  - sudoku1, sudoku2: Sudokufelder, belegt mit 0 bis 9
// Out: - are_Sudokus_same: Wahrheitswert (wahr/falsch), ob die beiden zur Pruefung
//                        uebergebenen Sudokus gleich sind
var     x, y         : integer;
begin
  are_Sudokus_same:= true;
  for y:= 1 to 9 do
    for x:= 1 to 9 do
      if ((sudoku1[x,y]) <> (sudoku2[x,y])) then are_Sudokus_same:=false;
end;



procedure copySudoku(sudoku: sudokuField; var copie_sudoku: sudokuField);
// Fertigt eine Kopie eines Sudoku-Feldes an.
// In:  - sudoku: Sudokufeld, belegt mit 0 bis 9.
// Out: - copie_sudoku: Eine Kopie des mit 'sudoku' uebergebenen Sudoku-Feldes.
var x,y:integer;
begin
  for x:=1 to 9 do
    for y:=1 to 9 do
      copie_sudoku[x,y]:=sudoku[x,y];
end;



procedure get_a_random_FieldPosition(sudoku: sudokuField; FreeOrUsed: integer;
                                     var x_SudokuFieldIndex, y_SudokuFieldIndex: integer);
// Sucht per Zufallsverfahren, je nach Aufruf, ein leeres bzw. belegtes Feld
// In: - sudoku: Sudokufeld, belegt mit Zahlen von 0 bis 9
//     - FreeOrUsed: free=0, used=1, suche nach einem freien oder belegten Feldplatz
// Out: - x_SudokuFieldIndex, y_SudokuFieldIndex:
//        Die Position des zufaellig ermittelten freien bzw. belegten Platzes
// Wenn Rueckgabe: x_Sudoku.../y_Sudoku... = 0, dann war kein freier bzw.
//                 belegter Feldplatz vorhanden.
// Aufruf ueber: --> get_FREE_random_FieldPosition oder get_USED_random_FieldPosition
const free=0;
var found: boolean;
    x, y:  integer;
    count: integer;
begin
//randomize;  // Aufruf im Hauptprogramm(main) zwingend noetig!
  x_SudokuFieldIndex:=0; y_SudokuFieldIndex:=0;
  x:=0; y:=0;
  found:=false;

  // Gibt es auf dem Sudokufeld ueberhaupt eine geeignete Feldposition?
  if FreeOrUsed=free then count:= count_fields_free(sudoku)
  else count:= count_fields_used(sudoku);

  if (count>=1) then
  begin
    x:=round((random*9)+0.5);
    y:=round((random*9)+0.5);

    while not(found) do
    begin
      // Pruefe und korrigiere die Indizes
      if x<1 then x:=1;
      if y<1 then y:=1;
      if (x>9) then
      begin
        x:=1; y:=y+1;
      end;
      if (y>9) then
      begin
        x:=1; y:=1;
      end;
      if debugPrintFlag10_random=1 then
        writeln('--------> Feldzaehlung: ',count, ', Zufallsindizes: x=', x, ' y=', y);
      if (((FreeOrUsed=free) and (sudoku[x,y]=free))
        or ((FreeOrUsed<>free) and (sudoku[x,y]>free))) then
      begin
        found:=true;
        x_SudokuFieldIndex:=x; y_SudokuFieldIndex:=y;
      end
      else x:=x+1;
    end;
  end;
end;



procedure get_FREE_random_FieldPosition(sudoku: sudokuField; var x, y: integer);
// Sucht per Zufallsverfahren ein leeres Feld, und gibt die Position zurueck
// In:  - sudoku: Sudokufeld, belegt mit Zahlen zwischen 0 bis 9
// Out: - x, y: Als Index auf den zufaellig ermittelten freien Feldplatzes
//        Wenn x /y = 0, dann war kein freier Feldplatz vorhanden.
const free=0;
begin
  get_a_random_FieldPosition(sudoku, free, x, y);
end;



procedure get_USED_random_FieldPosition(sudoku: sudokuField; var x, y: integer);
// Sucht per Zufallsverfahren ein belegtes Feld, und gibt die Position zurueck
// In:  - sudoku: Sudokufeld, belegt mit Zahlen zwischen 0 bis 9
// Out: - x, y: Als Index auf den zufaellig ermittelten belegten Feldplatzes
//        Wenn x /y = 0, dann war kein belegter Feldplatz vorhanden.
const used=1;
begin
  get_a_random_FieldPosition(sudoku, used, x,y);
end;



procedure text_screen_output(sudoku, fixSudoku: sudokuField;
                             cursor: cursorPosition; ScrMsg: ScreenMessage);
// Bildschirmausgabe des Sudoku-Feldes mit Rahmen und Hilfslinien und Cursor.
// Angezeigt wird das ueber 'sudoku' uebergebene Sudoku.
// ScrMsg ->Head, Status, Hint, KeyInfo, KeyInfo2
// In:  - sudoku: Sudokufeld, belegt mit Zahlen zwischen 0 bis 9
//                hier wird das bearbeitete, veraenderte Sudoku uebergeben
//      - fixSudoku: Das Sudokufeld, belegt mit Zahlen zwischen 0 bis 9
//                hier wird das unveraenderte Sudoku, das es zu loesen gilt
//      - cursor  .x/.y > 0 dann wird der Cursor auf dem Sudokufeld angezeigt
//      - ScrMsg: Die auszugebenden Mitteilungstexte
// Hinweis: Es werden zusaetzlich globale Variablen ausgewertet:
// - scrolling_displayMethod
// - color_fixSudokuPositions
// - mark_fixSudokuPositions
var x, y    : integer;
    useColor: integer;
begin
  if (scrolling_displayMethod=0) then ClrScr                        //
  else if (scrolling_displayMethod=1) then gotoXY(1,1);             //
                                                                    //
  useColor:=Black;                                                  // verwende
  if (color_fixSudokuPositions>0) then   // verwende ein Farbschema // globale
  begin                                                             // Settings
    TextBackground(Black);                                          //
    TextColor(White);                                               //
    if (color_fixSudokuPositions=1) then useColor:=DarkGray        //
    else if (color_fixSudokuPositions=2) then useColor:=Blue;       //
  end;

  writeln(ScrMsg.Head);
  writeln('**************************************');
  for y:= 9 downto 1 do
  begin
    if ((y mod 3 = 0) and (y<9)) then
      writeln('*-----------+-----------+------------*')
    else if (y<9) then
      writeln('*           |           |            *');
    write('*');
    for x:=1 to 9 do
    begin
      if (fixSudoku[x,y]>=1) and (mark_fixSudokuPositions=1) then
        write('!')                              // markiere unveraenderliche Z.
      else if ((x=cursor.x) and (y=cursor.y)) then write('>')  // Eingabecursor anzeigen
           else write(' ');

      if ((sudoku[x,y]=0) and (screenFlag1_useBlank=1)) then
        write(' ')                              // ein leeres Feld anstelle der Null ausgeben
      else if ((fixSudoku[x,y]=0) and (color_fixSudokuPositions>0)) then
        begin
          TextColor(useColor);
          write(sudoku[x,y]);                   // Zahlenausgabe
          TextColor(White);
        end
        else write(sudoku[x,y]);                // Zahlenausgabe

      if ((x=cursor.x) and (y=cursor.y)) then write('<')       //Eingabecursor anzeigen
      else write(' ');

      if ((x mod 3 = 0) and (x<9)) then write('|')
      else write(' ');
    end;
    writeln('*') ;
  end;
  writeln('**************************************');
  writeln(ScrMsg.Status, '   ');
  writeln(ScrMsg.Hint,   '   ');
  writeln(ScrMsg.KeyInfo,'   ');
  write(ScrMsg.KeyInfo2, '   ');
  NormVideo;
end;



procedure check_integrity (sudoku: sudokuField; var errorFound: boolean;
                                                var integrityError: integrityErrorField);
// Prueft die Integritaet (Definition) des uebergebenen Sudokus.
// Keine Zahl darf in einer Zeile, Spalte oder 3x3-Block mehrfach vorkommen.
// In:  - sudoku: Sudokufeld, belegt mit Zahlen zwischen 0 bis 9
// Out: - errorFound: Wahrheitswert ob die Integritaet/Definition verletzt ist
//      - integrityError:
//                 Grundsaetzlich:
//                 Fehlerausloesende Zeile  (integrityError[1]),
//                                   Spalte (integrityError[2]) des Sudokus und
//                   der ausloesende Zahlenwert (integrityError[3])
//       falls:
//       integrityError[1]=0 und integrityError[2]=0 ---> Integrity-check OK!
//       integrityError[1]=0 und integrityError[2]>0 ---> Zeilenfehler
//       integrityError[1]>0 und integrityError[2]=0 ---> Spaltenfehler
//       integrityError[1]>0 und integrityError[2]>0 ---> Blockfehler
var x, xh, y, yh, i      : integer;
    count: array[1..9] of integer;
begin
  errorFound:=false;
  integrityError[1]:=0;
  integrityError[2]:=0;
  integrityError[3]:=0;

  // Zeilenintegritaet pruefen
  for y:=1 to 9 do
  begin
    for i:=1 to 9 do count[i]:=0;  // initialisiere die Fehlerzaehlung

    for x:=1 to 9 do
      if ((sudoku[x,y]>=1) and (sudoku[x,y]<=9)) then
        count[sudoku[x,y]]:= count[sudoku[x,y]] + 1;

    for i:=1 to 9 do
      if (count[i]>= 2) then       // Integitaetsfehler gefunden:
      begin                        // Definition des Sudoku nicht erfuellt!
        errorFound:=true;
        integrityError[1]:=0;
        integrityError[2]:=y;      // Zeile mit integrity-error
        integrityError[3]:=i;
        if (debugPrintFlag1_integrityErrors=1) then
        begin
          write  ('---->Integrity-Check: Zeilen-FEHLER: ',integrityError[1]);
          writeln(', ', integrityError[2],', Zahl: ',integrityError[3]);
        end;
      end;
  end;

  // Spaltenintegritaet pruefen
  for x:=1 to 9 do
  begin
    for i:=1 to 9 do count[i]:=0;  // initialisiere die Fehlerzaehlung

    for y:=1 to 9 do
      if ((sudoku[x,y]>=1) and (sudoku[x,y]<=9)) then
          count[sudoku[x,y]]:= count[sudoku[x,y]] + 1;

    for i:=1 to 9 do
      if (count[i]>= 2) then       // Integitaetsfehler gefunden:
      begin                        // Definition des Sudoku nicht erfuellt!
          errorFound:=true;
          integrityError[1]:=x;    // Spalte mit integrity-error
          integrityError[2]:=0;
          integrityError[3]:=i;
          if (debugPrintFlag1_integrityErrors=1) then
          begin
            write  ('---->Integrity-Check: Spalten-FEHLER: ',integrityError[1]);
            writeln(', ', integrityError[2],', Zahl: ',integrityError[3]);
          end;
      end;
    end;

  // Blockintegritaet pruefen
  for xh:=0 to 2 do                  // Schleifen ueber die 3x3 Bloecke
    for yh:=0 to 2 do
    begin
      for i:=1 to 9 do count[i]:=0;  // initialisiere die Fehlerzaehlung

      for x:=1 to 3 do
        for y:=1 to 3 do
          if ((sudoku[(xh*3+x),(yh*3+y)]>=1) and (sudoku[(xh*3+x),(yh*3+y)]<=9)) then
             count[sudoku[(xh*3+x),(yh*3+y)]]:= count[sudoku[(xh*3+x),(yh*3+y)]] + 1;

      for i:=1 to 9 do
        if (count[i]>= 2) then       // Integitaetsfehler gefunden:
        begin                        // Definition des Sudoku nicht erfuellt!
          errorFound:=true;
          integrityError[1]:=xh+1;   // Block mit integrity-error
          integrityError[2]:=yh+1;
          integrityError[3]:=i;
          if (debugPrintFlag1_integrityErrors=1) then
          begin
            write  ('---->Integrity-Check: Block-FEHLER: ',integrityError[1]);
            writeln(', ', integrityError[2],', Zahl: ',integrityError[3]);
          end;
        end;
    end;
  if (not(errorFound) and (debugPrintFlag2_integrity_OK=1)) then
    writeln('---->Integrity-Check: SUDOKU-Definition is *** OK ***!');
end;



procedure solveSudoku(sudoku: sudokuField; solvingStrategy: integer;
                      var solvedSudoku: sudokuField; var solved: boolean);
// Loest das Sudoku mittels einer rekursiven Strategie!
// In:  - sudoku: Sudokufeld, belegt mit Zahlen zwischen 0 bis 9
//        solvingStrategy, drei Aufrufstrategieen:
//            1         (benutze const _upsearch=1)
//          <>1 und <>3 (benutze const _downsearch=2)
//            3         (benutze const _random=3)
// Out: - solvedSudoku: die gefundene Loesung, sonst leer
//      - solved: Wahrheitswert, ob eine Loesung zu 'sudoku' gefunden wurde
// Aufruf ueber: procedure solveSudoku_clearly
const solvingStrategy_upsearch=1;
      solvingStrategy_downsearch=2;
      solvingStrategy_random=3;
var x, y, i: integer;
    x_freePos, y_freePos: integer;
    freePos_found, integrityError_found: boolean;
    integrityError: integrityErrorField;

begin
  x:=0; y:=0;
  // Pruefe die Definition/Integritaet des Sudokus
  integrityError_found:=false;
  integrityError[1]:=0; integrityError[2]:=0; integrityError[3]:=0;
  check_integrity(sudoku, integrityError_found, integrityError);
  PositionsExamined:=PositionsExamined+1;

  if not(integrityError_found) then
  begin
    // Die Integritaet ist OK, jetzt ein freies Feld suchen
    x_freePos:=0; y_freePos:=0;
    freePos_found:=false;

    if (solvingStrategy=solvingStrategy_random) then
    begin                      // suche per Zufallsalg. ein freies Feld
      get_FREE_random_FieldPosition(sudoku, x, y);
      if (sudoku[x,y]=0) then          // ebenso: ((x>=1) and (y>=1))
      begin
        freePos_found:=true;
        x_freePos:=x; y_freePos:=y;
      end;
    end
    else
    begin                    // oder der Reihe nach einem freien Feld suchen
      x:=1;
      while ((x<=9) and not(freePos_found)) do
      begin
        y:=1;
        while ((y<=9) and not(freePos_found)) do
        begin
          if (sudoku[x,y]=0) then
          begin
            freePos_found:=true;
            x_freePos:=x; y_freePos:=y;
          end;
          y:=y+1;
        end;
        x:=x+1;
      end;
    end;

    if ((debugPrintFlag4_solve_emptyPos=1) and freePos_found) then
      writeln('-->Solve-Alg.: integrity OK, found empty Field: ', x_freePos, ', ' ,y_freePos);

    // Pruefe ob das Sudoku geloest ist (kein freies Feld mehr + Integritaet des S. ist OK)
    if (not(freePos_found) and not(integrityError_found)) then
    begin
      solved:=true;                      // Das Sudoku ist geloest!
      copySudoku(sudoku, solvedSudoku);  // Sichere das geloeste Sudoku fuer die Rueckgabe!
      if (debugPrintFlag6_solve_SolvMsg=1) then writeln('*** SOLVED!!! ***');
    end
    else if (freePos_found and not(integrityError_found)) then
    begin
      // Rekursiver Loesungsalgorithmus (zwei Moeglichkeiten) den Loesungsbaum aufzuspannen
      if (solvingStrategy = solvingStrategy_upsearch) then
      begin
        i:=1;
        while ((i<=9) and not(solved)) do
        begin
          if (debugPrintFlag5_solve_RecCall=1) then
          begin
            write  ('-->Solve-Alg.: integrity OK, recursive Call: Field: ');
            writeln(x_freePos, ', ' ,y_freePos, ' modified to ', i);
          end;
          sudoku[x_freePos, y_freePos]:=i;
          solved:=false;
          solveSudoku(sudoku, solvingStrategy, solvedSudoku, solved);     // starte Rekursion!
          i:=i+1;
        end;
      end
      else
      begin
        i:=9;
        while ((i>=1) and not(solved)) do
        begin
          if (debugPrintFlag5_solve_RecCall=1) then
          begin
            write  ('-->Solve-Alg.: integrity OK, recursive Call: Field: ');
            writeln(x_freePos, ', ' ,y_freePos, ' modified to ', i);
          end;
          sudoku[x_freePos, y_freePos]:=i;
          solved:=false;
          solveSudoku(sudoku, solvingStrategy, solvedSudoku, solved);     // starte Rekursion!
          i:=i-1;
        end;
      end;
    end
    else solved:=false;
  end
  else solved:=false;
end;



procedure solveSudoku_clearly(sudoku: sudokuField; var solvedSudoku: sudokuField;
                              var solved, oneSolution: boolean);
// Loest das Sudoku und klaert ob das Sudoku eindeutig oder mehrdeutig loesbar ist!
// In:  - sudoku: Sudokufeld, belegt mit Zahlen zwischen 0 bis 9
// Out: - solvedSudoku: die gefundene Loesung, sonst leer
//      - solved: Wahrheitswert, ob eine Loesung zu 'sudoku' gefunden wurde
const solvingStrategy_upsearch=1;
      solvingStrategy_downsearch=2;

var solvedSudoku2: sudokuField;
    solved2: boolean;
    upsearch_PositionsExamined, downsearch_PositionsExamined: Real;
begin
  PositionsExamined:=0;
  oneSolution:=false;
  solved:= false;
  clear_Sudoku(solvedSudoku);
  solved2:=false;
  clear_Sudoku(solvedSudoku2);
  solveSudoku(sudoku, solvingStrategy_upsearch, solvedSudoku, solved);
  upsearch_PositionsExamined:= PositionsExamined;

  PositionsExamined:=0;
  if solved then
  begin
    solveSudoku(sudoku, solvingStrategy_downsearch, solvedSudoku2, solved2);
    downsearch_PositionsExamined:= PositionsExamined;

    PositionsExamined:=0;
    if solved2 then               // Beide Loesungsstrategien waren erfolgreich,
    begin                         // pruefe nun ob die Loesungen identisch sind
      if (are_Sudokus_same(solvedSudoku, solvedSudoku2)) then oneSolution:=true
      else oneSolution:=false;
    end;
  end;

  if (debugPrintFlag7_solve_clearly=1) then
  begin
    if ((solved and not(solved2)) or (not(solved) and solved2)) then
      writeln('solve_clearly: *** ERROR *** Fehler im Loesungsalgorithmus!!!');
    write('>solve_clearly: ' );
    if solved  then write('Strategie#1: erfolgr. ');
    if solved2 then write('Strat.#2: erfolgr. ');
    if oneSolution then writeln('> eind. loesb.')
    else if (solved and solved2) then writeln('> mehrd. loesb.')
    else writeln;
    writeln('-> Loesungen gesucht (upsearch): ', upsearch_PositionsExamined);
    writeln('-> Loesungen gesucht (downsearch): ', downsearch_PositionsExamined);
  end;
end;



procedure changeSudoku_oneFieldInput(var sudoku, fixSudoku: sudokuField;
                           var cursor: cursorPosition; var Tastatur_eing: char);
// Haendische Eingabebearbeitung einer einzelnen erlaubten Eingabe auf dem
// Sudokufeld ('sudoku'). Alle korespondierende Sudokufeldpositionen von
// 'fixSudoku' (das unmodifizierte Sudoku, das es zu loesen gilt) sind
// unveraenderlich
// In:  - sudoku: Sudokufeld, belegt mit Zahlen zwischen 0 bis 9
//                hier wird das zu bearbeitende Sudoku uebergeben
//      - fixSudoku: Das Sudokufeld, belegt mit Zahlen zwischen 0 bis 9
//                hier steht das unveraenderte Sudoku, das es zu loesen gilt
//      - cursor  .x/.y der Eingabe-Cursor auf dem Sudokufeld
// Out: - sudoku: Das ggf. modifizierte Sudoku
//      - cursor: .x/.y der ggf. bewegte Eingabe-Cursor auf dem Sudokufeld
//      - Tastatur_eing: Rueckgabe der Tastatureingabe: 0-9,x,e,q,s,t,<ESC>
var SubCode_eing: char;
begin
  repeat
    repeat
      sleep(waitMSec_onEditSudoku);
    until keypressed;
    Tastatur_eing:=readkey;

    if (fixSudoku[cursor.x, cursor.y]=0) then         // Zahleneingabe erlauben?
    begin
      if Tastatur_eing=' ' then Tastatur_eing:='0';   // beh. Leerz. wie Null-Eing.
      case Tastatur_eing of
        '0' : sudoku[cursor.x, cursor.y]:= 0;
        '1' : sudoku[cursor.x, cursor.y]:= 1;
        '2' : sudoku[cursor.x, cursor.y]:= 2;
        '3' : sudoku[cursor.x, cursor.y]:= 3;
        '4' : sudoku[cursor.x, cursor.y]:= 4;
        '5' : sudoku[cursor.x, cursor.y]:= 5;
        '6' : sudoku[cursor.x, cursor.y]:= 6;
        '7' : sudoku[cursor.x, cursor.y]:= 7;
        '8' : sudoku[cursor.x, cursor.y]:= 8;
        '9' : sudoku[cursor.x, cursor.y]:= 9;
      end;
    end;
    if (Tastatur_eing=#0) then
    begin                              // Sondertaste gedrueckt
      SubCode_eing:=readkey;           // lese Tastatur Subcode!
      case SubCode_eing of
        #72 : begin                    // cursor up
                cursor.y:=cursor.y+1;
                if (cursor.y > 9) then
                begin
                  cursor.y:=9;
                  Tastatur_eing:='*';  // bleibe im Unterprogramm
                end;
              end;
        #75 : begin                    // cursor left
                cursor.x:=cursor.x-1;
                if (cursor.x < 1) then
                begin
                  cursor.x:=1;
                  Tastatur_eing:='*';  // bleibe im Unterprogramm
                end;
              end;
        #77 : begin                    // cursor right
                cursor.x:=cursor.x+1;
                if (cursor.x > 9) then
                begin
                  cursor.x:=9;
                  Tastatur_eing:='*';  // bleibe im Unterprogramm
                end;
              end;
        #80 : begin                    // cursor down
                cursor.y:=cursor.y-1;
                if (cursor.y < 1) then
                begin
                  cursor.y:=1;
                  Tastatur_eing:='*';  // bleibe im Unterprogramm
                end;
              end;
      end;
    end;
  until ((Tastatur_eing=#27) or (Tastatur_eing=#0)                      // #27 - ESC - Key
          or (Tastatur_eing='x') or (Tastatur_eing='e') or (Tastatur_eing='q')
          or (Tastatur_eing='s') or (Tastatur_eing='t')
          or (ord(Tastatur_eing)>=$30) and (ord(Tastatur_eing)<=$39))   // Zahl zw. 0 und 9
end;



procedure enterNewSudoku(var sudoku: sudokuField; var cursor: cursorPosition;
                                                  var Tastatur_eing: char);
// Haendische Eingabe eines neuen Sudokus. Waehrend der Eingabe wird die
// Integritaet (Einhaltung der Definition) geprueft und sichergestellt.
// In:  - sudoku: Sudokufeld, belegt mit Zahlen zwischen 0 bis 9
//                Das zu bearbeitende Sudoku wird uebergeben
// Out: - sudoku: Das ggf. modifizierte Sudoku
//      - cursor: .x/.y der ggf. bewegte Eingabe-Cursor auf dem Sudokufeld
//      - Tastatur_eing: Rueckgabe der Tastatureingabe: 0-9,x,e,q,s,t,<ESC>
// Hinweis: Es werden zusaetzlich globale Variablen ausgewertet.
var integrityErrorFound: boolean;
    integrityError: integrityErrorField;
    solved, oneSolution: boolean;
    solvedSudoku: sudokuField;
    ScrMsg: ScreenMessage;
    StrBuffer: string;
    Msg: ScrMessages;

begin
  set_ScreenMessages(Msg, lang);
  cursor.x:=1; cursor.y:=9;
  clear_sudoku(fixSudoku);
  clear_sudoku(sudoku);
  clear_ScreenMessages(ScrMsg);

//Rahmenbreite:    '**************************************'
  ScrMsg.Head:=    Msg[lang,entr,1].Head;    //'>> Ein zu loesendes Sudoku eingeben <<';
  ScrMsg.Status:=  Msg[lang,entr,1].Status;  //'OK (Feldintegrität)                   ';
  ScrMsg.Hint:=    Msg[lang,entr,1].Hint;    //'Eingabe: 0 - 9 und Cursor-Tasten      ';
  ScrMsg.KeyInfo:= Msg[lang,entr,1].KeyInfo; //'e, q     - Eingabe abschließen        ';
  ScrMsg.KeyInfo2:=Msg[lang,entr,1].KeyInfo2;//'<ESC>, x - alle Eingaben verwerfen    ';

  repeat                                         // eindeutige Loesbarkeit einfordern
    solved:=false; oneSolution:=false;
    repeat                                       // alle Feldeingaben durchfuehren
      text_screen_output(sudoku, fixSudoku, cursor, ScrMsg);
      ScrMsg.Status:=  Msg[lang,entr,1].Status;  //'OK (Feldintegrität)                   ';
      ScrMsg.Hint:=    Msg[lang,entr,1].Hint;    //'Eingabe: 0 - 9 und Cursor-Tasten      ';

      changeSudoku_oneFieldInput(sudoku, fixSudoku, cursor, Tastatur_eing);
      if debugPrintFlag12_edit=1 then
      begin
        write  ('### letzte Taste (Code): ', ord(Tastatur_eing));
        writeln('Cursor x:', cursor.x, ' y: ' ,cursor.y);
      end;
      integrityErrorFound:=false;
      integrityError[1]:=0; integrityError[2]:=0; integrityError[3]:=0;
      check_integrity(sudoku, integrityErrorFound, integrityError);
      if integrityErrorFound then                // letzte Eingabe war unzulässig
      begin
        Str(sudoku[cursor.x, cursor.y],StrBuffer);
        ScrMsg.Status:= Msg[lang,entr,2].Status               //'Zahl '
                      + StrBuffer + Msg[lang,entr,3].Status;  //' verletzt Integrität   ';

        ScrMsg.Hint:=    Msg[lang,entr,2].Hint;     //'Andere Zahl oder Feld belegen     ';
        sudoku[cursor.x, cursor.y]:=0;
      end;
    until ((Tastatur_eing=#27) or (Tastatur_eing='x')     // Abbruch (#27 - ESC-Key)
          or (Tastatur_eing='e') or (Tastatur_eing='q')); // Eingabeende

    if ((Tastatur_eing=#27) or (Tastatur_eing='x')) then
    begin                                      // Benutzer wuenscht Abbruch und
      clear_sudoku(sudoku);                    // bisherig eingegebenes Sudoku loeschen
      cursor.x:=1; cursor.y:=9;
    end
    else
    begin
      if (count_fields_used(sudoku) >= min_sudokuFields_used) then
      begin       // da genuegend Felder belegt sind, pruefe nun die Loesbarkeit
        ScrMsg.Status:= Msg[lang,entr,4].Status;    //'>>> PRÜFE DIE LÖSBARKEIT! <<<';
        ScrMsg.Hint:=   Msg[lang,entr,3].Hint;      //'Dies kann einige Zeit dauern! ';
        text_screen_output(sudoku, fixSudoku, cursor, ScrMsg);
        solved:=false; oneSolution:=false;
        solveSudoku_clearly(sudoku, solvedSudoku, solved, oneSolution);

        if (solved and (not(oneSolution))) then
        begin
          ScrMsg.Status:= Msg[lang,entr,5].Status; //'Lösbar, aber nicht eindeutig loesbar ';
          ScrMsg.Hint:=   Msg[lang,entr,4].Hint;   //'Weitere Feldbelegungen erforderlich ';
        end;
        if (not(solved)) then
        begin
          ScrMsg.Status:= Msg[lang,entr,6].Status; //'>>> >>> NICHT LÖSBAR!!! <<< <<<      ';
          ScrMsg.Hint:=   Msg[lang,entr,5].Hint;   //'Änderungen an den Feldbelegungen erf.';
        end;

      end
      else
      begin
        ScrMsg.Status:=  Msg[lang,entr,7].Status;   //'Nicht genuegend Felder belegt! ';
        ScrMsg.Hint:=    Msg[lang,entr,6].Hint;     //'Weitere Felder belegen         ';
      end;
    end
  until ((Tastatur_eing=#27) or (Tastatur_eing='x')  // Abbruch, (#27 - ESC-Key)
         or (solved and oneSolution));               // oder eindeutig Loesbar
end;



procedure edit_to_solve_sudoku(var sudoku: sudokuField; var cursor: cursorPosition;
                                                  var Tastatur_eing: char);
// Haendisches Loesen eines gegebenen Sudokus. Es werden drei Eingabemodi unterstuetzt:
// - keine Eingabepruefung
// - Pruefung der Integritaet waehrend der Eingabe
// - Pruefung auf Korrektheit waehrend der Eingabe
// In:  - sudoku: Sudokufeld, belegt mit Zahlen zwischen 0 bis 9,
//                Null entspricht einem unbelegen, leeren Platz
//                Das zu bearbeitende Sudoku wird uebergeben
//      - cursor  .x/.y der Eingabe-Cursor auf dem Sudokufeld
//                unerheblich, 0/0 zum Start moeglich
// Out: - sudoku: Das ggf. modifizierte Sudoku
//      - cursor: .x/.y der ggf. bewegte Eingabe-Cursor auf dem Sudokufeld
//      - Tastatur_eing: Rueckgabe der Tastatureingabe: 0-9,x,e,q,s,t,<ESC>
// Hinweis: Es werden zusaetzlich globale Variablen ausgewertet.
var integrityErrorFound: boolean;
    integrityError: integrityErrorField;
    solved, oneSolution, manually_solved: boolean;
    solvedSudoku, fixSudoku, backupSudoku_integrity: sudokuField;
    ScrMsg: ScreenMessage;
    StrBuffer, StrBuf_free, StrBuf_used: string;
    x, y: integer;
    way_of_inputChecking: integer;  // Eingabepruefung: 0-keine 1-Integritaet 2-Korrektheit
    Msg: ScrMessages;

begin
  set_ScreenMessages(Msg, lang);
  way_of_inputChecking:=0;
  solved:=false; oneSolution:=false;
  clear_sudoku(solvedSudoku);
  copySudoku(sudoku, fixSudoku);
  if ((cursor.x=0) or (cursor.y=0)) then
  begin
    cursor.x:=1; cursor.y:=9;
  end;

  clear_ScreenMessages(ScrMsg);
  // Rahmenbreite: '**************************************'
  ScrMsg.Head:=     Msg[lang,edit,1].Head;      //'>>> >>> >>>  Sudoku lösen  <<< <<< <<<';
  ScrMsg.KeyInfo:=  Msg[lang,edit,1].KeyInfo;   //'s - lösen, t - Eingabeprüfung ändern  ';
  ScrMsg.KeyInfo2:= Msg[lang,edit,1].KeyInfo2;  //'Eingabe: 0 - 9 und Cursor-Tasten      ';

  // Pruefe ob das Sudoku ueberhaupt loesbar ist. Dies sollte erfuellt sein
  integrityErrorFound:=false;
  integrityError[1]:=0; integrityError[2]:=0; integrityError[3]:=0;
  check_integrity(sudoku, integrityErrorFound, integrityError);
  if (not(integrityErrorFound) and (count_fields_used(sudoku) >= min_sudokuFields_used)) then
  begin
    ScrMsg.Status:= Msg[lang,edit,1].Status;          //'>>> PRÜFE DIE LÖSBARKEIT! <<<';
    ScrMsg.Hint:=   Msg[lang,edit,1].Hint;            //'Dies KANN einige Zeit dauern! ';
    text_screen_output(sudoku, fixSudoku, cursor, ScrMsg);
    solveSudoku_clearly(sudoku, solvedSudoku, solved, oneSolution);
    if (solved and oneSolution) then
    begin  // Alle Voraussetzungen fuer das haendische Loesen sind erfuellt
      manually_solved:=false;
      ScrMsg.Hint:=Msg[lang,edit,2].Hint;             //'Sudoku ist eindeutig lösbar';
      repeat
        integrityErrorFound:=false;
        integrityError[1]:=0; integrityError[2]:=0; integrityError[3]:=0;
        check_integrity(sudoku, integrityErrorFound, integrityError);
        if (not(integrityErrorFound)) then copySudoku(sudoku, backupSudoku_integrity);

        if (way_of_inputChecking=0) then
          ScrMsg.Status:=Msg[lang,edit,2].Status    //'ungel., keine Eingabeprüfung'
        else if (way_of_inputChecking=1) then
          ScrMsg.Status:=Msg[lang,edit,3].Status    //'ungel., Eing.-Integritätsprüf.'
        else if (way_of_inputChecking=2) then
          ScrMsg.Status:=Msg[lang,edit,4].Status;   //'ungel., Eing.-Korrektheitsprüf.';

        text_screen_output(sudoku, fixSudoku, cursor, ScrMsg);
        ScrMsg.KeyInfo2:= Msg[lang,edit,2].KeyInfo2;  //'<ESC>, x - Rückkehr zum Hauptmenü     ';

        Str(count_fields_free(sudoku) ,StrBuf_free);
        Str(count_fields_used(sudoku) ,StrBuf_used);
        // z.B.: Felder belegt: 30, frei: 51
        ScrMsg.Hint:= Msg[lang,edit,3].Hint                  //'Felder belegt: '
                    + StrBuf_used + Msg[lang,edit,4].Hint    //', frei: '
                    + StrBuf_free;

        manually_solved:=are_Sudokus_same(sudoku, solvedSudoku);
        if not(manually_solved) then
        begin    // die Loesung wurde noch nicht gefunden
          changeSudoku_oneFieldInput(sudoku, fixSudoku, cursor, Tastatur_eing);
          if debugPrintFlag12_edit=1 then
          begin
            write  ('### letzte Taste (Code): ', ord(Tastatur_eing));
            writeln('Cursor x:', cursor.x, ' y: ' ,cursor.y);
          end;

          if (Tastatur_eing='t') then way_of_inputChecking:=way_of_inputChecking+1;
          if (way_of_inputChecking>=3) then way_of_inputChecking:=0;

          // Eingabepruefungen durchfuehren

          if (way_of_inputChecking=1) then
          begin         // Integritaetspruefung waehrend der Eingabe
            integrityErrorFound:=false;
            integrityError[1]:=0; integrityError[2]:=0; integrityError[3]:=0;
            check_integrity(sudoku, integrityErrorFound, integrityError);
            if integrityErrorFound then                // letzte Eingabe war unzulässig
            begin
              if (sudoku[cursor.x, cursor.y]> 0) then
              begin
                Str(sudoku[cursor.x, cursor.y],StrBuffer);
                // z.B. 'Zahl 6 verletzt Integrität   '
                ScrMsg.Hint:=  Msg[lang,edit,5].Hint            //'Zahl '
                            + StrBuffer + Msg[lang,edit,6].Hint;//' verletzt Integrität   ';
              end
              else ScrMsg.Hint:=Msg[lang,edit,11].Hint; //'einige Eing. wurden korrigiert';

              copySudoku(backupSudoku_integrity, sudoku);     // hole Kopie
            end
            else if (sudoku[cursor.x, cursor.y]=0) then
              ScrMsg.Hint:= Msg[lang,edit,7].Hint             //'Feld belegen    '
          end;

          if (way_of_inputChecking=2) then
          begin         // Korrektheitspruefung waehrend der Eingabe
            if (sudoku[cursor.x, cursor.y]<>solvedSudoku[cursor.x, cursor.y]) then
            begin
              if (sudoku[cursor.x, cursor.y]>0) then
              begin
                Str(sudoku[cursor.x, cursor.y],StrBuffer);
                // z.B.  'Zahl 5 verletzt Korrektheit!'
                ScrMsg.Hint:=  Msg[lang,edit,8].Hint            //'Zahl '
                            + StrBuffer + Msg[lang,edit,9].Hint;//' verletzt Korrektheit!';
                sudoku[cursor.x, cursor.y]:=0;
              end
              else ScrMsg.Hint:= Msg[lang,edit,10].Hint;        //'Feld belegen    ';

              for x:=1 to 9 do         // Stelle die Korrektheit des Sudokus her
                for y:=1 to 9 do
                  if ((sudoku[x,y]>0) and (sudoku[x,y]<>solvedSudoku[x,y])) then
                  begin
                    sudoku[x,y]:=0;
                    ScrMsg.Hint:=Msg[lang,edit,11].Hint; //'einige Eing. wurden korrigiert';
                  end;

            end
          end;
        end

        else     // ab hier ist: -> manually_solved=true
        begin    //Die Loesung wurde haendisch gefunden
          ScrMsg.Status:=  Msg[lang,edit,5].Status;    //'Glückwunsch! Lösung gefunden.';
          ScrMsg.Hint:=    Msg[lang,edit,12].Hint;     //'=============================';
          ScrMsg.KeyInfo:= Msg[lang,edit,2].KeyInfo;   //'e, q     - Hauptmenü         ';
        end;

        if (Tastatur_eing='s') then
        begin
          copySudoku(solvedSudoku, sudoku);
          ScrMsg.Status:= Msg[lang,edit,6].Status;     //'Computerlösung ';
          ScrMsg.Hint:=   Msg[lang,edit,13].Hint;      //'Mehr Glück nächstes Mal ';
        end;

        if ((Tastatur_eing='s') or (manually_solved)) then
        begin             // Sudoku geloest, nun geloestes Sudoku anzeigen
          ScrMsg.KeyInfo:= Msg[lang,edit,2].KeyInfo;   //'e, q     - Hauptmenü         ';
          repeat
            text_screen_output(sudoku, fixSudoku, cursor, ScrMsg);
            repeat
              sleep(waitMSec_onGeneralSelection);
            until keypressed;
            Tastatur_eing:=readKey;
          until ((Tastatur_eing=#27) or (Tastatur_eing='x')      // Abbruch, (#27 - ESC-Key)
                or (Tastatur_eing='q') or (Tastatur_eing='e'));  // oder ENDE-Auswahl

        end;
      until ((Tastatur_eing=#27) or (Tastatur_eing='x')      // Abbruch, (#27 - ESC-Key)
            or (Tastatur_eing='q') or (Tastatur_eing='e')    // oder ENDE-Auswahl
            or manually_solved);                              // Loesung gefunden
    end;
  end;
end;



procedure setOneLineWithRandomNumbers(var sudoku: sudokuField; y_line: integer);
// Die Zeile (y_line) des Sudokus wird per Zufallszahlen mit Zahlen 1 bis 9 belegt.
// Jede Zahl wird hierbei nur einmal verwendet.
// Wird zur Computer-Generierung neuer Sudokus von 'generate_sudoku' verwendet!
// Hintergrund: Durch eine geschickte Vorbelegung eines Sudokufeldes koennen die
//              Freiheitsgrade erheblich reduziert werden
//              Denkbar ist eine Vorbelegung der Zeile 1 oder 2 oder 3
//              optional eine weitere Vorbelegung der Zeile 4 oder 5 oder 6 und
//              optional noch eine weitere Vorbelegung der Zeile 7 oder 8 oder 9
// In:  - sudoku: Sudokufeld, belegt mit Zahlen zwischen 0 bis 9,
//                Null entspricht einem unbelegen, leeren Platz
//                S. Ist entweder leer oder es sind schon ganze Zeilen vorbelegt!
//      - y_line: Wert von 1 bis 9 zulaessig!!!
// Out: - sudoku: Das modifizierte Sudoku
const max_random=10;                      // Anzahl der zufaelligen Vertauschungen
      max_retryOnIntegrityError=250;      // Verhindert Endlosschleife,
                                          // tatsaechlich benoetigte retries i.d.R. <10
var x, rand_count, oneCellBuffer: integer;
    integrityErrorFound: boolean;
    integrityError: integrityErrorField;
    x1_rnd_index, x2_rnd_index: integer;
    count_retriesOnError: integer;

begin
  integrityErrorFound:=false;
  integrityError[1]:=0; integrityError[2]:=0; integrityError[3]:=0;
  check_integrity (sudoku, integrityErrorFound, integrityError);
  if integrityErrorFound then clear_sudoku(sudoku);  // Korrigiere
  if y_line>9 then y_line:=9;                        // falsche
  if y_line<1 then y_line:=1;                        // Eingaben

  for x:=1 to 9 do sudoku[x, y_line]:=x;       // Belege die uebergebene Zeile

  count_retriesOnError:=0;
  repeat
    for rand_count:=1 to max_random do         // Mische die betreffende Zeile
    begin
      x1_rnd_index:= round((random*9)+0.5);
      x2_rnd_index:= round((random*9)+0.5);
      oneCellBuffer:= sudoku[x1_rnd_index, y_line];
      sudoku[x1_rnd_index, y_line]:= sudoku[x2_rnd_index, y_line];
      sudoku[x2_rnd_index, y_line]:= oneCellBuffer
    end;

    integrityErrorFound:=false;
    check_integrity (sudoku, integrityErrorFound, integrityError);
    count_retriesOnError:= count_retriesOnError+1;

  until((count_retriesOnError>=max_retryOnIntegrityError) or not(integrityErrorFound));

  if (debugPrintFlag14_randomLine=1) then
  begin
    write('GENERIERE EINE RANDOM-Zeile ( ', y_line);
    writeln(' ): Anz. Versuche: ', count_retriesOnError);
  end;
end;



procedure generate_sudoku(var sudoku: sudokuField; difficult_wanted, max_retry: integer;
                          var difficult_level_reached: integer);
// Es wird ein eindeutig loesbares Sudoku generiert !!! Rueckgabewerte sind das
// Sudoku und der Schwierigkeitsgrad des generierten Sudokus.
// Aufruf ueber: procedure "gen_sudoku_with_difficult_level" to force a difficult level
// In:  - difficult_wanted: gewuenschter Schwierigkeitsgrad (=belegte Felder)
//      - max_retry: 0..15 -> Anzahl der Versuche den Schwierigkeitsgrad zu verbessern
//                   -1    -> Waehle Werte in abh. des gewuenschten Schwierigkeitsgrads
// Out: - sudoku: Das generiertes Sudokufeld
//      - difficult_level_reached: erreichter Schwierigkeitsgrad (=belegte Felder)
var solvedSudoku, sudokuBackup: sudokuField;
    x, y: integer;
    retry_count: integer;
    solved, oneSolution, found: boolean;

begin
  clear_sudoku(sudoku); clear_sudoku(solvedSudoku);

  if (difficult_wanted>80) then difficult_wanted:=80;
  if (difficult_wanted<15) then difficult_wanted:=15;

  if max_retry > 5 then max_retry:=5;         // hoehere Werte bis ca. 10 moegl.
  if max_retry < 0 then
  begin      // Setze Standardwerte in abh. des gewuenschten Schwierigkeitsgrads
    max_retry:=0;
    if (difficult_wanted<36) then max_retry:=1;
    if (difficult_wanted<32) then max_retry:=2;
    if (difficult_wanted<30) then max_retry:=3;
    if (difficult_wanted<28) then max_retry:=5;
  end;

  setOneLineWithRandomNumbers(sudoku, (0 + round((random*3)+0.5)));
  setOneLineWithRandomNumbers(sudoku, (3 + round((random*3)+0.5)));
  setOneLineWithRandomNumbers(sudoku, (6 + round((random*3)+0.5)));

  PositionsExamined:=0; solved:=false; oneSolution:=false;
  solveSudoku_clearly(sudoku, solvedSudoku, solved, oneSolution);
  if solved then
  begin
    copySudoku(solvedSudoku, sudoku);
    found:=false;
    x:=0; y:=0; retry_count:=0;
    repeat
      retry_count:=retry_count+1;
      repeat
        copySudoku(sudoku, sudokuBackup);
        get_USED_random_FieldPosition(sudoku, x, y);
        sudoku[x,y]:=0;
        solveSudoku_clearly(sudoku, solvedSudoku, solved, oneSolution);
        if (solved and oneSolution) then
        begin
          retry_count:=0;
          if ((count_fields_used(sudoku)<= difficult_wanted)) then
          begin
            found:=true;
            copySudoku(sudoku, sudokuBackup);
            if (debugPrintFlag16_generateFound = 1) then
              writeln('generate: Sudoku mit gew. Schwierigkeitsgrad gef., beende Suche');
          end;
        end;
      until ((not(solved) or not(oneSolution)) or found);
      copySudoku(sudokuBackup, sudoku);
    until (found or (retry_count>=max_retry));
  end;
  difficult_level_reached:= count_fields_used(sudoku);
end;



procedure gen_sudoku_with_difficult_level(var sudoku: sudokuField;
                     level_wanted: oneDifficultLevel; var difficult_level_reached: integer);
// Es wird ein eindeutig loesbares Sudoku generiert, welches sich bestmöglich
// dem gewuenschten Schwierigkeitsgrad annaehert! Hierbei findet eine Abwaegung
// zwischen dem Erzwingen des gewuenschten Schwierigkeitslevels und dem
// Zeitaufwand /Rechenaufwand statt.
// In:  - level_wanted: einen gewuenschten Schwierigkeitslevel
// Out: - sudoku: Das generiertes Sudokufeld
//      - difficult_level_reached: erreichter Schwierigkeitsgrad (=belegte Felder)
// Hinweis: Es werden zusaetzlich globale Variablen ausgewertet.
var best_sudoku_reached: sudokuField;
    best_difficult_reached, difficult_reached: integer;
    count: integer;
    StrBuf_count, StrBuf_max, StrBuf_actual, StrBuf_best: string;

var ScrMsg: ScreenMessage;               // Variablen fuer die Ausgabe
    MenueSudoku, fixSudoku: sudokuField;
    cursor: cursorPosition;
    Msg: ScrMessages;
    StrBuffer: string;

begin
  set_ScreenMessages(Msg, lang);

  // Pruefe die uebergebenen Parameter und setze ggf. Standardwerte
  if ((level_wanted.lowValue<10) or (level_wanted.lowValue>80)
     or (level_wanted.highValue<10) or (level_wanted.highValue>80)
     or (level_wanted.lowValue > level_wanted.highValue)) then
  begin    // Dieser Fall sollte nicht vorkommen!
    level_wanted.MsgText  := '(33-36)';  level_wanted.callValue:= 34;
    level_wanted.lowValue := 33;         level_wanted.highValue:= 36;
  end;

  cursor.x:=0; cursor.y:=0;
  clear_ScreenMessages(ScrMsg);
  clear_sudoku(MenueSudoku); clear_sudoku(fixSudoku);
  //                            Rahmenbreite:   '**************************************'
  ScrMsg.Head:=    Msg[lang,gene,1].Head;     //'>>> >> >>> Generiere Sudoku <<< << <<<';
  ScrMsg.Status:=  Msg[lang,gene,1].Status;   //'Suche:                                '
  ScrMsg.Hint:=    Msg[lang,gene,1].Hint;     //'      - Bitte warten -        ';
  ScrMsg.KeyInfo:= Msg[lang,gene,1].KeyInfo;  //'Dies kann einige Zeit dauern! ';
  ScrMsg.KeyInfo2:=Msg[lang,gene,1].KeyInfo2; //'      - Bitte warten -        ';
  text_screen_output(MenueSudoku, fixSudoku, cursor, ScrMsg);

  cursor.x:=0; cursor.y:=0;
  clear_ScreenMessages(ScrMsg);
  clear_sudoku(MenueSudoku); clear_sudoku(fixSudoku);

  clear_sudoku(best_sudoku_reached);
  best_difficult_reached:=1000;   // Alles gefundene wird besser sein als dies!
  difficult_reached:=0;
  count:= 0;

  repeat
//  generate_sudoku(sudoku, level_wanted.callValue, -1, difficult_reached);
    generate_sudoku(sudoku, level_wanted.callValue, count, difficult_reached);

    if (debugPrintFlag16_generateFound = 1) then
      writeln('Generiertes Sudoku hat Schwierigkeitsgrad: ', difficult_reached);
    if (abs(difficult_reached - level_wanted.lowValue)
                          < abs(best_difficult_reached - level_wanted.lowValue)) then
    begin // Sudoku mit besserem, naeheren Schwierigkeitsgrad gefunden
      best_difficult_reached:= difficult_reached;
      copySudoku(sudoku, best_sudoku_reached);
      if (debugPrintFlag17_genFoundBetter = 1) then
        writeln('Finde besseren Schwierigkeitsgrad: ', difficult_reached);
    end;

    // Fortschrittsanzeige darstellen
    str(count , StrBuf_count); str(max_tries_genDifficultLevel, StrBuf_max);
    str(difficult_reached, StrBuf_actual); str(best_difficult_reached, StrBuf_best);
    // Rahmenbreite: '**************************************'
    ScrMsg.Head:= Msg[lang,gene,1].Head;     // '>>> >> >>> Generiere Sudoku <<< << <<<  ';
    ScrMsg.Status:= Msg[lang,gene,2].Status  // 'Suche: '
                  + level_wanted.MsgText;

    Str(level_wanted.callValue, StrBuffer);
    // z.Bsp: Vers.: 1/10, akt.: 30, best: 23
    ScrMsg.Hint:= Msg[lang,gene,2].Hint                   //' Ver.: '
                + StrBuf_count + '/'
                + StrBuf_max + Msg[lang,gene,3].Hint      //', akt.: '
                + StrBuf_actual + Msg[lang,gene,4].Hint   //', best: '
                + StrBuf_best;

    ScrMsg.KeyInfo:= Msg[lang,gene,1].KeyInfo;      //'Dies kann einige Zeit dauern! ';
    ScrMsg.KeyInfo2:=Msg[lang,gene,1].KeyInfo2;     //'      - Bitte warten -        ';
    text_screen_output(MenueSudoku, fixSudoku, cursor, ScrMsg);

    count:=count+1;
  until ((difficult_reached <= level_wanted.highValue)
        or (count > max_tries_genDifficultLevel));

  copySudoku(best_sudoku_reached, sudoku);
  difficult_level_reached:= count_fields_used(sudoku);
end;



Procedure mainMenue(var Tastatur_eing: char);
// Hauptauswahlmenue
// Auswahl: Ein Sudoku haendisch eingeben oder generieren lassen
// Sprache umstellen und Darstellungsart des Sudokus abaendern
// Out: -Tastatur_eing: x,<ESC> Abbruch, Ebene zurueck
//                      n - Sudoku händisch eingeben
//                      g - Sudoku generieren lassen
// Hinweis: Es werden zusaetzlich globale Variablen ausgewertet.
var ScrMsg: ScreenMessage;
    sudoku, fixSudoku: sudokuField;
    cursor: cursorPosition;
    Msg: ScrMessages;
    toggleLook: integer;

begin
  toggleLook:=1;
  set_ScreenMessages(Msg, lang);
  cursor.x:=0; cursor.y:=0;
  Tastatur_eing:=' ';
  clear_sudoku(sudoku);
  clear_sudoku(fixSudoku);
  clear_ScreenMessages(ScrMsg);

  load_internal_Sudoku(0, sudoku);
  load_internal_Sudoku(1, fixSudoku);
  repeat
    // Rahmenbreite:                              '**************************************'
    ScrMsg.Head:=    Msg[lang,menu,1].Head;     //'>>> >>>> >>> Auswahlmenü <<< <<<< <<<';
    ScrMsg.Status:=  Msg[lang,menu,1].Status;   //'Natürlich kein legales Sudoku!';
    ScrMsg.Hint:=    Msg[lang,menu,1].Hint;     //'Hinw.: l- Sprache, t- Aussehen ändern '
    ScrMsg.KeyInfo:= Msg[lang,menu,1].KeyInfo;  //'n - Sudoku händisch eingeben ';
    ScrMsg.KeyInfo2:=Msg[lang,menu,1].KeyInfo2; //'g - Sudoku generieren lassen ';
    text_screen_output(sudoku, fixSudoku, cursor, ScrMsg);

    repeat
      sleep(waitMSec_onGeneralSelection);
    until keypressed;
    Tastatur_eing:=readKey;

    if (Tastatur_eing='t') then
    begin
      toggleLook:=toggleLook+1;                      //
      if (toggleLook>=4) then toggleLook:=1;         //
      case toggleLook of                             // Bei Aenderungen vergl.
        1: begin                                     // mit
             mark_fixSudokuPositions:=1;             // procedure main_sudoku
             color_fixSudokuPositions:=0;            //
           end;                                      //
        2: begin                                     //
             mark_fixSudokuPositions:=0;             // Aendere
             color_fixSudokuPositions:=1;            // globale Settings!
           end;                                      //
        3: begin                                     //
             mark_fixSudokuPositions:=0;             //
             color_fixSudokuPositions:=2;            //
           end;                                      //
      end;                                           //
    end;                                             //
                                                     //
    if (Tastatur_eing='l') then lang:=lang+1;        // aendere die Sprache
    if (lang > max_languages) then lang:=1;          //

  until ((Tastatur_eing=#27) or (Tastatur_eing='x')       // Abbruch, (#27 - ESC-Key)
        or (Tastatur_eing='n') or (Tastatur_eing='g'));   // oder Auswahl erfolgt
end;



Procedure generiereSudokuMenue(var Tastatur_eing: char;
                               var difficult_wanted: oneDifficultLevel);
// Unter-/Auswahlmenue: Den Schwierigkeitsgrad des zu generierenden Sudokus
// waehlen.
// Out: - Tastatur_eing: x,<ESC> Abbruch, Ebene zurueck
//      - difficult_wanted: Der ausgewaehlte Schwierigkeitslevel
// Hinweis: Es werden zusaetzlich globale Variablen ausgewertet.
var ScrMsg: ScreenMessage;
    sudoku, fixSudoku: sudokuField;
    cursor: cursorPosition;
    level: integer;
    difficult: DifficultLevels;
    Msg: ScrMessages;

begin
  set_ScreenMessages(Msg, lang);
  level:=(max_difficult_levels div 2);   // ersteinmal: mittleren Schwierigkeitsgrad
  cursor.x:=0; cursor.y:=0;
  Tastatur_eing:=' ';
  clear_sudoku(sudoku);
  clear_sudoku(fixSudoku);
  load_internal_Sudoku(0, sudoku);
  set_difficult_level_definitions(difficult);
  clear_ScreenMessages(ScrMsg);

  repeat
    // Rahmenbreite:                               '**************************************'
    ScrMsg.Head:=    Msg[lang,gmen,1].Head;      //'>>> >> >>> Generiere Sudoku <<< << <<<';
    ScrMsg.Status:=  Msg[lang,gmen,1].Status     //'Level: '
                  + difficult[level].MsgText;    // oder: + Msg[lang,difc,level].Status;
    ScrMsg.KeyInfo:= Msg[lang,gmen,1].KeyInfo;   //'+/- Level ändern, g - generiere';
    ScrMsg.KeyInfo2:=Msg[lang,gmen,1].KeyInfo2;  //'<ESC>, x - ENDE              ';
    if difficult[level].lowValue<30 then
      ScrMsg.Hint:=  Msg[lang,gmen,1].Hint       //  'Dies KANN einige Zeit dauern!         '
    else
      ScrMsg.Hint:=  Msg[lang,gmen,2].Hint;      //  'Eingabe erwartet...           ';

    text_screen_output(sudoku, fixSudoku, cursor, ScrMsg);

    repeat
      sleep(waitMSec_onGeneralSelection);
    until keypressed;
    Tastatur_eing:=readKey;

    if (Tastatur_eing='+') then        // Schwierigkeitsgrad erhoehen
    begin
      level:=level+1;
      if level>max_difficult_levels then level:=max_difficult_levels;
    end;

    if (Tastatur_eing='-') then        // Schwierigkeitsgrad verringern
    begin
      level:=level-1;
      if level<1 then level:=1;
    end;
    difficult_wanted:= difficult[level];

  until ((Tastatur_eing=#27) or (Tastatur_eing='x')       // Abbruch, (#27 - ESC-Key)
         or (Tastatur_eing='e') or (Tastatur_eing='g'));  // oder Auswahl erfolgt
end;



procedure one_Sudoku_to_file(sudoku: sudokuField; fileName: string);
// Ein (sinnvollerweise zuvor generiertes) Sudoku-Feld, mit Rahmen und Hilfslinien
// in die Datei 'fileName' schreiben.
// Aufruf ueber: make_file_with_6_printable_Sudokus
// In: - sudoku: Sudokufeld, belegt mit Zahlen zwischen 0 bis 9,
//               Null entspricht einem unbelegen, leeren Platz
//     - fileName: Ein Dateinamen
var x, y: integer;
    f: text;
begin
  assign(f, fileName);
  rewrite(f);
  writeln(f, '**************************************');
  for y:= 9 downto 1 do
  begin
    if ((y mod 3 = 0) and (y<9)) then
      writeln(f, '*-----------+-----------+------------*')
    else if (y<9) then
      writeln(f, '*           |           |            *');
    write(f, '*');
    for x:=1 to 9 do
    begin
      write(f, ' ');
      if (sudoku[x,y]=0) then
        write(f, ' ')                 // ein leeres Feld anstelle der Null ausgeben
      else write(f, sudoku[x,y]);     // normale Zahlen-Ausgabe
      write(f, ' ');
      if ((x mod 3 = 0) and (x<9)) then write(f, '|')
      else write(f, ' ');
    end;
    writeln(f, '*') ;
  end;
  writeln(f, '**************************************');
  close(f);
end;



procedure make_file_with_6_printable_Sudokus(fileName, difficultLevelInfos: string);
// Generiert eine Datei mit 6 arangierten Sudokus zum Ausdrucken
// In: - difficultLevelInfos = 'easy', 'middle', 'hard','mixed'
//     - fileName: Ein Dateinamen
var sudoku: sudokuField;
    difficult_reached: integer;
    difficult:    DifficultLevels;
    level_wanted: oneDifficultLevel;
    i, count_nr, i_max:integer;
    f: text;
    printableSudoku: array[1..6,1..50] of string;
    six_difficult_levels: array[1..6] of oneDifficultLevel;
    Msg: ScrMessages;

begin
  // wenn es nicht ueber main_sudoku aufgerufen wird,
  // sind diese globalen Initialisierungen noetig
//lang:= default_language;                        // setzte die Sprache
//scrolling_displayMethod:=default_displayMethod; // Bildaufbau mit ClrScr oder gotoXY?
//mark_fixSudokuPositions:= default_mark_fixSudokuPositions; //markiere feste Zahlen mit !x ?

  // Initialisierungen
  set_ScreenMessages(Msg, lang);
  difficult_reached:=0;
  level_wanted.MsgText:=''; level_wanted.callValue:=0;
  level_wanted.lowValue:=0; level_wanted.highValue:=0;
  clear_sudoku(sudoku);

  set_difficult_level_definitions(difficult);
  if ((difficultLevelInfos<>'easy') and (difficultLevelInfos<>'middle')
  and (difficultLevelInfos<>'hard') and (difficultLevelInfos<>'mixed')) then
    difficultLevelInfos:='mixed';       // es wurde kein korrekte Level-Info uebergeben

  case difficultLevelInfos of //<easy/middle/hard/mixed>
    'easy':   begin
                six_difficult_levels[1]:=difficult[1];
                six_difficult_levels[2]:=difficult[1];
                six_difficult_levels[3]:=difficult[2];
                six_difficult_levels[4]:=difficult[2];
                six_difficult_levels[5]:=difficult[2];
                six_difficult_levels[6]:=difficult[3];
              end;
    'middle': begin
                six_difficult_levels[1]:=difficult[3];
                six_difficult_levels[2]:=difficult[4];
                six_difficult_levels[3]:=difficult[4];
                six_difficult_levels[4]:=difficult[5];
                six_difficult_levels[5]:=difficult[6];
                six_difficult_levels[6]:=difficult[5];
              end;
    'hard':   begin
                six_difficult_levels[1]:=difficult[5];
                six_difficult_levels[2]:=difficult[6];
                six_difficult_levels[3]:=difficult[6];
                six_difficult_levels[4]:=difficult[7];
                six_difficult_levels[5]:=difficult[7];
                six_difficult_levels[6]:=difficult[8];
              end;
    'mixed':  begin
                six_difficult_levels[1]:=difficult[2];
                six_difficult_levels[2]:=difficult[3];
                six_difficult_levels[3]:=difficult[4];
                six_difficult_levels[4]:=difficult[5];
                six_difficult_levels[5]:=difficult[6];
                six_difficult_levels[6]:=difficult[7];
              end;
  end;

  i_max:=0;
  for count_nr:=1 to 6 do
  begin
    level_wanted:=six_difficult_levels[count_nr];
    gen_sudoku_with_difficult_level(sudoku, level_wanted, difficult_reached);
    one_Sudoku_to_file(sudoku, fileName);
    assign(f, fileName);
    reset(f);
    i:=1;
    while not(eof(f)) do
    begin
      readln(f, printableSudoku[count_nr, i]);
      i:=i+1;
    end;
    i:=i-1;
    if i>=i_max then i_max:=i;
    close(f);
  end;

  assign(f, fileName);
  rewrite(f);
    for i:=1 to i_max do
    begin
      write(f, printableSudoku[1, i]);
      write(f,'   ');
      writeln(f, printableSudoku[2, i]);
    end;

    writeln(f,' ');
    for i:=1 to i_max do
    begin
      write(f, printableSudoku[3, i]);
      write(f,'   ');
      writeln(f, printableSudoku[4, i]);
    end;

    writeln(f,' ');
    for i:=1 to i_max do
    begin
      write(f, printableSudoku[5, i]);
      write(f,'   ');
      writeln(f, printableSudoku[6, i]);
    end;
  close(f);
end;



procedure main_sudoku(parameterOption_lang, parameterOption_displaymethod: string;
                      parameterOption_mode, parameterOption_genPrintable: string);
// >>> Das regulaere SUDOKU-Hauptprogramm
// In: - parameterOption_lang: uebergebener Startparameter <ger/eng>
//     - parameterOption_displaymethod: uebergebener Startparameter <normal/alternative>
//     - parameterOption_mode: uebergebener Startparameter <mark/gray/blue>
//     - parameterOption_genPrintable: uebergebener Startparameter <easy/middle/hard/mixed>
// Hinweis: Es werden zusaetzlich globale Variablen ausgewertet.
var sudoku, solvedSudoku: sudokufield;             // Sudoku-Spielfeld
    integrityErrorFound : boolean;                 // Fehler bei der Definitionsueberpruefung?
    integrityError      : integrityErrorField;     // Nennung eines Fehlers der Definitionspr.
    Tastatur_eing, eing : char;                    // Aufnahme von Tastatureingaben
    MenueWahl           : char;                    // Auf. der Tastatureingabe w. Menuewahl
    cursor              : cursorPosition;          // x und y Koordinaten innerhalb des S.
    difficult_reached   : integer;
    level_wanted        : oneDifficultLevel;
    Msg                 : ScrMessages;

begin
  // Start- Initialisierungen
  ClrScr;
  randomize;                           // Zufallszahlengenerator initialisieren
  difficult_reached:=0;
  level_wanted.MsgText:=''; level_wanted.callValue:=0;
  level_wanted.lowValue:=0; level_wanted.highValue:=0;
  PositionsExamined:=0;
  clear_sudoku(sudoku); clear_sudoku(fixSudoku); clear_sudoku(solvedSudoku);
  cursor.x:=1; cursor.y:=1;
  integrityErrorFound:=false;
  integrityError[1]:=0; integrityError[2]:=0; integrityError[3]:=0;

  // Verarbeite die uebergebenen Startparameter!!!
  // *********************************************
  // -l -language <ger/eng>
  lang:= default_language;                        // setzte die Sprache
  if (parameterOption_lang='ger') then lang:=ger
  else if (parameterOption_lang='eng') then lang:=eng;
  set_ScreenMessages(Msg, lang);                  // lade die Meldungstexte

  // -d -displaymethod <normal/alternative>
  scrolling_displayMethod:=default_displayMethod; // Bildaufbau mit ClrScr oder gotoXY?
   if (parameterOption_displaymethod='alternative') then scrolling_displayMethod:=0
   else if (parameterOption_displaymethod='normal') then scrolling_displayMethod:=1;

  // -m -mode <mark/gray/blue>
  mark_fixSudokuPositions:= default_mark_fixSudokuPositions; //markiere feste Zahlen mit !x ?
  color_fixSudokuPositions:=default_color_fixSudokuPositions;//use colors or not
  if (parameterOption_mode='mark') then
  begin                                                      // Bei Aenderungen
    mark_fixSudokuPositions:=1; color_fixSudokuPositions:=0; // beachte
  end                                                        // procedure
  else if (parameterOption_mode='gray') then                 // text_screen_output
  begin                                                      // und
    mark_fixSudokuPositions:=0; color_fixSudokuPositions:=1; // procedure
  end                                                        // mainMenu
  else if (parameterOption_mode='blue') then                 //
  begin                                                      //
    mark_fixSudokuPositions:=0; color_fixSudokuPositions:=2; //
  end;

//-p -makeprintablesudokus <easy/middle/hard/mixed>
  if ((parameterOption_genPrintable='easy') or (parameterOption_genPrintable='middle')
     or (parameterOption_genPrintable='hard') or (parameterOption_genPrintable='mixed')) then
  begin
    ClrScr;
    make_file_with_6_printable_Sudokus(default_fileName, parameterOption_genPrintable);
    writeln;
    write(Msg[lang,main,1].Hint);          //'>>> The SUDOKUS can be found in the file: '
    writeln(default_fileName);

    writeln;
    NormVideo;
    exit;
  end;

  repeat
    PositionsExamined:=0;
    clear_sudoku(sudoku); clear_sudoku(fixSudoku); clear_sudoku(solvedSudoku);
    cursor.x:=1; cursor.y:=1;
    integrityErrorFound:=false;
    integrityError[1]:=0; integrityError[2]:=0; integrityError[3]:=0;

    MenueWahl:=' '; Tastatur_eing:=' '; eing:=' ';
    mainMenue(MenueWahl);

    if ((MenueWahl='n') or (MenueWahl='g')) then
    begin

      if (MenueWahl='n') then
      begin
        // Neues Sudoku eingeben
        enterNewSudoku(sudoku, cursor, Tastatur_eing);
        copySudoku(sudoku, fixSudoku);
        cursor.x:=0; cursor.y:=0;
        if scrolling_displayMethod=1 then ClrScr; // normaler Bildaufbauf nur ueber gotoXY
      end
      else if (MenueWahl='g') then
      begin
        // Ein Sudoku generieren
        generiereSudokuMenue(Tastatur_eing, level_wanted);
        if ((Tastatur_eing<>'x') and (Tastatur_eing<>#27)) then
        begin
          cursor.x:=0; cursor.y:=0;
          gen_sudoku_with_difficult_level(sudoku,  level_wanted, difficult_reached);
          copySudoku(sudoku, fixSudoku);
          if scrolling_displayMethod=1 then ClrScr; // normaler Bildaufbauf nur ueber gotoXY
        end;
      end;
      // Sudoku loesen
      check_integrity (sudoku, integrityErrorFound, integrityError);
      if (not(integrityErrorFound) and (count_fields_used(sudoku)>0)) then
      begin
        edit_to_solve_sudoku(sudoku, cursor, eing);
        if scrolling_displayMethod=1 then ClrScr; // normaler Bildaufbauf nur ueber gotoXY
      end;
    end;
  until((MenueWahl='x') or (MenueWahl=#27));
  writeln;
  NormVideo;
end;



(* ************************************************************************** *)
(*                      Das Konsole Haendling                                 *)
(*                      =====================                                 *)
(* ************************************************************************** *)
{ Grundstruktur TMySudoku, generiert von der IDE fuer Konsole-Anwendung  }

procedure TMySudoku.DoRun;
// Beim Programmstart uebergebene Optionen und Parameter erkennen und auslesen
var
  parameterOption_lang: string;
  parameterOption_displaymethod: string;
  parameterOption_mode: string;
  parameterOption_genPrintable:string;

begin
  if ParamCount > 0 then
  begin
     // Uebergabeparameter abarbeiten
    if HasOption('h','help') then
    begin
      WriteHelp;
      Terminate;
      Exit;
    end;

    // -l -language <ger/eng>
    parameterOption_lang:='';
    if HasOption('l', 'language') then       // moegl. Parameter: <ger/eng>
    begin
      parameterOption_lang:=LowerCase(getOptionValue('l', 'language'));
      if (debugPrintFlag20_parameterFound= 1) then
      begin
        writeln('Ueberg. Startparameter LANGUAGE erkannt: ', parameterOption_lang);
      end;
    end;

    // -d -displaymethod <normal/alternative>
    parameterOption_displaymethod:='';
    if HasOption('d', 'displaymethod') then  //moegl. Parameter: <normal/alternative>
    begin
      parameterOption_displaymethod:=LowerCase(getOptionValue('d', 'displaymethod'));
      if (debugPrintFlag20_parameterFound= 1) then
      begin
        write('Ueberg. Startparameter DISPLAYMETHODE erkannt: ');
        writeln(parameterOption_displaymethod);
      end;
    end;

    //-m -mode <mark/gray/blue>
    parameterOption_mode:='';
    if HasOption('m', 'mode') then //moegl. Parameter: <mark/gray/blue>
    begin
      parameterOption_mode:=LowerCase(getOptionValue('m', 'mode'));
      if (debugPrintFlag20_parameterFound= 1) then
      begin
        write('Ueberg. Startparameter MODE erkannt: ');
        writeln(parameterOption_mode);
      end;
    end;

    //-p -makeprintablesudokus <easy/middle/hard/mixed>
    parameterOption_genPrintable:='';
    if HasOption('p', 'makeprintablesudokus') then //moegl.Parameter:<easy/middle/hard/mixed>
    begin
      parameterOption_genPrintable:=LowerCase(getOptionValue('p', 'makeprintablesudokus'));
      if parameterOption_genPrintable='' then   // setze default mixed
        parameterOption_genPrintable:='mixed';  // wenn -p ohne Option uebergeben
      if (debugPrintFlag20_parameterFound= 1) then
      begin
        write('Ueberg. Startparameter MAKEPRINTABLESUDOKU: ');
        writeln(parameterOption_genPrintable);
      end;
    end;

  end;
  // ***************************************************
  // * Das eigentliche "Sudoku-Hauptprogramm" aufrufen *
  // ***************************************************
  main_sudoku(parameterOption_lang, parameterOption_displaymethod,
              parameterOption_mode, parameterOption_genPrintable);

  // stop program loop
  Terminate;
end;



procedure TMySudoku.WriteHelp;
// Hielfe die bei Programmstart ueber Parameteroption -h oder -help angefordert
begin
  writeln('-d, -displaymethod [normal|alternative]        screen-output method');
  writeln('-d alternative                This maybe useful if problems occurs.');
  writeln('                              Default display-method is normal.');
//writeln;
  writeln('-h, -help                     This help text is shown.');
//writeln;
  writeln('-l, -language [ger|eng]       Select a language.');
  writeln('-language ger                 Program uses the German language.');
  writeln('-l eng                        Program uses the English language.');
  writeln('                              Default language is English.');
//writeln;
  writeln('-m, -mode [mark|gray|blue]    How to figure out the Sudoku field.');
  writeln('-m mark                       Mark the fix numbers with:(!)');
  writeln('-mode gray                    Use gray color, marker (!) is not used.');
  writeln('-m blue                       Use blue color, marker (!) is not used.');
  writeln('                              Default mode is mark.');
//writeln;
  writeln('-p, -makeprintablesudokus [easy|middle|hard|mixed]');
  writeln('                 make the file: "',default_fileName ,'" ');
  writeln('                 with six generated Sudokus, ready to print out.');
  writeln('-p easy          generates difficult levels: 1, 1, 2, 2, 2, 3');
  writeln('-p middle        generates difficult levels: 3, 4, 4, 5, 6, 5');
  writeln('-p hard          generates difficult levels: 5, 6, 6, 7, 7, 8');
  writeln('-p mixed         generates difficult levels: 2, 3, 4, 5, 6, 7');
  writeln('-p               if no option is given, mixed is used as default.');
  writeln('1:(>45) 2:(41-45) 3:(37-40) 4:(34-36) 5:(32-33) 6:(30-31) 7:(28-29) 8:(<=28)');
  writeln('Hint: Depending on the difficult level, the generating of six Sudokus');
  writeln('      >>> CAN TAKE up to SEVERAL MINUTES! <<<');

end;



constructor TMySudoku.Create(TheOwner: TComponent);
begin
  inherited Create(TheOwner);
  StopOnException:=True;
end;



destructor TMySudoku.Destroy;
begin
  inherited Destroy;
end;



begin
  Application:=TMySudoku.Create(nil);
  Application.Title:='consoleSudoku';
  Application.Run;
  Application.Free;
end.


