%{
/* Configuration file parser
 *
 * Copyright (c) 2002, Martin Hedenfalk <mhe@home.se>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 */

#include <string.h>
#include "confuse.h"

#if ENABLE_NLS
# include <libintl.h>
# define _(str) dgettext(PACKAGE, str)
#else
# define _(str) str
#endif
#define N_(str) str

typedef char * YYSTYPE;
extern YYSTYPE yylval;

#define YY_DECL int yylex YY_PROTO(( cfg_t *cfg ))

/* temporary buffer for the C-style string scanner
 * FIXME: doesn't check for buffer overflow!
 */
char *string_buf_ptr = 0;
char string_buf[1024];

#define MAX_INCLUDE_DEPTH 10
struct {
	YY_BUFFER_STATE state;
	char *filename;
	unsigned int line;
} include_stack[MAX_INCLUDE_DEPTH];
int include_stack_ptr = 0;

%}

%option noyywrap

 /* start conditions
  */
%x comment
%x str
%x sq_str
%x incl1
%x incl2
%x incl3

%%

[ \t]+    /* eat up whitespace */

[\n]	 cfg->line++; /* keep track of line number */

("#"|"//")[^\n]*     /* eat up one-line comments */

 /* special keywords/symbols
  */
"{"         { yylval = yytext; return '{'; }
"}"         { yylval = yytext; return '}'; }
"("         { yylval = yytext; return '('; }
")"         { yylval = yytext; return ')'; }
"="         { yylval = yytext; return '='; }
"+="        { yylval = yytext; return '+'; }
","         { yylval = yytext; return ','; }

 /* handle multi-line C-style comments
  */
"/*"         BEGIN(comment);
<comment>[^*\n]*        /* eat anything that's not a '*' */
<comment>"*"+[^*/\n]*   /* eat up '*'s not followed by '/'s */
<comment>\n             cfg->line++;
<comment>"*"+"/"        BEGIN(INITIAL);

 /* handle C-style strings
  */
"\""    string_buf_ptr = string_buf; BEGIN(str);
<str>\"  { /* saw closing quote - all done */
	BEGIN(INITIAL);
	*string_buf_ptr = '\0';
	yylval = string_buf;
	return CFGT_STR;
 }
<str>$\{[^}]*\} {
	char *var;
	char *e;
	yytext[strlen(yytext) - 1] = 0;
	e = strchr(yytext+2, ':');
	if(e && e[1] == '-')
		*e = 0;
	else
		e = 0;
	var = getenv(yytext+2);
	if(!var && e)
		var = e+2;
	while(var && *var)
		*string_buf_ptr++ = *var++;
}
<str>\\\n { /* allow continuing on next line */
    /* no-op */
}
<str>\\[0-7]{1,3} {
	/* octal escape sequence */
	int result;
	sscanf(yytext + 1, "%o", &result);
	if(result > 0xFF) {
		cfg_error(cfg, _("invalid octal number '%s'"), yytext);
		return 0;
	}
	*string_buf_ptr++ = result;
 }
<str>\\[0-9]+   {
	cfg_error(cfg, _("bad escape sequence '%s'"), yytext);
	return 0;
 }
<str>\\n  *string_buf_ptr++ = '\n';
<str>\\t  *string_buf_ptr++ = '\t';
<str>\\r  *string_buf_ptr++ = '\r';
<str>\\b  *string_buf_ptr++ = '\b';
<str>\\f  *string_buf_ptr++ = '\f';
<str>\\a  *string_buf_ptr++ = '\007';
<str>\\e  *string_buf_ptr++ = '\033';
<str>\\(.|\n)  *string_buf_ptr++ = yytext[1];
<str>[^\\\"]+  {
	char *yptr = yytext;
	while(*yptr)
		*string_buf_ptr++ = *yptr++;
 }

    /* single-quoted word ('...') */
"\'" {
    string_buf_ptr = string_buf;
    BEGIN(sq_str);
}
<sq_str>\' { /* saw closing quite - all done */
    BEGIN(INITIAL);
    *string_buf_ptr = '\0';
    yylval = string_buf;
    return CFGT_STR;
}
<sq_str>\\\n {
    /* no-op */
}
<sq_str>\\[\\\'] {
    *string_buf_ptr++ = yytext[1];
}
<sq_str>\\[^\\\'] {
    *string_buf_ptr++ = yytext[0];
    *string_buf_ptr++ = yytext[1];
}
<sq_str>[^\\\']+ {
    char *cp = yytext;
    while(*cp != '\0')
        *string_buf_ptr++ = *cp++;
}
<sq_str>(.|\n) {
    *string_buf_ptr++ = yytext[1];
}
<sq_str><<EOF>> {
    cfg_error(cfg, _("unterminated string constant"));
    return 0;
}

<<EOF>> {
             if ( --include_stack_ptr < 0 )
                 {
                 return EOF;
                 }

             else
                 {
                 yy_delete_buffer( YY_CURRENT_BUFFER );
                 yy_switch_to_buffer(
                      include_stack[include_stack_ptr].state );
                 free(cfg->filename);
                 cfg->filename = include_stack[include_stack_ptr].filename;
                 cfg->line = include_stack[include_stack_ptr].line;
                 }
}

$\{[^}]*\} {
	char *var;
	char *e;
	yytext[strlen(yytext) - 1] = 0;
	e = strchr(yytext+2, ':');
	if(e && e[1] == '-')
		*e = 0;
	else
		e = 0;
	var = getenv(yytext+2);
	if(!var && e)
		var = e+2;
	if(!var)
		var = "";
	yylval = var;
	return CFGT_STR;
}

 /* an unquoted string
  */
[^ \"\'\t\n={}()+,]+  yylval = yytext; return CFGT_STR;

%%

void cfg_dummy_function(void)
{
	/* please compiler :-)
	 * otherwise "defined but not used" warning
	 */
	yyunput(0, 0);
}

int cfg_lexer_include(cfg_t *cfg, const char *filename)
{
	char *xfilename;

	if(include_stack_ptr >= MAX_INCLUDE_DEPTH) {
		cfg_error(cfg, _("includes nested too deeply"));
		return 1;
	}

	include_stack[include_stack_ptr].state = YY_CURRENT_BUFFER;
	include_stack[include_stack_ptr].filename = cfg->filename;
	include_stack[include_stack_ptr].line = cfg->line;
	include_stack_ptr++;

	xfilename = cfg_tilde_expand(filename);

	/*cfg_error(cfg, "opening %s for parsing", xfilename);*/
	yyin = fopen(xfilename, "r" );

	if(!yyin) {
		cfg_error(cfg, "%s: %s", xfilename, strerror(errno));
		free(xfilename);
		return 1;
	}

	cfg->filename = xfilename;
	cfg->line = 1;

    yy_switch_to_buffer(yy_create_buffer(yyin, YY_BUF_SIZE));
	return 0;
}
