/***************************************************************************
 *   Copyright (C) 2001 by Rick L. Vinyard, Jr.                            *
 *   rvinyard@cs.nmsu.edu                                                  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU Lesser General Public License as        *
 *   published by the Free Software Foundation version 2.1.                *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU Lesser General Public      *
 *   License along with this library; if not, write to the                 *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA              *
 ***************************************************************************/
#ifndef CONEXUSTTY_H
#define CONEXUSTTY_H

#include <conexus/serial.h>

#include <termios.h>
#include <unistd.h>

namespace conexus {

/**
 * This class extends Serial to include operations upon TTY (TeleTYpe) serial
 * communications. This class should be used to control serial port baud rates
 * and other aspects of serial communications.
 *
 * @author Rick L Vinyard Jr
 * @ingroup conexus
 */
class TTY : public Serial
{
public:
    TTY();

    ~TTY();

    /**
     * Aliases for the tty function options relating to when an option change takes effect.
     */
    typedef enum SetOption {
      /** the setting change occurs immediately */
      SET_NOW=TCSANOW,

      /**
       * the setting change occurs after all output written to the object
       * has been transmitted, and all input that has been received but not
       * read will be discarded before the change is made.
      */
      SET_FLUSH=TCSAFLUSH,

      /**
       * the setting change occurs after all output has been transmitted;
       * This function should be used when changing parameters that affect output.
       */
      SET_DRAIN=TCSADRAIN
    } SetOption;

    /**
     * waits until all output written to the object has been transmitted.
     */
    void drain();

    /**
     * discards data received but not read
     */
    void flush_input();

    /**
     * discards data written to the object but not transmitted
     */
    void flush_output();

    /**
     * discards both data written to the object but not transmitted and data
     * received but not read
     */
    void flush();

    /**
     * Suspends transmission of data by the object
     */
    void suspend_output();

    /**
     * Resumes transmission of data by the object
     */
    void restart_output();

    /**
     * Suspends reception of data by the object
     */
    void suspend_input();

    /**
     * Resumes reception of data by the object
     */
    void restart_input();

    /**
     * Sets the input line speed
     */
    void set_input_speed(speed_t speed);

    /**
     * Sets the output line speed
     */
    void set_output_speed(speed_t speed);

    /**
     * Returns the current input line speed
     */
    speed_t get_input_speed();

    /**
     * Returns the current output line speed
     */
    speed_t get_output_speed();

    /**
     * Writes to the TTY
     */
//     virtual ssize_t write(const void* data, size_t size);

    /**
     * Reads up to s bytes from the TTY
     */
  virtual Data read(size_t s = 0) throw (read_error);

    virtual void open() throw (open_error) { Serial::open(); }

    /**
     * Opens the TTY in the specified mode
     */
    virtual void open(const std::string name, int s=READ|WRITE) throw (open_error);

    /**
     * Sets TTY modes [ see termios(3) ]
     */
    void set_input_modes(tcflag_t iflag, SetOption option=SET_NOW);

    /**
     * Sets TTY modes [ see termios(3) ]
     */
    void set_output_modes(tcflag_t oflag, SetOption option=SET_NOW);

    /**
     * Sets TTY modes [ see termios(3) ]
     */
    void set_control_modes(tcflag_t cflag, SetOption option=SET_NOW);

    /**
     * Sets TTY modes [ see termios(3) ]
     */
    void set_local_modes(tcflag_t lflag, SetOption option=SET_NOW);

    /**
     * Sets TTY modes [ see termios(3) ]
     */
    void set_control_characters(int index, cc_t value, SetOption option=SET_NOW);

    /**
     * Gets TTY attributes [ see termios(3) ]
     */
    struct termios get_attributes();

  protected:
    struct termios m_termios, m_origtermios;

    void tcsetattr(SetOption option);


};

}

#endif
