/***************************************************************************
 *   Copyright (C) 2001 by Rick L. Vinyard, Jr.                            *
 *   rvinyard@cs.nmsu.edu                                                  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU Lesser General Public License as        *
 *   published by the Free Software Foundation version 2.1.                *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU Lesser General Public      *
 *   License along with this library; if not, write to the                 *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA              *
 ***************************************************************************/
#ifndef CONEXUSSOCKET_H
#define CONEXUSSOCKET_H

#include <sys/types.h>
#include <utility>

#include <conexus/filedescriptor.h>
#include <conexus/address.h>
#include <conexus/error.h>

namespace conexus {

/**
 * This class encapsulates Linux's BSD Socket API and serves as the base class for all
 * objects performing socket I/O.
 *
 * This class provides the following propertymm properties:
 *  - domain (int)
 *  - type (int)
 *  - protocol (int)
 *
 * @author Rick L Vinyard Jr
 * @ingroup conexus
 */
class Socket: public FileDescriptor {
public:
/**
 * These enumerations are used in the socket class methods, and use is also encouraged in children.
 */
  typedef enum SocketState {
    BOUND    =LASTIOSTATE<<1, /**< The socket is bound to an interface */
    CONNECTED=LASTIOSTATE<<2, /**< The socket is connected (TCP) */
    LISTENING=LASTIOSTATE<<3, /**< The socket is listening for connections (TCP) */
    ACCEPTED =LASTIOSTATE<<4, /**< The socket is an accepted connection (TCP) */
    LASTSOCKETSTATE=ACCEPTED,
  } SocketState;

  /**
   * This default constructor is primarily for children to specify a domain, type and socket protocol.
   */
    Socket(int domain=-1, int type=-1, int protocol=0) throw ();

    ~Socket() throw ();

    /**
     * Creates the socket; similar to the socket() call. Ideally, the socket will
     * be in the CLOSED state before this call.
     *
     * If the socket is in the OPENED state (not BOUND, CONNECTED, LISTENING, ACCEPTED...)
     * this method will return without performing any action.
     *
     * If the socket is in any state other than CLOSED or OPENED the socket will be closed
     * and reopened.
     */
    virtual void open() throw (open_error);

  /**
   * Overloads the parent FileDescriptor class to ensure that the final state
   * clears the BOUND, CONNECTED, LISTENING and ACCEPTED flags.
   */
    virtual void close() throw (close_error);

    /**
     * Binding without an address (autobinding) is a child specific action.
     * By default an attempt to bind without providing an address will result in a thrown error condition.
     * Therefore children should modify this behavior if they wish to provide autobinding.
     */
    virtual void bind() throw (bind_error);

    /**
     * Binds the socket to the provided address.
     *
     * The socket should already be in the OPENED state before this call.
     *
     * If the socket is in the CLOSED state, then set_state(OPENED) will be automatically called.
     */
    virtual void bind(Address& a) throw (bind_error);

    /**
     * Connecting without an address (autoconnecting) is a child specific action.
     * By default an attempt to connect without providing an address will result in a thrown error condition.
     * Therefore children should modify this behavior if they wish to provide autoconnection.
     */
    virtual void connect() throw (connect_error);

    /**
     * Connects the socket to the provided address.
     *
     * If the socket is in the CLOSED state, then set_state(OPENED) will be automatically called.
     *
     * If the provided address is a broadcast address, will also set the broadcast socket option.
     */
    virtual void connect(Address& a) throw (connect_error);

    /**
     * @todo Need to create a default behavior for accepting connections
     */
    virtual void accept() { }

    /**
     * Places the socket in a listening mode; nearly identical to calling listen() on the socket.
     *
     * The socket should already be in the BOUND state before this call.
     *
     * If the socket is in the CLOSED or OPENED states, then set_state(BOUND) will be automatically called.
     */
  virtual void listen(int backlog=0);

    /**
     * Returns the socket descriptor; alias for the parent accessor get_fd()
     */
    int get_sd() throw ();

    /**
     * Returns the communication domain which specifies the protocol family of the socket.
     */
    int get_domain() throw ();

    /**
     * Sets the communication domain which specifies the protocol family of the socket;
     * this will only set the domain internally, but will not have an actual effect until the
     * socket is created, or if the socket is already created it will not have an effect until
     * the socket is closed and recreated.
     *
     * Linux currently defines the following protocol families:
     * <TABLE>
     * <TR><TD>Name</TD>             <TD>Purpose</TD>                       <TD>Man page</TD>  </TR>
     * <TR><TD>PF_UNIX, PF_LOCAL</TD><TD>Local communication</TD>           <TD>unix(7)</TD>   </TR>
     * <TR><TD>PF_INET</TD>          <TD>IPv4 Internet protocols</TD>       <TD>ip(7)</TD>     </TR>
     * <TR><TD>PF_INET6</TD>         <TD>IPv6 Internet protocols</TD>                          </TR>
     * <TR><TD>PF_IPX</TD>           <TD>IPX - Novell protocols</TD>                           </TR>
     * <TR><TD>PF_NETLINK</TD>       <TD>Kernel user interface device</TD>  <TD>netlink(7)</TD></TR>
     * <TR><TD>PF_X25</TD>           <TD>ITU-T X.25 / ISO-8208 protocol</TD><TD>x25(7)</TD>    </TR>
     * <TR><TD>PF_AX25</TD>          <TD>Amateur radio AX.25 protocol</TD>                     </TR>
     * <TR><TD>PF_ATMPVC</TD>        <TD>Access to raw ATM PVCs</TD>                           </TR>
     * <TR><TD>PF_APPLETALK</TD>     <TD>Appletalk</TD><TD>ddp(7)</TD>                         </TR>
     * <TR><TD>PF_PACKET</TD>        <TD>Low level packet interface</TD>    <TD>packet(7)</TD> </TR>
     * </TABLE>
     */
    void set_domain(int) throw ();

    /**
     * Returns the socket type, which defines the communication mechanism of the socket.
     */
    int get_type() throw ();

    /**
     * Sets the socket type, which defines the communication mechanism of the socket.
     *
     * Linux currently defines the following communication types:
     *
     *  - SOCK_STREAM
     *    - Provides sequenced,  reliable,  two-way,  connection-based  byte
     *      streams.  An out-of-band data transmission mechanism may be supported.
     *  - SOCK_DGRAM
     *    - Supports datagrams (connectionless,  unreliable  messages  of  a
     *      fixed maximum length).
     *  - SOCK_SEQPACKET
     *    - Provides  a  sequenced,  reliable, two-way connection-based data
     *      transmission path for datagrams of fixed maximum length; a  consumer
     *      is required to read an entire packet with each read system call.
     *  - SOCK_RAW
     *    - Provides raw network protocol access.
     *  - SOCK_RDM
     *    - Provides a reliable  datagram  layer  that  does  not  guarantee
     *      ordering.
     *  - SOCK_PACKET
     *    - Obsolete  and should not be used in new programs; see packet(7).
     *
     * Some socket types may not be implemented by all protocol families;  for
     * example, SOCK_SEQPACKET is not implemented for AF_INET.
     */
    void set_type(int) throw ();

    /**
     * Returns the specific protocol within the socket's protocol family.
     */
    int get_protocol() throw ();

    /**
     * Sets the specific protocol within the socket's protocol family, but is
     * normally 0.
     */
    void set_protocol(int) throw ();

  virtual Data read(size_t s = 0) throw (read_error);

  virtual ssize_t write(const void* data, size_t size) throw (write_error);
  virtual ssize_t writeto(Address& a, const void* data, size_t size) throw (write_error);


    virtual void set_option(int option, bool b);

  template <typename T>
    void set_option(int level, int optname, T& value);

  template <typename T>
    void get_option(int level, int optname, T& value);

  virtual void change_state(unsigned long states) throw (state_error);

  sigc::signal<void> signal_bound() { return m_signal_bound; }
  sigc::signal<void> signal_connected() { return m_signal_connected; }
  sigc::signal<void> signal_listening() { return m_signal_listening; }

  bool is_bound() { return m_state&BOUND; }
  bool is_connected() { return m_state&CONNECTED; }
  bool is_listening() { return m_state&LISTENING; }
  bool is_accepted() { return m_state&ACCEPTED; }

  protected:
    int m_domain;
    int m_type;
    int m_protocol;

  virtual void service_thread_main();

  virtual void set_state_closed();
  virtual void set_state_bound();
  virtual void set_state_connected();
  virtual void set_state_listening();

  sigc::signal<void> m_signal_bound;
  sigc::signal<void> m_signal_connected;
  sigc::signal<void> m_signal_listening;
};

template <typename T>
  inline
  void Socket::set_option(int level, int optname, T& value) {
    if ( ! is_bound() )
      try {
        change_state(BOUND);
      } catch (error::state::failed) {
        return;
      }

    ::setsockopt(m_fd, level, optname, &value, sizeof(T));
  }

template <typename T>
inline
  void Socket::get_option(int level, int optname, T& value) {
    if ( ! is_bound() )
      try {
        change_state(BOUND);
      } catch (error::state::failed) {
        return;
      }

    socklen_t size = sizeof(T);
    ::getsockopt(m_fd, level, optname, &value, &size);
  }
}

#endif
