/***************************************************************************
 *   Copyright (C) 2001 by Rick L. Vinyard, Jr.                            *
 *   rvinyard@cs.nmsu.edu                                                  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU Lesser General Public License as        *
 *   published by the Free Software Foundation version 2.1.                *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU Lesser General Public      *
 *   License along with this library; if not, write to the                 *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA              *
 ***************************************************************************/
#include "socket.h"

#include <errno.h>
#include <sys/socket.h>

using namespace conexus;

Socket::Socket(int domain, int type, int protocol) throw ():
    FileDescriptor(), m_domain(domain), m_type(type), m_protocol(protocol) {
}

Socket::~Socket() throw () {
}

int Socket::get_sd() throw () {
    return m_fd;
}

int Socket::get_domain() throw () {
    return m_domain;
}

void Socket::set_domain(int d) throw () {
    m_domain = d;
}

int Socket::get_type() throw () {
    return m_type;
}

void Socket::set_type(int t) throw () {
    m_type = t;
}

int Socket::get_protocol() throw () {
    return m_protocol;
}

void Socket::set_protocol(int p) throw () {
    m_protocol = p;
}

void Socket::open() throw (open_error) {
  if ( is_open() )
    change_state(CLOSED);

    m_fd = ::socket(m_domain, m_type, m_protocol);
    if (m_fd == -1)
      throw_open_error(errno);

  set_state_opened();
}

void Socket::close() throw (close_error) {
  int i = ::close(m_fd);
  if (i == -1)
    throw_close_error(errno);

  set_state_closed();
}


void Socket::bind() throw (bind_error) {
  throw error::bind::no_socket_bind();
}

void Socket::bind(Address& a) throw (bind_error) {
    int i;

  if ( is_closed() )
    try {
      change_state(OPENED);
    } catch (error::state::failed) {
      throw error::bind::not_open();
    }

    i = ::bind(m_fd, a.get_raw_address(), a.get_raw_address_size());
    if (i == -1)
      throw_bind_error(errno);

  set_state_bound();
}

void Socket::connect() throw (connect_error) {
  throw error::connect::no_socket_connect();
}

void Socket::connect(Address& a) throw (connect_error) {
    int i;

  if ( is_closed() )
    try {
      change_state(OPENED);
    } catch (error::state::failed) {
      throw error::connect::not_open();
    }

    if (a.is_broadcast())
      set_option(SO_BROADCAST, true);
    else
      set_option(SO_BROADCAST, false);

    i = ::connect(m_fd, a.get_raw_address(), a.get_raw_address_size());
    if (i == -1)
      throw_connect_error(errno);

  set_state_connected();
}

void conexus::Socket::listen( int backlog )
{
  int i;

  if (! is_bound() )
    try {
      change_state(BOUND);
    } catch (error::state::failed) {
      throw error::listen::not_bound();
    }

  i = ::listen(m_fd, backlog);

  if (i == -1)
    throw_listen_error(errno);

  set_state_connected();
}

ssize_t Socket::write(const void* data, size_t size) throw (write_error) {
    ssize_t ret;

  if ( is_write_blocked() )
    return 0;

  if ( ! is_connected() )
    try {
      change_state(CONNECTED);
    } catch (error::state::failed) {
      throw error::write::not_connected();
    }

    ret = ::send(m_fd, data, size, 0);
    if (ret == -1)
      throw_write_error(errno);

    return ret;
}

ssize_t Socket::writeto(Address& a, const void* data, size_t size) throw (write_error) {
    ssize_t ret;

  if ( is_write_blocked() )
    return 0;

  if ( is_closed() )
    try {
      change_state(OPENED);
    } catch (error::state::failed) {
      throw error::write::not_socket();
    }

    if (a.is_broadcast())
      set_option(SO_BROADCAST, true);
    else
      set_option(SO_BROADCAST, false);

    ret = ::sendto(m_fd, data, size, 0, a.get_raw_address(), a.get_raw_address_size());

    if (ret == -1)
      throw_write_error(errno);

      return ret;
}

Data Socket::read(size_t s) throw (read_error) {
    Data d;

  if (is_read_blocked())
    return d;

  if (! is_bound() )
    try {
      change_state(BOUND);
    } catch (error::state::failed) {
      throw error::read::not_bound();
    }

    d.data = Data::sptr(new Data::Octet[s+1]);
    d.size = ::recvfrom(m_fd, d.data.get(), s, 0, NULL, NULL);

    return d;
}

void Socket::set_option(int option, bool b) {
  int i = b;

  if ( ! is_bound() )
    try {
      change_state(BOUND);
    } catch (error::state::failed) {
      return;
    }

  ::setsockopt(m_fd, SOL_SOCKET, option, &i, sizeof(int));
}

void Socket::service_thread_main() {
  // ensure that the state is at least BOUND
  if ( ! is_bound() )
    try {
      change_state(BOUND);
    } catch (error::state::failed) {
      throw error::read::not_bound();
    }

  FileDescriptor::service_thread_main();
}

void conexus::Socket::change_state( unsigned long states ) throw (state_error)
{
  // check for inconsistent states
  // if it's CLOSED it can't be BOUND, CONNECTED, LISTENING or ACCEPTED
  if (states & CLOSED && ( states&BOUND || states&CONNECTED || states&LISTENING || states&ACCEPTED))
    throw error::state::inconsistent();

  // call for CLOSED, OPENED and UNCHANGED
  // also checks for existing state
  // if it does the job, go ahead and return
  IO::change_state(states);

  if ( ! is_bound() && states&BOUND )
    bind();
  if ( ! is_connected() && states&CONNECTED )
    connect();
  if ( ! is_listening() && states&LISTENING )
    listen();
}

void Socket::set_state_closed() {
  m_state &= ~BOUND;
  m_state &= ~CONNECTED;
  m_state &= ~LISTENING;
  m_state &= ~ACCEPTED;
  if (is_closed())
    return;
  IO::set_state_closed();
}

void Socket::set_state_bound() {
  if (is_bound())
    return;
  m_state |= BOUND;
  m_signal_bound.emit();
}

void Socket::set_state_connected() {
  if (is_connected())
    return;
  m_state |= CONNECTED;
  m_signal_connected.emit();
}

void Socket::set_state_listening() {
  if (is_listening())
    return;
  m_state |= LISTENING;
  m_signal_listening.emit();
}



