/***************************************************************************
 *   Copyright (C) 2001 by Rick L. Vinyard, Jr.                            *
 *   rvinyard@cs.nmsu.edu                                                  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU Lesser General Public License as        *
 *   published by the Free Software Foundation version 2.1.                *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU Lesser General Public      *
 *   License along with this library; if not, write to the                 *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA              *
 ***************************************************************************/
#ifndef CONEXUSSERIAL_H
#define CONEXUSSERIAL_H

#include <conexus/filedescriptor.h>

namespace conexus {

/**
 * This class encapsulates a raw serial port. It is likely that it will be extended one
 * level deeper in the inheritance hierarchy and replaced by an ancestor that provides
 * file I/O to be used with traditional files and with Unix named pipes.
 *
 * @author Rick L Vinyard Jr
 * @ingroup conexus
 */
class Serial : public FileDescriptor {
public:
    /**
     * The serial port open modes. These have been selected so that masks may be created through
     * the bitwise operators.
     */
    typedef enum SerialState {
      READ=LASTIOSTATE<<1,           /**< opened for reading */
      WRITE=LASTIOSTATE<<2,            /**< opened for writing */
      LASTSERIALSTATE=WRITE,
    } SerialState;

    Serial(unsigned long readwrite=READ|WRITE);

    ~Serial();

    /**
     * Overrides parent's pure virtual method to provide for writing to the serial port.
     */
  virtual ssize_t write(const void* data, size_t size) throw (write_error);

    /**
     * Overrides parent's pure virtual method to provide for reading from the serial port.
     */
  virtual Data read(size_t s = 0) throw (read_error);

    /**
     * Opens the named serial port device in the specified mode.
     */
    virtual void open(const std::string name, int state=READ|WRITE) throw (open_error);

    /**
     * Will attempt to open the serial port if a device has been specified. Otherwise an
     * error will be thrown.
     */
    virtual void open() throw (open_error);

    /**
     * Will return the string representing the device associated with this serial object.
     */
    std::string get_device();

    /**
     * Sets the string representing the device associated with this serial object.
     *
     * If the serial device is currently open and d is not equal to the current device,
     * the current device will be closed.
     *
     * Also, if auto_open is set, the new device will be immediately opened.
     *
     * Additionally, this method checks whether the parameter d is the same as the current device,
     * and parameter mode is the same as the current mode.
     * If both are equal, the device will not be closed and reopened unless force=true.
     */
    void set_device(std::string d, int state=READ|WRITE) throw (open_error);

    virtual void change_state(long unsigned new_state) throw (state_error);

    sigc::signal<void> signal_device_changed() { return m_signal_device_changed; }
  sigc::signal<void,bool> signal_readable_changed() { return m_signal_readable_changed; }
  sigc::signal<void,bool> signal_writable_changed() { return m_signal_writable_changed; }

  bool is_readable() { return m_state&READ; }
  bool is_writable() { return m_state&WRITE; }

  protected:
    std::string m_device;

    sigc::signal<void> m_signal_device_changed;
  sigc::signal<void,bool> m_signal_readable_changed;
  sigc::signal<void,bool> m_signal_writable_changed;

  void set_state_readable(bool b);
  void set_state_writable(bool b);
};

}

#endif
