/***************************************************************************
 *   Copyright (C) 2001 by Rick L. Vinyard, Jr.                            *
 *   rvinyard@cs.nmsu.edu                                                  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU Lesser General Public License as        *
 *   published by the Free Software Foundation version 2.1.                *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU Lesser General Public      *
 *   License along with this library; if not, write to the                 *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA              *
 ***************************************************************************/
#include "packetcapture.h"


using namespace conexus;

char PacketCapture::m_static_errbuf[PCAP_ERRBUF_SIZE];

PacketCapture::PacketCapture(std::string device, int snaplen, bool promiscuous, int timeout_ms):
    IO(),
    m_pcapd(NULL),
    m_device(device),
    m_snapshot_length(snaplen),
    m_promiscuous(promiscuous),
    m_timeout_ms(timeout_ms),
    m_capture_header(NULL),
    m_capture_data(NULL),
    m_responsiveness(1000)
{
  pthread_mutex_init(&m_mutex_working, NULL);
}


PacketCapture::~PacketCapture()
{
  close();
}

void PacketCapture::open() throw (open_error)
{
  if (m_device.size() > 0)
    m_pcapd = pcap_open_live(m_device.c_str(), m_snapshot_length, m_promiscuous, m_timeout_ms, m_errbuf);
  else
    m_pcapd = pcap_open_live(NULL, m_snapshot_length, m_promiscuous, m_timeout_ms, m_errbuf);
  if (m_pcapd == NULL)
    throw; // TODO: come back and create a real error set to throw
  set_state_opened();
}

void PacketCapture::close() throw (close_error)
{
  if (m_pcapd != NULL)
    pcap_close(m_pcapd);
  m_pcapd = NULL;
  set_state_closed();
}

Data PacketCapture::read(size_t s) throw (read_error)
{
  int result;

  if (is_read_blocked())
    return Data();

  if (is_closed())
    open();

  if (m_pcapd != NULL)
    result = pcap_next_ex(m_pcapd, &m_capture_header, (const u_char**)&m_capture_data);
  if (result == 1)
    return Data(m_capture_data, m_capture_header->caplen);
  else
    return Data();
}

ssize_t PacketCapture::write(CData data)
{
  throw error::write::read_only();
}

ssize_t PacketCapture::write(const void* data, size_t size) throw (write_error)
{
  throw error::write::read_only();
}

ssize_t PacketCapture::write(Data data)
{
  throw error::write::read_only();
}

PacketCapture::Devices conexus::PacketCapture::get_all_devices( )
{
  pcap_if_t *pcap_devices, *pcap_iter;
  pcap_addr_t *pcap_address;
  std::vector<Device> devices;
  int result;
  Device device;

  result = pcap_findalldevs(&pcap_devices, m_static_errbuf);
  pcap_iter = pcap_devices;
  if (result == 0) {
    while (pcap_iter != NULL) {
      device = Device();
      if (pcap_iter->name)
        device.name = pcap_iter->name;
      if (pcap_iter->description)
        device.description = pcap_iter->description;
      if (pcap_iter->flags == PCAP_IF_LOOPBACK)
        device.loopback = true;
      pcap_address = pcap_iter->addresses;
      while (pcap_address != NULL) {
        if (pcap_address->addr)
          device.addresses[ADDRESS] = IP::Address(*(pcap_address->addr));
        if (pcap_address->netmask)
          device.addresses[NETMASK] = IP::Address(*(pcap_address->netmask));
        if (pcap_address->broadaddr)
          device.addresses[BROADCAST] = IP::Address(*(pcap_address->broadaddr));
        if (pcap_address->dstaddr)
          device.addresses[DESTINATION] = IP::Address(*(pcap_address->dstaddr));
        pcap_address = pcap_address->next;
      }
      devices.push_back(device);
      pcap_iter = pcap_iter->next;
    }
    pcap_freealldevs(pcap_devices);
  }

  return devices;
}

std::string PacketCapture::device() const {
    return m_device;
}


void PacketCapture::set_device(const std::string& value) {
    m_device = value;
}


int PacketCapture::snapshot_length() const {
    return m_snapshot_length;
}


void PacketCapture::set_snapshot_length(const int& value) {
    m_snapshot_length = value;
}


bool PacketCapture::promiscuous() const {
    return m_promiscuous;
}


void PacketCapture::set_promiscuous(bool value) {
    m_promiscuous = value;
}


int PacketCapture::timeout_ms() const {
    return m_timeout_ms;
}


void PacketCapture::set_timeout_ms(const int& value) {
    m_timeout_ms = value;
}

void PacketCapture::pcap_callback(u_char* object, const struct pcap_pkthdr* pkthdr, const u_char* packet) {
  PacketCapture* pc = (PacketCapture*)object;
  pthread_mutex_lock(&(pc->m_mutex_working));
  pc->queue_received_data(Data(packet, pkthdr->caplen));
  pthread_mutex_unlock(&(pc->m_mutex_working));
}

void PacketCapture::service_thread_main( )
{
  m_running = false;

  if ( is_closed() )
    open();

  int result = pcap_loop(m_pcapd, -1, pcap_callback, (u_char*)this);

}

void PacketCapture::stop( )
{
  pcap_breakloop(m_pcapd);
  pthread_mutex_lock(&m_mutex_working);
  pthread_cancel(m_thread);
  Server::stop();
  pthread_mutex_unlock(&m_mutex_working);
}

void PacketCapture::set_responsiveness( long r )
{
  m_responsiveness = r;
}

long PacketCapture::get_responsiveness( )
{
  return m_responsiveness;
}
