/***************************************************************************
 *   Copyright (C) 2001 by Rick L. Vinyard, Jr.                            *
 *   rvinyard@cs.nmsu.edu                                                  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU Lesser General Public License as        *
 *   published by the Free Software Foundation version 2.1.                *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU Lesser General Public      *
 *   License along with this library; if not, write to the                 *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA              *
 ***************************************************************************/
#include "ip_udp.h"

#include <sys/ioctl.h>
#include <linux/sockios.h>

using namespace conexus::IP;

UDP::UDP():
    IP(SOCK_DGRAM, IPPROTO_UDP),
    m_write_without_connect(false)
{}


UDP::~UDP() throw()
{}

void UDP::set_interface(const std::string addr)
{
  m_local_address.set_hostname(addr);
  if (m_local_address.is_broadcast())
    set_option(SO_BROADCAST, true);
  else
    set_option(SO_BROADCAST, false);
}

void UDP::set_local_port(u_int16_t port)
{
  IP::set_local_port(port);
  // if the socket is already bound, close and reopen to same state with new local port
  if ( is_bound() )
    close_and_reopen();
}

void UDP::set_remote_port(u_int16_t port)
{
  IP::set_remote_port(port);
  // if the socket is already connected, close and reopen to same state with new remote port
  if ( is_connected() )
    close_and_reopen();
}

int UDP::get_pending_input_datagram_size()
{
  int rsize;
  int error;
  error = ioctl(m_fd, SIOCINQ, &rsize);
  return rsize;
}

int UDP::get_pending_output_datagram_size()
{
  int rsize;
  int error;
  error = ioctl(m_fd, SIOCOUTQ, &rsize);
  return rsize;
}

ssize_t UDP::write(const void* data, size_t size) throw (write_error)
{
  if (m_write_without_connect)
    {
      if (!m_remote_address_set)
        throw error::write::no_default_remote_address();

      if ( is_closed() )
        try
          {
            change_state(OPENED);
          }
        catch (error::state::failed)
          {
            throw error::write::not_opened();
          }

      return Socket::writeto(m_remote_address, data, size);
    }
  else
    {
      if ( ! is_connected() )
        try
          {
            change_state(CONNECTED);
          }
        catch (error::state::failed)
          {
            throw error::write::not_connected();
          }

      return Socket::write(data, size);
    }
}

conexus::Data UDP::read(size_t s) throw (read_error)
{
  if (! is_bound() )
    try {
      change_state(BOUND);
    } catch (error::state::failed) {
         throw error::read::not_bound();
    }

  return Socket::read(get_pending_input_datagram_size());
}

void UDP::set_local_address( Address addr )
{
  IP::set_local_address(addr);
  if ( is_bound() )
    close_and_reopen();
}


void UDP::set_remote_address( Address addr )
{
  IP::set_remote_address(addr);
  if ( is_connected() )
    close_and_reopen();
}

void UDP::connect( ) throw( connect_error )
{
  if (m_remote_address_set)
    Socket::connect(m_remote_address);
  else
    throw error::connect::no_remote_address();
}


bool UDP::get_write_without_connect() const
  {
    return m_write_without_connect;
  }


void UDP::set_write_without_connect(bool value)
{
  m_write_without_connect = value;
}
