/***************************************************************************
 *   Copyright (C) 2001 by Rick L. Vinyard, Jr.                            *
 *   rvinyard@cs.nmsu.edu                                                  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU Lesser General Public License as        *
 *   published by the Free Software Foundation version 2.1.                *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU Lesser General Public      *
 *   License along with this library; if not, write to the                 *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA              *
 ***************************************************************************/
#include "ip_address.h"

#include <netdb.h>

using namespace conexus::IP;

Address::Address(in_addr_t address, u_int16_t port) {
    set_family(AF_INET);
    set_address(address);
    set_port(port);
}

Address::Address(const std::string address, u_int16_t port) {
    set_family(AF_INET);
    set_address(address);
    set_port(port);
}


Address::Address(const std::string net,
                 const std::string local,
                 u_int16_t port) {
    set_family(AF_INET);
    set_address(net, local);
    set_port(port);
}

Address::Address( struct sockaddr_in & addr )
{
  set_family(AF_INET);
  memcpy(&m_sockaddr_in, &addr, sizeof(struct sockaddr_in));
}

Address::Address( struct sockaddr & addr )
{
  set_family(AF_INET);
  if (addr.sa_family == AF_INET)
    memcpy(&m_sockaddr_in, &addr, sizeof(struct sockaddr_in));
}

Address::~Address() {}

sa_family_t Address::get_family() const {
    return m_sockaddr_in.sin_family;
}

void Address::set_family(sa_family_t f) {
    m_sockaddr_in.sin_family = f;
}

u_int16_t Address::get_port() const {
    return ntohs(m_sockaddr_in.sin_port);
}

void Address::set_port(u_int16_t p) {
    m_sockaddr_in.sin_port = htons(p);
}

in_addr_t Address::get_address() const {
    return ntohl(m_sockaddr_in.sin_addr.s_addr);
}

in_addr_t Address::get_network() const {
    return inet_netof(m_sockaddr_in.sin_addr);
}

in_addr_t Address::get_local() const {
    return inet_lnaof(m_sockaddr_in.sin_addr);
}

void Address::set_address(in_addr_t a) {
    m_sockaddr_in.sin_addr.s_addr = htonl(a);
}

void Address::set_address(in_addr_t net, in_addr_t local) {
    m_sockaddr_in.sin_addr = inet_makeaddr(htonl(net), htonl(local));

}

void Address::set_hostname(const std::string name) {
  struct hostent *h;
  h = gethostbyname( name.c_str() );
  if (h == NULL)
    return;
  set_family(h->h_addrtype);
  memcpy((char *) &m_sockaddr_in.sin_addr.s_addr, h->h_addr_list[0], h->h_length);
}

bool Address::is_valid_hostname(const std::string name) const {
  struct hostent *h;
  h = gethostbyname( name.c_str() );
  if (h == NULL)
    return false;
  else
    return true;
}

std::string Address::get_hostname() const {
  struct hostent *h;
  h = gethostbyaddr((char*)&m_sockaddr_in.sin_addr.s_addr, 4, AF_INET);
  if (h != NULL)
    return h->h_name;
  else
    return "";
}

void Address::set_network(in_addr_t net) {
    set_address(net, get_local());
}

void Address::set_local(in_addr_t local) {
    set_address(get_network(), local);
}

std::string Address::get_address_string() const {
    return inet_ntoa(m_sockaddr_in.sin_addr);
}

std::string Address::get_network_string() const {
    struct in_addr sa;
    sa.s_addr = htonl(get_network());
    return inet_ntoa(sa);
}

std::string Address::get_local_string() const {
    struct in_addr sa;
    sa.s_addr = htonl(get_local());
    return inet_ntoa(sa);
}

void Address::set_address(const std::string addr) {
  inet_aton(addr.c_str(), (in_addr*)&m_sockaddr_in);
}

void Address::set_address(const std::string net, const std::string local) {
    set_address(inet_network(net.c_str()), inet_network(local.c_str()));
}

void Address::set_network_address(const std::string net) {
    set_network(inet_network(net.c_str()));
}

void Address::set_local_address(const std::string local) {
    set_local(inet_network(local.c_str()));
}

struct sockaddr_in& Address::get_sockaddr_in() {
    return m_sockaddr_in;
}

struct sockaddr_in* Address::get_sockaddr_in_ptr() {
  return &m_sockaddr_in;
}

struct sockaddr* Address::get_raw_address() {
  return (sockaddr*)&m_sockaddr_in;
}

socklen_t Address::get_raw_address_size() const {
  return sizeof(m_sockaddr_in);
}

bool Address::is_private() const {
  in_addr_t addr = get_address();
  if ((addr & 0xFF000000) == 0x10000000)
    return true;
  if ((addr & 0xFFF00000) == 0xAC100000)
    return true;
  if ((addr & 0xFFFF0000) == 0xC0A80000)
    return true;
  return false;
}

bool Address::is_reserved() const {
  in_addr_t addr = get_address();
  if ((addr & 0xFFFF0000) == 0x00000000)
    return true;
  if ((addr & 0xF8000000) == 0xF0000000)
    return true;
  return false;
}

bool Address::is_loopback() const {
  in_addr_t addr = get_address();
  if ((addr & 0xFF000000) == (IN_LOOPBACKNET << 24))
    return true;
  return false;
}

bool Address::is_broadcast() const {
  in_addr_t addr = get_address();
  if (addr == INADDR_BROADCAST)
    return true;
  return false;
}

bool Address::is_multicast() const {
  in_addr_t addr = get_address();
  if (IN_MULTICAST(addr))
    return true;
  return false;
}


