/***************************************************************************
 *   Copyright (C) 2001 by Rick L. Vinyard, Jr.                            *
 *   rvinyard@cs.nmsu.edu                                                  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU Lesser General Public License as        *
 *   published by the Free Software Foundation version 2.1.                *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU Lesser General Public      *
 *   License along with this library; if not, write to the                 *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA              *
 ***************************************************************************/
#ifndef CONEXUSIO_H
#define CONEXUSIO_H

#include <conexus/error.h>
#include <conexus/server.h>

namespace conexus
  {

  /**
   * This class is the top level object for all subclasses performing I/O (Input/Output)
   * operations.
   *
   * read/write methods are pure virtual methods in this base class and it is the responsibility
   * of all children to provide proper implementations that may go beyond the traditional read(2)
   * and write(2) functions standard in Linux/Unix.
   *
   * Children are also responsible for maintenance of the file descriptor member @em m_fd and the
   * integer state value @em m_state.
   *
   * This class inherits from propertymm::Object and thus makes some properties available via
   * the propertymm framework as well as providing direct accessor methods.
   *
   * @author Rick L Vinyard Jr
   * @ingroup conexus
   */
  class IO: public Server
    {
    public:
      /**
       * These enumerations are used in the socket class methods, and use is also encouraged in children.
       */
      typedef enum IOState {
        NOSTATE=0x00,
        CLOSED=1<<0,    /**< The IO object is in a closed state */
        OPENED=1<<1,   /**< The IO object is in an opened state */
        LASTIOSTATE=OPENED,
        UNCHANGED=~0x00, /**< setting this mode will keep the mode unchanged */
      } IOState;

      /**
       * Default constructor setting state to 0 and file descriptor to -1 (which should be an
       * invalid fd on just about every POSIX system.
       *
       * This constructor simply sets the values according to the parameters and performs no
       * real actions with the file descriptor.
       */
      IO(bool close_on_destruction=true);

      /**
       * Destructor does nothing. It will not call @em close() upon the file descriptor or perform
       * any other cleanup; these are the responsibility of the children.
       */
      ~IO();

      /**
       * Will attempt to open the object without any additional information.
       */
      virtual void open() throw (open_error) = 0;

      /**
       */
      virtual void close() throw (close_error) = 0;

      /**
       * A pure virtual method that must be reimplemented by children to perform whatever actions
       * necessary to ensure writing/transmission of a block of data pointed to by @em data and of
       * @em size bytes.
       *
       * @return The number of bytes actually written.
       * @param data Pointer to the raw data block to be written.
       * @param size Size in bytes of the raw data block to write.
       */
      virtual ssize_t write(const void* data, size_t size) throw (write_error) = 0;

      /**
       * Write data according to virtual method defined by children.
       * @return The number of bytes actually written.
       */
      virtual ssize_t write(Data data);

      /**
       * Write constant data according to virtual method defined by children.
       * @return The number of bytes actually written.
       */
      virtual ssize_t write(CData data);

      /**
       * A pure virtual method that must be reimplemented by children to perform whatever actions
       * necessary to read a block of data. Children may utilize the parameter @em s or may choose
       * to ignore the parameter entirely.
       *
       * @return A Data object containing the data and size of data in bytes. The data
       *    is dynamically allocated via a smart pointer, and deallocated in accordance
       *    with the rules for smart pointers. Therefore, if you want to keep the data
       *    from being automatically deallocated, keep the returned data object (or a
       *    copy of the returned data object) around as long as you need it. Call by value
       *    is sufficient to keep the object around, but extracting the pointer to the actual
       *    data via the dereference operator or through the smart pointer's get() method is
       *    not sufficient to keep the data block from being deallocated (again, keeping a
       *    copy around is easier).
       *
       * @param s The size of the data block to read. Suggested semantics for children are:
       *   - s=0: read as much as possible
       *   - s>0: read up to s bytes
       */
      virtual Data read(size_t s = 0) throw (read_error) = 0;

      /**
       * It is intended that children will provide their own implementation of
       * this method to take into account the actions necessary for any additional
       * states they may introduce.
       *
       * This method as implemented recognizes the OPEN, CLOSE and UNCHANGED states
       * and calls the virtual open, close (or no call) as is appropriate.
       *
       * This method does not actually modify the m_state member. It is the
       * responsibility of the called virtual methods open() and close() to
       * actually perform the modification.
       *
       * @return true if all requested state changes occurred, false otherwise
       */
      virtual void change_state(unsigned long new_state) throw (state_error);

      void set_close_on_destruction(bool value);

      bool close_on_destruction() const;

      /**
       * Close the I/O point and reopen to a new (or same state).
       *
       * If state is UNCHANGED will reopen to the previous state.
       *
       * close and reopen is accomplished by calling change_state(CLOSED) and then
       * change_state(new_state), not by calling close directly.
       */
      void close_and_reopen(unsigned long state=UNCHANGED);

      sigc::signal<void> signal_opened() { return m_signal_opened; }
      sigc::signal<void> signal_closed() { return m_signal_closed; }
      sigc::signal<void, bool, bool> signal_read_write_block_changed() { return m_signal_read_write_block_changed; }

      bool is_open() { return m_state&OPENED; }
      bool is_closed() { return m_state&CLOSED; }

      bool is_read_blocked() { return m_read_blocked; }
      bool is_write_blocked() { return m_write_blocked; }
      virtual void block_read(bool read_block=true);
      virtual void block_write(bool write_block=true);
      virtual void block_read_write(bool read_block=true, bool write_block=true);

      unsigned long get_state() { return m_state; }

    protected:
      bool m_close_on_destruction;
      long unsigned m_state;
      bool m_readable;
      bool m_writable;
      bool m_read_blocked;
      bool m_write_blocked;

      virtual void set_state_opened();
      virtual void set_state_closed();

      sigc::signal<void> m_signal_opened;
      sigc::signal<void> m_signal_closed;
      sigc::signal<void, bool, bool> m_signal_read_write_block_changed;
};

}

conexus::IO& operator<<(conexus::IO& io, conexus::Data d);
conexus::IO& operator<<(conexus::IO& io, conexus::CData d);
conexus::IO& operator>>(conexus::IO& io, conexus::Data d);

#endif
