/***************************************************************************
 *   Copyright (C) 2001 by Rick L. Vinyard, Jr.                            *
 *   rvinyard@cs.nmsu.edu                                                  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU Lesser General Public License as        *
 *   published by the Free Software Foundation version 2.1.                *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU Lesser General Public      *
 *   License along with this library; if not, write to the                 *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA              *
 ***************************************************************************/
#ifndef CONEXUSFDSERVER_H
#define CONEXUSFDSERVER_H

#include <set>

#include <sys/select.h>
#include <pthread.h>

#include <sigc++/sigc++.h>

#include <conexus/socket.h>
#include <conexus/server.h>

namespace conexus {

/**
@author Rick L Vinyard Jr
 * @ingroup conexus
 */
class FDServer: public Server {
public:
  FDServer();
  ~FDServer();

  void add_io(FileDescriptor& io);
  void remove_io(FileDescriptor& io);

  void add_socket(Socket& socket);
  void remove_socket(Socket& socket);

  /**
   * Sets the responsiveness to add_*, remove_* and stop commands in microseconds;
   * lower is more responsive; larger is less responsive; default is 5000 microseconds (or 5 ms).
   *
   * In general this refers to the timeout period of the @em select() call used internally.
   * Each cycle of the server I/O thread acquires several mutex locks at the beginning
   * of the cycle and releases them at the end of the cycle.
   *
   * This value shortens the cycle period, thereby making the server more responsive to commands at
   * the cost of increasing overhead.
   */
  void set_responsiveness(long r);

  /**
   * Returns the server responsiveness level.
   */
  long get_responsiveness();

protected:
  std::set<FileDescriptor*> m_io;
  fd_set m_readfd;
  int m_n;
  struct timeval m_timeval;
  struct timespec m_timespec;
  long m_responsiveness;

  pthread_mutex_t m_mutex_ioset;

  virtual void create_read_set();

  virtual void service_thread_main();

};

}

#endif
