/***************************************************************************
 *   Copyright (C) 2001 by Rick L. Vinyard, Jr.                            *
 *   rvinyard@cs.nmsu.edu                                                  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU Lesser General Public License as        *
 *   published by the Free Software Foundation version 2.1.                *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU Lesser General Public      *
 *   License along with this library; if not, write to the                 *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA              *
 ***************************************************************************/
#ifndef CONEXUSWRITE_ERROR_H
#define CONEXUSWRITE_ERROR_H

#include <conexus/error.h>

namespace conexus
  {

  /**
   * Base class for all writing errors.
   * @author Rick L Vinyard Jr
   * @ingroup errors
   */
  class write_error: public conexus_error
    {
    public:
      write_error(int e=0): conexus_error("Unknown error, write failed.",e,"write")
      { }
      write_error(const std::string s, int e=0): conexus_error(s,e,"write")
      { }
    }
  ;

  namespace error
    {

    namespace write
      {

      /**
       * EAGAIN or EWOULDBLOCK
       * The socket is marked non-blocking and  the  requested  operation
       * would block.
       * @author Rick L Vinyard Jr
       * @ingroup errors
       */
      class would_block: public write_error
        {
        public:
          would_block(): write_error("The socket is marked non-blocking and  the  requested  operation would block.", EWOULDBLOCK)
          { }
        }
      ;

      /**
       * EBADF: Thrown when equivalent BSD socket API error is encountered. May be caught alone
       * or via parent class.
       * @author Rick L Vinyard Jr
       * @ingroup errors
       */
      class bad_fd: public write_error
        {
        public:
          bad_fd(): write_error("sockfd is not a valid descriptor.", EBADF)
          { }
        }
      ;

      /**
       * ECONNRESET
       * Connection reset by peer.
       * @author Rick L Vinyard Jr
       * @ingroup errors
       */
      class connection_reset: public write_error
        {
        public:
          connection_reset(): write_error("Connection reset by peer.", ECONNRESET)
          { }
        }
      ;

      /**
       * EDESTADDRREQ
       * The socket is not connection-mode, and no peer address is set.
       * @author Rick L Vinyard Jr
       * @ingroup errors
       */
      class destination_address_required: public write_error
        {
        public:
          destination_address_required(): write_error("The socket is not connection-mode, and no peer address is set.", EDESTADDRREQ)
          { }
        }
      ;


      /**
       * EFAULT
       * An invalid user space address was specified for a parameter.
       * @author Rick L Vinyard Jr
       * @ingroup errors
       */
      class invalid_user_space_address: public write_error
        {
        public:
          invalid_user_space_address(): write_error("An invalid user space address was specified for a parameter.", EFAULT)
          { }
        }
      ;

      /**
       * EINTR
       * A signal occurred before any data was transmitted.
       * @author Rick L Vinyard Jr
       * @ingroup errors
       */
      class interrupted: public write_error
        {
        public:
          interrupted(): write_error("A signal occurred before any data was transmitted.", EINTR)
          { }
        }
      ;


      /**
       * EINVAL
       * Invalid argument passed.
       * @author Rick L Vinyard Jr
       * @ingroup errors
       */
      class invalid_argument: public write_error
        {
        public:
          invalid_argument(): write_error("Invalid argument passed.", EINVAL)
          { }
        }
      ;

      /**
       * EISCONN
       * The connection-mode socket was connected already but a recipient
       * was specified.  (Now either  this  error  is  returned,  or  the
       * recipient specification is ignored.)
       * @author Rick L Vinyard Jr
       * @ingroup errors
       */
      class is_connected: public write_error
        {
        public:
          is_connected(): write_error("The connection-mode socket was connected already but a recipient was specified.", EISCONN)
          { }
        }
      ;


      /**
       * EMSGSIZE
       * The  socket  type  requires that message be sent atomically, and
       * the size of the message to be sent made this impossible.
       * @author Rick L Vinyard Jr
       * @ingroup errors
       */
      class message_size: public write_error
        {
        public:
          message_size(): write_error("The  socket  type  requires that message be sent atomically, and the size of the message to be sent made this impossible.", EMSGSIZE)
          { }
        }
      ;


      /**
       * ENOBUFS
       * The output queue for a network interface was full.  This  gener-
       * ally  indicates  that the interface has stopped writing, but may
       * be caused by transient congestion.   (Normally,  this  does  not
       * occur  in Linux. Packets are just silently dropped when a device
       * queue overflows.)
       * @author Rick L Vinyard Jr
       * @ingroup errors
       */
      class no_buffers: public write_error
        {
        public:
          no_buffers(): write_error("The output queue for a network interface was full.", ENOBUFS)
          { }
        }
      ;


      /**
       * ENOMEM
       * No memory available.
       * @author Rick L Vinyard Jr
       * @ingroup errors
       */
      class no_memory: public write_error
        {
        public:
          no_memory(): write_error("No memory available.", ENOMEM)
          { }
        }
      ;

      /**
       * EACCES
       * Permission denied.
       * @author Rick L Vinyard Jr
       * @ingroup errors
       */
      class permission_denied: public write_error
        {
        public:
          permission_denied(): write_error("Permission denied.", EACCES)
          { }
        }
      ;

      /**
       * ENOTCONN
       * The socket is not connected, and no target has been given.
       * @author Rick L Vinyard Jr
       * @ingroup errors
       */
      class not_connected: public write_error
        {
        public:
          not_connected(): write_error("The socket is not connected, and no target has been given.", ENOTCONN)
          { }
        }
      ;

      /**
       * ENOTSOCK
       * The argument s is not a socket.
       * @author Rick L Vinyard Jr
       * @ingroup errors
       */
      class not_socket: public write_error
        {
        public:
          not_socket(): write_error("The argument s is not a socket.", ENOTSOCK)
          { }
        }
      ;


      /**
       * EOPNOTSUPP
       * Some bit in the flags argument is inappropriate for  the  socket
       * type.
       * @author Rick L Vinyard Jr
       * @ingroup errors
       */
      class operation_not_supported: public write_error
        {
        public:
          operation_not_supported(): write_error("Some bit in the flags argument is inappropriate for  the  socket type.", EOPNOTSUPP)
          { }
        }
      ;


      /**
       * EPIPE
       * The  local  end  has  been  shut  down  on a connection oriented
       * socket.  In this case the process will also  receive  a  SIGPIPE
       * unless MSG_NOSIGNAL is set.
       * @author Rick L Vinyard Jr
       * @ingroup errors
       */
      class pipe: public write_error
        {
        public:
          pipe(): write_error("The  local  end  has  been  shut  down  on a connection oriented socket.", EPIPE)
          { }
        }
      ;

      /**
       * ECONNREFUSED
       * With writes, this is most likely triggered by an ICMP UNREACHABLE message.
       * Therefore, the first one (or even several) messages may transmit before
       * the ICMP mechanism reports UNREACHABLE.
       * @author Rick L Vinyard Jr
       * @ingroup errors
       */
      class connection_refused: public write_error
        {
        public:
          connection_refused(): write_error("The connection has been refused. This is probably the result of an ICMP UNREACHABLE message, so the first few transmissions may have actually gone out before the ICMP UNREACHABLE message was processed. This probably means that the destination (host or port) doesn't exist or isn't active.", ECONNREFUSED)
          { }
        }
      ;

      /**
       * Thrown when attempting to send without specifying a default address.
       * @author Rick L Vinyard Jr
       * @ingroup errors
       */
      class no_default_remote_address: public write_error
        {
        public:
          no_default_remote_address(): write_error("Socket cannot send because a default remote address has not been set.", -1)
          { }
        }
      ;

      /**
       * Thrown when sendto without an address is requested of the Socket class
       * @author Rick L Vinyard Jr
       * @ingroup errors
       */
      class no_sendto_without_address: public write_error
        {
        public:
          no_sendto_without_address(): write_error("Socket is not capable of sendto without an address.", -1)
          { }
        }
      ;

      /**
       * Writing on an I/O than can't open
       * @author Rick L Vinyard Jr
       * @ingroup errors
       */
      class not_opened: public write_error
        {
        public:
          not_opened(): write_error("Trying to write on I/O that cannot open.", -1)
          { }
        }
      ;

      /**
       * Writing to a read-only I/O
       * @author Rick L Vinyard Jr
       * @ingroup errors
       */
      class read_only: public write_error
        {
        public:
          read_only(): write_error("Trying to write to a read only I/O.", -1)
          { }
        }
      ;


    }

  }

}

#endif // CONEXUSSEND_ERROR_H
